--////////////////////////////////////////////////////////////////////////////
--/  Copyright (C) by RivieraWaves.
--/  This module is a confidential and proprietary property of RivieraWaves
--/  and a possession or use of this module requires written permission
--/  from RivieraWaves.
--/---------------------------------------------------------------------------
--/ $Author: cvandebu $
--/ Company          : RivieraWaves
--/---------------------------------------------------------------------------
--/ $Revision: 50 $
--/ $Date: 2011-09-21 18:10:05 +0200 (Wed, 21 Sep 2011) $
--/ --------------------------------------------------------------------------
--/ Dependencies     : None
--/ Description      :  RW_WLAN Modem Deserializer  1 bytes => 2 bits serialized
--/ Application Note :
--/ Terms & concepts :
--/ Bugs             :
--/ Open issues and future enhancements :
--/ References       :
--/ Revision History :
--/ --------------------------------------------------------------------------
--/
--/ $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MODEM80211BCORE/serializer/vhdl/rtl/deserializer.vhd $
--/
--////////////////////////////////////////////////////////////////////////////



--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library IEEE; 
use IEEE.STD_LOGIC_1164.ALL; 
use ieee.std_logic_unsigned.all; 
use ieee.std_logic_arith.all;
 

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity deserializer is
  port (
    -- clock and reset
    clk            : in  std_logic;                   
    resetn         : in  std_logic;                  
     
    -- inputs
    deseria_in     : in std_logic_vector (1 downto 0); 
    --               2-bits outputs 
    psk_mode       : in  std_logic;
    --               BPSK = 0 - QPSK = 1 
    init_byte      : in  std_logic;
    --               synchronize (start a new byte)
    deseria_enable : in  std_logic;
    --               enable the deserializer. Beware to disable the deserializer
    --               when no transfer is performed to not get any 
    --               phy_data_ind pulse. 
    shift_period  : in  std_logic_vector (3 downto 0);
    --              nb of (periods-1) between 2 shifts(if clk=11MHz,1MHz is 10)
    
    -- outputs
    deseria_out   : out std_logic_vector (7 downto 0);
    --              byte for the buffer 
    phy_data_ind  : out std_logic
    --              The modem indicates that the Tx path has read the 
    --              new octet.
  );

end deserializer;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture RTL of deserializer is

  ------------------------------------------------------------------------------
  -- Type
  ------------------------------------------------------------------------------
  type DESERIA_STATE is  ( idle,          -- idle phase
                           memo,          -- memorization of type of transfer
                           shift_op       -- shift operations
                         );    

  ------------------------------------------------------------------------------
  -- Constants
  ------------------------------------------------------------------------------
  constant TRANS_VAL_BPSK_CT : std_logic_vector(2 downto 0):= "111";
  -- in bpsk there are 8 shifts to perform
  constant TRANS_VAL_QPSK_CT : std_logic_vector(2 downto 0):= "011";
  -- in qpsk there are 4 shifts to perform

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  signal deseria_int        : std_logic_vector (7 downto 0);
  --                          deseria register
  signal trans_count        : std_logic_vector (2 downto 0);
  --                          count the number of shift operation to execute  
  signal trans_c_init_val   : std_logic_vector (2 downto 0);
  --                          nb of shift op to perform : 
  --                          BPSK => "111" (8) - QPSK => "11"(4)
  signal shift_per_count    : std_logic_vector (3 downto 0);
  --                          counter that reduces shift frequency
  signal shift_period_reg   : std_logic_vector (3 downto 0);
  --                          registered shift_period
  signal psk_mode_reg       : std_logic;
  --                          registered psk_mode
  signal psk_mode_mem       : std_logic;
  --                          (psk_mode at init) else (psk_mode_reg) 
  signal current_state      :   DESERIA_STATE;
  signal next_state         :   DESERIA_STATE;


--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin
  ------------------------------------------------------------------------------
  -- State Machine
  ------------------------------------------------------------------------------
  next_state_p : process (deseria_enable,trans_count,shift_per_count,
                          current_state, init_byte)
  begin
    case current_state is

      -- idle State: the seria is disabled
      when idle => 
        if deseria_enable = '1'  then
          next_state  <= memo;   
        else
          next_state  <= idle;
        end if;

      -- memo State: memorization of the 8bit data info
      when memo =>
        if deseria_enable = '0'  then
          next_state  <= idle;   
        elsif init_byte = '1' then
          next_state  <= memo;
        else
          next_state <= shift_op;
        end if;
 
      -- shift_op State: perform the shift op
      when shift_op =>
        if deseria_enable = '0'  then
          next_state  <= idle;   
        elsif (trans_count = "000" and shift_per_count = "0000")
                or init_byte = '1' then
           next_state  <= memo;
        else
          next_state  <= shift_op;
        end if;

      when others =>
        next_state   <= idle;
    end case;
  end process next_state_p;
  ------------------------------------------------------------------------------
  state_p : process (resetn, clk)
  begin
    if (resetn = '0') then
      current_state <= idle;
    elsif clk'event and clk = '1' then
      current_state <= next_state;
    end if;
  end process state_p;

  ------------------------------------------------------------------------------
  -- Deserialization process
  ------------------------------------------------------------------------------
  deseria_proc:process (clk, resetn)
  begin
    if resetn ='0' then
      deseria_int      <= (others => '0');     -- reset registers
    
    elsif (clk'event and clk = '1') then
      if (next_state = shift_op or next_state = memo)
         and shift_per_count = "0000" then          
        -- shift op
        if psk_mode_mem = '0' then             -- BPSK mode
          deseria_int (7)          <= deseria_in(0);
          deseria_int (6 downto 0) <= deseria_int(7 downto 1);
        else                                   -- QPSK mode
          deseria_int (7 downto 6) <= deseria_in;
          deseria_int (5 downto 4) <= deseria_int(7 downto 6);
          deseria_int (3 downto 2) <= deseria_int(5 downto 4);
          deseria_int (1 downto 0) <= deseria_int(3 downto 2);
        end if;
      end if;
    end if;
  end process;
  
  ------------------------------------------------------------------------------
  -- Counters process
  ------------------------------------------------------------------------------
  counters_proc:process (clk, resetn)
  begin
    if resetn ='0' then
      trans_count          <= (others => '0');
      shift_per_count        <= (others => '0');
      shift_period_reg       <= (others => '0');
      psk_mode_reg           <= '0';

    elsif (clk'event and clk = '1') then
       
     if next_state = memo then
        trans_count      <= trans_c_init_val;
        shift_per_count  <= shift_period; 
        shift_period_reg <= shift_period;
        psk_mode_reg     <= psk_mode;
        
      elsif next_state = shift_op then  
        shift_per_count <= shift_per_count - '1';
        if shift_per_count = "0000" then   
          trans_count <= trans_count - '1';
          shift_per_count <= shift_period_reg;
        end if;                                                              
      end if;
    end if;
  end process;

  ------------------------------------------------------------------------------
  -- wiring....
  ------------------------------------------------------------------------------
  trans_c_init_val <= TRANS_VAL_BPSK_CT when psk_mode = '0' else TRANS_VAL_QPSK_CT;
  -- number of shift op to perform : BPSK => "111" (8) - QPSK => "11"(4)
 
  deseria_out <= deseria_int;

  phy_data_ind <= '1' when trans_count = "000" and (next_state = shift_op 
                           or next_state = memo) 
                       else '0';

  psk_mode_mem    <= psk_mode when next_state = memo else psk_mode_reg;

end RTL;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

