//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthournout $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 35952 $
// $Date: 2018-11-05 17:48:51 +0100 (Mon, 05 Nov 2018) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of RIUAGCMemCtrl module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module RIUAGCMemCtrl
( 
  /*****************************************************************************
  * Resets
  *****************************************************************************/
  input  wire                             nRegBusRst,
  input  wire                             nAGCRst,

  /*****************************************************************************
  * Clocks
  *****************************************************************************/
  /* regbus */
  input  wire                             RegBusClk,
  input  wire                             AGCMemGClk,

  /*****************************************************************************
  * RIU registers
  *****************************************************************************/
  input  wire                             AgcRegAgcFsmReset,
`ifdef RW_NX_FIQ_COMP_EN
  input  wire                             CfgRegfIQCalEn,
`endif // RW_NX_FIQ_COMP_EN

  /*****************************************************************************
  * registers bank bus
  *****************************************************************************/
  input  wire                             RegBusSelAGCMEM,
  input  wire  [10:0]                     RegBusAddr,
  input  wire                             RegBusWrEn,
  input  wire                             RegBusRdEn,
  input  wire  [31:0]                     RegBusWrData,
  output reg   [31:0]                     RegBusRdData,
  output wire                             RegBusRdDataValid,
  output reg                              RegBusReady,

  /***************************************************************************
  * HDMCore
  ***************************************************************************/
  input wire                              AGCOn,               

  /*****************************************************************************
  * AGC memory control from RIUCore
  *****************************************************************************/
  input wire [8:0]                        RIUAGCMemAddr,

  /*****************************************************************************
  * fIQ memory
  *****************************************************************************/
`ifdef RW_NX_FIQ_COMP_EN
  input wire [7:0]                        fIQMemAddr,
  input wire                              fIQMemWrEn,
  input wire                              fIQMemRdEn,
  input wire [63:0]                       fIQMemWrData,
`endif // RW_NX_FIQ_COMP_EN

  /*****************************************************************************
  * AGC memory
  *****************************************************************************/
  input  wire [63:0]                      AGCMemRdData,
  output reg  [7:0]                       AGCMemAddr,
  output reg                              AGCMemEn,
  output reg  [1:0]                       AGCMemWrEn,
  output reg  [63:0]                      AGCMemWrData
  );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
  localparam RDWR_WRITE=1'b1, RDWR_READ=1'b0;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

  // in AGC domain
  
  // memory access request
  wire                             AGCReqValid;
  wire                             AGCReqRdWr;
  wire  [8:0]                      AGCReqAddr;
  wire  [31:0]                     AGCReqWrData;
  reg                              AGCReqValidD;
  // read data saved after access
  reg   [63:0]                     AGCReqRdData;
  reg                              AGCReqRdDataValid;
  wire                             AGCReqRdDataValidFeedback;
  // AGC memory interface
  reg   [7:0]                      AGCMemBusAddr;
  reg   [1:0]                      AGCMemBusWrEn;
  reg                              AGCMemBusRdEn;
  reg   [63:0]                     AGCMemBusWrData;
  reg                              AGCMemBusRdValid; // read data valid, 1 cycle after RdEn

  // in RegBus domain

  // Keep track of RegBus state to insert wait states when needed
  reg                              RegBusReadyInt; // keeps track of internal wait states
  reg                              RegBusWaitData; // high when RegBus waits data from AGC memory
  wire                             RegBusIdle;

  // request towards memory
  reg                              ReqValid0;
  reg                              ReqRdWr0;
  reg   [8:0]                      ReqAddr0;
  reg   [31:0]                     ReqWrData0;
  wire                             ReqFeedback0;
  reg                              ReqFeedback0D;
  wire                             ReqBusy0;
  // next request in queue
  reg                              ReqValid1;
  reg                              ReqRdWr1;
  reg   [8:0]                      ReqAddr1;
  reg   [31:0]                     ReqWrData1;
  // read data from memory
  wire  [63:0]                     ReqRdData64;
  wire                             ReqRdDataValid;
  reg                              ReqRdDataValidD;
  wire                             RdDataWordSel; // Select 32bits from 64bits data
  // Keep track of valid read 32bit words between memory accesses
  reg   [8:0]                      PreviousReqRdAddr;
  reg   [1:0]                      PreviousReqRdAddrValid; // 2 bits to address the 2 32bits words
  reg                              hit;
  reg                              hitWordSel;

  wire                             RegBusError, RegBusRdError;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// RegBusError:
//////////////////////////////////////////////////////////////////////////////
  // Detect if a Regbus access is done while AGC FSM is controlled by AGC.
  // Write accesses are ignored; Read accesses are answered immediately with error code
  // without requesting data from AGC memory. This would crash the bus system with ready
  // low forever.
  assign RegBusError   = RegBusSelAGCMEM & (~AgcRegAgcFsmReset 
`ifdef RW_NX_FIQ_COMP_EN
                                            | CfgRegfIQCalEn
`endif //  RW_NX_FIQ_COMP_EN
                                            );
  assign RegBusRdError = RegBusError & RegBusRdEn; 

// RegBusRdData:
//////////////////////////////////////////////////////////////////////////////
// Same as in Registers block, RIUAGCMemCtrl does not implement completely the RegBus protocol.
// The read data shall be registered outside of this block, as part of the top-level read data mux.
// As RIUAGCMemCtrl inserts wait states on the RegBus, the read data can't be sampled using 
// the RegBus Sel signal, so also output a Valid signal. 

  // Read data is valid when available from memory read, or when an address hit was detected
  // In case of RegBusError, answer immediately without wait state.
  assign RegBusRdDataValid = (ReqRdDataValid & ~ReqRdDataValidD) | hit | RegBusRdError;
  // Select correct 32bits from 64bits memory data
  assign RdDataWordSel     = hit ? hitWordSel : ReqAddr0[0];
  always @(*)
  begin
    if (RegBusRdError==1'b1) begin
      RegBusRdData = 32'hAAAADEAD;
    end else begin
      if (RdDataWordSel==1'b1)
        RegBusRdData = ReqRdData64[63:32];
      else
        RegBusRdData = ReqRdData64[31:0];
    end
  end


// RegBusReady:
//////////////////////////////////////////////////////////////////////////////
// RegBusReadyInt keeps track of the internal wait states. In order not to block
// the RegBus, RegBusReadyInt is not sent out if the RegBus access is not addressed to 
// the AGC memory. So for instance, a single write access to the AGC Memory triggers RegBusReadyInt low, 
// but it does not trigger RegBusReady low unless another read or write to the AGC memory happens before the
// first write is over.

  // No access or access not addressed to AGC memory
  assign RegBusIdle = (~RegBusSelAGCMEM || (~RegBusRdEn && ~RegBusWrEn));

  always @(*)
  begin
    // No access or access not addressed to AGC memory, and RegBus not waiting for AGC memory data:
    // Set RegBusReady high.
    if (RegBusIdle && ~RegBusWaitData)
      RegBusReady = 1'b1;
    else
      RegBusReady = RegBusReadyInt;
  end


// Request management in RegBus domain
//////////////////////////////////////////////////////////////////////////////
  
  // Keep track of request sent to AGC memory, including feedback
  assign ReqBusy0 = ReqValid0 | ReqFeedback0;

  // Pulse detection
  wire ReqRdDataValidRiseP;
  wire ReqFeedback0FallP;
  assign ReqRdDataValidRiseP = ReqRdDataValid & ~ReqRdDataValidD;
  assign ReqFeedback0FallP = ~ReqFeedback0 && ReqFeedback0D;
  
  // Manage read and write requests in RegBus domain
  always @(posedge RegBusClk, negedge nRegBusRst)
    if(!nRegBusRst) begin
      // RegBus state
      RegBusReadyInt         <= 1'b1;
      RegBusWaitData         <= 1'b0;
      // Keep track of Read data validity
      PreviousReqRdAddr      <= 9'd0;
      PreviousReqRdAddrValid <= 2'd0;
      hitWordSel             <= 1'b0;
      hit                    <= 1'b0;
      // Request to AGC domain
      ReqValid0              <= 1'b0;
      ReqRdWr0               <= 1'b0;
      ReqAddr0               <= 9'd0;
      ReqWrData0             <= 32'd0;
      // Request in queue
      ReqValid1              <= 1'b0;
      ReqRdWr1               <= 1'b0;
      ReqAddr1               <= 9'd0;
      ReqWrData1             <= 32'd0;
      // Delay registers
      ReqFeedback0D          <= 1'b0;
      ReqRdDataValidD        <= 1'b0;

    end else begin
      // Delay registers
      ReqFeedback0D      <= ReqFeedback0;
      ReqRdDataValidD    <= ReqRdDataValid;
      // RTZ during wait state
      hit            <= 1'b0;

      // Read data available from AGC domain: set valid for both 32bit words
      if (ReqRdDataValidRiseP==1'b1)
        PreviousReqRdAddrValid <= 2'b11;

      // Request received in AGC domain: reset ReqValid0.
      // No other request can be sent before ReqFeedback0 goes to 0.
      // This ensures the low level of ReqValid0 can be sampled in AGC domain.  
      if (ReqFeedback0==1'b1)
        ReqValid0     <= 1'b0;
      
      if (RegBusReady==1'b1)// begin
        RegBusWaitData <= 1'b0;

      // Request from RegBus: answered only if no RegBusError
      if (RegBusReady & RegBusSelAGCMEM & ~RegBusError) begin

        // Read request
        if (RegBusRdEn==1'b1) begin
          RegBusWaitData <= 1'b1;

          // Check if read data is available from a previous memory read
          if ((RegBusAddr[10:3]==PreviousReqRdAddr[8:1]) &&                           // 64bits memory word match
              ( ((RegBusAddr[2]==1'b0) && (PreviousReqRdAddrValid[0]==1'b1)) ||       // lower 32bit word addressed and valid
                ((RegBusAddr[2]==1'b1) && (PreviousReqRdAddrValid[1]==1'b1)) )) begin // upper 32bit word addressed and valid
            RegBusReadyInt <= 1'b0;          // one cycle wait
            hit            <= 1'b1;
            hitWordSel     <= RegBusAddr[2]; // Save 32bit sel as RegBus address can change during the data phase

          // Process read request
          end else begin
            RegBusReadyInt      <= 1'b0; // Hold bus while waiting for read data
            if (ReqBusy0==1'b1) begin // Memory access on going, save request in queue
              ReqValid1         <= 1'b1;
              ReqRdWr1          <= 1'b0;
              ReqAddr1          <= RegBusAddr[10:2];
            end else begin            // Post read request
              ReqValid0         <= 1'b1;
              ReqRdWr0          <= 1'b0;
              ReqAddr0          <= RegBusAddr[10:2];
              PreviousReqRdAddr <= RegBusAddr[10:2]; // Save address associated with the future read data
            end
            
          end
          
        // Write request
        end else if (RegBusWrEn==1'b1) begin
          if (ReqBusy0==1'b1) begin // Memory access on going, save request in queue
            RegBusReadyInt <= 1'b0; // Internal wait state as both queues are full
            ReqValid1      <= 1'b1;
            ReqRdWr1       <= 1'b1;
            ReqAddr1       <= RegBusAddr[10:2];
            ReqWrData1     <= RegBusWrData;
          end else begin // Post write request without wait state
            ReqValid0      <= 1'b1;
            ReqRdWr0       <= 1'b1;
            ReqAddr0       <= RegBusAddr[10:2];
            ReqWrData0     <= RegBusWrData;
            // Reset read data valid flag in case of write to a previously read data
            if (RegBusAddr[10:3]==PreviousReqRdAddr[8:1]) begin
              if (RegBusAddr[2]==1'b0)
                PreviousReqRdAddrValid[0] <= 1'b0;
              else
                PreviousReqRdAddrValid[1] <= 1'b0;
            end
          end
        end
      end
      
      // end of wait state after hit
      if (hit==1'b1)
        RegBusReadyInt <= 1'b1;

      // Request done, process request queue if any
      if (ReqFeedback0FallP==1'b1) begin
        RegBusReadyInt       <= 1'b1; // Release internal wait state
        if (ReqValid1==1'b1) begin
          ReqValid1   <= 1'b0;
          ReqValid0   <= 1'b1;
          ReqRdWr0    <= ReqRdWr1;
          ReqAddr0    <= ReqAddr1;
          ReqWrData0  <= ReqWrData1;
          // Read access: set RegBusReady low and store address associated with the future read data
          if (ReqRdWr1==1'b0) begin
            RegBusReadyInt    <= 1'b0;
            PreviousReqRdAddr <= ReqAddr1;
          // Write access: reset read data valid flag in case of write to a previously read data
          end else begin
            if (ReqAddr1[8:1]==PreviousReqRdAddr[8:1]) begin
              if (ReqAddr1[0]==1'b0)
                PreviousReqRdAddrValid[0] <= 1'b0;
              else
                PreviousReqRdAddrValid[1] <= 1'b0;
            end
            
          end
        end
      end

    end
 

// Request management to AGCMem domain
//////////////////////////////////////////////////////////////////////////////

  // Resync Request to AGCMem domain
  ClkSyncParam  
  #(
    .PARAMSIZE(     42)
  ) 
  U_Reg2AGCReqSync (
    .dstclk(       AGCMemGClk),
    .dstresetn(    nAGCRst),
    .srcupdate(    ReqValid0),
    .srcdata(      {ReqAddr0,ReqRdWr0,ReqWrData0}),
    .dstupdate(    AGCReqValid),
    .dstdata(      {AGCReqAddr,AGCReqRdWr,AGCReqWrData})
  );

  // Pulses
  wire AGCReqValidRiseP;
  assign AGCReqValidRiseP = AGCReqValid & ~AGCReqValidD;
  
  // Process request in AGC domain  
  always @(posedge AGCMemGClk, negedge nAGCRst)
    if(!nAGCRst) begin
      // Memory interface
      AGCMemBusAddr     <= 8'b0;
      AGCMemBusWrEn     <= 2'b0;
      AGCMemBusWrData   <= 64'b0;
      AGCMemBusRdEn     <= 1'b0;
      AGCMemBusRdValid  <= 1'b0;
      // Sample read data
      AGCReqRdData      <= 64'b0;
      AGCReqRdDataValid <= 1'b0;
      // Delay
      AGCReqValidD      <= 1'b0;

    end else begin
      AGCMemBusWrEn     <= 2'b0; // RTZ
      AGCMemBusRdEn     <= 1'b0; // RTZ
      // Delays
      AGCReqValidD      <= AGCReqValid;
      AGCMemBusRdValid  <= AGCMemBusRdEn;

      // Reset AGCReqRdDataValid on feedback from RegBus domain
      if (AGCReqRdDataValidFeedback==1'b1)
        AGCReqRdDataValid <= 1'b0;

      // Request received in AGC domain
      if (AGCReqValidRiseP==1'b1) begin
        AGCMemBusAddr   <= AGCReqAddr[8:1];
        // Write access: set memory wren to correct 32bit word based on Request address
        if (AGCReqRdWr==1'b1) begin // write access
          AGCMemBusWrEn   <= {AGCReqAddr[0],~AGCReqAddr[0]};
          AGCMemBusRdEn   <= 1'b0;
          AGCMemBusWrData <= {AGCReqWrData,AGCReqWrData};
        end else begin
          AGCMemBusWrEn   <= 2'b0;
          AGCMemBusRdEn   <= 1'b1;
        end
      end
      
      // Sample read data
      if (AGCMemBusRdValid==1'b1) begin
        AGCReqRdData      <= AGCMemRdData;
        AGCReqRdDataValid <= 1'b1;
      end
      
    end


// Request feedback to RegBus domain
//////////////////////////////////////////////////////////////////////////////

  // Feedback for request
  ClkSyncSimple U_AGC2ReqFeedback0Sync ( 
    .dstclk     (RegBusClk),
    .dstresetn  (nRegBusRst),
    .srcdata    (AGCReqValid),
    .dstdata    (ReqFeedback0)
  );

  // Read data to RegBus domain
  ClkSyncParam  
  #(
    .PARAMSIZE(     64)
  ) 
  U_AGC2ReqRdDataSync (
    .dstclk(       RegBusClk),
    .dstresetn(    nRegBusRst),
    .srcupdate(    AGCReqRdDataValid),
    .srcdata(      AGCReqRdData),
    .dstupdate(    ReqRdDataValid),
    .dstdata(      ReqRdData64)
  );


// Read data feedback to RegBus domain
//////////////////////////////////////////////////////////////////////////////

  // Feedback for read data valid
  ClkSyncSimple U_Req2AGCRdValidSync ( 
    .dstclk     (AGCMemGClk),
    .dstresetn  (nAGCRst),
    .srcdata    (ReqRdDataValid),
    .dstdata    (AGCReqRdDataValidFeedback)
  );

  // ensure the AHC/AHB sram mux is synchronously switched
  wire  AgcRegAgcFsmReset_2t;
  ClkSyncSimple U_AgcRegAgcFsmResetSync ( 
    .dstclk     (AGCMemGClk),
    .dstresetn  (nAGCRst),
    .srcdata    (AgcRegAgcFsmReset),
    .dstdata    (AgcRegAgcFsmReset_2t)
  );


// Mux for AGC memory control
//////////////////////////////////////////////////////////////////////////////

// AGC memory can be accessed by the following blocks:
// - when CfgRegfIQCalEn=1, read and write accesses are controlled by the fIQ module
// - else, when the AGC FSM is under reset, read and write accessed are done through the RegBus
// - else, the memory is dedicated to the AGC: read when AGC is on, write accesses disabled 

always @(*)
  begin
`ifdef RW_NX_FIQ_COMP_EN
    if (CfgRegfIQCalEn==1'b1) begin
      AGCMemAddr    = fIQMemAddr;
      AGCMemWrEn    = {fIQMemWrEn,fIQMemWrEn};
      AGCMemEn      = fIQMemRdEn | fIQMemWrEn;
      AGCMemWrData  = fIQMemWrData;
    end else begin
`endif // RW_NX_FIQ_COMP_EN
      AGCMemWrData  = AGCMemBusWrData;
      if (AgcRegAgcFsmReset_2t==1'b1) begin
        AGCMemAddr    = AGCMemBusAddr;
        AGCMemWrEn    = AGCMemBusWrEn;
        AGCMemEn      = AGCMemBusRdEn | AGCMemBusWrEn[0] | AGCMemBusWrEn[1];
      end else begin
        AGCMemAddr    = RIUAGCMemAddr[8:1];
        AGCMemWrEn    = 2'b0;
        AGCMemEn      = AGCOn;
      end
`ifdef RW_NX_FIQ_COMP_EN
    end
`endif // RW_NX_FIQ_COMP_EN
  end

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
