/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author:$
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     :                                       
* Description      : phy2fe_fifo
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module phy2fe_fifo
#(
  parameter g_width = 10
)
(
  /*****************************************************************************
  * PHY domain
  *****************************************************************************/
  input  wire               phy_rst_n, 
  input  wire               phy_clk,
  
  input  wire [1:0]         phy_cbw,
  input  wire [1:0]         phy_fbw,
  input  wire               phy_valid,
  input  wire [g_width-1:0] phy_data,
     
  /*****************************************************************************
  * FE domain
  *****************************************************************************/
  input  wire               fe_rst_n, 
  input  wire               fe_clk,
  
  output wire [1:0]         fe_fbw,
  output reg                fe_valid,
  output reg  [g_width-1:0] fe_data
);

  /*****************************************************************************
  * PHY domain
  *****************************************************************************/
  reg  [ 5:0]           w_valid_sr;
  reg                   w_start;
  reg  [ 1:0]           w_fbw;
  reg  [ 1:0]           w_cbw;
  reg  [g_width-1:0]    w_fifo_d0,w_fifo_d1,w_fifo_d2;
  reg                   w_fifo_v0,w_fifo_v1,w_fifo_v2;
  reg  [ 1:0]           w_fifo_ptr;
  wire                  w_wen;               
  
  assign w_wen = phy_valid |
                 (w_valid_sr[2] & phy_cbw==2'd0) |
                 (w_valid_sr[5] & phy_cbw!=2'd0);
  
  always @(posedge phy_clk,negedge phy_rst_n)
  begin
    if(!phy_rst_n)
    begin
      w_valid_sr            <= 6'b0;
      w_start               <= 1'b0;
      w_cbw                 <= 2'b0;
      w_fbw                 <= 2'b0;
      {w_fifo_v0,w_fifo_d0} <= {1'b0,{g_width{1'b0}}};
      {w_fifo_v1,w_fifo_d1} <= {1'b0,{g_width{1'b0}}};
      {w_fifo_v2,w_fifo_d2} <= {1'b0,{g_width{1'b0}}};
      w_fifo_ptr            <= 2'd0;       
    end
    else
    begin
      /* track phy_valid pattern to determine end of frame */
      w_valid_sr <= {w_valid_sr[4:0],phy_valid};
      
      if(!w_start)                                                              
      begin 
        w_fifo_ptr <= 2'd0;
        if(phy_valid)
        begin
          /* first write */                                                       
          w_start <= 1'b1;                                                        
          w_fbw   <= phy_fbw;                                                     
          w_cbw   <= phy_cbw;                                                     
          {w_fifo_ptr,w_fifo_v0,w_fifo_d0} <= {2'd1,1'b1,phy_data};  
        end          
      end                                                                       
      else                                                                      
      begin                                                                     
        if(w_wen)
        begin
          /* subsequent write */                                                  
          case(w_fifo_ptr)                                                        
            2'd0:   {w_fifo_ptr,w_fifo_v0,w_fifo_d0} <= {2'd1,phy_valid,phy_data};  
            2'd1:   {w_fifo_ptr,w_fifo_v1,w_fifo_d1} <= {2'd2,phy_valid,phy_data};  
            default:{w_fifo_ptr,w_fifo_v2,w_fifo_d2} <= {2'd0,phy_valid,phy_data};  
          endcase
          
          /* eof condition */
          if(!phy_valid)
            w_start <= 1'b0;
        end                                                                 
      end                                                                       
    end
  end
 
  /*****************************************************************************
  * FE domain
  *****************************************************************************/
  wire       r_start;
  reg        r_start_1t;
  reg        r_state;
  reg  [1:0] r_fifo_ptr; 
  reg  [2:0] r_cycle;
  reg  [1:0] r_cbw; 
  reg  [1:0] r_fbw; 
  wire       r_clr_cycle;
  wire       r_ren;
  
  ClkSyncSimple u_phy2fe_start_resync
  ( 
    .dstclk(         fe_clk),
    .dstresetn(      fe_rst_n),
    .srcdata(        w_start),   
    .dstdata(        r_start)   
  );
               
  assign r_clr_cycle = r_cbw==2'd0 & r_cycle==3'd1 | /* 0 1             */
                       r_cbw==2'd1 & r_cycle==3'd3 | /* 0 1 2 3         */
                       r_cbw==2'd2 & r_cycle==3'd7;  /* 0 1 2 3 4 5 6 7 */
  
  assign r_ren = ~r_state & r_start & ~r_start_1t | 
                  r_state & (
                 r_cbw==2'd0 & r_cycle==3'd1  |     
                 r_cbw==2'd1 & r_fbw==2'd0 &  r_cycle==3'd3 |
                 r_cbw==2'd1 & r_fbw==2'd1 & (r_cycle==3'd1 | r_cycle==3'd3) |
                 r_cbw==2'd2 & r_fbw==2'd0 &  r_cycle==3'd7 |
                 r_cbw==2'd2 & r_fbw==2'd1 & (r_cycle==3'd3 | r_cycle==3'd7) |
                 r_cbw==2'd2 & r_fbw==2'd2 & (r_cycle==3'd1 | r_cycle==3'd3  | r_cycle==3'd5 | r_cycle==3'd7));
  
  always @(posedge fe_clk,negedge fe_rst_n)
  begin
    if(!fe_rst_n)
    begin
      r_start_1t     <= 1'b0;
      r_fifo_ptr     <= 2'd0;
      r_cycle        <= 3'd0;
      r_fbw          <= 2'b0;
      r_cbw          <= 2'b0;
      r_state        <= 1'b0;
      
      fe_valid       <= 1'b0;
      fe_data        <= {g_width{1'b0}};
    end
    else
    begin
      /* rising edge detection */
      r_start_1t     <= r_start;
      
      if(!r_state)
      begin
        /* idle */
        /* NOTE: do not clear fe_fbw when returning in idle,          */ 
        /*       because needed by other blocks during frontend flush */
        r_fifo_ptr <= 2'd0;  
        r_cycle    <= 3'd0;  
        if(r_start &&!r_start_1t)
        begin
          /* first read */                                              
          r_state <= 1'b1;
          r_fbw   <= w_fbw;                                              
          r_cbw   <= w_cbw;
          {r_fifo_ptr,fe_valid,fe_data} <= {2'd1,w_fifo_v0,w_fifo_d0};
        end
      end
      else
      begin
        /* track cycle */
        if(r_clr_cycle)
          r_cycle <= 3'd0;
        else
          r_cycle <= r_cycle + 3'd1;
        
        /* return to idle */
        if(!fe_valid)
          r_state <= 1'b0;
       
        /* fifo */
        if(r_ren)
        begin
          /* subsequent read */
          case(r_fifo_ptr)
            2'd0:    {r_fifo_ptr,fe_valid,fe_data} <= {2'd1,w_fifo_v0,w_fifo_d0};
            2'd1:    {r_fifo_ptr,fe_valid,fe_data} <= {2'd2,w_fifo_v1,w_fifo_d1};
            default: {r_fifo_ptr,fe_valid,fe_data} <= {2'd0,w_fifo_v2,w_fifo_d2};
          endcase
        end
      end
    end
  end

  assign fe_fbw = r_fbw;

endmodule
`default_nettype wire
