////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of rw_he_mpif_ax_ac module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_he_mpif_ax_ac( 
   //$port_g Clock and Reset
   input  wire         clk,                      // Clock

   //$port_g MAC MPIF (AX)
   input  wire         mac_mpif_txreq,           // Tx request
   input  wire   [7:0] mac_mpif_txdata,          // Tx data
   input  wire         mac_mpif_macdatavalid,    // Data valid
   input  wire         mac_mpif_mimocmdvalid,    // Mimo command
   output wire         mac_mpif_phyrdy,          // Data valid signal
   output wire         mac_mpif_txend_p,         // End of transaction
   input  wire         mac_mpif_rxreq,           // Rx request
   output reg    [7:0] mac_mpif_rxdata,          // Rx data
   output wire         mac_mpif_cca_pri20,       // CCA on Primary 20MHz channel
   output wire         mac_mpif_cca_sec20,       // CCA on Secondary 20MHz channel
   output wire         mac_mpif_cca_sec40,       // CCA on Secondary 40MHz channel
   output reg          mac_mpif_rxendfortiming_p,// End of transmission (antenna)
   output reg          mac_mpif_rxerr_p,         // Rx error
   output reg          mac_mpif_rxend_p,         // Rx end
   input  wire         mac_mpif_keeprfon,        // Keep RF on
   output wire         mac_mpif_phyerr_p,        // Phy error
   output wire         mac_mpif_rifsrxdetected,  // RIFS detected
                                                 
   //$port_g PHY MPIF (AC)                       
   output reg          phy_mpif_txreq,           // Tx request
   output reg    [7:0] phy_mpif_txdata,          // Tx data
   output reg          phy_mpif_macdatavalid,    // Data valid
   output wire         phy_mpif_mimocmdvalid,    // Mimo command
   input  wire         phy_mpif_phyrdy,          // Data valid signal
   input  wire         phy_mpif_txend_p,         // End of transaction
   output wire         phy_mpif_rxreq,           // Rx request
   input  wire   [7:0] phy_mpif_rxdata,          // Rx data
   input  wire         phy_mpif_cca_pri20,       // CCA on Primary 20MHz channel
   input  wire         phy_mpif_cca_sec20,       // CCA on Secondary 20MHz channel
   input  wire         phy_mpif_cca_sec40,       // CCA on Secondary 40MHz channel
   input  wire         phy_mpif_rxendfortiming_p,// End of transmission (antenna)
   input  wire         phy_mpif_rxerr_p,         // Rx error
   input  wire         phy_mpif_rxend_p,         // Rx end
   output wire         phy_mpif_keeprfon,        // Keep RF on
   input  wire         phy_mpif_phyerr_p,        // Phy error
   input  wire         phy_mpif_rifsrxdetected   // RIFS detected
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// Format and Modulation
localparam
   NON_HT       = 4'b0000,
   NON_HT_DUP   = 4'b0001,
   HT_MM        = 4'b0010,
   HT_GF        = 4'b0011,
   VHT          = 4'b0100,
   HE_SU        = 4'b0101,
   HE_MU        = 4'b0110,
   HE_EXT_SU    = 4'b0111,
   HE_TRIG      = 4'b1000;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
// TX Vector from MAC
reg  [7:0] mac_tx_vector_byte[0:25];
reg  [4:0] mac_tx_vector_cnt;
reg  [4:0] mac_tx_vector_len;
wire       mac_tx_vector_done;

// TX Vector to PHY
reg  [7:0] phy_tx_vector_byte[0:15];
reg  [4:0] phy_tx_vector_cnt;
wire       phy_tx_vector_done;

// RX Vector from PHY
reg  [7:0] phy_rx_vector_byte[0:14];
reg  [3:0] phy_rx_vector_cnt;
wire       phy_rx_vector_early_done;
wire       phy_rx_vector_done;
reg        phy_rx_vector_eft; // End For Timing received during RX Vector

// RX Vector to MAC
reg  [7:0] mac_rx_vector_byte[0:12];
reg  [3:0] mac_rx_vector_len;
reg        mac_mpif_phyrdy_rx;
reg        mac_mpif_phyerr_rx_p;

// TX Vector
reg  [3:0] tx_vector_format;
reg  [2:0] tx_vector_chbw;
reg        tx_vector_pretype;
reg  [7:0] tx_vector_antennaset;
reg  [7:0] tx_vector_txpwrlevel;
reg  [2:0] tx_vector_ntx;
reg        tx_vector_timeofdeparturereq;
reg        tx_vector_continuoustx;
reg [11:0] tx_vector_leglength;
reg  [3:0] tx_vector_legrate;
reg [15:0] tx_vector_service;
reg        tx_vector_triggerresponding;
reg  [7:0] tx_vector_smmindex;
reg        tx_vector_sounding;
reg        tx_vector_smoothing;
reg  [1:0] tx_vector_gitype;
reg        tx_vector_aggregation;
reg        tx_vector_stbc;
reg  [1:0] tx_vector_numextnss;
reg  [6:0] tx_vector_mcs;
reg  [2:0] tx_vector_nss;
reg        tx_vector_feccoding;
reg [19:0] tx_vector_htlength;
reg        tx_vector_beamformed;
reg        tx_vector_dozenotallowed;
reg  [8:0] tx_vector_partialaid;
reg  [5:0] tx_vector_groupid;
reg  [7:0] tx_vector_nuser;
reg  [4:0] tx_vector_userposition;

// RX Vector
reg [11:0] rx_vector_leglength;
reg  [3:0] rx_vector_legrate;
reg [19:0] rx_vector_htlength;
reg        rx_vector_shortgi;
reg  [1:0] rx_vector_stbc;
reg        rx_vector_smoothing;
reg  [6:0] rx_vector_mcs;
reg        rx_vector_pretype;
reg  [2:0] rx_vector_formatmod;
reg  [1:0] rx_vector_chbw;
reg  [2:0] rx_vector_nsts;
reg        rx_vector_lsigvalid;
reg        rx_vector_sounding;
reg  [1:0] rx_vector_numextss_chbwinnonht;
reg        rx_vector_aggregation;
reg        rx_vector_feccoding;
reg        rx_vector_dynbw;
reg        rx_vector_dozenotallowed;
reg  [7:0] rx_vector_antennaset;
reg  [8:0] rx_vector_partialaid;
reg  [5:0] rx_vector_groupid;
reg        rx_vector_firstuser;
reg  [7:0] rx_vector_rssi1;
reg  [7:0] rx_vector_rssi2;
reg  [7:0] rx_vector_rssi3;
reg  [7:0] rx_vector_rssi4;

// RX Buffer
reg [12:0] rx_buffer[0:15];
reg [12:0] rx_buffer_next[0:16];
reg [15:0] rx_buffer_wr_ptr;
wire       rx_buffer_rd;
wire       rx_buffer_wr;
reg        rx_buffer_wr_forced;


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
//
//******************************************************************************
assign phy_mpif_mimocmdvalid     = mac_mpif_mimocmdvalid;
assign phy_mpif_keeprfon         = mac_mpif_keeprfon;
                                 
assign mac_mpif_cca_pri20        = phy_mpif_cca_pri20;
assign mac_mpif_cca_sec20        = phy_mpif_cca_sec20;
assign mac_mpif_cca_sec40        = phy_mpif_cca_sec40;
assign mac_mpif_rifsrxdetected   = phy_mpif_rifsrxdetected;


//******************************************************************************
// MAC TX VECTOR STORE
//******************************************************************************
always @(posedge clk)
begin: mac_tx_vector_save
   // local variable
   integer v_i;

   if (~mac_mpif_txreq)
   begin
      for (v_i=0;v_i<=25;v_i=v_i+1) mac_tx_vector_byte[v_i] <= 8'h0;
      mac_tx_vector_cnt <= 5'h0;
   end
   else if (mac_mpif_macdatavalid & ~mac_tx_vector_done)
   begin
      mac_tx_vector_byte[mac_tx_vector_cnt] <= mac_mpif_txdata;
      mac_tx_vector_cnt                     <= mac_tx_vector_cnt+5'd1;
   end
end

always @(posedge clk)
begin
   if (~mac_mpif_txreq)
      mac_tx_vector_len <= 5'd10; // Default NON-HT
   else if (mac_mpif_macdatavalid & mac_tx_vector_cnt==5'd1)
      case (tx_vector_format)
      NON_HT    : mac_tx_vector_len <= 5'd10;
      NON_HT_DUP: mac_tx_vector_len <= 5'd10;
      HT_MM     : mac_tx_vector_len <= 5'd13;
      HT_GF     : mac_tx_vector_len <= 5'd13;
      VHT       : mac_tx_vector_len <= 5'd17;
      HE_SU     : mac_tx_vector_len <= 5'd18;
      HE_EXT_SU : mac_tx_vector_len <= 5'd18;
      //HE_MU   : mac_tx_vector_len <= 5'd26;
      HE_TRIG   : mac_tx_vector_len <= 5'd24;
      endcase
end
assign mac_tx_vector_done = (mac_tx_vector_cnt==mac_tx_vector_len);


//******************************************************************************
// MAC TX VECTOR DECODE
//******************************************************************************
always @*
begin
   // Default Value
   tx_vector_format                = 4'b0;
   tx_vector_chbw                  = 3'b0;
   tx_vector_pretype               = 1'b0;
   tx_vector_antennaset            = 8'b0;
   tx_vector_txpwrlevel            = 8'b0;
   tx_vector_ntx                   = 3'b0;
   tx_vector_timeofdeparturereq    = 1'b0;
   tx_vector_continuoustx          = 1'b0;
   tx_vector_leglength             = 12'b0;
   tx_vector_legrate               = 4'b0;
   tx_vector_service               = 16'b0;
   tx_vector_triggerresponding     = 1'b0;
   tx_vector_smmindex              = 8'b0;
   tx_vector_sounding              = 1'b0;
   tx_vector_smoothing             = 1'b0;
   tx_vector_gitype                = 2'b0;
   tx_vector_aggregation           = 1'b0;
   tx_vector_stbc                  = 1'b0;
   tx_vector_numextnss             = 2'b0;
   tx_vector_mcs                   = 7'b0;
   tx_vector_nss                   = 3'b0;
   tx_vector_feccoding             = 1'b0;
   tx_vector_htlength              = 20'b0;
   tx_vector_beamformed            = 1'b0;
   tx_vector_dozenotallowed        = 1'b0;
   tx_vector_partialaid            = 9'b0;
   tx_vector_groupid               = 6'b0;
   tx_vector_nuser                 = 8'b0;
   tx_vector_userposition          = 5'b0;

   // TX Vector Common Part for all frames
   tx_vector_format                = mac_tx_vector_byte[0][3:0];
   tx_vector_chbw                  = mac_tx_vector_byte[0][6:4];
   tx_vector_pretype               = mac_tx_vector_byte[0][7];
   tx_vector_antennaset            = mac_tx_vector_byte[1][7:0];
   tx_vector_txpwrlevel            = mac_tx_vector_byte[2][7:0];
   tx_vector_ntx                   = mac_tx_vector_byte[3][2:0];
   tx_vector_timeofdeparturereq    = mac_tx_vector_byte[3][6];
   tx_vector_continuoustx          = mac_tx_vector_byte[3][7];
   tx_vector_leglength[7:0]        = mac_tx_vector_byte[4][7:0];
   tx_vector_leglength[11:8]       = mac_tx_vector_byte[5][3:0];
   tx_vector_legrate               = mac_tx_vector_byte[5][7:4];
   tx_vector_service[7:0]          = mac_tx_vector_byte[6][7:0];
   tx_vector_service[15:8]         = mac_tx_vector_byte[7][7:0];

   case (tx_vector_format)
   NON_HT, NON_HT_DUP:
   begin
      // TX Vector for NON-HT frames
      tx_vector_triggerresponding  = mac_tx_vector_byte[8][0];
      tx_vector_smmindex           = mac_tx_vector_byte[9][7:0];
   end                             
   HT_MM, HT_GF:                   
   begin                           
      // TX Vector for HT frames  
      tx_vector_sounding           = mac_tx_vector_byte[8][0];
      tx_vector_smoothing          = mac_tx_vector_byte[8][1];
      tx_vector_gitype[0]          = mac_tx_vector_byte[8][2];
      tx_vector_aggregation        = mac_tx_vector_byte[8][3];
      tx_vector_stbc               = mac_tx_vector_byte[8][4];
      tx_vector_numextnss          = mac_tx_vector_byte[8][6:5];
      tx_vector_smmindex           = mac_tx_vector_byte[9][7:0];
      tx_vector_mcs                = mac_tx_vector_byte[10][6:0];
      tx_vector_feccoding          = mac_tx_vector_byte[10][7];
      tx_vector_htlength[7:0]      = mac_tx_vector_byte[11][7:0];
      tx_vector_htlength[15:8]     = mac_tx_vector_byte[12][7:0];
   end
   VHT:
   begin
      // TX Vector for VHT frames
      tx_vector_sounding           = mac_tx_vector_byte[8][0];
      tx_vector_beamformed         = mac_tx_vector_byte[8][1];
      tx_vector_gitype[0]          = mac_tx_vector_byte[8][2];
      tx_vector_aggregation        = 1'b1;
      tx_vector_stbc               = mac_tx_vector_byte[8][4];
      tx_vector_dozenotallowed     = mac_tx_vector_byte[8][5];
      tx_vector_partialaid[7:0]    = mac_tx_vector_byte[9][7:0];
      tx_vector_partialaid[8]      = mac_tx_vector_byte[10][0];
      tx_vector_groupid            = mac_tx_vector_byte[10][6:1];
      tx_vector_nuser[2:0]         = mac_tx_vector_byte[11][2:0];
      tx_vector_smmindex           = mac_tx_vector_byte[12][7:0];
      tx_vector_mcs[3:0]           = mac_tx_vector_byte[13][3:0];
      tx_vector_nss                = mac_tx_vector_byte[13][6:4];
      tx_vector_feccoding          = mac_tx_vector_byte[13][7];
      tx_vector_htlength[7:0]      = mac_tx_vector_byte[14][7:0];
      tx_vector_htlength[15:8]     = mac_tx_vector_byte[15][7:0];
      tx_vector_htlength[19:16]    = mac_tx_vector_byte[16][3:0];
      tx_vector_userposition[1:0]  = mac_tx_vector_byte[16][7:6];
   end
   default: // HE
   begin
      // TX Vector HE Common Part
      tx_vector_aggregation        = 1'b1;
      tx_vector_sounding           = mac_tx_vector_byte[8][0];
      tx_vector_beamformed         = mac_tx_vector_byte[8][1];
      tx_vector_gitype             = mac_tx_vector_byte[8][3:2];
      tx_vector_stbc               = mac_tx_vector_byte[8][4];
      case (tx_vector_format)
      HE_SU, HE_EXT_SU:
      begin
         // TX Vector for HE-SU frames
         tx_vector_smmindex        = mac_tx_vector_byte[13][7:0];
         tx_vector_mcs[3:0]        = mac_tx_vector_byte[14][3:0];
         tx_vector_nss             = mac_tx_vector_byte[14][6:4];
         tx_vector_feccoding       = mac_tx_vector_byte[14][7];
         tx_vector_htlength[7:0]   = mac_tx_vector_byte[15][7:0];
         tx_vector_htlength[15:8]  = mac_tx_vector_byte[16][7:0];
         tx_vector_htlength[19:16] = mac_tx_vector_byte[17][3:0];
      end
      HE_MU:
      begin
         // TX Vector for HE-MU frames
         tx_vector_nuser           = mac_tx_vector_byte[18][7:0];
         tx_vector_smmindex        = mac_tx_vector_byte[19][7:0];
         tx_vector_mcs[3:0]        = mac_tx_vector_byte[20][3:0];
         tx_vector_nss             = mac_tx_vector_byte[20][6:4];
         tx_vector_feccoding       = mac_tx_vector_byte[20][7];
         tx_vector_htlength[7:0]   = mac_tx_vector_byte[21][7:0];
         tx_vector_htlength[15:8]  = mac_tx_vector_byte[22][7:0];
         tx_vector_htlength[19:16] = mac_tx_vector_byte[23][3:0];
         tx_vector_userposition    = mac_tx_vector_byte[25][7:3];
      end
      HE_TRIG:
      begin
         // TX Vector for HE-TRIG frames
         tx_vector_smmindex        = mac_tx_vector_byte[18][7:0];
         tx_vector_mcs[3:0]        = mac_tx_vector_byte[19][3:0];
         tx_vector_nss             = mac_tx_vector_byte[19][6:4];
         tx_vector_feccoding       = mac_tx_vector_byte[19][7];
         tx_vector_htlength[7:0]   = mac_tx_vector_byte[20][7:0];
         tx_vector_htlength[15:8]  = mac_tx_vector_byte[21][7:0];
         tx_vector_htlength[19:16] = mac_tx_vector_byte[22][3:0];
      end
      endcase
   end
   endcase
end


//******************************************************************************
// PHY TX VECTOR ENCODE
//******************************************************************************
always @*
begin: tx_vector_encode
   // Local variable
   integer v_i;

   // Default Value
   for (v_i=0;v_i<=15;v_i=v_i+1) phy_tx_vector_byte[v_i] = 8'h0;

   phy_tx_vector_byte[0][7:0]   = tx_vector_txpwrlevel;
   phy_tx_vector_byte[1][0]     = tx_vector_sounding;
   phy_tx_vector_byte[1][1]     = tx_vector_feccoding;
   phy_tx_vector_byte[1][3]     = tx_vector_smoothing;
   phy_tx_vector_byte[1][4]     = tx_vector_continuoustx;
   phy_tx_vector_byte[1][7:6]   = tx_vector_chbw[1:0];
   phy_tx_vector_byte[2][7:0]   = tx_vector_antennaset;
   phy_tx_vector_byte[3][7:0]   = tx_vector_smmindex;
   phy_tx_vector_byte[4][6:0]   = tx_vector_mcs;
   phy_tx_vector_byte[4][7]     = tx_vector_pretype;
   phy_tx_vector_byte[5][2:0]   = tx_vector_format[2:0];
   phy_tx_vector_byte[5][4:3]   = tx_vector_numextnss;
   phy_tx_vector_byte[5][7:5]   = tx_vector_format<4'd4 ? 3'b0 :                     // NSTS=0
                                  tx_vector_stbc        ? {tx_vector_nss[1:0],1'b1}: // NSTS=2*Nss
                                                           tx_vector_nss[2:0];       // NSTS=Nss
   phy_tx_vector_byte[6][7:0]   = tx_vector_leglength[7:0];
   phy_tx_vector_byte[7][3:0]   = tx_vector_leglength[11:8];
   phy_tx_vector_byte[7][7:4]   = tx_vector_legrate[3:0];
   phy_tx_vector_byte[8][7:0]   = tx_vector_service[7:0];
   phy_tx_vector_byte[9][7:0]   = tx_vector_service[15:8];
   phy_tx_vector_byte[10][7:0]  = tx_vector_htlength[7:0];
   phy_tx_vector_byte[11][7:0]  = tx_vector_htlength[15:8];
   phy_tx_vector_byte[12][3:0]  = tx_vector_htlength[19:16];
   phy_tx_vector_byte[12][4]    = tx_vector_aggregation;
   phy_tx_vector_byte[12][5]    = tx_vector_dozenotallowed;
   phy_tx_vector_byte[12][6]    = tx_vector_gitype[0];
   phy_tx_vector_byte[12][7]    = 1'b0;                                          //disambiguitybit
   phy_tx_vector_byte[13][2:0]  = tx_vector_ntx[2:0];
   phy_tx_vector_byte[13][5:4]  = {1'b0,tx_vector_stbc}; // ToDo: userposition for MUMIMO VHT
   phy_tx_vector_byte[13][6]    = tx_vector_beamformed;
   phy_tx_vector_byte[14][7:0]  = tx_vector_partialaid[7:0];
   phy_tx_vector_byte[15][0]    = tx_vector_partialaid[8];
   phy_tx_vector_byte[15][6:1]  = tx_vector_groupid;
end


//******************************************************************************
// PHY TX VECTOR & DATA
//******************************************************************************
always @(posedge clk)
begin
   if (~mac_mpif_txreq)
      phy_mpif_txreq    <= 1'b0;
   else if (mac_tx_vector_done)
      phy_mpif_txreq    <= 1'b1;
end

always @(posedge clk)
begin
   if (~phy_mpif_txreq)
      phy_tx_vector_cnt <= 5'd0;
   else if (~phy_tx_vector_done)
      phy_tx_vector_cnt <= phy_tx_vector_cnt+5'd1;
end

assign phy_tx_vector_done = (phy_tx_vector_cnt==5'd16);

always @*
begin
   if (~phy_tx_vector_done)
   begin
      phy_mpif_txdata       = phy_tx_vector_byte[phy_tx_vector_cnt[3:0]];
      phy_mpif_macdatavalid = 1'b0;
   end
   else
   begin
      phy_mpif_txdata       = mac_mpif_txdata;
      phy_mpif_macdatavalid = mac_mpif_macdatavalid;
   end
end

assign mac_mpif_txend_p  = phy_mpif_txend_p;
assign mac_mpif_phyrdy   = mac_mpif_txreq ? phy_mpif_phyrdy   : mac_mpif_phyrdy_rx;
assign mac_mpif_phyerr_p = mac_mpif_txreq ? phy_mpif_phyerr_p : mac_mpif_phyerr_rx_p;


//******************************************************************************
// PHY RX VECTOR STORE
//******************************************************************************
always @(posedge clk)
begin: phy_rx_vector_save
   // local variable
   integer v_i;

   if (~phy_mpif_rxreq | mac_mpif_rxend_p)
   begin
      for (v_i=0;v_i<=14;v_i=v_i+1) phy_rx_vector_byte[v_i] <= 8'h0;
      phy_rx_vector_cnt <= 4'h0;
   end
   else if (phy_mpif_phyrdy & ~phy_rx_vector_done)
   begin
      phy_rx_vector_byte[phy_rx_vector_cnt] <= phy_mpif_rxdata;
      phy_rx_vector_cnt                     <= phy_rx_vector_cnt+4'd1;
   end
end

assign phy_rx_vector_early_done = (phy_rx_vector_cnt>=4'd12);
assign phy_rx_vector_done       = (phy_rx_vector_cnt==4'd15);

always @(posedge clk)
begin
   if (~phy_mpif_rxreq | phy_rx_vector_done)
      phy_rx_vector_eft <= 1'b0;
   else if (phy_mpif_rxendfortiming_p)
      phy_rx_vector_eft <= 1'b1;
end


//******************************************************************************
// PHY RX VECTOR DECODE
//******************************************************************************
always @*
begin
   rx_vector_leglength[7:0]       = phy_rx_vector_byte[0][7:0];
   rx_vector_leglength[11:8]      = phy_rx_vector_byte[1][3:0];
   rx_vector_legrate[3:0]         = phy_rx_vector_byte[1][7:4];
   rx_vector_htlength[7:0]        = phy_rx_vector_byte[2][7:0];
   rx_vector_htlength[15:8]       = phy_rx_vector_byte[3][7:0];
   rx_vector_htlength[19:16]      = phy_rx_vector_byte[4][3:0];
   rx_vector_shortgi              = phy_rx_vector_byte[4][4];
   rx_vector_stbc[1:0]            = phy_rx_vector_byte[4][6:5];
   rx_vector_smoothing            = phy_rx_vector_byte[4][7];
   rx_vector_mcs                  = phy_rx_vector_byte[5][6:0];
   rx_vector_pretype              = phy_rx_vector_byte[5][7];
   rx_vector_formatmod            = phy_rx_vector_byte[6][2:0];
   rx_vector_chbw                 = phy_rx_vector_byte[6][4:3];
   rx_vector_nsts                 = phy_rx_vector_byte[6][7:5];
   rx_vector_lsigvalid            = phy_rx_vector_byte[7][0];
   rx_vector_sounding             = phy_rx_vector_byte[7][1];
   rx_vector_numextss_chbwinnonht = phy_rx_vector_byte[7][3:2];
   rx_vector_aggregation          = phy_rx_vector_byte[7][4];
   rx_vector_feccoding            = phy_rx_vector_byte[7][5];
   rx_vector_dynbw                = phy_rx_vector_byte[7][6];
   rx_vector_dozenotallowed       = phy_rx_vector_byte[7][7];
   rx_vector_antennaset           = phy_rx_vector_byte[8][7:0];
   rx_vector_partialaid[7:0]      = phy_rx_vector_byte[9][7:0];
   rx_vector_partialaid[8]        = phy_rx_vector_byte[10][0];
   rx_vector_groupid              = phy_rx_vector_byte[10][6:1];
   rx_vector_firstuser            = phy_rx_vector_byte[10][7];
   rx_vector_rssi1                = phy_rx_vector_byte[11][7:0];
   rx_vector_rssi2                = phy_rx_vector_byte[12][7:0];
   rx_vector_rssi3                = phy_rx_vector_byte[13][7:0];
   rx_vector_rssi4                = phy_rx_vector_byte[14][7:0];
end


//******************************************************************************
// MAC RX VECTOR ENCODE
//******************************************************************************
always @*
begin: mac_rx_vector_encode
   // local variable
   integer v_i;

   // Default Value
   for (v_i=0;v_i<=12;v_i=v_i+1) mac_rx_vector_byte[v_i] = 8'h0;
   mac_rx_vector_len = 4'd7;

   // RX Vector Common Part for all frames
   mac_rx_vector_byte[0][3:0]    = {1'b0,rx_vector_formatmod};
   mac_rx_vector_byte[0][6:4]    = {1'b0,rx_vector_chbw};
   mac_rx_vector_byte[0][7]      = rx_vector_pretype;
   mac_rx_vector_byte[1][7:0]	   = rx_vector_antennaset;
   mac_rx_vector_byte[2][7:0]	   = rx_vector_rssi1;
   mac_rx_vector_byte[3][7:0]	   = rx_vector_leglength[7:0];
   mac_rx_vector_byte[4][3:0]	   = rx_vector_leglength[11:8];
   mac_rx_vector_byte[4][7:4]	   = rx_vector_legrate[3:0];
   mac_rx_vector_byte[5][7:0]	   = rx_vector_rssi1;

   case ({1'b0,rx_vector_formatmod})
   NON_HT, NON_HT_DUP:
   begin
      // RX Vector for NON-HT frames
      mac_rx_vector_byte[6][0]     = rx_vector_dynbw;
      mac_rx_vector_byte[6][2:1]   = rx_vector_numextss_chbwinnonht;
      mac_rx_vector_byte[6][7]     = rx_vector_lsigvalid;
      mac_rx_vector_len            = 4'd7;
   end
   HT_MM, HT_GF:
   begin
      // RX Vector for HT frames  
      mac_rx_vector_byte[6][0]     = rx_vector_sounding;
      mac_rx_vector_byte[6][1]     = rx_vector_smoothing;
      mac_rx_vector_byte[6][2]     = rx_vector_shortgi;
      mac_rx_vector_byte[6][3]     = rx_vector_aggregation;
      mac_rx_vector_byte[6][4]     = |rx_vector_stbc;
      mac_rx_vector_byte[6][6:5]   = rx_vector_numextss_chbwinnonht;
      mac_rx_vector_byte[6][7]     = rx_vector_lsigvalid;
      mac_rx_vector_byte[7][6:0]   = rx_vector_mcs;
      mac_rx_vector_byte[7][7]     = rx_vector_feccoding;
      mac_rx_vector_byte[8][7:0]   = rx_vector_htlength[7:0];
      mac_rx_vector_byte[9][7:0]   = rx_vector_htlength[15:8];
      mac_rx_vector_len            = 4'd10;
   end
   VHT:
   begin
      // RX Vector for VHT frames
      mac_rx_vector_byte[6][0]     = rx_vector_sounding;
      mac_rx_vector_byte[6][1]     = 1'b0;//BEAMFORMED
      mac_rx_vector_byte[6][2]     = rx_vector_shortgi;
      mac_rx_vector_byte[6][4]     = |rx_vector_stbc;
      mac_rx_vector_byte[6][5]     = rx_vector_dozenotallowed;
      mac_rx_vector_byte[6][6]     = rx_vector_firstuser;
      mac_rx_vector_byte[7][7:0]   = rx_vector_partialaid[7:0];
      mac_rx_vector_byte[8][0]     = rx_vector_partialaid[8];
      mac_rx_vector_byte[8][6:1]   = rx_vector_groupid;
      mac_rx_vector_byte[9][3:0]   = rx_vector_mcs[3:0];
      mac_rx_vector_byte[9][6:4]   = |rx_vector_stbc ? {1'b0,rx_vector_nsts[2:1]} : //Nss=Nsts/2
                                                       rx_vector_nsts;              //Nss=Nsts
      mac_rx_vector_byte[9][7]     = rx_vector_feccoding;
      mac_rx_vector_byte[10][7:0]  = rx_vector_htlength[7:0];
      mac_rx_vector_byte[11][7:0]  = rx_vector_htlength[15:8];
      mac_rx_vector_byte[12][3:0]  = rx_vector_htlength[19:16];
      mac_rx_vector_len            = 4'd13;
   end
   endcase
end


//******************************************************************************
// MAC RX VECTOR & DATA
//******************************************************************************
assign phy_mpif_rxreq = mac_mpif_rxreq;

always @*
begin
   if (~phy_rx_vector_early_done)
   begin
      mac_mpif_rxdata           = 8'h0;
      mac_mpif_phyrdy_rx        = 1'b0;
      mac_mpif_rxendfortiming_p = 1'b0;
      mac_mpif_rxend_p          = 1'b0;
      mac_mpif_rxerr_p          = 1'b0;
      mac_mpif_phyerr_rx_p      = 1'b0;
   end
   else
   begin
      mac_mpif_rxdata           = rx_buffer[0][7:0];
      mac_mpif_phyrdy_rx        = rx_buffer[0][8];
      mac_mpif_rxendfortiming_p = rx_buffer[0][9];
      mac_mpif_rxend_p          = rx_buffer[0][10];
      mac_mpif_rxerr_p          = rx_buffer[0][11];
      mac_mpif_phyerr_rx_p      = rx_buffer[0][12];
   end
end


//******************************************************************************
// PHY RX BUFFER
//******************************************************************************
always @(posedge clk)
begin
   if (~phy_rx_vector_early_done)
   begin
      rx_buffer_wr_ptr <= 16'h0;
   end
   else
   begin
      if (rx_buffer_wr_ptr==16'h0)
         rx_buffer_wr_ptr[mac_rx_vector_len] <= 1'b1;
      else if (rx_buffer_wr & ~rx_buffer_rd)
         rx_buffer_wr_ptr <= {rx_buffer_wr_ptr[14:0],1'b0};
      else if (~rx_buffer_wr & rx_buffer_rd)
         rx_buffer_wr_ptr <= {1'b0,rx_buffer_wr_ptr[15:1]};
   end
end

always @(posedge clk)
begin: rx_buffer_reg
   // local variable
   integer v_i;

   if (~phy_rx_vector_early_done)
   begin
      for (v_i=0;v_i<16;v_i=v_i+1) rx_buffer[v_i] <= 13'h0;
   end
   else
   begin
      if (rx_buffer_wr_ptr==16'h0)
         for (v_i=0;v_i<13;v_i=v_i+1)
            if (v_i<mac_rx_vector_len) rx_buffer[v_i] <= {5'b00001,mac_rx_vector_byte[v_i]};
            else                       rx_buffer[v_i] <= {5'b00000,mac_rx_vector_byte[v_i]};

      else
         for (v_i=0;v_i<16;v_i=v_i+1) rx_buffer[v_i] <= rx_buffer_next[v_i];
   end
end

always @*
begin: rx_buffer_comb
   // local variable
   integer v_i;

   rx_buffer_next[16] = 13'h0;

   // Write
   for (v_i=0;v_i<16;v_i=v_i+1)
      if (rx_buffer_wr & rx_buffer_wr_ptr[v_i])
      begin
         rx_buffer_next[v_i][7:0] = phy_mpif_rxdata;
         rx_buffer_next[v_i][8]   = phy_mpif_phyrdy;
         rx_buffer_next[v_i][9]   = phy_mpif_rxendfortiming_p | phy_rx_vector_eft;
         rx_buffer_next[v_i][10]  = phy_mpif_rxend_p;
         rx_buffer_next[v_i][11]  = phy_mpif_rxerr_p;
         rx_buffer_next[v_i][12]  = phy_mpif_phyerr_p;
      end
      else 
         rx_buffer_next[v_i] = rx_buffer[v_i];

   // Read
   for (v_i=0;v_i<16;v_i=v_i+1)
      if (rx_buffer_rd)
         rx_buffer_next[v_i] = rx_buffer_next[v_i+1];
      else
         rx_buffer_next[v_i] = rx_buffer_next[v_i];
end

assign rx_buffer_rd  = |rx_buffer_wr_ptr[15:1];
assign rx_buffer_wr  = phy_rx_vector_done & (phy_mpif_phyrdy           |
                                             phy_mpif_rxendfortiming_p | phy_rx_vector_eft |
                                             phy_mpif_rxend_p          |
                                             phy_mpif_rxerr_p          |
                                             phy_mpif_phyerr_p         |
                                             rx_buffer_wr_forced       );

always @(posedge clk)
begin
   if (~phy_rx_vector_done | phy_mpif_rxend_p)
      rx_buffer_wr_forced <= 1'b0;
   else if (phy_mpif_rxendfortiming_p | phy_mpif_rxerr_p | phy_mpif_phyerr_p)
      rx_buffer_wr_forced <= 1'b1;
end


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef  RW_SIMU_ON
`endif//RW_SIMU_ON


//////////////////////////////////////////////////////////
// System Verilog Assertions
//////////////////////////////////////////////////////////
`ifdef  RW_ASSERT_ON
`endif//RW_ASSERT_ON

endmodule
                 
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
