#!/bin/bash
set -e
trap get_error ERR
function get_error
{
  echo -e '\033[1;37m'
  echo "##############################################################"
  echo "script has encountered an error, please review :"
  echo "  script  : ${BASH_SOURCE/echo/:}"
  echo "  line    : ${BASH_LINENO/echo/:}"
  echo "  command : ${BASH_COMMAND/echo/:}"
  echo "##############################################################"
  echo -e '\033[0m'
  exit 1
}
function error_handler_bypass
{
  # Empty function
  echo "" > /dev/null
}
################################################################################
#  Description:
#
#@
#@ FPGA V7 synthesis frontend script
#@
#@environment variables used:
#@    RW_SYNDIR         : suffix used for the working folder name
#@    PROC              : define the processor
#@    CONFIG            : define the configuration
#@    RW_FPGA_SVNREV    : SVN revision for the FPGA  (default is LOCAL)
#@    RW_MAC_SVNREV     : SVN revision for the MAC   (default is LOCAL)
#@    RW_MODEM_SVNREV   : SVN revision for the MODEM (default is LOCAL)
#@
#@option:
#@
#@    init              : create synthesis workarea work.$RW_SYNDIR
#@
#@    build_coregen     : build the Xilinx memories
#@    build_vivado      : build the Vivado PCIe subsystem
#@    build_sources     : fetch and concat sources for synthesis.
#@                          $RW_FPGA_SVNREV=<svn tag>   default is local
#@                          $RW_MAC_SVNREV=<svn tag>    default is HEAD revision
#@                          $RW_MODEM_SVNREV=<svn tag>  default is HEAD revision
#@
#@    all               : same than synthesis + par options
#@
#@    synthesis         : do a synplify_pro synthesis in workarea work.$RW_SYNDIR
#@    par               : do place, route & bitfile generation in work.$RW_SYNDIR
#@
#@    release           : release the sources files, bit files, etc to team
#@
#
################################################################################
################################################################################
# initialize workarea
################################################################################
function init_workarea
{
  echo "Create working area"
  echo "==================="
  echo ""

  # some checks before going ahead
  if [ "$RW_SYNDIR" == "" ] ;
  then

    echo "error: RW_SYNDIR variable not defined !"
    exit

  elif [ -e "work.$RW_SYNDIR" ] ;
  then

    echo "error: this workarea already exists !"
    exit

  fi

  # create work directory
  mkdir ./work.$RW_SYNDIR || exit
  cd    ./work.$RW_SYNDIR

  # create synthesis directory
  mkdir synthesis

  # create par directory
  mkdir par
  #KDEBUG: Must add .bmm * .mem file
  
  cd -
  
  echo ""
  echo "Working area creation Done!"
  echo ""
}

################################################################################
# Check configuration
################################################################################
function check_config
{
   if [ -z ${CONFIG} ]
   then
      echo "ERROR : CONFIG variable not defined "
      echo "Please set the CONFIG variable to one of these configuration : "
      #------------------------------------------------------------------------#
      #--- List the available configuration                                 ---#
      #------------------------------------------------------------------------#
      CONFIGLIST=`find ${SOURCESLIB}/env/CONF/ -type f -name "config_*.v" | sed 's:.*env/CONF/::' | sed 's/\.v//' | sort`
      for CONFIGNAME in ${CONFIGLIST}
      do
         echo " - $CONFIGNAME"
      done
      echo
      exit
   fi

   if [ -z ${PROC} ]
   then
      echo "ERROR : PROC variable not defined "
      echo "Please set the PROC variable to one of these processor : "
      #------------------------------------------------------------------------#
      #--- List the available processor                                     ---#
      #------------------------------------------------------------------------#
      echo " - TL4"
      echo " - APS3"
      echo " - M0"
      echo " - X1"
      echo " - RISCV"
      echo
      exit
   fi

   # Check Processor
   case ${PROC} in
   TL4)   proc="tl4";;
   APS3)  proc="aps3";;
   M0)    proc="m0";;
   X1)    proc="x1";;
   RISCV) proc="riscv";;
   *)     echo "Error: ${PROC} processor is not supported"
          echo "Please set the PROC variable to one of these processor : "
          echo " - TL4"
          echo " - APS3"
          echo " - M0"
          echo " - X1"
          echo " - RISCV"
          exit
          ;;
   esac

   RW_SYNDIR="$RW_SYNDIR.${PROC}.${CONFIG#config_}"

   echo "CONFIG    = ${CONFIG}"
   echo "PROC      = ${PROC}"
   echo "RW_SYNDIR = ${RW_SYNDIR}"
   echo ""
}

################################################################################
# export_file
################################################################################
function export_file
{
   FILE_NAME=$1
   FILE_REV=$2

   if [ "${FILE_REV}" == 0 ] ;
   then
      cat ${FILE_NAME}                                || exit 1
   else
      # Resolve ".." in file path
      while [[ ${FILE_NAME} == */../* ]]
      do
         FILE_NAME=`echo ${FILE_NAME} | sed 's:/[^/]\+/\.\./:/:'`
      done

      svn cat -r ${FILE_REV} ${FILE_NAME}@${FILE_REV} || exit 1
   fi
}
################################################################################
# extract_svn_path
################################################################################
function extract_svn_path
{
   PATH_NAME=$1
   PATH_REV=$2

   if [ "${PATH_REV}" == 0 ] ;
   then
      echo ${PATH_NAME}
   elif [ "${PATH_NAME}" == "${SOURCESLIB}" ] ;
   then
      # Fix for $SOURCESLIB which is not a SVN path, need to be hard coded
      echo "$PATH_REV,https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/"
   else
      svn info ${PATH_NAME} | grep URL | sed "s/^URL: */$PATH_REV,/"
   fi
}
################################################################################
# create_defines_file
################################################################################
function create_defines_file
{
  [ "$PROC" == "APS3"  ] && SIGNATURE="32'hC0CA0000"
  [ "$PROC" == "TL4"   ] && SIGNATURE="32'hC0CA1000"
  [ "$PROC" == "M0"    ] && SIGNATURE="32'hC0CA2000"
  [ "$PROC" == "X1"    ] && SIGNATURE="32'hC0CA3000"
  [ "$PROC" == "RISCV" ] && SIGNATURE="32'hC0CA4000"

  # generate defines
  # Note: RELEASE_DATE,FPGA_DATE and FPGA_TIME are updated whenver the logic synthesis is run
  echo "//0000_00_00_00_00"                                     > ./defines.v
  echo "\`define RW_FPGA_SIGNATURE $SIGNATURE"                 >> ./defines.v
  echo "\`define RW_FPGA_DATE 32'h00000000"                    >> ./defines.v
  echo "\`define RW_FPGA_TIME 32'h000000"                      >> ./defines.v

  echo "\`define PROC_${PROC}"                                 >> ./defines.v
  echo "\`define RW_MAC_AC"                                    >> ./defines.v
  echo "\`define RW_SHARED_RAM_ADDR_WIDTH 19"                  >> ./defines.v

  # integrates fpga configurations options
  echo "\`define RW_SWPROF_DEBUG_PORT"                         >> ./defines.v
  echo "\`define RW_EMBEDDED_LA"                               >> ./defines.v
  echo "\`define RW_EMBEDDED_LA128"                            >> ./defines.v

  if [ "$PROC" == "TL4" ]
  then
    # Force fpga configurations options for CEVATL410
    echo "\`define RW_TL4_FPGA_SYNTHESIS"                      >> ./defines.v
    echo "\`define TL4_PROG_RAM_256kB"                         >> ./defines.v
    echo "\`define TL4_DATA_RAM_128kB"                         >> ./defines.v
  fi
  if [ "$PROC" == "RISCV" ]
  then
    echo "\`define ZERORISCY"                                  >> ./defines.v
    # Force fpga configurations options for RISC-V
    echo "\`define PULP_FPGA_EMUL"                             >> ./defines.v
    # ZERORISCY: remove debug code (assertion, tracer, etc...)
    echo "\`define VERILATOR"                                  >> ./defines.v
  fi
  echo "\`define RW_FPGA_SVNREV $RW_FPGA_SVNREV"               >> ./defines.v
  echo "\`define RW_MAC_SVNREV $RW_MAC_SVNREV"                 >> ./defines.v
  echo "\`define FPGA_SVNREV_MODEM $RW_MODEM_SVNREV"           >> ./defines.v
  echo "\`define RW_FPGA_AHB"                                  >> ./defines.v
  echo "\`define RW_CRM_FPGA"                                  >> ./defines.v
  echo "\`define RW_MEM_FPGA"                                  >> ./defines.v
  echo "\`define FPGA_CONFIG \"${CONFIG#config_}\""            >> ./defines.v

  if [ !  -z ${NOMODEM} ] ; then
    echo "\`define RW_FPGA_NOMODEM"                            >> ./defines.v
  fi

  echo "\`define RW_NX_RF_MODE 2"                              >> ./defines.v
  echo "\`define RW_RF_CACTUS"                                 >> ./defines.v

  if [ !  -z ${DUMMY_SVD} ] ; then
    echo "\`define DUMMY_SVD"                                  >> ./defines.v
  fi

  if [[ ${CONFIG} == *DECX2* ]]
  then
    echo "Info: RW_VITERBIX2_FPGA defined"
    echo "Info: RW_NX_LDPC_LDECX2 defined"
  fi

  echo "/* Comment to decrease SHARED Memory to 256KB */" >> ./defines.v
  echo "\`define RW_SHARED_RAM_384K"                      >> ./defines.v

  if [[ ${PROC} == "APS3" ]]
  then
     echo "/* Uncomment to increase APS3 Memory to 512KB */"   >> ./defines.v
     echo "//\`define APS3_RAM_512K"                           >> ./defines.v
  fi

  if [[ ${PROC} == "RISCV" ]]
  then
     echo "/* Uncomment to increase RISC-V Memory to 1MB */"   >> ./defines.v
     echo "//\`define RISCV_RAM_1M"                            >> ./defines.v
  fi

  echo "exporting ${SOURCESLIB}/env/CONF/${CONFIG}.v@${RW_MODEM_SVNREV}"
  echo "/* build from svn by ${BUILD_USER} on ${BUILD_DATE} */" | tee -a ./defines.v
  echo "/*   MODEM revision: ${RW_MODEM_SVNREV}             */" | tee -a ./defines.v
  export_file ${SOURCESLIB}/env/CONF/${CONFIG}.v ${RW_MODEM_SVNREV}   >> ./defines.v

  # Limit the MAC & Platform frequency for the FPGA
  MACFREQ=`grep "MAC_FREQ "       ${SOURCESLIB}/env/CONF/${CONFIG}.v | sed 's/^.*MAC_FREQ //'`
  PLFFREQ=`grep "MACPI_CLK_FREQ " ${SOURCESLIB}/env/CONF/${CONFIG}.v | sed 's/^.*MACPI_CLK_FREQ //'`

  if [ ${MACFREQ} -ge 80 ] && [ "$PROC" == "TL4" ]
  then
    MACFREQ=80
    PLFFREQ=90
    sed -i "s/\(\`define MAC_FREQ\) \+[0-9]\+/\1 80/"             ./defines.v
    sed -i "s/\(\`define MACPI_CLK_FREQ\) \+[0-9]\+/\1 90/"       ./defines.v

  elif [ ${MACFREQ} -ge 80 ] && [ "$PROC" == "M0" ]
  then
    MACFREQ=80
    PLFFREQ=90
    sed -i "s/\(\`define MAC_FREQ\) \+[0-9]\+/\1 80/"             ./defines.v
    sed -i "s/\(\`define MACPI_CLK_FREQ\) \+[0-9]\+/\1 100/"      ./defines.v

  elif [ ${MACFREQ} -ge 80 ] && [ "$PROC" == "X1" ]
  then
    MACFREQ=80
    PLFFREQ=90
    sed -i "s/\(\`define MAC_FREQ\) \+[0-9]\+/\1 80/"             ./defines.v
    sed -i "s/\(\`define MACPI_CLK_FREQ\) \+[0-9]\+/\1 100/"      ./defines.v

  elif [ ${MACFREQ} -ge 80 ] && [ "$PROC" == "RISCV" ]
  then
    MACFREQ=80
    PLFFREQ=90
    sed -i "s/\(\`define MAC_FREQ\) \+[0-9]\+/\1 80/"             ./defines.v
    sed -i "s/\(\`define MACPI_CLK_FREQ\) \+[0-9]\+/\1 100/"      ./defines.v

  elif [ ${MACFREQ} -gt 80 ] && [ "$PROC" == "APS3" ]
  then
    MACFREQ=80
    PLFFREQ=100
    sed -i "s/\(\`define MAC_FREQ\) \+[0-9]\+/\1 80/"             ./defines.v
    sed -i "s/\(\`define MACPI_CLK_FREQ\) \+[0-9]\+/\1 100/"      ./defines.v
  fi

  echo "//"                                                    >> ./defines.v
  echo "// Generated by the fpga.sh script"                    >> ./defines.v
  echo "//"                                                    >> ./defines.v
  echo "\`define MAC_FREQ_${MACFREQ}MHZ"                       >> ./defines.v
  echo "\`define MACPI_CLK_FREQ_${PLFFREQ}MHZ"                 >> ./defines.v

  if (( ${MACFREQ} > 60 ))
  then
    echo "\`define SBOX_RDATA_CAPT"                            >> ./defines.v
    echo "\`define RW_TXTIME_DIVIDER_ONECYCLE"                 >> ./defines.v
  fi

  ## Remove WAPI on the 2x2_CBW80 Config
  #if [[ ${CONFIG} == *2x2_CBW80* ]]
  #then
  #  echo "Info: Remove RW_WAPI_EN define"
  #  sed -i -e 's:`define RW_WAPI_EN://`define RW_WAPI_EN:g' defines.v
  #fi

}


################################################################################
# build sources
################################################################################
function build_sources
{
  ################################################################################
  # capture HEAD revision at script call
  #
  # IMPORTANT: only one call to get the head revision, if you do several
  #            'svn info' you might get a different version (race condition)
  ################################################################################
  #BUILD_HEAD=`svn info https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/ | \
  #            egrep "Last Changed Rev" | \
  #            egrep -o  "[0-9]+"`
  BUILD_HEAD=0
  BUILD_USER=`id -un`
  BUILD_DATE=`date`


  echo "Launch build source"
  echo "==================="
  echo ""

  if [ -z ${SOURCESLIB} ] ;
  then
    echo "error: SOURCESLIB environment variable not defined"
    exit
  fi

  if [ -e "work.$RW_SYNDIR" ] ;
  then
    cd ./work.$RW_SYNDIR
  else
    echo "error: this workarea doesn't exists !"
    exit
  fi

  if [ -n "$RW_FPGA_SVNREV" ]
  then
    echo "SVN revision for FPGA  = ${RW_FPGA_SVNREV}"
  else
    RW_FPGA_SVNREV=0
    echo "No SVN revision for FPGA. Forced to ${RW_FPGA_SVNREV}"
  fi

  if [ -n "$RW_MAC_SVNREV" ]
  then
    echo "SVN revision for MAC  = ${RW_MAC_SVNREV}"
  else
    RW_MAC_SVNREV=$BUILD_HEAD
    echo "No SVN revision for MAC. Forced to ${RW_MAC_SVNREV}"
  fi

  if [ -n "$RW_MODEM_SVNREV" ]
  then
    echo "SVN revision for MODEM  = ${RW_MODEM_SVNREV}"
  else
    RW_MODEM_SVNREV=$BUILD_HEAD
    echo "No SVN revision for MODEM. Forced to $RW_MODEM_SVNREV"
  fi

  if [[ ${CONFIG} == *LDPC* ]]
  then
    LDPC_ENABLE=1
  fi
    LDPC_ENABLE=1

  create_defines_file

  # generate verilog files
  #   Possible Issue: Verilog file list are taken from working copy, should be taken from svn.
  #   Possible Issue: Verilog file list included are taken from working copy
#  LISTFILE=../../../verilog/rtl/rw_he_v7_${proc}_rtl.list
  LISTFILE=../../../verilog/rtl/rw_he_v7_rtl.list
  LISTFILE=`          extract_svn_path ${LISTFILE}           ${RW_FPGA_SVNREV}`

  MACSUBSYSDIR=`      extract_svn_path ${MACSUBSYSDIR}       ${RW_MAC_SVNREV}`   \
  MACCOREDIR=`        extract_svn_path ${MACCOREDIR}         ${RW_MAC_SVNREV}`   \
  PHYSUBSYSDIR=`      extract_svn_path ${PHYSUBSYSDIR}       ${RW_MODEM_SVNREV}` \
  HDMCOREDIR=`        extract_svn_path ${HDMCOREDIR}         ${RW_MODEM_SVNREV}` \
  RIUCOREDIR=`        extract_svn_path ${RIUCOREDIR}         ${RW_MODEM_SVNREV}` \
  MDMCOMMONDIR=`      extract_svn_path ${MDMCOMMONDIR}       ${RW_MODEM_SVNREV}` \
  MODEM80211BCOREDIR=`extract_svn_path ${MODEM80211BCOREDIR} ${RW_MODEM_SVNREV}` \
  TOP11AXDIR=`        extract_svn_path ${TOP11AXDIR}         ${RW_FPGA_SVNREV}`  \
  HWCOMMONDIR=`       extract_svn_path ${HWCOMMONDIR}        ${RW_FPGA_SVNREV}`  \
  CPUSUBSYSDIR=`      extract_svn_path ${CPUSUBSYSDIR}       ${RW_FPGA_SVNREV}`  \
  SOURCESLIB=`        extract_svn_path ${SOURCESLIB}         ${RW_FPGA_SVNREV}`  \
  APS3_ROOT=`         extract_svn_path ${APS3_ROOT}          ${RW_FPGA_SVNREV}`  \
  ZERORISCY_ROOT=`    extract_svn_path ${ZERORISCY_ROOT}     ${RW_FPGA_SVNREV}`  \
  TL4_ROOT=`          extract_svn_path ${TL4_ROOT}           ${RW_FPGA_SVNREV}`  \
  X1_ROOT=`           extract_svn_path ${X1_ROOT}            ${RW_FPGA_SVNREV}`  \
  fpga_walk_list.pl ${LISTFILE} > ./rw_he_v7_rtl.list

  # export verilog sources from svn
  echo "/* build from svn by ${BUILD_USER} on ${BUILD_DATE} */" | tee    rw_he_v7_syn.v
  echo "/*   MAC   revision: ${RW_MAC_SVNREV}               */" | tee -a rw_he_v7_syn.v
  echo "/*   MODEM revision: ${RW_MODEM_SVNREV}             */" | tee -a rw_he_v7_syn.v
  echo "/*   FPGA  revision: ${RW_FPGA_SVNREV}              */" | tee -a rw_he_v7_syn.v

  echo "/* build from svn by ${BUILD_USER} on ${BUILD_DATE} */" | tee    rw_he_v7_syn.sv
  echo "/*   FPGA  revision: ${RW_FPGA_SVNREV}              */" | tee -a rw_he_v7_syn.sv

  rm -f rw_he_v7_revision
  rm -rf .export_svn
  mkdir  .export_svn
  #for file in `cat ./rw_he_v7_rtl.list | grep -v "defines"`
  for file in `cat ./rw_he_v7_rtl.list`
  do
    # Extract file information
    if [[ ${file} == *,https://* ]] ;
    then
       # SVN Path with revision
       SVNREV=`echo ${file} | cut -d , -f 1`
       FILE=`  echo ${file} | cut -d , -f 2`
    elif [[ ${file} == ^https://* ]] ;
    then
       # SVN Path without revision
       echo "ERROR: svn path must always be provided with revision (${file})"
       exit 1
    else
       # Not SVN Path
       SVNREV=0
       FILE=${file}
    fi

    # destination file (replace special char)
    dst_file=${SVNREV}_${FILE//[,:\/@]/_}

    echo "exporting ${FILE}@${SVNREV}" | tee -a rw_he_v7_revision
    export_file ${FILE} ${SVNREV} > .export_svn/${dst_file} || exit &

    # Create exported file list
    echo ${dst_file} >> .export_svn/rw_he_v7_rtl.list
  done

  # Wait all export done
  wait

  for file in `cat .export_svn/rw_he_v7_rtl.list`
  do
    # Select destination concat file (verilog or system verilog)
    if [[ ${file} == *.sv ]]
    then
       dst_file=rw_he_v7_syn.sv
    else
       dst_file=rw_he_v7_syn.v
    fi

    cat .export_svn/${file} | grep -v "ambit" \
                            | grep -v "Synthesis Notes" \
                            | grep -v "pragma coverage" >> ${dst_file} || exit
  done

  ### Disable second viterbi
  ##sed -i -e 's:`define rw_he_DERIV_NES2://`define rw_he_DERIV_NES2:g' rw_he_v7_syn.v

  # Generate vhdl files (modem B)
  echo "-- build from svn by ${BUILD_USER} on ${BUILD_DATE} --" | tee    modem11b_syn.vhd
  echo "--   MODEM revision: ${RW_MODEM_SVNREV}             --" | tee -a modem11b_syn.vhd
  for file in `cat ../../../verilog/rtl/build_modem11b.list`
  do
    FILE=`eval "echo $file"`

    # destination file (replace special char)
    dst_file=${RW_MODEM_SVNREV}_${FILE//[,:\/@]/_}

    echo "exporting MODEMB $FILE@${RW_MODEM_SVNREV}" | tee -a rw_he_v7_revision
    export_file ${FILE} ${RW_MODEM_SVNREV} > .export_svn/${dst_file} || exit &

    # Create exported file list
    echo ${dst_file} >> .export_svn/build_modem11b.list
  done

  # Wait all export done
  wait

  for file in `cat .export_svn/build_modem11b.list`
  do
    cat .export_svn/$file >> ./modem11b_syn.vhd || exit
  done
  ##${SOURCESLIB}/SB/rw_he_fpga_b_karst/verilog/rtl/build_modem11b.list


  # copy parameters/includes files
  if [ "$PROC" == "APS3" ]
  then
     export_file ${CPUSUBSYSDIR}/aps3_subsys/verilog/rtl/aps_params.v               ${RW_FPGA_SVNREV} > ./synthesis/aps_params.v
     export_file ${APS3_ROOT}/vlog/modules/aps3/aps3_constants.v                    ${RW_FPGA_SVNREV} > ./synthesis/aps3_constants.v
     export_file ${APS3_ROOT}/vlog/modules/aps3/aps3_copro_constants.v              ${RW_FPGA_SVNREV} > ./synthesis/aps3_copro_constants.v
  fi
  if [ "$PROC" == "TL4" ]
  then
     export_file ${TL4_ROOT}/design/top/genparams.v                                 ${RW_FPGA_SVNREV} > ./synthesis/genparams.v
     export_file ${TL4_ROOT}/design/top/param_file.v                                ${RW_FPGA_SVNREV} > ./synthesis/param_file.v
     export_file ${NDBDIR}/synopsys/syn_vH-2013.03/dw/sim_ver/DW_lod_function.inc   ${RW_FPGA_SVNREV} > ./synthesis/DW_lod_function.inc
     export_file ${NDBDIR}/synopsys/syn_vH-2013.03/dw/sim_ver/DW_lsd_function.inc   ${RW_FPGA_SVNREV} > ./synthesis/DW_lsd_function.inc
     export_file ${NDBDIR}/synopsys/syn_vH-2013.03/dw/sim_ver/DW01_bsh_function.inc ${RW_FPGA_SVNREV} > ./synthesis/DW01_bsh_function.inc
  fi
  if [ "$PROC" == "X1" ]
  then
     export_file ${X1_ROOT}/design/top/cevax_pmem_param.v                           ${RW_FPGA_SVNREV} > ./synthesis/cevax_pmem_param.v
     export_file ${X1_ROOT}/design/top/cevax_dmem_param.v                           ${RW_FPGA_SVNREV} > ./synthesis/cevax_dmem_param.v
     export_file ${X1_ROOT}/design/top/conf1_param_file.v                           ${RW_FPGA_SVNREV} > ./synthesis/conf1_param_file.v
     export_file ${X1_ROOT}/design/top/cevax_gen_param.v                            ${RW_FPGA_SVNREV} > ./synthesis/cevax_gen_param.v
  fi
  if [ "$PROC" == "RISCV" ]                                                                 
  then                                                                                      
     #export_file ${RI5CY_ROOT}/include/riscv_config.sv                              ${RW_FPGA_SVNREV} > ./synthesis/riscv_config.sv
     export_file ${ZERORISCY_ROOT}/include/zeroriscy_config.sv                      ${RW_FPGA_SVNREV} > ./synthesis/zeroriscy_config.sv
  fi


  echo "/* build from svn by ${BUILD_USER} on ${BUILD_DATE} */"
  echo "/*   MODEM revision: ${RW_MODEM_SVNREV}             */"
  MDMDIR=${SOURCESLIB}/IPs/HW/Modem/Src

  for file in `cat ../../../verilog/rtl/build_include_ldpc.list`
  do
     file=`eval "echo $file"`
     if [[ ( "${LDPC_ENABLE}" == "1" || -e ${file} ) ]]
     then
       echo "exporting LDPC Include $file@${RW_MODEM_SVNREV}" | tee -a rw_he_v7_revision
       echo "/* build from svn by ${BUILD_USER} on ${BUILD_DATE} */" >  ./$(basename $file)
       echo "/*   MODEM revision: ${RW_MODEM_SVNREV}             */" >> ./$(basename $file)
       export_file $file ${RW_MODEM_SVNREV}                          >> ./$(basename $file)
     fi
  done
  #${SOURCESLIB}/SB/rw_he_v7/verilog/rtl/build_include_ldpc.list


  # export project and constraint files from svn
  case ${CONFIG} in
  *1x1_CBW20*) constraint_basename="rw_he_v7_${proc}_1x1_20MHz";;
  *1x1_CBW40*) constraint_basename="rw_he_v7_${proc}_1x1_40MHz";;
  *1x1_CBW80*) constraint_basename="rw_he_v7_${proc}_1x1_80MHz";;
  *2x2_CBW20*) constraint_basename="rw_he_v7_${proc}_2x2_20MHz";;
  *2x2_CBW40*) constraint_basename="rw_he_v7_${proc}_2x2_40MHz";;
  *2x2_CBW80*) constraint_basename="rw_he_v7_${proc}_2x2_80MHz";;
  *)           echo "Error: No constraint found for CONFIG $CONFIG"
               exit
               ;;
  esac

  export_file ../_template/rw_he_v7.prj               ${RW_FPGA_SVNREV} > ./synthesis/rw_he_v7.prj
  export_file ../_template/rw_he_v7.tcl               ${RW_FPGA_SVNREV} > ./rw_he_v7.tcl
  export_file ../_template/rw_he_v7_smart.tcl         ${RW_FPGA_SVNREV} > ./rw_he_v7_smart.tcl
  export_file ../_template/rw_he_v7_smart.sh          ${RW_FPGA_SVNREV} > ./rw_he_v7_smart.sh
  chmod +x ./rw_he_v7_smart.sh
  export_file ../_template/rw_he_v7_func.tcl          ${RW_FPGA_SVNREV} > ./rw_he_v7_func.tcl
  export_file ../_template/${constraint_basename}.fdc ${RW_FPGA_SVNREV} > ./synthesis/rw_he_v7.fdc
  export_file ../_template/${constraint_basename}.xdc ${RW_FPGA_SVNREV} > ./rw_he_v7.xdc
  export_file ../_template/rw_he_v7_common.xdc        ${RW_FPGA_SVNREV} > ./rw_he_v7_common.xdc
  export_file ../_vivado/RW_sub_system_blackbox.v     0                 > ./synthesis/RW_sub_system_blackbox.v

  echo "exporting agcram.bin@${RW_MODEM_SVNREV}" | tee -a rw_he_v7_revision
  export_file ${SOURCESLIB}/SB/RIU_KARST/AGCKarstBinary/agcram.bin ${RW_MODEM_SVNREV} > ./agcram.bin

  case ${CONFIG} in
  *1x1_CBW20_LDPC*)  ldpcram="config_FPGA_1x1_CBW20_LDPC_ldpcram.bin" ;;
  *1x1_CBW40_LDPC*)  ldpcram="config_FPGA_1x1_CBW40_LDPC_ldpcram.bin" ;;
  *)                 ldpcram="config_FPGA_1x1_CBW20_LDPC_ldpcram.bin" ;;
  esac
  if [[ ( "${LDPC_ENABLE}" == "1" || -e ${HDMCOREDIR}/OFDMACORE/OFDMRXCORE/OFDMRXBD/ldpcDec/RAM_SWGEN/out/${ldpcram} ) ]]
  then
     echo "exporting ldpcram.bin@${RW_MODEM_SVNREV}" | tee -a rw_he_v7_revision
     export_file ${HDMCOREDIR}/OFDMACORE/OFDMRXCORE/OFDMRXBD/ldpcDec/RAM_SWGEN/out/${ldpcram} ${RW_MODEM_SVNREV} > ./ldpcram.bin
  fi

  # Extract FSM state definition
  echo "Extracting FSM state definition (rw_he_v7_syn_fsm.xml)"
  if [[ -e $BINDIR/docgen/fsm_extractor.pl && -e $BINDIR/docgen/fsm2svmap.pl ]]
  then
      rm -rf rw_he_v7_syn_fsm.xml fsm_mapping.svcf
      $BINDIR/docgen/fsm_extractor.pl rw_he_v7_syn.v rw_he_v7_syn_fsm.xml
      echo "Generating simvision FSM state name mapping file (fsm_mapping.svcf)"
      $BINDIR/docgen/fsm2svmap.pl rw_he_v7_syn_fsm.xml fsm_mapping.svcf
  fi    

  cd synthesis
  ln -sf ../*.v .
  ln -sf ../*.sv .
  ln -sf ../*.vh .
  ln -sf ../*.vhd .
  cd ../..

  echo ""
  echo "Build sources Done!"
  echo ""
}

################################################################################
# build memories view with coregen
################################################################################
function build_coregen
{
  echo "Launch Coregen"
  echo "=============="
  echo

  cd ./_coregen || exit
  rm -f *.edn *.tcl *.vh* *.ngc

  ##############################################################################
  # generate all memories
  ##############################################################################
  list=`find . -name '*.xco'`
  for i in $list ;
  do

    # synthesize memories
    coregen -p coregen.cgp -b $i

  done

  ##############################################################################
  # remove all nasty pragmas that prevent memory wrapper optimizations
  ##############################################################################
  list=`find . -name '*.ngc'`
  for i in $list ;
  do
    netgen -w -ofmt verilog ${i} ${i%.*}_syn.v
    sed -i 's/\/\* synthesis syn_black_box syn_noprune=1 \*\///g' ${i%.*}_syn.v
    sed -i 's/\/\/ synthesis translate_off//g' ${i%.*}_syn.v
    sed -i 's/\/\/ synthesis translate_on//g' ${i%.*}_syn.v
  done

  echo ""
  echo "Coregen Done!"
  echo ""

  cd ..
}

################################################################################
# build PCIe subsystem with Vivado
################################################################################
function build_vivado
{
  echo "Launch Vivado"
  echo "============="
  echo ""

  vivado -mode batch -notrace -nojournal -source _template/RW_sub_system.tcl \
         -log _vivado/RW_sub_system.log

  echo ""
  echo "Vivado Done!"
  echo ""
}

################################################################################
# Synplify Synthesis
################################################################################
function synplify_synthesis
{
  echo "Launch Synthesis"
  echo "================"
  echo ""

  # patch defines.v with the date
  FU=$(date +"%Y %m %d %H %M %S")
  YE=$(echo $FU | cut -f 1 -d " ")
  MO=$(echo $FU | cut -f 2 -d " ")
  DA=$(echo $FU | cut -f 3 -d " ")
  HO=$(echo $FU | cut -f 4 -d " ")
  MI=$(echo $FU | cut -f 5 -d " ")
  SE=$(echo $FU | cut -f 6 -d " ")

  RELEASE_DATE="${YE}_${MO}_${DA}_${HO}_${MI}"
  FPGA_DATE="32'h${YE}${MO}${DA}"
  FPGA_TIME="32'h${HO}${MI}${SE}"
  sed -i -e "s://[0-9]\+_[0-9]\+_[0-9]\+_[0-9]\+_[0-9]\+://$RELEASE_DATE:g" \
         -e "s/FPGA_DATE.\+/FPGA_DATE $FPGA_DATE/g"    \
         -e "s/FPGA_TIME.\+/FPGA_TIME $FPGA_TIME/g" ./work.$RW_SYNDIR/defines.v || exit

  cd ./work.$RW_SYNDIR/synthesis || exit

  # clean-up any previous run
  rm -rf rev_1/rw_he_v7.edf

  # If the working directory is on the space folder, add the /net/$HOSTNAME to the path
  HOST=`echo $HOSTNAME | sed -e "s/\.frso.*//"`
  WORKAREA=`pwd | sed -e "s:^/space/:/net/$HOST/space/:"`

  if [ -z ${SYNPLIFY_LOCAL} ]
  then
     #ssh newsodium "
     ssh -t tortank "
        cd ${WORKAREA}
        pwd
        source /share/tools/etc/bashrc
        set_project 300
        export SOURCESLIB=${SOURCESLIB}
        echo SOURCESLIB is \$SOURCESLIB
        synplify_pro -license_wait 1 -shell rw_he_v7.prj &
        wait"                                             || error_handler_bypass
  else
    synplify_pro -license_wait 1 -shell rw_he_v7.prj      || error_handler_bypass
  fi

  if [ ! -e "./rev_1/rw_he_v7.edf" ] ;
  then
     echo ""
     echo "error: ./synthesis/rev_1/rw_he_v7.edf not found !"
     echo ""
     echo "Synthesis Fail!"
     echo ""
     exit
  fi

  cd ../..

  echo ""
  echo "Synthesis Done!"
  echo ""
}

################################################################################
# VIVADO PAR
################################################################################
function xilinx_vivado
{
   echo "Launch Vivado PAR"
   echo "================="
   cd ./work.$RW_SYNDIR/ || exit

   if [ ! -e "./synthesis/rev_1/rw_he_v7.edf" ] ;
   then
      echo "error:vivado: ../synthesis/rev_1/rw_he_v7.edf not found !"
      exit
   fi

   # Create Place & Route directory
   if [ ! -e "par" ] ;
   then
      mkdir par
   fi
   chmod g+w par

   # Remove old bit file, if exist
   \rm -f ./par/rw_he_v7.bit

   # If the working directory is on the space folder, add the /net/$HOSTNAME to the path
   HOST=`echo $HOSTNAME | sed -e "s/\.frso.*//"`
   WORKAREA=`pwd | sed -e "s:^/space/:/net/$HOST/space/:"`

   # Number of Vivado runs
   case ${CONFIG} in
   *1x1_CBW20*) njobs=4;;
   *1x1_CBW40*) njobs=4;;
   *1x1_CBW80*) njobs=4;;
   *2x2_CBW20*) njobs=4;;
   *2x2_CBW40*) njobs=4;;
   *2x2_CBW80*) njobs=8;;
   *)           njobs=2;; # Default
   esac

   if [[ -n ${NJOBS} ]]
   then
      njobs=${NJOBS}
   fi

   # Launch Vivado
   if [ -z ${VIVADO_LOCAL} ]
   then
      ssh -t rwhw-srv1 sudo -u nxjen "bash -c '
         cd ${WORKAREA}
         pwd
         TERM=dumb
         source /share/tools/etc/bashrc
         set_project 300
         export SOURCESLIB=${SOURCESLIB}
         echo SOURCESLIB is \$SOURCESLIB
         umask 002
         vivado -mode batch -source ./rw_he_v7_smart.tcl \
                -log ./par/rw_he_v7.log -notrace -nojournal \
                -tclargs ${njobs} `whoami` &
         wait'"                                          || error_handler_bypass
   else
      vivado -mode batch -source ./rw_he_v7_smart.tcl \
             -log ./par/rw_he_v7.log -notrace -nojournal \
             -tclargs 1                                  || error_handler_bypass
   fi

   cd ..
   echo ""
   echo "Vivado PAR Done !"
   echo ""
}

################################################################################
# Release
################################################################################
function release
{
  echo "Release binaries"
  echo "================"

  DATE=`grep "//" ./work.$RW_SYNDIR/defines.v | head -1 | sed 's://::'`
  RELEASEPATH="/net/rwlab-srv1/nx_share/fpga_release/v7/v7_${proc}/HE_$DATE"

  #RELEASEMACFREQ=`grep "MAC_FREQ " ./work.$RW_SYNDIR/defines.v | sed 's/^.*MAC_FREQ //'`
  #RELEASEPATH=${RELEASEPATH}_${RELEASEMACFREQ}MHz

  RELEASECONFIG=`grep "FPGA_CONFIG " ./work.$RW_SYNDIR/defines.v | sed 's/.*"\(.*\)".*/\1/'`
  RELEASEPATH=${RELEASEPATH}_${RELEASECONFIG}

  RELEASESVN=`grep "RW_MAC_SVNREV" ./work.$RW_SYNDIR/defines.v | sed "s/^.*RW_MAC_SVNREV //" | sed "s/32'h//"`
  if [ "$RELEASESVN" != 0 ]
  then
    RELEASEPATH=${RELEASEPATH}_MAC$RELEASESVN
  fi

  RELEASESVN=`grep "FPGA_SVNREV_MODEM" ./work.$RW_SYNDIR/defines.v | sed "s/^.*FPGA_SVNREV_MODEM //" | sed "s/32'h//"`
  if [ "$RELEASESVN" != 0 ]
  then
    RELEASEPATH=${RELEASEPATH}_MODEM$RELEASESVN
  fi

  echo
  echo "Release path is $RELEASEPATH"
  echo
  if [ ! -e "./work.$RW_SYNDIR/par/rw_he_v7.bit" ] ;
  then

    echo "error:release: rw_he_v7.bit not found !"
    exit

  fi

  if [ -e "$RELEASEPATH/rw_he_v7.bit" ] ;
  then

    echo "error:release: $RELEASEPATH/rw_he_v7.bit already exists !"
    exit

  fi
  
  mkdir -p $RELEASEPATH
  
  RW_SYNDIR_ARCHIVE="./work.${RW_SYNDIR}/archive"
  rm  -rf  ${RW_SYNDIR_ARCHIVE}
  mkdir -p ${RW_SYNDIR_ARCHIVE}
  
  cp ./work.$RW_SYNDIR/par/rw_he_v7.bit                           $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/*ram.bin                                   $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7.log                           $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/*.v                                        $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/*.sv                                       $RW_SYNDIR_ARCHIVE/.
#  cp ./work.$RW_SYNDIR/*.vh                                       $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/synthesis/rw_he_v7.prj                     $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/synthesis/rw_he_v7.fdc                     $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/synthesis/rev_1/rw_he_v7.srr               $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/synthesis/rev_1/rw_he_v7.edf               $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/rw_he_v7.xdc                               $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/rw_he_v7_common.xdc                        $RW_SYNDIR_ARCHIVE/.
# cp ./work.$RW_SYNDIR/par/*.rpt                                  $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7_clock_utilization.rpt         $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7_datasheet.rpt                 $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7_drc.rpt                       $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7_io.rpt                        $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7_timing_summary_*.rpt          $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7_utilization*.rpt              $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/par/rw_he_v7_postroute_physopted.dcp       $RW_SYNDIR_ARCHIVE/.
  cp ./work.$RW_SYNDIR/fsm_mapping.svcf                           $RW_SYNDIR_ARCHIVE/.


  cd $RW_SYNDIR_ARCHIVE

  #create synbolic link
  ln -sf rw_he_v7.bit  rw_nx_v7.bit  

  #compress data
  gzip *.edf *.rpt *.dcp *.srr
  cd -
  #copy archive
  cp -rf $RW_SYNDIR_ARCHIVE/* $RELEASEPATH/.
  # Clean archive
  rm -rf $RW_SYNDIR_ARCHIVE
  
  chmod -R 777 $RELEASEPATH

  echo ""
  echo "Release Done!"
  echo ""
}

################################################################################
# Main
################################################################################

case $1 in

  init)          check_config
                 init_workarea
                 ;;
  build_sources) check_config
                 build_sources
                 ;;
  build_coregen) build_coregen
                 ;;
  build_vivado)  build_vivado
                 ;;
  synthesis)     check_config
                 synplify_synthesis
                 ;;
  par)           check_config
                 xilinx_vivado
                 ;;
  all_norelease) check_config
                 synplify_synthesis
                 xilinx_vivado
                 ;;
  all)           check_config
                 synplify_synthesis
                 xilinx_vivado
                 release
                 ;;
  release)       check_config
                 release
                 ;;
  *)             cat $0 | grep "#\@" | sed -e 's/\#\@//g'
                 echo "error: unknown argument $1 !"
                 exit
                 ;;
esac

#RW_SYNDIR=20180509 PROC=APS3 CONFIG=config_STA_1x1_CBW20 RW_FPGA_SVNREV=33921 RW_MAC_SVNREV=33921 RW_MODEM_SVNREV=33921  ./fpga.sh build_sources
