//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jandre $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 18174 $
// $Date: 2015-03-02 13:09:12 +0100 (Mon, 02 Mar 2015) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Serial controller for ANALOG DEVICE AD9613 ADC
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module ad9613_ctrl
(
  ///////////////////////////////////////////////
  //$port_g Clock and reset
  ///////////////////////////////////////////////
  input  wire        rst_n,
  input  wire        clk,
  
  ///////////////////////////////////////////////
  //$port_g Dynamic Control Interface
  ///////////////////////////////////////////////
  input  wire        adc0_on,
  input  wire        adc1_on,
  
  ///////////////////////////////////////////////
  //$port_g Register Interface
  ///////////////////////////////////////////////
  input  wire        regb_adc_autopower_en,
  input  wire [ 4:0] regb_adc_spi_prescaler, 
  input  wire [ 1:0] regb_adc_spi_sel,
  input  wire        regb_adc_spi_start,
  input  wire        regb_adc_spi_we,
  input  wire [12:0] regb_adc_spi_addr,
  input  wire [ 7:0] regb_adc_spi_wdata,
  //
  output reg  [ 7:0] regb_adc_spi_rdata,
  output reg         regb_adc_spi_done,

  ///////////////////////////////////////////////
  //$port_g ADC Interface
  ///////////////////////////////////////////////
  output reg         adc_pwdown0,
  output reg         adc_pwdown1,
  
  output reg         adc_spi_csn0, 
  output reg         adc_spi_csn1, 
  output reg         adc_spi_clk,
  input  wire        adc_spi_miso,
  output reg         adc_spi_mosi,
  output reg         adc_spi_oe
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam    IDLE=2'd0,C0=2'd1,C1=2'd2,DONE=2'd3;
  reg [ 1:0]    state;
  reg           regb_adc_spi_start_1t,regb_adc_spi_start_2t;
  reg [ 4:0]    prescaler;
  reg [ 4:0]    count; 
  reg [22:0]    wdata;
  reg [ 7:0]    rdata;
  
  wire          prescaler_done; 
  wire [ 4:0]   n_prescaler;
  wire [ 4:0]   n_count; 

  /*****************************************************************************
  * control
  *****************************************************************************/
  assign n_count         = count     + 5'd1;
  assign n_prescaler     = prescaler + 5'd1;
  assign prescaler_done  = prescaler==regb_adc_spi_prescaler;
  
  always @(posedge clk, negedge rst_n)
    if(!rst_n)
    begin
      adc_spi_clk           <= 1'b0;    
      adc_spi_csn0          <= 1'b1;    
      adc_spi_csn1          <= 1'b1;    
      adc_spi_mosi           <= 1'b0;    
      adc_spi_oe            <= 1'b0;    
      
      adc_pwdown0           <= 1'b0;    
      adc_pwdown1           <= 1'b0;    

      regb_adc_spi_start_1t <= 1'b0;    
      regb_adc_spi_start_2t <= 1'b0;    
      regb_adc_spi_rdata    <= 8'b0;    
      regb_adc_spi_done     <= 1'b0;    
      
      rdata                 <= 8'd0;    
      wdata                 <= 23'd0;   
      count                 <= 5'd0;    
      prescaler             <= 5'd0;    
      state                 <= IDLE;    
    end
    else
    begin
      /* resynchro */
      regb_adc_spi_start_1t <= regb_adc_spi_start;
      regb_adc_spi_start_2t <= regb_adc_spi_start_1t;
      
      /* auto-power down */
      adc_pwdown0 <= ~adc0_on & regb_adc_autopower_en;
      adc_pwdown1 <= ~adc1_on & regb_adc_autopower_en;
      
      /* FSM */
      case(state)
        IDLE:
        begin
          if(regb_adc_spi_start_2t)
          begin
            adc_spi_csn0 <= ~regb_adc_spi_sel[0];
            adc_spi_csn1 <= ~regb_adc_spi_sel[1]; 
            adc_spi_mosi  <= ~regb_adc_spi_we;
            adc_spi_oe   <= 1'b1;
            count        <= 5'd0;
            prescaler    <= 5'd0;
            rdata        <= 8'd0;
            wdata        <= {2'b0,regb_adc_spi_addr,regb_adc_spi_wdata};
            state        <= C0;
          end
        end
        
        C0:
        begin
          if(prescaler_done)
          begin
            /* RISING EDGE, CAPTURE */
            adc_spi_clk   <= 1'b1;
            rdata         <= {rdata[6:0],adc_spi_miso};
            prescaler     <= 5'd0;
            state         <= C1;
          end
          else
            prescaler <= n_prescaler;
        end
        
        C1:
        begin
          if(prescaler_done)
          begin
            /* FALLING EDGE, CAPTURE */
            prescaler   <= 5'd0;
            count       <= n_count;
            adc_spi_clk <= 1'b0;
            adc_spi_mosi <= wdata[22];
            wdata       <= {wdata[21:0],1'b0};
            
            if(count==5'd15 && !regb_adc_spi_we)
              adc_spi_oe <= 1'b0;
              
            if(count==5'd23)
            begin
              /* DONE */
              if(!regb_adc_spi_we)
                regb_adc_spi_rdata <= rdata;
              adc_spi_mosi        <= 1'b0;
              adc_spi_oe         <= 1'b0;
              state              <= DONE;
            end
            else
              state              <= C0;
          end
          else
            prescaler <= n_prescaler;
        end
        
        default: /* DONE */
        begin
          if(prescaler_done)
          begin
            regb_adc_spi_done <= 1'b1;
            adc_spi_csn0      <= 1'b1;
            adc_spi_csn1      <= 1'b1;
            if(!regb_adc_spi_start_2t)
            begin
              regb_adc_spi_done <= 1'b0;
              state             <= IDLE;
            end
          end
          else
             prescaler <= n_prescaler;
        end
      endcase
    end
      
endmodule
`default_nettype wire
