//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jandre $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 16919 $
// $Date: 2014-11-20 18:57:29 +0100 (Thu, 20 Nov 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Serial controller for Analog Device AD9780 DAC
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module ad9780_ctrl
(
  ///////////////////////////////////////////////
  //$port_g Clock and reset
  ///////////////////////////////////////////////
  input  wire        rst_n,
  input  wire        clk,
  
  ///////////////////////////////////////////////
  //$port_g Dynamic Control Interface
  ///////////////////////////////////////////////
  input  wire        dac0_on,
  input  wire        dac1_on,
  
  ///////////////////////////////////////////////
  //$port_g Register Interface
  ///////////////////////////////////////////////
  input  wire        regb_dac_reset,
  input  wire        regb_dac_autopower_en,
  input  wire [12:0] regb_dac_autopower_powerup,
  input  wire [12:0] regb_dac_autopower_powerdn,
  input  wire [ 4:0] regb_dac_spi_prescaler, 
  input  wire [ 1:0] regb_dac_spi_sel,
  input  wire        regb_dac_spi_start,
  input  wire        regb_dac_spi_we,
  input  wire [ 4:0] regb_dac_spi_addr,
  input  wire [ 7:0] regb_dac_spi_wdata,
  // 
  output reg   [7:0] regb_dac_spi_rdata,
  output reg         regb_dac_spi_done,

  ///////////////////////////////////////////////
  //$port_g DAC Interface
  ///////////////////////////////////////////////
  output reg         dac_reset,
  
  output reg         dac_spi_csn0, 
  output reg         dac_spi_csn1, 
  output reg         dac_spi_clk,
  input  wire        dac_spi_miso,
  output reg         dac_spi_mosi,
  output reg         dac_spi_oe
);
  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam    IDLE=2'd0,C0=2'd1,C1=2'd2,DONE=2'd3;
  reg [ 1:0]    state;
  reg           dac_state;
  reg           dac0_on_1t,dac0_on_2t,dac0_on_3t,dac0_on_4t;
  reg           dac1_on_1t,dac1_on_2t,dac1_on_3t,dac1_on_4t;
  reg           regb_dac_spi_start_1t,regb_dac_spi_start_2t;
  reg [ 4:0]    prescaler;
  reg [ 4:0]    count; 
  reg [14:0]    wdata;
  reg [ 7:0]    rdata;
  
  wire          prescaler_done; 
  wire [ 4:0]   n_prescaler;
  wire [ 4:0]   n_count; 

  /*****************************************************************************
  * control
  *****************************************************************************/
  assign n_count         = count     + 5'd1;
  assign n_prescaler     = prescaler + 5'd1;
  assign prescaler_done  = prescaler==regb_dac_spi_prescaler;
  
  always @(posedge clk, negedge rst_n)
    if(!rst_n)
    begin
      dac_spi_clk           <= 1'b0;    
      dac_spi_csn0          <= 1'b1;    
      dac_spi_csn1          <= 1'b1;    
      dac_spi_mosi          <= 1'b0;    
      dac_spi_oe            <= 1'b0;    
      
      dac_reset             <= 1'b0; 
      dac_state             <= 1'b0;   

      dac0_on_1t            <= 1'b0;
      dac0_on_2t            <= 1'b0;
      dac0_on_3t            <= 1'b0;
      dac0_on_4t            <= 1'b0;
      dac1_on_1t            <= 1'b0;
      dac1_on_2t            <= 1'b0;
      dac1_on_3t            <= 1'b0;
      dac1_on_4t            <= 1'b0;
     
      regb_dac_spi_start_1t <= 1'b0;    
      regb_dac_spi_start_2t <= 1'b0;    
      regb_dac_spi_rdata    <= 8'b0;    
      regb_dac_spi_done     <= 1'b0;    
      
      rdata                 <= 8'd0;    
      wdata                 <= 8'd0;   
      count                 <= 5'd0;    
      prescaler             <= 5'd0;    
      state                 <= IDLE;    
    end
    else
    begin
      /* resynchro */
      regb_dac_spi_start_1t <= regb_dac_spi_start;
      regb_dac_spi_start_2t <= regb_dac_spi_start_1t;
      dac0_on_1t            <= dac0_on;
      dac0_on_2t            <= dac0_on_1t;
      dac0_on_3t            <= dac0_on_2t;
      dac0_on_4t            <= dac0_on_3t;
      dac1_on_1t            <= dac1_on;
      dac1_on_2t            <= dac1_on_1t;
      dac1_on_3t            <= dac1_on_2t;
      dac1_on_4t            <= dac1_on_3t;
      
      dac_reset             <= regb_dac_reset;
      dac_state             <= dac_state & ~regb_dac_reset & regb_dac_autopower_en;
      
      /* FSM */
      case(state)
        IDLE:
        begin
          if(regb_dac_spi_start_2t)
          begin
            /* SOFTWARE PROGRAMMATION */
            dac_spi_csn0 <= ~regb_dac_spi_sel[0];
            dac_spi_csn1 <= ~regb_dac_spi_sel[1]; 
            dac_spi_mosi  <= ~regb_dac_spi_we;
            dac_spi_oe   <= 1'b1;
            count        <= 5'd0;
            prescaler    <= 5'd0;
            rdata        <= 8'd0;
            wdata        <= {2'b0,regb_dac_spi_addr,regb_dac_spi_wdata};
            state        <= C0;
          end
          else if((dac0_on_3t|dac1_on_3t)!=dac_state && regb_dac_autopower_en)
          begin
            /* AUTOMATIC PROGRAMMATION */
            dac_spi_csn0 <= dac0_on_4t==dac0_on_3t;
            dac_spi_csn1 <= dac1_on_4t==dac1_on_3t; 
            dac_spi_mosi <= 1'b0;
            dac_spi_oe   <= 1'b1;
            count        <= 5'd0;
            prescaler    <= 5'd0;
            rdata        <= 8'd0;
            state        <= C0;
            
            if(!dac_state)
            begin
              /* POWER DOWN->UP */
              wdata        <= {2'b0,regb_dac_autopower_powerup};
              dac_state    <= 1'b1;
            end
            else
            begin
              /* POWER UP->DOWN */
              wdata        <= {2'b0,regb_dac_autopower_powerdn};
              dac_state    <= 1'b0;
            end
          end
        end
        
        C0:
        begin
          if(prescaler_done)
          begin
            /* RISING EDGE, CAPTURE */
            dac_spi_clk   <= 1'b1;
            rdata         <= {rdata[6:0],dac_spi_miso};
            prescaler     <= 5'd0;
            state         <= C1;
          end
          else
            prescaler <= n_prescaler;
        end
        
        C1:
        begin
          if(prescaler_done)
          begin
            /* FALLING EDGE, CAPTURE */
            prescaler   <= 5'd0;
            count       <= n_count;
            dac_spi_clk <= 1'b0;
            dac_spi_mosi <= wdata[14];
            wdata       <= {wdata[13:0],1'b0};
            
            if(count==5'd7 && !regb_dac_spi_we)
              dac_spi_oe <= 1'b0;
              
            if(count==5'd15)
            begin
              /* DONE */
              if(!regb_dac_spi_we)
                regb_dac_spi_rdata <= rdata;
              dac_spi_mosi        <= 1'b0;
              dac_spi_oe         <= 1'b0;
              state              <= DONE;
            end
            else
              state <= C0;
          end
          else
            prescaler <= n_prescaler;
        end
        
        default: /* DONE */
        begin
          if(prescaler_done)
          begin
            regb_dac_spi_done <= 1'b1;
            dac_spi_csn0      <= 1'b1;
            dac_spi_csn1      <= 1'b1;
            if(!regb_dac_spi_start_2t)
            begin
              regb_dac_spi_done <= 1'b0;
              state             <= IDLE;
            end
          end
          else
             prescaler <= n_prescaler;
        end
      endcase
    end
  
 endmodule
`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
