/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author:$
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     :                                       
* Description      : fe2phy_fifo
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module karst_frame_buffer
( 
  ///////////////////////////////////////////////
  //$port_g resets
  ///////////////////////////////////////////////
  input  wire                   ahb_rst_n,
  input  wire                   ahb_clk,
 
  input  wire                   fe_rst_n,
  input  wire                   fe_clk,
  
  ///////////////////////////////////////////////
  //$port_g  AHB interface
  ///////////////////////////////////////////////
  input  wire                   hready_in,
  input  wire                   hsel_ch0,
  input  wire                   hsel_ch1,
  input  wire [15:0]            haddr,
  input  wire [1:0]             htrans,
  input  wire                   hwrite,
  input  wire [31:0]            hwdata,
  output wire [31:0]            hrdata_ch0,
  output wire [ 1:0]            hresp_ch0,
  output wire                   hready_ch0,
  output wire [31:0]            hrdata_ch1,
  output wire [ 1:0]            hresp_ch1,
  output wire                   hready_ch1,
  
  ///////////////////////////////////////////////
  //$port_g frame buffer
  ///////////////////////////////////////////////
  input  wire [15:0]            regb_playback_count,
  output reg                    regb_capture_done,
 
  input  wire                   regb_ch0_playback_en,
  input  wire                   regb_ch0_capture_en,
  input  wire                   regb_ch1_playback_en,
  input  wire                   regb_ch1_capture_en,
  
  ///////////////////////////////////////////////
  //$port_g Phy Interface
  ///////////////////////////////////////////////
  input  wire                   phy_ch0_adc_on,
  output reg  [11:0]            phy_ch0_adc_i,
  output reg  [11:0]            phy_ch0_adc_q,
  
  input  wire                   phy_ch1_adc_on,
  output reg  [11:0]            phy_ch1_adc_i,
  output reg  [11:0]            phy_ch1_adc_q,
  
  input  wire                   phy_ch0_trsw,
  input  wire                   phy_ch0_pa2g,
  input  wire                   phy_ch0_pa5g,
  input  wire                   phy_ch0_dac_on,
  input  wire [11:0]            phy_ch0_dac_i,
  input  wire [11:0]            phy_ch0_dac_q,
 
  input  wire                   phy_ch1_trsw,
  input  wire                   phy_ch1_pa2g,
  input  wire                   phy_ch1_pa5g,
  input  wire                   phy_ch1_dac_on,
  input  wire [11:0]            phy_ch1_dac_i,
  input  wire [11:0]            phy_ch1_dac_q,
  
  ///////////////////////////////////////////////
  //$port_g Karst Interface
  ///////////////////////////////////////////////
  output reg                    karst_ch0_adc_on,
  input  wire [11:0]            karst_ch0_adc_i,
  input  wire [11:0]            karst_ch0_adc_q,
  
  output reg                    karst_ch1_adc_on,
  input  wire [11:0]            karst_ch1_adc_i,
  input  wire [11:0]            karst_ch1_adc_q,
  
  output reg                    karst_ch0_trsw,
  output reg                    karst_ch0_pa2g,
  output reg                    karst_ch0_pa5g,
  output reg                    karst_ch0_dac_on,
  output reg  [11:0]            karst_ch0_dac_i,
  output reg  [11:0]            karst_ch0_dac_q,
  
  output reg                    karst_ch1_trsw,
  output reg                    karst_ch1_pa2g,
  output reg                    karst_ch1_pa5g,  
  output reg                    karst_ch1_dac_on,
  output reg  [11:0]            karst_ch1_dac_i,
  output reg  [11:0]            karst_ch1_dac_q
);
  
  /*****************************************************************************
  *
  *****************************************************************************/
  always @(posedge fe_clk,negedge fe_rst_n)
  begin
    if(!fe_rst_n)
    begin
      phy_ch0_adc_i <= 12'b0;
      phy_ch0_adc_q <= 12'b0;
      phy_ch1_adc_i <= 12'b0;
      phy_ch1_adc_q <= 12'b0;   
    end
    else
    begin
      phy_ch0_adc_i <= karst_ch0_adc_i;
      phy_ch0_adc_q <= karst_ch0_adc_q;
      phy_ch1_adc_i <= karst_ch1_adc_i;
      phy_ch1_adc_q <= karst_ch1_adc_q;
    end
  end
  
  /*****************************************************************************
  *
  * AHB DOMAIN
  *
  *****************************************************************************/
  reg  [13:0] addr;
  reg  [ 1:0] sel;
  reg         write_pending;
 
  always @(posedge ahb_clk,negedge ahb_rst_n)
  begin
    if(!ahb_rst_n)
    begin
      write_pending <= 1'b0;
      addr          <= 14'b0;
      sel           <= 2'b0;
    end
    else
    begin
      if(write_pending)
        write_pending <= 1'b0;
      
      if(hready_in && (hsel_ch0 || hsel_ch1) && htrans[1])
      begin
        addr <= haddr[15:2];
        sel  <= {hsel_ch1,hsel_ch0};
        if(hwrite)
          write_pending <= 1'b1;
      end
    end
  end
  
  reg  [13:0] ahbmem_addr;
  reg         ahbmem_ch0_we;
  reg         ahbmem_ch1_we;
  
  always @(*)
  begin
    if(write_pending)            
    begin                                   
      ahbmem_addr   = addr;     
      ahbmem_ch0_we = sel[0];                
      ahbmem_ch1_we = sel[1];                
    end                                  
    else
    begin
      ahbmem_ch0_we = 1'b0;
      ahbmem_ch1_we = 1'b0;
      ahbmem_addr   = haddr[15:2]; 
    end
  end

  assign {hresp_ch0,hready_ch0} = {2'b0,1'b1};
  assign {hresp_ch1,hready_ch1} = {2'b0,1'b1};

  /*****************************************************************************
  *
  * FE DOMAIN
  *
  *****************************************************************************/
  /* playback and capture start-up on both paths shall occur at the same timestamp.
  *  as regb_xxxx_en comes from the aysncronous  register bank,
  *  we resynchronize these couple of bits as vectors */
  reg         playback_en_1t,playback_en_2t,playback_en_3t,playback_en_4t;
  reg         playback_en_5t,playback_en_6t;
  reg         ch0_playback_en;
  reg         ch1_playback_en;
  
  always @(posedge fe_clk,negedge fe_rst_n)
    if(!fe_rst_n)
    begin
      playback_en_1t  <= 1'b0; 
      playback_en_2t  <= 1'b0; 
      playback_en_3t  <= 1'b0; 
      playback_en_4t  <= 1'b0; 
      playback_en_5t  <= 1'b0; 
      playback_en_6t  <= 1'b0; 
      ch0_playback_en <= 1'b0; 
      ch1_playback_en <= 1'b0;
    end
    else
    begin
      /* playback enable rsync */
      playback_en_1t <= regb_ch0_playback_en | regb_ch1_playback_en;
      playback_en_2t <= playback_en_1t;
      /* let regb_ch0_playback_en and regb_ch1_playback_en stbilize
         before capturing them as a vector */
      playback_en_3t <= playback_en_2t;
      playback_en_4t <= playback_en_3t;
      playback_en_5t <= playback_en_4t;
      playback_en_6t <= playback_en_5t;
      if(playback_en_5t && !playback_en_6t)
      begin
        ch0_playback_en <= regb_ch0_playback_en;
        ch1_playback_en <= regb_ch1_playback_en;
      end
      else if(!playback_en_5t && playback_en_6t)
      begin
        ch0_playback_en <= 1'b0;
        ch1_playback_en <= 1'b0;
      end
    end
    
  reg         capture_en_1t,capture_en_2t,capture_en_3t,capture_en_4t;
  reg         capture_en_5t,capture_en_6t;
  reg         ch0_capture_en;
  reg         ch1_capture_en;
  
  always @(posedge fe_clk,negedge fe_rst_n)
    if(!fe_rst_n)
    begin
      capture_en_1t  <= 1'b0; 
      capture_en_2t  <= 1'b0; 
      capture_en_3t  <= 1'b0; 
      capture_en_4t  <= 1'b0; 
      capture_en_5t  <= 1'b0; 
      capture_en_6t  <= 1'b0; 
      ch0_capture_en <= 1'b0; 
      ch1_capture_en <= 1'b0;
    end
    else
    begin
      /* capture enable rsync */
      capture_en_1t <= regb_ch0_capture_en | regb_ch1_capture_en;
      capture_en_2t <= capture_en_1t;
      /* let regb_ch0_capture_en and regb_ch1_capture_en stbilize
         before capturing them as a vector */
      capture_en_3t <= capture_en_2t;
      capture_en_4t <= capture_en_3t;
      capture_en_5t <= capture_en_4t;
      capture_en_6t <= capture_en_5t;
      if(capture_en_5t && !capture_en_6t)
      begin
        ch0_capture_en <= regb_ch0_capture_en;
        ch1_capture_en <= regb_ch1_capture_en;
      end
      else if(!capture_en_5t && capture_en_6t)
      begin
        ch0_capture_en <= 1'b0;
        ch1_capture_en <= 1'b0;
      end
    end
    
  /* keep away memory and pads from memories */
  wire [31:0] fbmem_ch0_rdata,fbmem_ch1_rdata;
  
  reg  [11:0] karst_ch0_adc_i_1t,karst_ch0_adc_i_2t,karst_ch0_adc_i_3t;
  reg  [11:0] karst_ch0_adc_q_1t,karst_ch0_adc_q_2t,karst_ch0_adc_q_3t;
  reg  [11:0] karst_ch1_adc_i_1t,karst_ch1_adc_i_2t,karst_ch1_adc_i_3t;
  reg  [11:0] karst_ch1_adc_q_1t,karst_ch1_adc_q_2t,karst_ch1_adc_q_3t;
  reg         karst_ch0_trsw_m2t,karst_ch0_trsw_m1t;
  reg         karst_ch0_pa2g_m2t,karst_ch0_pa2g_m1t;
  reg         karst_ch0_pa5g_m2t,karst_ch0_pa5g_m1t;
  reg         karst_ch0_dac_on_m2t,karst_ch0_dac_on_m1t;
  reg  [11:0] karst_ch0_dac_i_m2t,karst_ch0_dac_i_m1t;
  reg  [11:0] karst_ch0_dac_q_m2t,karst_ch0_dac_q_m1t;
  reg         karst_ch1_trsw_m2t,karst_ch1_trsw_m1t;
  reg         karst_ch1_pa2g_m2t,karst_ch1_pa2g_m1t;
  reg         karst_ch1_pa5g_m2t,karst_ch1_pa5g_m1t;
  reg         karst_ch1_dac_on_m2t,karst_ch1_dac_on_m1t;
  reg  [11:0] karst_ch1_dac_i_m2t,karst_ch1_dac_i_m1t;
  reg  [11:0] karst_ch1_dac_q_m2t,karst_ch1_dac_q_m1t;
  
  always @(posedge fe_clk,negedge fe_rst_n)
  begin
    if(!fe_rst_n)
    begin
      karst_ch0_adc_on     <= 1'b0;
      karst_ch0_adc_i_1t   <= 12'b0;
      karst_ch0_adc_q_1t   <= 12'b0; 
      karst_ch0_adc_i_2t   <= 12'b0; 
      karst_ch0_adc_q_2t   <= 12'b0; 
      karst_ch0_adc_i_3t   <= 12'b0; 
      karst_ch0_adc_q_3t   <= 12'b0;    
      karst_ch1_adc_on     <= 1'b0; 
      karst_ch1_adc_i_1t   <= 12'b0; 
      karst_ch1_adc_q_1t   <= 12'b0; 
      karst_ch1_adc_i_2t   <= 12'b0; 
      karst_ch1_adc_q_2t   <= 12'b0; 
      karst_ch1_adc_i_3t   <= 12'b0; 
      karst_ch1_adc_q_3t   <= 12'b0;    

      karst_ch0_trsw_m2t   <= 1'b1;  
      karst_ch0_pa2g_m2t   <= 1'b0; 
      karst_ch0_pa5g_m2t   <= 1'b0;  
      karst_ch0_dac_on_m2t <= 1'b0;  
      karst_ch0_dac_q_m2t  <= 12'b0;  
      karst_ch0_dac_i_m2t  <= 12'b0;     
      karst_ch0_trsw_m1t   <= 1'b1; 
      karst_ch0_pa2g_m1t   <= 1'b0; 
      karst_ch0_pa5g_m1t   <= 1'b0; 
      karst_ch0_dac_on_m1t <= 1'b0; 
      karst_ch0_dac_i_m1t  <= 12'b0;   
      karst_ch0_dac_q_m1t  <= 12'b0;      
      karst_ch0_trsw       <= 1'b1; 
      karst_ch0_pa2g       <= 1'b0; 
      karst_ch0_pa5g       <= 1'b0; 
      karst_ch0_dac_on     <= 1'b0; 
      karst_ch0_dac_i      <= 12'b0;   
      karst_ch0_dac_q      <= 12'b0;   
      karst_ch1_trsw_m2t   <= 1'b1;  
      karst_ch1_pa2g_m2t   <= 1'b0; 
      karst_ch1_pa5g_m2t   <= 1'b0;  
      karst_ch1_dac_on_m2t <= 1'b0;  
      karst_ch1_dac_q_m2t  <= 12'b0;  
      karst_ch1_dac_i_m2t  <= 12'b0;     
      karst_ch1_trsw_m1t   <= 1'b1; 
      karst_ch1_pa2g_m1t   <= 1'b0; 
      karst_ch1_pa5g_m1t   <= 1'b0; 
      karst_ch1_dac_on_m1t <= 1'b0; 
      karst_ch1_dac_i_m1t  <= 12'b0;   
      karst_ch1_dac_q_m1t  <= 12'b0;      
      karst_ch1_trsw       <= 1'b1; 
      karst_ch1_pa2g       <= 1'b0; 
      karst_ch1_pa5g       <= 1'b0; 
      karst_ch1_dac_on     <= 1'b0; 
      karst_ch1_dac_i      <= 12'b0;   
      karst_ch1_dac_q      <= 12'b0;   
    end
    else
    begin
    
      /* ADC KARST->PHY pipeline */
      if(ch0_capture_en)
        karst_ch0_adc_on  <= 1'b1;
      else
        karst_ch0_adc_on  <= phy_ch0_adc_on;
      karst_ch0_adc_i_1t <= karst_ch0_adc_i;
      karst_ch0_adc_q_1t <= karst_ch0_adc_q;
      karst_ch0_adc_i_2t <= karst_ch0_adc_i_1t;
      karst_ch0_adc_q_2t <= karst_ch0_adc_q_1t;
      karst_ch0_adc_i_3t <= karst_ch0_adc_i_2t;
      karst_ch0_adc_q_3t <= karst_ch0_adc_q_2t;

      if(ch1_capture_en)
        karst_ch1_adc_on  <= 1'b1;
      else
        karst_ch1_adc_on  <= phy_ch1_adc_on;
      karst_ch1_adc_i_1t <= karst_ch1_adc_i;
      karst_ch1_adc_q_1t <= karst_ch1_adc_q;
      karst_ch1_adc_i_2t <= karst_ch1_adc_i_1t;
      karst_ch1_adc_q_2t <= karst_ch1_adc_q_1t;
      karst_ch1_adc_i_3t <= karst_ch1_adc_i_2t;
      karst_ch1_adc_q_3t <= karst_ch1_adc_q_2t;
      
      /* DAC PHY->KARST pipeline */
      if(ch0_playback_en)
      begin
        karst_ch0_trsw_m2t   <= fbmem_ch0_rdata[27];
        karst_ch0_pa2g_m2t   <= fbmem_ch0_rdata[26];
        karst_ch0_pa5g_m2t   <= fbmem_ch0_rdata[25]; 
        karst_ch0_dac_on_m2t <= fbmem_ch0_rdata[24];
        karst_ch0_dac_q_m2t  <= fbmem_ch0_rdata[23:12]; 
        karst_ch0_dac_i_m2t  <= fbmem_ch0_rdata[11: 0];
      end
      else
      begin
        karst_ch0_trsw_m2t   <= phy_ch0_trsw;
        karst_ch0_pa2g_m2t   <= phy_ch0_pa2g;
        karst_ch0_pa5g_m2t   <= phy_ch0_pa5g;
        karst_ch0_dac_on_m2t <= phy_ch0_dac_on;
        karst_ch0_dac_i_m2t  <= phy_ch0_dac_i;
        karst_ch0_dac_q_m2t  <= phy_ch0_dac_q;
      end
      karst_ch0_trsw_m1t   <= karst_ch0_trsw_m2t;
      karst_ch0_pa2g_m1t   <= karst_ch0_pa2g_m2t;   
      karst_ch0_pa5g_m1t   <= karst_ch0_pa5g_m2t;   
      karst_ch0_dac_on_m1t <= karst_ch0_dac_on_m2t;  
      karst_ch0_dac_i_m1t  <= karst_ch0_dac_i_m2t;   
      karst_ch0_dac_q_m1t  <= karst_ch0_dac_q_m2t;       
      karst_ch0_trsw       <= karst_ch0_trsw_m1t;
      karst_ch0_pa2g       <= karst_ch0_pa2g_m1t;   
      karst_ch0_pa5g       <= karst_ch0_pa5g_m1t;   
      karst_ch0_dac_on     <= karst_ch0_dac_on_m1t;  
      karst_ch0_dac_i      <= karst_ch0_dac_i_m1t;   
      karst_ch0_dac_q      <= karst_ch0_dac_q_m1t;  
      
      if(ch1_playback_en)
      begin
        karst_ch1_trsw_m2t   <= fbmem_ch1_rdata[27];
        karst_ch1_pa2g_m2t   <= fbmem_ch1_rdata[26];
        karst_ch1_pa5g_m2t   <= fbmem_ch1_rdata[25]; 
        karst_ch1_dac_on_m2t <= fbmem_ch1_rdata[24];
        karst_ch1_dac_q_m2t  <= fbmem_ch1_rdata[23:12];
        karst_ch1_dac_i_m2t  <= fbmem_ch1_rdata[11: 0];
      end
      else
      begin
        karst_ch1_trsw_m2t   <= phy_ch1_trsw;
        karst_ch1_pa2g_m2t   <= phy_ch1_pa2g;
        karst_ch1_pa5g_m2t   <= phy_ch1_pa5g;
        karst_ch1_dac_on_m2t <= phy_ch1_dac_on;
        karst_ch1_dac_i_m2t  <= phy_ch1_dac_i;
        karst_ch1_dac_q_m2t  <= phy_ch1_dac_q;
      end
      karst_ch1_trsw_m1t   <= karst_ch1_trsw_m2t;
      karst_ch1_pa2g_m1t   <= karst_ch1_pa2g_m2t;   
      karst_ch1_pa5g_m1t   <= karst_ch1_pa5g_m2t;   
      karst_ch1_dac_on_m1t <= karst_ch1_dac_on_m2t;  
      karst_ch1_dac_i_m1t  <= karst_ch1_dac_i_m2t;   
      karst_ch1_dac_q_m1t  <= karst_ch1_dac_q_m2t;       
      karst_ch1_trsw       <= karst_ch1_trsw_m1t;
      karst_ch1_pa2g       <= karst_ch1_pa2g_m1t;   
      karst_ch1_pa5g       <= karst_ch1_pa5g_m1t;   
      karst_ch1_dac_on     <= karst_ch1_dac_on_m1t;  
      karst_ch1_dac_i      <= karst_ch1_dac_i_m1t;   
      karst_ch1_dac_q      <= karst_ch1_dac_q_m1t;
    end  
  end

  /*****************************************************************************
  * Modem clock side / frame buffer managment
  *****************************************************************************/
  reg [13:0] fbmem_addr;
  reg        fbmem_ch0_we;
  reg        fbmem_ch1_we;
  
  localparam IDLE=2'd0,PLAYBACK=2'd1,CAPTURE=2'd2;
  reg [ 1:0] state;
  
  wire [13:0] n_fbmem_addr;
  assign n_fbmem_addr = fbmem_addr + 14'd1;
  
  always @ (posedge fe_clk, negedge fe_rst_n)
  begin
    if (!fe_rst_n)
    begin
      fbmem_addr        <= 14'b0;
      fbmem_ch0_we      <= 1'b0;
      fbmem_ch1_we      <= 1'b0;
      regb_capture_done <= 1'b0;
      state             <= IDLE;
    end
    else
    begin
      case(state)
        /* wait for capture or playback command */
        IDLE:
        begin
          fbmem_addr <= 14'b0;
          if(ch0_playback_en||ch1_playback_en)
            state <= PLAYBACK;
          else if(ch0_capture_en||ch1_capture_en)
          begin
            fbmem_ch0_we <= ch0_capture_en;
            fbmem_ch1_we <= ch1_capture_en;
            state        <= CAPTURE;
          end
        end
        
        /* playback samples to DAC */
        PLAYBACK:
        begin
          if(!ch0_playback_en && !ch1_playback_en)
            state <= IDLE;
          else
          begin
            if(n_fbmem_addr==regb_playback_count[13:0])
              fbmem_addr <= 14'b0;
            else
              fbmem_addr <= n_fbmem_addr;
          end
        end
        
        default: /* CAPTURE */ 
        begin
          if(!ch0_capture_en && !ch1_capture_en)
          begin
            fbmem_ch0_we      <= 1'b0;
            fbmem_ch1_we      <= 1'b0;
            regb_capture_done <= 1'b0;
            state             <= IDLE;
          end
          else if(fbmem_addr==14'h3fff)
          begin
            fbmem_ch0_we      <= 1'b0;
            fbmem_ch1_we      <= 1'b0;
            regb_capture_done <= 1'b1;
          end
          else
            fbmem_addr <= n_fbmem_addr;
        end
      endcase
    end
  end

  /*****************************************************************************
  *
  * true dual port memories
  *
  *****************************************************************************/
  sram_tdp_16kx32 u_sram_ch0                        
  (                                             
    .clka(                   ahb_clk),          
    .ena(                    1'b1),      
    .wea(                    ahbmem_ch0_we),      
    .addra(                  ahbmem_addr),    
    .dina(                   hwdata),     
    .douta(                  hrdata_ch0),    
   
    .clkb(                   fe_clk),         
    .enb(                    1'b1),      
    .web(                    fbmem_ch0_we),      
    .addrb(                  fbmem_addr),    
    .dinb(                   {8'b0,karst_ch0_adc_q_3t,karst_ch0_adc_i_3t}),     
    .doutb(                  fbmem_ch0_rdata)     
  );                                            
 
  sram_tdp_16kx32 u_sram_ch1                        
  (                                             
    .clka(                   ahb_clk),          
    .ena(                    1'b1),      
    .wea(                    ahbmem_ch1_we),      
    .addra(                  ahbmem_addr),    
    .dina(                   hwdata),     
    .douta(                  hrdata_ch1),    
   
    .clkb(                   fe_clk),         
    .enb(                    1'b1),      
    .web(                    fbmem_ch1_we),      
    .addrb(                  fbmem_addr),    
    .dinb(                   {8'b0,karst_ch1_adc_q_3t,karst_ch1_adc_i_3t}),     
    .doutb(                  fbmem_ch1_rdata)     
  );  
                                            
endmodule
`default_nettype wire
