//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : RW HE FPGA registers
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
// 
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_he_v7_reg (
    ////////////////////////////////////////////
    //$port_g Clock and reset
    ////////////////////////////////////////////
    input wire            ahb_nrst,  // AHB Hard Reset
    input wire            ahb_clk,   // AHB clock

    ////////////////////////////////////////////
    // Registers
    ////////////////////////////////////////////
    //$port_g SIGNATURE register.
    input wire [31 : 0]   fpga_signature               , // The SIGNATURE register is a read only register containing a specific ID. 
    //$port_g DATE register.
    input wire [31 : 0]   fpga_date                    , // The DATE register is a read only register containing the date of the FPGA bitmap creation. The date format is : YYYYMMDD. 
    //$port_g FB_CNTL register.
    input wire            fb_capture_done              , // Indicate that the capture is completed
    //$port_g ADC_SPI_CNTL register.
    input wire            adc_spi_done                 , // Indicate the completion of the SPI access to ADC
    //$port_g ADC_SPI_DATA register.
    input wire [15 : 0]   adc_spi_rdata                , // Read Data
    //$port_g DAC_SPI_CNTL register.
    input wire            dac_spi_done                 , // Indicate the completion of the SPI access to DAC
    //$port_g DAC_SPI_DATA register.
    input wire [15 : 0]   dac_spi_rdata                , // Read Data
    //$port_g DIAG0_STAT register.
    input wire [31 : 0]   diag0_stat                   , // This register returns DiagPort0 current value
    //$port_g DIAG1_STAT register.
    input wire [31 : 0]   diag1_stat                   , // This register returns DiagPort1 current value
    //$port_g DIAG2_STAT register.
    input wire [31 : 0]   diag2_stat                   , // This register returns DiagPort2 current value
    //$port_g DIAG3_STAT register.
    input wire [31 : 0]   diag3_stat                   , // This register returns DiagPort3 current value
    //$port_g DIAG4_STAT register.
    input wire [31 : 0]   diag4_stat                   , // This register returns DiagPort4 current value
    //$port_g INTERRUPT_SET register.
    input wire            doorbell_int_set_in          , // Doorbell interrupt In
    input wire            doorbell_int_set_in_valid    , // Doorbell interrupt In Valid
    //$port_g INTERRUPT_CLEAR register.
    input wire            doorbell_int_clear_in        , // Doorbell interrupt In
    input wire            doorbell_int_clear_in_valid  , // Doorbell interrupt In Valid
    //$port_g INTERRUPT_STATUS register.
    input wire            doorbell_int_status          , // Doorbell interrupt
    //$port_g V7_GPIO register.
    input wire            v7_gpio3                     , // Control V7 GPIO3 (always input).
    input wire            v7_gpio2                     , // Control V7 GPIO2 (always input).
    input wire            v7_gpio1                     , // Control V7 GPIO1 (always input).
    input wire            v7_gpio0                     , // Control V7 GPIO0 (always input).
    //$port_g IODELAY_CNTL register.
    input wire            iodelay_done                 , // Once the IO delay procedure is completed, a 1 is written is this field 
    //$port_g RF_CNTRL register.
    input wire            rf_spi_done                  , // Indicate the completion of the SPI access to RF
    //$port_g RF_SPI_DATA register.
    input wire [15 : 0]   rf_spi_rdata                 , // Read Data
    //$port_g RF_IRQ register.
    input wire [7 : 0]    rf_irq                       , // This register provides status of RF Interrupt
    //$port_g SVNREV_MODEM register.
    input wire [31 : 0]   svnrev_modem                 , // This register provides the SVN revision of the Modem IP
    //$port_g SVNREV_FPGAB register.
    input wire [31 : 0]   svnrev_fpgab                 , // This register provides the SVN revision of the FPGA
    //
    //$port_g DIAGPORT_CONF1 register.
    output wire [7 : 0]    mictor2_sel                 , // Select the diag port connected to Mictor2 
    output wire [7 : 0]    mictor1_sel                 , // Select the diag port connected to Mictor1 
    //$port_g DIAGPORT_CONF2 register.
    output wire [15 : 0]   mictor0_sel                 , // Select the diag port connected to Mictor0
    //$port_g FB_CNTL register.
    output wire [1 : 0]    fb_capture_en               , // Enable the capture on the Channel [1:0]
    output wire [1 : 0]    fb_playback_en              , // Enable the playback on the Channel [1:0]
    output wire [15 : 0]   fb_playback_count           , // Configure the length of the playback pattern
    //$port_g FB_PATHMUX register.
    output wire [3 : 0]    fb_pathmux_sel              , // Configure the multiplexing of the ADC/DAC path with the frame buffer
    //$port_g ADC_SPI_CNTL register.
    output wire [4 : 0]    adc_spi_prescaler           , // Configure the SPI Clock pre-scaler
    output wire            adc_spi_autopower           , // Enable ADC Auto power mode
    output wire            adc_spi_start               , // Start an SPI access to ADC
    //$port_g ADC_SPI_ADDR register.
    output wire            adc_spi_we                  , // Configure R/W mode
    output wire [3 : 0]    adc_spi_sel                 , // Select the ADC
    output wire [15 : 0]   adc_spi_addr                , // This register controls  the ADC SPI Interface
    //$port_g ADC_SPI_DATA register.
    output wire [15 : 0]   adc_spi_wdata               , // Write Data
    //$port_g DAC_SPI_CNTL register.
    output wire [4 : 0]    dac_spi_prescaler           , // Configure the SPI Clock pre-scaler
    output wire            dac_reset                   , // Reset the DAC
    output wire            dac_spi_autopower           , // Enable DAC Auto power mode
    output wire            dac_spi_start               , // Start an SPI access to DAC
    //$port_g DAC_SPI_ADDR register.
    output wire            dac_spi_we                  , // Configure R/W mode
    output wire [3 : 0]    dac_spi_sel                 , // Select the DAC
    output wire [15 : 0]   dac_spi_addr                , // This register controls  the DAC SPI Interface
    //$port_g DAC_SPI_DATA register.
    output wire [15 : 0]   dac_spi_wdata               , // Write Data
    //$port_g DAC_SPI_AUTOPOWER register.
    output wire [15 : 0]   dac_powerup                 , // Configure DAC power UP
    output wire [15 : 0]   dac_powerdown               , // Configure DAC power down
    //$port_g INTERRUPT_SET register.
    output wire            doorbell_int_set            , // Doorbell interrupt
    //$port_g INTERRUPT_CLEAR register.
    output wire            doorbell_int_clear          , // Doorbell interrupt
    //$port_g INTERRUPT_UNMASK register.
    output wire            doorbell_int_mask           , // Doorbell interrupt
    //$port_g V7_GPIO register.
    output wire            v7_gpio_en                  , // Enable usage of the V7 GPIOs.
    output wire            v7_gpio6                    , // Control V7 GPIO6 (always output).
    output wire            v7_gpio5                    , // Control V7 GPIO5 (always output).
    //$port_g IODELAY_CNTL register.
    output wire            iodelay_start               , // Writing 1 in this field launch the IO delay procedure
    //$port_g IODELAY_ADC0 register.
    output wire [7 : 0]    iodelay_adc0                , // This register contains the IO delay for ADC0
    //$port_g IODELAY_DAC0 register.
    output wire [7 : 0]    iodelay_dac0                , // This register contains the IO delay for DAC0
    //$port_g IODELAY_ADC1 register.
    output wire [7 : 0]    iodelay_adc1                , // This register contains the IO delay for ADC1
    //$port_g IODELAY_DAC1 register.
    output wire [7 : 0]    iodelay_dac1                , // This register contains the IO delay for DAC1
    //$port_g IQCOMP_ADC0 register.
    output wire [9 : 0]    gcos_adc0                   , // IQ GCOS compensation on ADC0 output samples
    output wire [9 : 0]    gsin_adc0                   , // IQ GSIN compensation on ADC0 output samples
    //$port_g IQCOMP_DAC0 register.
    output wire [9 : 0]    gcos_dac0                   , // IQ GCOS compensation on DAC0 output samples
    output wire [9 : 0]    gsin_dac0                   , // IQ GSIN compensation on DAC0 output samples
    //$port_g IQCOMP_ADC1 register.
    output wire [9 : 0]    gcos_adc1                   , // IQ GCOS compensation on ADC1 output samples
    output wire [9 : 0]    gsin_adc1                   , // IQ GSIN compensation on ADC1 output samples
    //$port_g IQCOMP_DAC1 register.
    output wire [9 : 0]    gcos_dac1                   , // IQ GCOS compensation on DAC1 output samples
    output wire [9 : 0]    gsin_dac1                   , // IQ GSIN compensation on DAC1 output samples
    //$port_g RF_CNTRL register.
    output wire [3 : 0]    rf_spi_prescaler            , // Configure the SPI Clock pre-scaler
    output wire            rf_spi_start                , // Start an SPI access to RF
    //$port_g RF_SPI_ADDR register.
    output wire            rf_spi_we                   , // Configure R/W mode
    output wire [3 : 0]    rf_spi_sel                  , // Select the RF
    output wire [15 : 0]   rf_spi_addr                 , // This register controls  the RF SPI Interface
    //$port_g RF_SPI_DATA register.
    output wire [15 : 0]   rf_spi_wdata                , // Write Data
    //$port_g RF_FORCE register.
    output wire [15 : 0]   rf_force_en                 , // When set, indicate that the corresponding RF interface output is forced to the RF_FORCE value.
    output wire [15 : 0]   rf_force                    , // This register allows forcing RF interface output.
    //$port_g RF_GPIO register.
    output wire [7 : 0]    rf_gpio                     , // This register control RF GPIO interface

    ////////////////////////////////////////////
    //$port_g Bus interface
    ////////////////////////////////////////////
    input  wire            hready_in,
    input  wire            hsel,
    input  wire [10:0]     haddr,
    input  wire [ 1:0]     htrans,
    input  wire            hwrite,
    input  wire [31:0]     hwdata,
    output wire [31:0]     hrdata,
    output wire [ 1:0]     hresp,
    output wire            hready
    );
  
////////////////////////////////////////////////////////////////////////////////
// Port Declaration 
////////////////////////////////////////////////////////////////////////////////

  //////////////////////////////////////////////////////////////////////////////
  // Constants for registers addresses
  //////////////////////////////////////////////////////////////////////////////
  // Register configuration
  

  // Constants for register addresses.
  localparam RW_HE_V7_SIGNATURE_ADDR_CT        = 9'b000000000;
  localparam RW_HE_V7_DATE_ADDR_CT             = 9'b000000001;
  localparam RW_HE_V7_DIAGPORT_CONF1_ADDR_CT   = 9'b000000100;
  localparam RW_HE_V7_DIAGPORT_CONF2_ADDR_CT   = 9'b000000101;
  localparam RW_HE_V7_FB_CNTL_ADDR_CT          = 9'b000001000;
  localparam RW_HE_V7_FB_PATHMUX_ADDR_CT       = 9'b000001001;
  localparam RW_HE_V7_ADC_SPI_CNTL_ADDR_CT     = 9'b001000000;
  localparam RW_HE_V7_ADC_SPI_ADDR_ADDR_CT     = 9'b001000001;
  localparam RW_HE_V7_ADC_SPI_DATA_ADDR_CT     = 9'b001000010;
  localparam RW_HE_V7_DAC_SPI_CNTL_ADDR_CT     = 9'b010000000;
  localparam RW_HE_V7_DAC_SPI_ADDR_ADDR_CT     = 9'b010000001;
  localparam RW_HE_V7_DAC_SPI_DATA_ADDR_CT     = 9'b010000010;
  localparam RW_HE_V7_DAC_SPI_AUTOPOWER_ADDR_CT = 9'b010000011;
  localparam RW_HE_V7_DIAG0_STAT_ADDR_CT       = 9'b000010011;
  localparam RW_HE_V7_DIAG1_STAT_ADDR_CT       = 9'b000010100;
  localparam RW_HE_V7_DIAG2_STAT_ADDR_CT       = 9'b000010101;
  localparam RW_HE_V7_DIAG3_STAT_ADDR_CT       = 9'b000010110;
  localparam RW_HE_V7_DIAG4_STAT_ADDR_CT       = 9'b000010111;
  localparam RW_HE_V7_INTERRUPT_SET_ADDR_CT    = 9'b011100000;
  localparam RW_HE_V7_INTERRUPT_CLEAR_ADDR_CT  = 9'b011100001;
  localparam RW_HE_V7_INTERRUPT_UNMASK_ADDR_CT = 9'b011100010;
  localparam RW_HE_V7_INTERRUPT_STATUS_ADDR_CT = 9'b011100011;
  localparam RW_HE_V7_V7_GPIO_ADDR_CT          = 9'b011100100;
  localparam RW_HE_V7_IODELAY_CNTL_ADDR_CT     = 9'b100000000;
  localparam RW_HE_V7_IODELAY_ADC0_ADDR_CT     = 9'b100000001;
  localparam RW_HE_V7_IODELAY_DAC0_ADDR_CT     = 9'b100000010;
  localparam RW_HE_V7_IODELAY_ADC1_ADDR_CT     = 9'b100000011;
  localparam RW_HE_V7_IODELAY_DAC1_ADDR_CT     = 9'b100000100;
  localparam RW_HE_V7_IQCOMP_ADC0_ADDR_CT      = 9'b100100000;
  localparam RW_HE_V7_IQCOMP_DAC0_ADDR_CT      = 9'b100100001;
  localparam RW_HE_V7_IQCOMP_ADC1_ADDR_CT      = 9'b100100010;
  localparam RW_HE_V7_IQCOMP_DAC1_ADDR_CT      = 9'b100100011;
  localparam RW_HE_V7_RF_CNTRL_ADDR_CT         = 9'b101000000;
  localparam RW_HE_V7_RF_SPI_ADDR_ADDR_CT      = 9'b101000001;
  localparam RW_HE_V7_RF_SPI_DATA_ADDR_CT      = 9'b101000010;
  localparam RW_HE_V7_RF_FORCE_ADDR_CT         = 9'b101000011;
  localparam RW_HE_V7_RF_GPIO_ADDR_CT          = 9'b101000100;
  localparam RW_HE_V7_RF_IRQ_ADDR_CT           = 9'b101000101;
  localparam RW_HE_V7_SVNREV_MODEM_ADDR_CT     = 9'b000011001;
  localparam RW_HE_V7_SVNREV_FPGAB_ADDR_CT     = 9'b000011010;
 
  //////////////////////////////////////////////////////////////////////////////
  // Signals
  //////////////////////////////////////////////////////////////////////////////
  reg  [31: 0] int_reg_dr;
  wire [31: 0] int_reg_dw;
  reg          int_reg_rdy;
  reg          pending_write;
  reg          pending_read;
  reg   [8:0]  pending_addr;

  // DIAGPORT_CONF1 register.
  reg [7 : 0] int_mictor2_sel         ;
  reg [7 : 0] int_mictor1_sel         ;
  // DIAGPORT_CONF2 register.
  reg [15 : 0] int_mictor0_sel         ;
  // FB_CNTL register.
  reg [1 : 0] int_fb_capture_en       ;
  reg [1 : 0] int_fb_playback_en      ;
  reg [15 : 0] int_fb_playback_count   ;
  // FB_PATHMUX register.
  reg [3 : 0] int_fb_pathmux_sel      ;
  // ADC_SPI_CNTL register.
  reg [4 : 0] int_adc_spi_prescaler   ;
  reg         int_adc_spi_autopower   ;
  reg         int_adc_spi_start       ;
  // ADC_SPI_ADDR register.
  reg         int_adc_spi_we          ;
  reg [3 : 0] int_adc_spi_sel         ;
  reg [15 : 0] int_adc_spi_addr        ;
  // ADC_SPI_DATA register.
  reg [15 : 0] int_adc_spi_wdata       ;
  // DAC_SPI_CNTL register.
  reg [4 : 0] int_dac_spi_prescaler   ;
  reg         int_dac_reset           ;
  reg         int_dac_spi_autopower   ;
  reg         int_dac_spi_start       ;
  // DAC_SPI_ADDR register.
  reg         int_dac_spi_we          ;
  reg [3 : 0] int_dac_spi_sel         ;
  reg [15 : 0] int_dac_spi_addr        ;
  // DAC_SPI_DATA register.
  reg [15 : 0] int_dac_spi_wdata       ;
  // DAC_SPI_AUTOPOWER register.
  reg [15 : 0] int_dac_powerup         ;
  reg [15 : 0] int_dac_powerdown       ;
  // INTERRUPT_SET register.
  reg         int_doorbell_int_set    ;
  // INTERRUPT_CLEAR register.
  reg         int_doorbell_int_clear  ;
  // INTERRUPT_UNMASK register.
  reg         int_doorbell_int_mask   ;
  // V7_GPIO register.
  reg         int_v7_gpio_en          ;
  reg         int_v7_gpio6            ;
  reg         int_v7_gpio5            ;
  // IODELAY_CNTL register.
  reg         int_iodelay_start       ;
  // IODELAY_ADC0 register.
  reg [7 : 0] int_iodelay_adc0        ;
  // IODELAY_DAC0 register.
  reg [7 : 0] int_iodelay_dac0        ;
  // IODELAY_ADC1 register.
  reg [7 : 0] int_iodelay_adc1        ;
  // IODELAY_DAC1 register.
  reg [7 : 0] int_iodelay_dac1        ;
  // IQCOMP_ADC0 register.
  reg [9 : 0] int_gcos_adc0           ;
  reg [9 : 0] int_gsin_adc0           ;
  // IQCOMP_DAC0 register.
  reg [9 : 0] int_gcos_dac0           ;
  reg [9 : 0] int_gsin_dac0           ;
  // IQCOMP_ADC1 register.
  reg [9 : 0] int_gcos_adc1           ;
  reg [9 : 0] int_gsin_adc1           ;
  // IQCOMP_DAC1 register.
  reg [9 : 0] int_gcos_dac1           ;
  reg [9 : 0] int_gsin_dac1           ;
  // RF_CNTRL register.
  reg [3 : 0] int_rf_spi_prescaler    ;
  reg         int_rf_spi_start        ;
  // RF_SPI_ADDR register.
  reg         int_rf_spi_we           ;
  reg [3 : 0] int_rf_spi_sel          ;
  reg [15 : 0] int_rf_spi_addr         ;
  // RF_SPI_DATA register.
  reg [15 : 0] int_rf_spi_wdata        ;
  // RF_FORCE register.
  reg [15 : 0] int_rf_force_en         ;
  reg [15 : 0] int_rf_force            ;
  // RF_GPIO register.
  reg [7 : 0] int_rf_gpio             ;


  //////////////////////////////////////////////////////////////////////////////
  // Ouput linkage
  //////////////////////////////////////////////////////////////////////////////
  assign mictor2_sel         = int_mictor2_sel         ;
  assign mictor1_sel         = int_mictor1_sel         ;
  assign mictor0_sel         = int_mictor0_sel         ;
  assign fb_capture_en       = int_fb_capture_en       ;
  assign fb_playback_en      = int_fb_playback_en      ;
  assign fb_playback_count   = int_fb_playback_count   ;
  assign fb_pathmux_sel      = int_fb_pathmux_sel      ;
  assign adc_spi_prescaler   = int_adc_spi_prescaler   ;
  assign adc_spi_autopower   = int_adc_spi_autopower   ;
  assign adc_spi_start       = int_adc_spi_start       ;
  assign adc_spi_we          = int_adc_spi_we          ;
  assign adc_spi_sel         = int_adc_spi_sel         ;
  assign adc_spi_addr        = int_adc_spi_addr        ;
  assign adc_spi_wdata       = int_adc_spi_wdata       ;
  assign dac_spi_prescaler   = int_dac_spi_prescaler   ;
  assign dac_reset           = int_dac_reset           ;
  assign dac_spi_autopower   = int_dac_spi_autopower   ;
  assign dac_spi_start       = int_dac_spi_start       ;
  assign dac_spi_we          = int_dac_spi_we          ;
  assign dac_spi_sel         = int_dac_spi_sel         ;
  assign dac_spi_addr        = int_dac_spi_addr        ;
  assign dac_spi_wdata       = int_dac_spi_wdata       ;
  assign dac_powerup         = int_dac_powerup         ;
  assign dac_powerdown       = int_dac_powerdown       ;
  assign doorbell_int_set    = int_doorbell_int_set    ;
  assign doorbell_int_clear  = int_doorbell_int_clear  ;
  assign doorbell_int_mask   = int_doorbell_int_mask   ;
  assign v7_gpio_en          = int_v7_gpio_en          ;
  assign v7_gpio6            = int_v7_gpio6            ;
  assign v7_gpio5            = int_v7_gpio5            ;
  assign iodelay_start       = int_iodelay_start       ;
  assign iodelay_adc0        = int_iodelay_adc0        ;
  assign iodelay_dac0        = int_iodelay_dac0        ;
  assign iodelay_adc1        = int_iodelay_adc1        ;
  assign iodelay_dac1        = int_iodelay_dac1        ;
  assign gcos_adc0           = int_gcos_adc0           ;
  assign gsin_adc0           = int_gsin_adc0           ;
  assign gcos_dac0           = int_gcos_dac0           ;
  assign gsin_dac0           = int_gsin_dac0           ;
  assign gcos_adc1           = int_gcos_adc1           ;
  assign gsin_adc1           = int_gsin_adc1           ;
  assign gcos_dac1           = int_gcos_dac1           ;
  assign gsin_dac1           = int_gsin_dac1           ;
  assign rf_spi_prescaler    = int_rf_spi_prescaler    ;
  assign rf_spi_start        = int_rf_spi_start        ;
  assign rf_spi_we           = int_rf_spi_we           ;
  assign rf_spi_sel          = int_rf_spi_sel          ;
  assign rf_spi_addr         = int_rf_spi_addr         ;
  assign rf_spi_wdata        = int_rf_spi_wdata        ;
  assign rf_force_en         = int_rf_force_en         ;
  assign rf_force            = int_rf_force            ;
  assign rf_gpio             = int_rf_gpio             ;


  //////////////////////////////////////////////////////////////////////////////
  // Register write
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge ahb_clk or negedge ahb_nrst)
  begin
    if (!ahb_nrst) 
    begin
      pending_write             <= 1'b0;
      pending_read              <= 1'b0;
      pending_addr              <= 9'b0;
      int_reg_dr                <= 32'b0;
      int_reg_rdy               <= 1'b1;
      // DIAGPORT_CONF1 register.
      int_mictor2_sel          <= 8'b00000000;
      int_mictor1_sel          <= 8'b00000000;
      // DIAGPORT_CONF2 register.
      int_mictor0_sel          <= 16'b0000000000000000;
      // FB_CNTL register.
      int_fb_capture_en        <= 2'b00;
      int_fb_playback_en       <= 2'b00;
      int_fb_playback_count    <= 16'b0000000000000000;
      // FB_PATHMUX register.
      int_fb_pathmux_sel       <= 4'b0000;
      // ADC_SPI_CNTL register.
      int_adc_spi_prescaler    <= 5'b00000;
      int_adc_spi_autopower    <= 1'b0;
      int_adc_spi_start        <= 1'b0;
      // ADC_SPI_ADDR register.
      int_adc_spi_we           <= 1'b0;
      int_adc_spi_sel          <= 4'b0000;
      int_adc_spi_addr         <= 16'b0000000000000000;
      // ADC_SPI_DATA register.
      int_adc_spi_wdata        <= 16'b0000000000000000;
      // DAC_SPI_CNTL register.
      int_dac_spi_prescaler    <= 5'b00000;
      int_dac_reset            <= 1'b0;
      int_dac_spi_autopower    <= 1'b0;
      int_dac_spi_start        <= 1'b0;
      // DAC_SPI_ADDR register.
      int_dac_spi_we           <= 1'b0;
      int_dac_spi_sel          <= 4'b0000;
      int_dac_spi_addr         <= 16'b0000000000000000;
      // DAC_SPI_DATA register.
      int_dac_spi_wdata        <= 16'b0000000000000000;
      // DAC_SPI_AUTOPOWER register.
      int_dac_powerup          <= 16'b0000000000000000;
      int_dac_powerdown        <= 16'b0000000000000000;
      // INTERRUPT_SET register.
      int_doorbell_int_set     <= 1'b0;
      // INTERRUPT_CLEAR register.
      int_doorbell_int_clear   <= 1'b0;
      // INTERRUPT_UNMASK register.
      int_doorbell_int_mask    <= 1'b0;
      // V7_GPIO register.
      int_v7_gpio_en           <= 1'b0;
      int_v7_gpio6             <= 1'b0;
      int_v7_gpio5             <= 1'b0;
      // IODELAY_CNTL register.
      int_iodelay_start        <= 1'b0;
      // IODELAY_ADC0 register.
      int_iodelay_adc0         <= 8'b00000000;
      // IODELAY_DAC0 register.
      int_iodelay_dac0         <= 8'b00000000;
      // IODELAY_ADC1 register.
      int_iodelay_adc1         <= 8'b00000000;
      // IODELAY_DAC1 register.
      int_iodelay_dac1         <= 8'b00000000;
      // IQCOMP_ADC0 register.
      int_gcos_adc0            <= 10'b0100000000;
      int_gsin_adc0            <= 10'b0000000000;
      // IQCOMP_DAC0 register.
      int_gcos_dac0            <= 10'b0100000000;
      int_gsin_dac0            <= 10'b0000000000;
      // IQCOMP_ADC1 register.
      int_gcos_adc1            <= 10'b0100000000;
      int_gsin_adc1            <= 10'b0000000000;
      // IQCOMP_DAC1 register.
      int_gcos_dac1            <= 10'b0100000000;
      int_gsin_dac1            <= 10'b0000000000;
      // RF_CNTRL register.
      int_rf_spi_prescaler     <= 4'b0000;
      int_rf_spi_start         <= 1'b0;
      // RF_SPI_ADDR register.
      int_rf_spi_we            <= 1'b0;
      int_rf_spi_sel           <= 4'b0000;
      int_rf_spi_addr          <= 16'b0000000000000000;
      // RF_SPI_DATA register.
      int_rf_spi_wdata         <= 16'b0000000000000000;
      // RF_FORCE register.
      int_rf_force_en          <= 16'b0000000000000000;
      int_rf_force             <= 16'b0000000000000000;
      // RF_GPIO register.
      int_rf_gpio              <= 8'b00000000;
    end
    else
    begin
      int_reg_rdy <= 1'b1;

      if (pending_write==1'b1)
      begin
        
        pending_write <= 1'b0;
        
        case(pending_addr)

          // Write DIAGPORT_CONF1 register.
          RW_HE_V7_DIAGPORT_CONF1_ADDR_CT :
          begin
              int_mictor2_sel          <= int_reg_dw[15 : 8];
              int_mictor1_sel          <= int_reg_dw[7 : 0];
          end

          // Write DIAGPORT_CONF2 register.
          RW_HE_V7_DIAGPORT_CONF2_ADDR_CT :
          begin
              int_mictor0_sel          <= int_reg_dw[15 : 0];
          end

          // Write FB_CNTL register.
          RW_HE_V7_FB_CNTL_ADDR_CT :
          begin
              int_fb_capture_en        <= int_reg_dw[21 : 20];
              int_fb_playback_en       <= int_reg_dw[17 : 16];
              int_fb_playback_count    <= int_reg_dw[15 : 0];
          end

          // Write FB_PATHMUX register.
          RW_HE_V7_FB_PATHMUX_ADDR_CT :
          begin
              int_fb_pathmux_sel       <= int_reg_dw[3 : 0];
          end

          // Write ADC_SPI_CNTL register.
          RW_HE_V7_ADC_SPI_CNTL_ADDR_CT :
          begin
              int_adc_spi_prescaler    <= int_reg_dw[12 : 8];
              int_adc_spi_autopower    <= int_reg_dw[1];
              int_adc_spi_start        <= int_reg_dw[0];
          end

          // Write ADC_SPI_ADDR register.
          RW_HE_V7_ADC_SPI_ADDR_ADDR_CT :
          begin
              int_adc_spi_we           <= int_reg_dw[31];
              int_adc_spi_sel          <= int_reg_dw[19 : 16];
              int_adc_spi_addr         <= int_reg_dw[15 : 0];
          end

          // Write ADC_SPI_DATA register.
          RW_HE_V7_ADC_SPI_DATA_ADDR_CT :
          begin
              int_adc_spi_wdata        <= int_reg_dw[15 : 0];
          end

          // Write DAC_SPI_CNTL register.
          RW_HE_V7_DAC_SPI_CNTL_ADDR_CT :
          begin
              int_dac_spi_prescaler    <= int_reg_dw[12 : 8];
              int_dac_reset            <= int_reg_dw[2];
              int_dac_spi_autopower    <= int_reg_dw[1];
              int_dac_spi_start        <= int_reg_dw[0];
          end

          // Write DAC_SPI_ADDR register.
          RW_HE_V7_DAC_SPI_ADDR_ADDR_CT :
          begin
              int_dac_spi_we           <= int_reg_dw[31];
              int_dac_spi_sel          <= int_reg_dw[19 : 16];
              int_dac_spi_addr         <= int_reg_dw[15 : 0];
          end

          // Write DAC_SPI_DATA register.
          RW_HE_V7_DAC_SPI_DATA_ADDR_CT :
          begin
              int_dac_spi_wdata        <= int_reg_dw[15 : 0];
          end

          // Write DAC_SPI_AUTOPOWER register.
          RW_HE_V7_DAC_SPI_AUTOPOWER_ADDR_CT :
          begin
              int_dac_powerup          <= int_reg_dw[31 : 16];
              int_dac_powerdown        <= int_reg_dw[15 : 0];
          end

          // Write INTERRUPT_SET register.
          RW_HE_V7_INTERRUPT_SET_ADDR_CT :
          begin
              int_doorbell_int_set     <= int_reg_dw[0];
          end

          // Write INTERRUPT_CLEAR register.
          RW_HE_V7_INTERRUPT_CLEAR_ADDR_CT :
          begin
              int_doorbell_int_clear   <= int_reg_dw[0];
          end

          // Write INTERRUPT_UNMASK register.
          RW_HE_V7_INTERRUPT_UNMASK_ADDR_CT :
          begin
              int_doorbell_int_mask    <= int_reg_dw[0];
          end

          // Write V7_GPIO register.
          RW_HE_V7_V7_GPIO_ADDR_CT :
          begin
              int_v7_gpio_en           <= int_reg_dw[31];
              int_v7_gpio6             <= int_reg_dw[6];
              int_v7_gpio5             <= int_reg_dw[5];
          end

          // Write IODELAY_CNTL register.
          RW_HE_V7_IODELAY_CNTL_ADDR_CT :
          begin
              int_iodelay_start        <= int_reg_dw[0];
          end

          // Write IODELAY_ADC0 register.
          RW_HE_V7_IODELAY_ADC0_ADDR_CT :
          begin
              int_iodelay_adc0         <= int_reg_dw[7 : 0];
          end

          // Write IODELAY_DAC0 register.
          RW_HE_V7_IODELAY_DAC0_ADDR_CT :
          begin
              int_iodelay_dac0         <= int_reg_dw[7 : 0];
          end

          // Write IODELAY_ADC1 register.
          RW_HE_V7_IODELAY_ADC1_ADDR_CT :
          begin
              int_iodelay_adc1         <= int_reg_dw[7 : 0];
          end

          // Write IODELAY_DAC1 register.
          RW_HE_V7_IODELAY_DAC1_ADDR_CT :
          begin
              int_iodelay_dac1         <= int_reg_dw[7 : 0];
          end

          // Write IQCOMP_ADC0 register.
          RW_HE_V7_IQCOMP_ADC0_ADDR_CT :
          begin
              int_gcos_adc0            <= int_reg_dw[25 : 16];
              int_gsin_adc0            <= int_reg_dw[9 : 0];
          end

          // Write IQCOMP_DAC0 register.
          RW_HE_V7_IQCOMP_DAC0_ADDR_CT :
          begin
              int_gcos_dac0            <= int_reg_dw[25 : 16];
              int_gsin_dac0            <= int_reg_dw[9 : 0];
          end

          // Write IQCOMP_ADC1 register.
          RW_HE_V7_IQCOMP_ADC1_ADDR_CT :
          begin
              int_gcos_adc1            <= int_reg_dw[25 : 16];
              int_gsin_adc1            <= int_reg_dw[9 : 0];
          end

          // Write IQCOMP_DAC1 register.
          RW_HE_V7_IQCOMP_DAC1_ADDR_CT :
          begin
              int_gcos_dac1            <= int_reg_dw[25 : 16];
              int_gsin_dac1            <= int_reg_dw[9 : 0];
          end

          // Write RF_CNTRL register.
          RW_HE_V7_RF_CNTRL_ADDR_CT :
          begin
              int_rf_spi_prescaler     <= int_reg_dw[11 : 8];
              int_rf_spi_start         <= int_reg_dw[0];
          end

          // Write RF_SPI_ADDR register.
          RW_HE_V7_RF_SPI_ADDR_ADDR_CT :
          begin
              int_rf_spi_we            <= int_reg_dw[31];
              int_rf_spi_sel           <= int_reg_dw[19 : 16];
              int_rf_spi_addr          <= int_reg_dw[15 : 0];
          end

          // Write RF_SPI_DATA register.
          RW_HE_V7_RF_SPI_DATA_ADDR_CT :
          begin
              int_rf_spi_wdata         <= int_reg_dw[15 : 0];
          end

          // Write RF_FORCE register.
          RW_HE_V7_RF_FORCE_ADDR_CT :
          begin
              int_rf_force_en          <= int_reg_dw[31 : 16];
              int_rf_force             <= int_reg_dw[15 : 0];
          end

          // Write RF_GPIO register.
          RW_HE_V7_RF_GPIO_ADDR_CT :
          begin
              int_rf_gpio              <= int_reg_dw[7 : 0];
          end
        
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default : 
          begin
          end
          // pragma coverage block = on 

        endcase
      end

      if (pending_read==1'b1)
      begin
        
        pending_read <= 1'b0;
        int_reg_dr   <= 32'b0;
        
        case(pending_addr)

        // Read SIGNATURE register.
        RW_HE_V7_SIGNATURE_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= fpga_signature;
        end

        // Read DATE register.
        RW_HE_V7_DATE_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= fpga_date;
        end

        // Read DIAGPORT_CONF1 register.
        RW_HE_V7_DIAGPORT_CONF1_ADDR_CT :
        begin
          int_reg_dr[15 : 8]        <= mictor2_sel;
          int_reg_dr[7 : 0]         <= mictor1_sel;
        end

        // Read DIAGPORT_CONF2 register.
        RW_HE_V7_DIAGPORT_CONF2_ADDR_CT :
        begin
          int_reg_dr[15 : 0]        <= mictor0_sel;
        end

        // Read FB_CNTL register.
        RW_HE_V7_FB_CNTL_ADDR_CT :
        begin
          int_reg_dr[31]            <= fb_capture_done;
          int_reg_dr[21 : 20]       <= fb_capture_en;
          int_reg_dr[17 : 16]       <= fb_playback_en;
          int_reg_dr[15 : 0]        <= fb_playback_count;
        end

        // Read FB_PATHMUX register.
        RW_HE_V7_FB_PATHMUX_ADDR_CT :
        begin
          int_reg_dr[3 : 0]         <= fb_pathmux_sel;
        end

        // Read ADC_SPI_CNTL register.
        RW_HE_V7_ADC_SPI_CNTL_ADDR_CT :
        begin
          int_reg_dr[31]            <= adc_spi_done;
          int_reg_dr[12 : 8]        <= adc_spi_prescaler;
          int_reg_dr[1]             <= adc_spi_autopower;
          int_reg_dr[0]             <= adc_spi_start;
        end

        // Read ADC_SPI_ADDR register.
        RW_HE_V7_ADC_SPI_ADDR_ADDR_CT :
        begin
          int_reg_dr[31]            <= adc_spi_we;
          int_reg_dr[19 : 16]       <= adc_spi_sel;
          int_reg_dr[15 : 0]        <= adc_spi_addr;
        end

        // Read ADC_SPI_DATA register.
        RW_HE_V7_ADC_SPI_DATA_ADDR_CT :
        begin
          int_reg_dr[31 : 16]       <= adc_spi_rdata;
          int_reg_dr[15 : 0]        <= adc_spi_wdata;
        end

        // Read DAC_SPI_CNTL register.
        RW_HE_V7_DAC_SPI_CNTL_ADDR_CT :
        begin
          int_reg_dr[31]            <= dac_spi_done;
          int_reg_dr[12 : 8]        <= dac_spi_prescaler;
          int_reg_dr[2]             <= dac_reset;
          int_reg_dr[1]             <= dac_spi_autopower;
          int_reg_dr[0]             <= dac_spi_start;
        end

        // Read DAC_SPI_ADDR register.
        RW_HE_V7_DAC_SPI_ADDR_ADDR_CT :
        begin
          int_reg_dr[31]            <= dac_spi_we;
          int_reg_dr[19 : 16]       <= dac_spi_sel;
          int_reg_dr[15 : 0]        <= dac_spi_addr;
        end

        // Read DAC_SPI_DATA register.
        RW_HE_V7_DAC_SPI_DATA_ADDR_CT :
        begin
          int_reg_dr[31 : 16]       <= dac_spi_rdata;
          int_reg_dr[15 : 0]        <= dac_spi_wdata;
        end

        // Read DAC_SPI_AUTOPOWER register.
        RW_HE_V7_DAC_SPI_AUTOPOWER_ADDR_CT :
        begin
          int_reg_dr[31 : 16]       <= dac_powerup;
          int_reg_dr[15 : 0]        <= dac_powerdown;
        end

        // Read DIAG0_STAT register.
        RW_HE_V7_DIAG0_STAT_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= diag0_stat;
        end

        // Read DIAG1_STAT register.
        RW_HE_V7_DIAG1_STAT_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= diag1_stat;
        end

        // Read DIAG2_STAT register.
        RW_HE_V7_DIAG2_STAT_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= diag2_stat;
        end

        // Read DIAG3_STAT register.
        RW_HE_V7_DIAG3_STAT_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= diag3_stat;
        end

        // Read DIAG4_STAT register.
        RW_HE_V7_DIAG4_STAT_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= diag4_stat;
        end

        // Read INTERRUPT_SET register.
        RW_HE_V7_INTERRUPT_SET_ADDR_CT :
        begin
          int_reg_dr[0]             <= doorbell_int_set;
        end

        // Read INTERRUPT_CLEAR register.
        RW_HE_V7_INTERRUPT_CLEAR_ADDR_CT :
        begin
          int_reg_dr[0]             <= doorbell_int_clear;
        end

        // Read INTERRUPT_UNMASK register.
        RW_HE_V7_INTERRUPT_UNMASK_ADDR_CT :
        begin
          int_reg_dr[0]             <= doorbell_int_mask;
        end

        // Read INTERRUPT_STATUS register.
        RW_HE_V7_INTERRUPT_STATUS_ADDR_CT :
        begin
          int_reg_dr[0]             <= doorbell_int_status;
        end

        // Read V7_GPIO register.
        RW_HE_V7_V7_GPIO_ADDR_CT :
        begin
          int_reg_dr[31]            <= v7_gpio_en;
          int_reg_dr[6]             <= v7_gpio6;
          int_reg_dr[5]             <= v7_gpio5;
          int_reg_dr[3]             <= v7_gpio3;
          int_reg_dr[2]             <= v7_gpio2;
          int_reg_dr[1]             <= v7_gpio1;
          int_reg_dr[0]             <= v7_gpio0;
        end

        // Read IODELAY_CNTL register.
        RW_HE_V7_IODELAY_CNTL_ADDR_CT :
        begin
          int_reg_dr[31]            <= iodelay_done;
          int_reg_dr[0]             <= iodelay_start;
        end

        // Read IODELAY_ADC0 register.
        RW_HE_V7_IODELAY_ADC0_ADDR_CT :
        begin
          int_reg_dr[7 : 0]         <= iodelay_adc0;
        end

        // Read IODELAY_DAC0 register.
        RW_HE_V7_IODELAY_DAC0_ADDR_CT :
        begin
          int_reg_dr[7 : 0]         <= iodelay_dac0;
        end

        // Read IODELAY_ADC1 register.
        RW_HE_V7_IODELAY_ADC1_ADDR_CT :
        begin
          int_reg_dr[7 : 0]         <= iodelay_adc1;
        end

        // Read IODELAY_DAC1 register.
        RW_HE_V7_IODELAY_DAC1_ADDR_CT :
        begin
          int_reg_dr[7 : 0]         <= iodelay_dac1;
        end

        // Read IQCOMP_ADC0 register.
        RW_HE_V7_IQCOMP_ADC0_ADDR_CT :
        begin
          int_reg_dr[25 : 16]       <= gcos_adc0;
          int_reg_dr[9 : 0]         <= gsin_adc0;
        end

        // Read IQCOMP_DAC0 register.
        RW_HE_V7_IQCOMP_DAC0_ADDR_CT :
        begin
          int_reg_dr[25 : 16]       <= gcos_dac0;
          int_reg_dr[9 : 0]         <= gsin_dac0;
        end

        // Read IQCOMP_ADC1 register.
        RW_HE_V7_IQCOMP_ADC1_ADDR_CT :
        begin
          int_reg_dr[25 : 16]       <= gcos_adc1;
          int_reg_dr[9 : 0]         <= gsin_adc1;
        end

        // Read IQCOMP_DAC1 register.
        RW_HE_V7_IQCOMP_DAC1_ADDR_CT :
        begin
          int_reg_dr[25 : 16]       <= gcos_dac1;
          int_reg_dr[9 : 0]         <= gsin_dac1;
        end

        // Read RF_CNTRL register.
        RW_HE_V7_RF_CNTRL_ADDR_CT :
        begin
          int_reg_dr[31]            <= rf_spi_done;
          int_reg_dr[11 : 8]        <= rf_spi_prescaler;
          int_reg_dr[0]             <= rf_spi_start;
        end

        // Read RF_SPI_ADDR register.
        RW_HE_V7_RF_SPI_ADDR_ADDR_CT :
        begin
          int_reg_dr[31]            <= rf_spi_we;
          int_reg_dr[19 : 16]       <= rf_spi_sel;
          int_reg_dr[15 : 0]        <= rf_spi_addr;
        end

        // Read RF_SPI_DATA register.
        RW_HE_V7_RF_SPI_DATA_ADDR_CT :
        begin
          int_reg_dr[31 : 16]       <= rf_spi_rdata;
          int_reg_dr[15 : 0]        <= rf_spi_wdata;
        end

        // Read RF_FORCE register.
        RW_HE_V7_RF_FORCE_ADDR_CT :
        begin
          int_reg_dr[31 : 16]       <= rf_force_en;
          int_reg_dr[15 : 0]        <= rf_force;
        end

        // Read RF_GPIO register.
        RW_HE_V7_RF_GPIO_ADDR_CT :
        begin
          int_reg_dr[7 : 0]         <= rf_gpio;
        end

        // Read RF_IRQ register.
        RW_HE_V7_RF_IRQ_ADDR_CT :
        begin
          int_reg_dr[7 : 0]         <= rf_irq;
        end

        // Read SVNREV_MODEM register.
        RW_HE_V7_SVNREV_MODEM_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= svnrev_modem;
        end

        // Read SVNREV_FPGAB register.
        RW_HE_V7_SVNREV_FPGAB_ADDR_CT :
        begin
          int_reg_dr[31 : 0]        <= svnrev_fpgab;
        end
        
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default : int_reg_dr <= 32'b0;
          // pragma coverage block = on 

        endcase
      end

      // Pulse generation
      // Hardware update
        if (doorbell_int_set_in_valid)
          int_doorbell_int_set     <= doorbell_int_set_in;
        if (doorbell_int_clear_in_valid)
          int_doorbell_int_clear   <= doorbell_int_clear_in;

      if(hready_in==1'b1 && hsel==1'b1 && htrans[1]==1'b1)
      begin
        if(hwrite==1'b1)
        begin
          pending_addr  <= haddr[10:2];
          pending_write <= 1'b1;
        end
        else
        begin
          pending_addr  <= haddr[10:2];
          pending_read  <= 1'b1;
          int_reg_rdy   <= 1'b0;
        end
      end
    end
  end




  //////////////////////////////////////////////////////////////////////////////
  // Read data 
  //////////////////////////////////////////////////////////////////////////////
  assign  hrdata     = int_reg_dr;

  //////////////////////////////////////////////////////////////////////////////
  // Write data 
  //////////////////////////////////////////////////////////////////////////////
  assign  int_reg_dw = hwdata;

  assign  hready = int_reg_rdy;
  assign  hresp  = 2'b0;

`ifdef RW_ASSERT_ON

`endif // RW_ASSERT_ON
  
endmodule

////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////

