#!/bin/bash
#/////////////////////////////////////////////////////////////////////////////
#/  Copyright (C) by RivieraWaves.
#/  This module is a confidential and proprietary property of RivieraWaves
#/  and a possession or use of this module requires written permission
#/  from RivieraWaves.
#/----------------------------------------------------------------------------
#/ $Author: $
#/ Company          : RivieraWaves
#/----------------------------------------------------------------------------
#/ $Revision: $
#/ $Date: $
#/ ---------------------------------------------------------------------------
#/ Dependencies     : None
#/ Description      : Compilation and elaboration script
#/ Application Note :
#/ Terms & concepts :
#/ Bugs             :
#/ Open issues and future enhancements :
#/ References       :
#/ Revision History :
#/ ---------------------------------------------------------------------------
#/
#/ $HeadURL: $
#/
#/////////////////////////////////////////////////////////////////////////////


COMMAND_LINE="$@"


# Set log file directory if it is not set in the environment
RW_LOGDIR="${RW_LOGDIR:=../../logs}"

# Set work directory if it is not set in the environment
RW_SIMDIR="${RW_SIMDIR:=dev}"
WORKDIR="work.${RW_SIMDIR}"

# Set configuration directory
CONFDIR="${SOURCESLIB}/env/CONF"

# set env directory
ENVDIR="../../tb_env"

# set include directory
INCLUDEDIR="../../defines"

# set directory containing pli/dpi files
PLIDIR="${ENVDIR}/pli/"

# set directory containing this script
RUNDIR=`dirname "$(readlink -f "$0")"` #use absolute path

# set directory for reports
REPORTDIR="../../reports/"

# Matlab backup directory
MATLAB_BACKUP_DIR="./Pattern_backup"

# coverage directory
COVERAGEDIR=cov_work

# Name of tarball file to create
TARBALL_NAME="simu_backup.tgz"

# Current date
STARTDATE=`date`

# timeout command for killing simulation that hangs after setup amount of time
# duration can be set in seconds 's', minutes 'm', hours 'h'
TIMEOUT_DURATION=${TIMEOUT_DURATION:=90m}
TIMEOUT_CMD="timeout -s 9 $TIMEOUT_DURATION"
TIMEOUT_CMD=""

# WLAN_NX_TOP RTL Config
IP_CONFIG="${CONFDIR}/config_STA_1x1_CBW40_LDPC.v"

# define for log severity, by default normal
LOG_SEVERITY="+define+LOG_SEVERITY=2"

# verbosity for UVM messaging system (UVM_LOW, UVM_MEDIUM, UVM_HIGH, UVM_NONE, UVM_DEBUG)
UVM_VERBOSITY=UVM_LOW

# set verbosity for specific component in verification environment
# example: *.m_ahb_master_agent.monitor,_ALL_,UVM_DEBUG,run
UVM_INST_VERBOSITY=*.m_rst_agent.monitor,_ALL_,${UVM_VERBOSITY},build

# determine maximum error count before simulation stops, default value is infinite
UVM_ERR_COUNT=0

# setup watchdog timer to prevent simulation to hung infinitely
WDT_VALUE="+define+WDT_VALUE=300ms"

# UVM test case
TESTCASENAME=test_wlan_regs_reset

# list of UVM source directories for irun include directive
INCDIR_UVM=""

INCLUDE_CPU=""

CPU=""

# top tb module file
TOPTBFILE=${ENVDIR}/sv/tb/src/wlan_top_tb.sv

# top tb name
TOPTBNAME=wlan_top_tb

# Testcase root directory
TESTCASEPATH=${ENVDIR}/sv/tests

# shared objects required by PLI
PLI_SO="${ENVDIR}/pli/libpli.so"

# Matlab libdump shared object
MDMLIBDUMP=${LIBDUMPROOT}/../libhwmdmAX/libhwmdmAX.so

# C files used in environment
C_FILES=$(find ${SOURCESLIB}/SIMS_UVM/tb_env/c/ -name "*.c")

# run options
OPTIONS=""


# check which tool verstion we are using
# SIMTOOLVERSION="$(which simvision | grep XCELIUMMAIN)"
if [ ! -z $SIMTOOLVERSION ]
then
  echo "Simulation running with XCELIUM"
  SIMRUNTOOL=xrun
  OPTIONS="$OPTIONS -define XCELIUM"
  # Arguments required by NC Sim
  NCSIM_ARGS="-v93 -v200x -uvm -disable_sem2009 -xceligen on"
else
  echo "Simulation running with IES"
  SIMRUNTOOL=vcs
  # Arguments required by NC Sim
  NCSIM_ARGS="-full64 -sverilog +v2k -debug_access+all -ntb_opts uvm-1.2"
fi

# Coverage collection related arguments
NCSIM_COV="-ASSERT_COUNT_TRACES -covoverwrite -coverage all -covfile covfile -covworkdir ${COVERAGEDIR} -covdesign wlan_top_tb.u_rw_he_top_wrapper.u_rw_he_top -covdut wlan_top_tb -covtest tc_"${TEST}

#Assertion related arguments
FLAGS="$FLAGS"

#--------------------------------------------------------------------#
#--- Ctrl + C trap                                                ---#
#--------------------------------------------------------------------#
ctrl_c ()
{
if [ $RUNID != 0 ]
then
  if [ $RW_VERIFDB == 1 ]
  then
    verifdb.bash -u $RUNID aborted
  fi
  echo
  echo "------------------------------------------------------------"
  echo "-- SIMULATION ABORTED"
  echo "------------------------------------------------------------"
fi
echo "Received control-C"
exit 0
}
# trap keyboard interrupt (ctrl-c)
trap ctrl_c SIGINT


#--------------------------------------------------------------------#
#--- Print Help                                                   ---#
#--------------------------------------------------------------------#
DisplayHelp()
{
  echo "
    This script compiles the design and the TB and launch the simulation

    Options:

      -clean                 : Clean all temp files and logs

      -nodb                  : Run simulation without database

      -db                    : Run simulation with database

      -d | -define           : Add define for compilation

      -up_reg | -ur          : Update UVM register model ( HW/SIMS_UVM/tb_env/sv/wlan_env/common/regmodel/ )

      -seed <n>              : select seed randomization value
                             : to get a random seed set to: random

      -all                   : Run simulation of all the testcases.

      -f <filename>          : Run simulation of all the testcases defined in
                               tests/<filename>.

      -t <test1> <test2> ... : Run simulation of the listed testcases.

      -l                     : list the testcases available

      -gui                   : Start interactive session

      -loc                   : compile LOCALSOURCSELIB instead of SOURCESLIB

      -noverifdb             : Disable verif database logging

      -debug                 : Print debug messages in log - DEPRECATED use 'verbose' flag in UVM TB

      -simu.run <filename>   : Provide the simulation command script to be used by irun

      -verbose               : Setup UVM verbosity level

      -inst_verbose          : Set verbosity level of specific component in UVM environment
                               <comp>,<id>,<verbosity>,<phase>

      -err_cnt               : set error count before simulator stops

      -fsdb                  : Enable FSDB generation support

      -cov | -coverage       : Enable Code Coverage

      -config <name>    : Select configuration of the DUT "
  CONFIGLIST=`find ${CONFDIR}/ -type f -name "*.v" -exec basename {} \; | sed 's/\.v//' | sort -u`
  echo "                           Available configuration list :"
  for CONFIGNAME in ${CONFIGLIST}
  do
    echo "                            - $CONFIGNAME"
  done
  echo "

      -prof                  : Enable Profiling

      -cpu                   : Select the Top Level IP including a CPU (RISCV) and run simulation of rw_he_top_cpu.

      -jen                   : Enable jenkins log

      -comp_dpi              : Enable compilation of DPI code

      -keep_matlab           : Instead of deleting Matlab config files, move the files to ${MATLAB_BACKUP_DIR} for later debugging

      -for | -forensic       : Creates a tarball named ${TARBALL_NAME} of the working directory

      -h | -help             : Display this message



    Examples:
    ---------

    ./simulate.bash -t mytc                 - Will run the testcase mytc in the tests folder
    ./simulate.bash -gui -t mytc            - Will run the testcase mytc in the tests folder in interactive mode
    ./simulate.bash -nodb -seed 3 -t mytc   - Will run the testcase mytc in the tests folder without database and with random seed 3
  "
}

#--------------------------------------------------------------------#
#--- PrintErrorWarning <message>                                  ---#
#--------------------------------------------------------------------#
PrintErrorWarning ()
{
  echo "ERROR: $*"
  echo ""
  exit 2
}


#--------------------------------------------------------------------#
#--- Create working directory, create symlink noise file for Rx   ---#
#--- and change into working directory                            ---#
#--------------------------------------------------------------------#
GotoWorkDir()
{
  if [ ! -d "$WORKDIR" ]
  then
    # directory doesn't exist, create it
    mkdir $WORKDIR
  fi

  if [ ! -e "$WORKDIR/noise_0dB.txt" ]
  then
    ln -s $RUNDIR/noise_0dB.txt $WORKDIR/noise_0dB.txt
  fi

  cd $WORKDIR
}

#--------------------------------------------------------------------#
#--- Clean temporary files and recompile DPI if COMP_DPI is set  ---#
#--------------------------------------------------------------------#
CleanDB ()
{
  rm -rf INCA* csrc *.daidir .vcs* simv DVEfiles
  rm -rf *.log *.key *.mif
  rm -rf dirlist.txt

  if [ ! -e "./Pattern/mu" ]
  then
    mkdir -p ./Pattern/mu
  else
    rm -f ./Pattern/*.txt
    rm -f ./Pattern/mu/*.txt
  fi

  if [ $COMP_DPI == 1 ]
  then
    cd $PLIDIR && make -f Makefile clean && make -f Makefile lin
    cd $RUNDIR
    cd $WORKDIR
  fi
}

#--------------------------------------------------------------------#
#--------------------------------------------------------------------#
#--- Run a simulation                                             ---#
#--------------------------------------------------------------------#
#--------------------------------------------------------------------#
RunSimulation ()
{

  if [ -f ${TESTCASEPATH}/WLAN/src/${TESTCASENAME}.sv ] || [ -f ${TESTCASEPATH}/modem/src/${TESTCASENAME}.sv ] || [ -f ${TESTCASEPATH}/MAC/src/${TESTCASENAME}.sv ]
  then
        echo "RunSimulation $TESTCASENAME"
  else
        echo "Testcase $TESTCASENAME does not exists"
        exit 1
  fi

  # Set Coverage script
  if [ $COVERAGE == 1 ]
  then
    COV_OPTIONS="-ASSERT_COUNT_TRACES -covoverwrite -coverage all -covfile ${RUNDIR}/covfile -covworkdir ${COVERAGEDIR} -covdut  rw_he_top -covtest tc_${CONFIGNAME}_${TESTCASENAME}_seed_${SEED} ${OPTIONS}"
  else
    COV_OPTIONS=$OPTIONS
  fi

  STARTDATE2=`date +%s`

  # Clean old ncverilog database
  CleanDB

  # Create local copy of LDPC RAM
  if [[ $DEFINELIST == *"RW_NX_FPGA_SIM"* ]]; then
    # FPGA
    \cp -rf $PHYSUBSYSDIR/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXBD/ldpcDec/config/bin/8_6_3_2p_120_120.bin ./ldpc_cfg_ram.bin || PrintErrorWarning "ldpc_cfg_ram.bin not found !!!"
  else
    \cp -rf $PHYSUBSYSDIR/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXBD/ldpcDec/config/bin/ldpc_cfg_ram.bin . || PrintErrorWarning "ldpc_cfg_ram.bin not found !!!"
  fi

  # Create local copy of AGC RAM
  \cp -rf $SOURCESLIB/SB/RIU_KARST/AGCKarstBinary/agcram_simu.bin ram.bin || PrintErrorWarning "agcram_simu.bin not found !!!"

  # run the testcase
  echo "OPTIONS: $COV_OPTIONS"

  NIRUNLOOP=0
#           -PROF_MEM_CALLGRAPH -PROFILE -PROFTHREAD -PLIVERBOSE -PLIDEBUG -PERFSTAT -DPI_STACK_INT_C -64BIT \

  while [ "$NIRUNLOOP" != "2" ]
  do
    vcs ${NCSIM_ARGS} $COV_OPTIONS ${FLAGS} +define+UVM_VERBOSITY=${UVM_VERBOSITY} ${WDT_VALUE} +define+UVM_TESTNAME=${TESTCASENAME} \
           +define+UVM_MAX_QUIT_COUNT=${UVM_ERR_COUNT} \
           +define+UVM_REPORT_DISABLE_FILE \
           +incdir+${LIBDUMPROOT}/src \
           +incdir+${LIBDUMPROOT}/../libhwmdmAX \
           +incdir+${PHYSUBSYSDIR}/HDMCORE/OFDMACORE/OFDMTXCORE/ldpcEnc/verilog/rtl \
           +incdir+${PHYSUBSYSDIR}/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXBD/ldpcDec/verilog/rtl \
           ${INCDIR_UVM} \
           ${INCLUDE_CPU} \
           +incdir+${MCR_ROOT}/extern/include \
           +incdir+${LIBDUMPROOT}/src \
           +incdir+${LIBDUMPROOT}/distrib \
           +incdir+${LIBDUMPROOT}/../libhwmdmAX \
           +incdir+${ENVDIR}/pli \
           ${LIBDUMPROOT}/src/mdm_ref_func.c \
           ${MDMLIBDUMP} \
           ${IP_CONFIG} \
           ${INCLUDEDIR}/timescale.v \
           ${C_FILES} \
           ${PLI_SO} \
           ${INCLUDEDIR}/tbDefines.v \
           -f ${MODEM80211BCOREDIR}/modem802_11b_core/cfg/modem802_11b_core.list \
           -f ${TOPLEVEL_IP_NAME}_rtl_files.list \
           -f rw_he_top_tb_files.list \
           -f rw_he_wlan_uvm_files.list \
           ${TOPTBFILE} \
           -top ${TOPTBNAME} \
           -l ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log | tee ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}_with_matlab.log

    TESTNCSIMERR1=`grep -c "INTERNAL EXCEPTION" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log `
    TESTNCSIMERR2=`grep -c "SIGUSR" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log `
    TESTMATLABERR=`grep -c "Error in m_" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}_with_matlab.log`
    if [ $TESTNCSIMERR1$TESTNCSIMERR2 != 00 ]
    then
      if [ $TESTMATLABERR == 0 ]
      then
        NIRUNLOOP=$(($NIRUNLOOP+1))
        echo "NCSIM ERROR detected. Rerun the simulation one more time ($NIRUNLOOP)"
      else
        NIRUNLOOP=2
      fi
    else
      NIRUNLOOP=2
    fi

  done

  ENDDATE2=`date +%s`
  DURATION=$(($ENDDATE2-$STARTDATE2))

  # when random seed is used, log file needs to be renamed with actual seed value
  if [ $SEED == random ]
  then
    SEED_VALUE=$(grep "SEED" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log | sed 's/SVSEED set randomly from command line: //')
    mv ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}.log
    mv ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}_with_matlab.log ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}_with_matlab.log
    # rename coverage file
    if [ $COVERAGE == 1 ]
    then
      mv ${COVERAGEDIR}/scope/tc_${CONFIGNAME}_${TESTCASENAME}_seed_${SEED} ${COVERAGEDIR}/scope/tc_${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}
    fi
  else
    SEED_VALUE=${SEED}
  fi

  if [ $GUI == 0 ]
  then
    # Update Simulation Results Summary file
    TESTSIMU=`grep -c "= SCENARIO PASSED =" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}.log `
    TESTFATAL=`grep -c "UVM_FATAL :    0" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}.log `
    TESTERROR=`grep -c "UVM_ERROR :    0" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}.log `
    TESTASSERT=`egrep -c "Failing Assertions = 0|No assertions found." ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}.log`
    TESTNCSIMERR1=`grep -c "INTERNAL EXCEPTION" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}.log `
    TESTNCSIMERR2=`grep -c "SIGUSR" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}.log `
    TESTMATLABERR=`grep -c "Error in m_" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}_with_matlab.log`

    if [ $TESTFATAL$TESTERROR$TESTASSERT == 111 ]
#    if [ $TESTSIMU == 1 ]
    then
      if [ $RUNID != 0 ]
      then
        verifdb.bash -t $RUNID $TESTCASENAME pass
      fi
      echo "$1 PASSED ${SEED_VALUE}" >> simulationStatus.rpt
      echo
      echo "$1 PASSED ${SEED_VALUE}"
      echo
      if [ $JENKINS == 1 ]
      then
        echo "<testcase classname=\"$TBNAME\" name=\"${TESTCASENAME}_seed_${SEED_VALUE}\" time=\"$DURATION\">" >> jenkins.rpt
        echo "</testcase>" >> jenkins.rpt
      fi
    else
      if [ $JENKINS == 1 ]
      then
        echo "<testcase classname=\"$TBNAME\" name=\"${TESTCASENAME}_seed_${SEED_VALUE}\" time=\"$DURATION\">" >> jenkins.rpt
        if [ $TESTMATLABERR != 0 ]
        then
          echo "<failure type=\"MATLAB\"> MATLAB Error">> jenkins.rpt
          MESSAGE1=`grep -B10 "Error in m_" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}_with_matlab.log | sed 's/\[.>/ /g'| sed 's/\].//g' | sed 's/[<>]//g'`
          echo "$MESSAGE1">> jenkins.rpt
        elif [ $TESTNCSIMERR1$TESTNCSIMERR2 != 00 ]
        then
          echo "<skipped type=\"CADENCE\"> NCSIMERROR">> jenkins.rpt
          MESSAGE1=`grep -B5 "INTERNAL EXCEPTION" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log `
          MESSAGE2=`grep -B5 "SIGUSR" ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log `
          echo "$MESSAGE1">> jenkins.rpt
          echo "$MESSAGE2">> jenkins.rpt
        elif [ $TESTERROR != 1 ]
        then
          MESSAGE=`grep -B5 UVM_ERROR ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log | head -n 6`
          echo "<failure type=\"SIMU\"> UVM_ERROR">> jenkins.rpt
          echo "$MESSAGE">> jenkins.rpt
        elif [ $TESTFATAL != 1 ]
        then
          MESSAGE=`grep -B5 UVM_FATAL ${RW_LOGDIR}/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED}.log`
          echo "<failure type=\"SIMU\"> UVM_FATAL\n">> jenkins.rpt
          echo "$MESSAGE">> jenkins.rpt
        elif [ $TESTSIMU != 1 ]
        then
          echo "<failure type=\"SIMU\"> MYSIMUERROR">> jenkins.rpt
        else
          echo "<failure type=\"UNKNOW\"> UNKNOW">> jenkins.rpt
        fi
        echo "" >> jenkins.rpt
        echo "command line   : simulate.bash ${COMMAND_LINE}" >> jenkins.rpt
        echo "folder         : ${PWD}" >> jenkins.rpt
        echo "simu.log       : ${JOB_URL}/ws/SIMS_UVM/logs/${CONFIGNAME}_${TESTCASENAME}_seed_${SEED_VALUE}_with_matlab.log" >> jenkins.rpt
        echo "=============================================================================================================" >> jenkins.rpt

        if [ $TESTMATLABERR != 0 ]
        then
          echo "</failure>" >> jenkins.rpt
        elif [ $TESTNCSIMERR1$TESTNCSIMERR2 != 00 ]
        then
          echo "</skipped>" >> jenkins.rpt
        else
          echo "</failure>" >> jenkins.rpt
        fi
        echo "</testcase>" >> jenkins.rpt
      fi
      if [ $RUNID != 0 ]
      then
        verifdb.bash -t $RUNID $TESTCASENAME fail
      fi
      echo "$1 FAILED ${SEED_VALUE} (TESTFATAL=$TESTFATAL TESTERROR=$TESTERROR TESTASSERT=$TESTASSERT TESTMATLABERR=$TESTMATLABERR TESTNCSIMERR1,TESTNCSIMERR2=$TESTNCSIMERR1,$TESTNCSIMERR2)" >> simulationStatus.rpt
      echo
      echo "$1 FAILED ${SEED_VALUE} (TESTFATAL=$TESTFATAL TESTERROR=$TESTERROR TESTASSERT=$TESTASSERT TESTMATLABERR=$TESTMATLABERR TESTNCSIMERR1,TESTNCSIMERR2=$TESTNCSIMERR1,$TESTNCSIMERR2)"
      echo
     #exit 1
    fi
  fi

}

#--------------------------------------------------------------------#
#--- Generate compilation list                                    ---#
#--------------------------------------------------------------------#
complistgen()
{
  LOCALPATH=`pwd`
  SYSTEMVERILOGPATH=${ENVDIR}/sv

  if [ $LOC == 0 ]
  then
    SELECTEDSOURCESLIB=${SOURCESLIB}
    SELECTEDHWCOMMONDIR=${HWCOMMONDIR}
    SELECTEDPHYSUBSYSDIR=${PHYSUBSYSDIR}
    SELECTEDRIUCOREDIR=${RIUCOREDIR}
    SELECTEDMDMCOMMONDIR=${MDMCOMMONDIR}
    SELECTEDHDMCOREDIR=${HDMCOREDIR}
    SELECTEDMACSUBSYSDIR=${MACSUBSYSDIR}
    SELECTEDMACCOREDIR=${MACCOREDIR}
    SELECTEDTOP11AXDIR=${TOP11AXDIR}
  else
    SELECTEDSOURCESLIB=${LOCALSOURCESLIB}
    SELECTEDHWCOMMONDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax/HWCOMMON
    SELECTEDPHYSUBSYSDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax/PHYSUBSYS
    SELECTEDRIUCOREDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax/PHYSUBSYS/MDMCOMMON/RIUCORE
    SELECTEDMDMCOMMONDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax/PHYSUBSYS/MDMCOMMON
    SELECTEDHDMCOREDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE
    SELECTEDMACSUBSYSDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax/MACSUBSYS
    SELECTEDMACCOREDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax/MACSUBSYS/MACCORE
    SELECTEDTOP11AXDIR=${LOCALSOURCESLIB}/IPs/HW/TOP11ax
  fi
  cd ${SELECTEDSOURCESLIB}/IPs/HW/TOP11ax/${TOPLEVEL_IP_NAME}/simu
  HWCOMMONDIR="${SELECTEDHWCOMMONDIR}"   \
  PHYSUBSYSDIR="${SELECTEDPHYSUBSYSDIR}" \
  RIUCOREDIR="${SELECTEDRIUCOREDIR}"     \
  MDMCOMMONDIR="${SELECTEDMDMCOMMONDIR}" \
  HDMCOREDIR="${SELECTEDHDMCOREDIR}"     \
  MACSUBSYSDIR="${SELECTEDMACSUBSYSDIR}" \
  MACCOREDIR="${SELECTEDMACCOREDIR}"     \
  TOP11AXDIR="${SELECTEDTOP11AXDIR}"     \
  SOURCESLIB="${SELECTEDSOURCESLIB}" create_comp_file.pl ${TOPLEVEL_IP_NAME} rtl $LOCALPATH/${TOPLEVEL_IP_NAME}_rtl_files.list
  cd -
  cd ${SELECTEDSOURCESLIB}/IPs/HW/TOP11ax/rw_he_top/simu
  HWCOMMONDIR="${SELECTEDHWCOMMONDIR}"   \
  PHYSUBSYSDIR="${SELECTEDPHYSUBSYSDIR}" \
  RIUCOREDIR="${SELECTEDRIUCOREDIR}"     \
  MDMCOMMONDIR="${SELECTEDMDMCOMMONDIR}" \
  HDMCOREDIR="${SELECTEDHDMCOREDIR}"     \
  MACSUBSYSDIR="${SELECTEDMACSUBSYSDIR}" \
  MACCOREDIR="${SELECTEDMACCOREDIR}"     \
  TOP11AXDIR="${SELECTEDTOP11AXDIR}"     \
  SOURCESLIB="${SELECTEDSOURCESLIB}" create_comp_file.pl rw_he_top tb  $LOCALPATH/rw_he_top_tb_files.list
  cd -
  # create list of include directories for UVM componetns
  INCDIR_UVM=$(find $SYSTEMVERILOGPATH -name "src" -o -name "seq_lib" -o -name "callbacks" | awk '{print "+incdir+"$1}')
  # create list of all UVM testbench packages
  find $SYSTEMVERILOGPATH/security_lib/   -name "*_pkg.sv" | sort -o rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/misc/ -name "*_pkg.sv" | sort >> rw_he_wlan_uvm_files.list
  # regmodel needs special care, as the top level package has to be compiled last
  find $SYSTEMVERILOGPATH/wlan_env/common/regmodel/callbacks -type f \( -iname "*_pkg.sv" \)  | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/wlan_env/common/regmodel/src -type f \( -iname "*pkg.sv" ! -iname "REGMODEL_pkg.sv" \)  | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/wlan_env/common/regmodel/src -type f \( -iname "REGMODEL_pkg.sv" \)  | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/wlan_env/common/key_storage_ram/ -type f \( -iname "KeyStorageRAM_pkg.sv" \)  | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/wlan_env/common -type d \( -path '*common/regmodel' -o -path '*common/key_storage_ram' \) -prune -o -type f -name "*_pkg.sv" -print | sort >> rw_he_wlan_uvm_files.list
  # first compile all agents then UVC wrappers because of dependencies
  find $SYSTEMVERILOGPATH/wlan_env/UVCs   -name "*_pkg.sv" | grep    "_agent" | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/wlan_env/UVCs   -name "*_pkg.sv" | grep -v "_agent" | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/wlan_env/src    -name "*_pkg.sv" | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/tests/WLAN      -name "*_pkg.sv" | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/tests/modem     -name "*_pkg.sv" | sort >> rw_he_wlan_uvm_files.list
  find $SYSTEMVERILOGPATH/tests/MAC       -name "*_pkg.sv" | sort >> rw_he_wlan_uvm_files.list
}


#--------------------------------------------------------------------#
#--- Main                                                         ---#
#--------------------------------------------------------------------#
#--------------------------------------------------------------------#


# Display help if no arg
if test $# == 0
then
  DisplayHelp
  exit 1
fi


# Clear variable
RUNID=0
TBNAME="WLAN_HE_IP_UVM"
GUI=0
COVERAGE=0
SEED=1
LOC=0
VERBOSITY="NORMAL"
ALL=0
USERDEF=-1
NOVERIFDB=0
UP_REG=0
SIMURUN="simu.run"
JENKINS=0
DEFINELIST=""
AGC_RADAR_FSM_CONF=0
AGC_FPGA_FSM_CONF=0
AGC=""
FSDB=0
COMP_DPI=0
TOPLEVEL_IP_NAME="rw_he_top"
CPU="0"
KEEP_MATLAB=0

# Create working directory if it doesn't exist already and change into it
GotoWorkDir

# Check all arg
while test $# != 0
do

  arg="$1"
  OPTION=`echo $arg | sed 's/^-//'`
  shift
  case ${OPTION}
  in
    clean)
      echo "Cleaning all temp files"
      CleanDB
      rm -rf *.list
      # go back to run dir
      cd ../
      rm -rf $WORKDIR
      exit 1
    ;;
    seed)
      SEED=$1
      shift
    ;;
    all)
      #------------------------------------------------------------------------#
      #--- Run all the testcases contained in testcases folder              ---#
      #------------------------------------------------------------------------#
      TESTCASELIST=`find ${ENVDIR}/sv/tests -type f \( -iname "*.sv" ! -iname "*pkg.sv" \) -printf '%f\n' | sed 's/\.sv//' | sort`
      OPTIONS="+define+RW_SIMU_ON +define+SIMU_ON $OPTIONS"

      if [ $SIMURUN == "simu.run" ]
      then
        SIMURUN="simu.reg.run"
      fi
      ALL=1
      if [ $RW_VERIFDB == 1 ]
      then
         desc="Regression of WLAN NX IP done with SEED=$SEED<br>List of define used : $DEFINELIST"
         SVNREVMAC=`cat -n $SOURCESLIB/IPs/HW/MAC/Src/.svn/entries | grep -w 4 | cut -f 2`
         SVNREVMODEM=`cat -n $SOURCESLIB/IPs/HW/Modem/Src/.svn/entries | grep -w 4 | cut -f 2`
         SVNREVTB=`cat -n .svn/entries | grep -w 4 | cut -f 2`
         desc="$desc<br>Detailed SVN Revision : <br>MAC=$SVNREVMAC<br>MODEM=$SVNREVMODEM<br>TB=$SVNREVTB"
         #RUNID=`verifdb.bash -b $TBNAME -c "$desc"`
      fi
    ;;
    config)
      if test $# = 0
      then
        echo "ERROR : Configuration name missing"
        echo "Please choose between available configuration : "
        #------------------------------------------------------------------------#
        #--- List the available configuration                                 ---#
        #------------------------------------------------------------------------#
        CONFIGLIST=`find ${CONFDIR}/ -type f -name "*.v" -exec basename {} \; | sed 's/\.v//' | sort -u`
        echo "Available configuration list :"
        echo "------------------------------"
        for CONFIGNAME in ${CONFIGLIST}
        do
          echo " - $CONFIGNAME"
        done
        echo
        exit 1
      else
        CONFIGNAME=${1}
        echo
        echo "Configured with ${CONFDIR}/$CONFIGNAME.v"
        echo
        IP_CONFIG="${CONFDIR}/$CONFIGNAME.v"
        shift
      fi
    ;;
    d | define)
      #------------------------------------------------------------------------#
      #--- Run frequency clock                                              ---#
      #------------------------------------------------------------------------#
      echo "define $1"
      OPTIONS="$OPTIONS +define+$1"
      DEFINELIST="$DEFINELIST<br>$1"
      shift
    ;;
    up_reg | ur)
      #------------------------------------------------------------------------#
      #--- Update verilog macro for register                                ---#
      #------------------------------------------------------------------------#
      UP_REG=1
    ;;
    nodb)
      #------------------------------------------------------------------------#
      #--- Run without database                                             ---#
      #------------------------------------------------------------------------#
      if [ $SIMURUN == "simu.run" ]
      then
      SIMURUN="simu.reg.run"
      fi
    ;;
    db)
      #------------------------------------------------------------------------#
      #--- Run with database                                                ---#
      #------------------------------------------------------------------------#
      SIMURUN="simu.run"
    ;;
    simu.run)
      #------------------------------------------------------------------------#
      #--- Override command script name for irun                            ---#
      #------------------------------------------------------------------------#
      echo "Command script : \"$1\""
      SIMURUN=$1
      shift
    ;;
    tbname)
      #------------------------------------------------------------------------#
      #--- Override testbench name for verifdb logging                      ---#
      #------------------------------------------------------------------------#
      echo "Testbench name \"$1\""
      TBNAME=$1
      shift
    ;;
    fsdb)
      #------------------------------------------------------------------------#
      #--- Run with fsdb support                                            ---#
      #------------------------------------------------------------------------#
      FSDB=1
    ;;
    prof)
      #------------------------------------------------------------------------#
      #--- Run with profiling                                               ---#
      #------------------------------------------------------------------------#
      echo "Run with profiling..."
      OPTIONS="$OPTIONS -iprof"
    ;;
    noverifdb)
      #------------------------------------------------------------------------#
      #--- Run without verif database logging                               ---#
      #------------------------------------------------------------------------#
      NOVERIFDB=1
    ;;
    noagc)
      #------------------------------------------------------------------------#
      #--- Run without AGC                                                  ---#
      #------------------------------------------------------------------------#
      AGC="+define+AGC_ON=0"
      echo " Running without AGC..."
    ;;
    f)
      #------------------------------------------------------------------------#
      #--- Run the testcases defined in a file                              ---#
      #------------------------------------------------------------------------#
      if test $# = 0
      then
        echo "ERROR : filename missing"
        exit 1
      fi

      if [ -f ${TESTCASEPATH}/WLAN/regression_lists/$1 ]
      then
        TMPLIST=`cat ${TESTCASEPATH}/WLAN/regression_lists/$1  | egrep -v "^#|^//|^--" | sed 's/\n//'`
        TESTCASELIST="$TESTCASELIST"$'\n'"$TMPLIST"
        echo " testcase list : "
        echo "$TESTCASELIST"
        echo ""
      elif [ -f ${TESTCASEPATH}/modem/regression_lists/$1 ]
      then
        TMPLIST=`cat ${TESTCASEPATH}/modem/regression_lists/$1  | egrep -v "^#|^//|^--" | sed 's/\n//'`
        TESTCASELIST="$TESTCASELIST"$'\n'"$TMPLIST"
        echo " testcase list : "
        echo "$TESTCASELIST"
        echo ""
        echo "Modem standalone test detected, adding 'define STANDALONE_PHY'"
        OPTIONS="-define STANDALONE_PHY $OPTIONS"
      elif  [ -f ${TESTCASEPATH}/MAC/regression_lists/$1 ]
      then
        TMPLIST=`cat ${TESTCASEPATH}/MAC/regression_lists/$1 | egrep -v "^#|^//|^--" | sed 's/\n//'`
        TESTCASELIST="$TESTCASELIST"$'\n'"$TMPLIST"
        echo " testcase list : "
        echo "$TESTCASELIST"
        echo ""
        echo "MAC standalone test detected, adding 'define STANDALONE_MAC'"
        OPTIONS="-define STANDALONE_MAC $OPTIONS"
      else
        echo "ERROR : file named $1 does not exist in test folders"
        exit 1
      fi
      OPTIONS="+define+RW_SIMU_ON +define+SIMU_ON $OPTIONS"
      if [ $SIMURUN == "simu.run" ]
      then
      SIMURUN="simu.reg.run"
      fi
      if [ $RW_VERIFDB == 1 -a $NOVERIFDB == 0 ]
      then
         desc="Regression of WLAN NX IP based on file $1 done with SEED=$SEED<br>List of define used : $DEFINELIST"
         SVNREVMAC=`cat -n $SOURCESLIB/IPs/HW/MAC/Src/.svn/entries | grep -w 4 | cut -f 2`
         SVNREVMODEM=`cat -n $SOURCESLIB/IPs/HW/Modem/Src/.svn/entries | grep -w 4 | cut -f 2`
         SVNREVTB=`cat -n .svn/entries | grep -w 4 | cut -f 2`
         desc="$desc<br>Detailed SVN Revision : <br>MAC=$SVNREVMAC<br>MODEM=$SVNREVMODEM<br>TB=$SVNREVTB"
         #RUNID=`verifdb.bash -b $TBNAME -c "$desc"`
      fi

      shift
    ;;
    t)
      #------------------------------------------------------------------------#
      #--- Run a list of testcases                                          ---#
      #------------------------------------------------------------------------#
      TESTCASELIST=""
      while [ $# != 0 ] && [[ $1 != -* ]]
      do
        arg="$1"
        TESTCASELIST="$TESTCASELIST $1"
        TESTCASENAME=$1
        shift
      done
      OPTIONS="+define+RW_SIMU_ON +define+SIMU_ON $OPTIONS"
      if [[ $TESTCASENAME =~ ^test_modem.* ]]
      then
        echo "Modem standalone test detected, adding 'define STANDALONE_PHY'"
        OPTIONS="-define STANDALONE_PHY $OPTIONS"
      fi
      if [[ $TESTCASENAME =~ ^test_mac.* ]]
      then
        echo "MAC standalone test detected, adding 'define STANDALONE_MAC'"
        OPTIONS="-define STANDALONE_MAC $OPTIONS"
      fi

    ;;
    l)
      #------------------------------------------------------------------------#
      #--- List the available testcases                                     ---#
      #------------------------------------------------------------------------#

      TESTCASELIST=`find ${ENVDIR}/sv/tests -type f \( -iname "test_*.sv" ! -iname "*pkg.sv" ! -iname "*_base*" \) -printf '%f\n' | sed 's/\.sv//' | sort`
      echo "Available testcases list:"
      echo "-------------------------"
      for TESTCASENAME in ${TESTCASELIST}
      do
        echo " - $TESTCASENAME"
      done
      echo
      TESTCASELIST=`find ${ENVDIR}/sv/tests -type f -name "*.list" -printf '%f\n' | sort`
      echo "Available testsuites list:"
      echo "--------------------------"
      for TESTCASENAME in ${TESTCASELIST}
      do
        echo " - $TESTCASENAME"
      done
      echo
      exit 0
    ;;
    gui )
      GUI=1
      # when running in GUI no timeout command should be set
      TIMEOUT_CMD=""
    ;;
    radar )
      AGC_RADAR_FSM_CONF=1
      OPTIONS="${OPTIONS} -define RADARDETEN"
    ;;
    fpga )
      AGC_FPGA_FSM_CONF=1
      OPTIONS="${OPTIONS} -define FPGA_AGC_FSM"
    ;;
    loc )
      LOC=1
    ;;
    debug )
      LOG_SEVERITY=" +define+LOG_SEVERITY=3 "
      echo "!!! debug flag is DEPRECATED, use 'verbose' flag in UVM TB !!!"
    ;;
    verbose )
      echo "verbosity set to $1"
      UVM_VERBOSITY=${1}
      shift
    ;;
    inst_verbose )
      echo "set component verbosity $1"
      UVM_INST_VERBOSITY=${1}
      shift
    ;;
    err_cnt )
      echo "set error count to $1"
      UVM_ERR_COUNT=${1}
      shift
    ;;
    cov | coverage)
      COVERAGE=1
    ;;
    userdef)
      USERDEF=$1
    ;;
    jen)
      JENKINS=1
    ;;
    cpu )
      CPU=1
      DEFINE="$DEFINE +define+RISCV_CPU +define+ZERORISCY";
      INCLUDE_CPU="$INCLUDE_CPU +incdir+$ZERORISCY_ROOT/include"
      OPTIONS="${OPTIONS} ${DEFINE} +define+MEMMAP_OFFSET=32'h60000000 +define+WLAN_HE_WITH_CPU"
      TOPLEVEL_IP_NAME="rw_he_top_cpu"
    ;;
    comp_dpi)
      #------------------------------------------------------------------------#
      #--- Compile DPI code                                                 ---#
      #------------------------------------------------------------------------#
      COMP_DPI=1
    ;;
    keep_matlab)
      #------------------------------------------------------------------------#
      #--- Backup sysparam.txt/phyPayload.txt instead of deleting it        ---#
      #------------------------------------------------------------------------#
      KEEP_MATLAB=1
      OPTIONS="${OPTIONS} -define KEEP_MATLAB"
    ;;
    for | forensic)
      #------------------------------------------------------------------------#
      #--- Create a tarball of the work directory                           ---#
      #------------------------------------------------------------------------#
        EXCLUDELIST="${WORKDIR}/INCA_libs\n${WORKDIR}/.simvision\n${WORKDIR}/*.err\n${WORKDIR}/*.diag\n${WORKDIR}/tmp_reg\n${WORKDIR}/.reinvoke.sim\n${WORKDIR}/REINVOKE_SIMVISION\n${WORKDIR}/irun.key\n"
        cd ${RUNDIR}
        echo -e $EXCLUDELIST > excludeme.txt
        if [ -e "$TARBALL_NAME" ]
        then
          echo "WARNING: tarball already exists. Delete ?"
          select USER_ANSWER in "Yes" "No"; do
            case $USER_ANSWER in
              Yes ) rm -rf ${TARBALL_NAME}; break;;
              No ) exit 1;;
            esac
          done
        fi
        echo "Please stand by while the tarball is created ... "
        tar cpvfz ${TARBALL_NAME} ${WORKDIR} -X ${RUNDIR}/excludeme.txt
#         tar -cvzf ${TARBALL_NAME} ${WORKDIR} -X ${RUNDIR}/excludeme.txt
        rm excludeme.txt
        exit 0
    ;;
    h | help)
      #------------------------------------------------------------------------#
      #--- Display Help                                                     ---#
      #------------------------------------------------------------------------#
      DisplayHelp
      exit 1
    ;;
    *)
      #------------------------------------------------------------------------#
      #--- Unknown Option                                                   ---#
      #------------------------------------------------------------------------#
      echo "Unknown Option"
      DisplayHelp
      exit 1
    ;;
  esac
done

OS_OPTION=""
# Add 64 bit option if needed
if [ `uname -m` = "x86_64"  ]
then
 OS_OPTION="-full64"
fi

# Create directory to store all matlab sysparam/payload files
if [ $KEEP_MATLAB == 1 ]
then
  if [ ! -e "$MATLAB_BACKUP_DIR" ]
  then
    mkdir ${MATLAB_BACKUP_DIR}
 fi
fi

# Add gui option
if [ $GUI == 1 ]
then
  if [ $SIMURUN == "simu.run" ]
  then
    SIMURUN="simu.gui.run"
  fi
  OPTIONS="-gui -linedebug -uvmlinedebug $OPTIONS"
fi

if [ $COMP_DPI == 1 ]
then
  cd $PLIDIR && make -f Makefile clean && make -f Makefile lin
  cd $RUNDIR
  cd $WORKDIR
fi

# update verilog macros for register files
if [ $UP_REG == 1 ]
then
  gen_all_reg_models.sh
fi

echo "set top level to ${TOPLEVEL_IP_NAME}"

#generate compilation list
complistgen $LOC

# Clear simulationStatus.rpt
echo > simulationStatus.rpt
echo "--------------------------------------------------" >> simulationStatus.rpt
echo "  test case name         |   status    |   seed   |" >> simulationStatus.rpt
echo "--------------------------------------------------" >> simulationStatus.rpt


LD_LIBRARY_PATH=${MCR_ROOT}/bin/glnxa64:${MCR_ROOT}/runtime/glnxa64:${MCR_ROOT}/sys/os/glnxa64:$LD_LIBRARY_PATH
echo $LD_LIBRARY_PATH

if [ "$FSDB" == 1 ]
then
  VERDIPATH="/share/eda/synopsys/verdi3-I-201403-SP2"
  LD_LIBRARY_PATH=${VERDIPATH}/share/PLI/IUS/LINUX64:${VERDIPATH}/share/PLI/IUS/LINUX64/boot:$LD_LIBRARY_PATH
fi

# Main loop to run all the selected testcases
for TESTCASENAME in ${TESTCASELIST}
do
  RunSimulation $TESTCASENAME
  echo
  echo "$TESTCASENAME Simulation Completed"
  echo "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
  echo
done


# Generate Code coverage Database with iccr
if [ $COVERAGE == 1 ]
then
  echo "Generate Code Coverage Merge database"
  echo
  imc -64bit -exec ../cov_report.cmd || exit 1
  echo
  echo "Code Coverage database generation done"
fi

echo
echo " Simulation Results"
cat simulationStatus.rpt
echo
if [ $ALL == 1 ]
then
  echo
  echo
  egrep FAILED simulationStatus.rpt
  echo
fi

# Rw verif db set
if [ $RUNID != 0 ]
then
  if [ $RW_VERIFDB == 1 ]
  then
    verifdb.bash -u $RUNID success
    mkdir -p  ${REPORTDIR}
    if [ -e  ${REPORTDIR}/simulate_report_links.htm ]
    then
      echo "http://verif.frso.rivierawaves.com/index.php?run=$RUNID" >> ${REPORTDIR}/simulate_report_links.htm
    else
      echo "http://verif.frso.rivierawaves.com/index.php?run=$RUNID" >  ${REPORTDIR}/simulate_report_links.htm
    fi
    PASSED=`egrep -c PASSED simulationStatus.rpt`
    FAILED=`egrep -c FAILED simulationStatus.rpt`
    FAILURE=`egrep FAILED simulationStatus.rpt`
    echo "PASSED = $PASSED" >> ${REPORTDIR}/simulate_report_links.htm
    echo "FAILED = $FAILED" >> ${REPORTDIR}/simulate_report_links.htm
    echo ""                 >> ${REPORTDIR}/simulate_report_links.htm
    if [ $FAILED != 0 ]
    then
      echo "$FAILURE"         >> ${REPORTDIR}/simulate_report_links.htm
      echo ""                 >> ${REPORTDIR}/simulate_report_links.htm
      echo ""                 >> ${REPORTDIR}/simulate_report_links.htm
    fi
  fi
  echo
  echo
  egrep FAILED simulationStatus.rpt
  echo
fi

ENDDATE=`date`
echo $STARTDATE
echo $ENDDATE
echo "STARTED : $STARTDATE " >> simulationStatus.rpt
echo "ENDED   : $ENDDATE " >> simulationStatus.rpt
#exit 0


#-------------------------------------------------------------------------------
#- End of file
#-------------------------------------------------------------------------------
