//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthou $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 26496 $
// $Date: 2016-05-27 09:24:12 +0200 (Fri, 27 May 2016) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
//   This is the PHY Tx/Rx vector. It contains control parameters to/from the
//   Modem. It also contains a function for TX-TIME calculation.
//
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <math.h>
#include "phyVectorAX.h"

// Function DPI-C to print through verilog testbench in irun.log
extern void tb_print(const char *STR);
extern void tb_fire_error(const char *STR);

// calculate number of symbols in HESIGB
int GetNSymbHESIGB(int isSIGBCompression, int NUserFields, int NDBPS) {

  int numBitsCC = 0; //number of bits in common content
  int numBitsUF = 0; //numner of bits in user field
  int Nsymb     = 0;
  int CRCTailBits = 10;

  if (isSIGBCompression) {
    numBitsCC = 0;
  } else {
    numBitsCC = 8 + CRCTailBits;
  }

  numBitsUF = floor(NUserFields/2)*(2*21+CRCTailBits) + (NUserFields%2)*(21+CRCTailBits);

  Nsymb     = floor((numBitsCC + numBitsUF)/NDBPS) + (((numBitsCC + numBitsUF)%NDBPS) > 0);

  return Nsymb;
}

// get number of user fields from RU allocation
int GetNumUFHESIGB(int RUAllocationDec) {

  int NUFs = 0;

  RUAllocationDec = RUAllocationDec % 256;
  if ((RUAllocationDec >= 0) && (RUAllocationDec <= 15)) {
    switch (RUAllocationDec) {
      case 0:
        NUFs = 9;
        break;
      case 1:
      case 2:
      case 4:
      case 8:
        NUFs = 8;
        break;
      case 3:
      case 5:
      case 6:
      case 9:
      case 10:
      case 12:
        NUFs = 7;
        break;
      case 7:
      case 11:
      case 13:
      case 14:
        NUFs = 6;
        break;
      case 15:
        NUFs = 5;
        break;
    }
  } else if ((RUAllocationDec >= 16) && (RUAllocationDec <= 23)) {
    NUFs = RUAllocationDec - 16 + 3;
  } else if ((RUAllocationDec >= 24) && (RUAllocationDec <= 31)) {
    NUFs = RUAllocationDec - 24 + 4;
  } else if ((RUAllocationDec >= 32) && (RUAllocationDec <= 39)) {
    NUFs = RUAllocationDec - 32 + 6;
  } else if ((RUAllocationDec >= 40) && (RUAllocationDec <= 47)) {
    NUFs = RUAllocationDec - 40 + 5;
  } else if ((RUAllocationDec >= 48) && (RUAllocationDec <= 55)) {
    NUFs = RUAllocationDec - 48 + 5;
  } else if ((RUAllocationDec >= 56) && (RUAllocationDec <= 63)) {
    NUFs = RUAllocationDec - 56 + 4;
  } else if ((RUAllocationDec >= 64) && (RUAllocationDec <= 71)) {
    NUFs = RUAllocationDec - 64 + 6;
  } else if ((RUAllocationDec >= 72) && (RUAllocationDec <= 79)) {
    NUFs = RUAllocationDec - 72 + 5;
  } else if ((RUAllocationDec >= 80) && (RUAllocationDec <= 87)) {
    NUFs = RUAllocationDec - 80 + 5;
  } else if ((RUAllocationDec >= 88) && (RUAllocationDec <= 95)) {
    NUFs = RUAllocationDec - 88 + 4;
  } else if ((RUAllocationDec >= 96) && (RUAllocationDec <= 111)) {
    RUAllocationDec = RUAllocationDec - 96;
    NUFs = (RUAllocationDec % 4) + 1 + floor(RUAllocationDec/4) + 1;
  } else if (RUAllocationDec == 112){
    NUFs = 4;
  } else if ((RUAllocationDec >= 113) && (RUAllocationDec <= 115)) {
    NUFs = 0;
  } else if ((RUAllocationDec >= 128) && (RUAllocationDec <= 191)) {
    RUAllocationDec = RUAllocationDec - 128;
    NUFs = (RUAllocationDec%8) + 1 + floor(RUAllocationDec/8) + 1;
  } else if ((RUAllocationDec >= 192) && (RUAllocationDec <= 199)) {
    NUFs = RUAllocationDec - 192 + 1;
  } else if ((RUAllocationDec >= 200) && (RUAllocationDec <= 207)) {
    NUFs = RUAllocationDec - 200 + 1;
  } else if ((RUAllocationDec >= 208) && (RUAllocationDec <= 215)) {
    NUFs = RUAllocationDec - 208 + 1;
  } else {
    NUFs = 0;
  }

  return NUFs;
}

// Function DPI-C to calculate TXTIME
int ComputeTimeOnAirAX (
   int length_i,
   int preType_i,
   int mcsIndex_i,
   int giType_i,
   int ruType_i,
   int heLtfType_i,
   int numHeLtf_i,
   int dcm_i,
   int packetExtension_i,
   int heTbLength_i,
   int triggerMethod_i,
   int doppler_i,
   int mma_i,
   int stbc_i,
   int woPreamble_i,
   const HESIGB_s *heSigB_i,
   int debug_i)
{

  // Max of 1024 character for debug printing through tb_print
  char   STR[1024];
  // Total TXTIME
  float  timeOnAirFloat        = 0.0;
  int    timeOnAir             = 0;
  // Preamble Time
  float  preambleTimeOnAir     = 0.0;
  // Midamble Time
  float  midambleTimeOnAir     = 0.0;
  // HE-LTF Time
  float  heLtfTimeOnAir        = 0.0;
  // PPDU Time
  float  dataTimeOnAir         = 0.0;
  // Number of HE-SIGB symbols
  int    numHeSigb             = 0;
  // Num He-SIGB Time
  float  numHeSigbTimeOnAir    = 0.0;
  int    nDbpsHeSigB           = 0;
  int    numUserFields         = 0;

  // IEEE corresponding variables
  int    nSS                   = 0;
  float  tGiData               = 0.0;
  int    nSym                  = 0.0;
  float  tSym                  = 0.0;
  float  tGiHeLtf              = 0.0;
  float  tHeLtf                = 0.0;
  float  tHeLtfSym             = 0.0;
  int    nHeLtf                = 0;
  int    nDbps                 = 0;
  int    nTail                 = 0;
  float  r                     = 0.0;
  int    nBpscs                = 0;
  int    nSdShort              = 0;
  int    nCbpsShort            = 0;
  int    nDbpsShort            = 0;
  int    nExcess               = 0;
  int    aInit                 = 0;
  int    mMa                   = 0;
  int    nMa                   = 0;
  int    nominalPacketPadding  = 0;
  int    defaultPeDuration     = 0;
  int    tPe                   = 0;
  int    mSTBC                 = 0;

  // First Step: Get all the IEEE Variables
  // Nss
  nSS        = 1; // 1x1 20MHz, Nss is always '1'

  if(debug_i==1){
    sprintf(STR,"\nnSS               = %0d (preType_i = %0d, mcsIndex_i = %0d)",nSS,preType_i,mcsIndex_i);
    tb_print(STR);
  }

  // tGiData
  switch (giType_i){
    case 0:  tGiData = TGI1_DATA; break;
    case 1:  tGiData = TGI2_DATA; break;
    case 2:  tGiData = TGI4_DATA; break;
    default: tGiData = 0;
  }

  // tSym
  tSym = TDFT_HE + tGiData;

  if(debug_i==1){
    sprintf(STR,"Tsym              = %0f (giType = %0d)",tSym,giType_i);
    tb_print(STR);
  }

  // tGiHeLtf
  switch (giType_i){
    case 0:  tGiHeLtf = TGI1_DATA; break;
    case 1:  tGiHeLtf = TGI2_DATA; break;
    case 2:  tGiHeLtf = TGI4_DATA; break;
    default: tGiHeLtf = 0;
  }

  // tHeLtf
  switch (heLtfType_i){
    case 0:  tHeLtf = THE_LTF_1X; break;
    case 1:  tHeLtf = THE_LTF_2X; break;
    case 2:  tHeLtf = THE_LTF_4X; break;
    default: tHeLtf = 0;
  }

  // tHeLtfSym
  tHeLtfSym = tHeLtf + tGiHeLtf;

  if(debug_i==1){
    sprintf(STR,"The-ltf-sym       = %0f (heLtfType = %0d, giType = %0d)",tHeLtfSym,heLtfType_i,giType_i);
    tb_print(STR);
  }

  // nHeLtf
  nHeLtf = numHeLtf_i + 1;

  if(debug_i==1){
    sprintf(STR,"Nhe-ltf           = %0d",nHeLtf);
    tb_print(STR);
  }

  // Ndbps,short
  r          = R[mcsIndex_i];
  nBpscs     = NBPSCS[mcsIndex_i];
  nSdShort   = HE_NSDSHORT[dcm_i][ruType_i];
  nCbpsShort = nSdShort*nSS*nBpscs;
  nDbpsShort = nCbpsShort*r;

  if(debug_i==1){
    sprintf(STR,"Nsd,short         = %0d (dcm_i = %0d, ruType_i = %0d)",nSdShort,dcm_i,ruType_i);
    tb_print(STR);
    sprintf(STR,"Ncbps,short       = %0d (nSdShort = %0d, nBpscs = %0d)",nCbpsShort,nSdShort,nBpscs);
    tb_print(STR);
    sprintf(STR,"Ndbps,short       = %0d (r = %0f)",nDbpsShort,r);
    tb_print(STR);
  }

  // Mma
  if (mma_i==0) {
     mMa = 10;
  } else {
     mMa = 20;
  }


  // number of DATA BIT PER SYMBOL
  // ruType_i       : from 0 to 7
  // nSS            : 1 only
  // mcsIndex_i[3:0]: from 0 to 11
  // dcm_i          : from 0 to 1
  nDbps = HE_NDBPS[ruType_i][nSS-1][mcsIndex_i & 0xF][dcm_i];
  if (nDbps==0) {
     // in case of Incorrect parameters, Ndbps is forced to 1.
     // This prevent crash when divide by 0 (Ndbps)
     nDbps=1;
  }

  if(debug_i==1){
    sprintf(STR,"nDBPS             = %0d (ruType = %0d)",nDbps,ruType_i);
    tb_print(STR);
  }

  // Ntail
  nTail = 6;

  // HE-LTF Calculation
  // heLtfTimeOnAir = NHE-LTF * THE-LTF-SYM
  heLtfTimeOnAir =  nHeLtf * tHeLtfSym;

  // calculate HE-SIG duration, present only in HE-MU frame
  if (preType_i == 6) {

    // number of DATA BIT PER SYMBOL in HE-SIGB, OFDM
    nDbpsHeSigB   = HESIGB_NDBPS[heSigB_i->mcs & 0xF] >> (heSigB_i->dcm);
    numUserFields = GetNumUFHESIGB(heSigB_i->ru_allocation);
    numHeSigb     = GetNSymbHESIGB(heSigB_i->compressed_mode, numUserFields, nDbpsHeSigB);
    if (debug_i==1) {
      sprintf(STR,"SIGB MCS = %0d, SIGB DCM = %0d, RU ALLOCATION = %0d",
      heSigB_i->mcs, heSigB_i->dcm, heSigB_i->ru_allocation);
      tb_print(STR);
      sprintf(STR,"nDbpsHeSigB = %0d, numUserFields = %0d, numHeSigb = %0d",nDbpsHeSigB,numUserFields,numHeSigb);
      tb_print(STR);
    }
    // calculate duration of HE-SIBB field
    numHeSigbTimeOnAir = numHeSigb * THE_SIG_B;
    if (debug_i==1) {
      sprintf(STR,"numHeSigbTimeOnAir = %0f",numHeSigbTimeOnAir);
      tb_print(STR);
    }
  }

  // preamble time on air
  switch(preType_i){
    case 5:  preambleTimeOnAir = HE_SU_PRE_BASE    + heLtfTimeOnAir; break;
    case 6:  preambleTimeOnAir = HE_MU_PRE_BASE    + numHeSigbTimeOnAir + heLtfTimeOnAir; break;
    case 7:  preambleTimeOnAir = HE_ER_SU_PRE_BASE + heLtfTimeOnAir; break;
    case 8:  preambleTimeOnAir = HE_TB_PRE_BASE    + heLtfTimeOnAir; break;
    default: preambleTimeOnAir = 0;
  }

  if(debug_i==1){
    sprintf(STR,"heLtfTimeOnAir    = %0f (Nhe-ltf = %0d)",heLtfTimeOnAir,nHeLtf);
    tb_print(STR);
    sprintf(STR,"preambleTimeOnAir = %0f (woPreamble_i = %0d)",preambleTimeOnAir,woPreamble_i);
    tb_print(STR);
  }

  // Nominal Packet Padding & default PE Duration
  if (preType_i==8 && triggerMethod_i==1) {
     // Trigger frame with UMRS trigger method
     switch (packetExtension_i & 0x7) {
     case 0:  defaultPeDuration = 0;  break;
     case 1:  defaultPeDuration = 4;  break;
     case 2:  defaultPeDuration = 8; break;
     case 3:  defaultPeDuration = 12; break;
     case 4:  defaultPeDuration = 16; break;
     default: defaultPeDuration = 0;  break;
     }
  } else {
     // Trigger frame with UMRS trigger method
     switch (packetExtension_i) {
     case 0:  nominalPacketPadding = 0;  break;
     case 2:  nominalPacketPadding = 8;  break;
     case 4:  nominalPacketPadding = 16; break;
     default: nominalPacketPadding = 0;  break;
     }
  }

  // mSTBC is the common STBC setting among all the users, as described in 27.3.11.7
  if(stbc_i == 1)
  {
    mSTBC = 2;
  } else {
    mSTBC = 1;
  }

  // Compute the initial number of symbol segments in the last OFDM symbol(s)
  nExcess = ((length_i*8) + nTail + NSERVICE) % (nDbps*mSTBC);

  if (nExcess==0){
     aInit = 4;
  } else {
     aInit = ceilf( ((float)nExcess) / ((float)nDbpsShort*mSTBC) );

     if (aInit > 4) {
        aInit = 4;
     }
  }

  // Data Time on Air

  if (preType_i==8 && triggerMethod_i==1) {
     // Total of symbols is provided in case of HE-TB PPDU with UMRS trigger.
     nSym = heTbLength_i+1;

     if(debug_i==1){
       sprintf(STR,"Nsym              = %0d (preType_i= %0d  heTbLength_i = %0d)",nSym,preType_i,heTbLength_i);
       tb_print(STR);
     }
  } else {

     // Total of Symbols = ceil( (8*length_i + Ntail + Nservice) / NDBPS )
     nSym = mSTBC * ceilf( ((float)((length_i*8) + nTail + NSERVICE)) / ((float)(nDbps*mSTBC)) );

     if(debug_i==1){
       sprintf(STR,"Nsym              = %0d (length_i = %0d  nDBPS = %0d, stbc_i = %0d, mSTBC = %0d)",nSym,length_i,nDbps,stbc_i,mSTBC);
       tb_print(STR);
     }
  }

  // Data time on air
  dataTimeOnAir = nSym * tSym;

  if(debug_i==1){
    sprintf(STR,"dataTimeOnAir     = %0f (giType_i = %0d)",dataTimeOnAir,giType_i);
    tb_print(STR);
  }

  // Compute the number of midamble periods
  if (nSym<=1 || doppler_i==0) {
     nMa = 0;
  } else {
     nMa = ceilf( ((float)(nSym-1)) / ((float)mMa) ) - 1;
  }

  if(debug_i==1) {
    sprintf(STR,"Nma               = %0d (Mma = %0d)",nMa,mMa);
    tb_print(STR);
  }

  // Compute the midamble time on air
  midambleTimeOnAir = nMa * nHeLtf * tHeLtfSym;

  if(debug_i==1){
    sprintf(STR,"midambleTimeOnAir = %0f (Nma = %0d)",midambleTimeOnAir,nMa);
    tb_print(STR);
  }

  if(debug_i==1){
    sprintf(STR,"aInit             = %0d (Nexcess = %0d, Ndbps,short = %0d)",aInit,nExcess,nDbpsShort);
    tb_print(STR);
  }

  // Compute the packet extension duration
  if (preType_i==8 && triggerMethod_i==1) {
     // Trigger frame with UMRS trigger method
     tPe = defaultPeDuration;
  } else if ((aInit==1 && nominalPacketPadding==16) ||
             (aInit==3 && nominalPacketPadding==8)) {
     tPe = 4;
  } else if ((aInit==2 && nominalPacketPadding==16) ||
             (aInit==4 && nominalPacketPadding==8)) {
     tPe = 8;
  } else if ((aInit==3 && nominalPacketPadding==16)) {
     tPe = 12;
  } else if ((aInit==4 && nominalPacketPadding==16)) {
     tPe = 16;
  } else {
     tPe = 0;
  }

  if(debug_i==1) {
    sprintf(STR,"Tpe               = %0d (preType_i = %0d, triggerMethod_i = %0d, defaultPeDuration = %0d, nominalPacketPadding = %0d)",tPe,preType_i,triggerMethod_i,defaultPeDuration,nominalPacketPadding);
    tb_print(STR);
  }

  // Calculate the full time on air of the frame (without signal extension)
  if(woPreamble_i == 0){
     timeOnAirFloat = 20 + preambleTimeOnAir + dataTimeOnAir + midambleTimeOnAir + tPe;
  } else {
     timeOnAirFloat =                          dataTimeOnAir + midambleTimeOnAir + tPe;
  }
  timeOnAir      = (int)timeOnAirFloat;

  // Round Up the timeOnAir
  if ((timeOnAirFloat-timeOnAir)>0.1) {
     timeOnAir = timeOnAir+1;
  }

  if(debug_i==1){
    sprintf(STR,"timeOnAir         = %0d",timeOnAir);
    tb_print(STR);
  }

  if (timeOnAir > 5484) {
    sprintf(STR,"timeOnAir         = %0d",timeOnAir);
    tb_print(STR);
    sprintf(STR,"TXTIME exeeds limitation of 5.484ms!!!");
    tb_fire_error(STR);
  }
  // Return the value
  return (timeOnAir);
}

// get HE length for HE-TB frame from TXTIME
int GetHeTbLength (
  int lsigLength_i,
  int heLtfType_i,
  int giType_i,
  int numHeLtf_i,
  int ruType_i,
  int mcsIndex_i,
  int dcm_i,
  int stbc_i,
  int doppler_i,
  int fec_coding_i,
  int ldpc_extra_symbol_i,
  int pre_fec_padding_i,
  int midamble_periodicity_i,
  int bpe_dis_i,
  int debug_i)
{

  int    he_length             = 0;
  float  tSym                  = 0.0;
  float  tGiHeLtf              = 0.0;
  float  tHeLtf                = 0.0;
  float  tHeLtfSym             = 0.0;
  float  tMa                   = 0.0;
  int    nHeLtf                = 0;
  int    nDbps                 = 0;
  int    nTail                 = 0;
  int    nMa                   = 0;
  int    mSTBC                 = 0;
  int    aInit                 = 0;
  int    nDbpsShort            = 0;
  int    nCbpsShort            = 0;
  int    nSdShort              = 0;
  int    nBpscs                = 0;
  float  r                     = 0.0;
  int    nSym                  = 0;
  int    nSymInit              = 0;
  int    nDbpsInit             = 0;
  int    Tpe                   = 0;
  int    ldpc_npld             = 0;
  int    nbits_pack            = 0;

  // Max of 1024 character for debug printing through tb_print
  char   STR[1024];

  nTail = 6;

  if(pre_fec_padding_i == 0)
    pre_fec_padding_i = 4;

  // tGiHeLtf
  switch (giType_i){
    case 0:  tGiHeLtf = TGI1_DATA; break;
    case 1:  tGiHeLtf = TGI2_DATA; break;
    case 2:  tGiHeLtf = TGI4_DATA; break;
    default: tGiHeLtf = 0;
  }

  // tHeLtf
  switch (heLtfType_i){
    case 0:  tHeLtf = THE_LTF_1X; break;
    case 1:  tHeLtf = THE_LTF_2X; break;
    case 2:  tHeLtf = THE_LTF_4X; break;
    default: tHeLtf = 0;
  }

  // leg_length
  if(debug_i==1) {
    sprintf(STR,"lsigLength_i     = %0d", lsigLength_i);
    tb_print(STR);
    sprintf(STR,"giType_i = %0d", giType_i);
    tb_print(STR);
  }

  // tHeLtfSym
  if(debug_i==1) {
    sprintf(STR,"tGiHeLtf     = %0f, tHeLtf     = %0f", tGiHeLtf, tHeLtf);
    tb_print(STR);
  }
  tHeLtfSym = tHeLtf + tGiHeLtf;

  if(debug_i==1) {
    sprintf(STR,"tHeLtfSym      = %0f",tHeLtfSym);
    tb_print(STR);
  }

  // tSym
  tSym = TDFT_HE + tGiHeLtf;

  if(debug_i==1) {
    sprintf(STR,"tSym     = %0f",tSym);
    tb_print(STR);
  }

  nHeLtf = numHeLtf_i + 1;

  if(debug_i==1) {
    sprintf(STR,"nHeLtf     = %0d",nHeLtf);
    tb_print(STR);
  }

  // mSTBC
  mSTBC = stbc_i + 1;

  if(debug_i==1) {
    sprintf(STR,"mSTBC     = %0d",mSTBC);
    tb_print(STR);
  }

  nDbps = HE_NDBPS[ruType_i][0][mcsIndex_i & 0xF][dcm_i];

  if(debug_i==1) {
    sprintf(STR,"nDbps     = %0d",nDbps);
    tb_print(STR);
  }

  tMa = ((midamble_periodicity_i+1)*10)*tSym + numHeLtf_i*tHeLtfSym;

  if(doppler_i == 1){
    nMa = floor((((lsigLength_i + 5)/3.0)*4 - HE_TB_PRE_BASE - nHeLtf*tHeLtfSym - (bpe_dis_i+2)*tSym)/tMa);
  } else {
    nMa = 0;
  }

  if(debug_i==1) {
    sprintf(STR,"nMa     = %0d",nMa);
    tb_print(STR);
  }

  nSym = floor(((lsigLength_i+5)/3)*4 - HE_TB_PRE_BASE - nHeLtf*tHeLtfSym - nMa*nHeLtf*tHeLtfSym)/tSym - bpe_dis_i;

  Tpe = floor((((lsigLength_i + 5)/3)*4 - HE_TB_PRE_BASE - nHeLtf*tHeLtfSym - nSym*tSym - nMa*nHeLtf*tHeLtfSym)/4) * 4;

  // Chapter 27.3.11.5.5 Encoding process for an HE TB PPDU - IEEE 802.11ax 4.1 draft
  if (fec_coding_i == 0) {
    // BCC coding
    nSymInit = nSym;
    aInit = pre_fec_padding_i;
  } else {
    // LDPC coding
    if (ldpc_extra_symbol_i == 1){
      // LDPC Extra Symbol is set
      if (pre_fec_padding_i == 1){
        aInit = 4;
        nSymInit = nSym - mSTBC;
      } else {
        aInit = pre_fec_padding_i - 1;
        nSymInit = nSym;
      }
    } else {
      nSymInit = nSym;
      aInit = pre_fec_padding_i;
    }
  }

  if(debug_i==1) {
    sprintf(STR,"pre_fec_padding_i     = %0d",pre_fec_padding_i);
    tb_print(STR);
    sprintf(STR,"fec_coding_i     = %0d",fec_coding_i);
    tb_print(STR);
  }

  r           = R[mcsIndex_i];
  nBpscs      = NBPSCS[mcsIndex_i];
  nSdShort    = HE_NSDSHORT[dcm_i][ruType_i];
  nCbpsShort  = nSdShort*nBpscs;

  nDbpsShort = nCbpsShort*r;

  if(debug_i==1){
    sprintf(STR,"Nsd,short         = %0d (dcm_i = %0d, ruType_i = %0d)",nSdShort,dcm_i,ruType_i);
    tb_print(STR);
    sprintf(STR,"Ncbps,short       = %0d (nSdShort = %0d, nBpscs = %0d)",nCbpsShort,nSdShort,nBpscs);
    tb_print(STR);
    sprintf(STR,"Ndbps,short       = %0d (r = %0f)",nDbpsShort,r);
    tb_print(STR);
  }

  if (pre_fec_padding_i == 4) {
    nDbpsInit = nDbps;
  } else {
    nDbpsInit = aInit*nDbpsShort;
  }

  if(debug_i==1) {
    sprintf(STR,"nSym     = %0d",nSym);
    tb_print(STR);
    sprintf(STR,"nSymInit = %0d",nSymInit);
    tb_print(STR);
    sprintf(STR,"nDbps    = %0d",nDbps);
    tb_print(STR);
    sprintf(STR,"nDbpsInit= %0d",nDbpsInit);
    tb_print(STR);
    sprintf(STR,"aInit = %0d", aInit);
    tb_print(STR);
  }

  he_length = floor(((nSymInit - mSTBC)*nDbps + mSTBC*nDbpsInit - NSERVICE - nTail)/8);

  ldpc_npld = (nSymInit-mSTBC)*nDbps + nDbpsInit;
  nbits_pack = he_length*8 + 16;

  if(debug_i==1) {
    if (ldpc_npld < nbits_pack) {
      sprintf(STR,"Binary source exceeds the max num of bits");
      tb_print(STR);
    } else {
      sprintf(STR,"Everything is ok");
      tb_print(STR);
    }
  }

  if (debug_i == 1){
    sprintf(STR,"ldpcNpld = %d", ldpc_npld);
    tb_print(STR);
    sprintf(STR,"nbits_pack = %d", nbits_pack);
    tb_print(STR);
  }

  // Additional part for constraining the psdulength
  if (he_length >= 10000)
    while(he_length > 10000) {
      he_length = he_length - he_length*0.1;
    }
  else if (he_length > 105)
    he_length = he_length - 100;
  else if (he_length > 10)
    he_length = he_length - 4;

  if(debug_i==1) {
    sprintf(STR,"Tpe     = %0d",Tpe);
    tb_print(STR);
    sprintf(STR,"PSDU Length     = %0d",he_length);
    tb_print(STR);
  }

  return (he_length);
}

