//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $RCSmodulefile   :
// $Author          :
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 0.1
// $Date: 22-Dec-2008
// $State: Complete
// $Locker:
// ---------------------------------------------------------------------------
// Dependencies     :
// Description      : CCMP Class For TB
//
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

// IEEE
bit  [7:0] IEEE_CCMP_MAC_HEADER[24] = '{8'h08,8'h48,8'hc3,8'h2c,
                                        8'h0f,8'hd2,8'he1,8'h28,
                                        8'ha5,8'h7c,8'h50,8'h30,
                                        8'hf1,8'h84,8'h44,8'h08,
                                        8'hab,8'hae,8'ha5,8'hb8,
                                        8'hfc,8'hba,8'h80,8'h33};

bit  [7:0] IEEE_CCMP_HEADER[8]      = '{8'h0c,8'he7,8'h00,8'h20,8'h76,8'h97,8'h03,8'hb5};


bit  [7:0] IEEE_CCMP_PLAIN_DATA[20] = '{8'hf8,8'hba,8'h1a,8'h55,
                                        8'hd0,8'h2f,8'h85,8'hae,
                                        8'h96,8'h7b,8'hb6,8'h2f,
                                        8'hb6,8'hcd,8'ha8,8'heb,
                                        8'h7e,8'h78,8'ha0,8'h50};

bit  [7:0] IEEE_CCMP_KEY[16]        = '{8'hc9,8'h7c,8'h1f,8'h67,
                                        8'hce,8'h37,8'h11,8'h85,
                                        8'h51,8'h4a,8'h8a,8'h19,
                                        8'hf2,8'hbd,8'hd5,8'h2f};

bit  [7:0] IEEE_CCMP_PN[6]          = '{8'h0C,8'hE7,8'h76,8'h97,8'h03,8'hB5};

bit  [7:0] IEEE_CCMP_NONCE[13]      = '{8'h00,8'h50,8'h30,8'hf1,8'h84,8'h44,8'h08,8'hb5,8'h03,8'h97,8'h76,8'he7,8'h0c};

bit  [7:0] IEEE_CCMP_AAD[22]        = '{8'h08,8'h40,8'h0f,8'hd2,8'he1,8'h28,8'ha5,8'h7c,
                                        8'h50,8'h30,8'hf1,8'h84,8'h44,8'h08,8'hab,8'hae,
                                        8'ha5,8'hb8,8'hfc,8'hba,8'h00,8'h00};

bit  [7:0] IEEE_CCMP_MIC[8]         = '{8'h78,8'h45,8'hce,8'h0b,
                                        8'h16,8'hf9,8'h76,8'h23};

bit  [7:0] IEEE_CCMP_CIPHER[28]     = '{8'hf3,8'hd0,8'ha2,8'hfe,
                                        8'h9a,8'h3d,8'hbf,8'h23,
                                        8'h42,8'ha6,8'h43,8'he4,
                                        8'h32,8'h46,8'he8,8'h0c,
                                        8'h3c,8'h04,8'hd0,8'h19,
                                        8'h78,8'h45,8'hce,8'h0b,
                                        8'h16,8'hf9,8'h76,8'h23};

// IEEE CCMP-256
bit  [7:0] IEEE_CCMP256_MAC_HEADER[24] = '{8'h08,8'h48,8'hc3,8'h2c,8'h0f,8'hd2,8'he1,8'h28,
                                           8'ha5,8'h7c,8'h50,8'h30,8'hf1,8'h84,8'h44,8'h08,
                                           8'hab,8'hae,8'ha5,8'hb8,8'hfc,8'hba,8'h80,8'h33};
bit  [7:0] IEEE_CCMP256_HEADER[8]      = '{8'h0c,8'he7,8'h00,8'h20,8'h76,8'h97,8'h03,8'hb5};
bit  [7:0] IEEE_CCMP256_PLAIN_DATA[20] = '{8'hf8,8'hba,8'h1a,8'h55,8'hd0,8'h2f,8'h85,8'hae,
                                           8'h96,8'h7b,8'hb6,8'h2f,8'hb6,8'hcd,8'ha8,8'heb,
                                           8'h7e,8'h78,8'ha0,8'h50};
bit  [7:0] IEEE_CCMP256_KEY[32]        = '{8'hc9,8'h7c,8'h1f,8'h67,8'hce,8'h37,8'h11,8'h85,
                                           8'h51,8'h4a,8'h8a,8'h19,8'hf2,8'hbd,8'hd5,8'h2f,
                                           8'h00,8'h01,8'h02,8'h03,8'h04,8'h05,8'h06,8'h07,
                                           8'h08,8'h09,8'h0a,8'h0b,8'h0c,8'h0d,8'h0e,8'h0f};
bit  [7:0] IEEE_CCMP256_PN[6]          = '{8'h0c,8'he7,8'h76,8'h97,8'h03,8'hb5};
bit  [7:0] IEEE_CCMP256_NONCE[13]      = '{8'h00,8'h50,8'h30,8'hf1,8'h84,8'h44,8'h08,8'hb5,
                                           8'h03,8'h97,8'h76,8'he7,8'h0c};
bit  [7:0] IEEE_CCMP256_AAD[22]        = '{8'h08,8'h40,8'h0f,8'hd2,8'he1,8'h28,8'ha5,8'h7c,
                                           8'h50,8'h30,8'hf1,8'h84,8'h44,8'h08,8'hab,8'hae,
                                           8'ha5,8'hb8,8'hfc,8'hba,8'h00,8'h00};
bit  [7:0] IEEE_CCMP256_CIPHER[36]     = '{8'h6d,8'h15,8'h5d,8'h88,8'h32,8'h66,8'h82,8'h56,
                                           8'hd6,8'ha9,8'h2b,8'h78,8'he1,8'h1d,8'h8e,8'h54,
                                           8'h49,8'h5d,8'hd1,8'h74,8'h80,8'haa,8'h56,8'hc9,
                                           8'h49,8'h2e,8'h88,8'h2b,8'h97,8'h64,8'h2f,8'h80,
                                           8'hd5,8'h0f,8'he9,8'h7b};

// RCF
bit  [7:0] RFC_3610_AES_Key[16]  =  '{8'hC0,8'hC1,8'hC2,8'hC3,8'hC4,8'hC5,8'hC6,8'hC7,8'hC8,8'hC9,8'hCA,8'hCB,8'hCC,8'hCD,8'hCE,8'hCF};

bit  [7:0] RFC_3610_Nonce[13]    =  '{8'h00,8'h00,8'h00,8'h03,8'h02,8'h01,8'h00,8'hA0,8'hA1,8'hA2,8'hA3,8'hA4,8'hA5};

bit  [7:0] RFC_3610_Adata[8]     =  '{8'h00,8'h01,8'h02,8'h03,8'h04,8'h05,8'h06,8'h07};

bit  [7:0] RCF_3610_DATA[23]     =  '{8'h08,8'h09,8'h0A,8'h0B,8'h0C,8'h0D,8'h0E,8'h0F,
                                      8'h10,8'h11,8'h12,8'h13,8'h14,8'h15,8'h16,8'h17,
                                      8'h18,8'h19,8'h1A,8'h1B,8'h1C,8'h1D,8'h1E};

bit  [7:0] RCF_3610_MIC[8]       =  '{8'h2D,8'hC6,8'h97,8'hE4,8'h11,8'hCA,8'h83,8'hA8};

bit  [7:0] RCF_3610_CIPHER[31]   =  '{8'h58,8'h8C,8'h97,8'h9A,8'h61,8'hC6,8'h63,8'hD2,
                                      8'hF0,8'h66,8'hD0,8'hC2,8'hC0,8'hF9,8'h89,8'h80,
                                      8'h6D,8'h5F,8'h6B,8'h61,8'hDA,8'hC3,8'h84,8'h17,
                                      8'hE8,8'hD1,8'h2C,8'hFD,8'hF9,8'h26,8'hE0};

bit  [7:0] RFC_3610_Nonce_1[13]  =  '{8'h00 ,8'h00 ,8'h00 ,8'h04 ,8'h03 ,8'h02 ,8'h01 ,8'hA0 ,8'hA1 ,8'hA2 ,8'hA3 ,8'hA4 ,8'hA5};

bit  [7:0] RCF_3610_DATA_1[24]   =  '{8'h08 ,8'h09 ,8'h0A ,8'h0B ,8'h0C ,8'h0D ,8'h0E ,8'h0F,
                                      8'h10 ,8'h11 ,8'h12 ,8'h13 ,8'h14 ,8'h15 ,8'h16 ,8'h17,
                                      8'h18 ,8'h19 ,8'h1A ,8'h1B ,8'h1C ,8'h1D ,8'h1E ,8'h1F};

bit  [7:0] RCF_3610_MIC_1[8]     =  '{8'hF7 ,8'hB9 ,8'h05 ,8'h6A  ,8'h86 ,8'h92 ,8'h6C ,8'hF3};

bit  [7:0] RCF_3610_CIPHER_1[32] =  '{8'h72 ,8'hC9 ,8'h1A ,8'h36 ,8'hE1 ,8'h35 ,8'hF8 ,8'hCF,
                                      8'h29 ,8'h1C ,8'hA8 ,8'h94 ,8'h08 ,8'h5C ,8'h87 ,8'hE3,
                                      8'hCC ,8'h15 ,8'hC4 ,8'h39 ,8'hC9 ,8'hE4 ,8'h3A ,8'h3B,
                                      8'hA0 ,8'h91 ,8'hD5 ,8'h6E ,8'h10 ,8'h40 ,8'h09 ,8'h16};

class ccmp;
  int        M;
  int        L;

  aes aes_obj;

  bit  [7:0] PN[6];

  bit  [7:0] FCS[4];
  bit [31:0] FCS32;

  bit  [7:0] Key[];
  bit  [7:0] AAD[];
  bit  [7:0] CCMnonce[13];

  bit  [7:0] macheader[];
  bit  [7:0] plain_data[];
  bit  [7:0] enc_data[];
  bit  [7:0] full_enc_data[];
  bit  [7:0] dec_data[];

  bit  [7:0] S_0[16];
  bit  [7:0] S_i[16];
  bit  [7:0] A_i[];
  bit  [7:0] ccmp_data[];

  bit  [7:0] B_mic[];
  bit  [7:0] Mic_in[16];
  bit  [7:0] Mic_out[16];
  bit  [7:0] MIC[];
  bit  [7:0] MIC_dec[];
  bit  [7:0] CCMPHeader[8];

  bit        htMode       = 0;
  bit        sspCapable   = 0;

  int  ccmp_err_cnt = 0;

  function new();
    aes_obj = new();
  endfunction : new

  function void gen_plain_data(input bit [7:0] plain_data_in[] = {}, input bit valid = 1'b0,bit debug_en = 1'b0);
    int plain_data_length = $urandom_range(1024,1);
    if(valid && (plain_data_in.size() != 0))
    begin
      plain_data = new[plain_data_in.size()](plain_data_in);
    end
    else
    begin
      plain_data = new[plain_data_length];
      foreach(plain_data[i])
        plain_data[i] = $urandom_range(255,0);
    end

    if(debug_en)
    begin
      for(int i=0; i<plain_data.size();i=i+4)
      begin
        $display("%02h%02h%02h%02h",plain_data[i+3],plain_data[i+2],plain_data[i+1],plain_data[i]);
      end
    end

  endfunction : gen_plain_data

  function void gen_PN(input bit [7:0] PN_in[6] = {8'h0,8'h0,8'h0,8'h0,8'h0,8'h0}, input bit valid = 1'b0);
    foreach(PN[i])
    begin
      PN[i] = valid ? PN_in[i] : $urandom_range(255,0);
    end
  endfunction : gen_PN

  function void gen_CCMPHeader(input bit [7:0] CCMPHeader_in[8] = {8'h0,8'h0,8'h0,8'h0,8'h0,8'h0,8'h0,8'h0}, input bit valid = 1'b0);
    if(valid)
    begin
      foreach(CCMPHeader[i])
      begin
        CCMPHeader[i] = CCMPHeader_in[i];
      end
    end
    else
    begin
      CCMPHeader[0] = PN[0];
      CCMPHeader[1] = PN[1];
      CCMPHeader[2] = 8'h00;
      CCMPHeader[3] = {$urandom_range(3,0),1'b1,5'h00};
      CCMPHeader[4] = PN[2];
      CCMPHeader[5] = PN[3];
      CCMPHeader[6] = PN[4];
      CCMPHeader[7] = PN[5];
    end
  endfunction : gen_CCMPHeader

  function void gen_Key(input bit [7:0] Key_in[] = {}, input bit valid = 1'b0, input bit key_size = 1'b0);

    if (valid & Key_in.size==32 | ~valid & key_size==1)
    begin
       // CCMP-256
       M   = 16;
       L   = 2;
       Key = new[32];
    end
    else
    begin
       // CCMP-128
       M   = 8;
       L   = 2;
       Key = new[16];
    end

    foreach(Key[i])
    begin
      Key[i] = valid ? Key_in[i] : $urandom_range(255,0);
    end
  endfunction : gen_Key

  function void gen_macheader(input bit [7:0] macheader_in[] = {},input bit valid = 1'b0, input bit forced_add4 = 1'b0, input bit forced_qos = 1'b0);
    if(valid && macheader_in.size() >= 24)
    begin
      macheader =new[macheader_in.size()](macheader_in);
    end
    else
    begin
      macheader    = new[24];
      foreach(macheader[i])
      begin
        macheader[i] = $urandom_range(255,0);
      end
      macheader[0][2] = 1'b0; // Force Management or Data Frame

      macheader[1][6]   = 1'b1;
      if(forced_add4)
      begin
        macheader[1][1:0] = 2'b11;
        macheader[0][3:2] = 2'b10;
      end

      if(forced_qos)
      begin
        macheader[0][7]   = 1'b1;
        macheader[0][6:4] = $urandom_range(7,0);
        macheader[0][3:2] = 2'b10;
      end

      if(macheader[1][3:2] == 2'b00)
      begin
        macheader[1][7] = 1'b0;
        macheader[1][1:0] = 2'b01;
      end

      if(macheader[1][1:0] == 2'b11)
      begin
        macheader = new[macheader.size()+6](macheader);
        for(int i=macheader.size()-6; i<macheader.size(); i=i+1)
          macheader[i] = $urandom_range(255,0);
      end

      if(macheader[0][3:2]==2'b10 && macheader[0][7] && macheader[0][7:4]!=4'b1101)
      begin
        // QoS Field
        macheader = new[macheader.size()+2](macheader);
        for(int i=macheader.size()-2; i<macheader.size(); i=i+1)
          macheader[i] = $urandom_range(255,0);
      end
      macheader[1][6]   = 1'b1;
      //$display("gen_macheader : FC = %h %h", macheader[0],macheader[1]);
    end
  endfunction : gen_macheader

  function void gen_full_enc_data();
    bit [31:0] temp_crc;
    full_enc_data.delete();

    foreach(macheader[i])
    begin
      full_enc_data = new[full_enc_data.size() + 1](full_enc_data);
      full_enc_data[full_enc_data.size() - 1] = macheader[i];
    end

    foreach(CCMPHeader[i])
    begin
      full_enc_data = new[full_enc_data.size() + 1](full_enc_data);
      full_enc_data[full_enc_data.size() - 1] = CCMPHeader[i];
    end

    foreach(enc_data[i])
    begin
      full_enc_data = new[full_enc_data.size() + 1](full_enc_data);
      full_enc_data[full_enc_data.size() - 1] = enc_data[i];
    end

    FCS_calc(full_enc_data,temp_crc,1);
  endfunction : gen_full_enc_data

  // G(x) = x32 + x26 + x23 + x22 + x16 + x12 + x11 + x10 + x8 + x7 + x5 + x4 + x2 + x + 1
  function void FCS_calc(inout bit [7:0] data2crc[], output bit [31:0] crc_out,input bit append_crc = 1'b0);
    bit lfsr_in;
    bit [31:0] crc_result = 32'hFFFF_FFFF;

    for(int i = 0; i<data2crc.size(); i=i+1)
    begin
      for(int g=0; g<8; g=g+1)
      begin
        lfsr_in             =  crc_result[31] ^ data2crc[i][g];
        crc_result[31]      =  crc_result[30];
        crc_result[30]      =  crc_result[29];
        crc_result[29]      =  crc_result[28];
        crc_result[28]      =  crc_result[27];
        crc_result[27]      =  crc_result[26];
        crc_result[26]      =  crc_result[25] ^ lfsr_in;
        crc_result[25]      =  crc_result[24];
        crc_result[24]      =  crc_result[23];
        crc_result[23]      =  crc_result[22] ^ lfsr_in;
        crc_result[22]      =  crc_result[21] ^ lfsr_in;
        crc_result[21]      =  crc_result[20];
        crc_result[20]      =  crc_result[19];
        crc_result[19]      =  crc_result[18];
        crc_result[18]      =  crc_result[17];
        crc_result[17]      =  crc_result[16];
        crc_result[16]      =  crc_result[15] ^ lfsr_in;
        crc_result[15]      =  crc_result[14];
        crc_result[14]      =  crc_result[13];
        crc_result[13]      =  crc_result[12];
        crc_result[12]      =  crc_result[11] ^ lfsr_in;
        crc_result[11]      =  crc_result[10] ^ lfsr_in;
        crc_result[10]      =  crc_result[9]  ^ lfsr_in;
        crc_result[9]       =  crc_result[8];
        crc_result[8]       =  crc_result[7]  ^ lfsr_in;
        crc_result[7]       =  crc_result[6]  ^ lfsr_in;
        crc_result[6]       =  crc_result[5];
        crc_result[5]       =  crc_result[4]  ^ lfsr_in;
        crc_result[4]       =  crc_result[3]  ^ lfsr_in;
        crc_result[3]       =  crc_result[2];
        crc_result[2]       =  crc_result[1]  ^ lfsr_in;
        crc_result[1]       =  crc_result[0]  ^ lfsr_in;
        crc_result[0]       =                   lfsr_in;
      end
    end

    crc_result[31:0] = ~crc_result[31:0];

    for(int i=0;i<32;i=i+1)
    begin
      if(i<8)
        crc_out[7-i]     = crc_result[i];
      else if(i<16)
        crc_out[15-i+8]  = crc_result[i];
      else if(i<24)
        crc_out[23-i+16] = crc_result[i];
      else
        crc_out[31-i+24] = crc_result[i];
    end

    if(append_crc)
    begin
      data2crc = new[data2crc.size() + 4](data2crc);
      data2crc[data2crc.size()-4] = crc_out[31:24];
      data2crc[data2crc.size()-3] = crc_out[23:16];
      data2crc[data2crc.size()-2] = crc_out[15:8];
      data2crc[data2crc.size()-1] = crc_out[7:0];
    end
  endfunction : FCS_calc

  function void ConstructAAD();
    AAD.delete();
    AAD = new[22];

    if (macheader[0][3:2] == 2)
		// In case of Data Frame, mask bits 4, 5 & 6
      AAD[0] =  macheader[0] & 8'h8F; // bit 4 5 6   = 0;
    else
      AAD[0] =  macheader[0];

    AAD[1] = (macheader[1] | 8'h40) & 8'hC7; // bit 11 12 13 = 0; bit 14 = 1;
    for(int i=2; i<(6*3)+2; i=i+1)
      AAD[i] = macheader[i+2];
    AAD[20] = macheader[22] & 8'h0F; // bit 4-7 = 0
    AAD[21] = 8'h00; // bit 8-15 = 0

    if(macheader[1][1:0] == 2'b11)
    begin
      AAD = new[AAD.size()+6](AAD);
      AAD[AAD.size()-6] = macheader[24];
      AAD[AAD.size()-5] = macheader[25];
      AAD[AAD.size()-4] = macheader[26];
      AAD[AAD.size()-3] = macheader[27];
      AAD[AAD.size()-2] = macheader[28];
      AAD[AAD.size()-1] = macheader[29];
      //$display("AAD with ADDR4");
    end

    if(macheader[0][7] == 1'b1 && macheader[0][7:4] != 4'b1101 && macheader[0][3:2] == 2'b10)
    begin
      AAD = new[AAD.size()+2](AAD);
      AAD[AAD.size()-2] = macheader[macheader.size()-2] & 8'h0F;
      AAD[AAD.size()-1] = 8'h00;
      //$display("AAD with QoS");

      // HT Patching
      if(htMode)
      begin
        AAD[1][7] = 0; // FC 15 mask to 0 with QoS frame
        if(sspCapable)
        begin
          AAD[AAD.size()-2][7] = macheader[macheader.size()-2][7]; // QoS 7 unmask (A-MSDU)
        end
      end
    end

    //foreach(AAD[i])
    //begin
    //  $display("AAD[%0d]=%h",i,AAD[i]);
    //end

  endfunction : ConstructAAD

  function void ConstructCCMnonce();
    if (macheader[0][3:2] == 0)
    begin
      CCMnonce[0] =  8'h10;
      //$display("NONCE Mgt Frame =%h",macheader[0][3:2]);
    end
    else
    begin
      CCMnonce[0] =  (macheader[0][7] == 1'b1 && macheader[0][7:4] != 4'b1101) ? macheader[macheader.size()-2] & 8'h0F : 8'h00;
     // $display("NONCE Others Frame =%h",macheader[0][3:2]);
    end

    for(int i=1; i<7; i=i+1)
      CCMnonce[i] = macheader[i-1 + 4 + 6];

    for(int i=7; i<13; i=i+1)
       CCMnonce[i] = PN[5-(i-7)];

    //foreach(CCMnonce[i])
    //begin
    //  $display("CCMnonce[%0d]=%h",i,CCMnonce[i]);
    //end
  endfunction : ConstructCCMnonce

  function void CCMP_AES_MIC(input bit [7:0] Nonce_in[], input bit [7:0] Message_in[],input bit [7:0] Adata_in[] = {},bit  debug_en = 1'b0);
    // Create the new B-mic array
    B_mic.delete();

    // Create the new MAC array
    if (Key.size()==32)
      // CCMP-256
      MIC = new[16];
    else
      // CCMP-128
      MIC = new[8];

    // ET RCF 3610
    B_mic          = new[1];
    B_mic[0][7]    = 1'b0; // Rsvd
    B_mic[0][6]    = Adata_in.size() ? 1'b1 : 1'b0; // Adata
    B_mic[0][5:3]  = (M-2) / 2;
    B_mic[0][2:0]  = (L-1);

    // Add nonce
    for(int i=B_mic.size(), int j=B_mic.size(); i<(j + Nonce_in.size()); i=i+1)
    begin
      B_mic = new[B_mic.size() + 1](B_mic);
      B_mic[B_mic.size() - 1] = Nonce_in[i-j];
    end

    // Add message length order reverse
    B_mic = new[B_mic.size() + 2](B_mic);
    {B_mic[B_mic.size()-2],B_mic[B_mic.size()-1]} = Message_in.size();

    // Add Adata
    if(Adata_in.size())
    begin
      // length in B_mic reverse order
      if(Adata_in.size() < (2**16 - 2**8))
      begin
        B_mic = new[B_mic.size()+2](B_mic);
        {B_mic[B_mic.size()-2],B_mic[B_mic.size()-1]} = Adata_in.size();
      end
      else
      begin
        $display("length of Adata  > (2**16 - 2**8) is not supported");
      end

      // Append Adata
      for(int i=B_mic.size(), int j=B_mic.size(); i<(j + Adata_in.size()); i=i+1)
      begin
        B_mic = new[B_mic.size() + 1](B_mic);
        B_mic[B_mic.size() - 1] = Adata_in[i-j];
      end

      // Pad with 0 if the length is not modulo 16
      if(B_mic.size() % 16)
      begin
        do
        begin
          B_mic = new[B_mic.size() + 1](B_mic);
          B_mic[B_mic.size() - 1] = 8'h00;
        end while(B_mic.size() % 16);
      end
    end

    // Append the message
    if(Message_in.size())
    begin
      for(int i=B_mic.size(), int j=B_mic.size(); i<(j + Message_in.size()); i=i+1)
      begin
        B_mic = new[B_mic.size() + 1](B_mic);
        B_mic[B_mic.size() - 1] = Message_in[i-j];
      end

      // Pad with 0 if the length is not modulo 16
      if(B_mic.size() % 16)
      begin
        do
        begin
          B_mic = new[B_mic.size() + 1](B_mic);
          B_mic[B_mic.size() - 1] = 8'h00;
        end while(B_mic.size() % 16);
      end
    end

//    for(int i=0; i<B_mic.size(); i=i+16)
//    begin
//      $display(" %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h",B_mic[i+15],B_mic[i+14],B_mic[i+13],B_mic[i+12],B_mic[i+11],B_mic[i+10],B_mic[i+9],B_mic[i+8],B_mic[i+7],B_mic[i+6],B_mic[i+5],B_mic[i+4],B_mic[i+3],B_mic[i+2],B_mic[i+1],B_mic[i+0]);
//    end

    // Mic calculation
    aes_obj.gen_Key(Key,1'b1); // setup Key
    for(int i=0; i<B_mic.size(); i=i+16)
    begin
      if(i==0)
      begin
        foreach(Mic_in[j])
          Mic_in[j] = B_mic[i+j];
      end
      else
      begin
        foreach(Mic_in[j])
        begin
          Mic_in[j] = B_mic[i+j] ^ Mic_out[j];
        end
      end

      aes_obj.gen_plain_data(Mic_in,1'b1);
      aes_obj.aes_encrypt();
      foreach(Mic_out[j])
        Mic_out[j] = aes_obj.enc_data[j];

      if(debug_en)
      begin
        $display("Mic in  %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h",Mic_in[15],Mic_in[14],Mic_in[13],Mic_in[12],Mic_in[11],Mic_in[10],Mic_in[9],Mic_in[8],Mic_in[7],Mic_in[6],Mic_in[5],Mic_in[4],Mic_in[3],Mic_in[2],Mic_in[1],Mic_in[0]);
        $display("Mic out %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h %02h",Mic_out[15],Mic_out[14],Mic_out[13],Mic_out[12],Mic_out[11],Mic_out[10],Mic_out[9],Mic_out[8],Mic_out[7],Mic_out[6],Mic_out[5],Mic_out[4],Mic_out[3],Mic_out[2],Mic_out[1],Mic_out[0]);
      end
    end

    // Set MIC value
    foreach(MIC[j])
      MIC[j] = Mic_out[j];
  endfunction : CCMP_AES_MIC

  function void CCMP_AES_Enc( bit [7:0] Nonce_in[], bit [7:0] Message_in[],bit debug_en = 1'b0);
    // init the compute data
    enc_data = new[Message_in.size()];

    // Setup A_i[0]
    A_i = new[1];
    A_i[0][7]  = 1'b0;   // Rsvd
    A_i[0][6]  = 1'b0;   // Rsvd
    A_i[0][5:3]= 3'b000; // Set to 0
    A_i[0][2:0]= 3'b001; // L-1

    // Add nonce
    for(int i=A_i.size(), int j=A_i.size(); i<(j + Nonce_in.size()); i=i+1)
    begin
      A_i = new[A_i.size() + 1](A_i);
      A_i[A_i.size() - 1] = Nonce_in[i-j];
    end

    // Setup the first counter value
    A_i = new[A_i.size() + 2](A_i);
    {A_i[A_i.size()-2],A_i[A_i.size()-1]} = 16'h0000;

    // Calculate S_0
    aes_obj.gen_plain_data(A_i,1'b1);
    aes_obj.aes_encrypt();
    foreach(S_0[j])
      S_0[j] = aes_obj.enc_data[j];

    // encrypt the message
    for(int i=0; i<Message_in.size(); i=i+16)
    begin
      // update the A_i counter
      {A_i[A_i.size()-2],A_i[A_i.size()-1]} = (i/16)+1;

      // get the new S_i
      aes_obj.gen_plain_data(A_i,1'b1);
      aes_obj.aes_encrypt();
      foreach(S_i[j])
        S_i[j] = aes_obj.enc_data[j];

      foreach(S_i[j])
      begin
        if(i+j < Message_in.size())
          enc_data[i+j] = Message_in[i+j] ^ S_i[j];
        else
          break;
      end

      if(debug_en)
      begin
        array_sprint("aes data in : ",A_i,1);
        array_sprint("aes data out: ",S_i,1);
        array_sprint("enc data out: ",enc_data[i+:16],1);
        array_sprint("msg data in : ",Message_in[i+:16],1);
      end
    end

    // compute the encrypt MIC
    foreach(MIC[i])
    begin
      enc_data = new[enc_data.size()+1](enc_data);
      enc_data[enc_data.size()-1] = MIC[i] ^ S_0[i];
    end

  endfunction :CCMP_AES_Enc

  function void CCMP_AES_Dec( bit [7:0] Nonce_in[], bit [7:0] Message_in[], bit debug_en = 1'b0);
    // init the compute data : exclude encrypted MIC
    if (Key.size()==32)
    begin
       // CCMP-256
       dec_data = new[Message_in.size()-16];
       MIC_dec  = new[16];
    end
    else
    begin
       // CCMP-128
       dec_data = new[Message_in.size()-8];
       MIC_dec  = new[8];
    end

    // Setup A_i[0]
    A_i = new[1];
    A_i[0][7]  = 1'b0;   // Rsvd
    A_i[0][6]  = 1'b0;   // Rsvd
    A_i[0][5:3]= 3'b000; // Set to 0
    A_i[0][2:0]= 3'b001; // L-1

    // Add nonce
    for(int i=A_i.size(), int j=A_i.size(); i<(j + Nonce_in.size()); i=i+1)
    begin
      A_i = new[A_i.size() + 1](A_i);
      A_i[A_i.size() - 1] = Nonce_in[i-j];
    end

    // Setup the first counter value
    A_i = new[A_i.size() + 2](A_i);
    {A_i[A_i.size()-2],A_i[A_i.size()-1]} = 16'h0000;

    // Calculate S_0
    aes_obj.gen_plain_data(A_i,1'b1);
    aes_obj.aes_encrypt();
    foreach(S_0[j])
      S_0[j] = aes_obj.enc_data[j];

    // decrypt the message
    for(int i=0; i<(Message_in.size()-MIC_dec.size()); i=i+16)
    begin
      // update the A_i counter
      {A_i[A_i.size()-2],A_i[A_i.size()-1]} = (i/16)+1;

      // get the new S_i
      aes_obj.gen_plain_data(A_i,1'b1);
      aes_obj.aes_encrypt();
      foreach(S_i[j])
        S_i[j] = aes_obj.enc_data[j];

      foreach(S_i[j])
      begin
        if(i+j < (Message_in.size()-MIC_dec.size()))
          dec_data[i+j] = Message_in[i+j] ^ S_i[j];
        else
          break;
      end

      if(debug_en)
      begin
        array_sprint("aes data in : ",A_i,1);
        array_sprint("aes data out: ",S_i,1);
        array_sprint("dec data out: ",dec_data[i+:16],1);
        array_sprint("msg data in : ",Message_in[i+:16],1);
      end
    end

    // compute the decrypt MIC
    foreach(MIC_dec[i])
    begin
      MIC_dec[i] = Message_in[Message_in.size()-MIC_dec.size()+i] ^ S_0[i];
    end
  endfunction :CCMP_AES_Dec

  function void testCCMP();

    // Test against RCF 3610 test vector
    gen_Key(RFC_3610_AES_Key,1);

    CCMP_AES_MIC(RFC_3610_Nonce,RCF_3610_DATA,RFC_3610_Adata);

    foreach(MIC[i])
    begin
      if(MIC[i] != RCF_3610_MIC[i])
      begin
        $display("MIC[%02h] %02h differs from RCF_3610_MIC[%02h] %02h",i,MIC[i],i,RCF_3610_MIC[i]);
        ccmp_err_cnt = ccmp_err_cnt + 1;
      end
    end

    CCMP_AES_Enc(RFC_3610_Nonce,RCF_3610_DATA);

    if(enc_data.size() != $size(RCF_3610_CIPHER))
    begin
      $display("AES ecn_data size (%0d) differs from RCF_3610_CIPHER size (%0d)",enc_data.size(),$size(RCF_3610_CIPHER));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(enc_data[i])
      begin
        if(enc_data[i] != RCF_3610_CIPHER[i])
        begin
          $display("AES enc_data[%02h] %02h differs from RCF_3610_CIPHER[%02h] size %02h",i,enc_data[i],i,RCF_3610_CIPHER[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    CCMP_AES_Dec(RFC_3610_Nonce,RCF_3610_CIPHER);

    if(dec_data.size() != $size(RCF_3610_DATA))
    begin
      $display("AES dec_data size (%0d) differs from RCF_3610_DATA size (%0d)",dec_data.size(),$size(RCF_3610_DATA));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(dec_data[i])
      begin
        if(dec_data[i] != RCF_3610_DATA[i])
        begin
          $display("AES dec_data[%02h] %02h differs from RCF_3610_DATA[%02h] size %02h",i,dec_data[i],i,RCF_3610_DATA[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    if(MIC_dec.size() != $size(RCF_3610_MIC))
    begin
      $display("MIC_dec size (%0d) differs from RCF_3610_MIC size (%0d)",MIC_dec.size(),$size(RCF_3610_MIC));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(MIC_dec[i])
      begin
        if(MIC_dec[i] != RCF_3610_MIC[i])
        begin
          $display("MIC_dec[%02h] %02h differs from RCF_3610_MIC[%02h] %02h",i,MIC_dec[i],i,RCF_3610_MIC[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    gen_Key(RFC_3610_AES_Key,1);

    CCMP_AES_MIC(RFC_3610_Nonce_1,RCF_3610_DATA_1,RFC_3610_Adata);

    foreach(MIC[i])
    begin
      if(MIC[i] != RCF_3610_MIC_1[i])
      begin
        $display("MIC[%02h] %02h differs from RCF_3610_MIC_1[%02h] %02h",i,MIC[i],i,RCF_3610_MIC_1[i]);
        ccmp_err_cnt = ccmp_err_cnt + 1;
      end
    end

    CCMP_AES_Enc(RFC_3610_Nonce_1,RCF_3610_DATA_1);

    if(enc_data.size() != $size(RCF_3610_CIPHER_1))
    begin
      $display("AES ecn_data size (%0d) differs from RCF_3610_CIPHER_1 size (%0d)",enc_data.size(),$size(RCF_3610_CIPHER_1));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(enc_data[i])
      begin
        if(enc_data[i] != RCF_3610_CIPHER_1[i])
        begin
          $display("AES enc_data[%02h] %02h differs from RCF_3610_CIPHER_1[%02h] size %02h",i,enc_data[i],i,RCF_3610_CIPHER_1[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    // Test against IEEE CCMP-128 test vector
    gen_PN(IEEE_CCMP_PN,1);
    gen_macheader(IEEE_CCMP_MAC_HEADER,1);

    ConstructAAD();

    if(AAD.size() != $size(IEEE_CCMP_AAD))
    begin
      $display("AAD.size() (%0d) differs from IEEE_CCMP_AAD size (%0d)",AAD.size(),$size(IEEE_CCMP_AAD));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(AAD[i])
      begin
        if(AAD[i] != IEEE_CCMP_AAD[i])
        begin
          $display("AAD[%02h] %02h differs from IEEE_CCMP_AAD[%02h] size %02h",i,AAD[i],i,IEEE_CCMP_AAD[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    ConstructCCMnonce();

    foreach(CCMnonce[i])
    begin
      if(CCMnonce[i] != IEEE_CCMP_NONCE[i])
      begin
        $display("CCMnonce[%02h] %02h differs from IEEE_CCMP_NONCE[%02h] size %02h",i,CCMnonce[i],i,IEEE_CCMP_NONCE[i]);
        ccmp_err_cnt = ccmp_err_cnt + 1;
      end
    end

    gen_Key(IEEE_CCMP_KEY,1);
    gen_plain_data(IEEE_CCMP_PLAIN_DATA,1);

    CCMP_AES_MIC(CCMnonce,plain_data,AAD);
    CCMP_AES_Enc(CCMnonce,plain_data,1);

    if(enc_data.size() != $size(enc_data))
    begin
      $display("enc_data.size() (%0d) differs from enc_data size (%0d)",enc_data.size(),$size(enc_data));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(enc_data[i])
      begin
        if(enc_data[i] != IEEE_CCMP_CIPHER[i])
        begin
          $display("enc_data[%02h] %02h differs from IEEE_CCMP_CIPHER[%02h] size %02h",i,enc_data[i],i,IEEE_CCMP_CIPHER[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    CCMP_AES_Dec(CCMnonce,IEEE_CCMP_CIPHER,1);

    if(dec_data.size() != $size(IEEE_CCMP_PLAIN_DATA))
    begin
      $display("AES dec_data size (%0d) differs from IEEE_CCMP_PLAIN_DATA size (%0d)",dec_data.size(),$size(IEEE_CCMP_PLAIN_DATA));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(dec_data[i])
      begin
        if(dec_data[i] != IEEE_CCMP_PLAIN_DATA[i])
        begin
          $display("AES dec_data[%02h] %02h differs from IEEE_CCMP_PLAIN_DATA[%02h] size %02h",i,dec_data[i],i,IEEE_CCMP_PLAIN_DATA[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    if(MIC_dec.size() != MIC.size())
    begin
      $display("MIC_dec size (%0d) differs from MIC size (%0d)",MIC_dec.size(),MIC.size());
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(MIC_dec[i])
      begin
        if(MIC_dec[i] != MIC[i])
        begin
          $display("MIC_dec[%02h] %02h differs from MIC[%02h] %02h",i,MIC_dec[i],i,MIC[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    // Test against IEEE CCMP-256 test vector
    gen_PN(IEEE_CCMP256_PN,1);
    gen_macheader(IEEE_CCMP256_MAC_HEADER,1);

    ConstructAAD();

    if(AAD.size() != $size(IEEE_CCMP256_AAD))
    begin
      $display("AAD.size() (%0d) differs from IEEE_CCMP256_AAD size (%0d)",AAD.size(),$size(IEEE_CCMP256_AAD));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(AAD[i])
      begin
        if(AAD[i] != IEEE_CCMP256_AAD[i])
        begin
          $display("AAD[%02h] %02h differs from IEEE_CCMP256_AAD[%02h] size %02h",i,AAD[i],i,IEEE_CCMP256_AAD[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    ConstructCCMnonce();

    foreach(CCMnonce[i])
    begin
      if(CCMnonce[i] != IEEE_CCMP256_NONCE[i])
      begin
        $display("CCMnonce[%02h] %02h differs from IEEE_CCMP256_NONCE[%02h] size %02h",i,CCMnonce[i],i,IEEE_CCMP256_NONCE[i]);
        ccmp_err_cnt = ccmp_err_cnt + 1;
      end
    end

    gen_Key(IEEE_CCMP256_KEY,1);
    gen_plain_data(IEEE_CCMP256_PLAIN_DATA,1);

    CCMP_AES_MIC(CCMnonce,plain_data,AAD);
    CCMP_AES_Enc(CCMnonce,plain_data,1);

    if(enc_data.size() != $size(enc_data))
    begin
      $display("enc_data.size() (%0d) differs from enc_data size (%0d)",enc_data.size(),$size(enc_data));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(enc_data[i])
      begin
        if(enc_data[i] != IEEE_CCMP256_CIPHER[i])
        begin
          $display("enc_data[%02h] %02h differs from IEEE_CCMP256_CIPHER[%02h] size %02h",i,enc_data[i],i,IEEE_CCMP256_CIPHER[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    CCMP_AES_Dec(CCMnonce,IEEE_CCMP256_CIPHER,1);

    if(dec_data.size() != $size(IEEE_CCMP256_PLAIN_DATA))
    begin
      $display("AES dec_data size (%0d) differs from IEEE_CCMP256_PLAIN_DATA size (%0d)",dec_data.size(),$size(IEEE_CCMP256_PLAIN_DATA));
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(dec_data[i])
      begin
        if(dec_data[i] != IEEE_CCMP256_PLAIN_DATA[i])
        begin
          $display("AES dec_data[%02h] %02h differs from IEEE_CCMP256_PLAIN_DATA[%02h] size %02h",i,dec_data[i],i,IEEE_CCMP256_PLAIN_DATA[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

    if(MIC_dec.size() != MIC.size())
    begin
      $display("MIC_dec size (%0d) differs from MIC size (%0d)",MIC_dec.size(),MIC.size());
      ccmp_err_cnt = ccmp_err_cnt + 1;
    end
    else
    begin
      foreach(MIC_dec[i])
      begin
        if(MIC_dec[i] != MIC[i])
        begin
          $display("MIC_dec[%02h] %02h differs from MIC[%02h] %02h",i,MIC_dec[i],i,MIC[i]);
          ccmp_err_cnt = ccmp_err_cnt + 1;
        end
      end
    end

  endfunction : testCCMP

  function void array_sprint(input string msg = "",input [7:0] array_in[] = {},input bit revert = 1'b0);
    string s_out;

    foreach(array_in[i])
    begin
      if((i%128 == 0) && (i > 0))
        $swrite(s_out,"%s\n",s_out);
      if(revert)
        $swrite(s_out,"%02h%s",array_in[i],s_out);
      else
        $swrite(s_out,"%s%02h",s_out,array_in[i]);
    end

    $display("%s%s",msg,s_out);
  endfunction : array_sprint
endclass : ccmp
