//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef AHB_MASTER_DRIVER_SV
`define AHB_MASTER_DRIVER_SV


class ahb_master_driver extends uvm_driver #(ahb_master_seq_item);

  `uvm_component_utils(ahb_master_driver)

  virtual ahb_master_if vif;
  ahb_master_config        cfg;
  int                      burst_counter;
  bit                      burst_completed;
  bit                      transfer_completed;
  ahb_transfer_response_e  hresp;

  function new (string name = "ahb_master_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual ahb_master_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    fork
      get_and_drive();
      observe_reset();
      /* add other tasks here */
    join_none;
  endtask : run_phase

  task initialize();
    vif.haddr  <= 'h0;
    vif.hwdata <= 'h0;
    vif.htrans <= 'h0;
    vif.hsize  <= 'h0;
    vif.hburst <= 'h0;
    vif.hprot  <= 'h0;
    vif.hwrite <= 'h0;
  endtask : initialize

  task observe_reset();
    forever begin
       @(negedge vif.hresetn);
      `uvm_info(get_type_name(), "Reset observed on HRESETn.", UVM_DEBUG);
      initialize();
    end
  endtask : observe_reset

  task get_and_drive();
    initialize();

    forever begin
      seq_item_port.get_next_item(req);
      burst_counter      =   0;
      burst_completed    = 'b0;
      transfer_completed = 'b0;

      `uvm_info(get_type_name(), "Start of an AHB bus cycle detected.", UVM_DEBUG)
      `uvm_info(get_type_name(), $sformatf("Transaction received :\n%s", req.sprint()), UVM_DEBUG)

        if ((req.ahb_trans == IDLE) && (cfg.ahb_use_htrans_idle == 0)) begin
          `uvm_info(get_type_name(), "Got IDLE transaction type, but configuration prevents usage!", UVM_LOW)
        end else begin
          while (transfer_completed == 'b0) begin
            @(negedge vif.hclk); // the AHB master drives the address and control signals onto
                                 // the bus after the rising edge of HCLK
            vif.haddr  <= req.ahb_addr;
            vif.htrans <= req.ahb_trans;
            vif.hburst <= req.ahb_burst;
            vif.hsize  <= req.ahb_size;
            vif.hprot  <= req.ahb_prot;
            vif.hwrite <= (req.ahb_access == WRITE) ? 1'b1 : 1'b0;

            while (burst_completed != 'b1) begin
              // the AHB slave samples these signals on the rising edge. Next change
              // is again, after the rising edge -> we wait for falling edge
              @(negedge vif.hclk);

              if ((req.ahb_burst == SINGLE) || (burst_counter == (req.ahb_burst_size-1))) begin
                vif.haddr  <= 'h0;
                vif.htrans <= 'h0;
                vif.hburst <= 'h0;
                vif.hsize  <= 'h0;
                vif.hprot  <= 'h0;
                vif.hwrite <= 'h0;
              end else begin
                vif.haddr  <= calculate_address(req.ahb_addr, req.ahb_burst, burst_counter+1, req.ahb_size);
                vif.htrans <= ahb_transfer_type_e'(SEQ);
              end

              if (req.ahb_access == WRITE) begin
                // if the access is a WRITE access, HWDATA has to be set regardless
                // of the state of HREADY
                vif.hwdata <= req.ahb_data[burst_counter];
                `uvm_info(get_type_name(), $sformatf("DEBUG: ahb_data[%1d]=0x%x",burst_counter, req.ahb_data[burst_counter]), UVM_DEBUG)
              end

              // the master samples slave responses on rising edge
              @(posedge vif.hclk);

              while (vif.hready != 1'b1)
                @(posedge vif.hclk);

              hresp = ahb_transfer_response_e'(vif.hresp);

              if (hresp == OKAY) begin
                burst_counter++;

                if (req.ahb_access == READ) begin
                  req.ahb_data = vif.hrdata;
                end

                `uvm_info(get_type_name(), $sformatf("DEBUG: HRESP = %s, burst_counter = %4d, ahb_burst_size = %4d", hresp.name(), burst_counter, req.ahb_burst_size), UVM_DEBUG)

                if (burst_counter == req.ahb_burst_size) begin
                  transfer_completed = 'b1;
                  burst_completed = 'b1;
                end

              end else if (hresp == ERROR) begin
                `uvm_info(get_type_name(), $sformatf("Got HRESP = ERROR, aborting transaction immediatelly..."), UVM_LOW)
                transfer_completed = 'b1;
              end else if (hresp == RETRY) begin
                `uvm_info(get_type_name(), $sformatf("Got HRESP = RETRY, retrying transaction immediatelly..."), UVM_LOW)
              end else if (hresp == SPLIT) begin
                int split_delay;
                split_delay = $urandom_range (`SPLIT_DELAY_MIN , `SPLIT_DELAY_MAX);
                `uvm_info(get_type_name(), $sformatf("Got HRESP = SPLIT, releasing AHB bus and retrying in %d clock cycles.", split_delay), UVM_LOW)
                initialize();
                while (split_delay > 0) begin
                  @(posedge vif.hclk);
                  split_delay--;
                end
              end

            end // while (burst_completed)
          end // while(transfer_completed)
         end //ahb_htrans == IiDLE
        `uvm_info(get_type_name(), "AHB bus cycle finished.", UVM_DEBUG)
      initialize();
      seq_item_port.item_done();
    end // forever
  endtask : get_and_drive

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : ahb_master_driver

`endif
