//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef AXI_SLAVE_IF_SV
`define AXI_SLAVE_IF_SV

interface axi_slave_if (
                  input bit aclk,
                  input bit aresetn
                       );

  import uvm_pkg::*;
  `include "uvm_macros.svh"

  // Check flags
  bit   has_checks = 1;
  bit   has_coverage = 1;

  //
  logic[`AXI_SLAVE_ID_WIDTH-1:0]     awid;
  logic[`AXI_SLAVE_ADDR_WIDTH-1:0]   awaddr;
  logic[7:0]                         awlen;
  logic[2:0]                         awsize;
  logic[1:0]                         awburst;
  logic[`AXI_SLAVE_QOS_WIDTH-1:0]    awqos;
  logic[`AXI_SLAVE_USER_WIDTH-1:0]   awuser;
  logic                              awvalid;
  logic                              awready;

  //
  logic[`AXI_SLAVE_ID_WIDTH-1:0]     wid;          // Not supported in AXI 4
  logic[`AXI_SLAVE_DATA_WIDTH-1:0]   wdata;
  logic[`AXI_SLAVE_DATA_WIDTH/8-1:0] wstrb;
  logic                              wlast;
  logic[`AXI_SLAVE_USER_WIDTH-1:0]   wuser;
  logic                              wvalid;
  logic                              wready;

  //
  logic[`AXI_SLAVE_ID_WIDTH-1:0]     bid;
  logic[1:0]                         bresp;
  logic[`AXI_SLAVE_USER_WIDTH-1:0]   buser;
  logic                              bvalid;
  logic                              bready;

  //
  //
  logic[`AXI_SLAVE_ID_WIDTH-1:0]     arid;
  logic[`AXI_SLAVE_ADDR_WIDTH-1:0]   araddr;
  logic[7:0]                         arlen;
  logic[2:0]                         arsize;
  logic[1:0]                         arburst;
  logic[`AXI_SLAVE_QOS_WIDTH-1:0]    arqos;
  logic[`AXI_SLAVE_USER_WIDTH-1:0]   aruser;
  logic                              arvalid;
  logic                              arready;

  //
  logic[`AXI_SLAVE_ID_WIDTH-1:0]     rid;
  logic[`AXI_SLAVE_DATA_WIDTH-1:0]   rdata;
  logic[1:0]                         rresp;
  logic                              rlast;
  logic[`AXI_SLAVE_USER_WIDTH-1:0]   ruser;
  logic                              rvalid;
  logic                              rready;

  /***************************/
  /* ASSERTIONS              */
  /***************************/

 // Cross 4Kb boundary
 bit[`AXI_SLAVE_ADDR_WIDTH-1:0] aw_addr_final;
 bit[`AXI_SLAVE_ADDR_WIDTH-1:0] ar_addr_final;

 bit[6:0] aw_align_msk;
 bit[6:0] ar_align_msk;

localparam Ax_BURST_INCR = 2'b01;
localparam Ax_BURST_WRAP = 2'b10;

localparam Ax_BYTE_1   = 3'b000,
           Ax_BYTE_2   = 3'b001,
           Ax_BYTE_4   = 3'b010,
           Ax_BYTE_8   = 3'b011,
           Ax_BYTE_16  = 3'b100,
           Ax_BYTE_32  = 3'b101,
           Ax_BYTE_64  = 3'b110,
           Ax_BYTE_128 = 3'b111;

 always@*
  begin
   aw_addr_final = awaddr + (awlen << awsize);
  end

 always@*
  begin
   ar_addr_final = araddr + (awlen << arsize);
  end

// INCR
property AW_ADR_4K_BOUNDARY_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awburst,awaddr})) & awvalid & (awburst == Ax_BURST_INCR) |->  awaddr[`AXI_SLAVE_ADDR_WIDTH-1:12] == aw_addr_final[`AXI_SLAVE_ADDR_WIDTH-1:12];
endproperty // AW_ADR_4K_BOUNDARY_P

property AR_ADR_4K_BOUNDARY_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arburst,araddr})) & arvalid & (arburst == Ax_BURST_INCR) |->  araddr[`AXI_SLAVE_ADDR_WIDTH-1:12] == ar_addr_final[`AXI_SLAVE_ADDR_WIDTH-1:12];
endproperty // AR_ADR_4K_BOUNDARY_P

//assert property(AW_ADR_4K_BOUNDARY_P) else `uvm_error("AXI UVC", "4Kb boundary crossing WRITE")
//assert property(AR_ADR_4K_BOUNDARY_P) else `uvm_error("AXI UVC", "4Kb boundary crossing READ")


// Aling mask

 always@*
  begin
   if(awvalid) begin
    case(awsize)
      Ax_BYTE_1   : aw_align_msk = 7'b1111111;
      Ax_BYTE_2   : aw_align_msk = 7'b1111110;
      Ax_BYTE_4   : aw_align_msk = 7'b1111100;
      Ax_BYTE_8   : aw_align_msk = 7'b1111000;
      Ax_BYTE_16  : aw_align_msk = 7'b1110000;
      Ax_BYTE_32  : aw_align_msk = 7'b1100000;
      Ax_BYTE_64  : aw_align_msk = 7'b1000000;
      Ax_BYTE_128 : aw_align_msk = 7'b0000000;
      default     : aw_align_msk = 7'b1111111;
    endcase
   end
   else begin
    aw_align_msk = '1;
   end
   if(arvalid) begin
    case(arsize)
      Ax_BYTE_1   : ar_align_msk = 7'b1111111;
      Ax_BYTE_2   : ar_align_msk = 7'b1111110;
      Ax_BYTE_4   : ar_align_msk = 7'b1111100;
      Ax_BYTE_8   : ar_align_msk = 7'b1111000;
      Ax_BYTE_16  : ar_align_msk = 7'b1110000;
      Ax_BYTE_32  : ar_align_msk = 7'b1100000;
      Ax_BYTE_64  : ar_align_msk = 7'b1000000;
      Ax_BYTE_128 : ar_align_msk = 7'b0000000;
      default     : ar_align_msk = 7'b1111111;
    endcase
   end
   else begin
    ar_align_msk = '1;
   end
  end


// For Wrapping bursts
property AR_ADR_ALING_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arburst,araddr})) & arvalid & (arburst == Ax_BURST_WRAP) |->  (araddr[6:0] & ar_align_msk) == araddr[6:0];
endproperty // AR_ADR_ALING_P

property AW_ADR_ALING_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awburst,awaddr})) & awvalid & (awburst == Ax_BURST_WRAP) |->  (awaddr[6:0] & aw_align_msk) == awaddr[6:0];
endproperty // AW_ADR_ALING_P

assert property(AR_ADR_ALING_P) else `uvm_error("AXI UVC", "Non-aling WRAP access WRITE")
assert property(AW_ADR_ALING_P) else `uvm_error("AXI UVC", "Non-aling WRAP access READ")


// Correct burst for both READ and WRITE
property AW_BURST_CORRECT_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awburst})) & awvalid  |->  awburst != 2'b11 ;
endproperty // AW_BURST_CORRECT_P

property AR_BURST_CORRECT_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arburst})) & arvalid  |->  arburst != 2'b11 ;
endproperty // AR_BURST_CORRECT_P

assert property(AW_BURST_CORRECT_P) else `uvm_error("AXI UVC", "Incorrect value of AWBURST for WRITE transaction")
assert property(AR_BURST_CORRECT_P) else `uvm_error("AXI UVC", "Incorrect value of ARBURST for READ transaction")


// Stable for both READ and WRITE
property AW_WID_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awready,awid})) & awvalid & !awready |-> ##1 $stable(awid) ;
endproperty // AW_WID_STABLE_P

property AW_AWADDR_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awready,awaddr})) & awvalid & !awready |-> ##1 $stable(awaddr) ;
endproperty // AW_AWADDR_STABLE_P

property AW_AWLEN_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awready,awlen})) & awvalid & !awready |-> ##1 $stable(awlen) ;
endproperty // AW_AWLEN_STABLE_P

property AW_AWSIZE_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awready,awsize})) & awvalid & !awready |-> ##1 $stable(awsize) ;
endproperty // AW_AWSIZE_STABLE_P

property AW_AWBURST_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awready,awburst})) & awvalid & !awready |-> ##1 $stable(awburst) ;
endproperty // AW_AWBURST_STABLE_P

property AW_AWUSER_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awready,awuser})) & awvalid & !awready |-> ##1 $stable(awuser) ;
endproperty // AW_AWUSER_STABLE_P

property AW_AWQOS_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({awvalid,awready,awqos})) & awvalid & !awready |-> ##1 $stable(awqos) ;
endproperty // AW_AWQOS_STABLE_P

assert property(AW_WID_STABLE_P)      else `uvm_error("AXI UVC", "Signal AWID is not stable.")
assert property(AW_AWADDR_STABLE_P)   else `uvm_error("AXI UVC", "Signal AWADDR is not stable.")
assert property(AW_AWLEN_STABLE_P)    else `uvm_error("AXI UVC", "Signal AWLEN is not stable.")
assert property(AW_AWSIZE_STABLE_P)   else `uvm_error("AXI UVC", "Signal AWSIZE is not stable.")
assert property(AW_AWBURST_STABLE_P)  else `uvm_error("AXI UVC", "Signal AWBURST is not stable.")
assert property(AW_AWUSER_STABLE_P)   else `uvm_error("AXI UVC", "Signal AWUSER is not stable.")
assert property(AW_AWQOS_STABLE_P)    else `uvm_error("AXI UVC", "Signal AWQOS is not stable.")

// $$$$

property AR_WID_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arready,arid})) & arvalid & !arready |-> ##1 $stable(arid) ;
endproperty // AR_WID_STABLE_P

property AR_ARADDR_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arready,araddr})) & arvalid & !arready |-> ##1 $stable(araddr) ;
endproperty // AR_ARADDR_STABLE_P

property AR_ARLEN_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arready,arlen})) & arvalid & !arready |-> ##1 $stable(arlen) ;
endproperty // AR_ARLEN_STABLE_P

property AR_ARSIZE_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arready,arsize})) & arvalid & !arready |-> ##1 $stable(arsize) ;
endproperty // AR_ARSIZE_STABLE_P

property AR_ARBURST_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arready,arburst})) & arvalid & !arready |-> ##1 $stable(arburst) ;
endproperty // AR_ARBURST_STABLE_P

property AR_ARUSER_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arready,aruser})) & arvalid & !arready |-> ##1 $stable(aruser) ;
endproperty // AR_ARUSER_STABLE_P

property AR_ARQOS_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({arvalid,arready,arqos})) & arvalid & !arready |-> ##1 $stable(arqos) ;
endproperty // AR_ARQOS_STABLE_P


assert property(AR_WID_STABLE_P)      else `uvm_error("AXI UVC", "Signal ARID is not stable.")
assert property(AR_ARADDR_STABLE_P)   else `uvm_error("AXI UVC", "Signal ARADDR is not stable.")
assert property(AR_ARLEN_STABLE_P)    else `uvm_error("AXI UVC", "Signal ARLEN is not stable.")
assert property(AR_ARSIZE_STABLE_P)   else `uvm_error("AXI UVC", "Signal ARSIZE is not stable.")
assert property(AR_ARBURST_STABLE_P)  else `uvm_error("AXI UVC", "Signal ARBURST is not stable.")
assert property(AR_ARUSER_STABLE_P)   else `uvm_error("AXI UVC", "Signal ARUSER is not stable.")
assert property(AR_ARQOS_STABLE_P)    else `uvm_error("AXI UVC", "Signal ARQOS is not stable.")

// Write response channel
property W_BID_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({bvalid,bready,bid})) & bvalid & !bready |-> ##1 $stable(bid) ;
endproperty // W_BID_STABLE_P

property W_BRESP_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({bvalid,bready,bresp})) & bvalid & !bready |-> ##1 $stable(bresp) ;
endproperty // W_BRESP_STABLE_P

assert property(W_BID_STABLE_P)  else `uvm_error("AXI UVC", "Signal BID is not stable.")
assert property(W_BRESP_STABLE_P)   else `uvm_error("AXI UVC", "Signal BRESP is not stable.")


property W_BVALID_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({bvalid,bready})) & bvalid & !bready |-> ##1 $stable(bvalid) ;
endproperty // W_BVALID_STABLE_P

assert property(W_BVALID_STABLE_P)   else `uvm_error("AXI UVC", "Unexpected deassertion of BVALID.")


bit[`AXI_SLAVE_DATA_WIDTH-1:0]   wr_data_mask;

 always@*
  begin
   for(int unsigned i = 0; i < `AXI_SLAVE_DATA_WIDTH/8; i++) begin
    for(int unsigned j = i * 8; j < (i * 8) + 7; j++) begin
     wr_data_mask[j] =  wstrb[i];
    end
   end
  end

property AW_DATA_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({wvalid,wready,wdata})) & wvalid & !wready |-> ##1 $stable(wdata) ;
endproperty // AW_DATA_STABLE_P

property AW_WLAST_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({wvalid,wready,wlast})) & wvalid & !wready |-> ##1 $stable(wlast) ;
endproperty // AW_WLAST_STABLE_P

property AW_WSTRB_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({wvalid,wready,wstrb})) & wvalid & !wready |-> ##1 $stable(wstrb) ;
endproperty // AW_WSTRB_STABLE_P

assert property(AW_DATA_STABLE_P)   else `uvm_error("AXI UVC", "Signal WDATA is not stable when wvalid is asserted and wready is deasserted.")
assert property(AW_WLAST_STABLE_P)   else `uvm_error("AXI UVC", "Signal WLAST is not stable when wvalid is asserted and wready is deasserted..")
assert property(AW_WSTRB_STABLE_P)   else `uvm_error("AXI UVC", "Signal WSTRB is not stable when wvalid is asserted and wready is deasserted..")

// Read channel
property AR_DATA_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({rvalid,rready,rdata})) & rvalid & !rready |-> ##1 $stable(rdata) ;
endproperty // AR_DATA_STABLE_P

property AR_RLAST_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({rvalid,rready,rlast})) & rvalid & !rready |-> ##1 $stable(rlast) ;
endproperty // AR_RLAST_STABLE_P

property AR_RRESP_STABLE_P;
 @(posedge aclk iff (aresetn === 1'b1 && has_checks == 1'b1))
 !($isunknown({rvalid,rready,rresp})) & rvalid & !rready |-> ##1 $stable(rresp) ;
endproperty // AR_RRESP_STABLE_P

assert property(AR_DATA_STABLE_P)   else `uvm_error("AXI UVC", "Signal RDATA is not stable when wvalid is asserted and wready is deasserted.")
assert property(AR_RLAST_STABLE_P)  else `uvm_error("AXI UVC", "Signal RLAST is not stable when wvalid is asserted and wready is deasserted..")
assert property(AR_RRESP_STABLE_P)  else `uvm_error("AXI UVC", "Signal RRESP is not stable when wvalid is asserted and wready is deasserted..")


  /***************************/
  /* COVERAGE                */
  /***************************/

endinterface : axi_slave_if

`endif
