//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef AXI_SLAVE_READ_DRIVER_SV
`define AXI_SLAVE_READ_DRIVER_SV


class axi_slave_read_driver extends uvm_driver #(axi_slave_seq_item);

  `uvm_component_utils(axi_slave_read_driver)

  virtual axi_slave_if vif;
  axi_slave_config     cfg;

  axi_slave_seq_item   read_data_ch_q[$];

  // get_and_drive
   int unsigned       rdata_cnt;
   bit                rdata_ch_is_active;
   int unsigned       rready_delay_cnt;
   axi_slave_seq_item req_cloned;
   axi_slave_seq_item read_item;
   int unsigned       read_delay_cnt;

  extern function new (string name = "axi_slave_read_driver", uvm_component parent = null);
  extern virtual function void build_phase(uvm_phase phase);
  extern virtual function void connect_phase(uvm_phase phase);
  extern virtual task run_phase(uvm_phase phase);
  extern virtual function void shuffle_data_queue();
  extern virtual task initialize();
  extern virtual task observe_reset();
  extern virtual task address_channel();
  extern virtual task data_channel();
  extern virtual task get_and_drive();
  extern virtual function void report_phase(uvm_phase phase);
endclass : axi_slave_read_driver

  function axi_slave_read_driver::new (string name = "axi_slave_read_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  function void axi_slave_read_driver::build_phase(uvm_phase phase);
    super.build_phase(phase);
    if(!uvm_config_db#(virtual axi_slave_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");
  endfunction : build_phase

  function void axi_slave_read_driver::connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  // "Randomize" queue of data responses
  function void axi_slave_read_driver::shuffle_data_queue();
    axi_slave_seq_item              res_q[$];
    bit[`AXI_SLAVE_ID_WIDTH-1:0]    ref_rid_q[$];
    int                             idx[$];

    if(read_data_ch_q.size() > 1) begin
      foreach(read_data_ch_q[k]) begin
       ref_rid_q[k] = read_data_ch_q[k].arid;
      end
      idx = ref_rid_q.unique_index();
      if(idx.size() < 2) begin
        // Only single ID is found
        return;
      end
      idx = {};
      ref_rid_q.shuffle();
      foreach(ref_rid_q[k]) begin
       // Reorganize queue
       idx = read_data_ch_q.find_first_index() with (item.arid == ref_rid_q[k]);
       res_q[k] = read_data_ch_q[idx[0]];
       read_data_ch_q.delete(idx[0]);
      end
      read_data_ch_q = res_q;
    end
  endfunction : shuffle_data_queue

  task axi_slave_read_driver::run_phase(uvm_phase phase);
    super.run_phase(phase);
    `uvm_info(get_type_name(), "AXI slave driver...", UVM_DEBUG)
    fork
      get_and_drive();
      observe_reset();
      /* add other tasks here */
    join_none;
  endtask : run_phase

  task axi_slave_read_driver::initialize();
    vif.arready  <= '0;
    vif.rid      <= '0;
    vif.rresp    <= '0;
    vif.rvalid   <= '0;
    vif.rlast    <= '0;
    vif.rdata    <= '0;
  endtask : initialize

  task axi_slave_read_driver::observe_reset();
    forever begin
      @(negedge vif.aresetn);
      `uvm_info(get_type_name(), "Reset observed on HRESETn.", UVM_DEBUG)
      initialize();
    end
  endtask : observe_reset

  task axi_slave_read_driver::address_channel();
    // Setup phase
    // Read channel -> address
    if(vif.arvalid === 1'b1) begin
      // Read request is detected -> Get next item
      seq_item_port.get_next_item(req); // For read channel
      // Sample address signals
     `uvm_info(get_type_name(), $sformatf("ARVALID -> READ address = 0x%x", vif.araddr), UVM_DEBUG)
      req.arid     = vif.arid;
      req.araddr   = vif.araddr;
      req.arlen    = vif.arlen;
      req.arsize   = vif.arsize;
      req.arburst  = vif.arburst;
      req.arqos    = vif.arqos;
      req.aruser   = vif.aruser;
      req.read_fl  = 1'b1; // Indicates acitivity in the READ DATA CHANNEL

      if(vif.arready !== 1'b1) begin // Arready was previously asserted
        // Wait and drive Arready
        repeat(req.arvalid_delay) begin
         @(posedge vif.aclk iff vif.aresetn == 1'b1);
        end
        vif.arready <= 1'b1; // Assert Arready
        @(posedge vif.aclk iff vif.aresetn == 1'b1);
      end // if(vif.arready !== 1'b1
     vif.arready <= 1'b0; // Deassert Arready
     seq_item_port.item_done();        // For address
     seq_item_port.get_next_item(req); // For read data
     $cast(req_cloned, req.clone());
     `uvm_info(get_type_name(), $sformatf("READ DEBUG A rdata.size %d.", req_cloned.rdata.size()), UVM_DEBUG)
     read_data_ch_q.push_back(req_cloned);
     seq_item_port.item_done();        // For read data
    end
  endtask : address_channel

  task axi_slave_read_driver::data_channel();

  if(!rdata_ch_is_active) begin : READ_CH_IS_IDLE
   if(read_data_ch_q.size() > 0) begin
    read_item = read_data_ch_q.pop_front();
    `uvm_info(get_type_name(), $sformatf("READ DEBUG 0 rdata.size %d. RID = 0x%x ARLEN = = 0x%x", read_item.rdata.size(), read_item.arid, read_item.arlen), UVM_DEBUG)
    rdata_cnt = 0;
    read_delay_cnt = read_item.rvalid_delay[0]; // Get the first delay
    rdata_ch_is_active = 1'b1;
    if(read_delay_cnt == 0) begin
      // Special case -> Read delay is 0
      `uvm_info(get_type_name(), $sformatf("DEBUG ONLY. READ data channel was idle and the first data beat had zero delay ARLEN = 0x%x. rdata.size %d. RID = 0x%x.", read_item.arlen, read_item.rdata.size(), read_item.arid), UVM_DEBUG)
      vif.rvalid <= 1'b1;
      vif.rdata  <= read_item.rdata[rdata_cnt]; // Drive data
      vif.rid    <= read_item.arid;
      if(rdata_cnt == read_item.arlen) begin
      // Current data beat will be the last one in the transaction so RLAST is asserted
      vif.rlast <= 1'b1;
      end // if(rdata_cnt == read_item.arlen)
    end
   end
  end // READ_CH_IS_IDLE
  else begin : READ_CH_IS_ACTIVE
    if(vif.rvalid === 1'b1 && vif.rready === 1'b1) begin : RDATA_ACCEPTED
      `uvm_info(get_type_name(), $sformatf("READ DATA BEAT. DATA = 0x%x. RLAST = %b.", vif.rdata, vif.rlast), UVM_DEBUG)
     if(vif.rlast === 1'b0) begin
      rdata_cnt++; // Icrement beat counter
      read_delay_cnt = read_item.rvalid_delay[rdata_cnt]; // Get the first delay
      `uvm_info(get_type_name(), $sformatf("Popping new READ data  Data = 0x%x. RVALID delay = %d.", read_item.rdata[rdata_cnt], read_delay_cnt), UVM_DEBUG)
      if(read_delay_cnt == 0) begin
      // Delay is expired -> start to drive data and RID
       `uvm_info(get_type_name(), $sformatf("DEBUG ONLY. Data beat in the burst = rdata_cnt %d. Data = 0x%x.", rdata_cnt, read_item.rdata[rdata_cnt]), UVM_DEBUG)
       vif.rdata  <= read_item.rdata[rdata_cnt]; // Drive data
       vif.rid    <= read_item.arid;
       if(rdata_cnt == read_item.arlen) begin
       // Next data beat will be the last one in the transaction so RLAST is asserted
        vif.rlast <= 1'b1;
       end // if(rdata_cnt == read_item.arlen)
      end // if(read_delay_cnt == 0)
      else begin : NEW_DATA_IS_NOT_YET_READY
       vif.rdata  <= ~vif.rdata; // Corrupt data
       vif.rvalid <= 1'b0;       // Deassert RVALID
       vif.rid    <= read_item.arid;
      end // NEW_DATA_IS_NOT_YET_READY
     end // if(vif.rlast === 1'b0)
     else begin : CURRENT_TRANS_IS_COMPLETED
      // Transfer is done -> deassert control / data signals
      shuffle_data_queue();
      if(read_data_ch_q.size() > 0 && read_data_ch_q[0].rvalid_delay[0] == 0) begin : NEW_TRANSFER_DELAY_ZERO
       // Read queue is not empty and the delay before the first data beat is zero -> the first data beat of the transfer sould be driven and RVALID is kept ASSERTED
      `uvm_info(get_type_name(), $sformatf("DEBUG ONLY. Immediate transfer in the READ data channel is found ARLEN = 0x%x. rdata.size %d. RID = 0x%x.", read_item.arlen, read_item.rdata.size(), read_item.arid), UVM_DEBUG)
       read_item = read_data_ch_q.pop_front();
       rdata_cnt = 0;
       read_delay_cnt = read_item.rvalid_delay[0]; // Get the first delay -> dummy  operation
       // Fix -> Drive DATA and read ID
       vif.rdata  <= read_item.rdata[rdata_cnt]; // Drive data
       vif.rid    <= read_item.arid;
       if(read_item.arlen != 0) begin
        // Deassert RLAST only if number of data beats is greater than 1 (ARLEN != 0)
        vif.rlast <= 1'b0;
       end // if(read_item.arlen != 0)
      end // NEW_TRANSFER_DELAY_ZERO
      else begin : END_OF_TRANSFER
       // Put READ DATA CHANNEL into the idle state -> Deassert control signals and "corrupt" data signals
       `uvm_info(get_type_name(), $sformatf("DEBUG ONLY. End of READ data transfer without immediate next READ data transfer is detected."), UVM_DEBUG)
       vif.rlast  <= 1'b0;       // Deassert RLAST
       vif.rvalid <= 1'b0;       // Deassert RVALID
       vif.rdata  <= ~vif.rdata; // "Corrupt" data
       rdata_ch_is_active = 1'b0;
      end // END_OF_TRANSFER
     end // CURRENT_TRANS_IS_COMPLETED
    end // RDATA_ACCEPTED
    else begin : NO_ACTIVE_BEAT
     if(read_delay_cnt == 0) begin : READ_DELAY_CNT_EXPIRED
     // Read delay counter is expired(in the middle of the READ DATA CHANNEL transfer ) -> start to drive rvlalid and rdata
      `uvm_info(get_type_name(), $sformatf("DEBUG ONLY. rdata_cnt %d.", rdata_cnt), UVM_DEBUG)
      vif.rvalid <= 1'b1;                       // Assert RVALID
      vif.rdata  <= read_item.rdata[rdata_cnt]; // Drive data
      vif.rid    <= read_item.arid;
      if(rdata_cnt == read_item.arlen) begin
       // Current data beat will be the last one in the transaction so RLAST is asserted
       vif.rlast <= 1'b1;
      end
     end // READ_DELAY_CNT_EXPIRED
    end // NO_ACTIVE_BEAT
    read_delay_cnt = (read_delay_cnt) ? read_delay_cnt - 1 : 0; // Decrement read delay counter
   end // READ_CH_IS_ACTIVE

    endtask : data_channel

   task axi_slave_read_driver::get_and_drive();
    initialize();
    fork
     forever begin : AR_CHANNEL
      @(posedge vif.aclk iff vif.aresetn == 1'b1);
      address_channel();
     end // AR_CHANNEL
     // Read channel ->  driven by the slave
     forever begin : READ_CHANNEL
      @(posedge vif.aclk iff vif.aresetn == 1'b1);
      data_channel();
     end // READ_CHANNEL
   join_none
  endtask : get_and_drive

   function void axi_slave_read_driver::report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

`endif // AXI_SLAVE_READ_DRIVER_SV
