//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company      : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies   : None
// Description    :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator    :
// Parameters     :
// Terms & concepts :
// Bugs       :
// Open issues and future enhancements :
// References     :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef AXI_SLAVE_READ_MONITOR_SV
`define AXI_SLAVE_READ_MONITOR_SV

class axi_slave_read_monitor extends uvm_monitor;

  virtual axi_slave_if  vif;
  axi_slave_config    cfg;
  uvm_analysis_port #(axi_slave_seq_item) ap;

  axi_slave_seq_item  address_item_q[$];

  `uvm_component_utils(axi_slave_read_monitor)
  extern function new(string name = "axi_slave_read_monitor", uvm_component parent = null);
  extern virtual function void build_phase(uvm_phase phase);
  extern virtual function void connect_phase(uvm_phase phase);
  extern virtual task run_phase(uvm_phase phase);
  extern virtual function axi_slave_seq_item find_addr_item(input bit[`AXI_SLAVE_ID_WIDTH-1:0] rid);
  extern virtual task observe_reset();
  extern virtual task collect_transaction();
  extern virtual function void perform_checks();
  extern virtual function void report_phase(uvm_phase phase);

endclass : axi_slave_read_monitor

  function axi_slave_read_monitor::new(string name = "axi_slave_read_monitor", uvm_component parent = null);
   super.new(name, parent);
  endfunction : new

  function void axi_slave_read_monitor::build_phase(uvm_phase phase);
    super.build_phase(phase);

    if (!uvm_config_db#(virtual axi_slave_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

    ap = new("ap", this);
  endfunction : build_phase

  function void axi_slave_read_monitor::connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  task axi_slave_read_monitor::run_phase(uvm_phase phase);
    super.run_phase(phase);
    fork
      collect_transaction();
      observe_reset();
    join_none;
  endtask : run_phase

  function axi_slave_seq_item axi_slave_read_monitor::find_addr_item(input bit[`AXI_SLAVE_ID_WIDTH-1:0] rid);
    int                             idx[$];
    axi_slave_seq_item              tmp;
    idx = address_item_q.find_first_index() with (item.arid == rid);
    if (idx.size() == 0) begin
      `uvm_error(get_full_name(), $sformatf("Transfer in data channel is detected but corresponded address transaction is not found. RID = 0x%x.", rid ))
    end
    tmp = address_item_q[idx[0]];
    address_item_q.delete(idx[0]);
    return tmp;
  endfunction : find_addr_item

  task axi_slave_read_monitor::observe_reset();
    wait(vif.aresetn === 1'b0);  // Reset assertion
    `uvm_info(get_type_name(), "Transaction recording interrupted by reset assertion.", UVM_DEBUG);
    wait(vif.aresetn === 1'b1);  // Reset deassertion
  endtask : observe_reset

  task axi_slave_read_monitor::collect_transaction();
    int unsigned beat_cnt;
    axi_slave_seq_item  item_cloned;
    axi_slave_seq_item  item_resp;
    axi_slave_seq_item  item;
    bit                 data_ch_is_active;

    if (!cfg.direction) begin
      // Write transactions are not monitored by the READ monitor
      return;
    end

    item = axi_slave_seq_item ::type_id::create("item");
    item_resp = axi_slave_seq_item ::type_id::create("item_resp");

    forever begin : AR_CHANNEL
      // Setup phase
      @(posedge vif.aclk iff vif.aresetn == 1'b1);
      // Handshaking in the ADDRESS READ channel
      if (vif.arvalid === 1'b1 && vif.arready === 1'b1 ) begin

        // Workaround for  assert(item.randomize()); -> Disable randomization + Create data areas
        item.disable_randomization();
        item.create_read_arrays(item.axi_max_beat);

        `uvm_info(get_type_name(), $sformatf("DEBUG size of created rdata buffer %d", item.rdata.size()), UVM_DEBUG)
        // Collect address related information from the READ ADDRESS CHANNEL
        item.arid     = vif.arid;
        item.araddr   = vif.araddr;
        item.arlen    = vif.arlen;
        item.arsize   = vif.arsize;
        item.arburst  = vif.arburst;
        item.arqos    = vif.arqos;
        item.aruser   = vif.aruser;
        $cast(item_cloned, item.clone());
        address_item_q.push_back(item_cloned);
      end // if (vif.arvalid === 1'b1 && vif.arready === 1'b1 )

      // Handshaking in the DATA READ channel
      if (vif.rvalid === 1'b1 && vif.rready === 1'b1) begin
        `uvm_info(get_type_name(), $sformatf("DEBUG: >>>>>>> data_ch_is_active = %d", data_ch_is_active), UVM_DEBUG)
        if (data_ch_is_active == 1'b0) begin
          // The first data beat in the transaction -> find corresponded item (order and rid)
          item_resp = find_addr_item(vif.rid);
          item_resp.rid = vif.rid; // Sample RID in in the first beat of data transfer
          `uvm_info(get_type_name(), $sformatf("DEBUG: RID 0x%x,size of created rdata buffer %d", item_resp.rid, item_resp.rdata.size()), UVM_DEBUG)
          beat_cnt = 0; // Clear data beat conter
          // Sample data and response (for the first data beat)
          item_resp.rdata[beat_cnt] = vif.rdata;
          item_resp.rresp[beat_cnt] = vif.rresp;
          data_ch_is_active = 1'b1;

          if (vif.rlast === 1'b1) begin
            // The last(and the only one) data beat in the transfer
            data_ch_is_active = 1'b0;
            item_resp.read_fl = 1'b1; // Item is READ
            `uvm_info(get_type_name(), $sformatf("Collected AXI transfer = \n %s", item_resp.convert2string), UVM_DEBUG)
            ap.write(item_resp);
          end // if (vif.rlast === 1'b1)

        end
        else begin
          // We are in the middle of the data transfer
          beat_cnt++; // Increment data beat conter
          // Sample data and response
          item_resp.rdata[beat_cnt] = vif.rdata;
          item_resp.rresp[beat_cnt] = vif.rresp;

          if (vif.rlast === 1'b1) begin
            // The last data beat in the transfer
            data_ch_is_active = 1'b0;
            item_resp.read_fl = 1'b1; // Item is READ
            `uvm_info(get_type_name(), $sformatf("Collected AXI transfer = \n %s", item_resp.convert2string), UVM_DEBUG)
            ap.write(item_resp);
          end // if (vif.rlast === 1'b1)
        end
      end // if (vif.rvalid === 1'b1 && vif.rready === 1'b1)
    end // AR_CHANNEL

  endtask : collect_transaction


  function void axi_slave_read_monitor::perform_checks();
   // Empty for the moment
  endfunction : perform_checks

  function void axi_slave_read_monitor::report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

`endif // AXI_SLAVE_READ_MONITOR_SV
