//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef AXI_SLAVE_WRITE_DRIVER_SV
`define AXI_SLAVE_WRITE_DRIVER_SV

class axi_slave_write_driver extends uvm_driver #(axi_slave_seq_item);

  `uvm_component_utils(axi_slave_write_driver)

  virtual axi_slave_if          vif;
  axi_slave_config              cfg;
  bresp_s                       bid_q[$];


  // get_and_drive
  bit                           wr_ch_is_active;      // Driver expects transfer in WRITE DATA channel
  bit                           wr_rsp_ch_is_active;  // Active transfer in WRITE RESPONSE channel
  int unsigned                  wdata_beat_cnt;
  int unsigned                  wready_delay_cnt;
  bresp_s                       bid_tmp;

  extern function new (string name = "axi_slave_write_driver", uvm_component parent = null);
  extern virtual function void fill_bresp(
                                          input int unsigned                 bvalid_delay,
                                          input axi_resp_type_e              bresp,
                                          input bit[`AXI_SLAVE_ID_WIDTH-1:0] bid
                                          );
  extern virtual function void build_phase(uvm_phase phase);
  extern virtual function void connect_phase(uvm_phase phase);
  extern virtual function void shuffle_bresp_queue();
  extern virtual function void report_phase(uvm_phase phase);
  extern virtual task run_phase(uvm_phase phase);
  extern virtual task initialize();
  extern virtual task observe_reset();
  extern virtual task address_channel();
  extern virtual task data_channel();
  extern virtual task response_channel();
  extern virtual task get_and_drive();

endclass : axi_slave_write_driver

  function axi_slave_write_driver::new (string name = "axi_slave_write_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

   function void axi_slave_write_driver::fill_bresp(
                                                    input int unsigned                 bvalid_delay,
                                                    input axi_resp_type_e              bresp,
                                                    input bit[`AXI_SLAVE_ID_WIDTH-1:0] bid
                                                    );
   bresp_s tmp;
   tmp.bvalid_delay = bvalid_delay;
   tmp.bresp        = bresp;
   tmp.bid          = bid;
   bid_q.push_back(tmp);
  endfunction : fill_bresp

   function void axi_slave_write_driver::build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual axi_slave_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name()," if not configured");

  endfunction : build_phase

  function void axi_slave_write_driver::connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


   task axi_slave_write_driver::run_phase(uvm_phase phase);
    super.run_phase(phase);
    `uvm_info(get_type_name(), "AXI slave driver...", UVM_DEBUG)
    fork
      get_and_drive();
      observe_reset();
      /* add other tasks here */
    join_none;
  endtask : run_phase

   task axi_slave_write_driver::initialize();
   vif.awready <= '0;
   vif.bid     <= '0;
   vif.bresp   <= '0;
   vif.bvalid  <= '0;
   vif.wready  <= '0;
  endtask : initialize

  // "Randomize" queue of responses
   function void axi_slave_write_driver::shuffle_bresp_queue();
   bresp_s                         res_q[$];
   bit[`AXI_SLAVE_ID_WIDTH-1:0]    ref_rid_q[$];
   int                             idx[$];
   if(bid_q.size() > 1) begin
    foreach(bid_q[k]) begin
     ref_rid_q[k] = bid_q[k].bid;
    end
    idx = ref_rid_q.unique_index();
    if(idx.size() < 2) begin
     // Only single ID is found
     return;
    end
    idx = {};
    ref_rid_q.shuffle();
    foreach(ref_rid_q[k]) begin
     // Reorganize queue
     idx = bid_q.find_first_index() with (item.bid == ref_rid_q[k]);
     res_q[k] = bid_q[idx[0]];
     bid_q.delete(idx[0]);
    end
    bid_q = res_q;
   end
  endfunction : shuffle_bresp_queue


   task axi_slave_write_driver::observe_reset();
    forever begin
       @(negedge vif.aresetn);
      `uvm_info(get_type_name(), "Reset observed on HRESETn.", UVM_DEBUG)
       initialize();
    end
  endtask : observe_reset


   task axi_slave_write_driver::address_channel();

   // Setup phase
   // Write channel -> address
   if(vif.awvalid === 1'b1) begin
   // Write request is detected
   // Get next item
     seq_item_port.get_next_item(req); // For write channel
     wdata_beat_cnt = 0;
   // Sample
    `uvm_info(get_type_name(), $sformatf("AWVALID -> Write address = 0x%x", vif.awaddr), UVM_DEBUG)

     req.awid     = vif.awid;
     req.awaddr   = vif.awaddr;
     req.awlen    = vif.awlen;
     req.awsize   = vif.awsize;
     req.awburst  = vif.awburst;
     req.awqos    = vif.awqos;
     req.awuser   = vif.awuser;

     if(vif.awready !== 1'b1) begin // AWREADY was previously asserted
     // Wait and drive AWREADY
     repeat(req.awvalid_delay) begin
      @(posedge vif.aclk iff vif.aresetn == 1'b1);
     end
     vif.awready <= 1'b1; // Assert AWREADY
    @(posedge vif.aclk iff vif.aresetn == 1'b1);
    end // if(vif.awready !== 1'b1
    vif.awready <= 1'b0; // Deassert AWREADY
    seq_item_port.item_done();        // For address
    seq_item_port.get_next_item(req); // For write data
    wr_ch_is_active = 1'b1;
    wready_delay_cnt = req.wvalid_delay[0]; // WREADY assertion delay for the first data beat -> trick
   end

  endtask : address_channel

   task axi_slave_write_driver::data_channel();

   if(wr_ch_is_active) begin : wr_ch_is_active_lbl
    wready_delay_cnt = (wready_delay_cnt > 0) ? wready_delay_cnt - 1 : 0;
    if(!wready_delay_cnt) begin
     vif.wready <= 1'b1; // Assert AWREADY
    end // if(!wready_delay_cnt)
   if(vif.wvalid === 1'b1) begin
    if(vif.wready === 1'b1) begin
    // WVALID and WREADY are high ->  End of the current data beat
    `uvm_info(get_type_name(), $sformatf("WVALID -> Write data = 0x%x", vif.wdata), UVM_DEBUG)
     req.wdata[wdata_beat_cnt] = vif.wdata;
     req.wstrb[wdata_beat_cnt] = vif.wstrb;
     wdata_beat_cnt++;
     if(vif.wlast === 1'b1) begin
      // End of transfer in the write channel
      wr_ch_is_active = 1'b0; // End of wdata phase
      fill_bresp(req.bvalid_delay, axi_resp_type_e'(req.bresp), vif.wid);
      `uvm_info(get_type_name(), $sformatf("Debug BVALID DELAY 1 = %d", req.bvalid_delay), UVM_DEBUG)
      seq_item_port.item_done();
     end // if(vif.wlast)

     // WREADY manipulation
     if (!wr_ch_is_active) begin
      vif.wready <= 1'b0; // WLAST was detected
     end
     else if(!wready_delay_cnt) begin
      if(!req.wvalid_delay[wdata_beat_cnt+1]) begin
       vif.wready <= 1'b1; // !!! next delay is 0
      end
      else begin
       vif.wready <= 1'b0; // Deassert WREADY
      end
     end
     // New value of delay counter
     wready_delay_cnt = req.wvalid_delay[wdata_beat_cnt];

    end // if(vif.wready === 1'b1)
   end // if(vif.wvalid === 1'b1)
  end // wr_ch_is_active_lbl

  endtask : data_channel

   task axi_slave_write_driver::response_channel();
    if(vif.bvalid === 1'b0) begin
     if(!wr_rsp_ch_is_active) begin
      if(bid_q.size()) begin
       shuffle_bresp_queue(); // Provide random order of responses
       bid_tmp = bid_q.pop_front();
       `uvm_info(get_type_name(), $sformatf("Debug BVALID DELAY 2 = %d", bid_tmp.bvalid_delay), UVM_DEBUG)
       wr_rsp_ch_is_active = 1'b1;
      end // if(bid_q.size())
     end // if(!wr_rsp_ch_is_active)
     else begin
      if(bid_tmp.bvalid_delay == 0) begin
       vif.bvalid <= 1'b1;
       vif.bresp  <= bid_tmp.bresp;
       vif.bid    <= bid_tmp.bid;
      end // if(bid_tmp.bvalid_delay == 0)
      `uvm_info(get_type_name(), $sformatf("Debug BVALID DELAY 3 = %d", bid_tmp.bvalid_delay), UVM_DEBUG)
       bid_tmp.bvalid_delay = (bid_tmp.bvalid_delay) ? bid_tmp.bvalid_delay - 1 : 0; // Decrement delay counter
     end
    end // if(vif.bvalid === 1'b0)
    else  begin // vif.bvalid === 1'b1
     if(vif.bready === 1'b1) begin
      if(bid_q.size() && bid_q[0].bvalid_delay == 0) begin
       bid_tmp = bid_q.pop_front();
       vif.bresp  <= bid_tmp.bresp;
       vif.bid    <= bid_tmp.bid;
      end
      else begin
       vif.bvalid <= 1'b0;
       wr_rsp_ch_is_active = 1'b0;
      end
     end // if(vif.bready === 1'b1)
    end
  endtask : response_channel


   task axi_slave_write_driver::get_and_drive();
   initialize();
   fork
    forever begin : AW_WD_CHANNELS
     @(posedge vif.aclk iff vif.aresetn == 1'b1);
     address_channel();
     data_channel();
    end // AW_WD_CHANNELS
    // Response channel -> the only channel driven by the slave
    forever begin : WSP_CHANNEL
     @(posedge vif.aclk iff vif.aresetn == 1'b1);
     response_channel();
    end // WSP_CHANNEL
   join_none
  endtask : get_and_drive

   function void axi_slave_write_driver::report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

`endif // AXI_SLAVE_WRITE_DRIVER_SV
