//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company      : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies   : None
// Description    :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator    :
// Parameters     :
// Terms & concepts :
// Bugs       :
// Open issues and future enhancements :
// References     :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef AXI_SLAVE_WRITE_MONITOR_SV
`define AXI_SLAVE_WRITE_MONITOR_SV

class axi_slave_write_monitor extends uvm_monitor;

  `uvm_component_utils(axi_slave_write_monitor)

  virtual axi_slave_if  vif;
  axi_slave_config    cfg;
  uvm_analysis_port #(axi_slave_seq_item) ap;
  uvm_analysis_port #(axi_slave_seq_item) ap_wd; // Analysis port for wdata channel -> item contains address and data related infor but not the response information

  axi_slave_seq_item  aw_channel_item_q[$];
  axi_slave_seq_item  w_channel_item_q[$];

  extern function new(string name = "axi_slave_write_monitor", uvm_component parent = null);
  extern virtual task run_phase(uvm_phase phase);
  extern virtual task observe_reset();
  extern virtual task collect_transaction();
  extern virtual function void build_phase(uvm_phase phase);
  extern virtual function void connect_phase(uvm_phase phase);
  extern virtual function axi_slave_seq_item find_addr_item(input bit[`AXI_SLAVE_ID_WIDTH-1:0] wid);
  extern virtual function axi_slave_seq_item find_data_item(input bit[`AXI_SLAVE_ID_WIDTH-1:0] bid);
  extern virtual function void perform_checks();
  extern virtual function void report_phase(uvm_phase phase);

endclass : axi_slave_write_monitor


  function axi_slave_write_monitor::new(string name = "axi_slave_write_monitor", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  function void axi_slave_write_monitor::build_phase(uvm_phase phase);
    super.build_phase(phase);
    if (!uvm_config_db#(virtual axi_slave_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");
    ap    = new("ap", this);
    ap_wd = new("ap_wd", this);
  endfunction : build_phase

  function void axi_slave_write_monitor::connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task axi_slave_write_monitor::run_phase(uvm_phase phase);
    super.run_phase(phase);
    fork
      collect_transaction();
      observe_reset();
    join_none;
  endtask : run_phase

  function axi_slave_seq_item axi_slave_write_monitor::find_addr_item(input bit[`AXI_SLAVE_ID_WIDTH-1:0] wid);
    int                             idx[$];
    axi_slave_seq_item              tmp;
    idx = aw_channel_item_q.find_first_index() with (item.awid == wid);
    if (idx.size() == 0) begin
      `uvm_error(get_full_name(), $sformatf("Transfer in data channel is detected but corresponded address transaction is not found. wid = 0x%x.", wid ))
    end
    tmp = aw_channel_item_q[idx[0]];
    aw_channel_item_q.delete(idx[0]);
    return tmp;
  endfunction : find_addr_item

  function axi_slave_seq_item axi_slave_write_monitor::find_data_item(input bit[`AXI_SLAVE_ID_WIDTH-1:0] bid);
    int                             idx[$];
    axi_slave_seq_item              tmp;
    idx = w_channel_item_q.find_first_index() with (item.wid == bid);
    if (idx.size() == 0) begin
     `uvm_error(get_full_name(), $sformatf("Transfer in response channel is detected but corresponded data transaction is not found. bid = 0x%x.", bid ))
    end
    tmp = w_channel_item_q[idx[0]];
    w_channel_item_q.delete(idx[0]);
    return tmp;
  endfunction : find_data_item

  task axi_slave_write_monitor::observe_reset();
    wait(vif.aresetn === 1'b0);  // Reset assertion
    `uvm_info(get_type_name(), "Transaction recording interrupted by reset assertion.", UVM_DEBUG);
    wait(vif.aresetn === 1'b1);  // Reset deassertion
  endtask : observe_reset

  task axi_slave_write_monitor::collect_transaction();
    int unsigned beat_cnt;
    axi_slave_seq_item  item_cloned;
    axi_slave_seq_item  item_resp;
    axi_slave_seq_item  item_bresp;
    axi_slave_seq_item  item;
    bit                 data_ch_is_active;

    if (cfg.direction) begin
     // Read transactions are not monitored by the WRITE monitor
     return;
    end

    item       = axi_slave_seq_item ::type_id::create("item");
    item_resp  = axi_slave_seq_item ::type_id::create("item_resp");
    item_bresp = axi_slave_seq_item ::type_id::create("item_bresp");

    forever begin : AW_CHANNEL
      // Setup phase
      @(posedge vif.aclk iff vif.aresetn == 1'b1);
      // Handshaking in the ADDRESS WRITE channel
      if (vif.awvalid === 1'b1 && vif.awready === 1'b1 ) begin

        // Workaround for  assert(item.randomize()); -> Disable randomization + Create data areas
        item.disable_randomization();
        item.create_write_arrays(item.axi_max_beat);

        `uvm_info(get_type_name(), $sformatf("DEBUG size of created rdata buffer %d", item.wdata.size()), UVM_DEBUG)
        // Collect address related information from the WRITE ADDRESS CHANNEL
        item.awid     = vif.awid;
        item.awaddr   = vif.awaddr;
        item.awlen    = vif.awlen;
        item.awsize   = vif.awsize;
        item.awburst  = vif.awburst;
        item.awqos    = vif.awqos;
        item.awuser   = vif.awuser;
        $cast(item_cloned, item.clone());
        aw_channel_item_q.push_back(item_cloned);
      end // if (vif.awvalid === 1'b1 && vif.awready === 1'b1 )

      // Handshaking in the DATA WRITE channel
      if (vif.wvalid === 1'b1 && vif.wready === 1'b1 ) begin
        `uvm_info(get_type_name(), $sformatf("DEBUG: >>>>>>> data_ch_is_active = %d", data_ch_is_active), UVM_DEBUG)
        if (data_ch_is_active == 1'b0) begin
          // The first data beat in the transaction -> find corresponded item (order and rid)
          item_resp = find_addr_item(vif.wid);
          item_resp.wid = vif.wid; // Sample RID in in the first beat of data transfer
          `uvm_info(get_type_name(), $sformatf("DEBUG: WID 0x%x,size of created rdata buffer %d", item_resp.wid, item_resp.wdata.size()), UVM_DEBUG)
          beat_cnt = 0; // Clear data beat conter
          // Sample data and response (for the first data beat)
          item_resp.wdata[beat_cnt] = vif.wdata;
          // item_resp.wresp = vif.wresp;
          data_ch_is_active = 1'b1;

          if (vif.wlast === 1'b1) begin
            // The last(and the only one) data beat in the transfer
            data_ch_is_active = 1'b0;
            // `uvm_info(get_type_name(), $sformatf("Collected AXI transfer = \n %s", item_resp.convert2string), UVM_DEBUG)
            ap_wd.write(item_resp); // Provides Address/Data (but not response) related information for the subscribers
            w_channel_item_q.push_back(item_resp);
          end // if (vif.wlast === 1'b1)

        end
        else begin
          // We are in the middle of the data transfer
          beat_cnt++; // Increment data beat conter
          // Sample data and response
          item_resp.wdata[beat_cnt] = vif.wdata;
          //item_resp.wresp = vif.wresp;

          if (vif.wlast === 1'b1) begin
            // The last data beat in the transfer
            data_ch_is_active = 1'b0;
            item_bresp.read_fl = 1'b0; // Item is WRITE
            ap_wd.write(item_resp); // Provides Address/Data (but not response) related information for the subscribers
            w_channel_item_q.push_back(item_resp);
          end // if (vif.wlast === 1'b1)

        end
      end // if (vif.wvalid === 1'b1 && vif.wready === 1'b1 )

      // Handshaking in the WRITE RESPONSE channel
      if (vif.bvalid === 1'b1 && vif.bready === 1'b1) begin
        item_bresp = find_data_item(vif.bid);
        item_bresp.bid = vif.bid;
        item_bresp.bresp = vif.bresp;
        item_bresp.read_fl = 1'b0; // Item is WRITE
        `uvm_info(get_type_name(), $sformatf("Collected AXI transfer = \n %s", item_bresp.convert2string), UVM_DEBUG)
        ap.write(item_bresp);
      end // if (vif.bvalid === 1'b1 && vif.bready === 1'b1)
     end // AW_CHANNEL
  endtask : collect_transaction

  function void axi_slave_write_monitor::perform_checks();
    // Empty for the moment
  endfunction : perform_checks

  function void axi_slave_write_monitor::report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

`endif // AXI_SLAVE_WRITE_MONITOR_SV
