//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef H_MEM_DRIVER_SV
`define H_MEM_DRIVER_SV


class h_mem_driver extends uvm_driver #(h_mem_seq_item);

  `uvm_component_utils(h_mem_driver)

  virtual h_mem_if vif;
  h_mem_config     cfg;
  uvm_analysis_port #(h_mem_seq_item) ap;

  //---------------------------------------------------------------------------
  // variables
  //---------------------------------------------------------------------------
  bit signed  [7:0] lower_subcarrier;   // First subcarrier that should be checked
  bit signed  [7:0] upper_subcarrier;   // Last subcarrier that should be checked
  bit         [7:0] matlab_offset;      // Subcarrier offset for the matlab buffers
  bit signed  [7:0] pilot_subcarrier[]; // Pilot & DC subcarriers that should not be checked
  bit signed  [7:0] subcarrier_index;   // Loop index for subcarriers

  bit signed [`HMEM_WIDTH-1:0] h_val_re; // H mem complex memory content
  bit signed [`HMEM_WIDTH-1:0] h_val_im; // H mem complex memory content

  bit [`HMEMDATAWIDTH-1:0]     rdata;    // read data from H memory
  //---------------------------------------------------------------------------


  function new (string name = "h_mem_driver", uvm_component parent = null);
    super.new(name, parent);

    ap = new("ap", this);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual h_mem_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  //-----------------------------------------------------------------------
  // list of tasks and functions
  //-----------------------------------------------------------------------
  extern task execute_hmem_readout();
  extern task writeHmem(bit [`HMEMDATAWIDTH-1:0] address, bit [`HMEMDATAWIDTH-1:0] data);
  extern task readHmem (bit [`HMEMDATAWIDTH-1:0] address, ref bit [`HMEMDATAWIDTH-1:0] data);
  //-----------------------------------------------------------------------

  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    fork
      get_and_drive();
      /* add other tasks here */
    join_none;
  endtask : run_phase

  task initialize();
    vif.wr_data <= 0;
    vif.wr_addr <= 0;
    vif.rd_data <= 0;
    vif.rd_addr <= 0;
  endtask : initialize

  task get_and_drive();
    initialize();

    forever begin
      `uvm_info(get_type_name(), "Start of a bus cycle detected.", UVM_DEBUG)
      seq_item_port.get_next_item(req);
      `uvm_info(get_type_name(), "Got new item.", UVM_DEBUG)

      // Define list of subcarriers to check
      case (req.ch_bw)
        2'b00: begin
          lower_subcarrier = -28;
          upper_subcarrier =  28;
          pilot_subcarrier = {-21,-7,0,7,21};
        end
        2'b01: begin
          lower_subcarrier = -58;
          upper_subcarrier =  58;
          pilot_subcarrier = {-53,-25,-11,-1,0,1,11,25,53};
        end
        2'b10:  begin
          lower_subcarrier = -122;
          upper_subcarrier =  122;
          pilot_subcarrier = {-103,-75,-39,-11,-1,0,1,11,39,75,103};
        end
       endcase

       case (req.cmd)
         H_MEM_WR: begin
           `uvm_warning(get_type_name(),$sformatf("Write to H memory not implemented"))
         end
         H_MEM_RD: begin
           execute_hmem_readout();
         end
       endcase

      `uvm_info(get_type_name(), $sformatf("Transaction requested :%s", req.sprint()), UVM_HIGH)
      ap.write(req);
      seq_item_port.item_done();
    end // forever
  endtask : get_and_drive

endclass : h_mem_driver

  //------------------------------------------------------
  // performe read out data from H memory
  //------------------------------------------------------
  task h_mem_driver::execute_hmem_readout();
    // Check loop
    for (subcarrier_index=lower_subcarrier; subcarrier_index<=upper_subcarrier; subcarrier_index++) begin
      matlab_offset = (req.ch_bw == 2'b00) ? subcarrier_index+32 :
                      (req.ch_bw == 2'b01) ? subcarrier_index+64 :
                                             subcarrier_index+128;

      if (subcarrier_index inside {pilot_subcarrier}) begin
        `uvm_info(get_type_name(),$sformatf("Hmem(subcarrier %3d) => Not Checked (pilot or DC subcarrier)", subcarrier_index), UVM_DEBUG);
      end
      else begin
        for (int rx_index=0; rx_index < req.num_rx; rx_index++) begin
          for (int ss_index=0; ss_index < req.num_ss; ss_index++) begin

            // Extract from RTL the H value for subcarrier_index, rx_index & ss_index
            readHmem($unsigned(subcarrier_index), rdata);
            h_val_re = (rdata >> ((((`RW_NX_DERIV_NESS+`RW_NX_DERIV_NSTS)*(rx_index+1))-(ss_index+1))*26   )) & 13'h1FFF;
            h_val_im = (rdata >> ((((`RW_NX_DERIV_NESS+`RW_NX_DERIV_NSTS)*(rx_index+1))-(ss_index+1))*26+13)) & 13'h1FFF;
            // store read data
            req.hmem_re = new[req.hmem_re.size()+1](req.hmem_re);
            req.hmem_re[req.hmem_re.size()-1] = h_val_re;
            req.hmem_im = new[req.hmem_im.size()+1](req.hmem_im);
            req.hmem_im[req.hmem_im.size()-1] = h_val_im;

          end// for ss_index
        end// for rx_index
      end// else
    end// for subcarrier_index
  endtask : execute_hmem_readout

  //------------------------------------------------------
  // write to memory via backdoor access
  //------------------------------------------------------
  task h_mem_driver::writeHmem(bit [`HMEMDATAWIDTH-1:0] address, bit [`HMEMDATAWIDTH-1:0] data);
    begin
      `uvm_info(get_type_name(), $sformatf("writeHmem : Writing 0x%0h @ 0x%0h",data,address), UVM_DEBUG)
      vif.wr_data   <= data;
      vif.wr_addr   <= address;
      #(`HMEM_DELAY_TICK);
    end
  endtask : writeHmem

  //------------------------------------------------------
  // read from memory via backdoor access
  //------------------------------------------------------
  task h_mem_driver::readHmem(bit [`HMEMDATAWIDTH-1:0] address, ref bit [`HMEMDATAWIDTH-1:0] data);
    begin
      vif.rd_addr <= address;
      #(`HMEM_DELAY_TICK);
      data        = vif.rd_data;
      `uvm_info(get_type_name(), $sformatf("readHmem : Read 0x%0h @ 0x%0h",data,address), UVM_DEBUG)
    end
  endtask: readHmem

`endif
