//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_DRIVER_SV
`define MAC_DRIVER_SV


class mac_driver extends uvm_driver #(mac_seq_item);

  `uvm_component_utils(mac_driver)

  virtual mac_phy_if vif;
  mac_config         cfg;

  // Tx vectors for up to 4 users
  octet_t            tx_vector[];
  bit                mu_mimo;          // indicate that MU-MIMO is sent
  bit [3:0]          dd_ff[$];         // delay signals for MAC data valid generation
  bit [3:0]          delayed_mac_valid;
  bit [7:0]          tx_data_uid;


  function new (string name = "mac_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual mac_phy_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    initialize();

    fork
      get_and_drive();
      drive_mac_data_valid();
    join_none;
  endtask : run_phase

  task initialize();
    mu_mimo = 0;
    // init interface signals
    vif.tx_req            <= 0;
    vif.rx_req            <= 0;
    vif.tx_data           <= 0;
    vif.mac_data_valid    <= 0;
    vif.mimo_cmd_valid    <= 0;
    vif.keep_rf_on        <= 0;
    vif.tx_data_uid       <= 0;
    wait (vif.rst_n == 1'b1);
  endtask : initialize

  //-------------------------------------------------------
  // MAC data valid signal is PHY ready signal
  // delayed for minimum 2 clock cycles
  //-------------------------------------------------------
  task drive_mac_data_valid();

    fork : DELAY_VALID_2_CLOCKS

      forever begin
        @(posedge vif.clk);
        dd_ff.push_front(vif.phy_rdy);
      end

      begin
        @(posedge vif.clk);
        forever begin
          @(posedge vif.clk);
          delayed_mac_valid = dd_ff.pop_back();
        end
      end

    join_none
  endtask : drive_mac_data_valid

  //-------------------------------------------------------
  // wait for new sequence item to drive signals
  //-------------------------------------------------------
  task get_and_drive();
    int      sample_cnt;  //number of received samples

    forever begin
      `uvm_info(get_type_name(), "Start of a bus cycle detected.", UVM_HIGH)
      seq_item_port.get_next_item(req);
      `uvm_info(get_type_name(), "Got new MAC item.", UVM_HIGH)

      if (req.trans == TX) begin
        //NOTE: Check for user order
        `uvm_info(get_type_name(), "MAC-PHY transmit generation", UVM_HIGH)
        // TX vector will be created from Matlab file not PPDU frame
        if (cfg.use_preset_files) begin
          create_tx_vector_from_file(tx_vector);
        end
        else begin
          // TX vector created from PPDU frame
          case (req.frame.num_of_users)
            0, 1: begin
              // in case of NDP just send TxVector
              mu_mimo = 0;
              create_tx_vector(tx_vector, req.frame.preamble_header, 0, req.frame.num_of_users);
            end
            2, 3, 4: begin
              mu_mimo = 1;
              create_tx_vector(tx_vector, req.frame.preamble_header, req.frame.user_order[0], req.frame.num_of_users);
            end
          endcase
        end//else

        send_tx_vector();
        // run in 2 threads sending data and waiting for Tx end signal
        fork : SEND_AND_WAIT_END
          if (cfg.use_preset_files)
            send_data_from_file();
          else
            send_data(req.frame);
          wait_tx_end();
        join
        `uvm_info(get_type_name(), "MAC sequence for TX driving ended", UVM_HIGH)
      end// TX
      else if (req.trans == RX) begin

        `uvm_info(get_type_name(), $sformatf("MAC-PHY receive %0d number of frames",req.rx_frame_num), UVM_HIGH)
        @(posedge vif.clk);
        vif.rx_req <= 1'b1;
        repeat (req.rx_frame_num) begin
          while (vif.rx_end == 1'b0) begin
            // receive data until Rx end occurs or Rx error or Rx end
            @(posedge vif.clk iff  (vif.phy_rdy == 1'b1
                                 || vif.rx_err == 1'b1
                                 || vif.rx_end == 1'b1
                                 || cfg.stop_wating_rx_end == 1'b1));

            // when Rx error occurs
            if (vif.rx_err == 1'b1) continue;
            // to prevent driver to be stuck because of absence
            // of Rx end, trigger stop signal
            if (cfg.stop_wating_rx_end == 1'b1) break;
          end//while
          if (cfg.stop_wating_rx_end == 1'b1) break;
          wait (vif.rx_end == 1'b0);
        end//repeat
        // de-assert Rx request
        @(posedge vif.clk);
        vif.rx_req <= 1'b0;

        `uvm_info(get_type_name(), "MAC sequence for RX driving ended", UVM_HIGH)
      end// RX
      else begin // RX_RIFS

        `uvm_info(get_type_name(), $sformatf("MAC-PHY receive %0d number of frames",req.rx_frame_num), UVM_HIGH)
        @(posedge vif.clk);
        vif.rx_req <= 1'b1;
        sample_cnt = 0;
        repeat (req.rx_frame_num) begin
          while (vif.rx_end == 1'b0) begin
            // receive data until Rx end occurs or Rx error or Rx end
            @(posedge vif.clk iff vif.phy_rdy == 1'b1
              or vif.rx_err == 1'b1
              or vif.rx_end == 1'b1
              or cfg.stop_wating_rx_end == 1'b1);

            sample_cnt++;
            if (vif.rx_err == 1'b1) break;
            // to prevent driver to be stuck because of absence
            // of Rx end, trigger stop signal
            if (cfg.stop_wating_rx_end == 1'b1) break;
            if (sample_cnt == req.idle_cycles) vif.keep_rf_on = 1'b1;
          end//while
        end//repeat
        // de-assert Rx request
        @(posedge vif.clk);
        vif.rx_req     <= 1'b0;
        vif.keep_rf_on <= 1'b0;

        `uvm_info(get_type_name(), "MAC sequence for RX RIFS driving ended", UVM_HIGH)

      end// RX_RIFS

      seq_item_port.item_done();
    end // forever
  endtask : get_and_drive

  //-------------------------------------------------------
  // create Tx vectors from preset files prepared for Matlab
  //-------------------------------------------------------
  task create_tx_vector_from_file(ref octet_t vec[]);
    string         txvec[$];
    string         fname;
    string         value, format_mod;
    int            num_of_users = 1;
    format_mod_e   ppdu_format;
    bit [15:0]     spatial_reuse;
    bit            doppler;
    bit            trigger_method;

    // create TC name from global settings
    fname = {`STRINGIFY(`SYSPARAM_TXT_DIR),"/",`TESTCASE_NAME,`TESTCASE_EXT};
    // extract lines from file with TXV pattern
    txvec = extract_pattern_from_file("TXV", fname);

    //determine size of vector
    format_mod = get_param_value(txvec, "FORMAT");
    case (format_mod)
      "'NON_HT'"            : vec = new[10];
      "'HT_MF'","'HT_GF'"   : vec = new[13];
      "'VHT'"               : vec = new[12 + 5*num_of_users];
      "'HE_SU'","'HE_ER_SU'": vec = new[18];
      "'HE_MU'"             : vec = new[19 + 7*num_of_users];
      "'HE_TB'"             : vec = new[24];
    endcase

    //set txvector common part
    vec[0][3:0] = (format_mod == "'NON_HT'" && get_param_value(txvec,"NON_HT_MODULATION") == "'NON_HT_DUP_OFDM'") ? 4'b0001 :
                  (format_mod == "'HT_MF'")    ? 4'b0010 :
                  (format_mod == "'HT_GF'")    ? 4'b0011 :
                  (format_mod == "'VHT'")      ? 4'b0100 :
                  (format_mod == "'HE_SU'")    ? 4'b0101 :
                  (format_mod == "'HE_MU'")    ? 4'b0110 :
                  (format_mod == "'HE_ER_SU'") ? 4'b0111 :
                  (format_mod == "'HE_TB'")    ? 4'b1000 :
                                                 4'b0000; //NON-HT

    //cast to enum type to ease parsing
    ppdu_format = format_mod_e'(vec[0][3:0]);

    value = get_param_value(txvec, "CH_BANDWIDTH");
    vec[0][6:4] = value.atoi();
    value = get_param_value(txvec, "PREAMBLE_TYPE");
    vec[0][7]   = value.atoi();
`ifdef RW_TXRX_1X1
    vec[1][7:0] = 8'b0000_0001;
`elsif RW_TXRX_2X2
    vec[1][7:0] = 8'b0000_0011;
`endif
    value = get_param_value(txvec, "TXPWR_LEVEL_INDEX");
    vec[2][7:0] = value.atoi();
`ifdef RW_TXRX_1X1
    vec[3][2:0] = 3'b000;
`elsif RW_TXRX_2X2
    vec[3][2:0] = 3'b001;
`endif
    value = get_param_value(txvec, "L_LENGTH");
    {vec[5][3:0],vec[4][7:0]} = value.atoi();
    value = get_param_value(txvec, "L_DATARATE");
    vec[5][7:4] = decode_leg_rate(value.atoi());
    // SERVICE
    vec[6][7:0] = 0;
    vec[7][7:0] = 0;

    //NON-HT and NON_HT_DUP_OFDM
    if (ppdu_format inside {NON_HT, NON_HT_DUP_OFDM}) begin
      value = get_param_value(txvec, "TRIGGER_RESPONDING");
      vec[8][0] = value.atoi();
      // SMM INDEX
      vec[9][7:0] = 0;
    end

    // HT-MF and HT-GF
    if (ppdu_format inside {HT_MF, HT_GF}) begin
      value = get_param_value(txvec, "SOUNDING");
      vec[8][0] = value.atoi();
      value = get_param_value(txvec, "SMOOTHING");
      vec[8][1] = value.atoi();
      value = get_param_value(txvec, "GI_TYPE");
      // 0.8 - LONG_GI
      // 0.4 - SHORT_GI
      if (value == "0.8")
        vec[8][2] = 1'b0;
      else if (value == "0.4")
        vec[8][2] = 1'b1;
      value = get_param_value(txvec, "AGGREGATION");
      vec[8][3] = value.atoi();
      value = get_param_value(txvec, "STBC");
      vec[8][4] = value.atoi();
      // NUM EXT SS
      vec[8][6:5] = 2'b0;
      // SMM INDEX
      vec[9][7:0] = 0;
      value = get_param_value(txvec, "MCS");
      vec[10][6:0] = value.atoi();
      value = get_param_value(txvec, "FEC_CODING");
      vec[10][7] = value.atoi();
      value = get_param_value(txvec, "APEP_LENGTH");
      {vec[12],vec[11]} = value.atoi();
    end

    // VHT
    if (ppdu_format == VHT) begin
      value = get_param_value(txvec, "SOUNDING");
      vec[8][0] = value.atoi();
      value = get_param_value(txvec, "BEAMFORMED");
      vec[8][1] = value.atoi();
      value = get_param_value(txvec, "GI_TYPE");
      // 0.8 - LONG_GI
      // 0.4 - SHORT_GI
      if (value == "0.8")
        vec[8][2] = 1'b0;
      else if (value == "0.4")
        vec[8][2] = 1'b1;
      value = get_param_value(txvec, "STBC");
      vec[8][4] = value.atoi();
      vec[8][5] = 0; // doze_not_allowed
      value = get_param_value(txvec, "PARTIAL_AID");
      {vec[10][0],vec[9]} = value.atoi();
      value = get_param_value(txvec, "GROUP_ID");
      vec[10][6:1] = value.atoi();
      value = get_param_value(txvec, "NUM_USERS");
      vec[11][2:0] = value.atoi();
      //loop by user
      for (int i=0; i<num_of_users; i++) begin
        // SMM INDEX
        vec[12][7:0] = 0;
        value = get_param_value(txvec, "MCS");
        vec[13][3:0] = value.atoi();
        value = get_param_value(txvec, "NUM_STS");
        vec[13][6:4] = value.atoi()-1;
        value = get_param_value(txvec, "FEC_CODING");
        vec[13][7] = value.atoi();
        value = get_param_value(txvec, "APEP_LENGTH");
        {vec[16][3:0],vec[15],vec[14]} = value.atoi();
        value = get_param_value(txvec, "USER_POSITION");
        vec[16][7:6] = value.atoi();
      end
    end

    // common part for HE frames
    if (ppdu_format >= HE_SU) begin
      value = get_param_value(txvec, "SOUNDING");
      vec[8][0] = value.atoi();
      value = get_param_value(txvec, "BEAMFORMED");
      vec[8][1] = value.atoi();
      value = get_param_value(txvec, "GI_TYPE");
      // 0.8 - LONG_GI
      // 0.4 - SHORT_GI
      if (value == "0.8")
        vec[8][3:2] = 2'b00;
      else if (value == "1.6")
        vec[8][3:2] = 2'b01;
      else if (value == "3.2")
        vec[8][3:2] = 2'b10;
      value = get_param_value(txvec, "STBC");
      vec[8][4] = value.atoi();
      value = get_param_value(txvec, "UPLINK_FLAG");
      vec[9][0] = value.atoi();
      value = get_param_value(txvec, "BEAM_CHANGE");
      vec[9][1] = value.atoi();
      value = get_param_value(txvec, "DCM");
      vec[9][2] = value.atoi();
      value = get_param_value(txvec, "HE_LTF_TYPE");
      vec[9][4:3] = (value.atoi() == 1) ? 2'b00 :
                    (value.atoi() == 2) ? 2'b01 :
                    (value.atoi() == 4) ? 2'b10 :
                                          2'b00;
      value = get_param_value(txvec, "DOPPLER");
      vec[9][5] = value.atoi();
      doppler = value.atoi();
      // midamble_periodicity
      if (doppler) begin
        value = get_param_value(txvec, "MIDAMBLE_PERIODICITY");
        vec[9][6] = (value.atoi() == 10) ? 1'b0 : 1'b1;
      end
      else begin
        vec[9][6] = 0;
      end
      value = get_param_value(txvec, "BSS_COLOR");
      vec[10][5:0] = value.atoi();
      value = get_param_value(txvec, "TXOP_DURATION");
      vec[11][6:0] = value.atoi();
      value = get_param_value(txvec, "SPATIAL_REUSE");
      vec[12][3:0] = value.atoi();
    end

    // HE-SU and HE-EXT-SU
    if (ppdu_format inside {HE_SU, HE_EXT_SU}) begin
      // SMM INDEX
      vec[13][7:0] = 0;
      value = get_param_value(txvec, "MCS");
      vec[14][3:0] = value.atoi();
      value = get_param_value(txvec, "NUM_STS");
      vec[14][6:4] = value.atoi()-1;
      value = get_param_value(txvec, "FEC_CODING");
      vec[14][7] = value.atoi();
      value = get_param_value(txvec, "APEP_LENGTH");
      {vec[17][3:0],vec[16],vec[17]} = value.atoi();
      value = get_param_value(txvec, "NOMINAL_PACKET_PADDING");
      vec[17][6:4] = value.atoi() / 4;
    end

    // HE-MU
    if (ppdu_format == HE_MU) begin
      value = get_param_value(txvec, "SIG_B_COMPRESSION_MODE");
      vec[13][0] = value.atoi();
      value = get_param_value(txvec, "DCM_SIG_B");
      vec[13][1] = value.atoi();
      value = get_param_value(txvec, "MCS_SIG_B");
      vec[13][4:2] = value.atoi();
      value = get_param_value(txvec, "RU_ALLOCATION");
      vec[14][7:0] = value.atoi();
      vec[15][7:0] = 0;
      vec[16][7:0] = 0;
      vec[17][7:0] = 0;
      //N_USER
      vec[18][7:0] = 1;
      //loop by user
      for (int i=0; i<num_of_users; i++) begin
        // SMM INDEX
        vec[19][7:0] = 0;
        value = get_param_value(txvec, "MCS");
        vec[20][3:0] = value.atoi();
        value = get_param_value(txvec, "NUM_STS");
        vec[20][6:4] = value.atoi()-1;
        value = get_param_value(txvec, "FEC_CODING");
        vec[20][7] = value.atoi();
        value = get_param_value(txvec, "APEP_LENGTH");
        {vec[23][3:0],vec[22],vec[21]} = value.atoi();
        value = get_param_value(txvec, "NOMINAL_PACKET_PADDING");
        vec[23][6:4] = value.atoi() / 4;
        {vec[25][2:0],vec[24]} = (`STAID_OFFSET+2+i); // to be aligned with MDM
        // user position
        vec[25][7:3] = 0;
      end
    end

    //HE-TB
    if (ppdu_format == HE_TB) begin
      value = get_param_value(txvec, "SPATIAL_REUSE");
      spatial_reuse = value.atoi();
      {vec[13],vec[12][7:4]} = spatial_reuse[15:4];
      value = get_param_value(txvec, "HE_SIG_A2_RESERVED");
      {vec[15][0],vec[14]} = value.atoi();
      value = get_param_value(txvec, "NUM_HE_LTF");
      vec[15][3:1] = value.atoi()-1;
      //HE_LTF_MODE
      vec[15][4] = 0;
      value = get_param_value(txvec, "LDPC_EXTRA_SYMBOL");
      vec[15][5] = value.atoi();
      //Starting STS number
      vec[16][2:0] = 0;
      value = get_param_value(txvec, "RU_ALLOCATION");
      vec[17][7:0] = value.atoi();
      // SMM INDEX
      vec[18][7:0] = 0;
      value = get_param_value(txvec, "MCS");
      vec[19][3:0] = value.atoi();
      value = get_param_value(txvec, "NUM_STS");
      vec[19][6:4] = value.atoi()-1;
      value = get_param_value(txvec, "FEC_CODING");
      vec[19][7] = value.atoi();
      value = get_param_value(txvec, "APEP_LENGTH");
      {vec[22][3:0],vec[21],vec[20]} = value.atoi();
      value = get_param_value(txvec, "TRIGGER_METHOD");
      vec[22][7] = (value == "'TRIGGER_FRAME'") ? 1'b0 : 1'b1;
      trigger_method = (value == "'TRIGGER_FRAME'") ? 1'b0 : 1'b1;
      if (trigger_method) begin
        value = get_param_value(txvec, "DEFAULT_PE_DURATION");
        vec[22][6:4] = value.atoi() / 4;
      end
      else begin
        value = get_param_value(txvec, "HE_TB_PreFEC_Padding_Factor");
        vec[22][5:4] = value.atoi();
        value = get_param_value(txvec, "HE_TB_PE_DISAMBIGUITY");
        vec[22][6] = value.atoi();
      end
      // RU tone set
      vec[23][6:0] = 0;
      // feedback status
      vec[23][7] = 0;
    end

  endtask : create_tx_vector_from_file

  //-------------------------------------------------------
  // create Tx vectors for up to 4 users
  //-------------------------------------------------------
  task create_tx_vector(ref octet_t vec[], input PPDU_preamble_header ph, input int user, input int num_of_users);
    // prepare Tx vector for user 0
    int user_index = 1;
    case(ph.format_mod)
      NON_HT, NON_HT_DUP_OFDM : begin
                                  vec = new[10];
                                end
      HT_MF, HT_GF :            begin
                                  vec = new[13];
                                end
      VHT :                     begin
                                  vec = new[12 + 5*num_of_users];
                                end
      HE_SU, HE_EXT_SU :        begin
                                  vec = new[18];
                                end
      HE_MU :                   begin
                                  vec = new[19 + 7*num_of_users];
                                end
      HE_TB :                   begin
                                  vec = new[24];
                                end
    endcase

    vec[0] = {ph.preamble_type,
              ph.ch_bw,
              ph.format_mod
    };
    vec[1] = ph.antenna_set;
    vec[2] = ph.tx_pwr_level;
    vec[3] = {ph.continuous_tx,
              ph.time_dep_req,
              3'b000, // Reserved
              ph.num_tx
    };
    vec[4] = ph.leg_length[7:0];
    vec[5] = {ph.leg_rate,
              ph.leg_length[11:8]
    };
    vec[6] = ph.service[7:0];
    vec[7] = ph.service[15:8];

    // Common fields for all HE frames
    if (ph.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin

      vec[8] = {3'b000,
                ph.stbc,
                ph.gi_type,
                ph.beamformed,
                ph.sounding
      };
      vec[9] = {1'b0,
                ph.midamble_periodicity,
                ph.doppler,
                ph.he_ltf_type,
                ph.dcm,
                ph.beam_change,
                ph.uplink_flag
      };
      vec[10] = {2'b00,ph.bss_color};
      vec[11] = {1'b0,ph.txop_duration};
      vec[12] = {4'h0,ph.spatial_reuse[0]};

    end

    case(ph.format_mod)
      NON_HT, NON_HT_DUP_OFDM : begin
                                  vec[8] = ph.trigger_responding;
                                  vec[9] = ph.user_header[0].smm_index_f;
                                end
      HT_MF, HT_GF :            begin
                                  vec[8] = {1'b0,
                                            ph.num_extn_ss,
                                            ph.stbc,
                                            ph.aggregated,
                                            ph.gi_type[0],
                                            ph.smoothing,
                                            ph.sounding
                                  };
                                  vec[9] = ph.user_header[0].smm_index_f;
                                  vec[10] = {ph.user_header[0].fec_coding_f,
                                             ph.user_header[0].mcs_f
                                  };
                                  vec[11] = ph.user_header[0].ht_length_f[7:0];
                                  vec[12] = ph.user_header[0].ht_length_f[15:8];
                                end
      VHT :                     begin
                                  vec[8] = {2'b00,
                                            ph.doze_not_allowed,
                                            ph.stbc,
                                            1'b0,
                                            ph.gi_type[0],
                                            ph.beamformed,
                                            ph.sounding
                                  };
                                  vec[9] = ph.partial_aid[7:0];
                                  vec[10] = {ph.group_id,
                                             ph.partial_aid[8]
                                  };
                                  vec[11] = ph.n_user;
                                  for (int i = 0; i<num_of_users; i++) begin
                                    vec[12 + 5*i] = ph.user_header[i].smm_index_f;
                                    vec[13 + 5*i] = {ph.user_header[i].fec_coding_f,
                                                     ph.user_header[i].num_sts_f,
                                                     ph.user_header[i].mcs_f[3:0]
                                    };
                                    vec[14 + 5*i] = ph.user_header[i].ht_length_f[7:0];
                                    vec[15 + 5*i] = ph.user_header[i].ht_length_f[15:8];
                                    vec[16 + 5*i] = {ph.user_header[i].user_position_f,
                                                     ph.user_header[i].ht_length_f[19:16]
                                    };
                                 end
                                end
      HE_SU, HE_EXT_SU :        begin
                                  vec[13] = ph.user_header_he[0].smm_index_f;
                                  vec[14] = {ph.user_header_he[0].fec_coding_f,
                                             ph.user_header_he[0].nss_f,
                                             ph.user_header_he[0].mcs_f
                                  };
                                  vec[15] = ph.user_header_he[0].he_length_f[7:0];
                                  vec[16] = ph.user_header_he[0].he_length_f[15:8];
                                  vec[17] = {ph.user_header_he[0].pkt_ext_f,
                                             ph.user_header_he[0].he_length_f[19:16]
                                  };
                                end
      HE_MU :                   begin
                                  vec[13] = {3'b000,
                                             ph.mcs_sig_b,
                                             ph.dcm_sig_b,
                                             ph.sig_b_comp_mode
                                  };
                                  vec[14] = ph.ru_allocation;
                                  vec[15] = 0;
                                  vec[16] = 0;
                                  vec[17] = 0;
                                  vec[18] = ph.n_user;
                                  for (int i = 0; i < num_of_users; i++) begin
                                    vec[19 + 7*i] = ph.user_header_he[i].smm_index_f;
                                    vec[20 + 7*i] = {ph.user_header_he[i].fec_coding_f,
                                                     ph.user_header_he[i].nss_f,
                                                     ph.user_header_he[i].mcs_f
                                    };
                                    vec[21 + 7*i] = ph.user_header_he[i].he_length_f[7:0];
                                    vec[22 + 7*i] = ph.user_header_he[i].he_length_f[15:8];
                                    vec[23 + 7*i] = {ph.user_header_he[i].pkt_ext_f,
                                                     ph.user_header_he[i].he_length_f[19:16]
                                    };
                                    vec[24 + 7*i] = ph.user_header_he[i].staid_f[7:0];
                                    vec[25 + 7*i] = {ph.user_header_he[i].user_position_f,
                                                     ph.user_header_he[i].staid_f[10:8]
                                    };
                                  end
                                end
      HE_TB :                   begin
                                  vec[12] = {ph.spatial_reuse[1],vec[12][3:0]};
                                  vec[13] = {ph.spatial_reuse[3],
                                             ph.spatial_reuse[2]
                                  };
                                  vec[14] = ph.he_siga_reserved[7:0];
                                  vec[15] = {2'h0,
                                             ph.ldpc_extra_symbol,
                                             ph.he_ltf_mode,
                                             ph.num_he_ltf,
                                             ph.he_siga_reserved[8]
                                  };
                                  vec[16] = {5'h0, ph.starting_sts_num};
                                  vec[17] = ph.ru_allocation;
                                  vec[18] = ph.user_header_he[0].smm_index_f;
                                  vec[19] = {ph.user_header_he[0].fec_coding_f,
                                             ph.user_header_he[0].nss_f,
                                             ph.user_header_he[0].mcs_f
                                  };
                                  vec[20] = ph.user_header_he[0].he_length_f[7:0];
                                  vec[21] = ph.user_header_he[0].he_length_f[15:8];
                                  vec[22] = {ph.trigger_method,
                                             ph.user_header_he[0].pkt_ext_f,
                                             ph.user_header_he[0].he_length_f[19:16]
                                  };
                                  vec[23] = {ph.feedback_status,
                                             ph.ru_tone_set_index
                                  };
                                end
    endcase
  endtask : create_tx_vector

  //-------------------------------------------------------
  // send Tx vector to PHY
  //-------------------------------------------------------
  task send_tx_vector();
    int user_num = 1;
    int start_count = 0;

    for (int i=0; i < tx_vector.size(); i++)begin
      @(posedge vif.clk);
      vif.tx_req     <= 1'b1; // assert requenst signal and start sending data
      vif.tx_data    <= tx_vector[i];
      vif.mac_data_valid <= 1'b1;
      // When user fields are being sent, the tx_UID should have, the
      // value of the ID of the user for which those parameters apply
      if (req.frame.kind == MU_MIMO) begin
        if (req.frame.ppdu_format == VHT) begin
          if (i > 11) begin
            start_count++;
            if (start_count == 6) begin
              start_count = 0;
              vif.tx_data_uid <= user_num;
              user_num++;
            end
          end
        end
        else if (req.frame.ppdu_format== HE_MU) begin
          if (i > 18) begin
            start_count++;
            if (start_count == 7) begin
              start_count = 0;
              vif.tx_data_uid <= user_num;
              user_num++;
            end
          end
        end
      end
    end
    start_count = 0;
    // Set the UID to 0, it will be changed, when data will be sent
    vif.tx_data_uid <= 0;
    @(posedge vif.clk);
    vif.mac_data_valid <= 1'b0;
    `uvm_info(get_type_name(), "TX vector sent to PHY...", UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("Tx vector sent to PHY: %0p",tx_vector), UVM_HIGH)
  endtask : send_tx_vector

  //-------------------------------------------------------
  // send data with payload from file
  //-------------------------------------------------------
  task send_data_from_file();
    string        fname;
    int           fhandle;
    bit [7:0]     rdata, txdata;
    int           ret;
    string        line, value;

    fname = {`STRINGIFY(`SYSPARAM_TXT_DIR),"/",`PAYLOAD_NAME,`PAYLOAD_EXT};
    fhandle = $fopen(fname, "r");

    if (fhandle == 0)
      `uvm_fatal(get_type_name(), $sformatf("Unable to open %s for reading", fname))
    else
      `uvm_info(get_type_name(), $sformatf("Opened %s for reading",fname), UVM_LOW)

    // read data from file and drive pins
    while (!$feof(fhandle)) begin

      vif.tx_data_uid <= 0;
      //-----------------------------------------------------------------------
      // send A-MPDU payload
      //-----------------------------------------------------------------------
      @(posedge vif.clk or posedge vif.phy_err or posedge vif.tx_end);
      vif.mac_data_valid = delayed_mac_valid[0];
      if (delayed_mac_valid) begin
        ret = $fscanf(fhandle,"%h",rdata);
        //invert MSB and LSB since matlab stores it like that
        for (int i=0; i<8; i++) txdata[i] = rdata[7-i];
        `uvm_info(get_type_name(),$sformatf("Data from file %0h, txdata %0h",rdata,txdata),UVM_DEBUG)
        vif.tx_data <= txdata;
      end
      // when PHY error occurs stop sending data
      if (vif.phy_err == 1 || vif.tx_end) break;
    end

    @(posedge vif.clk);
    vif.mac_data_valid  <= 1'b0;
    vif.tx_data         <= 8'h0;

    $fclose(fhandle);

  endtask : send_data_from_file

  //-------------------------------------------------------
  // send data to all active users
  //-------------------------------------------------------
`ifdef STANDALONE_PHY
  task send_data(ref PPDU_frame frm);
    case (frm.kind)
      SINGLETON, AGGREGATED: begin
        send_aggregated_data(frm, 0, 0);
      end
      MU_MIMO: begin
        for (int i = 0; i < frm.num_of_users; i++) begin
          send_aggregated_data(frm, i, frm.user_order[i]);
        end
      end
      NDP: begin
        vif.tx_data <= 8'h00;
      end
    endcase
  endtask : send_data
`else
  task send_data(ref PPDU_frame frm);

    case (frm.kind)

      SINGLETON: begin
        `uvm_info(get_type_name(), $sformatf("Frame content: %0p",frm.ampdu_frame[0].mpdu_frame[0].frame), UVM_HIGH)
        if (frm.ampdu_frame[0].ppdu_format == VHT) begin
          `uvm_info(get_type_name(), $sformatf("Sending SINGLETON VHT frame."), UVM_HIGH)
          send_aggregated_data(frm, 0, 0);
        end else begin
          for (int i=0; i < frm.ampdu_frame[0].mpdu_frame[0].size();) begin

            //-----------------------------------------------------------------------
            // send MPDU frame
            //-----------------------------------------------------------------------
            @(posedge vif.clk or posedge vif.phy_err == 1'b1);
            vif.mac_data_valid <= delayed_mac_valid[0];
            if (delayed_mac_valid[0] == 1'b1) begin
              vif.tx_data <= frm.ampdu_frame[0].mpdu_frame[0].frame[i];
              i++;
            end
            // when PHY error occurs stop sending data
            if (vif.phy_err == 1'b1) return;

          end//for
          @(posedge vif.clk);
          vif.mac_data_valid   <= 1'b0;
          vif.tx_data          <= 8'h0;

          `uvm_info(get_type_name(), $sformatf("TX data sent to PHY, bytes sent %0d",
                    frm.ampdu_frame[0].mpdu_frame[0].size()), UVM_HIGH)
        end // else begin
      end

      AGGREGATED: begin
        // send data on primary chennel
        send_aggregated_data(frm, 0, 0);
      end

      MU_MIMO: begin
        fork : MU_MIMO_TX
          if (frm.num_of_users >= 1) send_aggregated_data(frm, 0, frm.user_order[0]);
          if (frm.num_of_users >= 2) send_aggregated_data(frm, 1, frm.user_order[1]);
          if (frm.num_of_users >= 3) send_aggregated_data(frm, 2, frm.user_order[2]);
          if (frm.num_of_users == 4) send_aggregated_data(frm, 3, frm.user_order[3]);
        join
      end

      NDP: begin
        vif.tx_data[0] <= 8'h00;
        vif.tx_data[1] <= 8'h00;
        vif.tx_data[2] <= 8'h00;
        vif.tx_data[3] <= 8'h00;
      end
    endcase
  endtask : send_data
`endif//STANDALONE_PHY

  //-------------------------------------------------------
  // send agreggated MPDU frame to primary or secondary user
  //-------------------------------------------------------
`ifdef STANDALONE_PHY
  task send_aggregated_data(ref PPDU_frame frm, input int ch, input int user);

    `uvm_info(get_type_name(), $sformatf("Frame content: %0p",frm.ampdu_frame[user].ampdu_payload), UVM_HIGH)
    vif.tx_data_uid <= user;
    for (int j=0; j < frm.ampdu_frame[user].ampdu_payload.size();) begin
      //-----------------------------------------------------------------------
      // send A-MPDU payload
      //-----------------------------------------------------------------------
      @(posedge vif.clk or posedge vif.phy_err);
      vif.mac_data_valid = delayed_mac_valid[0];
      if (delayed_mac_valid) begin
        vif.tx_data <= frm.ampdu_frame[user].ampdu_payload[j];
        j++;
      end
      // when PHY error occurs stop sending data
      if (vif.phy_err == 1) return;
    end//for

    @(posedge vif.clk);
    vif.mac_data_valid  <= 1'b0;
    vif.tx_data         <= 8'h0;

    `uvm_info(get_type_name(), $sformatf("U%0d on channel %0d - TX data sent to PHY, bytes sent %0d",user,ch,
            frm.ampdu_frame[user].ampdu_payload.size()), UVM_HIGH)

  endtask : send_aggregated_data
`else
  task send_aggregated_data(ref PPDU_frame frm, input int ch, input int user);
    bit [31:0]  delimiter;
    bit [31:0]  blank_delimiter;
    int         del_cnt = 0;
    int         pad_cnt = 0;

    for (int i=0; i < frm.ampdu_frame[user].mpdu_frame.size(); i++) begin

      //-----------------------------------------------------------------------
      // blank delimiter needs to be sent if it exists
      //-----------------------------------------------------------------------
      if (frm.ampdu_frame[user].blank_delimit.size() != 0) begin
        // repeat number blank delimiters as set
        repeat (frm.ampdu_frame[user].blank_delimit[i].blank_delimiter_num) begin
          blank_delimiter = frm.ampdu_frame[user].blank_delimit[i].get_data();

          del_cnt = 0;
          while (del_cnt < 4) begin
            @(posedge vif.clk or posedge vif.phy_err);
            vif.mac_data_valid = delayed_mac_valid[ch];
            if (delayed_mac_valid[ch] == 1'b1) begin
              vif.tx_data[ch] <= blank_delimiter[7:0];
              blank_delimiter = blank_delimiter >> 8;
              del_cnt++;
            end
            // when PHY error occurs stop sending data
            if (vif.phy_err == 1) return;
          end//while
        end//repeat
        `uvm_info(get_type_name(), $sformatf("blank_delimit[%0d]: \n%0p",i,frm.ampdu_frame[user].blank_delimit[i].sprint()), UVM_HIGH)
      end// if blank_delimiter exists

      //-----------------------------------------------------------------------
      // first delimiter needs to be sent
      //-----------------------------------------------------------------------
      delimiter = frm.ampdu_frame[user].delimit[i].get_data();
      `uvm_info(get_type_name(), $sformatf("delimiter[%0d]: \n%0p",i,frm.ampdu_frame[user].delimit[i].sprint()), UVM_HIGH)

      del_cnt = 0;
      while (del_cnt < 4) begin
        @(posedge vif.clk or posedge vif.phy_err);
        vif.mac_data_valid = delayed_mac_valid[ch];
        if (delayed_mac_valid[ch] == 1'b1) begin
          vif.tx_data[ch] <= delimiter[7:0];
          delimiter = delimiter >> 8;
          del_cnt++;
        end
        // when PHY error occurs stop sending data
        if (vif.phy_err == 1) return;
      end//while

      `uvm_info(get_type_name(), $sformatf("U%0d on channel %0d - Delimiter[%0d] sent",user,ch,i), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("Frame content: %0p",frm.ampdu_frame[user].mpdu_frame[i].frame), UVM_HIGH)

      for (int j=0; j < frm.ampdu_frame[user].mpdu_frame[i].size();) begin
        //-----------------------------------------------------------------------
        // send MPDU frame
        //-----------------------------------------------------------------------
        @(posedge vif.clk or posedge vif.phy_err);
        vif.mac_data_valid = delayed_mac_valid[ch];
        if (delayed_mac_valid[ch]) begin
          vif.tx_data[ch] <= frm.ampdu_frame[user].mpdu_frame[i].frame[j];
          j++;
        end
        // when PHY error occurs stop sending data
        if (vif.phy_err == 1) return;
      end//for

      `uvm_info(get_type_name(), $sformatf("U%0d on channel %0d - TX data sent to PHY, bytes sent %0d",user,ch,
              frm.ampdu_frame[user].mpdu_frame[i].size()), UVM_HIGH)

      //-----------------------------------------------------------------------
      // send padding if it is present for current MPDU frame
      //-----------------------------------------------------------------------
      if (frm.ampdu_frame[user].padding.exists(i)) begin
        // send padding bytes
        pad_cnt = 0;
        while (pad_cnt < frm.ampdu_frame[user].padding[i]) begin
          @(posedge vif.clk or posedge vif.phy_err);
          vif.mac_data_valid = delayed_mac_valid[ch];
          if (delayed_mac_valid[ch] == 1'b1) begin
            vif.tx_data[ch] <= padding_c;
            pad_cnt++;
          end
          // when PHY error occurs stop sending data
          if (vif.phy_err == 1) return;
        end//while

        `uvm_info(get_type_name(), $sformatf("U%0d on channel %0d - %0d padding bytes sent",
                  user,ch,frm.ampdu_frame[user].padding[i]), UVM_HIGH)
      end//if padding
    end//for

    @(posedge vif.clk);
    vif.mac_data_valid <= 1'b0;
    vif.tx_data[ch]        <= 8'h0;
  endtask : send_aggregated_data
`endif//STANDALONE_PHY

  //-------------------------------------------------------
  // wait for PHY to assert TX end signal indicating that
  // transaction of frame has finished
  //-------------------------------------------------------
  task wait_tx_end();
    @(posedge vif.tx_end);
    `uvm_info(get_type_name(), "TX end signal asserted by PHY...", UVM_HIGH)
    // when end signal is asserted TX req needs to go low after 1 clock cycle
    @(posedge vif.clk);
    vif.tx_req <= 1'b0;
  endtask : wait_tx_end

  //-------------------------------------------------------
  // wait for PHY to assert RX end signal indicating that
  // reception of frame has finished
  //-------------------------------------------------------
  task wait_rx_end();
    @(posedge vif.rx_end);
    `uvm_info(get_type_name(), "RX end signal asserted by PHY...", UVM_HIGH)
  endtask : wait_rx_end


  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : mac_driver

`endif
