//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef PHY_DRIVER_SV
`define PHY_DRIVER_SV


class phy_driver extends uvm_driver #(phy_seq_item);

  `uvm_component_utils(phy_driver)

  virtual mac_phy_if vif;
  phy_config         cfg;

  // signals and variables
  // Tx vectors for up to 4 users
  octet_t            tx_vector_u0[]; // TX vector contains 16 bytes of data
  octet_t            tx_vector_u1[]; // TX vector contains 16 bytes of data
  octet_t            tx_vector_u2[]; // TX vector contains 16 bytes of data
  octet_t            tx_vector_u3[]; // TX vector contains 16 bytes of data
  octet_t            tx_vector2[6];
  int                u0_data_len;
  int                u1_data_len;
  int                u2_data_len;
  int                u3_data_len;
  // Rx vectors for start and end parts
  octet_t            rx_vector_start[];
  octet_t            rx_vector_end[8];
  int                rx_vector_length;

  bit                phy_ready_gen_u0[][]; // 2 dimension array of phy ready signal per symbol
  bit                phy_ready_gen_u1[][]; // 2 dimension array of phy ready signal per symbol
  bit                phy_ready_gen_u2[][]; // 2 dimension array of phy ready signal per symbol
  bit                phy_ready_gen_u3[][]; // 2 dimension array of phy ready signal per symbol
  bit                tmp_array_short[`SHORT_LEN];
  bit                tmp_array_long[`LONG_LEN];
  bit                tmp_array_he[];

  function new (string name = "phy_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual mac_phy_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    initialize();

    fork
      get_and_drive();
      /* add other tasks here */
    join_none;
  endtask : run_phase

  task initialize();
    vif.phy_rdy            <= 0;
    vif.tx_end             <= 0;
    vif.rx_data            <= 0;
    vif.cca_primary_20     <= 0;
    vif.cca_secondary_20   <= 0;
    vif.cca_secondary_40   <= 0;
    vif.rx_end_for_timing  <= 0;
    vif.rx_err             <= 0;
    vif.rx_end             <= 0;
    vif.phy_err            <= 0;
    vif.rifs_rx_detected   <= 0;
    wait (vif.rst_n == 1'b1);
  endtask : initialize


  //-------------------------------------------------------
  // wait for new sequence item to drive signals
  //-------------------------------------------------------
  task get_and_drive();

    forever begin
      `uvm_info(get_type_name(), "Start of a bus cycle detected.", UVM_HIGH)
      seq_item_port.get_next_item(req);
      //~ tx_vector2[0] = req.time_of_departure[7:0];
      //~ tx_vector2[1] = req.time_of_departure[15:8];
      //~ tx_vector2[2] = req.time_of_departure[23:16];
      //~ tx_vector2[3] = req.time_of_departure[31:24];
      //~ tx_vector2[4] = req.time_of_departure_clk_rate[7:0];
      //~ tx_vector2[5] = req.time_of_departure_clk_rate[15:8];
      `uvm_info(get_type_name(), $sformatf("Got new PHY item %s", req.sprint()), UVM_HIGH)
      // wait for Tx or Rx request to trigger
      @(posedge vif.clk iff (   (vif.tx_req == 1'b1 && req.trans == TX && vif.mac_data_valid == 1'b1)
                             || (vif.rx_req == 1'b1 && req.trans == RX)));
      if (vif.tx_req) begin

        collect_tx_vector();
        create_phy_ready_array(1);
        #(`TX_DELAY_TO_FIRST_PHY_RDY_OFDM * 1us);

        // monitor if MAC aborts transmission
        fork : TX_DATA_THREAD
          drive_phy_ready_tx();
          begin
            @(negedge vif.tx_req);
            `uvm_warning(get_type_name(), "MAC aborted transmission!!!")
            cfg.aborted_transmission = 1'b1;
          end
        join_any

        disable fork;

        drive_tx_end();

        //send_txvector2();

        // wait MAC to end transmission
        wait (vif.tx_req == 1'b0);

      end//tx_req = 1
      else if (vif.rx_req) begin
        create_rx_vector_start(rx_vector_start, req.frame.preamble_header);
        create_rx_vector_end(rx_vector_end, req.frame.preamble_header);
        create_phy_ready_array(0);
        #(`CCA_DELAY * 1us);
        fork
          // execute in separate thread because driving
          // of CCA is independant from data on MAC-PHY vif
          drive_cca(req.frame.preamble_header);
        join_none

        if (req.err_inj == PHY_ERR_RX_VECTOR || req.err_inj == FORMAT_VIOLATION) begin
          send_rx_vector_start(req.frame.preamble_header.format_mod, rx_vector_length);
        end
        else begin
          send_rx_vector_start(req.frame.preamble_header.format_mod, rx_vector_length);
          send_rx_data(req.frame);
        end
        // wait for reception over the air to finish then send Rx vector
        drive_rx_end_for_timing();
        if (req.err_inj == NO_ERROR) begin
          send_rx_vector_end();
        end
        drive_rx_end();

      end//rx_req = 1
      seq_item_port.item_done();
    end // forever
  endtask : get_and_drive

  //-------------------------------------------------------
  // drive CCA signal in relative to channel bandwith
  //-------------------------------------------------------
  task drive_cca(PPDU_preamble_header ph);
    int frame_duration;

    @(posedge vif.clk);
    case (ph.ch_bw)
      2'b00: vif.cca_primary_20   <= 1'b1;
      2'b01: begin
        vif.cca_primary_20   <= 1'b1;
        vif.cca_secondary_20 <= 1'b1;
      end
      2'b10, 2'b11: begin
        vif.cca_primary_20   <= 1'b1;
        vif.cca_secondary_20 <= 1'b1;
        vif.cca_secondary_40 <= 1'b1;
      end
    endcase
    `uvm_info(get_type_name(), "CCA signal asserted", UVM_HIGH)

    wait(vif.rx_end_for_timing == 1);

    if (cfg.set_cca_to_busy == 0) begin
      // de-assert CCA signal after last sample in symbol is received
      @(posedge vif.clk);
      vif.cca_primary_20     <= 0;
      vif.cca_secondary_20   <= 0;
      vif.cca_secondary_40   <= 0;
      `uvm_info(get_type_name(), "CCA signal de-asserted", UVM_HIGH)
    end

  endtask : drive_cca

  //-------------------------------------------------------
  // send Rx vector 1 (start) to MAC-PHY interface
  //-------------------------------------------------------
  task send_rx_vector_start(format_mod_e mode, int rx_vector_length);

    // time to wait before sending Rx vector
    case (mode)
      NON_HT, NON_HT_DUP_OFDM, HT_MF, VHT: begin
        #(`OFDM_NON_HT_PRE*1us);
      end
      HT_GF: begin
        #(`OFDM_HT_GF_PRE_BASE*1us);
      end
      HE_SU: begin
        #(`HE_SU_PRE_BASE*1us);
      end
      HE_MU: begin
        #(`HE_MU_PRE_BASE*1us);
      end
      HE_EXT_SU: begin
        #(`HE_ER_SU_PRE_BASE*1us);
      end
      HE_TB: begin
        #(`HE_TB_PRE_BASE*1us);
      end
    endcase

    for (int i=0; i < rx_vector_length; i++) begin
      @(posedge vif.clk);
      vif.phy_rdy <= 1'b1;
      vif.rx_data    <= rx_vector_start[i];
      // for HT_MF and VHT 1st byte is send then
      // wait specific amount of time before sending
      // rest of Rx vector bytes to MAC
      if (i == 0 && (mode == HT_MF || mode == VHT)) begin
        if (mode == HT_MF) begin
          @(posedge vif.clk);
          vif.phy_rdy <= 1'b0;
          vif.rx_data    <= 8'h0;
          #((`OFDM_HT_MM_PRE_BASE-`OFDM_NON_HT_PRE)*1us);
        end
        else if (mode == VHT) begin
          @(posedge vif.clk);
          vif.phy_rdy <= 1'b0;
          vif.rx_data    <= 8'h0;
          #((`OFDM_VHT_PRE_BASE-`OFDM_NON_HT_PRE)*1us);
        end
      end//if
    end//for

    `uvm_info(get_type_name(), "Rx vector 1 sent", UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("Rx vector part 1: %0p",rx_vector_start), UVM_HIGH)

    // stop sending data
    @(posedge vif.clk);
    vif.phy_rdy <= 1'b0;
    vif.rx_data    <= 8'h0;
  endtask : send_rx_vector_start

  //-------------------------------------------------------
  // send Rx vector 2 (end) to MAC-PHY interface
  //-------------------------------------------------------
  task send_rx_vector_end();
    for (int i=0; i < 8; i++) begin
      @(posedge vif.clk);
      vif.phy_rdy <= 1'b1;
      vif.rx_data <= rx_vector_end[i];
    end//for

    `uvm_info(get_type_name(), "Rx vector 2 sent", UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("Rx vector part 2: %0p",rx_vector_end), UVM_HIGH)

    // stop sending data
    @(posedge vif.clk);
    vif.phy_rdy <= 1'b0;
    vif.rx_data    <= 8'h0;
  endtask : send_rx_vector_end

  //-------------------------------------------------------
  // send Rx data to MAC-PHY interface
  //-------------------------------------------------------
  task send_rx_data(PPDU_frame frm);
    int         data_cnt = 0;
    int         del_cnt  = 0;
    int         pad_cnt  = 0;
    int         mpdu_cnt = 0;
    bit [31:0]  delimiter;
    bit         del_finished = 0; //finished sending delimiter
    bit         frm_finished = 0; //finished MPDU frame sending
    bit         pad_finished = 0; //finished padding
    int         accu;
    ppdu_frame_kind_e tmp_kind;
    int         tmp_cnt = 0;

    // in cases when we send VHT and HE singleton frame it is sent as aggregated
    if (frm.kind == SINGLETON && frm.ppdu_format >= VHT)
      tmp_kind = AGGREGATED;
    else
      tmp_kind = frm.kind;

    case (tmp_kind)

      SINGLETON: begin
        foreach (phy_ready_gen_u0[symbol]) begin
          foreach (phy_ready_gen_u0[symbol][valid]) begin

            @(posedge vif.clk);
            vif.phy_rdy <= phy_ready_gen_u0[symbol][valid];
            if (phy_ready_gen_u0[symbol][valid] == 1'b1)
              vif.rx_data <= frm.ampdu_frame[0].mpdu_frame[0].frame[data_cnt++];
            else
              vif.rx_data <= 8'h0;
          end//foreach
          if (req.err_inj inside {PHY_ERR_BEFORE_RX_END, UNSUPPORTED_RATE} && data_cnt > 14) begin
            if (req.err_inj == UNSUPPORTED_RATE) begin
              vif.rx_err <= 1'b1;
              @(posedge vif.clk);
              vif.rx_err <= 1'b0;
            end
            break; //foreach
          end
        end//foreach
      end//SINGLETON

      AGGREGATED: begin
        delimiter = frm.ampdu_frame[0].delimit[mpdu_cnt].get_data();
        `uvm_info(get_type_name(), $sformatf("Delimiter[%0d]: %0h", mpdu_cnt,delimiter), UVM_HIGH)
        `uvm_info(get_type_name(),
          $sformatf("MPDU_FRAME[%0d]: %0p", mpdu_cnt,frm.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame), UVM_HIGH)
        `uvm_info(get_type_name(),
          $sformatf("Padding for [%0d] present: %0d", mpdu_cnt, frm.ampdu_frame[0].padding.exists(mpdu_cnt)), UVM_HIGH)
        `uvm_info(get_type_name(),
          $sformatf("Blank delimiter for [%0d] present: %0d",
          mpdu_cnt, frm.ampdu_frame[0].blank_delimit.exists(mpdu_cnt)), UVM_HIGH)

        foreach (phy_ready_gen_u0[symbol]) begin
          foreach (phy_ready_gen_u0[symbol][valid]) begin

            @(posedge vif.clk);
            vif.phy_rdy <= phy_ready_gen_u0[symbol][valid];

            // when valid is 1 send first delimiter then frame then padding,
            // if present
            if (phy_ready_gen_u0[symbol][valid] == 1'b1) begin
              if (del_finished == 1'b0) begin
                vif.rx_data <= delimiter[7:0];
                delimiter    = delimiter >> 8;
                del_cnt++;
                if (del_cnt == 4) del_finished = 1'b1;
              end
              else if (del_finished == 1'b1 && frm_finished == 1'b0) begin
                vif.rx_data <= frm.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame[data_cnt++];
                if (data_cnt == frm.ampdu_frame[0].mpdu_frame[mpdu_cnt].size()) begin
                  frm_finished = 1'b1;
                  del_cnt = 0;
                end
              end
              // send padding bytes if delimiter and frame are sent and
              // padding exist for that MPDU frame
              else if (   frm_finished == 1'b1
                       && del_finished == 1'b1
                       && pad_finished == 1'b0
                       && frm.ampdu_frame[0].padding.exists(mpdu_cnt)) begin

                // padding is sent at the end of frame, if needed
                vif.rx_data <= 8'h0;
                pad_cnt++;
                if (pad_cnt == frm.ampdu_frame[0].padding[mpdu_cnt]) begin

                  // check if blank delimiter is present
                  if (frm.ampdu_frame[0].blank_delimit.exists(mpdu_cnt)) begin
                    pad_finished = 1'b1;
                    del_cnt = 0;
                  end
                  // fetch new data only if they are MPDUs left
                  else if ((mpdu_cnt+1) < frm.ampdu_frame[0].mpdu_frame_type.size()) begin
                    // reset all flags before sending next MDPU frame
                    del_cnt  = 0;
                    del_finished = 0;
                    data_cnt = 0;
                    frm_finished = 0;
                    pad_cnt = 0;
                    pad_finished = 0;
                    mpdu_cnt++; // send next MPDU frame
                    // get delimiter data
                    delimiter = frm.ampdu_frame[0].delimit[mpdu_cnt].get_data();
                    `uvm_info(get_type_name(), $sformatf("Delimiter[%0d]: %0h", mpdu_cnt,delimiter), UVM_HIGH)
                    `uvm_info(get_type_name(),
                      $sformatf("MPDU_FRAME[%0d]: %0p", mpdu_cnt,frm.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame), UVM_HIGH)
                    `uvm_info(get_type_name(),
                      $sformatf("Padding for [%0d] present: %0d", mpdu_cnt,frm.ampdu_frame[0].padding.exists(mpdu_cnt)), UVM_HIGH)
                    `uvm_info(get_type_name(),
                      $sformatf("Blank delimiter for [%0d] present: %0d",
                      mpdu_cnt, frm.ampdu_frame[0].blank_delimit.exists(mpdu_cnt)), UVM_HIGH)
                  end
                end
              end
              // send blank delimietr bytes if delimiter, frame and paddidng
              // are sent and padding exist for that MPDU frame
              else if (   frm_finished == 1'b1
                       && del_finished == 1'b1
                       && frm.ampdu_frame[0].blank_delimit.exists(mpdu_cnt)) begin

                delimiter = frm.ampdu_frame[0].blank_delimit[mpdu_cnt].get_data();
                case (del_cnt % 4)
                  0: vif.rx_data <= delimiter[ 7: 0];
                  1: vif.rx_data <= delimiter[15: 8];
                  2: vif.rx_data <= delimiter[23:16];
                  3: vif.rx_data <= delimiter[31:24];
                endcase
                del_cnt++;
                if (del_cnt == frm.ampdu_frame[0].blank_delimit[mpdu_cnt].size()) begin
                  // reset all flags before sending next MDPU frame
                  del_cnt  = 0;
                  del_finished = 0;
                  data_cnt = 0;
                  frm_finished = 0;
                  pad_cnt = 0;
                  pad_finished = 0;

                  // fetch new data only if they are MPDUs left
                  if ((mpdu_cnt+1) < frm.ampdu_frame[0].mpdu_frame_type.size()) begin
                    mpdu_cnt++; // send next MPDU frame
                    delimiter = frm.ampdu_frame[0].delimit[mpdu_cnt].get_data();
                    `uvm_info(get_type_name(), $sformatf("Delimiter[%0d]: %0h", mpdu_cnt,delimiter), UVM_HIGH)
                    `uvm_info(get_type_name(), $sformatf("MPDU_FRAME[%0d]: %0p",
                      mpdu_cnt,frm.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame), UVM_HIGH)
                    `uvm_info(get_type_name(), $sformatf("Padding for [%0d] present: %0d",
                      mpdu_cnt,frm.ampdu_frame[0].padding.exists(mpdu_cnt)), UVM_HIGH)
                    `uvm_info(get_type_name(),
                      $sformatf("Blank delimiter for [%0d] present: %0d",
                      mpdu_cnt, frm.ampdu_frame[0].blank_delimit.exists(mpdu_cnt)), UVM_HIGH)
                  end
                end
              end
              // when there is no padding and blank delimiters reset all flags
              else begin
                // reset all flags before sending next MDPU frame
                del_cnt  = 0;
                del_finished = 0;
                data_cnt = 0;
                frm_finished = 0;
                pad_cnt = 0;
                pad_finished = 0;
                mpdu_cnt++; // send next MPDU frame

                // fetch new data only if they are MPDUs left
                if (mpdu_cnt < frm.ampdu_frame[0].mpdu_frame_type.size()) begin
                  delimiter = frm.ampdu_frame[0].delimit[mpdu_cnt].get_data();
                  `uvm_info(get_type_name(), $sformatf("Delimiter[%0d]: %0h", mpdu_cnt,delimiter), UVM_HIGH)
                  `uvm_info(get_type_name(),
                    $sformatf("MPDU_FRAME[%0d]: %0p", mpdu_cnt,frm.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame), UVM_HIGH)
                  `uvm_info(get_type_name(),
                    $sformatf("Padding for [%0d] present: %0d", mpdu_cnt,frm.ampdu_frame[0].padding.exists(mpdu_cnt)), UVM_HIGH)
                  `uvm_info(get_type_name(),
                    $sformatf("Blank delimiter for [%0d] present: %0d",
                    mpdu_cnt, frm.ampdu_frame[0].blank_delimit.exists(mpdu_cnt)), UVM_HIGH)
                  // when there is not padding present for MDPU, next byte
                  // sent needs to be first byte of delimiter
                  vif.rx_data <= delimiter[7:0];
                  delimiter    = delimiter >> 8;
                  del_cnt++;
                end
              end
            end //phy_rdy == 1
            else begin // when valid = 0 no data is sent
              vif.rx_data <= 8'h0;
            end
            // when error is injected
            if (req.err_inj inside {PHY_ERR_BEFORE_RX_END, UNSUPPORTED_RATE} && data_cnt > 14) begin
              break;
            end
          end//foreach valid

          if (req.err_inj inside {PHY_ERR_BEFORE_RX_END, UNSUPPORTED_RATE} && data_cnt > 14) begin
            if (req.err_inj == UNSUPPORTED_RATE) begin
              vif.rx_err <= 1'b1;
              @(posedge vif.clk);
              vif.rx_err <= 1'b0;
            end
            break; //foreach
          end

        end//foreach symbol

      end//AGGREGATED

      MU_MIMO: `uvm_error(get_type_name(), "MU-MIMO receive is not supported!")

      NDP: vif.rx_data <= 8'h00;
    endcase

    // put PHY ready to low when all samples are driven
    if (req.err_inj == NO_ERROR)
      @(posedge vif.clk);
    vif.phy_rdy <= 1'b0;

  endtask : send_rx_data

  //-------------------------------------------------------
  // create Rx vectors for the start of reception
  //-------------------------------------------------------
  task create_rx_vector_start(output octet_t vec[], input PPDU_preamble_header ph);
    // combine all fields of Rx vector
     case (ph.format_mod)
        NON_HT, NON_HT_DUP_OFDM : begin
          rx_vector_length = 7;
        end
        HT_MF, HT_GF : begin
          rx_vector_length = 10;
        end
        VHT : begin
          rx_vector_length = 13;
        end
        HE_SU, HE_EXT_SU : begin
          rx_vector_length = 16;
        end
        HE_MU : begin
          rx_vector_length = 16;
        end
        HE_TB : begin
          // as AP
          rx_vector_length = 12 + ph.user_header_he.size()*6;
          // as STA
          //rx_vector_length = 12;
          // as HE-NDP-TRIG
          //rx_vector_length = 30;
        end
      endcase

    vec = new[rx_vector_length];

    vec[0][3:0]   = ph.format_mod;
    vec[0][6:4]   = ph.ch_bw;
    vec[0][7]     = ph.preamble_type;
    vec[1]        = ph.antenna_set;
    vec[2]        = ph.rssi_legacy;
    vec[3]        = ph.leg_length[7:0];
    vec[4][3:0]   = ph.leg_length[11:8];
    vec[4][7:4]   = ph.leg_rate;
    vec[5]        = ph.rssi;

    if (ph.format_mod inside {NON_HT, NON_HT_DUP_OFDM, HT_MF, HT_GF, VHT}) begin
      case(ph.format_mod)
        NON_HT, NON_HT_DUP_OFDM: begin
          vec[6][0]   = ph.dyn_bw;
          vec[6][2:1] = ph.chbw_in_non_ht;
          vec[6][7]   = ph.l_sig_valid;
        end
        HT_MF, HT_GF: begin
          vec[6][0]   = ph.sounding;
          vec[6][1]   = ph.smoothing;
          vec[6][2]   = ph.gi_type[0];
          vec[6][3]   = ph.aggregated;
          vec[6][4]   = ph.stbc;
          vec[6][6:5] = ph.num_extn_ss;
          vec[6][7]   = ph.l_sig_valid;
          vec[7][6:0] = ph.user_header[0].mcs_f;
          vec[7][7]   = ph.user_header[0].fec_coding_f;
          vec[8]      = ph.user_header[0].ht_length_f[7:0];
          vec[9]      = ph.user_header[0].ht_length_f[15:8];
        end
        VHT: begin
          vec[6][0]   = ph.sounding;
          vec[6][1]   = ph.beamformed;
          vec[6][2]   = ph.gi_type[0];
          vec[6][4]   = ph.stbc;
          vec[6][5]   = ph.doze_not_allowed;
          vec[6][6]   = ph.first_user;
          vec[7]      = ph.partial_aid[7:0];
          vec[8][0]   = ph.partial_aid[8];
          vec[8][6:1] = ph.group_id;
          vec[9][3:0] = ph.user_header[0].mcs_f[3:0];
          vec[9][6:4] = (ph.stbc) ? ((ph.user_header[0].num_sts_f+1)/2)-1 : ph.user_header[0].num_sts_f;
          vec[9][7]   = ph.user_header[0].fec_coding_f;
          vec[10]     = ph.user_header[0].ht_length_f[7:0];
          vec[11]     = ph.user_header[0].ht_length_f[15:8];
          vec[12]     = ph.user_header[0].ht_length_f[19:16];
        end
      endcase
    // HE format mod
    end else begin
      vec[6][0]     = ph.sounding;
      vec[6][1]     = ph.beamformed;
      vec[6][3:2]   = ph.gi_type;
      vec[6][4]     = ph.stbc;
      vec[7][0]     = ph.uplink_flag;
      vec[7][1]     = ph.beam_change;
      vec[7][2]     = ph.dcm;
      vec[7][4:3]   = ph.he_ltf_type;
      vec[7][5]     = ph.doppler;
      vec[8][5:0]   = ph.bss_color;
      vec[9][6:0]   = ph.txop_duration;

      case(ph.format_mod)
        HE_SU, HE_EXT_SU: begin
          vec[10][3:0]  = ph.pe_duration;
          vec[10][7:4]  = ph.spatial_reuse[0];
          vec[11]       = 0; // Reserved
          vec[12][3:0]  = ph.user_header_he[0].mcs_f;
          vec[12][6:4]  = ph.user_header_he[0].nss_f;
          vec[12][7]    = ph.user_header_he[0].fec_coding_f;
          vec[13]       = ph.user_header_he[0].he_length_f[7:0];
          vec[14]       = ph.user_header_he[0].he_length_f[15:8];
          vec[15]       = ph.user_header_he[0].he_length_f[19:16];
        end
        HE_MU: begin
          vec[10][3:0]  = ph.pe_duration;
          vec[10][7:4]  = ph.spatial_reuse[0];
          vec[11][0]    = ph.sig_b_comp_mode;
          vec[11][1]    = ph.dcm_sig_b;
          vec[11][4:2]  = ph.mcs_sig_b;
          vec[11][7:5]  = get_ru_type_from_ru_allocation(ph.ru_allocation,
                                                         ph.user_header_he[0].dut_location_f);
          vec[12][3:0]  = ph.user_header_he[0].mcs_f;
          vec[12][6:4]  = ph.user_header_he[0].nss_f;
          vec[12][7]    = ph.user_header_he[0].fec_coding_f;
          vec[13]       = ph.user_header_he[0].he_length_f[7:0];
          vec[14]       = ph.user_header_he[0].he_length_f[15:8];
          vec[15]       = ph.user_header_he[0].he_length_f[19:16];
        end
        HE_TB: begin
          // as AP
          vec[10] = 0;
          vec[11] = ph.n_user;
          foreach (ph.user_header_he[i]) begin
            vec[12+6*i][3:0]  = ph.user_header_he[0].mcs_f;
            vec[12+6*i][6:4]  = ph.user_header_he[0].nss_f;
            vec[12+6*i][7]    = ph.user_header_he[0].fec_coding_f;
            vec[13+6*i]       = ph.user_header_he[0].he_length_f[7:0];
            vec[14+6*i]       = ph.user_header_he[0].he_length_f[15:8];
            vec[15+6*i]       = ph.user_header_he[0].he_length_f[19:16];
            vec[16+6*i]       = ph.user_header_he[0].staid_f[7:0];
            vec[17+6*i][2:0]  = ph.user_header_he[0].staid_f[10:8];
          end
          // as STA
          // vec[10] = ph.spatial_reuse[0];
          // vec[11] = ph.spatial_reuse[1];
          // HE-NDP-TRIG
          // vec[10] = 0;
          // vec[11] = 0; // N_USER = 0 to indicate NDP report
          // vec[29:12] = NDP report data;
        end
      endcase
    end
  endtask : create_rx_vector_start

  //-------------------------------------------------------
  // create Rx vector for the end of reception
  //-------------------------------------------------------
  task create_rx_vector_end(output octet_t vec[8], input PPDU_preamble_header ph);
    // combine all fields of Rx vector
    vec[0] = ph.rcpi[0];
    vec[1] = ph.rcpi[1];
    vec[2] = ph.rcpi[2];
    vec[3] = ph.rcpi[3];
    vec[4] = ph.evm[0];
    vec[5] = ph.evm[1];
    vec[6] = ph.evm[2];
    vec[7] = ph.evm[3];
  endtask : create_rx_vector_end

  //-------------------------------------------------------
  // collect Tx vectors and then calculate duration and
  // legth of trasmitted frame
  //-------------------------------------------------------
  task collect_tx_vector();
    int tx_vec_len = 0;
    int tx_vector_length;

    // determine format mode
    case (format_mod_e'(vif.tx_data[3:0]))
      NON_HT, NON_HT_DUP_OFDM : begin
        tx_vector_length = 10;
      end
      HT_MF, HT_GF : begin
        tx_vector_length = 13;
      end
      VHT : begin
        tx_vector_length = 17;
      end
      HE_SU, HE_EXT_SU : begin
        tx_vector_length = 18;
      end
      HE_MU : begin
        tx_vector_length = 19;
      end
      HE_TB : begin
        tx_vector_length = 24;
      end
    endcase

    tx_vector_u0 = new[tx_vector_length];
    tx_vector_u1 = new[tx_vector_length];
    tx_vector_u2 = new[tx_vector_length];
    tx_vector_u3 = new[tx_vector_length];

    // collect Tx data from all users in parallel
    while (tx_vec_len < tx_vector_length) begin
      if (tx_vec_len != 0)
        @(posedge vif.clk iff vif.mac_data_valid);

      case (vif.tx_data_uid)
        0 : tx_vector_u0[tx_vec_len] = vif.tx_data;
        1 : tx_vector_u1[tx_vec_len] = vif.tx_data;
        2 : tx_vector_u2[tx_vec_len] = vif.tx_data;
        3 : tx_vector_u3[tx_vec_len] = vif.tx_data;
      endcase
      tx_vec_len++;
    end//while
    `uvm_info(get_type_name(), $sformatf("Tx vector 0 collected by PHY: %0p",tx_vector_u0), UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("Tx vector 1 collected by PHY: %0p",tx_vector_u1), UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("Tx vector 2 collected by PHY: %0p",tx_vector_u2), UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("Tx vector 3 collected by PHY: %0p",tx_vector_u3), UVM_HIGH)


    // determine Tx data len per user, for NON_HT format use legacy length
    // for HT format use HT length parameter
    u0_data_len = (tx_vector_u0[0][3:0] inside {NON_HT,NON_HT_DUP_OFDM}) ?
                   {tx_vector_u0[5][3:0],  tx_vector_u0[4]} :                     // when NON_HT use legacy length
                   (tx_vector_u0[0][3:0] inside {HT_GF, HT_MF}) ?
                   {tx_vector_u0[12], tx_vector_u0[11]} :                         // HT-MF and HT-GF
                   (tx_vector_u0[0][3:0] == VHT) ?
                   {tx_vector_u0[16][3:0], tx_vector_u0[15], tx_vector_u0[14]} :  // VHT
                   (tx_vector_u0[0][3:0] inside {HE_SU, HE_EXT_SU}) ?
                   {tx_vector_u0[17][3:0], tx_vector_u0[16], tx_vector_u0[15]} :  // HE_SU, HE_EXT_SU
                   (tx_vector_u0[0][3:0] == HE_MU) ?
                   {tx_vector_u0[23][3:0], tx_vector_u0[22], tx_vector_u0[21]} :  // HE-MU
                   {tx_vector_u0[22][3:0], tx_vector_u0[21], tx_vector_u0[20]};   // HE-TRIG
    `uvm_info(get_type_name(), $sformatf("U0 data length %0d",u0_data_len), UVM_HIGH)

    u1_data_len = (tx_vector_u1[0][3:0] inside {NON_HT,NON_HT_DUP_OFDM}) ?
                   {tx_vector_u1[5][3:0],  tx_vector_u1[4]} :                     // when NON_HT use legacy length
                   (tx_vector_u1[0][3:0] inside {HT_GF, HT_MF}) ?
                   {tx_vector_u1[12], tx_vector_u1[11]} :                         // HT-MF and HT-GF
                   (tx_vector_u1[0][3:0] == VHT) ?
                   {tx_vector_u1[16][3:0], tx_vector_u1[15], tx_vector_u1[14]} :  // VHT
                   (tx_vector_u1[0][3:0] inside {HE_SU, HE_EXT_SU}) ?
                   {tx_vector_u1[17][3:0], tx_vector_u1[16], tx_vector_u1[15]} :  // HE_SU, HE_EXT_SU
                   (tx_vector_u1[0][3:0] == HE_MU) ?
                   {tx_vector_u1[23][3:0], tx_vector_u1[22], tx_vector_u1[21]} :  // HE-MU
                   {tx_vector_u1[22][3:0], tx_vector_u1[21], tx_vector_u1[20]};   // HE-TRIG
    `uvm_info(get_type_name(), $sformatf("U1 data length %0d",u1_data_len), UVM_HIGH)

    u2_data_len = (tx_vector_u2[0][3:0] inside {NON_HT,NON_HT_DUP_OFDM}) ?
                   {tx_vector_u2[5][3:0],  tx_vector_u2[4]} :                     // when NON_HT use legacy length
                   (tx_vector_u2[0][3:0] inside {HT_GF, HT_MF}) ?
                   {tx_vector_u2[12], tx_vector_u2[11]} :                         // HT-MF and HT-GF
                   (tx_vector_u2[0][3:0] == VHT) ?
                   {tx_vector_u2[16][3:0], tx_vector_u2[15], tx_vector_u2[14]} :  // VHT
                   (tx_vector_u2[0][3:0] inside {HE_SU, HE_EXT_SU}) ?
                   {tx_vector_u2[17][3:0], tx_vector_u2[16], tx_vector_u2[15]} :  // HE_SU, HE_EXT_SU
                   (tx_vector_u2[0][3:0] == HE_MU) ?
                   {tx_vector_u2[23][3:0], tx_vector_u2[22], tx_vector_u2[21]} :  // HE-MU
                   {tx_vector_u2[22][3:0], tx_vector_u2[21], tx_vector_u2[20]};   // HE-TRIG
    `uvm_info(get_type_name(), $sformatf("U2 data length %0d",u2_data_len), UVM_HIGH)

    u3_data_len = (tx_vector_u3[0][3:0] inside {NON_HT,NON_HT_DUP_OFDM}) ?
                   {tx_vector_u3[5][3:0],  tx_vector_u3[4]} :                     // when NON_HT use legacy length
                   (tx_vector_u3[0][3:0] inside {HT_GF, HT_MF}) ?
                   {tx_vector_u3[12], tx_vector_u3[11]} :                         // HT-MF and HT-GF
                   (tx_vector_u3[0][3:0] == VHT) ?
                   {tx_vector_u3[16][3:0], tx_vector_u3[15], tx_vector_u3[14]} :  // VHT
                   (tx_vector_u3[0][3:0] inside {HE_SU, HE_EXT_SU}) ?
                   {tx_vector_u3[17][3:0], tx_vector_u3[16], tx_vector_u3[15]} :  // HE_SU, HE_EXT_SU
                   (tx_vector_u3[0][3:0] == HE_MU) ?
                   {tx_vector_u3[23][3:0], tx_vector_u3[22], tx_vector_u3[21]} :  // HE-MU
                   {tx_vector_u3[22][3:0], tx_vector_u3[21], tx_vector_u3[20]};   // HE-TRIG
    `uvm_info(get_type_name(), $sformatf("U3 data length %0d",u3_data_len), UVM_HIGH)

  endtask : collect_tx_vector

  //-------------------------------------------------------
  // create PHY ready signal array for driving ready signal
  // on interface, with random content. It is a matrix of
  // 2 dimensions of number of symbols to send in duration
  // of 1us * 4 (3.6) regarding of long or short GI
  // @tx - for Tx or Rx path (1/0)
  //-------------------------------------------------------
  task create_phy_ready_array(int tx);
    int         num_of_bits_in_symbol;
    int         num_of_bytes_in_symbol;
    int         num_of_symbols_in_frame;
    int         Nss; // number of spatial streams
    int         symbol_time;
    int         ones;
    int         mcs_index;
    int         format;
    int         ch_bw;
    int         n_sts;
    int         stbc;
    bit [1:0]   gi_type;
    bit [3:0]   leg_rate;
    bit         dcm;
    int         sym_dur;
    ru_type_e   ru_type;
    bit [7:0]   ru_allocation;

    phy_ready_gen_u0.delete();
    phy_ready_gen_u1.delete();
    phy_ready_gen_u2.delete();
    phy_ready_gen_u3.delete();

    //--------------------------------------------------
    // PHY ready for user 0
    //--------------------------------------------------
    if (u0_data_len != 0 && tx == 1'b1) begin

      format = tx_vector_u0[0][3:0];
      case (format)
        NON_HT, NON_HT_DUP_OFDM: begin
          mcs_index   = 0;
          ch_bw       = tx_vector_u0[0][6:4];
          n_sts       = 0;
          stbc        = 0;
          dcm         = 0;
          gi_type     = tx_vector_u0[0][7];
        end
        HT_MF, HT_GF: begin
          mcs_index   = tx_vector_u0[10][6:0];
          ch_bw       = tx_vector_u0[0][6:4];
          n_sts       = 0;
          stbc        = tx_vector_u0[8][4];
          gi_type     = tx_vector_u0[8][2];
          dcm         = 0;
        end
        VHT: begin
          mcs_index   = tx_vector_u0[13][3:0];
          format      = tx_vector_u0[0][3:0];
          ch_bw       = tx_vector_u0[0][6:4];
          n_sts       = tx_vector_u0[13][6:4];
          stbc        = tx_vector_u0[8][4];
          gi_type     = tx_vector_u0[8][2];
          dcm         = 0;
        end
        HE_SU, HE_EXT_SU: begin
          mcs_index   = tx_vector_u0[14][3:0];
          ch_bw       = tx_vector_u0[0][6:4];
          n_sts       = tx_vector_u0[14][6:4];
          stbc        = tx_vector_u0[8][4];
          gi_type     = tx_vector_u0[8][3:2];
          dcm         = tx_vector_u0[9][2];
        end
        HE_MU: begin
          mcs_index   = tx_vector_u0[20][3:0];
          ch_bw       = tx_vector_u0[0][6:4];
          n_sts       = tx_vector_u0[20][6:4];
          stbc        = tx_vector_u0[8][4];
          gi_type     = tx_vector_u0[8][3:2];
          dcm         = tx_vector_u0[9][2];
          ru_allocation = tx_vector_u0[14];
        end
        HE_TB: begin
          mcs_index   = tx_vector_u0[19][3:0];
          ch_bw       = tx_vector_u0[0][6:4];
          n_sts       = tx_vector_u0[19][6:4];
          stbc        = tx_vector_u0[8][4];
          gi_type     = tx_vector_u0[8][3:2];
          dcm         = tx_vector_u0[9][2];
          ru_allocation = tx_vector_u0[17];
        end
      endcase
      leg_rate = tx_vector_u0[5][7:4];

      Nss = get_num_ss_func(mcs_index,    //txMCS
                            format,       //txFormat
                            ch_bw,        //txChBW
                            n_sts,        //txnSTS
                            stbc);        //txSTBC
      `uvm_info(get_type_name(), $sformatf("Number of spatial streams Nss = %0d",Nss), UVM_HIGH)

      if (format ==  HE_MU)
        ru_type = get_ru_type_from_ru_allocation(ru_allocation, 0);
      else if (format == HE_TB)
        ru_type = get_ru_type(ru_allocation);
      else
        ru_type = get_ru_type_he_su(ch_bw);

      `uvm_info(get_type_name(), $sformatf("txFormatFn  = %0h", format), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txLRateFn   = %0h", leg_rate), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txChBWFn    = %0h", ch_bw), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txMCSFn     = %0h", mcs_index), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txRUType    = %s", ru_type.name()), UVM_HIGH)

      num_of_bits_in_symbol = get_bits_in_one_symbol_func(Nss,        //txNumSSFn
                                                          format,     //txFormatFn
                                                          leg_rate,   //txLRateFn
                                                          ch_bw,      //txChBWFn
                                                          mcs_index,  //txMCSFn
                                                          dcm,        //txDCMFn
                                                          ru_type);   // RUtype

      // if num of bytes is not rounded then apply floor function
      num_of_bytes_in_symbol  = (num_of_bits_in_symbol % 8) ?
                                 $floor(real'(num_of_bits_in_symbol)/8) :
                                 (num_of_bits_in_symbol/8);
      // when there are less then 8bits in symbol make minimum 1 byte
      if (num_of_bytes_in_symbol == 0) num_of_bytes_in_symbol = 1;

      num_of_symbols_in_frame = (u0_data_len % num_of_bytes_in_symbol) ?
                                 $ceil(real'(real'(u0_data_len)) / num_of_bytes_in_symbol) :
                                 (u0_data_len / num_of_bytes_in_symbol);

      `uvm_info(get_type_name(), $sformatf("u0_data_len = %d", u0_data_len), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("num_of_bytes_in_symbol = %d", num_of_bytes_in_symbol), UVM_HIGH)
      `uvm_info(get_type_name(),
        $sformatf("~ u0 ~ Number of bits in symbol %0d\nNumber of bytes in symbol %0d\nNumber of symbols in frame %0d",
        num_of_bits_in_symbol,num_of_bytes_in_symbol,num_of_symbols_in_frame), UVM_HIGH)



      // create and randomize ready signals
      phy_ready_gen_u0 = new[num_of_symbols_in_frame];
      foreach (phy_ready_gen_u0[i]) begin
        // randomize number of PHY ready signals (value 1) per symbol not to be larger
        // then number of bytes per symbol
        // determine if short or long GI
        if (format inside {NON_HT, NON_HT_DUP_OFDM, HT_GF, HT_MF, VHT}) begin
          if (gi_type) begin // SHORT GI

            symbol_time = `SHORT_LEN;
            assert (randomize(tmp_array_short) with { (tmp_array_short.sum(b) with (int'(b)) == num_of_bytes_in_symbol); });
            phy_ready_gen_u0[i] = new[symbol_time](tmp_array_short); // create array with tmp_array_short content

          end
          else begin // LONG GI

            symbol_time = `LONG_LEN;
            assert (randomize(tmp_array_long) with { (tmp_array_long.sum(b) with (int'(b)) == num_of_bytes_in_symbol); });
            phy_ready_gen_u0[i] = new[symbol_time](tmp_array_long); // create array with tmp_array_long content

          end
        end
        else begin
          symbol_time = `HE_LEN + (`GI_LEN << gi_type);
          tmp_array_he = new[symbol_time];
          assert (randomize(tmp_array_he) with { (tmp_array_he.sum(b) with (int'(b)) == num_of_bytes_in_symbol);});
          phy_ready_gen_u0[i] = new[symbol_time](tmp_array_he); // create array with tmp_array_long content
        end

      end//foreach

      // put to 0 PHY ready that are overabundant
      ones = 0;
      foreach (phy_ready_gen_u0[i]) begin
        foreach (phy_ready_gen_u0[i][j]) begin
          if (phy_ready_gen_u0[i][j] == 1'b1 && ones < u0_data_len)
            ones++;
          else if (phy_ready_gen_u0[i][j] == 1'b1 && ones == u0_data_len)
            phy_ready_gen_u0[i][j] = 1'b0; // set overabundant to 0
        end
        phy_ready_gen_u0[i].shuffle();
      end//foreach

    end// user 0

    // generate PHY ready for Rx path
    else if (tx == 1'b0) begin
      format = rx_vector_start[0][3:0];

      case (format)
        NON_HT, NON_HT_DUP_OFDM: begin
          mcs_index   = 0;
          ch_bw       = rx_vector_start[0][6:4];
          n_sts       = 0;
          stbc        = 0;
          dcm         = 0;
          gi_type     = rx_vector_start[0][7];
        end
        HT_MF, HT_GF: begin
          mcs_index   = rx_vector_start[7][6:0];
          ch_bw       = rx_vector_start[0][6:4];
          n_sts       = 0;
          stbc        = rx_vector_start[6][4];
          gi_type     = rx_vector_start[6][2];
          dcm         = 0;
        end
        VHT: begin
          mcs_index   = rx_vector_start[9][3:0];
          ch_bw       = rx_vector_start[0][6:4];
          stbc        = rx_vector_start[6][4];
          gi_type     = rx_vector_start[6][2];
          n_sts       = (2**stbc)*(rx_vector_start[9][6:4]+1)-1;
          dcm         = 0;
        end
        HE_SU, HE_EXT_SU: begin
          mcs_index   = rx_vector_start[12][3:0];
          ch_bw       = rx_vector_start[0][6:4];
          n_sts       = rx_vector_start[12][6:4];
          stbc        = rx_vector_start[6][4];
          dcm         = rx_vector_start[7][2];
          gi_type     = rx_vector_start[6][3:2];
        end
        HE_MU: begin
          mcs_index   = rx_vector_start[12][3:0];
          ch_bw       = rx_vector_start[0][6:4];
          n_sts       = rx_vector_start[12][6:4];
          stbc        = rx_vector_start[6][4];
          dcm         = rx_vector_start[7][2];
          gi_type     = rx_vector_start[6][3:2];
        end
        HE_TB: begin
          mcs_index   = rx_vector_start[12][3:0];
          ch_bw       = rx_vector_start[0][6:4];
          n_sts       = rx_vector_start[12][6:4];
          stbc        = rx_vector_start[6][4];
          dcm         = rx_vector_start[7][2];
          gi_type     = rx_vector_start[6][3:2];
        end
      endcase
      leg_rate = rx_vector_start[4][7:4];

      Nss = get_num_ss_func(mcs_index,    //txMCS
                            format,       //txFormat
                            ch_bw,        //txChBW
                            n_sts,        //txnSTS
                            stbc);        //txSTBC

      `uvm_info(get_type_name(), $sformatf("Number of spatial streams Nss = %0d",Nss), UVM_HIGH)

      if (format ==  HE_MU)
        ru_type = get_ru_type_from_ru_allocation(req.frame.preamble_header.ru_allocation,
                                                 req.frame.preamble_header.user_header_he[0].dut_location_f);
      else if (format == HE_TB)
        ru_type = get_ru_type(req.frame.preamble_header.ru_allocation);
      else
        ru_type = get_ru_type_he_su(ch_bw);

      `uvm_info(get_type_name(), $sformatf("txFormatFn  = %h", format), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txLRateFn   = %h", leg_rate), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txChBWFn    = %h", ch_bw), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txMCSFn     = %h", mcs_index), UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("txRUType    = %s", ru_type.name()), UVM_HIGH)

      num_of_bits_in_symbol = get_bits_in_one_symbol_func(Nss,        //txNumSSFn
                                                          format,     //txFormatFn
                                                          leg_rate,   //txLRateFn
                                                          ch_bw,      //txChBWFn
                                                          mcs_index,  //txMCSFn
                                                          dcm,        //txDCMFn
                                                          ru_type);   //RUType

      // determine Rx data length
      u0_data_len = (format inside {NON_HT, NON_HT_DUP_OFDM}) ?
                     {rx_vector_start[4][3:0],  rx_vector_start[3][7:0]} : // when NON_HT use legacy length
                     (format inside {HT_GF, HT_MF}) ?
                     {rx_vector_start[9][7:0], rx_vector_start[8][7:0]}  : // HT lenght when HT_GF or HT_MF format
                     (format == VHT) ?
                     {rx_vector_start[12][3:0], rx_vector_start[11], rx_vector_start[10]} : // VHT format
                     {rx_vector_start[15][3:0], rx_vector_start[14], rx_vector_start[13]};  // HE format

      `uvm_info(get_type_name(), $sformatf("U0 RX data length %0d",u0_data_len), UVM_HIGH)

      // if num of bytes is not rounded then apply ceiling function
      num_of_bytes_in_symbol  = (num_of_bits_in_symbol % 8) ?
                                 $floor(real'(num_of_bits_in_symbol)/8) :
                                 (num_of_bits_in_symbol/8);

      // when there are less then 8bits in symbol make minimum 1 byte
      if (num_of_bytes_in_symbol == 0) num_of_bytes_in_symbol = 1;

      num_of_symbols_in_frame = (u0_data_len % num_of_bytes_in_symbol) ?
                                 $ceil(real'(u0_data_len) / num_of_bytes_in_symbol) :
                                 (u0_data_len / num_of_bytes_in_symbol);

      `uvm_info(get_type_name(),
        $sformatf("~ u0 ~ Number of bits in symbol %0d\nNumber of bytes in symbol %0d\nNumber of symbols in frame %0d",
        num_of_bits_in_symbol,num_of_bytes_in_symbol,num_of_symbols_in_frame), UVM_HIGH)


      // create and randomize ready signals
      phy_ready_gen_u0 = new[num_of_symbols_in_frame];
      foreach (phy_ready_gen_u0[i]) begin
        // randomize number of PHY ready signals (value 1) per symbol not to be larger
        // then number of bytes per symbol
        // determine if short or long GI
        if (format inside {NON_HT, NON_HT_DUP_OFDM, HT_MF, HT_GF, VHT}) begin
          if (gi_type) begin // SHORT GI

            symbol_time = `SHORT_LEN;
            assert (randomize(tmp_array_short) with { (tmp_array_short.sum(b) with (int'(b)) == num_of_bytes_in_symbol); });
            phy_ready_gen_u0[i] = new[symbol_time](tmp_array_short); // create array with tmp_array_short content

          end
          else begin // LONG GI

            symbol_time = `LONG_LEN;
            assert (randomize(tmp_array_long) with { (tmp_array_long.sum(b) with (int'(b)) == num_of_bytes_in_symbol); });
            phy_ready_gen_u0[i] = new[symbol_time](tmp_array_long); // create array with tmp_array_long content

          end
        end
        else begin
          symbol_time = `HE_LEN + (`GI_LEN << gi_type);
          tmp_array_he = new[symbol_time];
          assert (randomize(tmp_array_he) with { (tmp_array_he.sum(b) with (int'(b)) == num_of_bytes_in_symbol);});
          phy_ready_gen_u0[i] = new[symbol_time](tmp_array_he); // create array with tmp_array_long content
        end
      end//foreach

      // put to 0 PHY ready that are overabundant
      ones = 0;
      foreach (phy_ready_gen_u0[i]) begin
        foreach (phy_ready_gen_u0[i][j]) begin
          if (phy_ready_gen_u0[i][j] == 1'b1 && ones < u0_data_len)
            ones++;
          else if (phy_ready_gen_u0[i][j] == 1'b1 && ones == u0_data_len)
            phy_ready_gen_u0[i][j] = 1'b0; // set overabundant to 0
        end
        phy_ready_gen_u0[i].shuffle();
      end//foreach

    end// else if tx = 0

  endtask : create_phy_ready_array

  //-------------------------------------------------------
  // drive PHY ready signal to all active users
  // from generated random array
  //-------------------------------------------------------
  task drive_phy_ready_tx();

    fork : DRIVE_PHY_RDY

      //-----------------------------------------
      // start sample counting for all 4 users
      // if some user is not present that thread
      // will end immediately
      //-----------------------------------------
      cnt_tx_samples(u0_data_len, 0);
      //-----------------------------------------
      // User 0
      //-----------------------------------------
      begin : USER0
        for (int symbol=0; symbol < phy_ready_gen_u0.size(); symbol++) begin
          for (int valid=0; valid < phy_ready_gen_u0[symbol].size(); valid++) begin
            @(posedge vif.clk);
            vif.phy_rdy <= phy_ready_gen_u0[symbol][valid];
          end// for valid
        end// for symbol
        // be sure to put PHY ready to low after all samples from array are driven
        @(posedge vif.clk);
        vif.phy_rdy <= 1'b0;
      end//USER0

    join

  endtask : drive_phy_ready_tx

  //-------------------------------------------------------
  // count MAC data valid until it reaches byte limit for
  // that frame
  //-------------------------------------------------------
  task cnt_tx_samples(int limit, int user);
    int sample_cnt = 0;

    while (sample_cnt < limit) begin
      @(posedge vif.clk iff vif.mac_data_valid == 1'b1);
      sample_cnt++;
    end

    `uvm_info(get_type_name(), $sformatf("Received %0d samples from user %0d",sample_cnt,user), UVM_HIGH)
  endtask : cnt_tx_samples

  //-------------------------------------------------------
  // generate Tx end signal after all data is sent
  //-------------------------------------------------------
  task drive_tx_end();
    @(posedge vif.clk);
    vif.tx_end <= 1'b1;
    @(posedge vif.clk);
    vif.tx_end <= 1'b0;
    `uvm_info(get_type_name(),$sformatf("Drive TX END signal"), UVM_HIGH)
  endtask : drive_tx_end

  //-------------------------------------------------------
  // generate txvector2, after tx_end
  //-------------------------------------------------------
  task send_txvector2();
    vif.phy_rdy <= 1'b1;
    vif.rx_data <= tx_vector2[0];
    for (int i = 1; i <= 5; i++) begin
      @(posedge vif.clk);
      vif.rx_data <= tx_vector2[i];
    end
    vif.rx_data <= 0;
    vif.phy_rdy <= 1'b0;
  endtask
  //-------------------------------------------------------
  // generate Rx end signal after all data is sent
  //-------------------------------------------------------
  task drive_rx_end();
    if (!(req.err_inj inside {PHY_ERR_RX_VECTOR, PHY_ERR_BEFORE_RX_END})) begin
      vif.rx_end <= 1'b1;
      @(posedge vif.clk);
      vif.rx_end <= 1'b0;
      `uvm_info(get_type_name(),$sformatf("Drive RX END signal"), UVM_HIGH)
    end
    else begin
      vif.rx_end <= 1'b1;
      vif.phy_err <= 1'b1;
      @(posedge vif.clk);
      vif.phy_err <= 1'b0;
      vif.rx_end <= 1'b0;
      `uvm_info(get_type_name(),$sformatf("Drive RX END signal with PHY_ERR"), UVM_HIGH)
    end
  endtask : drive_rx_end

  //-------------------------------------------------------
  // generate Rx end for timing signal after all data is sent
  //-------------------------------------------------------
  task drive_rx_end_for_timing();
    // inform MAC that last sample of symbol is received
    if (req.err_inj == FORMAT_VIOLATION) begin
      vif.rx_err <= 1'b1;
      vif.rx_end_for_timing <= 1'b1;
      @(posedge vif.clk);
      vif.rx_err <= 1'b0;
      vif.rx_end_for_timing <= 1'b0;
      `uvm_info(get_type_name(),$sformatf("Drive RX END FOR TIMING signal with RX_ERR"), UVM_HIGH)
    end
    else begin
      vif.rx_end_for_timing <= 1'b1;
      @(posedge vif.clk);
      vif.rx_end_for_timing <= 1'b0;
      `uvm_info(get_type_name(),$sformatf("Drive RX END FOR TIMING signal"), UVM_HIGH)
    end
  endtask : drive_rx_end_for_timing

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : phy_driver

`endif //PHY_DRIVER_SV
