//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_PHY_MONITOR_SV
`define MAC_PHY_MONITOR_SV


class mac_phy_monitor extends uvm_monitor;

  virtual mac_phy_if                    vif;
  mac_phy_config                        cfg;
  uvm_analysis_port #(mac_phy_seq_item) ap;
  uvm_analysis_port #(mac_phy_rx_stream_seq_item) ap_rx_stream;
  uvm_analysis_port #(PPDU_frame)       ap_frame; // used for frame coverage model
  mac_phy_seq_item                      item;
  mac_phy_rx_stream_seq_item            rx_stream_item;

  //---------------------------------------
  // signals for collecting item
  //---------------------------------------
  bit                  phy_err;
  bit                  rx_err;
  octet_t              tx_stream[4][$]; // queue for tx vector and data
  octet_t              rx_stream[$];    // queue for rx vector and data
  // Tx and Rx vectors
  octet_t              tx_vector2[6];
  octet_t              tx_vector[];
  octet_t              tx_response[6];
  octet_t              rx_vector_start[];
  octet_t              rx_vector_end[8];
  int                  blk_del_cnt = 0;
  int                  num_of_users = 1;

  `uvm_component_utils(mac_phy_monitor)

  function new(string name = "mac_phy_monitor", uvm_component parent = null);
    super.new(name, parent);

  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual mac_phy_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

    ap = new("ap", this);
    ap_rx_stream = new("ap_rx_stream", this);
    ap_frame = new("ap_frame", this);

    item = mac_phy_seq_item::type_id::create("item");
    rx_stream_item = mac_phy_rx_stream_seq_item::type_id::create("rx_stream_item");
  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    wait (vif.rst_n == 1'b1);

    fork
      collect_transaction();
      observe_reset();
      detect_rx_err();
      detect_phy_err();
      en_dis_checking();
    join_none
  endtask : run_phase


  //-------------------------------------------------------
  // detect reset assetion
  //-------------------------------------------------------
  task observe_reset();
    @(negedge vif.rst_n);
    `uvm_info(get_type_name(), "Transaction recording interrupted by reset assertion.", UVM_HIGH);
  endtask : observe_reset

  //-------------------------------------------------------
  // task that enables/disables checkes/assertions inside
  // interface module
  //-------------------------------------------------------
  task en_dis_checking();
    bit has_checks_old;

    has_checks_old = cfg.has_checks;
    vif.has_checks = cfg.has_checks;
    forever begin
      wait (cfg.has_checks != has_checks_old);
      vif.has_checks = cfg.has_checks;
    end
  endtask : en_dis_checking

  //-------------------------------------------------------
  // collect MAC-PHY transactions
  //-------------------------------------------------------
  task collect_transaction();

    forever begin
      // clear all queues
      tx_stream[0] = {};
      tx_stream[1] = {};
      tx_stream[2] = {};
      tx_stream[3] = {};
      rx_stream    = {};

      fork : WAIT_RX_TX_REQ
        mac_phy_tx();
        mac_phy_rx();
      join_any

      disable fork;

      // put error indication if they occured
      item.rx_err  = rx_err;
      item.phy_err = phy_err;

      `uvm_info(get_type_name(), $sformatf("Transaction collected :\n%s", item.sprint()), UVM_HIGH)

      if (cfg.has_checks)
        perform_checks();

      ap.write(item);
      // added extra analysis port for coverage
      if (!rx_err && !phy_err) begin
        ap_frame.write(item.frame);
      end
      // reset error flags
      rx_err  = 0;
      phy_err = 0;
      // add delay when RX trasaction has occured, because of delay
      // that is from rx_end to fall of rx_req. This will prevent unintentional
      // detection of Rx request
      if (item.trans == RX) begin
        #(`RX_END_2_RX_REQ_FALL_DELAY);
      end

    end// forever
  endtask : collect_transaction

  //-------------------------------------------------------
  // wait for tx request to occur and collect data
  //-------------------------------------------------------
  task mac_phy_tx();

    int   tx_vector_length;

    nonht_txvector_s      nonht_txv;
    ht_txvector_s         ht_txv;
    vht_txvector_s        vht_txv;
    he_su_txvector_s      he_su_txv;
    he_mu_txvector_s      he_mu_txv;
    he_trigg_txvector_s   he_tb_txv;

    @(posedge vif.clk iff  vif.tx_req == 1'b1 && vif.mac_data_valid == 1'b1);
    `uvm_info(get_type_name(), "TX request received...", UVM_HIGH)
    item.trans = TX;
    item.tx_req_rise = $time();

    tx_vector = new[1];
    tx_vector[0] = vif.tx_data;

    case (format_mod_e'(vif.tx_data[3:0]))
      NON_HT, NON_HT_DUP_OFDM : begin
        tx_vector_length = 10;
      end
      HT_MF, HT_GF : begin
        tx_vector_length = 13;
      end
      VHT : begin
        tx_vector_length = 17;
      end
      HE_SU, HE_EXT_SU : begin
        tx_vector_length = 18;
      end
      HE_MU : begin
        tx_vector_length = 19;
      end
      HE_TB : begin
        tx_vector_length = 24;
      end
    endcase


    forever begin
      if (tx_vector.size() == tx_vector_length)
        break;

      @(posedge vif.clk iff vif.mac_data_valid == 1'b1);
        tx_vector = new[tx_vector.size()+1](tx_vector);
        tx_vector[tx_vector.size()-1] = vif.tx_data;

    end // forever

    foreach (tx_vector[i]) begin
      case(format_mod_e'(tx_vector[0][3:0]))
        NON_HT, NON_HT_DUP_OFDM : nonht_txv[i*8 +: 8] = tx_vector[i];
        HT_MF, HT_GF            : ht_txv[i*8 +: 8] = tx_vector[i];
        VHT                     : vht_txv[i*8 +: 8] = tx_vector[i];
        HE_SU, HE_EXT_SU        : he_su_txv[i*8 +: 8] = tx_vector[i];
        HE_MU                   : he_mu_txv[i*8 +: 8] = tx_vector[i];
        HE_TB                   : he_tb_txv[i*8 +: 8] = tx_vector[i];
      endcase
    end // foreach

    case(format_mod_e'(tx_vector[0][3:0]))
      NON_HT, NON_HT_DUP_OFDM : `uvm_info(get_type_name(), $sformatf("Tx vector collected: %p",nonht_txv), UVM_LOW)
      HT_MF, HT_GF            : `uvm_info(get_type_name(), $sformatf("Tx vector collected: %p",ht_txv), UVM_LOW)
      VHT                     : `uvm_info(get_type_name(), $sformatf("Tx vector collected: %p",vht_txv), UVM_LOW)
      HE_SU, HE_EXT_SU        : `uvm_info(get_type_name(), $sformatf("Tx vector collected: %p",he_su_txv), UVM_LOW)
      HE_MU                   : `uvm_info(get_type_name(), $sformatf("Tx vector collected: %p",he_mu_txv), UVM_LOW)
      HE_TB                   : `uvm_info(get_type_name(), $sformatf("Tx vector collected: %p",he_tb_txv), UVM_LOW)
    endcase

    // collect rest of data on every data valid signal per user
    forever begin
      @(posedge vif.clk iff vif.mac_data_valid or vif.tx_end); // TODO: add tx_data_uid, when implemented
      if (vif.mac_data_valid) tx_stream[0].push_front(vif.tx_data);
      // when Tx end asserted quit collecting data
      if (vif.tx_end) begin
        @(negedge vif.tx_req);
        break;
      end
    end

    // put collected stream to item
    convert_stream2item(TX, num_of_users);
    wait (vif.tx_end == 1'b0);
    //~ // get TX_VECTOR 2
    //~ wait(vif.phy_rdy == 1'b1);
    //~ for (int i = 0; i < 6; i++) begin
      //~ @(posedge vif.clk);
      //~ tx_vector2[i] = vif.rx_data;
    //~ end
//~
    //~ `uvm_info(get_type_name(), $sformatf("Tx vector 2 collected: %0p",tx_vector2), UVM_LOW)

    item.rx_end_for_timing = 0;
    item.end_of_frame = 0;
    item.start_of_frame = 0;

  endtask : mac_phy_tx

  //-------------------------------------------------------
  // wait for rx request to occur and collect data
  //-------------------------------------------------------
  task mac_phy_rx();

    @(posedge vif.clk iff  vif.rx_req == 1'b1);
    `uvm_info(get_type_name(), "RX request received...", UVM_HIGH)
    item.trans = RX;
    item.rx_req_rise = $time();
    // wait CCA signal to rise and fall
    fork
      begin : CCA_COLLECTING
        @(posedge vif.clk iff (   vif.cca_primary_20
                               || vif.cca_secondary_20
                               || vif.cca_secondary_40));
        `uvm_info(get_type_name(), "CCA signal asserted...", UVM_HIGH)

        item.start_of_frame = $time();
        item.cca = {vif.cca_primary_20,
                    vif.cca_secondary_20,
                    vif.cca_secondary_40};
        // wait for fall of asserted signal
        wait (!vif.cca_primary_20 && !vif.cca_secondary_20 && !vif.cca_secondary_40);
      end

      begin : END_OF_TIMING
        @(posedge vif.rx_end_for_timing);
        item.rx_end_for_timing = 1;
        item.end_of_frame = $time();
        `uvm_info(get_type_name(), "Rx end of timing asserted...", UVM_HIGH)
      end

      begin : KEEP_RF_ON
        @(posedge vif.keep_rf_on);
        item.trans = RX_RIFS;
        `uvm_info(get_type_name(), "Keep RF ON asserted...", UVM_HIGH)
      end
    join_none

    // collect stream data
    forever begin
      @(posedge vif.clk iff (vif.phy_rdy && vif.rx_req) or vif.rx_end);
      if (vif.rx_end) break; // end of reception
      rx_stream.push_front(vif.rx_data);
    end//forever

    `uvm_info(get_type_name(), $sformatf("RX stream collected: %0p",rx_stream), UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("RX stream size: 0x%0h",rx_stream.size()), UVM_HIGH)
    // put collected stream to item
    convert_stream2item(RX, num_of_users);
    // sent item
    ap_rx_stream.write(rx_stream_item);

    item.tx_req_rise = 0;

  endtask : mac_phy_rx

  //-------------------------------------------------------
  // Check if remaining octests are all zeros
  //-------------------------------------------------------
  function bit check_zero_rx_stream();
    check_zero_rx_stream = 1;
    for (int i = 0; i< rx_stream.size(); i++) begin
      if (rx_stream[i] != 'b0) check_zero_rx_stream = 0;
    end
    `uvm_info(get_type_name(), $sformatf("check_zero_rx_stream = %0d", check_zero_rx_stream), UVM_HIGH)
    return check_zero_rx_stream;
  endfunction: check_zero_rx_stream

  //-------------------------------------------------------
  // convert data stream that is collected on Tx path and
  // put to proper fields of PPDU frame object
  //-------------------------------------------------------
  task convert_stream2item(transaction_e tr, int num_of_users);
    int            byte_cnt;
    bit [15:0]     frame_ctrl;
    int            frame_body_len;
    int            frame_len;
    bit [31:0]     delimiter;
    int            mpdu_cnt;
    int            pad_num;
    bit [7:0]      dummy;
    int            rx_vector_length;

    //---------------------------------------------
    // clear dynamic arrays from item
    //---------------------------------------------
    item.frame.preamble_header.user_header.delete();
    item.frame.preamble_header.user_header_he.delete();
    item.frame.ampdu_frame.delete();
    //---------------------------------------------
    if (tr == TX) begin

      item.frame.ppdu_format = format_mod_e'(tx_vector[0][3:0]);
      item.frame.create_preamble_header();

      item.frame.preamble_header.format_mod       = format_mod_e'(tx_vector[0][3:0]);
      item.frame.preamble_header.ch_bw            = tx_vector[0][6:4];
      item.frame.preamble_header.preamble_type    = tx_vector[0][7];
      item.frame.preamble_header.antenna_set      = tx_vector[1];
      item.frame.preamble_header.tx_pwr_level     = tx_vector[2];
      item.frame.preamble_header.num_tx           = tx_vector[3][2:0];
      item.frame.preamble_header.continuous_tx    = tx_vector[3][7];
      item.frame.preamble_header.leg_length       = {tx_vector[5][3:0], tx_vector[4]};
      item.frame.preamble_header.leg_rate         = tx_vector[5][7:4];
      item.frame.preamble_header.service          = {tx_vector[7], tx_vector[6]};

      if (item.frame.ppdu_format != NON_HT && item.frame.ppdu_format != NON_HT_DUP_OFDM) begin
        item.frame.preamble_header.sounding       = tx_vector[8][0];
        item.frame.preamble_header.stbc           = tx_vector[8][4];
      end

      if (item.frame.ppdu_format inside {HT_MF, HT_GF}) begin
        item.frame.preamble_header.smoothing        = tx_vector[8][1];
      end

      if (item.frame.ppdu_format inside {HT_MF, HT_GF, VHT}) begin
        item.frame.preamble_header.gi_type        = {1'b0, tx_vector[8][2]};
      end

      if (item.frame.ppdu_format inside {VHT, HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
        item.frame.preamble_header.beamformed       = tx_vector[8][1];
      end

      // create right number of user headers
      if (item.frame.ppdu_format == VHT) begin
        item.frame.preamble_header.aggregated       = 1;
        item.frame.preamble_header.doze_not_allowed = tx_vector[8][5];
        item.frame.preamble_header.partial_aid      = {tx_vector[10][0], tx_vector[9]};
        item.frame.preamble_header.group_id         = tx_vector[10][6:1];
        item.frame.preamble_header.n_user           = tx_vector[11][2:0];
        item.frame.preamble_header.user_header = new[num_of_users];
        for (int i = 0; i < num_of_users; i++) begin
          item.frame.preamble_header.user_header[i].smm_index_f     = tx_vector[12 + 6*i];
          item.frame.preamble_header.user_header[i].mcs_f           = tx_vector[13 + 6*i][3:0];
          item.frame.preamble_header.user_header[i].num_sts_f       = tx_vector[13 + 6*i][6:4];
          item.frame.preamble_header.user_header[i].fec_coding_f    = tx_vector[13 + 6*i][7];
          item.frame.preamble_header.user_header[i].ht_length_f     = {tx_vector[16 + 6*i][3:0],
                                                                       tx_vector[15 + 6*i],
                                                                       tx_vector[14 + 6*i]};
          item.frame.preamble_header.user_header[i].user_position_f = tx_vector[16 + 6*i][7:6];
        end
      end

      if (item.frame.ppdu_format inside {NON_HT, NON_HT_DUP_OFDM}) begin
        item.frame.preamble_header.trigger_responding         = tx_vector[8][0];
        item.frame.preamble_header.user_header                = new[num_of_users];
        item.frame.preamble_header.user_header[0].smm_index_f = tx_vector[9];
      end

      if (item.frame.ppdu_format inside {HT_MF, HT_GF}) begin
        item.frame.preamble_header.num_extn_ss                 = tx_vector[8][6:5];
        item.frame.preamble_header.aggregated                  = tx_vector[8][3];
        item.frame.preamble_header.user_header                 = new[num_of_users];
        item.frame.preamble_header.user_header[0].smm_index_f  = tx_vector[9];
        item.frame.preamble_header.user_header[0].mcs_f        = tx_vector[10][6:0];
        item.frame.preamble_header.user_header[0].fec_coding_f = tx_vector[10][7];
        item.frame.preamble_header.user_header[0].ht_length_f  = {tx_vector[12],
                                                                  tx_vector[11]};
      end

      if (item.frame.ppdu_format inside {HE_MU, HE_TB, HE_EXT_SU, HE_SU})begin
        item.frame.preamble_header.aggregated           = 1'b1;
        item.frame.preamble_header.user_header_he       = new[num_of_users];
        item.frame.preamble_header.gi_type              = tx_vector[8][3:2];
        item.frame.preamble_header.uplink_flag          = tx_vector[9][0];
        item.frame.preamble_header.beam_change          = tx_vector[9][1];
        item.frame.preamble_header.dcm                  = tx_vector[9][2];
        item.frame.preamble_header.he_ltf_type          = tx_vector[9][4:3];
        item.frame.preamble_header.doppler              = tx_vector[9][5];
        item.frame.preamble_header.midamble_periodicity = tx_vector[9][6];
        item.frame.preamble_header.bss_color            = tx_vector[10][5:0];
        item.frame.preamble_header.txop_duration        = tx_vector[11][6:0];
        item.frame.preamble_header.spatial_reuse[0]     = tx_vector[12][3:0];
      end

      if (item.frame.ppdu_format inside {HE_SU, HE_EXT_SU}) begin
        item.frame.preamble_header.user_header_he[0].smm_index_f  = tx_vector[13];
        item.frame.preamble_header.user_header_he[0].mcs_f        = tx_vector[14][3:0];
        item.frame.preamble_header.user_header_he[0].nss_f        = tx_vector[14][6:4];
        item.frame.preamble_header.user_header_he[0].fec_coding_f = tx_vector[14][7];
        item.frame.preamble_header.user_header_he[0].he_length_f  = {tx_vector[17][3:0],
                                                                     tx_vector[16],
                                                                     tx_vector[15]};
        item.frame.preamble_header.user_header_he[0].pkt_ext_f    = tx_vector[17][6:4];
      end

      if (item.frame.ppdu_format == HE_MU) begin
        item.frame.preamble_header.n_user = tx_vector[18];
        // common for all users
        item.frame.preamble_header.sig_b_comp_mode = tx_vector[13][0];
        item.frame.preamble_header.dcm_sig_b       = tx_vector[13][1];
        item.frame.preamble_header.mcs_sig_b       = tx_vector[13][4:2];
        item.frame.preamble_header.ru_allocation   = tx_vector[14];
        for (int i = 0; i < num_of_users; i++) begin
          // per user
          item.frame.preamble_header.user_header_he[i].smm_index_f  = tx_vector[19 + 7*i];
          item.frame.preamble_header.user_header_he[i].mcs_f        = tx_vector[20 + 7*i][3:0];
          item.frame.preamble_header.user_header_he[i].nss_f        = tx_vector[20 + 7*i][6:4];
          item.frame.preamble_header.user_header_he[i].fec_coding_f = tx_vector[20 + 7*i][7];
          item.frame.preamble_header.user_header_he[i].he_length_f  = {tx_vector[23 + 7*i][3:0],
                                                                       tx_vector[22 + 7*i],
                                                                       tx_vector[21 + 7*i]};
          item.frame.preamble_header.user_header_he[i].pkt_ext_f    = tx_vector[23 + 7*i][6:4];
          item.frame.preamble_header.user_header_he[i].staid_f      = {tx_vector[25 + 7*i][2:0],
                                                                       tx_vector[24 + 7*i]};
          item.frame.preamble_header.user_header_he[i].user_position_f = tx_vector[25 + 7*i][7:3];
        end
      end

      if (item.frame.ppdu_format == HE_TB) begin
        item.frame.preamble_header.spatial_reuse[1]  = tx_vector[12][7:4];
        item.frame.preamble_header.spatial_reuse[2]  = tx_vector[13][3:0];
        item.frame.preamble_header.spatial_reuse[3]  = tx_vector[13][7:4];
        item.frame.preamble_header.he_siga_reserved  = {tx_vector[15][0],
                                                        tx_vector[14]};
        item.frame.preamble_header.num_he_ltf        = tx_vector[15][3:1];
        item.frame.preamble_header.he_ltf_mode       = tx_vector[15][4];
        item.frame.preamble_header.ldpc_extra_symbol = tx_vector[15][5];
        item.frame.preamble_header.starting_sts_num  = tx_vector[16][2:0];
        item.frame.preamble_header.ru_allocation     = tx_vector[17];
        item.frame.preamble_header.user_header_he[0].smm_index_f  = tx_vector[18];
        item.frame.preamble_header.user_header_he[0].mcs_f        = tx_vector[19][3:0];
        item.frame.preamble_header.user_header_he[0].nss_f        = tx_vector[19][6:4];
        item.frame.preamble_header.user_header_he[0].fec_coding_f = tx_vector[19][7];
        item.frame.preamble_header.user_header_he[0].he_length_f  = {tx_vector[22][3:0],
                                                                     tx_vector[21],
                                                                     tx_vector[20]};
        item.frame.preamble_header.user_header_he[0].pkt_ext_f    = tx_vector[22][6:4];
        item.frame.preamble_header.trigger_method    = tx_vector[22][7];
        item.frame.preamble_header.ru_tone_set_index = tx_vector[23][6:0];
        item.frame.preamble_header.feedback_status   = tx_vector[23];
      end

      item.frame.num_of_users = num_of_users;
      item.frame.preamble_header.tr = TX;
      // save preamble header to configuration
      cfg.pre_hdr.copy(item.frame.preamble_header);

      if (item.frame.ppdu_format inside {HE_SU, HE_EXT_SU, HE_TB, HE_MU}) begin
        if (num_of_users > 1)
          item.frame.kind = MU_MIMO;
        else if (item.frame.preamble_header.user_header_he[0].he_length_f == 0)
          item.frame.kind = NDP;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 0)
          item.frame.kind = SINGLETON;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 1)
          item.frame.kind = AGGREGATED;
      end
      else begin
        if (num_of_users > 1)
          item.frame.kind = MU_MIMO;
        else if (   (item.frame.preamble_header.user_header[0].ht_length_f == 0)
                 && (item.frame.ppdu_format == VHT))
          item.frame.kind = NDP;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 0)
          item.frame.kind = SINGLETON;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 1)
          item.frame.kind = AGGREGATED;
      end

`ifdef STANDALONE_PHY
      item.frame.ampdu_frame = new[num_of_users];
      foreach (item.frame.ampdu_frame[user]) begin
        item.frame.ampdu_frame[user] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[%0d]",user));
        item.frame.ampdu_frame[user].ampdu_payload = new[tx_stream[user].size()];

        foreach (item.frame.ampdu_frame[user].ampdu_payload[i]) begin
          item.frame.ampdu_frame[user].ampdu_payload[i] = tx_stream[user].pop_back();
        end
      end
      // set number of users when NDP is transmitted
      if (item.frame.kind == NDP) begin
        item.frame.num_of_users = 0;
      end// if NDP
      // return from task because only payload is valid
      return;
`endif

//---------------------------------------------------------------------------------------
      // fetch A-MPDU and MPDU frames from stream and store to PPDU object
      if (item.frame.kind == SINGLETON) begin
//---------------------------------------------------------------------------------------
        // create one AMPDU frame
        item.frame.ampdu_frame = new[1];
        item.frame.ampdu_frame[0] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[0]"));
        item.frame.ampdu_frame[0].mpdu_frame_type = new[1];
        item.frame.ampdu_frame[0].mpdu_frame      = new[1];

        // set format in A-MPDU frame
        item.frame.ampdu_frame[0].ppdu_format = item.frame.ppdu_format;

        // determine if frame is DSSS
        if (   (item.frame.ppdu_format == NON_HT)
            && (item.frame.preamble_header.leg_rate inside {[0:3]})) begin
          item.frame.ampdu_frame[0].is_dsss = 1;
        end else begin
          item.frame.ampdu_frame[0].is_dsss = 0;
        end

        // in sigleton frame MAC header is first sent, first 2 bytes are frame
        // control bytes, from which type and sub-type of MPDU frame is
        // determined
        frame_ctrl[ 7:0] = tx_stream[0].pop_back();
        frame_ctrl[15:8] = tx_stream[0].pop_back();
        // determine type and sub-type of frame
        item.frame.ampdu_frame[0].mpdu_frame_type[0] = mpdu_frame_type_e'({frame_ctrl[3:2], frame_ctrl[7:4]});
        // if HE-SU and frame is VHT_NDP_ANNOUNCEMENT then it needs to be
        // converted to HE_NDP_ANNOUNCEMENT
        if (   item.frame.ppdu_format == HE_SU
            && item.frame.ampdu_frame[0].mpdu_frame_type[0] == VHT_NDP_ANNOUNCEMENT) begin
          item.frame.ampdu_frame[0].mpdu_frame_type[0] = HE_NDP_ANNOUNCEMENT;
        end

        item.frame.ampdu_frame[0].create_mpdu_frame_by_type(0);
        // when we know type of frame, randomize it to apply constraints on
        // all fields inside, to make it easy when storing data
        assert (item.frame.ampdu_frame[0].mpdu_frame[0].randomize() with {
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.frame_ctrl == frame_ctrl;
        });

        item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.duration_id[ 7:0] = tx_stream[0].pop_back();
        item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.duration_id[15:8] = tx_stream[0].pop_back();
        // save transmited frame duration in configuration
        cfg.tx_frm_duration = item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.duration_id;
        // store MAC address
        foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i]) begin
          if (i == 3) break; // don't store 4th address before sequence control
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][ 7: 0] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][15: 8] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][23:16] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][31:24] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][39:32] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][47:40] = tx_stream[0].pop_back();
        end
        foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.seq_ctrl[i]) begin
          // MAC seq_ctrl is contained of 2 ostets
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.seq_ctrl[i][ 7:0] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.seq_ctrl[i][15:8] = tx_stream[0].pop_back();
        end//for
        // 4th MAC address can only apear after sequence control
        if (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr.size() == 4) begin
          // MAC address is contained of 6 ostets
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][ 7: 0] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][15: 8] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][23:16] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][31:24] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][39:32] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][47:40] = tx_stream[0].pop_back();
        end
        foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.qos_ctrl[i]) begin
          // MAC qos_ctrl is contained of 2 ostets
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.qos_ctrl[i][ 7:0] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.qos_ctrl[i][15:8] = tx_stream[0].pop_back();
        end//for
        foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i]) begin
          // MAC ht_ctrl is contained of 4 ostets
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][ 7: 0] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][15: 8] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][23:16] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][31:24] = tx_stream[0].pop_back();
        end//for

        // store frame body data
        item.frame.ampdu_frame[0].mpdu_frame[0].frame_body.delete();
        frame_body_len = (   item.frame.preamble_header.format_mod == NON_HT
                          || item.frame.preamble_header.format_mod == NON_HT_DUP_OFDM) ?
                          item.frame.preamble_header.leg_length :                 // non HT mode
                          (  item.frame.preamble_header.format_mod inside {HT_MF, HT_GF, VHT}) ?
                          item.frame.preamble_header.user_header[0].ht_length_f :  // HT mode
                          item.frame.preamble_header.user_header_he[0].he_length_f;// HE mode

        // subtract MAC header length
        frame_body_len -= item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.size();
        // subtract 4 bytes of FCS
        frame_body_len -= 4;

        if (frame_body_len > 0) begin
          item.frame.ampdu_frame[0].mpdu_frame[0].frame_body = new[frame_body_len];
          for (int i=0; i<frame_body_len; i++) begin
            item.frame.ampdu_frame[0].mpdu_frame[0].frame_body[i] = tx_stream[0].pop_back();
          end//for
        end
        // store last 4 bytes for FCS
        item.frame.ampdu_frame[0].mpdu_frame[0].FCS[ 7: 0] = tx_stream[0].pop_back();
        item.frame.ampdu_frame[0].mpdu_frame[0].FCS[15: 8] = tx_stream[0].pop_back();
        item.frame.ampdu_frame[0].mpdu_frame[0].FCS[23:16] = tx_stream[0].pop_back();
        item.frame.ampdu_frame[0].mpdu_frame[0].FCS[31:24] = tx_stream[0].pop_back();

        //make sure frame and frame_size are correct
        item.frame.ampdu_frame[0].mpdu_frame[0].post_monitor();
        `uvm_info(get_type_name(), $sformatf("MPDU transmited: %s",item.frame.ampdu_frame[0].mpdu_frame[0].sprint()),UVM_DEBUG)

      end// if SINGLETON
//---------------------------------------------------------------------------------------
      else if (item.frame.kind == AGGREGATED) begin
//---------------------------------------------------------------------------------------

        mpdu_cnt = 0;

        // create one AMPDU frame
        item.frame.ampdu_frame = new[1];
        item.frame.ampdu_frame[0] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[0]"));

        // set format in A-MPDU frame
        item.frame.ampdu_frame[0].ppdu_format = item.frame.ppdu_format;

        while (tx_stream[0].size() != 0) begin

          item.frame.ampdu_frame[0].aggregated = 1'b1;
          blk_del_cnt = 0;
          // in aggregated frame delimiter is first sent then MPDU frame and padding
          while (1) begin
            delimiter[ 7: 0] = tx_stream[0].pop_back();
            delimiter[15: 8] = tx_stream[0].pop_back();
            delimiter[23:16] = tx_stream[0].pop_back();
            delimiter[31:24] = tx_stream[0].pop_back();
            // determine if blank delimiter is received, length is 0 is this
            // case
            if (delimiter[15:4] == 12'h0  && delimiter[31:24] == 8'h4E && !check_delimiter_crc(delimiter)) begin
              item.frame.ampdu_frame[0].blank_delimit[mpdu_cnt-1] = new();
              blk_del_cnt++;
              item.frame.ampdu_frame[0].blank_delimit[mpdu_cnt-1].blank_delimiter_num = blk_del_cnt;
              `uvm_info(get_type_name(), $sformatf("blank_delimit: %s",item.frame.ampdu_frame[0].blank_delimit[mpdu_cnt-1].sprint()),UVM_DEBUG)
              continue; // while loop
            end
            break; // while loop
          end //while (1)

          // check if queue is empty
          if (tx_stream[0].size() == 0) break; // exit while loop

          // extend 3 dynamic arrays by 1 and append previous content to it
          // for mpdu_frame_type, mpdu_frame and delimiter
          item.frame.ampdu_frame[0].mpdu_frame_type =
            new[item.frame.ampdu_frame[0].mpdu_frame_type.size() + 1](item.frame.ampdu_frame[0].mpdu_frame_type);
          item.frame.ampdu_frame[0].mpdu_frame      =
            new[item.frame.ampdu_frame[0].mpdu_frame.size() + 1](item.frame.ampdu_frame[0].mpdu_frame);
          item.frame.ampdu_frame[0].delimit =
            new[item.frame.ampdu_frame[0].delimit.size() + 1](item.frame.ampdu_frame[0].delimit);

          item.frame.ampdu_frame[0].delimit[mpdu_cnt] = frame_model_pkg::delimiter::type_id::create($sformatf("delimiter[%0d]",mpdu_cnt));
          item.frame.ampdu_frame[0].delimit[mpdu_cnt].set_data(delimiter);
          `uvm_info(get_type_name(), $sformatf("delimiter: %s",item.frame.ampdu_frame[0].delimit[mpdu_cnt].sprint()),UVM_DEBUG)

          // in singleton frame MAC header is sent first, first 2 bytes are frame
          // control bytes, from which type and sub-type of MPDU frame is
          // determined
          frame_ctrl[ 7:0] = tx_stream[0].pop_back();
          frame_ctrl[15:8] = tx_stream[0].pop_back();
          // determine type and sub-type of frame
          item.frame.ampdu_frame[0].mpdu_frame_type[mpdu_cnt] = mpdu_frame_type_e'({frame_ctrl[3:2], frame_ctrl[7:4]});
          item.frame.ampdu_frame[0].create_mpdu_frame_by_type(mpdu_cnt);
          // when we know type of frame, randomize it to apply constraints on
          // all fields inside, to make it easy when storing data
          assert (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].randomize() with {
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.frame_ctrl == frame_ctrl;
          });

          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.duration_id[ 7:0] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.duration_id[15:8] = tx_stream[0].pop_back();
          // save transmited frame duration in configuration
          cfg.tx_frm_duration = item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.duration_id;
          // store MAC address
          foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i]) begin
            if (i == 3) break; // don't store 4th address before sequence control
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][ 7: 0] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][15: 8] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][23:16] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][31:24] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][39:32] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][47:40] = tx_stream[0].pop_back();
          end
          foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i]) begin
            // MAC seq_ctrl is contained of 2 ostets
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i][ 7:0] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i][15:8] = tx_stream[0].pop_back();
          end//for
          // 4th MAC address can only apear after sequence control
          if (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr.size() == 4) begin
            // MAC address is contained of 6 ostets
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][ 7: 0] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][15: 8] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][23:16] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][31:24] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][39:32] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][47:40] = tx_stream[0].pop_back();
          end
          foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i]) begin
            // MAC qos_ctrl is contained of 2 ostets
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i][ 7:0] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i][15:8] = tx_stream[0].pop_back();
          end//for
          foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i]) begin
            // MAC ht_ctrl is contained of 4 ostets
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][ 7: 0] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][15: 8] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][23:16] = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][31:24] = tx_stream[0].pop_back();
          end//for

          // store frame body data
          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame_body.delete();
          frame_body_len = item.frame.ampdu_frame[0].delimit[mpdu_cnt].mpdu_length;

          // subtract MAC header length
          frame_body_len -= item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.size();
          // subtract 4 bytes of FCS
          frame_body_len -= 4;

          if (frame_body_len > 0) begin
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame_body = new[frame_body_len];
            for (int i=0; i<frame_body_len; i++) begin
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame_body[i] = tx_stream[0].pop_back();
            end//for
          end

          // store last 4 bytes for FCS
          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[ 7: 0] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[15: 8] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[23:16] = tx_stream[0].pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[31:24] = tx_stream[0].pop_back();

          //make sure frame and frame_size are correct
          item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].post_monitor();
          `uvm_info(get_type_name(), $sformatf("MPDU transmited: %s",item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].sprint()),UVM_DEBUG)

          // store padding bits
          // padding number is equal to number of bytes that are needed for
          // length to be in power of 4
          pad_num = item.frame.ampdu_frame[0].delimit[mpdu_cnt].mpdu_length % 4;
          pad_num = (pad_num == 0) ? 0 : 4 - pad_num;
          // padding is not added to the last MPDU in A-MPDU
          if (tx_stream[0].size() != 0 && pad_num != 0) begin
            repeat (pad_num) dummy = tx_stream[0].pop_back();
            item.frame.ampdu_frame[0].padding[mpdu_cnt] = pad_num;
            `uvm_info(get_type_name(),$sformatf("Number of padding bytes %0d",pad_num),UVM_DEBUG)
          end

          mpdu_cnt++;

        end// while not empty

        // when one frame is sent as VHT or HE, MAC sends it as agreggated
        // but is is singleton frame
        if (mpdu_cnt == 1 && item.frame.ppdu_format inside {VHT, HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
          item.frame.ampdu_frame[0].aggregated = 0;
          item.frame.kind = SINGLETON;
        end

      end// if AGGREGATED

//---------------------------------------------------------------------------------------
      else if (item.frame.kind == MU_MIMO) begin
//---------------------------------------------------------------------------------------
        mpdu_cnt = 0;
        // create one AMPDU frame
        item.frame.ampdu_frame = new[num_of_users];
        foreach (item.frame.ampdu_frame[user]) begin
          item.frame.ampdu_frame[user] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[%0d]",user));
          // set format in A-MPDU frame
          item.frame.ampdu_frame[user].ppdu_format = item.frame.ppdu_format;
        end

        foreach (item.frame.ampdu_frame[user]) begin
          mpdu_cnt = 0;

          while (tx_stream[user].size() != 0) begin

            item.frame.ampdu_frame[user].aggregated = 1'b1;
            blk_del_cnt = 0;
            // in aggregated frame delimiter is first sent then MPDU frame and padding
            while (1) begin
              delimiter[ 7: 0] = tx_stream[user].pop_back();
              delimiter[15: 8] = tx_stream[user].pop_back();
              delimiter[23:16] = tx_stream[user].pop_back();
              delimiter[31:24] = tx_stream[user].pop_back();
              // determine if blank delimiter is received, length is 0 is this
              // case
              if (delimiter[15:4] == 12'h0  && delimiter[31:24] == 8'h4E && !check_delimiter_crc(delimiter)) begin
                item.frame.ampdu_frame[user].blank_delimit[mpdu_cnt-1] = new();
                blk_del_cnt++;
                item.frame.ampdu_frame[user].blank_delimit[mpdu_cnt-1].blank_delimiter_num = blk_del_cnt;
                `uvm_info(get_type_name(), $sformatf("blank_delimit: %s",item.frame.ampdu_frame[user].blank_delimit[mpdu_cnt-1].sprint()),UVM_DEBUG)
                continue; // while loop
              end
              break; // while loop
            end //while (1)

            if (tx_stream[user].size() == 0) break; // exit while loop

            // extend 3 dynamic arrays by 1 and append previous content to it
            // for mpdu_frame_type, mpdu_frame and delimiter
            item.frame.ampdu_frame[user].mpdu_frame_type =
              new[item.frame.ampdu_frame[user].mpdu_frame_type.size() + 1](item.frame.ampdu_frame[user].mpdu_frame_type);
            item.frame.ampdu_frame[user].mpdu_frame      =
              new[item.frame.ampdu_frame[user].mpdu_frame.size() + 1](item.frame.ampdu_frame[user].mpdu_frame);
            item.frame.ampdu_frame[user].delimit =
              new[item.frame.ampdu_frame[user].delimit.size() + 1](item.frame.ampdu_frame[user].delimit);

            item.frame.ampdu_frame[user].delimit[mpdu_cnt] = frame_model_pkg::delimiter::type_id::create($sformatf("delimiter[%0d]",mpdu_cnt));
            item.frame.ampdu_frame[user].delimit[mpdu_cnt].set_data(delimiter);
            `uvm_info(get_type_name(), $sformatf("delimiter: %s",item.frame.ampdu_frame[user].delimit[mpdu_cnt].sprint()),UVM_DEBUG)

            // in sigleton frame MAC header is first sent, first 2 bytes are frame
            // control bytes, from which type and sub-type of MPDU frame is
            // determined
            frame_ctrl[ 7:0] = tx_stream[user].pop_back();
            frame_ctrl[15:8] = tx_stream[user].pop_back();
            // determine type and sub-type of frame
            item.frame.ampdu_frame[user].mpdu_frame_type[mpdu_cnt] = mpdu_frame_type_e'({frame_ctrl[3:2], frame_ctrl[7:4]});
            item.frame.ampdu_frame[user].create_mpdu_frame_by_type(mpdu_cnt);
            // when we know type of frame, randomize it to apply constraints on
            // all fields inside, to make it easy when storing data
            assert (item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].randomize() with {
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.frame_ctrl == frame_ctrl;
            });

            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.duration_id[ 7:0] = tx_stream[user].pop_back();
            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.duration_id[15:8] = tx_stream[user].pop_back();
            // save transmited frame duration in configuration
            cfg.tx_frm_duration = item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.duration_id;
            // store MAC address
            foreach (item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[i]) begin
              if (i == 3) break; // don't store 4th address before sequence control
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[i][ 7: 0] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[i][15: 8] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[i][23:16] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[i][31:24] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[i][39:32] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[i][47:40] = tx_stream[user].pop_back();
            end
            foreach (item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i]) begin
              // MAC seq_ctrl is contained of 2 ostets
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i][ 7:0] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i][15:8] = tx_stream[user].pop_back();
            end//for
            // 4th MAC address can only apear after sequence control
            if (item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr.size() == 4) begin
              // MAC address is contained of 6 ostets
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[3][ 7: 0] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[3][15: 8] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[3][23:16] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[3][31:24] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[3][39:32] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.addr[3][47:40] = tx_stream[user].pop_back();
            end
            foreach (item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i]) begin
              // MAC qos_ctrl is contained of 2 ostets
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i][ 7:0] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i][15:8] = tx_stream[user].pop_back();
            end//for
            foreach (item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i]) begin
              // MAC ht_ctrl is contained of 4 ostets
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][ 7: 0] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][15: 8] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][23:16] = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][31:24] = tx_stream[user].pop_back();
            end//for

            // store frame body data
            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].frame_body.delete();
            frame_body_len = item.frame.ampdu_frame[user].delimit[mpdu_cnt].mpdu_length;

            // subtract MAC header length
            frame_body_len -= item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.size();
            // subtract 4 bytes of FCS
            frame_body_len -= 4;

            if (frame_body_len > 0) begin
              item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].frame_body = new[frame_body_len];
              for (int i=0; i<frame_body_len; i++) begin
                item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].frame_body[i] = tx_stream[user].pop_back();
              end//for
            end

            // store last 4 bytes for FCS
            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].FCS[ 7: 0] = tx_stream[user].pop_back();
            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].FCS[15: 8] = tx_stream[user].pop_back();
            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].FCS[23:16] = tx_stream[user].pop_back();
            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].FCS[31:24] = tx_stream[user].pop_back();

            //make sure frame and frame_size are correct
            item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].post_monitor();
            `uvm_info(get_type_name(), $sformatf("MPDU transmited: %s",item.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].sprint()),UVM_DEBUG)

            // store padding bits
            // padding number is equal to number of bytes that are needed for
            // length to be in power of 4
            pad_num = item.frame.ampdu_frame[user].delimit[mpdu_cnt].mpdu_length % 4;
            pad_num = (pad_num == 0) ? 0 : 4 - pad_num;
            // padding is not added to the last MPDU in A-MPDU
            if (tx_stream[user].size() != 0 && pad_num != 0) begin
              repeat (pad_num) dummy = tx_stream[user].pop_back();
              item.frame.ampdu_frame[user].padding[mpdu_cnt] = pad_num;
              `uvm_info(get_type_name(),$sformatf("Number of padding bytes %0d",pad_num),UVM_DEBUG)
            end

            mpdu_cnt++;

          end// while not empty stream per user
        end// foreach user

      end// if MU_MIMO

//---------------------------------------------------------------------------------------
      else if (item.frame.kind == NDP) begin
//---------------------------------------------------------------------------------------
        item.frame.num_of_users = 0;
      end// if NDP

    end // Tx
    else if (tr == RX) begin
      nonht_rxvector_s        nonht_rxv;
      ht_rxvector_s           ht_rxv;
      vht_rxvector_s          vht_rxv;
      he_su_rxvector_s        he_su_rxv;
      he_mu_rxvector_s        he_mu_rxv;
      he_trig_rxvector_sta_s  he_tb_sta_rxv;
      he_trig_rxvector_ap_s   he_tb_ap_rxv;

      byte_cnt = 1;
      rx_vector_start = new[1];
      rx_vector_start[0] = rx_stream.pop_back();
      case (format_mod_e'(rx_vector_start[0][3:0]))
        NON_HT, NON_HT_DUP_OFDM : begin
          rx_vector_length = 7;
        end
        HT_MF, HT_GF : begin
          rx_vector_length = 10;
        end
        VHT : begin
          rx_vector_length = 13;
        end
        HE_SU, HE_EXT_SU : begin
          rx_vector_length = 16;
        end
        HE_MU : begin
          rx_vector_length = 16;
        end
        HE_TB : begin
          rx_vector_length = 18;
        end
      endcase
      rx_vector_start = new[rx_vector_length](rx_vector_start);
      // fetch RX vector start from input stream
      while (byte_cnt < rx_vector_length) begin
        // store RxVector1 to item data
        rx_vector_start[byte_cnt] = rx_stream.pop_back();
        byte_cnt++;
      end

      foreach (rx_vector_start[i]) begin
        case(format_mod_e'(rx_vector_start[0][3:0]))
          NON_HT, NON_HT_DUP_OFDM : nonht_rxv[i*8 +: 8] = rx_vector_start[i];
          HT_MF, HT_GF            : ht_rxv[i*8 +: 8] = rx_vector_start[i];
          VHT                     : vht_rxv[i*8 +: 8] = rx_vector_start[i];
          HE_SU, HE_EXT_SU        : he_su_rxv[i*8 +: 8] = rx_vector_start[i];
          HE_MU                   : he_mu_rxv[i*8 +: 8] = rx_vector_start[i];
          HE_TB                   : he_tb_ap_rxv[i*8 +: 8] = rx_vector_start[i];
        endcase
      end // foreach

      case(format_mod_e'(rx_vector_start[0][3:0]))
        NON_HT, NON_HT_DUP_OFDM : `uvm_info(get_type_name(), $sformatf("Rx vector collected: %p",nonht_rxv), UVM_LOW)
        HT_MF, HT_GF            : `uvm_info(get_type_name(), $sformatf("Rx vector collected: %p",ht_rxv), UVM_LOW)
        VHT                     : `uvm_info(get_type_name(), $sformatf("Rx vector collected: %p",vht_rxv), UVM_LOW)
        HE_SU, HE_EXT_SU        : `uvm_info(get_type_name(), $sformatf("Rx vector collected: %p",he_su_rxv), UVM_LOW)
        HE_MU                   : `uvm_info(get_type_name(), $sformatf("Rx vector collected: %p",he_mu_rxv), UVM_LOW)
        HE_TB                   : `uvm_info(get_type_name(), $sformatf("Rx vector collected: %p",he_tb_ap_rxv), UVM_LOW)
      endcase

      // store stream item rxvector content
      rx_stream_item.format_mod = format_mod_e'(rx_vector_start[0][3:0]);

      case(format_mod_e'(rx_vector_start[0][3:0]))
        NON_HT, NON_HT_DUP_OFDM : rx_stream_item.nonht_rxv = nonht_rxv;
        HT_MF, HT_GF            : rx_stream_item.ht_rxv    = ht_rxv;
        VHT                     : rx_stream_item.vht_rxv   = vht_rxv;
        HE_SU, HE_EXT_SU        : rx_stream_item.he_su_rxv = he_su_rxv;
        HE_MU                   : rx_stream_item.he_mu_rxv = he_mu_rxv;
        HE_TB                   : rx_stream_item.he_tb_ap_rxv = he_tb_ap_rxv;
      endcase

      //  fill frame fields
      num_of_users = 1;
      item.frame.num_of_users = num_of_users;
      item.frame.ppdu_format  = format_mod_e'(rx_vector_start[0][3:0]);
      item.frame.create_preamble_header();
      item.frame.preamble_header.format_mod = format_mod_e'(rx_vector_start[0][3:0]);
      if (item.frame.ppdu_format inside {HE_SU, HE_EXT_SU, HE_TB, HE_MU}) begin
        item.frame.preamble_header.aggregated = 1'b1;
        item.frame.preamble_header.user_header_he = new[num_of_users];
      end else begin
        item.frame.preamble_header.user_header = new[num_of_users];
      end

      // set preamble and header out of stream data
      item.frame.preamble_header.leg_length    = {rx_vector_start[4][3:0], rx_vector_start[3]};
      item.frame.preamble_header.leg_rate      = rx_vector_start[4][7:4];
      item.frame.preamble_header.ch_bw         = rx_vector_start[0][6:4];
      item.frame.preamble_header.preamble_type = rx_vector_start[0][7];
      item.frame.preamble_header.antenna_set   = rx_vector_start[1];
      item.frame.preamble_header.rssi_legacy   = rx_vector_start[2];
      item.frame.preamble_header.rssi          = rx_vector_start[5];
      // GI type
      if (item.frame.ppdu_format inside {HT_GF, HT_MF, VHT})
        item.frame.preamble_header.gi_type     = {1'b0, rx_vector_start[6][2]};
      if (!(item.frame.ppdu_format inside {NON_HT, NON_HT_DUP_OFDM}))
        item.frame.preamble_header.stbc          = rx_vector_start[4][6:5];
      if (item.frame.ppdu_format inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
        item.frame.preamble_header.sounding         = rx_vector_start[6][0];
        item.frame.preamble_header.beamformed       = rx_vector_start[6][1];
        item.frame.preamble_header.gi_type          = rx_vector_start[6][3:2];
        item.frame.preamble_header.stbc             = rx_vector_start[6][4];
        item.frame.preamble_header.uplink_flag      = rx_vector_start[7][0];
        item.frame.preamble_header.beam_change      = rx_vector_start[7][1];
        item.frame.preamble_header.dcm              = rx_vector_start[7][2];
        item.frame.preamble_header.he_ltf_type      = rx_vector_start[7][4:3];
        item.frame.preamble_header.doppler          = rx_vector_start[7][5];
        item.frame.preamble_header.bss_color        = rx_vector_start[8][5:0];
        item.frame.preamble_header.txop_duration    = rx_vector_start[9][6:0];
      end

      case (item.frame.ppdu_format)
        NON_HT, NON_HT_DUP_OFDM : begin
          item.frame.preamble_header.dyn_bw         = rx_vector_start[6][0];
          item.frame.preamble_header.chbw_in_non_ht = rx_vector_start[6][2:1];
          item.frame.preamble_header.l_sig_valid    = rx_vector_start[6][7];
        end
        HT_MF, HT_GF : begin
          item.frame.preamble_header.sounding                     = rx_vector_start[6][0];
          item.frame.preamble_header.smoothing                    = rx_vector_start[6][1];
          item.frame.preamble_header.aggregated                   = rx_vector_start[6][3];
          item.frame.preamble_header.stbc                         = rx_vector_start[6][4];
          item.frame.preamble_header.num_extn_ss                  = rx_vector_start[6][6:5];
          item.frame.preamble_header.l_sig_valid                  = rx_vector_start[6][7];
          item.frame.preamble_header.user_header[0].mcs_f         = rx_vector_start[7][6:0];
          item.frame.preamble_header.user_header[0].fec_coding_f  = rx_vector_start[7][7];
          item.frame.preamble_header.user_header[0].ht_length_f   = {rx_vector_start[9], rx_vector_start[8]};
        end
        VHT : begin
          item.frame.preamble_header.sounding                     = rx_vector_start[6][0];
          item.frame.preamble_header.beamformed                   = rx_vector_start[6][1];
          item.frame.preamble_header.gi_type[0]                   = rx_vector_start[6][2];
          item.frame.preamble_header.stbc                         = rx_vector_start[6][4];
          item.frame.preamble_header.doze_not_allowed             = rx_vector_start[6][5];
          item.frame.preamble_header.first_user                   = rx_vector_start[6][6];
          item.frame.preamble_header.partial_aid                  = {rx_vector_start[8][0],rx_vector_start[7]};
          item.frame.preamble_header.group_id                     = rx_vector_start[8][6:1];
          item.frame.preamble_header.user_header[0].mcs_f         = rx_vector_start[9][3:0];
          item.frame.preamble_header.user_header[0].num_sts_f     = rx_vector_start[9][6:4];
          item.frame.preamble_header.user_header[0].fec_coding_f  = rx_vector_start[9][7];
          item.frame.preamble_header.user_header[0].ht_length_f   = {rx_vector_start[12],
                                                                     rx_vector_start[11],
                                                                     rx_vector_start[10]};
          item.frame.preamble_header.aggregated                   = 1;
        end
        HE_SU, HE_EXT_SU : begin
          item.frame.preamble_header.pe_duration                      = rx_vector_start[10][3:0];
          item.frame.preamble_header.spatial_reuse[0]                 = rx_vector_start[10][7:4];
          item.frame.preamble_header.user_header_he[0].mcs_f          = rx_vector_start[12][3:0];
          item.frame.preamble_header.user_header_he[0].nss_f          = rx_vector_start[12][6:4];
          item.frame.preamble_header.user_header_he[0].fec_coding_f   = rx_vector_start[12][7];
          item.frame.preamble_header.user_header_he[0].he_length_f    = {rx_vector_start[15][3:0],
                                                                         rx_vector_start[14],
                                                                         rx_vector_start[13]};
          item.frame.preamble_header.aggregated                       = 1;
        end
        HE_MU : begin
          item.frame.preamble_header.pe_duration                      = rx_vector_start[10][3:0];
          item.frame.preamble_header.spatial_reuse[0]                 = rx_vector_start[10][7:4];
          item.frame.preamble_header.sig_b_comp_mode                  = rx_vector_start[11][0];
          item.frame.preamble_header.dcm_sig_b                        = rx_vector_start[11][1];
          item.frame.preamble_header.mcs_sig_b                        = rx_vector_start[11][4:2];
          item.frame.preamble_header.ru_type                          = ru_type_e'(rx_vector_start[11][7:5]);
          item.frame.preamble_header.user_header_he[0].mcs_f          = rx_vector_start[12][3:0];
          item.frame.preamble_header.user_header_he[0].nss_f          = rx_vector_start[12][6:4];
          item.frame.preamble_header.user_header_he[0].fec_coding_f   = rx_vector_start[12][7];
          item.frame.preamble_header.user_header_he[0].he_length_f    = {rx_vector_start[15][3:0],
                                                                         rx_vector_start[14],
                                                                         rx_vector_start[13]};
        end
        HE_TB : begin
          item.frame.preamble_header.n_user                           = rx_vector_start[11];
          item.frame.preamble_header.user_header_he[0].mcs_f          = rx_vector_start[12][3:0];
          item.frame.preamble_header.user_header_he[0].nss_f          = rx_vector_start[12][6:4];
          item.frame.preamble_header.user_header_he[0].fec_coding_f   = rx_vector_start[12][7];
          item.frame.preamble_header.user_header_he[0].he_length_f    = {rx_vector_start[15][3:0],
                                                                         rx_vector_start[14],
                                                                         rx_vector_start[13]};
        end
      endcase

      byte_cnt = 8;
      // fetch RX vector end from input stream, second part of RX vector is
      // sent on the end of reception so data is taken from front
      while (byte_cnt != 0) begin
        rx_vector_end[byte_cnt-1] = rx_stream.pop_front();
        // store RxVector2 data
        rx_stream_item.rx_vector_end[byte_cnt-1] = rx_vector_end[byte_cnt-1];
        byte_cnt--;
      end

      item.frame.preamble_header.rcpi[0] = rx_vector_end[0];
      item.frame.preamble_header.rcpi[1] = rx_vector_end[1];
      item.frame.preamble_header.rcpi[2] = rx_vector_end[2];
      item.frame.preamble_header.rcpi[3] = rx_vector_end[3];
      item.frame.preamble_header.evm[0]  = rx_vector_end[4];
      item.frame.preamble_header.evm[1]  = rx_vector_end[5];
      item.frame.preamble_header.evm[2]  = rx_vector_end[6];
      item.frame.preamble_header.evm[3]  = rx_vector_end[7];

      item.frame.preamble_header.tr = RX;
      // save preable header to configuration
      cfg.pre_hdr.copy(item.frame.preamble_header);

      //------------------------------------------------------------------
      // store Rx payload data to item data
      //------------------------------------------------------------------
      rx_stream_item.rx_stream = new[rx_stream.size()];
      foreach (rx_stream_item.rx_stream[i])
        rx_stream_item.rx_stream[i] = rx_stream[rx_stream.size()-i-1];

      rx_stream_item.no_valid_frames = 0;
      //------------------------------------------------------------------

      if (item.frame.ppdu_format inside {HE_SU, HE_EXT_SU, HE_TB, HE_MU}) begin
        if (num_of_users > 1)
          item.frame.kind = MU_MIMO;
        else if (item.frame.preamble_header.user_header_he[0].he_length_f == 0)
          item.frame.kind = NDP;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 0)
          item.frame.kind = SINGLETON;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 1)
          item.frame.kind = AGGREGATED;
      end
      else begin
        if (num_of_users > 1)
          item.frame.kind = MU_MIMO;
        else if (   (item.frame.preamble_header.user_header[0].ht_length_f == 0)
                 && (item.frame.ppdu_format == VHT))
          item.frame.kind = NDP;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 0)
          item.frame.kind = SINGLETON;
        else if (num_of_users == 1 && item.frame.preamble_header.aggregated == 1)
          item.frame.kind = AGGREGATED;
      end

`ifdef STANDALONE_PHY
      item.frame.ampdu_frame = new[num_of_users];
      foreach (item.frame.ampdu_frame[user]) begin
        item.frame.ampdu_frame[user] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[%0d]",user));
        item.frame.ampdu_frame[user].ampdu_payload = new[rx_stream.size()];

        foreach (item.frame.ampdu_frame[user].ampdu_payload[i]) begin
          item.frame.ampdu_frame[user].ampdu_payload[i] = rx_stream.pop_back();
        end
      end
      // set number of users in case of NDP
      if (item.frame.kind == NDP) begin
        item.frame.num_of_users = 0;
      end// if NDP
      // return from task because only payload is valid
      return;
`endif

//---------------------------------------------------------------------------------------
      // fetch A-MPDU and MPDU frames from stream and store to PPDU object
      if (item.frame.kind == SINGLETON) begin
//---------------------------------------------------------------------------------------
        // create one AMPDU frame
        item.frame.ampdu_frame = new[1];
        item.frame.ampdu_frame[0] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[0]"));
        item.frame.ampdu_frame[0].mpdu_frame_type = new[1];
        item.frame.ampdu_frame[0].mpdu_frame      = new[1];

        // set format in A-MPDU frame
        item.frame.ampdu_frame[0].ppdu_format = item.frame.ppdu_format;

        // determine if frame is DSSS
        if (   (item.frame.ppdu_format == NON_HT)
            && (item.frame.preamble_header.leg_rate inside {[0:3]})) begin
          item.frame.ampdu_frame[0].is_dsss = 1;
        end else begin
          item.frame.ampdu_frame[0].is_dsss = 0;
        end

        // determine frame body length
        frame_len = (   item.frame.preamble_header.format_mod == NON_HT
                     || item.frame.preamble_header.format_mod == NON_HT_DUP_OFDM) ?
                     // then
                     item.frame.preamble_header.leg_length :                 // non HT mode
                     // else
                     item.frame.preamble_header.user_header[0].ht_length_f;  // HT mode

        // when using Matlab model for generating Rx data, there is
        // a possibility that some bytes will be corrupted so those MPDU
        // frames have FCS error and need to be created like that
        if (!check_mpdu_fcs(frame_len-4)) begin

          // in a sigleton frame MAC header is sent first. The first 2 bytes are frame
          // control bytes, from which type and sub-type of MPDU frame is
          // determined
          frame_ctrl[ 7:0] = rx_stream.pop_back();
          frame_ctrl[15:8] = rx_stream.pop_back();
          // determine type and sub-type of frame
          item.frame.ampdu_frame[0].mpdu_frame_type[0] = mpdu_frame_type_e'({frame_ctrl[3:2], frame_ctrl[7:4]});
          // if HE-SU and frame is VHT_NDP_ANNOUNCEMENT then it needs to be
          // converted to HE_NDP_ANNOUNCEMENT
          if (   item.frame.ppdu_format == HE_SU
              && item.frame.ampdu_frame[0].mpdu_frame_type[0] == VHT_NDP_ANNOUNCEMENT) begin
            item.frame.ampdu_frame[0].mpdu_frame_type[0] = HE_NDP_ANNOUNCEMENT;
          end
          item.frame.ampdu_frame[0].create_mpdu_frame_by_type(0);
          // when we know type of frame, randomize it to apply constraints on
          // all fields inside, to make it easy when storing data
          assert (item.frame.ampdu_frame[0].mpdu_frame[0].randomize() with {
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.frame_ctrl == frame_ctrl;
          });

          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.duration_id[ 7:0] = rx_stream.pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.duration_id[15:8] = rx_stream.pop_back();
          // store MAC address
          foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i]) begin
            if (i == 3) break; // don't store 4th address before sequence control
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][ 7: 0] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][15: 8] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][23:16] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][31:24] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][39:32] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[i][47:40] = rx_stream.pop_back();
          end
          foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.seq_ctrl[i]) begin
            // MAC seq_ctrl is contained of 2 ostets
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.seq_ctrl[i][ 7:0] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.seq_ctrl[i][15:8] = rx_stream.pop_back();
          end//for
          // 4th MAC address can only apear after sequence control
          if (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr.size() == 4) begin
            // MAC address is contained of 6 ostets
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][ 7: 0] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][15: 8] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][23:16] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][31:24] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][39:32] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.addr[3][47:40] = rx_stream.pop_back();
          end
          foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.qos_ctrl[i]) begin
            // MAC qos_ctrl is contained of 2 ostets
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.qos_ctrl[i][ 7:0] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.qos_ctrl[i][15:8] = rx_stream.pop_back();
          end//for
          foreach (item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i]) begin
            // MAC ht_ctrl is contained of 4 ostets
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][ 7: 0] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][15: 8] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][23:16] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl[i][31:24] = rx_stream.pop_back();
          end//for

          // store frame body data
          item.frame.ampdu_frame[0].mpdu_frame[0].frame_body.delete();
          frame_body_len = frame_len;
          // subtract MAC header length
          frame_body_len -= item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.size();
          // subtract 4 bytes of FCS
          frame_body_len -= 4;

          item.frame.ampdu_frame[0].mpdu_frame[0].frame_body = new[frame_body_len];
          for (int i=0; i<frame_body_len; i++) begin
            item.frame.ampdu_frame[0].mpdu_frame[0].frame_body[i] = rx_stream.pop_back();
          end//for

          // store last 4 bytes for FCS
          item.frame.ampdu_frame[0].mpdu_frame[0].FCS[ 7: 0] = rx_stream.pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].FCS[15: 8] = rx_stream.pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].FCS[23:16] = rx_stream.pop_back();
          item.frame.ampdu_frame[0].mpdu_frame[0].FCS[31:24] = rx_stream.pop_back();

          //make sure frame and frame_size are correct
          item.frame.ampdu_frame[0].mpdu_frame[0].post_monitor();
          `uvm_info(get_type_name(), $sformatf("MPDU received: %s",item.frame.ampdu_frame[0].mpdu_frame[0].sprint()),UVM_DEBUG)
        end // if check_mpdu_fcs = 0
        else begin
          // create ERROR frame
          item.frame.ampdu_frame[0].mpdu_frame_type[0] = ERROR_FRAME;
          item.frame.ampdu_frame[0].create_mpdu_frame_by_type(0);
          // store Rx stream bytes to MPDU frame array
          item.frame.ampdu_frame[0].mpdu_frame[0].frame = new[rx_stream.size()];
          foreach (item.frame.ampdu_frame[0].mpdu_frame[0].frame[i]) begin
            item.frame.ampdu_frame[0].mpdu_frame[0].frame[i] = rx_stream.pop_back();
          end

          // call post_monitor to set fcs_error flag
          item.frame.ampdu_frame[0].mpdu_frame[0].post_monitor();
          `uvm_info(get_type_name(), $sformatf("MPDU received: %s",item.frame.ampdu_frame[0].mpdu_frame[0].sprint()),UVM_DEBUG)

          // indicate that frame is corrupted
          rx_stream_item.no_valid_frames = 1;
        end // if check_mpdu_fcs = 1

      end// if SINGLETON
//---------------------------------------------------------------------------------------
      else if (item.frame.kind == AGGREGATED) begin
//---------------------------------------------------------------------------------------

        mpdu_cnt = 0;
        // create one AMPDU frame
        item.frame.ampdu_frame = new[1];
        item.frame.ampdu_frame[0] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[0]"));

        // set format in A-MPDU frame
        item.frame.ampdu_frame[0].ppdu_format = item.frame.ppdu_format;

        while (rx_stream.size() != 0) begin
          // extend 3 dynamic arrays by 1 and append previous content to it
          // for mpdu_frame_type, mpdu_frame and delimiter
          item.frame.ampdu_frame[0].mpdu_frame_type =
            new[item.frame.ampdu_frame[0].mpdu_frame_type.size() + 1](item.frame.ampdu_frame[0].mpdu_frame_type);
          item.frame.ampdu_frame[0].mpdu_frame      =
            new[item.frame.ampdu_frame[0].mpdu_frame.size() + 1](item.frame.ampdu_frame[0].mpdu_frame);
          item.frame.ampdu_frame[0].delimit =
            new[item.frame.ampdu_frame[0].delimit.size() + 1](item.frame.ampdu_frame[0].delimit);

          item.frame.ampdu_frame[0].aggregated = 1'b1;
          blk_del_cnt = 0;
          // in aggregated frame delimiter is first sent then MPDU frame and padding,
          // until correct delimiter is found, loop through Rx stream bytes
          while (rx_stream.size() != 0) begin
            delimiter[ 7: 0] = rx_stream.pop_back();
            delimiter[15: 8] = rx_stream.pop_back();
            delimiter[23:16] = rx_stream.pop_back();
            delimiter[31:24] = rx_stream.pop_back();
            // determine if blank delimiter is received, length is 0 is this
            // case and signature 'h4E
            if (delimiter[15:4] == 12'h0 && delimiter[31:24] == 8'h4E) begin
              item.frame.ampdu_frame[0].blank_delimit[mpdu_cnt-1] = new();
              blk_del_cnt++;
              item.frame.ampdu_frame[0].blank_delimit[mpdu_cnt-1].blank_delimiter_num = blk_del_cnt;
              `uvm_info(get_type_name(),
              $sformatf("blank_delimit: %s",item.frame.ampdu_frame[0].blank_delimit[mpdu_cnt-1].sprint()),UVM_DEBUG)
              continue; // while loop
            end
            else if (delimiter[31:24] == 8'h4E && check_delimiter_crc(delimiter) == 0) begin
              break; // exit while loop because delimiter is found
            end
          end //while (1)

          // if there was no valid delimiter in the stream, exit while loop
          if (rx_stream.size() == 0) begin
            if (mpdu_cnt == 0) begin
              `uvm_info(get_type_name(),$sformatf("No valid delimiters in the Rx stream"),UVM_HIGH)
              item.frame.ampdu_frame[0].clear_all_arrays(); // delete allocated objects
              item.frame.ampdu_frame[0].no_valid_frames = 1;
              rx_stream_item.no_valid_frames = 1;
            end
            else begin
              // remove dynamic entries that are not collected
              item.frame.ampdu_frame[0].mpdu_frame_type = new[mpdu_cnt](item.frame.ampdu_frame[0].mpdu_frame_type);
              item.frame.ampdu_frame[0].mpdu_frame      = new[mpdu_cnt](item.frame.ampdu_frame[0].mpdu_frame);
              item.frame.ampdu_frame[0].delimit         = new[mpdu_cnt](item.frame.ampdu_frame[0].delimit);
            end
            break; // exit while loop rx_stream.size()
          end

          item.frame.ampdu_frame[0].delimit[mpdu_cnt] = frame_model_pkg::delimiter::type_id::create($sformatf("delimiter[%0d]",mpdu_cnt));
          item.frame.ampdu_frame[0].delimit[mpdu_cnt].set_data(delimiter);
          `uvm_info(get_type_name(),$sformatf("delimiter[%0d] = %0h",mpdu_cnt,delimiter),UVM_HIGH)

          // when using Matlab model for generating Rx data, there is
          // a possibility that some bytes will be corrupted so those MPDU
          // frames have FCS error and need to be created like that
          if (!check_mpdu_fcs(item.frame.ampdu_frame[0].delimit[mpdu_cnt].mpdu_length-4)) begin

            // in sigleton frame MAC header is first sent, first 2 bytes are frame
            // control bytes, from which type and sub-type of MPDU frame is
            // determined
            frame_ctrl[ 7:0] = rx_stream.pop_back();
            frame_ctrl[15:8] = rx_stream.pop_back();
            `uvm_info(get_type_name(),$sformatf("frame_ctrl[%0d] = %0h",mpdu_cnt,frame_ctrl),UVM_HIGH)
            // determine type and sub-type of frame
            item.frame.ampdu_frame[0].mpdu_frame_type[mpdu_cnt] = mpdu_frame_type_e'({frame_ctrl[3:2], frame_ctrl[7:4]});
            // if HE-SU and frame is VHT_NDP_ANNOUNCEMENT then it needs to be
            // converted to HE_NDP_ANNOUNCEMENT
            if (   item.frame.ppdu_format == HE_SU
                && item.frame.ampdu_frame[0].mpdu_frame_type[mpdu_cnt] == VHT_NDP_ANNOUNCEMENT) begin
              item.frame.ampdu_frame[0].mpdu_frame_type[mpdu_cnt] = HE_NDP_ANNOUNCEMENT;
            end
            item.frame.ampdu_frame[0].create_mpdu_frame_by_type(mpdu_cnt);
            // when we know type of frame, randomize it to apply constraints on
            // all fields inside, to make it easy when storing data
            assert (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].randomize() with {
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.frame_ctrl == frame_ctrl;
            });

            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.duration_id[ 7:0] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.duration_id[15:8] = rx_stream.pop_back();
            // store MAC address
            foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i]) begin
              if (i == 3) break; // don't store 4th address before sequence control
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][ 7: 0] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][15: 8] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][23:16] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][31:24] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][39:32] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[i][47:40] = rx_stream.pop_back();
            end
            foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i]) begin
              // MAC seq_ctrl is contained of 2 ostets
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i][ 7:0] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.seq_ctrl[i][15:8] = rx_stream.pop_back();
            end//for
            // 4th MAC address can only apear after sequence control
            if (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr.size() == 4) begin
              // MAC address is contained of 6 ostets
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][ 7: 0] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][15: 8] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][23:16] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][31:24] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][39:32] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.addr[3][47:40] = rx_stream.pop_back();
            end
            foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i]) begin
              // MAC qos_ctrl is contained of 2 ostets
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i][ 7:0] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.qos_ctrl[i][15:8] = rx_stream.pop_back();
            end//for
            foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i]) begin
              // MAC ht_ctrl is contained of 4 ostets
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][ 7: 0] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][15: 8] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][23:16] = rx_stream.pop_back();
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.ht_ctrl[i][31:24] = rx_stream.pop_back();
            end//for

            // store frame body data
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame_body.delete();
            frame_body_len = item.frame.ampdu_frame[0].delimit[mpdu_cnt].mpdu_length;

            // subtract MAC header length
            frame_body_len -= item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].MAC_header.size();
            // subtract 4 bytes of FCS
            frame_body_len -= 4;

            if (frame_body_len > 0) begin
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame_body = new[frame_body_len];
              for (int i=0; i<frame_body_len; i++) begin
                item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame_body[i] = rx_stream.pop_back();
              end//for
            end

            // store last 4 bytes for FCS
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[ 7: 0] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[15: 8] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[23:16] = rx_stream.pop_back();
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].FCS[31:24] = rx_stream.pop_back();

            //make sure frame and frame_size are correct
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].post_monitor();
            `uvm_info(get_type_name(), $sformatf("MPDU received: %s",item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].sprint()),UVM_DEBUG)

            // store padding bits
            // padding number is equal to number of bytes that are needed for
            // length to be in power of 4
            pad_num = item.frame.ampdu_frame[0].delimit[mpdu_cnt].mpdu_length % 4;
            pad_num = (pad_num == 0) ? 0 : 4 - pad_num;
            if (rx_stream.size() != 0 && pad_num != 0) begin
              repeat (pad_num) dummy = rx_stream.pop_back();
              item.frame.ampdu_frame[0].padding[mpdu_cnt] = pad_num;
            end

            if ((rx_stream.size() != 0) && (check_zero_rx_stream() == 1'b1)) begin
              `uvm_info(get_type_name(), $sformatf("Removing %0d zero octets from stream",rx_stream.size()),UVM_DEBUG)
              repeat(rx_stream.size()) dummy = rx_stream.pop_back();
            end

          end// check_mpdu_fcs = 0
          else begin
            // create ERROR frame
            item.frame.ampdu_frame[0].mpdu_frame_type[mpdu_cnt] = ERROR_FRAME;
            item.frame.ampdu_frame[0].create_mpdu_frame_by_type(mpdu_cnt);
            // store Rx stream bytes to MPDU frame array
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame
              = new[item.frame.ampdu_frame[0].delimit[mpdu_cnt].mpdu_length];

            foreach (item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame[i]) begin
              item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].frame[i] = rx_stream.pop_back();
            end

            // store padding bits
            // padding number is equal to number of bytes that are needed for
            // length to be in power of 4
            pad_num = item.frame.ampdu_frame[0].delimit[mpdu_cnt].mpdu_length % 4;
            pad_num = (pad_num == 0) ? 0 : 4 - pad_num;
            if (rx_stream.size() != 0 && pad_num != 0) begin
              repeat (pad_num) dummy = rx_stream.pop_back();
              item.frame.ampdu_frame[0].padding[mpdu_cnt] = pad_num;
            end

            if ((rx_stream.size() != 0) && (check_zero_rx_stream() == 1'b1)) begin
              `uvm_info(get_type_name(), $sformatf("Removing %0d zero octets from stream",rx_stream.size()),UVM_DEBUG)
              repeat(rx_stream.size()) dummy = rx_stream.pop_back();
            end

            // call post_monitor to set fcs_error flag
            item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].post_monitor();
            `uvm_info(get_type_name(), $sformatf("MPDU received: %s",item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt].sprint()),UVM_DEBUG)
          end // if check_mpdu_fcs = 1

          mpdu_cnt++;

        end// while not empty

        // when one frame is sent as VHT, MAC sends it as agreggated
        // but it is a singleton frame
        if (   mpdu_cnt == 1
            && item.frame.ampdu_frame[0].delimit[0].eof == 1
            && item.frame.ppdu_format inside {VHT, HE_SU, HE_EXT_SU, HE_MU, HE_TB}
           ) begin

          item.frame.ampdu_frame[0].aggregated = 0;
          item.frame.kind = SINGLETON;
        end

      end// if AGGREGATED

//---------------------------------------------------------------------------------------
      else if (item.frame.kind == NDP) begin
//---------------------------------------------------------------------------------------
        item.frame.num_of_users = 0;
      end// if NDP

    end// RX
  endtask : convert_stream2item

  //-------------------------------------------------------
  // detect PHY error during Rx
  //-------------------------------------------------------
  task detect_rx_err();
    rx_err = 0;
    forever begin
      @(posedge vif.rx_err);
      `uvm_info(get_type_name(), "PHY RX ERR detected...", UVM_LOW)
      rx_err = 1;
    end
  endtask : detect_rx_err

  //-------------------------------------------------------
  // detect PHY error during Tx
  //-------------------------------------------------------
  task detect_phy_err();
    phy_err = 0;
    forever begin
      @(posedge vif.phy_err);
      `uvm_info(get_type_name(), "PHY ERR detected...", UVM_LOW)
      phy_err = 1;
    end
  endtask : detect_phy_err

  //-------------------------------------------------------
  // function that will check FCS of recevied frame before
  // creating PPDU frame objects, this will avoid creation
  // of wrong Rx data.
  // @return : 1 - ERROR FCS
  //           0 - OK FCS
  //-------------------------------------------------------
  function bit check_mpdu_fcs(int len);
    octet_t    frame[];
    bit [31:0] fcs;
    bit [31:0] fcs_rcvd;

    frame = new[len]; // create temp frame for received bytes

    for (int i=0; i<len; i++) begin
      frame[i] = rx_stream[rx_stream.size()-i-1];
    end
    fcs = fcs_calc_func(len, frame);
    fcs_rcvd[ 7: 0] = rx_stream[rx_stream.size()-len-1];
    fcs_rcvd[15: 8] = rx_stream[rx_stream.size()-len-2];
    fcs_rcvd[23:16] = rx_stream[rx_stream.size()-len-3];
    fcs_rcvd[31:24] = rx_stream[rx_stream.size()-len-4];

    `uvm_info(get_type_name(), $sformatf("check_mpdu_fcs: FCS (%0h), FCS RCVD (%0h)",fcs,fcs_rcvd),UVM_DEBUG)

    return (fcs == fcs_rcvd) ? 0 : 1;
  endfunction : check_mpdu_fcs

  //-------------------------------------------------------
  // function that will check CRC of recevied delimiter before
  // creating PPDU frame objects, this will avoid creation
  // of wrong Rx data.
  // @return : 1 - ERROR
  //           0 - OK
  //-------------------------------------------------------
  function bit check_delimiter_crc(bit [31:0] del);
    bit [7:0] crc;

    crc = delimiter_crc_func({del[15:4], del[3:2], del[1], del[0]});

    `uvm_info(get_type_name(), $sformatf("check_delimiter_crc: CRC (%0h), CRC RCVD (%0h)",crc,del[23:16]),UVM_DEBUG)

    return (crc == del[23:16]) ? 0 : 1;
  endfunction : check_delimiter_crc

  function void perform_checks();
  endfunction : perform_checks

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase


endclass : mac_phy_monitor

`endif //MAC_MONITOR_SV
