//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MEM_DRIVER_SV
`define MEM_DRIVER_SV


class mem_driver#(int WIDTH = 8) extends uvm_driver #(mem_seq_item#(WIDTH));

  `uvm_component_param_utils(mem_driver#(WIDTH))

  virtual mem_if #(WIDTH) vif;
  mem_config              cfg;
  symb_duration_s         symb_dur;
  time                    fft_wr_time; // time driver needs to write symbol data to FFT (@120MHz)
  int                     num_he_ltf;

  function new (string name = "mem_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual mem_if #(WIDTH))::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    fork
      get_and_drive();
      /* add other tasks here */
    join_none;
  endtask : run_phase

  task initialize();
    vif.td_done <= 0;
    if (cfg.is_read_mem) begin
      vif.ren    <= 0;
      vif.rsel   <= 0;
      vif.rindex <= 0;
      vif.rlen   <= 0;
      vif.rps    <= 0;
    end
    else begin
      vif.wpi    <= 0;
      vif.wpq    <= 0;
      vif.wsi    <= 0;
      vif.wsq    <= 0;
      vif.wen    <= 0;
      vif.wsel   <= 0;
      vif.windex <= 0;
      vif.wlen   <= 3'b111;
      vif.wscale <= 0;
      vif.start  <= 0;
      vif.dir    <= 0;
      vif.sel    <= 0;
    end

    wait (vif.rst_n == 1'b1);
  endtask : initialize

  //-----------------------------------------------------------------
  // get sequence item and drive signals
  //-----------------------------------------------------------------
  task get_and_drive();
    initialize();

    forever begin
      seq_item_port.get_next_item(req);

      if (cfg.is_read_mem)
        drive_read_path();
      else
        drive_write_path();

      seq_item_port.item_done(req);
    end // forever
  endtask : get_and_drive

  //-----------------------------------------------------------------
  // drive signals for write path of memory
  //-----------------------------------------------------------------
  task drive_write_path();
    int symb = 0;
    int offset = 0;
    int hesigb_cnt = 0;
    int l_ltf_sig_cnt = 0;

    `uvm_info(get_type_name(),$sformatf("Write to FFT input (%0d samples)",req.fft_in_data.size()),UVM_LOW)

    @(posedge vif.clk iff vif.enable == 1'b1);
    vif.dir    <= 1'b0;
    vif.sel    <= 3'b000;
    symb_dur   = DUR_4_0US;

    // Num HE-LTF enconding
    num_he_ltf = (req.num_he_ltf <= 6) ? (req.num_he_ltf+1) : 8;

    for (int i=0; i<req.fft_in_data.size();) begin
      if (symb == req.fft_in_data[i].symb_idx_f) begin
        @(posedge vif.clk);
        vif.wen    <= 1'b1;
        vif.windex <= req.fft_in_data[i].index_f;
        vif.wpi    <= req.fft_in_data[i].re_f;
        vif.wpq    <= req.fft_in_data[i].im_f;
`ifdef RW_NX_CHBW4020
        if (req.fft_in_data[i].rxfield_f inside {L_LTF,L_SIG}) begin
          offset = req.fft_in_data[i].fft_size_f;
          vif.wsi  <= req.fft_in_data[i+offset].re_f;
          vif.wsq  <= req.fft_in_data[i+offset].im_f;
          l_ltf_sig_cnt++;
        end
        else if (req.ch_bw == 2'b01 && req.fft_in_data[i].rxfield_f == HE_SIGB) begin
          // matlab stores samples for HESIGB: PPPSSS (3 symbol case)
          offset = req.num_he_sig_b * req.fft_in_data[i].fft_size_f;
          vif.wsi  <= req.fft_in_data[i+offset].re_f;
          vif.wsq  <= req.fft_in_data[i+offset].im_f;
          hesigb_cnt++;
        end
        else begin
          offset = 0;
          hesigb_cnt = 0;
          l_ltf_sig_cnt = 0;
          vif.wsi  <= 0;
          vif.wsq  <= 0;
        end
`endif//RW_NX_CHBW4020

        // WSEL driving
        case (req.fft_in_data[i].fft_size_f)
          12'd64:  begin

`ifdef RW_NX_CHBW4020
            if (req.fft_in_data[i].rxfield_f inside {L_LTF,L_SIG,HE_SIGB})
              vif.wlen <= 3'b110;
            else
              vif.wlen <= 3'b000;
`else
            vif.wlen <= 3'b000;
`endif//RW_NX_CHBW4020

            vif.wscale <= 4'b0000;
            fft_wr_time = 0.5us;
          end
          12'd128: begin
            vif.wlen <= 3'b001;
            vif.wscale <= 4'b0001;
            fft_wr_time = 1.0us;
          end
          12'd256: begin
            vif.wlen <= 3'b010;
            vif.wscale <= 4'b0010;
            fft_wr_time = 2.1us;
          end
          12'd512: begin
            vif.wlen <= 3'b011;
            vif.wscale <= 4'b0011;
            fft_wr_time = 4.2us;
          end
          default: begin
            vif.wlen <= 3'b111;
            vif.wscale <= 4'b1111;
            fft_wr_time = 0.0us;
          end
        endcase

        if (req.format_mod inside {HE_MU, HE_SU})begin
          // Starting
          if (symb == (req.num_he_sig_b + 5)) begin
            case (req.he_ltf_type)
              2'b00: symb_dur = DUR_3_2US;
              2'b01: symb_dur = DUR_6_4US;
              2'b10: symb_dur = DUR_12_8US;
            endcase
          end
          else if (symb > (num_he_ltf) + req.num_he_sig_b + 4) begin
            case (req.gi_type)
              2'b01: symb_dur = DUR_13_6US;
              2'b10: symb_dur = DUR_14_4US;
              2'b11: symb_dur = DUR_16_0US;
            endcase
          end
        end
        // increment counter in case of secondary samples by offset
        if (l_ltf_sig_cnt == offset || hesigb_cnt == offset) begin
          i = i + offset;
        end
`ifdef RW_NX_CHBW4020
        // toggle WEN in case of secondary
        if (req.fft_in_data[i].rxfield_f inside {L_LTF,L_SIG,HE_SIGB}) begin
          @(posedge vif.clk);
          vif.wen <= 1'b0;
        end
`endif//RW_NX_CHBW4020

        i++;
      end// if (symb == req.fft_in_data[i].symb_idx_f)
      else begin// symbol number changed
        // increment symbol count in case of secondary
        if (offset != 0 && l_ltf_sig_cnt == offset) begin
          symb += 2;
          l_ltf_sig_cnt = 0;
        end
        else if (offset != 0 && hesigb_cnt == offset) begin
          symb += req.num_he_sig_b + 1;
          hesigb_cnt = 0;
        end
        else begin
          symb++;
        end

        `uvm_info(get_type_name(),$sformatf("Symbol count %0d, Symbol duration: %s",symb, symb_dur.name()),UVM_DEBUG)
        drive_start_and_wait_done();
      end//else
    end//for

    if (req.format_mod == NON_HT) begin
      @(posedge vif.clk);
      vif.td_done <= 1'b1;
      @(posedge vif.clk);
      vif.td_done <= 1'b0;
    end
    drive_start_and_wait_done(1);
    if (!(req.format_mod == NON_HT)) begin
      vif.td_done <= 1'b1;
      @(posedge vif.clk);
      vif.td_done <= 1'b0;
    end
    initialize();
  endtask : drive_write_path

  //-----------------------------------------------------------------
  // trigger start signal and wait for done
  //-----------------------------------------------------------------
  task drive_start_and_wait_done(bit wait_fddone = 0);
    @(posedge vif.clk);
    vif.wpi    <= 0;
    vif.wpq    <= 0;
    vif.wsi    <= 0;
    vif.wsq    <= 0;
    vif.wen    <= 0;
    vif.windex <= 0;
    vif.wsel   <= ~vif.wsel;
    // start FFT processing
    @(posedge vif.clk);
    vif.start <= 1'b1;
    @(posedge vif.clk);
    vif.start <= 1'b0;

    fork
      begin
        // wait FFT done
        @(posedge vif.done);
        vif.sel[0] <= ~vif.sel[0];
        if (wait_fddone)
          @(posedge vif.done);
      end

      case (symb_dur)
        DUR_3_2US:  #(3.2us  - fft_wr_time);
        DUR_4_0US:  #(4.0us  - fft_wr_time);
        DUR_6_4US:  #(6.4us  - fft_wr_time);
        DUR_12_8US: #(12.8us - fft_wr_time);
        DUR_13_6US: #(13.6us - fft_wr_time);
        DUR_14_4US: #(14.4us - fft_wr_time);
        DUR_16_0US: #(16.0us - fft_wr_time);
      endcase
    join
  endtask : drive_start_and_wait_done

  //-----------------------------------------------------------------
  // drive signals for read path of memory
  //-----------------------------------------------------------------
  task drive_read_path();

    `uvm_info(get_type_name(),
    $sformatf("Read from memory %0d symbols (%s)",req.read_nsymb,req.format_mod.name()),UVM_LOW)

    // Num HE-LTF enconding
    num_he_ltf = (req.num_he_ltf <= 6) ? (req.num_he_ltf+1) : 8;

    // loop read out for every symbol
    for (int symb=0; symb<req.read_nsymb; symb++) begin

      @(posedge vif.clk iff vif.done); //wait FFT done to read
      case (req.format_mod)
        NON_HT, NON_HT_DUP_OFDM: begin
          foreach (NON_HT_INDEX[i]) begin
            @(posedge vif.clk);
            vif.rindex <= NON_HT_INDEX[i];
            vif.ren    <= 1'b1;
          end
        end//NON_HT

        HT_MF: begin
          // first read legacy part
          if (symb <= 3) begin
            foreach (NON_HT_INDEX[i]) begin
              @(posedge vif.clk);
              vif.rindex <= NON_HT_INDEX[i];
              vif.ren    <= 1'b1;
            end
          end
          else begin
            foreach (HT_INDEX[req.ch_bw][i]) begin
              @(posedge vif.clk);
              vif.rindex <= HT_INDEX[req.ch_bw][i];
              vif.ren    <= 1'b1;
            end
          end//if
        end//HT_MF

        HT_GF: begin
          // first read legacy part
          if (symb == 1 || symb == 2) begin
            foreach (NON_HT_INDEX[i]) begin
              @(posedge vif.clk);
              vif.rindex <= NON_HT_INDEX[i];
              vif.ren    <= 1'b1;
            end
          end
          else begin
            foreach (HT_INDEX[i]) begin
              @(posedge vif.clk);
              vif.rindex <= HT_INDEX[req.ch_bw][i];
              vif.ren    <= 1'b1;
            end
          end//if
        end//HT_GF

        VHT: begin
          // first read legacy part
          if (symb <= 3) begin
            foreach (NON_HT_INDEX[i]) begin
              @(posedge vif.clk);
              vif.rindex <= NON_HT_INDEX[i];
              vif.ren    <= 1'b1;
            end
          end
          else begin
            foreach (VHT_INDEX[req.ch_bw][i]) begin
              @(posedge vif.clk);
              vif.rindex <= VHT_INDEX[req.ch_bw][i];
              vif.ren    <= 1'b1;
            end
          end//if
        end//VHT

        HE_SU, HE_EXT_SU: begin
          // legacy part
          if (symb <= 4) begin
            foreach (NON_HT_INDEX[i]) begin
              @(posedge vif.clk);
              vif.rindex <= NON_HT_INDEX[i];
              vif.ren    <= 1'b1;
            end
          end
          // HE-LTF
          else if ((symb > 4) && (symb <= 4 + num_he_ltf)) begin
            foreach (HE_LTF_INDEX[req.ch_bw][req.he_ltf_type][i]) begin
              @(posedge vif.clk);
              vif.rindex <= HE_LTF_INDEX[req.ch_bw][req.he_ltf_type][i];
              vif.ren    <= 1'b1;
            end
          end
          // data
          else begin
            foreach (HE_SU_INDEX[req.ch_bw][i]) begin
              @(posedge vif.clk);
              vif.rindex <= HE_SU_INDEX[req.ch_bw][i];
              vif.ren    <= 1'b1;
            end
          end//if
        end//HE

        HE_MU: begin
          // legacy part + HE-SIGA + HE-SIGB
          if (symb <= 4 + req.num_he_sig_b) begin
            foreach (NON_HT_INDEX[i]) begin
              @(posedge vif.clk);
              vif.rindex <= NON_HT_INDEX[i];
              vif.ren    <= 1'b1;
            end
          end
          // HE-LTF
          else if ((symb > (4 + req.num_he_sig_b)) && (symb <= (4 + req.num_he_sig_b + num_he_ltf))) begin
            foreach (HE_LTF_INDEX[req.ch_bw][req.he_ltf_type][i]) begin
              @(posedge vif.clk);
              vif.rindex <= HE_LTF_INDEX[req.ch_bw][req.he_ltf_type][i];
              vif.ren    <= 1'b1;
            end
          end
          // data
          else begin
            // 20 MHz
            if (req.ch_bw == 0) begin
              case (req.ru_type)
                RU26  : begin
                          foreach (HE_INDEX_RU26[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU26[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU52  : begin
                          foreach (HE_INDEX_RU52[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU52[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU106 : begin
                          foreach (HE_INDEX_RU106[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU106[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU242 : begin
                          foreach (HE_INDEX_RU242[i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU242[i];
                            vif.ren    <= 1'b1;
                          end
                        end

              endcase
            end
            //40 MHz
            else if (req.ch_bw == 1) begin
              case (req.ru_type)
                RU26  : begin
                          foreach (HE_INDEX_RU26_40[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU26_40[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU52  : begin
                          foreach (HE_INDEX_RU52_40[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU52_40[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU106 : begin
                          foreach (HE_INDEX_RU106_40[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU106_40[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU242 : begin
                          foreach (HE_INDEX_RU242_40[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU242_40[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU484 : begin
                          foreach (HE_INDEX_RU484[i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU484[i];
                            vif.ren    <= 1'b1;
                          end
                        end
              endcase
            end
            // 80 MHz
            else if (req.ch_bw == 2) begin
              case (req.ru_type)
                RU26  : begin
                          foreach (HE_INDEX_RU26_80[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU26_80[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU52  : begin
                          foreach (HE_INDEX_RU52_80[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU52_80[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU106 : begin
                          foreach (HE_INDEX_RU106_80[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU106_80[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU242 : begin
                          foreach (HE_INDEX_RU242_80[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU242_80[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU484 : begin
                          foreach (HE_INDEX_RU484_80[req.dut_location][i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU484_80[req.dut_location][i];
                            vif.ren    <= 1'b1;
                          end
                        end
                RU996 : begin
                          foreach (HE_INDEX_RU996[i]) begin
                            @(posedge vif.clk);
                            vif.rindex <= HE_INDEX_RU996[i];
                            vif.ren    <= 1'b1;
                          end
                        end
              endcase
            end
          end//if
        end//HE
      endcase
      @(posedge vif.clk);
      vif.rindex <= 0;
      vif.ren    <= 0;
      vif.rps    <= 0;

`ifdef RW_NX_CHBW4020
      // read secondary for L-LTF and L-SIG
      if (symb inside {0,1}) begin
        //wait some time before read out of secondary
        repeat (15) @(posedge vif.clk);

        foreach (NON_HT_INDEX[i]) begin
          @(posedge vif.clk);
          vif.rps    <= 1'b1;
          vif.rindex <= NON_HT_INDEX[i];
          vif.ren    <= 1'b1;
        end
      end
      // read secondary for HE-SIGB
      else if (   req.format_mod == 4'd6
               && req.ch_bw == 2'b01
               && symb > 4
               && symb <= (4 + req.num_he_sig_b)
              ) begin

        //wait some time before read out of secondary
        repeat (15) @(posedge vif.clk);

        foreach (NON_HT_INDEX[i]) begin
          @(posedge vif.clk);
          vif.rps    <= 1'b1;
          vif.rindex <= NON_HT_INDEX[i];
          vif.ren    <= 1'b1;
        end
      end

      @(posedge vif.clk);
      vif.rindex <= 0;
      vif.ren    <= 0;
      vif.rps    <= 0;
`endif

      @(posedge vif.clk);
      vif.rsel   <= ~vif.rsel;

    end// for
    `uvm_info(get_type_name(),"Read memory done!",UVM_LOW)
  endtask : drive_read_path

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : mem_driver

`endif
