//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MEM_MONITOR_SV
`define MEM_MONITOR_SV


class mem_monitor#(int WIDTH = 8) extends uvm_monitor;

  virtual mem_if #(WIDTH) vif;
  mem_config              cfg;
  uvm_analysis_port #(mem_seq_item#(WIDTH)) ap;
  mem_seq_item#(WIDTH)    item;

  `uvm_component_param_utils(mem_monitor#(WIDTH))

  function new(string name = "mem_monitor", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual mem_if#(WIDTH))::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

    ap = new("ap", this);
  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    set_active_flag();

    fork
      collect_transaction();
      /* add some task calls here */
    join_none;
  endtask : run_phase


  task collect_transaction();
    forever begin
      // collect if not in reset
      wait (vif.rst_n == 1'b1);

      item = mem_seq_item#(WIDTH)::type_id::create("item");
      void'(begin_tr(item));

      fork
        if (cfg.is_read_mem)
          collect_read_transaction();
        else
          collect_write_transaction();
      join

      `uvm_info(get_type_name(), $sformatf("Transaction collected :\n%s", item.sprint()), UVM_HIGH)

      if (cfg.has_checks)
        perform_checks();

      ap.write(item);
      end_tr(item);
    end// forever
  endtask : collect_transaction

  //-------------------------------------------------------
  // collect write transactions per symbol
  //-------------------------------------------------------
  task collect_write_transaction();
    int        index;
    FFT_data_s sample;
    FFT_data_s sample_sec;
    int        fft_size;

    wait (vif.wlen != 3'b111); //symbol start

    forever begin
      @(posedge vif.clk iff vif.wen or posedge vif.start);
      if (vif.start) break; // exit loop
      // determine index range
      case (vif.wlen)
        3'b000:  begin index = vif.windex[5:0]; fft_size = 64;   end// FFT64
        3'b001:  begin index = vif.windex[6:0]; fft_size = 128;  end// FFT128
        3'b010:  begin index = vif.windex[7:0]; fft_size = 256;  end// FFT256
        3'b011:  begin index = vif.windex[8:0]; fft_size = 512;  end// FFT512
        3'b100:  begin index = vif.windex;      fft_size = 1024; end// FFT1024
        3'b110:  begin index = vif.windex;      fft_size = 64;   end// 2xFFT64
        default: begin index = vif.windex;      fft_size = 0;    end// no size
      endcase
      sample.re_f          = vif.wpi;
      sample.im_f          = vif.wpq;
      sample.index_f       = index;
      sample.symb_idx_f    = vif.symb_cnt;
      sample.nsymb_f       = 0;
      sample.fft_size_f    = fft_size;
      item.fft_data[index] = sample;
      // collect secondary samples
      if (vif.wlen == 3'b110) begin
        sample_sec.re_f          = vif.wsi;
        sample_sec.im_f          = vif.wsq;
        sample_sec.index_f       = index;
        sample_sec.symb_idx_f    = vif.symb_cnt;
        sample_sec.nsymb_f       = 0;
        sample_sec.fft_size_f    = fft_size;
        item.fft_data_sec[index] = sample_sec;
      end//if
    end//forever
    @(posedge vif.clk);
  endtask : collect_write_transaction

  //-------------------------------------------------------
  // collect read transactions per symbol
  //-------------------------------------------------------
  task collect_read_transaction();
    bit signed [12:0] re[$];
    bit signed [12:0] im[$];
    int               index[$], i;
    bit               secondary;

    fork
      // collect address
      forever begin
        @(posedge vif.clk iff vif.ren or negedge vif.ren);
        if (vif.ren == 1'b0) break;
        index.push_front(signed'(vif.rindex));//save address
      end//forever
      // collect data samples
      forever begin
        @(posedge vif.clk iff (vif.rvalid || vif.last) or negedge vif.rvalid);
        if (vif.rvalid == 0 && vif.last == 0) break;
        if (vif.rvalid) begin
          re.push_front(vif.ri);
          im.push_front(vif.rq);
        end
        if (vif.last == 1'b1) begin
          @(negedge vif.last);
          break;
        end
      end//forever
      // determine primary or secondary
      begin
        @(posedge vif.clk iff vif.ren)
        secondary = (vif.rps === 1'b1) ? 1'b1 : 1'b0;
      end
    join
    // store samples to array in item
    while (index.size() > 0) begin
      i = index.pop_back();
      // secondary
      if (secondary) begin
        item.fft_data_sec[i].re_f       = re.pop_back();
        item.fft_data_sec[i].im_f       = im.pop_back();
        item.fft_data_sec[i].index_f    = i;
        item.fft_data_sec[i].symb_idx_f = vif.symb_cnt-1;
        item.fft_data_sec[i].nsymb_f    = 0;
        item.fft_data_sec[i].fft_size_f = 0;
      end
      // primary
      else begin
        item.fft_data[i].re_f       = re.pop_back();
        item.fft_data[i].im_f       = im.pop_back();
        item.fft_data[i].index_f    = i;
        item.fft_data[i].symb_idx_f = vif.symb_cnt-1;
        item.fft_data[i].nsymb_f    = 0;
        item.fft_data[i].fft_size_f = 0;
      end
    end//while
  endtask : collect_read_transaction

  //-------------------------------------------------------
  // set flag inside virtual interface to indicate is
  // interface active or passive, needed when foricing signals
  //-------------------------------------------------------
  function void set_active_flag();
    if (cfg.is_active == UVM_PASSIVE) begin
      vif.is_read_active  = 0;
      vif.is_write_active = 0;
    end
    else if (cfg.is_active == UVM_ACTIVE && cfg.is_read_mem == 1) begin
      vif.is_read_active  = 1;
      vif.is_write_active = 0;
    end
    else begin
      vif.is_read_active  = 0;
      vif.is_write_active = 1;
    end
  endfunction : set_active_flag

  function void perform_checks();
  endfunction : perform_checks

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : mem_monitor

`endif //MEM_MONITOR_SV
