//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MODEM_STREAM_DRIVER_SV
`define MODEM_STREAM_DRIVER_SV


class modem_stream_driver#(int WIDTH = 8) extends uvm_driver #(modem_stream_seq_item#(WIDTH));

  `uvm_component_param_utils(modem_stream_driver#(WIDTH))

  virtual modem_stream_if #(WIDTH) vif;
  modem_stream_config              cfg;

  bit phy_disabled_adc;   // set to 1 on falling edge of ADCOn0
  bit ADCOn0_d;

  function new (string name = "modem_stream_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual modem_stream_if #(WIDTH))::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    fork
      get_and_drive();
      detect_adc_on_edges();
      /* add other tasks here */
    join_none;
  endtask : run_phase

  // Matlab generates trailing noise samples, but PHY RTL finishes reception before all samples have been sent.
  // This task detects when this happens and prevents the driver from waiting for ADCOn0 indefinitely
  task detect_adc_on_edges();
    forever begin
      ADCOn0_d = vif.On0;
      @(posedge vif.clk);
      if ((ADCOn0_d === 1'b1) && (vif.On0 === 1'b0)) begin
        // PHY RTL disabled ADC, do not send any data
        `uvm_info(get_type_name(), "ADC has been disabled.", UVM_HIGH)
        phy_disabled_adc = 1'b1;
      end else if ((ADCOn0_d === 1'b0) && (vif.On0 === 1'b1)) begin
        // ADC has been (re-)enabled, clear phy_disabled_adc
        `uvm_info(get_type_name(), "ADC has been enabled.", UVM_HIGH)
        phy_disabled_adc = 1'b0;
      end
    end
  endtask : detect_adc_on_edges

  task initialize();
    vif.DataI0      <= 'h0;
    vif.DataQ0      <= 'h0;
    vif.DataI1      <= 'h0;
    vif.DataQ1      <= 'h0;
    vif.DataEn0     <= 'h0;
    vif.DataEn1     <= 'h0;
    vif.data_valid  <= 'h0;
    phy_disabled_adc = 'b0;
    ADCOn0_d         = 'b0;
    wait (vif.rst_n == 1'b1);
  endtask : initialize

  task get_and_drive();
    initialize();

    forever begin
      seq_item_port.get_next_item(req);
      if (phy_disabled_adc == 1'b0) begin
        `uvm_info(get_type_name(), "Start of ADC bus cycle detected.", UVM_FULL)
        `uvm_info(get_type_name(), $sformatf("Transaction received :\n%s", req.sprint()), UVM_FULL)
        fork
          drive_channel(0, req);
  `ifdef RW_TXRX_2X2
          drive_channel(1, req);
  `endif
        join
      end else begin
        `uvm_info(get_type_name(), $sformatf("ADC has been disabled, skipping item :\n%s", req.sprint()), UVM_FULL)
        @(posedge vif.clk);
        vif.DataEn0 <= 'b0;
        vif.DataEn1 <= 'b0;
        vif.data_valid <= 'b0;
      end
      req.phy_disabled_adc = phy_disabled_adc;
      seq_item_port.item_done(req);
    end // forever
  endtask : get_and_drive

  task drive_channel(int chan, modem_stream_seq_item#(WIDTH) req);
      vif.data_valid <= req.data_valid;
      if (req.last_adc_sample == 1'b1) begin
        @(posedge vif.clk);
        if (chan == 0) begin
          vif.DataEn0 <= 1'b0;
        end else if (chan == 1) begin
          vif.DataEn1 <= 1'b0;
        end
      end else begin
        if (chan == 0) begin
          wait ((vif.On0 === 1'b1) || (phy_disabled_adc == 1'b1));
        end else if (chan == 1) begin
          wait ((vif.On1 === 1'b1) || (phy_disabled_adc == 1'b1));
        end
        @(posedge vif.clk);
        if (chan == 0) begin
          vif.DataI0  <= req.DataI0;
          vif.DataQ0  <= req.DataQ0;
          vif.DataEn0 <= ~phy_disabled_adc;
        end else if (chan == 1) begin
          vif.DataI1  <= req.DataI1;
          vif.DataQ1  <= req.DataQ1;
          vif.DataEn1 <= ~phy_disabled_adc;
        end
      end
  endtask: drive_channel


  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : modem_stream_driver

`endif
