//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef RUI_DRIVER_SV
`define RUI_DRIVER_SV


class rui_driver extends uvm_driver #(rui_seq_item);

  `uvm_component_utils(rui_driver)

  virtual rui_if vif;
  rui_config     cfg;

  bit        phy_disabled_adc; // set to 1 on falling edge of ADCOn0
  bit        ADCOn0_d;         // ADCOn delayed by 1 clk cycle
  int        RxDataPhase;
  int        RxDataStartDelay; // Counter for generating delay between AGC Lock & First data
  int        RxDataIn20Ptr;    // Pointer for 20MHz samples
  int        RxDataIn40Ptr;    // Pointer for 40MHz samples
  int        RxDataIn80Ptr;    // Pointer for 80MHz samples
  bit [1:0]  nRx;              // Number of Rx antennae
  // file handling veriables
  string     value;
  bit [31:0] snr_from_file;
  string     simvector[$];

  function new (string name = "rui_driver", uvm_component parent = null);
    super.new(name, parent);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual rui_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase


  task run_phase(uvm_phase phase);
    super.run_phase(phase);
    fork
      get_and_drive();
      detect_adc_on_edges();
      if (!cfg.sig_error_injection) begin
        drive_rxparams();
      end
      /* add other tasks here */
    join_none;
  endtask : run_phase

  // Matlab generates more samples than RTL needs (PHY RTL finishes reception before all samples have been sent).
  // This task detects when this happens and prevents the driver from waiting for ADCOn0 indefinitely
  task detect_adc_on_edges();
    forever begin
      ADCOn0_d = vif.ADC0On;
      @(posedge vif.clk);
      if ((ADCOn0_d === 1'b1) && (vif.ADC0On === 1'b0)) begin
        // PHY RTL disabled ADC, do not send any data
        `uvm_info(get_type_name(), "ADC has been disabled.", UVM_LOW)
        phy_disabled_adc = 1'b1;
      end else if ((ADCOn0_d === 1'b0) && (vif.ADC0On === 1'b1)) begin
        // ADC has been (re-)enabled, clear phy_disabled_adc
        `uvm_info(get_type_name(), "ADC has been enabled.", UVM_LOW)
        phy_disabled_adc = 1'b0;
      end
    end
  endtask : detect_adc_on_edges

  task init_param_update();
    vif.frm_param1_update  <= 'h0;
    vif.frm_param2_update  <= 'h0;
    vif.frm_param3_update  <= 'h0;
    vif.frm_param4_update  <= 'h0;
    vif.frm_param5_update  <= 'h0;
    vif.frm_param6_update  <= 'h0;
    vif.frm_param_cbw      <= 'h0;
    vif.frm_param_gi_type  <= 'h0;
    vif.frm_param_he_ltf_type <= 'h0;
    vif.frm_param_ness     <= 'h0;
    vif.frm_param_he_sigb  <= 'h0;
    vif.frm_param_nsts     <= 'h0;
    vif.frm_param_nsymb    <= 'h0;
    vif.frm_param_num_he_ltf <= 'h0;
    vif.frm_param_vht_ndp  <= 'h0;
    vif.frm_param_tpe      <= 'h0;
    vif.frm_format_mod    <= 'h0;
    vif.frm_param_nsd     <= 'h0;
    vif.frm_param_nbpsc0  <= 'h0;
    vif.frm_param_nbpsc1  <= 'h0;
    vif.frm_param_nss     <= 'h0;
    vif.frm_param_stbc    <= 'h0;
    vif.frm_param_fec     <= 'h0;
    vif.frm_param_smoothing <= 'h0;
    vif.frm_param_htndp     <= 'h0;
    vif.frm_param_lsig6m    <= 'h0;
    vif.frm_param5_rlsig    <= 'h0;
    vif.frm_param4_hemu     <= 'h0;
    vif.frm_param_nbpsc_hesigb <= 'h0;
    vif.frm_param_rutype <= 'h0;
    vif.frm_param_ruindex <= 'h0;
    vif.frm_param_doppler <= 0;
    vif.frm_param_midamble <= 0;
    vif.frm_param_nma <= 0;
    vif.frm_param_dcm <= 0;
    vif.frm_param_hesigb_mcs <= 0;
    vif.frm_param_hesigb_dcm <= 0;
    vif.FDCpeSlope <= 'h0;
  endtask : init_param_update

  task initialize();
    vif.agcBypass         <= 'h0;
    vif.AGCOFDMLock       <= 'h0;
    vif.agc_update        <= 'h0;
    vif.RxDataValidIn20   <= 'h0;
    vif.RxDataIn20PRe0    <= 'h0;
    vif.RxDataIn20PIm0    <= 'h0;
    vif.RxDataIn20PRe1    <= 'h0;
    vif.RxDataIn20PIm1    <= 'h0;
    vif.RxDataIn20SRe     <= 'h0;
    vif.RxDataIn20SIm     <= 'h0;
    vif.RxDataValidIn40   <= 'h0;
    vif.RxDataIn40PRe0    <= 'h0;
    vif.RxDataIn40PIm0    <= 'h0;
    vif.RxDataIn40PRe1    <= 'h0;
    vif.RxDataIn40PIm1    <= 'h0;
    vif.RxDataValidIn80   <= 'h0;
    vif.RxDataIn80PRe0    <= 'h0;
    vif.RxDataIn80PIm0    <= 'h0;
    vif.RxDataIn80PRe1    <= 'h0;
    vif.RxDataIn80PIm1    <= 'h0;
    vif.bfrSigmadB        <= 'h0;
    vif.noise_variance    <= 'h0;
    vif.RatioNoiseSignal  <= 'h0;
    vif.RatioShift        <= 'h0;

    RxDataPhase       = 0;
    RxDataStartDelay  = 0;
    RxDataIn20Ptr     = 0;
    RxDataIn40Ptr     = 0;
    RxDataIn80Ptr     = 0;
    nRx               = `RW_NX_DERIV_NRX;
  endtask : initialize

  // drive parameters for RxTD block
  task drive_rxparams();
    forever begin
      @(posedge vif.agcBypass);
      // Time domain parameters
      vif.frm_param_cbw           <=  req.rxparams.cbw_f;
      vif.frm_param_gi_type       <=  req.rxparams.gi_type_f;
      vif.frm_param_he_ltf_type   <=  req.rxparams.he_ltf_type_f;
      vif.frm_param_ness          <=  req.rxparams.ness_f;
      vif.frm_param_he_sigb       <=  req.rxparams.he_sigb_f;
      vif.frm_param_num_he_ltf    <=  req.rxparams.num_he_ltf_f;
      vif.frm_param_nsts          <=  req.rxparams.nsts_f;
      vif.frm_param_nsymb         <=  req.rxparams.nsymb_f;
      vif.frm_param_vht_ndp       <=  req.rxparams.vht_ndp_f;
      vif.frm_param_tpe           <=  req.rxparams.tpe_f;
      vif.frm_param1_update       <= 'h1;
      vif.frm_param2_update       <= 'h1;
      vif.frm_param3_update       <= 'h1;
      vif.frm_param4_update       <= 'h1;
      vif.frm_param5_update       <= 'h1;
      vif.frm_param6_update       <= 'h1;
      vif.frm_format_mod          <= req.rxparams.format_mod_f;
      vif.frm_param5_rlsig        <= req.rxparams.rlsig_f;
      vif.frm_param_doppler       <= req.rxparams.doppler_f;
      vif.frm_param_midamble      <= req.rxparams.midamble_f;
      vif.frm_param_nma           <= req.rxparams.nma_f;
      vif.frm_param_dcm           <= req.rxparams.dcm_f;
      vif.frm_param_hesigb_mcs    <= req.rxparams.mcs_sigb_f;
      vif.frm_param_hesigb_dcm    <= req.rxparams.dcm_sigb_f;
      vif.FDCpeSlope              <= req.rxparams.FDCpeSlope;

      // HE-MU
      if (req.rxparams.format_mod_f == 6)
        vif.frm_param4_hemu       <= 'h1;

      vif.frm_param_nbpsc_hesigb  <= req.rxparams.nbpsc_he_sigb_f;
      vif.frm_param_rutype        <= req.rxparams.rutype_f;
      vif.frm_param_ruindex       <= req.rxparams.ruindex_f;

      vif.RatioNoiseSignal        <= req.rxparams.RatioNoiseSignal;
      vif.RatioShift              <= req.rxparams.RatioShift;
      // Additional frequency domain parameters
      case(req.rxparams.nsd_f)
        24,48 : vif.frm_param_nsd           <= 2'd0;
        52    : vif.frm_param_nsd           <= 2'd1;
        108   : vif.frm_param_nsd           <= 2'd2;
        234   : vif.frm_param_nsd           <= 2'd3;
      endcase

      case(req.rxparams.nbpsc0_f)
        1       : vif.frm_param_nbpsc0        <= 3'd0;
        2       : vif.frm_param_nbpsc0        <= 3'd1;
        4       : vif.frm_param_nbpsc0        <= 3'd2;
        6       : vif.frm_param_nbpsc0        <= 3'd3;
        8       : vif.frm_param_nbpsc0        <= 3'd4;
        10      : vif.frm_param_nbpsc0        <= 3'd5;
        default : `uvm_error(get_type_name(),$sformatf("Illegal value of NBPSC0"))
      endcase

      case(req.rxparams.nbpsc1_f)
        0       : vif.frm_param_nbpsc1        <= 3'd0;
        1       : vif.frm_param_nbpsc1        <= 3'd0;
        2       : vif.frm_param_nbpsc1        <= 3'd1;
        4       : vif.frm_param_nbpsc1        <= 3'd2;
        6       : vif.frm_param_nbpsc1        <= 3'd3;
        8       : vif.frm_param_nbpsc1        <= 3'd4;
        10      : vif.frm_param_nbpsc0        <= 3'd5;
        default : `uvm_error(get_type_name(),$sformatf("Illegal value of NBPSC1"))
      endcase

      vif.frm_param_nss           <= req.rxparams.nss_f - 1;
      vif.frm_param_stbc          <= req.rxparams.stbc_f;
      vif.frm_param_fec           <= req.rxparams.fec_f;
      vif.frm_param_smoothing     <= req.rxparams.smoothing_f;
      vif.frm_param_htndp         <= req.rxparams.htndp_f;
      vif.frm_param_lsig6m        <= req.rxparams.lsig6m_f;
    end//forever
  endtask : drive_rxparams

  // get sequence item and drive
  task get_and_drive();
    initialize();
    init_param_update();

    forever begin
      seq_item_port.get_next_item(req);

      // drive SNR from TC file or from sequence item
      if (cfg.set_snr_from_file) begin
        simvector     = extract_pattern_from_file("SIM", cfg.fname);
        value         = get_param_value(simvector, "SNR_v");
        snr_from_file = value.atoi();
        // drive pin
        vif.agc_snr <= snr_from_file;
      end
      else begin
        vif.agc_snr <= req.snr_dB;
      end

      @(posedge vif.clk iff phy_disabled_adc == 1'b0);
      `uvm_info(get_type_name(), "Start of RUI bus cycle detected.", UVM_HIGH)
      `uvm_info(get_type_name(), $sformatf("Transaction received :\n%s", req.sprint()), UVM_HIGH)

      RxDataPhase       = 0;
      RxDataStartDelay  = 0;
      RxDataIn20Ptr     = 0;
      RxDataIn40Ptr     = 0;
      RxDataIn80Ptr     = 0;
      vif.agcBypass    <= 1;

      fork
        do begin // while
          @(posedge vif.clk); // wait for rising edge of `PHYCLK

          if (vif.ADC0On == 1'b1) begin

            vif.AGCOFDMLock     <= 1;
            vif.RxDataValidIn20 <= 0;
            vif.RxDataValidIn40 <= 0;
            vif.RxDataValidIn80 <= 0;

            // Delay between AGC Lock & First data
            if (RxDataStartDelay < `AGC_DATA_START_DELAY) begin
              RxDataStartDelay++;
            end else begin

                vif.agc_update   <= 1'b1;
                vif.bfrSigmadB   <= req.bfrSigmadB[0];
                vif.noise_variance <= req.noise_variance;

                // 20MHz Data From Matlab
                if (RxDataPhase == 1) begin
                  vif.RxDataValidIn20   <= 1;
                  vif.RxDataIn20PRe0    <= req.rxFEout20_re[RxDataIn20Ptr][0];
                  vif.RxDataIn20PIm0    <= req.rxFEout20_im[RxDataIn20Ptr][0];

                  if (nRx>=2) begin
                    vif.RxDataIn20PRe1    <= req.rxFEout20_re[RxDataIn20Ptr][1];
                    vif.RxDataIn20PIm1    <= req.rxFEout20_im[RxDataIn20Ptr][1];
                  end else begin
                    vif.RxDataIn20PRe1    <= 'h0;
                    vif.RxDataIn20PIm1    <= 'h0;
                  end

                  if ((`BW_PARAM_CONFIG == 40) || (`BW_PARAM_CONFIG == 80)) begin
                      vif.RxDataIn20SRe  <= req.rxFEout20S_re[RxDataIn20Ptr][0];
                      vif.RxDataIn20SIm  <= req.rxFEout20S_im[RxDataIn20Ptr][0];
                  end
                  RxDataIn20Ptr++;
                end

                // 40MHz Data From Matlab
                if (((`BW_PARAM_CONFIG == 40) || (`BW_PARAM_CONFIG == 80)) &&
                  ((RxDataPhase == 1) || (RxDataPhase == 4))) begin
                  vif.RxDataValidIn40 <= 1;
                  vif.RxDataIn40PRe0  <= req.rxFEout40_re[RxDataIn40Ptr][0];
                  vif.RxDataIn40PIm0  <= req.rxFEout40_im[RxDataIn40Ptr][0];

                  if (nRx>=2) begin
                    vif.RxDataIn40PRe1  <= req.rxFEout40_re[RxDataIn40Ptr][1];
                    vif.RxDataIn40PIm1  <= req.rxFEout40_im[RxDataIn40Ptr][1];
                  end else begin
                    vif.RxDataIn40PRe1  <= 'h0;
                    vif.RxDataIn40PIm1  <= 'h0;
                  end

                  RxDataIn40Ptr++;
                end

                // 80MHz Data From Matlab
                if ((`BW_PARAM_CONFIG == 80) &&
                  ((RxDataPhase == 1) || (RxDataPhase == 2) || (RxDataPhase == 4) || (RxDataPhase == 5)))
                begin
                  vif.RxDataValidIn80 <= 1;
                  vif.RxDataIn80PRe0  <= req.rxFEout80_re[RxDataIn80Ptr][0];
                  vif.RxDataIn80PIm0  <= req.rxFEout80_im[RxDataIn80Ptr][0];

                  if (nRx>=2) begin
                    vif.RxDataIn80PRe1  <= req.rxFEout80_re[RxDataIn80Ptr][1];
                    vif.RxDataIn80PIm1  <= req.rxFEout80_im[RxDataIn80Ptr][1];
                  end else begin
                    vif.RxDataIn80PRe1  <= 'h0;
                    vif.RxDataIn80PIm1  <= 'h0;
                  end

                  RxDataIn80Ptr++;
                end

                // Timing reference for 20MHz, 40MHz & 80MHz Data
                if (((RxDataPhase == 5) && (`BW_PARAM_CONFIG == 20)) ||
                    ((RxDataPhase == 5) && (`BW_PARAM_CONFIG != 20)))
                  RxDataPhase = 0;
                else begin
                  RxDataPhase++;
                end

            end
          end else if (phy_disabled_adc == 1'b1) begin
            if (RxDataIn20Ptr < req.rxFEout20_re.size()) begin
              `uvm_info(get_type_name(), $sformatf("ADC has been disabled by PHY RTL, skipping remaining samples (%0d)", (req.rxFEout20_re.size()-RxDataIn20Ptr)), UVM_HIGH)
              RxDataIn20Ptr = req.rxFEout20_re.size();
            end else begin
            `uvm_info(get_type_name(), $sformatf("ADC has been disabled by PHY RTL, all samples have been sent"), UVM_HIGH)
            end
          end
        end while ( RxDataIn20Ptr < req.rxFEout20_re.size());

        // wait for TD/FD block to finish with processing
        if (req.rxparams.format_mod_f != -1 && cfg.sig_error_injection == 0)
          @(posedge vif.clk iff vif.data_done === 1'b1);
      join

      initialize();

      seq_item_port.item_done();
    end // forever
  endtask : get_and_drive

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : rui_driver

`endif
