//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef SRAM_DRIVER_SV
`define SRAM_DRIVER_SV


class sram_driver extends uvm_driver #(sram_seq_item);

  `uvm_component_utils(sram_driver)

  virtual sram_if                    vif;
  sram_config                        cfg;
  uvm_analysis_port #(sram_seq_item) ap;

  // -----------------------------------
  // descriptors
  // -----------------------------------
  THD                     thd;
  TBD                     tbd;
  RHD                     rhd;
  RPD                     rpd;
  policy_table            pt;
  policy_table            bar_pt;
  compressed_policy_table c_pt;
  // ----------------------------------
  bit [7:0]    data_frame[][];   // dynamic array used for storing frame data to memory
  int          tbd_num;          // number of TBDs in memory
  bit [31:0]   mem_addr;         // used when storing data for THD data pointers
  // addresses for MU-MIMO THDs
  bit [31:0]   primary_user_addr;
  bit [31:0]   first_user_addr;
  bit [31:0]   second_user_addr;
  bit [31:0]   third_user_addr;
  bit [31:0]   save_addr;
  // addresses for BAR
  bit [31:0]   primary_bar_addr;
  bit [31:0]   first_bar_addr;
  bit [31:0]   second_bar_addr;
  bit [31:0]   third_bar_addr;
  bit          rx_frm_received = 0;
  int          rhd_list_cnt = 0;
  int          rpd_list_cnt = 0;
  int          thd_list_cnt = 0;
  int          tbd_list_cnt = 0;
  bit [31:0]   first_buff_addr;
  bit [31:0]   last_thd_addr;
  // index of user that is mapped in MU-MIMO
  int          user;


  function new (string name = "sram_driver", uvm_component parent = null);
    super.new(name, parent);

    ap   = new("ap", this);
    thd  = new("thd");
    tbd  = new("tbd");
    pt   = new("pt");
    c_pt = new("c_pt");
    rhd  = new("rhd");
    rpd  = new("rpd");
    bar_pt = new("bar_pt");
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if(!uvm_config_db#(virtual sram_if)::get(this, "", "vif", vif))
      `uvm_fatal(get_type_name(),"virtual if not configured");

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  //----------------------------------------------------------------------------
  // list of tasks and functions used in driver
  //----------------------------------------------------------------------------
  extern function int number_of_tbd_func(int user, int mpdu_cnt, const ref PPDU_frame frm);
  extern task create_thd(bit is_ndp, bit a_thd, bit [1:0] which_descr, int user,
                         int mpdu_cnt, bit mu_mimo, const ref sram_seq_item req);
  extern task create_and_write_tbd(int user, int mpdu_cnt, const ref sram_seq_item req);
  extern task partition_data_from_frame(AMPDU_frame frm, int tbd_num,
                                        int mpdu_cnt, ref bit [7:0] data[][]);
  extern task clear_buff_array(ref bit [7:0] data[][]);
  extern task write_thd(THD thd, bit [31:0] addr);
  extern task read_thd(bit [31:0] addr, ref THD thd);
  extern task write_tbd(TBD tbd, bit [31:0] addr, int buff_cnt, const ref bit [7:0] buff[][]);
  extern task write_policy_table(policy_table pt, bit [31:0] addr);
  extern task write_compr_policy_table(compressed_policy_table c_pt, bit [31:0] addr);
  extern task write_rhd(RHD rhd, bit [31:0] addr);
  extern task write_rpd(RPD rpd, bit [31:0] addr);
  extern task writeSysMem32(bit [31:0] address, bit [31:0] data);
  extern task writeSysMem8(bit [31:0] address, bit [7:0] data);
  extern task readSysMem32(bit [31:0] address, ref bit [31:0] data);
  extern task readSysMem8(bit [31:0] address, ref bit [7:0] data);
  extern task read_rhd(bit [31:0] addr, ref RHD rhd);
  extern task read_rpd(bit [31:0] addr, ref RPD rpd);
  //----------------------------------------------------------------------------

  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    fork
      get_and_drive();
      /* add other tasks here */
    join_none;
  endtask : run_phase

  task get_and_drive();

    vif.wr_byte  <= 1'b0;
    vif.wr_data  <= 32'h0;
    vif.wr_addr  <= 32'h0;
    vif.rd_byte  <= 1'b0;
    vif.rd_data  <= 32'h0;
    vif.rd_addr  <= 32'h0;

    forever begin
      `uvm_info(get_type_name(), "Start of a bus cycle detected.", UVM_HIGH)
      seq_item_port.get_next_item(req);

//********************************************************
      if (req.cmd == WRITE_TX_FRAME) begin
//********************************************************
        // save current free position
        cfg.thd_head_ptr = req.thd_head_addr;
        cfg.thd_curr_ptr = req.thd_head_addr;
        if (!cfg.use_cfg_pt_ptr)
          cfg.pt_curr_ptr  = req.pt_head_addr;
        else
          req.pt_head_addr = cfg.pt_curr_ptr;


        case (req.frame.kind)
//--------------------------------------------------------
          SINGLETON: begin
//--------------------------------------------------------

            tbd_num = number_of_tbd_func(0, 0, req.frame);

            `uvm_info(get_type_name(),
              $sformatf("Storing singleton frame to SRAM, num of TBDs: %0d, data ptr used %0d",tbd_num-req.use_data_ptr,req.use_data_ptr), UVM_HIGH)

            create_thd( 0,     // is_ndp
                        0,     // a_thd,
                        2'b00, // which_descr,
                        0,     // user,
                        0,     // mpdu_cnt,
                        0,     // mu_mimo,
                        req    // sram_seq_item
            );

            // write THD
            write_thd(thd, cfg.thd_curr_ptr);
            // save the current free position for next THD
            cfg.thd_curr_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);

            create_and_write_tbd(0,  //user,
                                 0,  //mpdu_cnt,
                                 req //sram_seq_item req
            );

            //-----------------------------------------------------------------
            // write policy table to memory
            //-----------------------------------------------------------------
            pt = new("pt");
            pt.pt_phy_ctrl_info_1.spatial_reuse_f   = req.frame.preamble_header.spatial_reuse[0];
            pt.pt_phy_ctrl_info_1.doppler_f         = req.frame.preamble_header.doppler;
            pt.pt_phy_ctrl_info_1.midamble_f        = req.frame.preamble_header.midamble_periodicity;
            pt.pt_phy_ctrl_info_1.num_tx_prot_f     = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.num_tx_f          = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.stbc_f            = req.frame.preamble_header.stbc;
            pt.pt_phy_ctrl_info_1.num_extn_ss_f     = req.frame.preamble_header.num_extn_ss;
            pt.pt_phy_ctrl_info_1.beamforming_f     = 0;
            pt.pt_phy_ctrl_info_1.smoothing_prot_f  = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.smoothing_f       = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.sounding_f        = req.frame.preamble_header.sounding;

            if (req.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = req.frame.preamble_header.user_header_he[0].pkt_ext_f;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header_he[0].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header_he[0].smm_index_f;
            end
            else begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = 0;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header[0].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header[0].smm_index_f;
            end
            pt.pt_phy_ctrl_info_2.bss_color_f     = req.frame.preamble_header.bss_color;
            pt.pt_phy_ctrl_info_2.uplink_flag_f   = req.frame.preamble_header.uplink_flag;
            pt.pt_phy_ctrl_info_2.beamchange_f    = req.frame.preamble_header.beam_change;
            pt.pt_phy_ctrl_info_2.beamformed_f    = req.frame.preamble_header.beamformed;
            pt.pt_phy_ctrl_info_2.antenna_set_f   = req.frame.preamble_header.antenna_set;

            pt.pt_mac_ctrl_info_1.key_sram_idx_rcv_addr_f = req.pt_ksr_index[0];
            pt.pt_mac_ctrl_info_1.key_sram_idx_f          = req.pt_ksr_index[0];
            pt.pt_mac_ctrl_info_2                         = req.pt_mac_ctrl_info_2;

            // store rate control fields
            foreach (req.pt_rate_ctrl[i]) begin
              pt.pt_rate_ctrl[i] = req.pt_rate_ctrl[i];
              pt.pt_pwr_ctrl_info[i].dcm_f                = req.frame.preamble_header.dcm;
              pt.pt_pwr_ctrl_info[i].he_ltf_type_f        = req.frame.preamble_header.he_ltf_type;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_prot_f  = req.frame.preamble_header.tx_pwr_level;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_f       = req.frame.preamble_header.tx_pwr_level;
            end

            // in case when frame stored is trigger based PHY control
            // information needs to be null
            if (cfg.trigger_based_frame) begin
              pt.pt_phy_ctrl_info_1 = 0;
              pt.pt_phy_ctrl_info_2 = 0;
              foreach (pt.pt_rate_ctrl[i]) pt.pt_rate_ctrl[i] = 0;
            end
            //-----------------------------------------------------------------

            // write policy table to memory
            write_policy_table(pt, cfg.pt_curr_ptr);
            // save the position of free memory for new policy table
            cfg.pt_curr_ptr = cfg.pt_curr_ptr + (4*`PT_LEN);

          end// SINGLETON
//--------------------------------------------------------
          AGGREGATED: begin
//--------------------------------------------------------

            `uvm_info(get_type_name(),
              $sformatf("Storing aggregated frame to SRAM, data ptr used %0d",req.use_data_ptr), UVM_HIGH)

            // first store A-THD to memory
            create_thd( 0,     // is_ndp
                        1,     // a_thd,
                        2'b00, // which_descr,
                        0,     // user,
                        0,     // mpdu_cnt,
                        0,     // mu_mimo,
                        req    // sram_seq_item
            );

            // write A-THD
            write_thd(thd, cfg.thd_curr_ptr);
            // save the current free position for next THD
            cfg.thd_curr_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);

            // go through every MPDU frame in A-MPDU and store THD and TBDs
            foreach (req.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt]) begin

              tbd_num = number_of_tbd_func(0, mpdu_cnt, req.frame);

              // every aggregated MPDU has BAR attached to the end of frame,
              // so this BAR needs to be stored in memory as singleton frame
              if (mpdu_cnt == req.frame.ampdu_frame[0].mpdu_frame.size()-1 && req.frame.tx_frame) begin
                // store THD of BAR
                create_thd( 0,        // is_ndp
                            0,        // a_thd,
                            2'b00,    // which descriptor
                            0,        // user,
                            mpdu_cnt, // mpdu_cnt,
                            0,        // mu_mimo,
                            req       // sram_seq_item
                );
              end
              else begin
                // store THD per MPDU
                create_thd( 0,  // is_ndp
                            1,  // a_thd,
                            // -------------------------
                            // which descriptor
                            // -------------------------
                            (    (mpdu_cnt == req.frame.ampdu_frame[0].mpdu_frame.size()-2 && req.frame.tx_frame)
                              || (mpdu_cnt == req.frame.ampdu_frame[0].mpdu_frame.size()-1 && !req.frame.tx_frame))
                            ? 2'b11 // last THD
                            : (mpdu_cnt == 0) ? 2'b01 // first THD
                            : 2'b10 , // intermediate THD
                            // -------------------------
                            0,        // user,
                            mpdu_cnt, // mpdu_cnt,
                            0,        // mu_mimo,
                            req       // sram_seq_item
                );
              end

              // save last stored THD address for later update of next MPDU address
              last_thd_addr = cfg.thd_curr_ptr;
              // write THD
              write_thd(thd, cfg.thd_curr_ptr);

              // save the current free position for next THD
              cfg.thd_curr_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);

              create_and_write_tbd(0,         //user,
                                   mpdu_cnt,  //mpdu_cnt,
                                   req        //sram_seq_item req
              );

              // store next MPDU address when every byte of TBD is
              // finished, so alocating is correct
              if (mpdu_cnt != req.frame.ampdu_frame[0].mpdu_frame.size()-1) begin
                writeSysMem32(last_thd_addr + `NEXT_MPDU_OFFSET, cfg.thd_curr_ptr);
                read_thd(last_thd_addr, thd);//for easier debug
              end
              else begin
                writeSysMem32(last_thd_addr + `NEXT_MPDU_OFFSET, 32'd0);
                read_thd(last_thd_addr, thd);//for easier debug
              end
            end// foreach

            //-----------------------------------------------------------------
            // write policy table to memory
            //-----------------------------------------------------------------
            pt = new("pt");
            pt.pt_phy_ctrl_info_1.spatial_reuse_f   = req.frame.preamble_header.spatial_reuse[0];
            pt.pt_phy_ctrl_info_1.doppler_f         = req.frame.preamble_header.doppler;
            pt.pt_phy_ctrl_info_1.midamble_f        = req.frame.preamble_header.midamble_periodicity;
            pt.pt_phy_ctrl_info_1.num_tx_prot_f     = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.num_tx_f          = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.stbc_f            = req.frame.preamble_header.stbc;
            pt.pt_phy_ctrl_info_1.num_extn_ss_f     = req.frame.preamble_header.num_extn_ss;
            pt.pt_phy_ctrl_info_1.beamforming_f     = 0;
            pt.pt_phy_ctrl_info_1.smoothing_prot_f  = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.smoothing_f       = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.sounding_f        = req.frame.preamble_header.sounding;

            if (req.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = req.frame.preamble_header.user_header_he[0].pkt_ext_f;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header_he[0].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header_he[0].smm_index_f;
            end
            else begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = 0;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header[0].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header[0].smm_index_f;
            end
            pt.pt_phy_ctrl_info_2.bss_color_f     = req.frame.preamble_header.bss_color;
            pt.pt_phy_ctrl_info_2.uplink_flag_f   = req.frame.preamble_header.uplink_flag;
            pt.pt_phy_ctrl_info_2.beamchange_f    = req.frame.preamble_header.beam_change;
            pt.pt_phy_ctrl_info_2.beamformed_f    = req.frame.preamble_header.beamformed;
            pt.pt_phy_ctrl_info_2.antenna_set_f   = req.frame.preamble_header.antenna_set;

            pt.pt_mac_ctrl_info_1.key_sram_idx_rcv_addr_f = req.pt_ksr_index[0];
            pt.pt_mac_ctrl_info_1.key_sram_idx_f          = req.pt_ksr_index[0];
            pt.pt_mac_ctrl_info_2                         = req.pt_mac_ctrl_info_2;

            // store rate control fields
            foreach (req.pt_rate_ctrl[i]) begin
              pt.pt_rate_ctrl[i] = req.pt_rate_ctrl[i];
              pt.pt_pwr_ctrl_info[i].dcm_f                = req.frame.preamble_header.dcm;
              pt.pt_pwr_ctrl_info[i].he_ltf_type_f        = req.frame.preamble_header.he_ltf_type;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_prot_f  = req.frame.preamble_header.tx_pwr_level;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_f       = req.frame.preamble_header.tx_pwr_level;
            end

            //-----------------------------------------------------------------
            // create policy table for BAR that is attached
            //-----------------------------------------------------------------
            bar_pt = new("bar_pt");
            bar_pt.pt_phy_ctrl_info_1.num_tx_f      = req.frame.preamble_header.num_tx;
            bar_pt.pt_phy_ctrl_info_2.antenna_set_f = req.frame.preamble_header.antenna_set;
            bar_pt.pt_mac_ctrl_info_1               = pt.pt_mac_ctrl_info_1;
            bar_pt.pt_mac_ctrl_info_2               = pt.pt_mac_ctrl_info_2;
            foreach (bar_pt.pt_rate_ctrl[i]) begin
              bar_pt.pt_rate_ctrl[i].bw_f = 0;
              bar_pt.pt_rate_ctrl[i].format_mode_f = NON_HT;
              bar_pt.pt_rate_ctrl[i].mcs_idx_f = get_legacy_rate_from_mcs(format_mod_e'(req.pt_rate_ctrl[0].format_mode_f),
                                                                          req.pt_rate_ctrl[0].mcs_idx_f);
              // power control
              bar_pt.pt_pwr_ctrl_info[i].tx_pwr_level_prot_f  = req.frame.preamble_header.tx_pwr_level;
              bar_pt.pt_pwr_ctrl_info[i].tx_pwr_level_f       = req.frame.preamble_header.tx_pwr_level;
            end
            //-----------------------------------------------------------------

            // write policy table to memory:
            write_policy_table(pt, cfg.pt_curr_ptr);
            // save the position of free memory for new policy table
            cfg.pt_curr_ptr = cfg.pt_curr_ptr + (4*`PT_LEN);

            // write policy table for BAR to memory
            write_policy_table(bar_pt, cfg.pt_curr_ptr);
            // save the position of free memory for new policy table
            cfg.pt_curr_ptr = cfg.pt_curr_ptr + (4*`PT_LEN);

          end//AGGREGATED
//--------------------------------------------------------
          MU_MIMO: begin
//--------------------------------------------------------

            // go through every user A-MPDU
            foreach (req.frame.ampdu_frame[mapped_user]) begin
              user = req.frame.user_order[mapped_user]; // use longest user as first one

              `uvm_info(get_type_name(),
                $sformatf("Storing MU-MIMO (user %0d) frame to SRAM, data ptr used %0d",user,req.use_data_ptr), UVM_HIGH)

              // first store A-THD for MU-MIMO or A-THD to memory
              create_thd( 0,     // is_ndp
                          1,     // a_thd,
                          2'b00, // which_descr,
                          user,  // user,
                          0,     // mpdu_cnt,
                          (mapped_user == 0) ? 1 : 0, // mu_mimo,
                          req    // sram_seq_item
              );

              // write THD
              write_thd(thd, cfg.thd_curr_ptr);
              // save address of primary and secondary A-THDs
              case (mapped_user)
                0: primary_user_addr = cfg.thd_curr_ptr;
                1: first_user_addr  = cfg.thd_curr_ptr;
                2: second_user_addr = cfg.thd_curr_ptr;
                3: third_user_addr  = cfg.thd_curr_ptr;
                default: begin
                  primary_user_addr = 32'h0;
                  first_user_addr  = 32'd0;
                  second_user_addr = 32'd0;
                  third_user_addr  = 32'd0;
                end
              endcase
              // save the current free position for next THD
              cfg.thd_curr_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);

              // go through every MPDU frame in A-MPDU and store THD and TBDs
              foreach (req.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt]) begin

                tbd_num = number_of_tbd_func(user, mpdu_cnt, req.frame);

                if (mpdu_cnt == req.frame.ampdu_frame[user].mpdu_frame.size()-1 && req.frame.tx_frame) begin
                  // store BAR per A-MPDU
                  create_thd( 0,        // is_ndp
                              0,        // a_thd,
                              2'b00,    // which descriptor
                              user,     // user,
                              mpdu_cnt, // mpdu_cnt,
                              0,        // mu_mimo,
                              req       // sram_seq_item
                  );

                  // save address of BAR frame for later linking
                  case (mapped_user)
                    0: primary_bar_addr = cfg.thd_curr_ptr;
                    1: first_bar_addr   = cfg.thd_curr_ptr;
                    2: second_bar_addr  = cfg.thd_curr_ptr;
                    3: third_bar_addr   = cfg.thd_curr_ptr;
                  endcase
                end//if (mpdu_cnt == req.frame.ampdu_frame[user].mpdu_frame.size()-1 && req.frame.tx_frame)
                else begin
                  // store THD per MPDU
                  create_thd( 0,  // is_ndp
                              1,  // a_thd,
                              // -------------------------
                              // which descriptor
                              // -------------------------
                              (  (mpdu_cnt == req.frame.ampdu_frame[user].mpdu_frame.size()-2 && req.frame.tx_frame)
                               ||(mpdu_cnt == req.frame.ampdu_frame[user].mpdu_frame.size()-1 && !req.frame.tx_frame))
                              ? 2'b11 // last THD
                              : (mpdu_cnt == 0) ? 2'b01 // first THD
                              : 2'b10 , // intermediate THD
                              // -------------------------
                              user,     // user,
                              mpdu_cnt, // mpdu_cnt,
                              0,        // mu_mimo,
                              req       // sram_seq_item
                  );
                end//else

                // save last stored THD address for later update of next MPDU address
                last_thd_addr = cfg.thd_curr_ptr;
                // write THD
                write_thd(thd, cfg.thd_curr_ptr);
                // save the current free position for next THD
                cfg.thd_curr_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);

                create_and_write_tbd(user,      //user,
                                     mpdu_cnt,  //mpdu_cnt,
                                     req        //sram_seq_item req
                );

                // store next MPDU address when every byte of TBD is
                // finished, so alocating is correct
                if (mpdu_cnt != req.frame.ampdu_frame[user].mpdu_frame.size()-1) begin
                  writeSysMem32(last_thd_addr + `NEXT_MPDU_OFFSET, cfg.thd_curr_ptr);
                end//if (mpdu_cnt != req.frame.ampdu_frame[user].mpdu_frame.size()-1)
                else begin
                  writeSysMem32(last_thd_addr + `NEXT_MPDU_OFFSET, 32'd0);
                end//else

              end// foreach per MPDU

            end// foreach per user

            // when all A-MPDUs are stored in memory, pointers to user THD
            // need to be set in first A-THD
            writeSysMem32(primary_user_addr + `FIRST_USER_OFFSET,  first_user_addr);
            writeSysMem32(primary_user_addr + `SECOND_USER_OFFSET, second_user_addr);
            writeSysMem32(primary_user_addr + `THIRD_USER_OFFSET,  third_user_addr);

            // link BAR together
            foreach (req.frame.ampdu_frame[mapped_user]) begin
              user = req.frame.user_order[mapped_user]; // use longest user as first one
              case (user)
                0: writeSysMem32(primary_bar_addr + `NEXT_MPDU_OFFSET, first_bar_addr);
                1: writeSysMem32(first_bar_addr   + `NEXT_MPDU_OFFSET, second_bar_addr);
                2: writeSysMem32(second_bar_addr  + `NEXT_MPDU_OFFSET, third_bar_addr);
                3: writeSysMem32(third_bar_addr   + `NEXT_MPDU_OFFSET, 32'd0); // null
              endcase
            end

            //-----------------------------------------------------------------
            // write policy table to memory
            //-----------------------------------------------------------------
            pt = new("pt");
            user = req.frame.user_order[0]; // use longest user as first one
            pt.pt_phy_ctrl_info_1.spatial_reuse_f   = req.frame.preamble_header.spatial_reuse[0];
            pt.pt_phy_ctrl_info_1.doppler_f         = req.frame.preamble_header.doppler;
            pt.pt_phy_ctrl_info_1.midamble_f        = req.frame.preamble_header.midamble_periodicity;
            pt.pt_phy_ctrl_info_1.num_tx_prot_f     = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.num_tx_f          = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.stbc_f            = req.frame.preamble_header.stbc;
            pt.pt_phy_ctrl_info_1.num_extn_ss_f     = req.frame.preamble_header.num_extn_ss;
            pt.pt_phy_ctrl_info_1.beamforming_f     = 0;
            pt.pt_phy_ctrl_info_1.smoothing_prot_f  = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.smoothing_f       = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.sounding_f        = req.frame.preamble_header.sounding;

            if (req.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = req.frame.preamble_header.user_header_he[user].pkt_ext_f;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header_he[user].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header_he[user].smm_index_f;
            end
            else begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = 0;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header[user].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header[user].smm_index_f;
            end
            pt.pt_phy_ctrl_info_2.bss_color_f     = req.frame.preamble_header.bss_color;
            pt.pt_phy_ctrl_info_2.uplink_flag_f   = req.frame.preamble_header.uplink_flag;
            pt.pt_phy_ctrl_info_2.beamchange_f    = req.frame.preamble_header.beam_change;
            pt.pt_phy_ctrl_info_2.beamformed_f    = req.frame.preamble_header.beamformed;
            pt.pt_phy_ctrl_info_2.antenna_set_f   = req.frame.preamble_header.antenna_set;

            pt.pt_mac_ctrl_info_1.key_sram_idx_rcv_addr_f = req.pt_ksr_index[0];
            pt.pt_mac_ctrl_info_1.key_sram_idx_f          = req.pt_ksr_index[0];
            pt.pt_mac_ctrl_info_2                         = req.pt_mac_ctrl_info_2;

            // store rate control fields
            foreach (req.pt_rate_ctrl[i]) begin
              pt.pt_rate_ctrl[i] = req.pt_rate_ctrl[i];
              pt.pt_rate_ctrl[i].mcs_idx_f = (req.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB})
                                             ? req.frame.preamble_header.user_header_he[user].mcs_f
                                             : req.frame.preamble_header.user_header[user].mcs_f;
              pt.pt_pwr_ctrl_info[i].dcm_f                = req.frame.preamble_header.dcm;
              pt.pt_pwr_ctrl_info[i].he_ltf_type_f        = req.frame.preamble_header.he_ltf_type;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_prot_f  = req.frame.preamble_header.tx_pwr_level;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_f       = req.frame.preamble_header.tx_pwr_level;
            end
            //-----------------------------------------------------------------
            // create policy table for BAR that is attached
            //-----------------------------------------------------------------
            bar_pt = new("bar_pt");
            bar_pt.pt_phy_ctrl_info_1.num_tx_prot_f = req.frame.preamble_header.num_tx;
            bar_pt.pt_phy_ctrl_info_1.num_tx_f      = req.frame.preamble_header.num_tx;
            bar_pt.pt_phy_ctrl_info_2.antenna_set_f = req.frame.preamble_header.antenna_set;
            bar_pt.pt_mac_ctrl_info_1               = pt.pt_mac_ctrl_info_1;
            bar_pt.pt_mac_ctrl_info_2               = pt.pt_mac_ctrl_info_2;
            foreach (bar_pt.pt_rate_ctrl[i]) begin
              bar_pt.pt_rate_ctrl[i].bw_f = 0;
              bar_pt.pt_rate_ctrl[i].format_mode_f = NON_HT;
              bar_pt.pt_rate_ctrl[i].mcs_idx_f = get_legacy_rate_from_mcs(format_mod_e'(req.pt_rate_ctrl[0].format_mode_f),
                                                                          req.pt_rate_ctrl[0].mcs_idx_f);
              // power control
              bar_pt.pt_pwr_ctrl_info[i].tx_pwr_level_prot_f  = req.frame.preamble_header.tx_pwr_level;
              bar_pt.pt_pwr_ctrl_info[i].tx_pwr_level_f       = req.frame.preamble_header.tx_pwr_level;
            end
            //-----------------------------------------------------------------

            // write policy table to memory
            write_policy_table(pt, cfg.pt_curr_ptr);
            // save the position of free memory for new policy table
            cfg.pt_curr_ptr = cfg.pt_curr_ptr + (4*`PT_LEN);

            // write policy table for BAR to memory
            write_policy_table(bar_pt, cfg.pt_curr_ptr);
            // save the position of free memory for new policy table
            cfg.pt_curr_ptr = cfg.pt_curr_ptr + (4*`PT_LEN);

            // store compressed_policy_table to memory per user
            foreach (req.frame.preamble_header.user_header[mapped_user]) begin
              user = req.frame.user_order[mapped_user]; // use longest user as first one
              // only secondary users have compressed policy table
              if (mapped_user != 0) begin

                c_pt.pt_mumimo_ctrl_info.key_sram_idx_f = req.pt_ksr_index[user];
                if (req.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
                  c_pt.pt_mumimo_ctrl_info.smm_index_f    = req.frame.preamble_header.user_header_he[user].smm_index_f;
                  c_pt.pt_mumimo_ctrl_info.fec_coding_f   = req.frame.preamble_header.user_header_he[user].fec_coding_f;
                  c_pt.pt_mumimo_ctrl_info.mcs_idx_f      = req.frame.preamble_header.user_header_he[user].mcs_f;
                end
                else begin
                  c_pt.pt_mumimo_ctrl_info.smm_index_f    = req.frame.preamble_header.user_header[user].smm_index_f;
                  c_pt.pt_mumimo_ctrl_info.fec_coding_f   = req.frame.preamble_header.user_header[user].fec_coding_f;
                  c_pt.pt_mumimo_ctrl_info.mcs_idx_f      = req.frame.preamble_header.user_header[user].mcs_f;
                end

                // write policy table to memory
                write_compr_policy_table(c_pt, cfg.pt_curr_ptr);

                // store policy table address on the proper location in userN THD
                case (mapped_user)
                  1: writeSysMem32(first_user_addr  + `PT_OFFSET, cfg.pt_curr_ptr);
                  2: writeSysMem32(second_user_addr + `PT_OFFSET, cfg.pt_curr_ptr);
                  3: writeSysMem32(third_user_addr  + `PT_OFFSET, cfg.pt_curr_ptr);
                endcase

                // save the position of free memory for new policy table
                cfg.pt_curr_ptr = cfg.pt_curr_ptr + 8;
              end// if mapped_user != 0
            end// foreach user header

          end//MU_MIMO
//--------------------------------------------------------
          NDP: begin
//--------------------------------------------------------
            tbd_num = 0;

            `uvm_info(get_type_name(),$sformatf("Storing NDP frame to SRAM"), UVM_HIGH)

            create_thd( 1,     // is_ndp
                        0,     // a_thd,
                        2'b00, // which_descr,
                        0,     // user,
                        0,     // mpdu_cnt,
                        0,     // mu_mimo,
                        req    // sram_seq_item
            );

            // write THD
            write_thd(thd, cfg.thd_curr_ptr);
            // save the current free position for next THD
            cfg.thd_curr_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);

            //-----------------------------------------------------------------
            // write policy table to memory
            //-----------------------------------------------------------------
            pt = new("pt");
            pt.pt_phy_ctrl_info_1.spatial_reuse_f   = req.frame.preamble_header.spatial_reuse[0];
            pt.pt_phy_ctrl_info_1.doppler_f         = req.frame.preamble_header.doppler;
            pt.pt_phy_ctrl_info_1.midamble_f        = req.frame.preamble_header.midamble_periodicity;
            pt.pt_phy_ctrl_info_1.num_tx_prot_f     = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.num_tx_f          = req.frame.preamble_header.num_tx;
            pt.pt_phy_ctrl_info_1.stbc_f            = req.frame.preamble_header.stbc;
            pt.pt_phy_ctrl_info_1.num_extn_ss_f     = req.frame.preamble_header.num_extn_ss;
            pt.pt_phy_ctrl_info_1.beamforming_f     = 0;
            pt.pt_phy_ctrl_info_1.smoothing_prot_f  = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.smoothing_f       = req.frame.preamble_header.smoothing;
            pt.pt_phy_ctrl_info_1.sounding_f        = req.frame.preamble_header.sounding;

            if (req.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = req.frame.preamble_header.user_header_he[0].pkt_ext_f;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header_he[0].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header_he[0].smm_index_f;
            end
            else begin
              pt.pt_phy_ctrl_info_2.packet_extension_f = 0;
              pt.pt_phy_ctrl_info_1.fec_coding_f  = req.frame.preamble_header.user_header[0].fec_coding_f;
              pt.pt_phy_ctrl_info_2.smm_index_f   = req.frame.preamble_header.user_header[0].smm_index_f;
            end
            pt.pt_phy_ctrl_info_2.bss_color_f     = req.frame.preamble_header.bss_color;
            pt.pt_phy_ctrl_info_2.uplink_flag_f   = req.frame.preamble_header.uplink_flag;
            pt.pt_phy_ctrl_info_2.beamchange_f    = req.frame.preamble_header.beam_change;
            pt.pt_phy_ctrl_info_2.beamformed_f    = req.frame.preamble_header.beamformed;
            pt.pt_phy_ctrl_info_2.antenna_set_f   = req.frame.preamble_header.antenna_set;

            pt.pt_mac_ctrl_info_1   = 0;
            pt.pt_mac_ctrl_info_2   = 0;

            // store rate control fields
            foreach (req.pt_rate_ctrl[i]) begin
              pt.pt_rate_ctrl[i] = req.pt_rate_ctrl[i];
              pt.pt_pwr_ctrl_info[i].dcm_f                = req.frame.preamble_header.dcm;
              pt.pt_pwr_ctrl_info[i].he_ltf_type_f        = req.frame.preamble_header.he_ltf_type;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_prot_f  = req.frame.preamble_header.tx_pwr_level;
              pt.pt_pwr_ctrl_info[i].tx_pwr_level_f       = req.frame.preamble_header.tx_pwr_level;
            end

            // write policy table to memory
            write_policy_table(pt, cfg.pt_curr_ptr);
            // save the position of free memory for new policy table
            cfg.pt_curr_ptr = cfg.pt_curr_ptr + (4*`PT_LEN);
          end

        endcase
      end// WRITE_TX_FRAME
//********************************************************
      else if (req.cmd == READ_RX_FRAME) begin
//********************************************************
        rx_frm_received = 0;
        first_user_addr = req.rhd_head_addr;

        while (!rx_frm_received) begin
          // read first RHD
          read_rhd(first_user_addr, rhd);
          // link RHD to list
          rhd_list_cnt = req.rhd_list.size()+1;
          req.rhd_list = new[rhd_list_cnt](req.rhd_list);
          req.rhd_list[rhd_list_cnt-1] = new;
          req.rhd_list[rhd_list_cnt-1].copy(rhd);

          // read payload buffer if it exists
          if (rhd.first_payload_buff_ptr != 32'd0) begin
            // add RPD to linked list
            req.rpd_list = new[rhd_list_cnt](req.rpd_list);
            rpd_list_cnt = 1;
            first_buff_addr = rhd.first_payload_buff_ptr;

            do begin
              read_rpd(first_buff_addr, rpd);
              req.rpd_list[rhd_list_cnt-1] = new[rpd_list_cnt](req.rpd_list[rhd_list_cnt-1]);
              req.rpd_list[rhd_list_cnt-1][rpd_list_cnt-1] = new;
              // copy content of read RPD to one in the linked list
              req.rpd_list[rhd_list_cnt-1][rpd_list_cnt-1].copy(rpd);
              rpd_list_cnt++;
              first_buff_addr = rpd.next_payload_buff_ptr;
            end while (!rpd.buff_status_info.last_buff_f && rpd.next_payload_buff_ptr != 32'h0);
          end

          // determine is it a last RHD of received frame
          // else read next RHD
          if (rhd.mpdu_statu_info.rx_vector2_valid_f)
            rx_frm_received = 1;
          // if frame was not stored in memory send item with no read data
          else if (   rhd.first_payload_buff_ptr == 32'd0
                   && rhd.mpdu_statu_info.descr_done_f == 1'b0
                   && rhd.mpdu_statu_info.frm_successful_f == 1'b0)
            rx_frm_received = 1;
          // if there was some kind of error during reception
          else if (   rhd.mpdu_statu_info.descr_done_f == 1'b1
                   && (   rhd.mpdu_statu_info.fcs_err_f
                       || rhd.mpdu_statu_info.decry_err_f
                       || rhd.mpdu_statu_info.undef_err_f
                      ))
            rx_frm_received = 1;
          // in case when frame had some error it wont be linked with previous
          // RHD (AMPDU with one of MPDUs has FCS error)
          else if (rhd.next_hader_descr_ptr == 32'h0)
            rx_frm_received = 1;
          else
            first_user_addr = rhd.next_hader_descr_ptr;
        end// while

      end// READ_RX_FRAME
//********************************************************
      else if (req.cmd == PREPARE_RD_BUFF) begin
//********************************************************
        `uvm_info(get_type_name(), $sformatf("Prepare Rx buffer for reception of frame"), UVM_HIGH)

        cfg.rhd_curr_ptr = req.rhd_head_addr;
        cfg.rpd_curr_ptr = req.rpd_head_addr;
        cfg.rhd_head_ptr = req.rhd_head_addr;
        cfg.rpd_head_ptr = req.rpd_head_addr;
        `uvm_info(get_type_name(), $sformatf("RHD head address %0h", req.rhd_head_addr), UVM_HIGH)
        `uvm_info(get_type_name(), $sformatf("RPD head address %0h", req.rpd_head_addr), UVM_HIGH)
        `uvm_info(get_type_name(), $sformatf("num_of_rhd %0d, num_of_rpd %0d",req.num_of_rhd,req.num_of_rpd), UVM_HIGH)

        rhd_list_cnt = 0;
        // create link of RHDs
        repeat (req.num_of_rhd) begin
          rhd_list_cnt++;

          // last RHD should point to null since it is last one in chain list
          rhd.next_hader_descr_ptr      = (rhd_list_cnt == req.num_of_rhd)
                                           ? 32'h0
                                           : cfg.rhd_curr_ptr + (4*`RHD_LEN);
          rhd.first_payload_buff_ptr    = 32'd0;
          rhd.sw_descr_ptr              = 32'd0;
          rhd.data_start_ptr            = 32'd0;
          rhd.data_end_ptr              = 32'd0;
          rhd.header_ctrl_info.int_en_f = 1'b1;
          rhd.ampdu_status_info         = 32'd0;
          rhd.payload_len               = 32'd0;
          rhd.tsf_lo                    = 32'd0;
          rhd.tsf_hi                    = 32'd0;
          rhd.received_vector_1a        = 32'd0;
          rhd.received_vector_1b        = 32'd0;
          rhd.received_vector_1c        = 32'd0;
          rhd.received_vector_1d        = 32'd0;
          rhd.received_vector_2a        = 32'd0;
          rhd.received_vector_2b        = 32'd0;
          rhd.mpdu_statu_info           = 32'd0;

          // store RHD in memory
          write_rhd(rhd, cfg.rhd_curr_ptr);
          // save pointer location and increment by number of bytes stored
          cfg.rhd_curr_ptr = cfg.rhd_curr_ptr + (4*`RHD_LEN);
        end

        rpd_list_cnt = 0;
        // create link of RPDs
        repeat (req.num_of_rpd) begin
          rpd_list_cnt++;

          // last RPD should point to null since it is last one in chain
          rpd.next_payload_buff_ptr = (rpd_list_cnt == req.num_of_rpd)
                                      ? 32'h0
                                      : cfg.rpd_curr_ptr + (4*`RPD_LEN) + `BUFF_MAX_SIZE;
          rpd.data_start_ptr        = cfg.rpd_curr_ptr + (4*`RPD_LEN);
          rpd.data_end_ptr          = cfg.rpd_curr_ptr + (4*`RPD_LEN) + `BUFF_MAX_SIZE-1;
          rpd.buff_status_info      = 0;

          // store RPD in memory
          write_rpd(rpd, cfg.rpd_curr_ptr);
          // save pointer to next free location
          cfg.rpd_curr_ptr = cfg.rpd_curr_ptr + (4*`RPD_LEN) + `BUFF_MAX_SIZE;
        end

      end// PREPARE_RD_BUFF
//********************************************************
      else if (req.cmd == RD_TX_STATUS) begin
//********************************************************
        first_user_addr = req.thd_head_addr;
        // if expected ACK is equal to normal ACK and compressed BA
        // A-THD will not be updated by MAC HW so status information
        // will not be checked for this descriptor
        if (req.skip_athd) begin
          read_thd(first_user_addr, thd);
          first_user_addr = thd.next_mpdu_descr_ptr;
        end

        do begin
          read_thd(first_user_addr, thd);
          thd_list_cnt = req.status_info_list.size()+1;
          req.status_info_list = new[thd_list_cnt](req.status_info_list);
          // store read status information
          req.status_info_list[thd_list_cnt-1] = thd.status_info;

          first_user_addr = thd.next_mpdu_descr_ptr;

        // read THD status information until all valid are read
        end while (   thd.status_info.which_descr_sw_f != 4'b0000   // singleton MPDU
                   && thd.status_info.which_descr_sw_f != 4'b1000   // singleton MPDU under BA
                   && thd.status_info.which_descr_sw_f != 4'b0111   // A-MPDU
                   && thd.status_info.which_descr_sw_f != 4'b1111); // A-MPDU under BA

      end // RD_TX_STATUS
//********************************************************
      else if (req.cmd == GET_NEXT_RHD_PTR) begin
//********************************************************
        rsp = sram_seq_item::type_id::create("rsp");

        save_addr = req.rhd_head_addr;
        read_rhd(save_addr, rhd);
        // if rx vector 2 is not valid fetch next RHD pointer
        while (!rhd.mpdu_statu_info.rx_vector2_valid_f) begin
          save_addr = rhd.next_hader_descr_ptr;
          if (save_addr == 32'h0)
            break; //exit while loop
          else
            read_rhd(rhd.next_hader_descr_ptr, rhd);
        end
        // read RPD to find last data pointer, which means read pointer
        if (rhd.first_payload_buff_ptr != 32'd0) begin
          save_addr = rhd.first_payload_buff_ptr;
          read_rpd(rhd.first_payload_buff_ptr, rpd);
          while (rpd.next_payload_buff_ptr != 32'd0) begin
            save_addr = rpd.next_payload_buff_ptr;
            read_rpd(rpd.next_payload_buff_ptr, rpd);
          end
          read_rpd(save_addr, rpd);
          save_addr = rpd.data_end_ptr;
        end
        else if (rhd.payload_len == 0) begin
          save_addr = save_addr + 4*`RHD_LEN - 4;
          rsp.payload_zero = 1;
          req.payload_zero = 1;
        end
        else begin
          save_addr = rhd.data_end_ptr;
        end
        rsp.get_last_read_ptr = save_addr;
        req.get_last_read_ptr = save_addr;
        rsp.set_id_info(req);
      end // GET_NEXT_RHD_PTR
//********************************************************
      else if (req.cmd == WRITE_MEM) begin
//********************************************************
        writeSysMem32(req.rd_wr_addr, req.rd_wr_data);
      end // WRITE_MEM
//********************************************************
      else if (req.cmd == READ_MEM) begin
//********************************************************
        rsp = sram_seq_item::type_id::create("rsp");
        rsp.set_id_info(req);
        readSysMem32(req.rd_wr_addr, rsp.rd_wr_data);
        req.rd_wr_data = rsp.rd_wr_data;
      end // READ_MEM

      `uvm_info(get_type_name(), $sformatf("Transaction requested:\n%s", req.sprint()), UVM_HIGH)
      ap.write(req);

      seq_item_port.item_done(rsp);
    end // forever
  endtask : get_and_drive

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

endclass : sram_driver

//==================================================================
//======================= functions and tasks ======================
//==================================================================

  //-----------------------------------------------------------------
  // create THD out of frame content inside sequence item
  // @is_ndp  - indication that frame is NDP
  // @a_thd   - is THD an A-THD
  // @which_descr - define which descriptor is this
  // @user    - user location inside MU-MIMO frames
  // @mpdu_cnt- MPDU location inside A-MPDU frame
  // @mu_mimo - is frame MU_MIMO
  // @req     - sequence item to be sent
  //-----------------------------------------------------------------
  task sram_driver::create_thd(bit is_ndp, bit a_thd, bit [1:0] which_descr, int user,
                               int mpdu_cnt, bit mu_mimo, const ref sram_seq_item req);

    // determine if this MPDU is BAR, if it is different policy table needs to be linked
    // with this BAR
    bit is_bar = (   req.frame.kind inside {AGGREGATED, MU_MIMO}
                  && mpdu_cnt == req.frame.ampdu_frame[user].mpdu_frame.size()-1
                  && req.frame.tx_frame);

    // create THD
    thd.next_atomic_frm_ptr          = 32'd0;
    thd.next_mpdu_descr_ptr          = (   (a_thd == 1 && which_descr <= 2'b10)
                                        || (a_thd == 1 && which_descr == 2'b11 && user == 0))
    // when A-THD, skip length of A-THD to point to THD
                                       ? cfg.thd_curr_ptr + (4*`THD_LEN)
                                       : 32'd0;

    if ((a_thd == 1 && which_descr == 2'b00) || is_ndp == 1) begin
      thd.first_tbd_ptr_user1_thd_ptr  = 32'd0;
    end
    else begin
      // when data size is over the buffer limit use TBD
      thd.first_tbd_ptr_user1_thd_ptr  = (tbd_num == 0 && req.use_data_ptr)
                                         ? 32'd0
                                         // else we must skip THD and
                                         // data that THD points to, if
                                         // it exists
                                         : (!req.use_data_ptr)
                                         // skip THD
                                         ? cfg.thd_curr_ptr + (4*`THD_LEN)
                                         // skip THD and data stored after
                                         : cfg.thd_curr_ptr + (4*`THD_LEN) + `BUFF_MAX_SIZE;
    end

    // if data pointer is used, put data after THD but data buffer can
    // point only on maximum number of bytes
    if (is_ndp) begin
      thd.data_start_ptr_user2_thd_ptr = 0;
      thd.data_end_ptr_user3_thd_ptr   = 0;
    end
    else if (req.use_data_ptr == 1 && tbd_num != 0 && (a_thd == 0 || (a_thd == 1 && which_descr != 2'b00))) begin
      thd.data_start_ptr_user2_thd_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);
      thd.data_end_ptr_user3_thd_ptr   =   thd.data_start_ptr_user2_thd_ptr
                                         + `BUFF_MAX_SIZE-1;
    end
    else if (req.use_data_ptr == 1 && (a_thd == 0 || (a_thd == 1 && which_descr != 2'b00))) begin
      thd.data_start_ptr_user2_thd_ptr = cfg.thd_curr_ptr + (4*`THD_LEN);
      thd.data_end_ptr_user3_thd_ptr   =   thd.data_start_ptr_user2_thd_ptr
                                         + req.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].size();
    end
    else begin
      thd.data_start_ptr_user2_thd_ptr = 32'd0;
      thd.data_end_ptr_user3_thd_ptr   = 32'd0;
    end
    thd.frm_length                   = (is_ndp == 1)
                                       ? 32'd0
                                       : (a_thd == 1 && which_descr == 2'b00) // Extra THD (A-THD)
                                       ? req.frame.ampdu_frame[user].size()
                                       : req.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].size();

    thd.frm_lifetime                 = 32'd0;
    //PHY CTRL INFO
    thd.phy_ctrl_info.partial_aid_f      = req.frame.preamble_header.partial_aid;
    thd.phy_ctrl_info.group_id_f         = req.frame.preamble_header.group_id;
    thd.phy_ctrl_info.mu_mimo_f          = mu_mimo;
    thd.phy_ctrl_info.user_position_f    = (req.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB})
                                           ? req.frame.preamble_header.user_header_he[user].user_position_f
                                           : req.frame.preamble_header.user_header[user].user_position_f;
    thd.phy_ctrl_info.ptit_f             = cfg.pti; // store PTI from configuration, to be able to change it from test case
    thd.phy_ctrl_info.continous_tx_f     = req.frame.preamble_header.continuous_tx;
    thd.phy_ctrl_info.doze_not_allowed_f = req.frame.preamble_header.doze_not_allowed;
    thd.phy_ctrl_info.dyn_bw_f           = req.frame.preamble_header.dyn_bw;
    thd.phy_ctrl_info.use_bw_signaling_f = 0;
    thd.policy_table_addr                = (is_bar) ? cfg.pt_curr_ptr + (4*`PT_LEN) : cfg.pt_curr_ptr;
    thd.optional_frm_len_20mhz           = 32'd0;
    thd.optional_frm_len_40mhz           = 32'd0;
    thd.optional_frm_len_80mhz           = 32'd0;
    //MAC CTRL INFO 1
    thd.mac_ctrl_info_1                  = req.mac_ctrl_info_1;
    //MAC CTRL INFO 2
    thd.mac_ctrl_info_2                  = req.mac_ctrl_info_2;
    thd.mac_ctrl_info_2.num_blank_delimit_f = (is_ndp == 1)
                                              ? 0
                                              : (req.frame.ampdu_frame[user].blank_delimit.exists(mpdu_cnt))
                                              ? req.frame.ampdu_frame[user].blank_delimit[mpdu_cnt].blank_delimiter_num
                                              : 0;
    thd.mac_ctrl_info_2.ampdu_f          = a_thd;
    thd.mac_ctrl_info_2.which_descr_f    = which_descr;
    thd.mac_ctrl_info_2.subtype_f        = (is_ndp == 1)
                                           ? 0
                                           : req.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.frame_ctrl.subtype_f;
    thd.mac_ctrl_info_2.type_f           = (is_ndp == 1)
                                           ? 0
                                           : req.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].MAC_header.frame_ctrl.type_f;
    thd.mac_ctrl_info_2.ts_valid_f       = 1'b1;
    thd.status_info                      = 32'd0;
    thd.medium_time_used                 = 32'd0;

  endtask : create_thd

  //-------------------------------------------------------------
  // create TBDs chain from MPDU content
  // @user    - user location inside MU-MIMO frames
  // @mpdu_cnt- MPDU location inside A-MPDU frame
  // @req     - sequnece item to be sent
  //-------------------------------------------------------------
  task sram_driver::create_and_write_tbd(int user, int mpdu_cnt, const ref sram_seq_item req);
    // write TBD,
    // 1 - when only TBDs are used for storing data
    // 2 - when one TBD is used
    // 3 - when part is stored via data pointers and rest with TBDs
    tbd.next_payload_buff_ptr         = (tbd_num == 0) // if there is only one TBD
                                        ? 32'd0        // next payload is null
                                        //else, skip TBD size + buffer max size
                                        : cfg.thd_curr_ptr + `BUFF_MAX_SIZE + (4*`TBD_LEN);

    tbd.data_start_ptr                = (req.use_data_ptr)
                                        ? cfg.thd_curr_ptr   + (4*`TBD_LEN) + `BUFF_MAX_SIZE
                                        : cfg.thd_curr_ptr   + (4*`TBD_LEN);

    tbd.data_end_ptr                  = (tbd_num == 0)
                                        ? tbd.data_start_ptr + req.frame.ampdu_frame[user].size()-1
                                        : tbd.data_start_ptr + `BUFF_MAX_SIZE-1;
    tbd.buff_ctrl_info.buff_done_sw_f = 1;
    tbd.buff_ctrl_info.buff_int_en_f  = 1;

    // when frame size is smaller than maximum size, create at least
    // one buffer
    if (tbd_num == 0) tbd_num++;
    partition_data_from_frame(req.frame.ampdu_frame[user], tbd_num, mpdu_cnt, data_frame);

    foreach (data_frame[buff_cnt]) begin
      // store THD data pointers if enabled
      if (buff_cnt == 0 && req.use_data_ptr == 1) begin

        mem_addr = thd.data_start_ptr_user2_thd_ptr;
        foreach (data_frame[buff_cnt][i]) begin
          writeSysMem8(mem_addr, data_frame[buff_cnt][i]);
          mem_addr++;
        end
        // save the current free position, skip buffer data
        cfg.thd_curr_ptr = cfg.thd_curr_ptr + `BUFF_MAX_SIZE;

      end
      else begin
        // when this is the last TBD there is no next payload pointer
        tbd.next_payload_buff_ptr = (buff_cnt == data_frame.size()-1)
                                    ? 32'd0
                                    : cfg.thd_curr_ptr + `BUFF_MAX_SIZE + (4*`TBD_LEN);
        tbd.data_start_ptr        = cfg.thd_curr_ptr   + (4*`TBD_LEN);
        tbd.data_end_ptr          = (buff_cnt == data_frame.size()-1)
                                    // last TBD end pointer should point to
                                    // frame size minus amount of data already
                                    // writen in memory
                                    ? tbd.data_start_ptr + (req.frame.ampdu_frame[user].mpdu_frame[mpdu_cnt].size()-((data_frame.size()-1)*`BUFF_MAX_SIZE)-1)
                                    : tbd.data_start_ptr + `BUFF_MAX_SIZE-1;

        write_tbd(tbd, cfg.thd_curr_ptr, buff_cnt, data_frame);

        // move current pointer to skip TBD and max buffer bytes
        cfg.thd_curr_ptr          = cfg.thd_curr_ptr + `BUFF_MAX_SIZE + (4*`TBD_LEN);

      end
    end//foreach
  endtask : create_and_write_tbd

  //-----------------------------------------------------------------
  // Determine how many TBD are needed in order to store data,
  // buffer can have maximum byte number so it needs to be divided
  // into several TBDs. If no TBDs are needed then data pointers are
  // used instead, but this can be controlled by random decision.
  //-----------------------------------------------------------------
  function int sram_driver::number_of_tbd_func(int user, int mpdu_cnt, const ref PPDU_frame frm);
    int ret;

    ret = (frm.ampdu_frame[user].mpdu_frame[mpdu_cnt].size() <= `BUFF_MAX_SIZE)
          ? 0 // when frame size is smaller then buffer size
          : $ceil(real'(frm.ampdu_frame[user].mpdu_frame[mpdu_cnt].size()) / `BUFF_MAX_SIZE);

    return ret;
  endfunction : number_of_tbd_func

  //------------------------------------------------------
  // Partition of A-MPDU or MPDU frame to buffer sized arrays
  // which can be stored with TBDs.
  // Returns 2-dimension array X * Y;
  //  X - number of buffers to be stored (number of TBDs)
  //  Y - buffer data with maximum size possible (BUFF_MAX_SIZE)
  //------------------------------------------------------
  task sram_driver::partition_data_from_frame(AMPDU_frame frm, int tbd_num, int mpdu_cnt, ref bit [7:0] data[][]);
    int        data_cnt;
    bit [31:0] del;

    clear_buff_array(data);
    data_cnt = 0;
    // create adequate number of buffers
    data = new[tbd_num];
    foreach (data[i]) data[i] = new[`BUFF_MAX_SIZE];

    `uvm_info(get_type_name(), $sformatf("Partition frame to buffer sized arrays (%0d bytes)",`BUFF_MAX_SIZE), UVM_HIGH)
    `uvm_info(get_type_name(), $sformatf("Number of buffers %0d", tbd_num), UVM_HIGH)

    // store data from frame to buffers
    foreach (data[i]) begin
      foreach (data[i][j]) begin
        data[i][j] = frm.mpdu_frame[mpdu_cnt].frame[data_cnt];
        data_cnt++;
        // if all data is stored to buffer and buffer still has space just
        // leave zeros
        if (frm.mpdu_frame[mpdu_cnt].size() == data_cnt) break;
      end//foreach
    end//foreach

  endtask : partition_data_from_frame

  //------------------------------------------------------
  // clear all dynamic buffer data arrays
  //------------------------------------------------------
  task sram_driver::clear_buff_array(ref bit [7:0] data[][]);
    foreach (data[i]) data[i].delete();
    data.delete();
  endtask : clear_buff_array

  //------------------------------------------------------
  // write THD to memory @ given address
  //------------------------------------------------------
  task sram_driver::write_thd(THD thd, bit [31:0] addr);
    bit [31:0] thd_addr;

    if (!(addr inside {[`MEM_TX_SEGMENT:`MEM_RX_SEGMENT]}))
      `uvm_error(get_type_name(),$sformatf("THD address (%0h) out of range! %0h-%0h",
      addr, `MEM_TX_SEGMENT, `MEM_RX_SEGMENT))

    thd_addr = addr;
    writeSysMem32(thd_addr,   `THD_UNIQUE_PATTERN           ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.next_atomic_frm_ptr         ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.next_mpdu_descr_ptr         ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.first_tbd_ptr_user1_thd_ptr ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.data_start_ptr_user2_thd_ptr); thd_addr += 4;
    writeSysMem32(thd_addr, thd.data_end_ptr_user3_thd_ptr  ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.frm_length                  ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.frm_lifetime                ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.phy_ctrl_info               ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.policy_table_addr           ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.optional_frm_len_20mhz      ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.optional_frm_len_40mhz      ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.optional_frm_len_80mhz      ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.mac_ctrl_info_1             ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.mac_ctrl_info_2             ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.status_info                 ); thd_addr += 4;
    writeSysMem32(thd_addr, thd.medium_time_used            );
    `uvm_info(get_type_name(), $sformatf("THD written to SRAM @%0h, %s", addr,thd.sprint()), UVM_HIGH)
  endtask : write_thd

  //------------------------------------------------------
  // read THD from memory @ given address
  //------------------------------------------------------
  task sram_driver::read_thd(bit [31:0] addr, ref THD thd);
    bit [31:0] thd_addr;
    bit [31:0] pattern;

    thd_addr = addr;
    readSysMem32(thd_addr, pattern                          ); thd_addr += 4;
    // check if pattern is valid
    if (pattern != `THD_UNIQUE_PATTERN)
      `uvm_error(get_type_name(), $sformatf("THD pattern mismatch! Read %0h", pattern))

    readSysMem32(thd_addr, thd.next_atomic_frm_ptr         ); thd_addr += 4;
    readSysMem32(thd_addr, thd.next_mpdu_descr_ptr         ); thd_addr += 4;
    readSysMem32(thd_addr, thd.first_tbd_ptr_user1_thd_ptr ); thd_addr += 4;
    readSysMem32(thd_addr, thd.data_start_ptr_user2_thd_ptr); thd_addr += 4;
    readSysMem32(thd_addr, thd.data_end_ptr_user3_thd_ptr  ); thd_addr += 4;
    readSysMem32(thd_addr, pattern                         ); thd_addr += 4;
    thd.frm_length = pattern[19:0];
    readSysMem32(thd_addr, thd.frm_lifetime                ); thd_addr += 4;
    readSysMem32(thd_addr, thd.phy_ctrl_info               ); thd_addr += 4;
    readSysMem32(thd_addr, thd.policy_table_addr           ); thd_addr += 4;
    readSysMem32(thd_addr, pattern                         ); thd_addr += 4;
    thd.optional_frm_len_20mhz = pattern[19:0];
    readSysMem32(thd_addr, pattern                         ); thd_addr += 4;
    thd.optional_frm_len_40mhz = pattern[19:0];
    readSysMem32(thd_addr, pattern                         ); thd_addr += 4;
    thd.optional_frm_len_80mhz = pattern[19:0];
    readSysMem32(thd_addr, thd.mac_ctrl_info_1             ); thd_addr += 4;
    readSysMem32(thd_addr, thd.mac_ctrl_info_2             ); thd_addr += 4;
    readSysMem32(thd_addr, thd.status_info                 ); thd_addr += 4;
    readSysMem32(thd_addr, thd.medium_time_used            );
    `uvm_info(get_type_name(), $sformatf("THD read from SRAM @%0h, %s",addr, thd.sprint()), UVM_HIGH)
  endtask : read_thd

  //------------------------------------------------------
  // write TBD to memory @ given address
  //------------------------------------------------------
  task sram_driver::write_tbd(TBD tbd, bit [31:0] addr, int buff_cnt, const ref bit [7:0] buff[][]);
    int        data_ptr;
    bit [31:0] tbd_addr;

    if (!(addr inside {[`MEM_TX_SEGMENT:`MEM_RX_SEGMENT]}))
      `uvm_error(get_type_name(),$sformatf("TBD address (%0h) out of range! %0h-%0h",
      addr, `MEM_TX_SEGMENT, `MEM_RX_SEGMENT))

    tbd_addr = addr;
    writeSysMem32(tbd_addr,   `TBD_UNIQUE_PATTERN    ); tbd_addr += 4;
    writeSysMem32(tbd_addr, tbd.next_payload_buff_ptr); tbd_addr += 4;
    writeSysMem32(tbd_addr, tbd.data_start_ptr       ); tbd_addr += 4;
    writeSysMem32(tbd_addr, tbd.data_end_ptr         ); tbd_addr += 4;
    writeSysMem32(tbd_addr, tbd.buff_ctrl_info       );

    data_ptr = tbd.data_start_ptr;
    // put data after TBD
    for (int i=0; i < buff[buff_cnt].size(); i++) begin
      writeSysMem8(data_ptr, buff[buff_cnt][i]);
      data_ptr = data_ptr + 1;
    end
    `uvm_info(get_type_name(), $sformatf("TBD written to SRAM @%0h, %s",addr, tbd.sprint()), UVM_HIGH)
  endtask : write_tbd

  //------------------------------------------------------
  // write policy table to memory @ given address
  //------------------------------------------------------
  task sram_driver::write_policy_table(policy_table pt, bit [31:0] addr);
    bit [31:0] pt_addr;

    if (!(addr inside {[`MEM_PT_SEGMENT:`MAX_ADDR_SIZE]}) && cfg.use_cfg_pt_ptr == 0)
      `uvm_error(get_type_name(),$sformatf("PT address (%0h) out of range! %0h-%0h",
      addr, `MEM_PT_SEGMENT, `MAX_ADDR_SIZE))

    pt_addr = addr;
    writeSysMem32(pt_addr, `PT_UNIQUE_PATTERN   ); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_phy_ctrl_info_1); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_phy_ctrl_info_2); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_mac_ctrl_info_1); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_mac_ctrl_info_2); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_rate_ctrl[0]    ); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_rate_ctrl[1]    ); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_rate_ctrl[2]    ); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_rate_ctrl[3]    ); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_pwr_ctrl_info[0]); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_pwr_ctrl_info[1]); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_pwr_ctrl_info[2]); pt_addr += 4;
    writeSysMem32(pt_addr, pt.pt_pwr_ctrl_info[3]);
    `uvm_info(get_type_name(), $sformatf("Policy table written to SRAM @%0h, %s",addr, pt.sprint()), UVM_HIGH)
  endtask : write_policy_table

  //------------------------------------------------------
  // write compressed policy table to memory @ given address
  //------------------------------------------------------
  task sram_driver::write_compr_policy_table(compressed_policy_table c_pt, bit [31:0] addr);
    bit [31:0] pt_addr;

    if (!(addr inside {[`MEM_PT_SEGMENT:`MAX_ADDR_SIZE]}))
      `uvm_error(get_type_name(),$sformatf("PT address (%0h) out of range! %0h-%0h",
      addr, `MEM_PT_SEGMENT, `MAX_ADDR_SIZE))

    pt_addr = addr;
    writeSysMem32(pt_addr, `PT_UNIQUE_PATTERN      ); pt_addr += 4;
    writeSysMem32(pt_addr, c_pt.pt_mumimo_ctrl_info);
    `uvm_info(get_type_name(), $sformatf("Compressed policy table written to SRAM @%0h, %s",addr, c_pt.sprint()), UVM_HIGH)
  endtask : write_compr_policy_table

  //------------------------------------------------------
  // write RHD to memory @ given address
  //------------------------------------------------------
  task sram_driver::write_rhd(RHD rhd, bit [31:0] addr);
    bit [31:0] rhd_addr;
    bit [7:0]  reserved = 8'h00;

    if (addr > `MEM_PT_SEGMENT)
      `uvm_error(get_type_name(),$sformatf("RHD address (%0h) out of range! last addr %0h",
      addr, `MEM_PT_SEGMENT))

    rhd_addr = addr;
    writeSysMem32(rhd_addr, `RHD_UNIQUE_PATTERN        ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.next_hader_descr_ptr   ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.first_payload_buff_ptr ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.sw_descr_ptr           ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.data_start_ptr         ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.data_end_ptr           ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.header_ctrl_info       ); rhd_addr += 4;
    writeSysMem32(rhd_addr, {rhd.ampdu_status_info,
                             reserved,
                             rhd.payload_len}          ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.tsf_lo                 ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.tsf_hi                 ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.received_vector_1a     ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.received_vector_1b     ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.received_vector_1c     ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.received_vector_1d     ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.received_vector_2a     ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.received_vector_2b     ); rhd_addr += 4;
    writeSysMem32(rhd_addr, rhd.mpdu_statu_info        );
    `uvm_info(get_type_name(), $sformatf("RHD written to SRAM @%0h, %s", addr,rhd.sprint()), UVM_HIGH)
  endtask : write_rhd

  //------------------------------------------------------
  // read RHD from memory @ given address
  //------------------------------------------------------
  task sram_driver::read_rhd(bit [31:0] addr, ref RHD rhd);
    bit [31:0] rhd_addr;
    bit [7:0]  reserved = 8'h00;
    bit [31:0] pattern;
    bit [7:0]  buff_data;

    rhd_addr = addr;
    readSysMem32(rhd_addr, pattern                    ); rhd_addr += 4;
    // check if pattern is valid
    if (pattern != `RHD_UNIQUE_PATTERN)
      `uvm_error(get_type_name(), $sformatf("RHD pattern mismatch! Read %0h @%0h", pattern,rhd_addr-4))

    readSysMem32(rhd_addr, rhd.next_hader_descr_ptr   ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.first_payload_buff_ptr ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.sw_descr_ptr           ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.data_start_ptr         ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.data_end_ptr           ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.header_ctrl_info       ); rhd_addr += 4;
    readSysMem32(rhd_addr, pattern                    ); rhd_addr += 4;
    // map read data to fields
    {rhd.ampdu_status_info,reserved,rhd.payload_len} = pattern;
    readSysMem32(rhd_addr, rhd.tsf_lo                 ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.tsf_hi                 ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.received_vector_1a     ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.received_vector_1b     ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.received_vector_1c     ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.received_vector_1d     ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.received_vector_2a     ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.received_vector_2b     ); rhd_addr += 4;
    readSysMem32(rhd_addr, rhd.mpdu_statu_info        );

    // read MAC header bytes
    rhd.mac_header.delete();
    // allocate memory needed for buffer size
    rhd.mac_header = new[rhd.data_end_ptr-rhd.data_start_ptr+1];
    rhd_addr = rhd.data_start_ptr;
    // read data for memory and store to RPD buffer
    foreach (rhd.mac_header[i]) begin
      readSysMem8(rhd_addr, buff_data);
      rhd.mac_header[i] = buff_data;
      rhd_addr++;
    end
    `uvm_info(get_type_name(), $sformatf("RHD read from SRAM @%0h, %s",addr, rhd.sprint()), UVM_HIGH)
  endtask : read_rhd

  //------------------------------------------------------
  // write RPB to memory @ given address
  //------------------------------------------------------
  task sram_driver::write_rpd(RPD rpd, bit [31:0] addr);
    bit [31:0] rpd_addr;

    if (addr > `MEM_PT_SEGMENT)
      `uvm_error(get_type_name(),$sformatf("RPD address (%0h) out of range! last addr %0h",
      addr, `MEM_PT_SEGMENT))

    rpd_addr = addr;
    writeSysMem32(rpd_addr,   `RPD_UNIQUE_PATTERN    ); rpd_addr += 4;
    writeSysMem32(rpd_addr, rpd.next_payload_buff_ptr); rpd_addr += 4;
    writeSysMem32(rpd_addr, rpd.data_start_ptr       ); rpd_addr += 4;
    writeSysMem32(rpd_addr, rpd.data_end_ptr         ); rpd_addr += 4;
    writeSysMem32(rpd_addr, rpd.buff_status_info     );

    `uvm_info(get_type_name(), $sformatf("RPD written to SRAM @%0h, %s",addr, rpd.sprint()), UVM_HIGH)
  endtask : write_rpd

  //------------------------------------------------------
  // read RPB from memory @ given address
  //------------------------------------------------------
  task sram_driver::read_rpd(bit [31:0] addr, ref RPD rpd);
    bit [31:0] rpd_addr;
    bit [31:0] pattern;
    bit [7:0]  buff_data;

    rpd_addr = addr;
    readSysMem32(rpd_addr, pattern                  ); rpd_addr += 4;
    // check if pattern is valid
    if (pattern != `RPD_UNIQUE_PATTERN)
      `uvm_error(get_type_name(), $sformatf("RPD pattern mismatch! Read %0h", pattern))
    readSysMem32(rpd_addr, rpd.next_payload_buff_ptr); rpd_addr += 4;
    readSysMem32(rpd_addr, rpd.data_start_ptr       ); rpd_addr += 4;
    readSysMem32(rpd_addr, rpd.data_end_ptr         ); rpd_addr += 4;
    readSysMem32(rpd_addr, rpd.buff_status_info     );

    rpd.payload.delete();
    // allocate memory needed for buffer size
    rpd.payload = new[rpd.data_end_ptr-rpd.data_start_ptr+1];
    rpd_addr = rpd.data_start_ptr;
    // read data for memory and store to RPD buffer
    foreach (rpd.payload[i]) begin
      readSysMem8(rpd_addr, buff_data);
      rpd.payload[i] = buff_data;
      rpd_addr++;
    end

    `uvm_info(get_type_name(), $sformatf("RPD read from SRAM @%0h, %s",addr, rpd.sprint()), UVM_HIGH)
  endtask : read_rpd

  //------------------------------------------------------
  // write to memory via backdoor access
  //------------------------------------------------------
  task sram_driver::writeSysMem32(bit [31:0] address, bit [31:0] data);
    begin
      if(address[2] == 1'b0) begin
        `uvm_info(get_type_name(), $sformatf("writeSysMem32 : Writing 0x00000000_%8h @ 0x%8h",data,address[31:3]), UVM_DEBUG)
        vif.wr_byte   <= 1'b0;
        vif.wr_data   <= data;
        vif.wr_addr   <= address;
      end
      else begin
        `uvm_info(get_type_name(), $sformatf("writeSysMem32 : Writing 0x%8h_00000000 @ 0x%8h",data,address[31:3]), UVM_DEBUG)
        vif.wr_byte   <= 1'b0;
        vif.wr_data   <= data;
        vif.wr_addr   <= address;
      end
      #(`SRAM_DELAY_TICK);
    end
  endtask : writeSysMem32

  task sram_driver::writeSysMem8(bit [31:0] address, bit [7:0] data);
    begin
      vif.wr_byte   <= 1'b1;
      vif.wr_data   <= data;
      vif.wr_addr   <= address;
      #(`SRAM_DELAY_TICK);
    end
  endtask : writeSysMem8

  //------------------------------------------------------
  // read from memory via backdoor access
  //------------------------------------------------------
  task sram_driver::readSysMem32(bit [31:0] address, ref bit [31:0] data);
    begin
      if(address[1:0] != 2'b00)
        `uvm_warning(get_type_name(), $sformatf("readSysMem32 : address not aligned : %8h",address))

      vif.rd_byte <= 1'b0;
      vif.rd_addr <= address;
      #(`SRAM_DELAY_TICK);
      data        = vif.rd_data;
    end
  endtask: readSysMem32

  task sram_driver::readSysMem8(bit [31:0] address, ref bit [7:0] data);
    begin
      vif.rd_byte <= 1'b1;
      vif.rd_addr <= address;
      #(`SRAM_DELAY_TICK);
      data        = vif.rd_data[7:0];
    end
  endtask : readSysMem8

`endif
