//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthou $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 32750 $
// $Date: 2017-12-01 17:34:43 +0100 (Fri, 01 Dec 2017) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef SRAM_SEQ_ITEM_SV
`define SRAM_SEQ_ITEM_SV


class sram_seq_item extends uvm_sequence_item;

  rand mem_access_e          cmd;
       PPDU_frame            frame;
  rand bit [31:0]            thd_head_addr;
  rand bit [31:0]            rhd_head_addr;
  rand bit [31:0]            rpd_head_addr;
  rand bit [31:0]            pt_head_addr;
  rand bit                   use_data_ptr;     // use data pointer in THD
  rand int unsigned          num_of_rhd;       // number of RHDs to be linked in memory
  rand int unsigned          num_of_rpd;       // number of RPDs to be linked in memory
  rand bit                   smaller_rx_segment; //in Tx tests use smaller Rx segnemt for storing frames
  // descriptor fields
       mac_ctrl_info_1_s     mac_ctrl_info_1;
       mac_ctrl_info_2_s     mac_ctrl_info_2;
       pt_mac_ctrl_info_2_s  pt_mac_ctrl_info_2;
  rand pt_rate_ctrl_s        pt_rate_ctrl[];
  // read frame from memory arrays
  RHD                        rhd_list[];
  RPD                        rpd_list[][];
  bit [31:0]                 get_last_read_ptr;
  bit                        payload_zero;
  // read status information for THD and TBD
  status_info_s              status_info_list[];
  // fields for read and write from memory
  rand bit [31:0]            rd_wr_addr;
  rand bit [31:0]            rd_wr_data;
  rand bit                   skip_athd;
  // KSR index that will be used in policy table and
  // compressed policy table
       bit [9:0]             pt_ksr_index[4];

  `uvm_object_utils_begin(sram_seq_item)
    `uvm_field_enum(mem_access_e, cmd, UVM_DEFAULT)
    `uvm_field_int(thd_head_addr, UVM_DEFAULT)
    `uvm_field_int(rhd_head_addr, UVM_DEFAULT)
    `uvm_field_int(rpd_head_addr, UVM_DEFAULT)
    `uvm_field_int(pt_head_addr, UVM_DEFAULT)
    `uvm_field_int(use_data_ptr, UVM_DEFAULT)
    `uvm_field_int(num_of_rhd, UVM_DEFAULT)
    `uvm_field_int(num_of_rpd, UVM_DEFAULT)
    `uvm_field_int(get_last_read_ptr, UVM_DEFAULT)
    `uvm_field_int(payload_zero, UVM_DEFAULT)
  `uvm_object_utils_end

  function new (string name = "sram_seq_item");
    super.new(name);
  endfunction : new

  //----------------------------------------------
  // constraints
  //----------------------------------------------

  constraint c_quadlet_aligned {
    solve rhd_head_addr before rpd_head_addr;
    solve num_of_rhd before rpd_head_addr;
    solve num_of_rpd before rpd_head_addr;

    if (cmd == WRITE_TX_FRAME) {
      thd_head_addr[1:0] == 0;
      thd_head_addr inside {[`MEM_TX_SEGMENT:`MEM_RX_SEGMENT]};

      rhd_head_addr == 0;
      rpd_head_addr == 0;
      num_of_rhd == 0;
      num_of_rpd == 0;

      pt_head_addr == `MEM_PT_SEGMENT;

    } else if (cmd == PREPARE_RD_BUFF) {
      thd_head_addr == 0;
      pt_head_addr == 0;

      if (smaller_rx_segment) {
        rhd_head_addr[1:0] == 0;
        rhd_head_addr == `MEM_RX_SEGMENT;
        rpd_head_addr[1:0] == 0;
        rpd_head_addr == rhd_head_addr+(num_of_rhd*(4*`RHD_LEN));
      } else {
        rhd_head_addr == `MEM_RX_SEGMENT_START;
        rpd_head_addr[1:0] == 0;
        rpd_head_addr == rhd_head_addr+(num_of_rhd*(4*`RHD_LEN));
      }
    }
    rd_wr_addr[1:0] == 0;
  }

  constraint c_pt_rate_control_size {
    if (cmd == WRITE_TX_FRAME) {
      pt_rate_ctrl.size() inside {[1:4]};
    } else {
      pt_rate_ctrl.size() == 0;
    }
  }

  constraint c_pt_rate_control {
    if (cmd == WRITE_TX_FRAME) {
        foreach (pt_rate_ctrl[i]) {
          pt_rate_ctrl[i].num_retry_rc_f  inside {[1:7]};
  `ifdef RW_NX_CHBW20
          // prevent sending protection frames in VHT
          if (frame.preamble_header.format_mod >= VHT) {
            pt_rate_ctrl[i].format_mode_prot_f < VHT;
            pt_rate_ctrl[i].format_mode_prot_f != NON_HT_DUP_OFDM;
          } else {
            if (frame.preamble_header.format_mod != HE_TB)
              pt_rate_ctrl[i].format_mode_prot_f <= frame.preamble_header.format_mod;
            pt_rate_ctrl[i].format_mode_prot_f != NON_HT_DUP_OFDM;
          }
  `else
          // prevent sending protection frames in VHT and match channel
          // bandwidth
          if (frame.preamble_header.ch_bw == 2'b00 && frame.preamble_header.format_mod >= VHT) {//only 20MHz
            pt_rate_ctrl[i].format_mode_prot_f < VHT;
            pt_rate_ctrl[i].format_mode_prot_f != NON_HT_DUP_OFDM;
          } else if (frame.preamble_header.ch_bw == 2'b00 && frame.preamble_header.format_mod < VHT) {
            pt_rate_ctrl[i].format_mode_prot_f <= frame.preamble_header.format_mod;
            pt_rate_ctrl[i].format_mode_prot_f != NON_HT_DUP_OFDM;
          } else if (frame.preamble_header.ch_bw == 2'b01) {//only 40MHz
            pt_rate_ctrl[i].format_mode_prot_f inside {NON_HT_DUP_OFDM,HT_MF};
          } else {//only 80MHz
            pt_rate_ctrl[i].format_mode_prot_f == NON_HT_DUP_OFDM;
          }
  `endif
          if (frame.preamble_header.format_mod <= VHT) {
            pt_rate_ctrl[i].mcs_idx_prot_f     <= frame.preamble_header.user_header[0].mcs_f;
            pt_rate_ctrl[i].mcs_idx_prot_f     inside {[0:7]};
          } else {
            pt_rate_ctrl[i].mcs_idx_prot_f     <= frame.preamble_header.user_header_he[0].mcs_f;
            pt_rate_ctrl[i].mcs_idx_prot_f     inside {[0:7]};
          }
          pt_rate_ctrl[i].bw_prot_f          == frame.preamble_header.ch_bw;
          pt_rate_ctrl[i].bw_f               == frame.preamble_header.ch_bw;
          if (frame.preamble_header.format_mod == NON_HT) {
            pt_rate_ctrl[i].gi_type_f          == {frame.preamble_header.preamble_type,1'b0};
          } else {
            pt_rate_ctrl[i].gi_type_f          == frame.preamble_header.gi_type;
          }
          if (frame.preamble_header.format_mod != HE_TB)
            pt_rate_ctrl[i].format_mode_f      == frame.preamble_header.format_mod;

          // HE rates
          if (frame.preamble_header.format_mod > VHT) {
            pt_rate_ctrl[i].mcs_idx_f == {frame.preamble_header.user_header_he[0].nss_f,
                                          frame.preamble_header.user_header_he[0].mcs_f[3:0]};
          } else if (frame.preamble_header.format_mod == VHT) {
            if (frame.preamble_header.stbc == 0 ) {
              pt_rate_ctrl[i].mcs_idx_f == {frame.preamble_header.user_header[0].num_sts_f,
                                            frame.preamble_header.user_header[0].mcs_f[3:0]};
            } else {// STBC = 1
              pt_rate_ctrl[i].mcs_idx_f == {((frame.preamble_header.user_header[0].num_sts_f + 1)/2) - 1,
                                            frame.preamble_header.user_header[0].mcs_f[3:0]};
            }
          } else if (   frame.preamble_header.format_mod == HT_GF
                     || frame.preamble_header.format_mod == HT_MF){

            pt_rate_ctrl[i].mcs_idx_f == frame.preamble_header.user_header[0].mcs_f;

          } else {

            pt_rate_ctrl[i].mcs_idx_f == get_mcs_from_legacy_rate(frame.preamble_header.leg_rate);

          }
        }//foreach
      //}
    }//if WRITE_TX_FRAME
  }// constraint c_pt_rate_control
  //----------------------------------------------

  function void post_randomize();
    super.post_randomize();

    // prevent DSSS/CCK frames in protection
    foreach (pt_rate_ctrl[i]) begin
      if (pt_rate_ctrl[i].format_mode_prot_f inside {NON_HT, NON_HT_DUP_OFDM}) begin
        pt_rate_ctrl[i].mcs_idx_prot_f = $urandom_range(4,11);
      end
    end//foreach
  endfunction : post_randomize

  //------------------------------------------------------------
  // custom print function
  //------------------------------------------------------------
  function void do_print(uvm_printer printer);
    super.do_print(printer);

    if (cmd == READ_RX_FRAME) begin
      foreach (rhd_list[i])
        printer.print_string("RHD list", $sformatf("[%0d] %s",i,rhd_list[i].sprint()));
      foreach (rpd_list[i])
        foreach (rpd_list[i][j])
          printer.print_string("RPD list", $sformatf("[%0d][%0d] %s",i,j,rpd_list[i][j].sprint()));
    end
    else if (cmd == WRITE_TX_FRAME) begin
      printer.print_string("mac ctrl info 1", $sformatf("%p",mac_ctrl_info_1));
      printer.print_string("mac ctrl info 2", $sformatf("%p",mac_ctrl_info_2));
      printer.print_string("pt_mac_ctrl_info_2", $sformatf("%p",pt_mac_ctrl_info_2));
      printer.print_string("pt_ksr_index", $sformatf("%p",pt_ksr_index));
      foreach (pt_rate_ctrl[i]) begin
        printer.print_string("pt_rate_ctrl", $sformatf("%p",pt_rate_ctrl[i]));
      end
    end
    else if (cmd == RD_TX_STATUS) begin
      printer.print_int("Skip A-THD", skip_athd, 1, UVM_HEX);
      foreach (status_info_list[i])
        printer.print_string("status_info_list", $sformatf("%p",status_info_list[i]));
    end
    else if (cmd inside {WRITE_MEM, READ_MEM}) begin
      printer.print_int("RD/WR address", rd_wr_addr, 32, UVM_HEX);
      printer.print_int("RD/WR data", rd_wr_data, 32, UVM_HEX);
    end

  endfunction : do_print

  //------------------------------------------------------------
  // custom copy function
  //------------------------------------------------------------
  function void do_copy(uvm_object rhs);
    sram_seq_item rhs_;

    if (!$cast(rhs_,rhs)) begin
      `uvm_fatal(get_type_name(), "do_copy cast failed!")
    end
    super.do_copy(rhs);

    cmd                = rhs_.cmd;
    thd_head_addr      = rhs_.thd_head_addr;
    rhd_head_addr      = rhs_.rhd_head_addr;
    rpd_head_addr      = rhs_.rpd_head_addr;
    pt_head_addr       = rhs_.pt_head_addr;
    use_data_ptr       = rhs_.use_data_ptr;
    num_of_rhd         = rhs_.num_of_rhd;
    num_of_rpd         = rhs_.num_of_rpd;
    mac_ctrl_info_1    = rhs_.mac_ctrl_info_1;
    mac_ctrl_info_2    = rhs_.mac_ctrl_info_2;
    pt_mac_ctrl_info_2 = rhs_.pt_mac_ctrl_info_2;

    rhd_list = new[rhs_.rhd_list.size()];
    foreach (rhs_.rhd_list[i])
      rhd_list[i] = rhs_.rhd_list[i];

    rpd_list = new[rhs_.rpd_list.size()];
    foreach (rhs_.rpd_list[i]) begin
      rpd_list[i] = new[rhs_.rpd_list[i].size()];
      foreach (rhs_.rpd_list[i][j])
        rpd_list[i][j] = rhs_.rpd_list[i][j];
    end

    pt_rate_ctrl = new[rhs_.pt_rate_ctrl.size()];
    foreach (rhs_.pt_rate_ctrl[i])
      pt_rate_ctrl[i] = rhs_.pt_rate_ctrl[i];

    status_info_list = new[rhs_.status_info_list.size()];
    foreach (rhs_.status_info_list[i])
      status_info_list[i] = rhs_.status_info_list[i];

    if (cmd == WRITE_TX_FRAME) begin
      frame = PPDU_frame::type_id::create("frame");
      // create default container frame for monitor to store collected data
      assert (frame.randomize() with { frame.kind == SINGLETON;
                                       frame.ppdu_format == NON_HT;
                                       frame.num_of_users == 1;
                                     });
      frame.do_copy(rhs_.frame);
    end

  endfunction : do_copy

endclass : sram_seq_item

`endif //SRAM_SEQ_ITEM_SV
