//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef AMPDU_FRAME_SV
`define AMPDU_FRAME_SV

class AMPDU_frame extends uvm_object;

  rand bit                       aggregated; // 0-not, 1-aggregation
  rand mpdu_frame_type_e         mpdu_frame_type[];
  rand format_mod_e              ppdu_format;
       bit                       tx_frame;        // indication that frame is for TX
       bit                       is_dsss;         // flag indicating that singleton frame is NON-HT with DSSS mod.
       int                       payload_size;    // this is used for constraining the MPDU payload size
       security_wrapper_engine   security_wrap;   // security wrapper object for encryption and decryption
       MPDU_frame                mpdu_frame[];
       delimiter                 delimit[];
       blank_delimiter           blank_delimit[int];
       int                       min_mpdu_start_spacing;
       int                       padding[int];
       bit                       bar_attached;
       bit                       no_valid_frames;  // flag indicating that there
                                                   // are no valid MPDU frames collect
  rand int                       dsss_length;      // limit DSSS-CCK frame length
`ifdef STANDALONE_PHY
  rand octet_t                   ampdu_payload[]; // in PHY standalone we just want random content
`endif

  `uvm_object_utils_begin(AMPDU_frame)
`ifdef STANDALONE_PHY
    `uvm_field_array_int(ampdu_payload, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
`else
    `uvm_field_int(aggregated, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_array_enum(mpdu_frame_type_e, mpdu_frame_type, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(is_dsss, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_object(security_wrap, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_array_object(mpdu_frame, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_array_object(delimit, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_aa_object_int(blank_delimit, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_aa_int_int(padding, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(no_valid_frames, UVM_DEFAULT | UVM_NOPRINT | UVM_NOCOMPARE | UVM_NOCOPY)
`endif
  `uvm_object_utils_end

  //---------------------------------------------
  // constraints
  //---------------------------------------------
`ifdef STANDALONE_PHY
  constraint c_mpdu_type_size {
    mpdu_frame_type.size() == 0;
  }
  constraint c_no_error_frames {}
`else
  constraint c_mpdu_type_size {
    solve ppdu_format before mpdu_frame_type;

    mpdu_frame_type.size() > 0;
    mpdu_frame_type.size() <= `MAX_NUM_MPDU_FRAMES;
  }

  constraint c_no_error_frames {
    if (aggregated) {
      foreach (mpdu_frame_type[i])
        mpdu_frame_type[i] inside {QOS_DATA,
                                   QOS_DATA_CF_ACK,
                                   QOS_DATA_CF_POLL,
                                   QOS_DATA_CF_ACK_CF_POLL};
    } else {
      foreach (mpdu_frame_type[i])
        mpdu_frame_type[i] != ERROR_FRAME;
    }
  }

  constraint c_dsss_length {
    dsss_length dist {
      [`MAX_MAC_HEADER_SIZE_DSSS:500] :/ 85,
      [500:1000]                      :/ 10,
      [1000:`MAX_MPDU_SIZE_DSSS]      :/ 5
    };
  }
`endif
  //---------------------------------------------
  // end of constraints
  //---------------------------------------------

  function new (string name = "AMPDU_frame");
    super.new(name);
    payload_size = 0;
    bar_attached = 0;
    no_valid_frames = 0;
  endfunction

  // size of A-MPDU frame in bytes
  function int size();
    int accu = 0, i;
`ifdef STANDALONE_PHY
    accu = ampdu_payload.size();
`else
    foreach (mpdu_frame[i]) accu += mpdu_frame[i].size();
    foreach (delimit[i])    accu += delimit[i].size();
    if (padding.first(i)) do
      accu += padding[i];
    while (padding.next(i));
    if (blank_delimit.first(i)) do
      accu += blank_delimit[i].size;
    while (blank_delimit.next(i));

    // when BAR is attached substract BAR frame size
    if (bar_attached) accu -= mpdu_frame[mpdu_frame.size()-1].size();
`endif
    `uvm_info(get_type_name(), $sformatf("A-MPDU frame size is %d",accu), UVM_FULL)
    return accu;
  endfunction : size

  // --------------------------------------------------------------
  // in post randomize all MPDU freames are created and randomized,
  // delimiters are set and calculated, padding added when needed
  // --------------------------------------------------------------
`ifndef STANDALONE_PHY
  function void post_randomize();
    // no need to attach BAR when testing PHY standalone
    if (tx_frame) attach_bar_frame();
    create_mpdu_frames();
    randomize_mpdu_frames();
    create_delimiter();
    add_padding();
    create_blank_delimiter();
  endfunction : post_randomize
`endif

  // --------------------------------------------------------------
  // function for creating all MPDU frames in relative to type
  // --------------------------------------------------------------
  function void create_mpdu_frames();
    mpdu_frame = new[mpdu_frame_type.size()];

    for (int i=0; i < mpdu_frame_type.size(); i++) begin
      case (mpdu_frame_type[i])
        ASSOCIATION_REQUEST     : mpdu_frame[i] = ASSOCIATION_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
        ASSOCIATION_RESPONSE    : mpdu_frame[i] = ASSOCIATION_RESPONSE_frame::type_id::create(mpdu_frame_type[i].name());
        REASSOCIATION_REQUEST   : mpdu_frame[i] = REASSOCIATION_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
        REASSOCIATION_RESPONSE  : mpdu_frame[i] = REASSOCIATION_RESPONSE_frame::type_id::create(mpdu_frame_type[i].name());
        PROBE_REQUEST           : mpdu_frame[i] = PROBE_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
        PROBE_RESPONSE          : mpdu_frame[i] = PROBE_RESPONSE_frame::type_id::create(mpdu_frame_type[i].name());
        TIMING_ADVERTISEMENT    : mpdu_frame[i] = TIMING_ADVERTISEMENT_frame::type_id::create(mpdu_frame_type[i].name());
        BEACON                  : mpdu_frame[i] = BEACON_frame::type_id::create(mpdu_frame_type[i].name());
        ATIM                    : mpdu_frame[i] = ATIM_frame::type_id::create(mpdu_frame_type[i].name());
        DISASSOCIATION          : mpdu_frame[i] = DISASSOCIATION_frame::type_id::create(mpdu_frame_type[i].name());
        AUTHENTICATION          : mpdu_frame[i] = AUTHENTICATION_frame::type_id::create(mpdu_frame_type[i].name());
        DEAUTHENTICATION        : mpdu_frame[i] = DEAUTHENTICATION_frame::type_id::create(mpdu_frame_type[i].name());
        ACTION                  : mpdu_frame[i] = ACTION_frame::type_id::create(mpdu_frame_type[i].name());
        ACTION_NOACK            : mpdu_frame[i] = ACTION_NOACK_frame::type_id::create(mpdu_frame_type[i].name());
        BEAMFORMING_REPORT_POLL : mpdu_frame[i] = BEAMFORMING_REPORT_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        VHT_NDP_ANNOUNCEMENT    : mpdu_frame[i] = VHT_NDP_ANNOUNCEMENT_frame::type_id::create(mpdu_frame_type[i].name());
        HE_NDP_ANNOUNCEMENT     : mpdu_frame[i] = HE_NDP_ANNOUNCEMENT_frame::type_id::create(mpdu_frame_type[i].name());
        CONTROL_WRAPPER         : mpdu_frame[i] = CONTROL_WRAPPER_frame::type_id::create(mpdu_frame_type[i].name());
        BLOCK_ACK_REQUEST       : mpdu_frame[i] = BLOCK_ACK_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
        BLOCK_ACK               : mpdu_frame[i] = BLOCK_ACK_frame::type_id::create(mpdu_frame_type[i].name());
        PS_POLL                 : mpdu_frame[i] = PS_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        RTS                     : mpdu_frame[i] = RTS_frame::type_id::create(mpdu_frame_type[i].name());
        CTS                     : mpdu_frame[i] = CTS_frame::type_id::create(mpdu_frame_type[i].name());
        ACK                     : mpdu_frame[i] = ACK_frame::type_id::create(mpdu_frame_type[i].name());
        TRIGGER                 : mpdu_frame[i] = TRIGGER_frame::type_id::create(mpdu_frame_type[i].name());
        CF_END                  : mpdu_frame[i] = CF_END_frame::type_id::create(mpdu_frame_type[i].name());
        CF_END_CF_ACK           : mpdu_frame[i] = CF_END_CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
        DATA                    : mpdu_frame[i] = DATA_frame::type_id::create(mpdu_frame_type[i].name());
        DATA_CF_ACK             : mpdu_frame[i] = DATA_CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
        DATA_CF_POLL            : mpdu_frame[i] = DATA_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        DATA_CF_ACK_CF_POLL     : mpdu_frame[i] = DATA_CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        NULL                    : mpdu_frame[i] = NULL_frame::type_id::create(mpdu_frame_type[i].name());
        CF_ACK                  : mpdu_frame[i] = CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
        CF_POLL                 : mpdu_frame[i] = CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        CF_ACK_CF_POLL          : mpdu_frame[i] = CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        QOS_DATA                : mpdu_frame[i] = QOS_DATA_frame::type_id::create(mpdu_frame_type[i].name());
        QOS_DATA_CF_ACK         : mpdu_frame[i] = QOS_DATA_CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
        QOS_DATA_CF_POLL        : mpdu_frame[i] = QOS_DATA_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        QOS_DATA_CF_ACK_CF_POLL : mpdu_frame[i] = QOS_DATA_CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        QOS_NULL                : mpdu_frame[i] = QOS_NULL_frame::type_id::create(mpdu_frame_type[i].name());
        QOS_CF_POLL             : mpdu_frame[i] = QOS_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        QOS_CF_ACK_CF_POLL      : mpdu_frame[i] = QOS_CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
        BEAMFORMING_REPORT      : mpdu_frame[i] = BEAMFORMING_REPORT_frame::type_id::create(mpdu_frame_type[i].name());
        ERROR_FRAME             : mpdu_frame[i] = error_MPDU_frame::type_id::create(mpdu_frame_type[i].name());
        default                 : mpdu_frame[i] = error_MPDU_frame::type_id::create(mpdu_frame_type[i].name());
      endcase
    end
  endfunction : create_mpdu_frames

  // --------------------------------------------------------------
  // function for creating MPDU frame in dynamic array by its type
  // @i - index in mpdu_frame_type array
  // --------------------------------------------------------------
  function void create_mpdu_frame_by_type(int i);
    case (mpdu_frame_type[i])
      ASSOCIATION_REQUEST     : mpdu_frame[i] = ASSOCIATION_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
      ASSOCIATION_RESPONSE    : mpdu_frame[i] = ASSOCIATION_RESPONSE_frame::type_id::create(mpdu_frame_type[i].name());
      REASSOCIATION_REQUEST   : mpdu_frame[i] = REASSOCIATION_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
      REASSOCIATION_RESPONSE  : mpdu_frame[i] = REASSOCIATION_RESPONSE_frame::type_id::create(mpdu_frame_type[i].name());
      PROBE_REQUEST           : mpdu_frame[i] = PROBE_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
      PROBE_RESPONSE          : mpdu_frame[i] = PROBE_RESPONSE_frame::type_id::create(mpdu_frame_type[i].name());
      TIMING_ADVERTISEMENT    : mpdu_frame[i] = TIMING_ADVERTISEMENT_frame::type_id::create(mpdu_frame_type[i].name());
      BEACON                  : mpdu_frame[i] = BEACON_frame::type_id::create(mpdu_frame_type[i].name());
      ATIM                    : mpdu_frame[i] = ATIM_frame::type_id::create(mpdu_frame_type[i].name());
      DISASSOCIATION          : mpdu_frame[i] = DISASSOCIATION_frame::type_id::create(mpdu_frame_type[i].name());
      AUTHENTICATION          : mpdu_frame[i] = AUTHENTICATION_frame::type_id::create(mpdu_frame_type[i].name());
      DEAUTHENTICATION        : mpdu_frame[i] = DEAUTHENTICATION_frame::type_id::create(mpdu_frame_type[i].name());
      ACTION                  : mpdu_frame[i] = ACTION_frame::type_id::create(mpdu_frame_type[i].name());
      ACTION_NOACK            : mpdu_frame[i] = ACTION_NOACK_frame::type_id::create(mpdu_frame_type[i].name());
      BEAMFORMING_REPORT_POLL : mpdu_frame[i] = BEAMFORMING_REPORT_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      VHT_NDP_ANNOUNCEMENT    : mpdu_frame[i] = VHT_NDP_ANNOUNCEMENT_frame::type_id::create(mpdu_frame_type[i].name());
      HE_NDP_ANNOUNCEMENT     : mpdu_frame[i] = HE_NDP_ANNOUNCEMENT_frame::type_id::create(mpdu_frame_type[i].name());
      CONTROL_WRAPPER         : mpdu_frame[i] = CONTROL_WRAPPER_frame::type_id::create(mpdu_frame_type[i].name());
      BLOCK_ACK_REQUEST       : mpdu_frame[i] = BLOCK_ACK_REQUEST_frame::type_id::create(mpdu_frame_type[i].name());
      BLOCK_ACK               : mpdu_frame[i] = BLOCK_ACK_frame::type_id::create(mpdu_frame_type[i].name());
      PS_POLL                 : mpdu_frame[i] = PS_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      RTS                     : mpdu_frame[i] = RTS_frame::type_id::create(mpdu_frame_type[i].name());
      CTS                     : mpdu_frame[i] = CTS_frame::type_id::create(mpdu_frame_type[i].name());
      ACK                     : mpdu_frame[i] = ACK_frame::type_id::create(mpdu_frame_type[i].name());
      TRIGGER                 : mpdu_frame[i] = TRIGGER_frame::type_id::create(mpdu_frame_type[i].name());
      CF_END                  : mpdu_frame[i] = CF_END_frame::type_id::create(mpdu_frame_type[i].name());
      CF_END_CF_ACK           : mpdu_frame[i] = CF_END_CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
      DATA                    : mpdu_frame[i] = DATA_frame::type_id::create(mpdu_frame_type[i].name());
      DATA_CF_ACK             : mpdu_frame[i] = DATA_CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
      DATA_CF_POLL            : mpdu_frame[i] = DATA_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      DATA_CF_ACK_CF_POLL     : mpdu_frame[i] = DATA_CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      NULL                    : mpdu_frame[i] = NULL_frame::type_id::create(mpdu_frame_type[i].name());
      CF_ACK                  : mpdu_frame[i] = CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
      CF_POLL                 : mpdu_frame[i] = CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      CF_ACK_CF_POLL          : mpdu_frame[i] = CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      QOS_DATA                : mpdu_frame[i] = QOS_DATA_frame::type_id::create(mpdu_frame_type[i].name());
      QOS_DATA_CF_ACK         : mpdu_frame[i] = QOS_DATA_CF_ACK_frame::type_id::create(mpdu_frame_type[i].name());
      QOS_DATA_CF_POLL        : mpdu_frame[i] = QOS_DATA_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      QOS_DATA_CF_ACK_CF_POLL : mpdu_frame[i] = QOS_DATA_CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      QOS_NULL                : mpdu_frame[i] = QOS_NULL_frame::type_id::create(mpdu_frame_type[i].name());
      QOS_CF_POLL             : mpdu_frame[i] = QOS_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      QOS_CF_ACK_CF_POLL      : mpdu_frame[i] = QOS_CF_ACK_CF_POLL_frame::type_id::create(mpdu_frame_type[i].name());
      BEAMFORMING_REPORT      : mpdu_frame[i] = BEAMFORMING_REPORT_frame::type_id::create(mpdu_frame_type[i].name());
      ERROR_FRAME             : mpdu_frame[i] = error_MPDU_frame::type_id::create(mpdu_frame_type[i].name());
      default                 : mpdu_frame[i] = error_MPDU_frame::type_id::create(mpdu_frame_type[i].name());
    endcase
  endfunction : create_mpdu_frame_by_type

  // --------------------------------------------------------------
  // attach Block Acknowledge Request to the end of A-MPDU frame
  // --------------------------------------------------------------
  function void attach_bar_frame();
    if (aggregated == 1'b1 && bar_attached == 1'b0) begin
      mpdu_frame_type = new[mpdu_frame_type.size()+1](mpdu_frame_type);
      // put BLOCK_ACK_REQUEST type on the end of array
      mpdu_frame_type[mpdu_frame_type.size()-1] = BLOCK_ACK_REQUEST;
      bar_attached = 1'b1;
    end
  endfunction : attach_bar_frame

  // --------------------------------------------------------------
  // randomize all MPDU frames that are created
  // --------------------------------------------------------------
  function void randomize_mpdu_frames();
    BLOCK_ACK_REQUEST_frame bar;

    foreach (mpdu_frame[i]) begin

      // when sending A-MPDU, QoS control fields for EOSP, ACK policy,
      // to/from DS, MAC addresses, duration, sequence control in incremental
      if (aggregated && mpdu_frame_type[i] != BLOCK_ACK_REQUEST) begin
        assert (mpdu_frame[i].randomize() with {
          if (i>0) {
            // sequence control number should be incremented by 1,
            // prevent randomization error
            if ((mpdu_frame[i-1].MAC_header.seq_ctrl[0].sequence_number_f + 1) <= 12'hFFF) {
              mpdu_frame[i].MAC_header.seq_ctrl[0].sequence_number_f == mpdu_frame[i-1].MAC_header.seq_ctrl[0].sequence_number_f + 1;
              mpdu_frame[i].MAC_header.seq_ctrl[0].fragment_number_f == 0;
            } else {
              mpdu_frame[i].MAC_header.seq_ctrl[0].sequence_number_f == 0;
              mpdu_frame[i].MAC_header.seq_ctrl[0].fragment_number_f == 0;
            }
            // every QoC control is same in MPDUs inside A-MPDU
            mpdu_frame[i].MAC_header.qos_ctrl[0] == mpdu_frame[i-1].MAC_header.qos_ctrl[0];
            // RA,TA addresses are same in A-MPDU, protection and duration also
            mpdu_frame[i].MAC_header.frame_ctrl.to_DS_f   == mpdu_frame[i-1].MAC_header.frame_ctrl.to_DS_f;
            mpdu_frame[i].MAC_header.frame_ctrl.from_DS_f == mpdu_frame[i-1].MAC_header.frame_ctrl.from_DS_f;
            mpdu_frame[i].MAC_header.frame_ctrl.protected_frame_f == mpdu_frame[i-1].MAC_header.frame_ctrl.protected_frame_f;
            mpdu_frame[i].MAC_header.frame_ctrl.order_f   == mpdu_frame[i-1].MAC_header.frame_ctrl.order_f;
            mpdu_frame[i].MAC_header.duration_id          == mpdu_frame[i-1].MAC_header.duration_id;
            mpdu_frame[i].MAC_header.ht_ctrl[0]           == mpdu_frame[i-1].MAC_header.ht_ctrl[0];
          } else {
            mpdu_frame[i].MAC_header.frame_ctrl.order_f == 1'b1;

            if (ppdu_format >= HE_SU)
              mpdu_frame[i].MAC_header.qos_ctrl[0].ack_policy_f inside {2'b00,2'b10}; // receive BLOCK_ACK, HTP ACK
            else
              mpdu_frame[i].MAC_header.qos_ctrl[0].ack_policy_f == 0; // receive BLOCK_ACK

            mpdu_frame[i].MAC_header.seq_ctrl[0].fragment_number_f == 0;

            if (mpdu_frame[i].MAC_header.ht_ctrl.size()) {
              // set VHT field
              mpdu_frame[i].MAC_header.ht_ctrl[0][0] == (ppdu_format >= VHT);
              // set HE field
              mpdu_frame[i].MAC_header.ht_ctrl[0][1] == (ppdu_format >= HE_SU);
              // constraint HE control
              if (ppdu_format >= HE_SU) {
                mpdu_frame[i].MAC_header.ht_ctrl[0][5:2] < 7;   // control ID
                mpdu_frame[i].MAC_header.ht_ctrl[0][10:6] != 0; // UL data symbols
                mpdu_frame[i].MAC_header.ht_ctrl[0][30:29] inside {0,1,3}; // UL MCS
                // RU allocation constraining for the HE_TB response
                `ifdef RW_NX_CHBW20
                  mpdu_frame[i].MAC_header.ht_ctrl[0][18:12] inside {[0:8],[37:40],[53:54],61};
                `elsif RW_NX_CHBW4020
                  mpdu_frame[i].MAC_header.ht_ctrl[0][18:12] inside {[0:17],[37:44],[53:56],61,62,65};
                `elsif RW_NX_CHBW804020
                  mpdu_frame[i].MAC_header.ht_ctrl[0][18:12] inside {[0:67]};
                `endif
                mpdu_frame[i].MAC_header.ht_ctrl[0][11] == 0;
              }//HT control exists
            }
          }

          // limit MPDU size inside AMPDU
          if (payload_size > 0){
            if (mpdu_frame[i].MAC_header.frame_ctrl.protected_frame_f)
              //make frame body shorter because of protection header and footer
              mpdu_frame[i].frame_body.size() <= (payload_size - 40);
            else
              mpdu_frame[i].frame_body.size() <= payload_size;
          }
        });
      end// aggregated
      // BAR should be randomized as compressed BAR
      else if (bar_attached &&  mpdu_frame_type[i] == BLOCK_ACK_REQUEST) begin
        bar = new();
        assert (bar.randomize() with {
          bar_variant == COMPRESSED_BAR;
          // set starting sequence number to be starting sequence number of
          // first MPDU frame inside A-MPDU
          starting_seq_num == mpdu_frame[0].MAC_header.seq_ctrl[0].sequence_number_f;
        });
        mpdu_frame[i].copy(bar);
      end// bar_attached
      else begin
        assert (mpdu_frame[i].randomize() with {
          // constraint order/HTC bit
          if (   ppdu_format >= HT_MF
              && mpdu_frame_type[i] inside {QOS_DATA,
                                            QOS_DATA_CF_ACK,
                                            QOS_DATA_CF_POLL,
                                            QOS_DATA_CF_ACK_CF_POLL,
                                            QOS_NULL,
                                            ACTION}){

            mpdu_frame[i].MAC_header.frame_ctrl.order_f == 1'b1;
          } else {
            mpdu_frame[i].MAC_header.frame_ctrl.order_f == 1'b0;
          }

          // constraint HT control
          if (mpdu_frame[i].MAC_header.ht_ctrl.size()) {
            // set VHT field
            mpdu_frame[i].MAC_header.ht_ctrl[0][0] == (ppdu_format >= VHT);
            // set HE field
            mpdu_frame[i].MAC_header.ht_ctrl[0][1] == (ppdu_format >= HE_SU);
            // constraint HE control ID and HE TB PPDU length
            if (ppdu_format >= HE_SU) {
              mpdu_frame[i].MAC_header.ht_ctrl[0][5:2] < 7;
              mpdu_frame[i].MAC_header.ht_ctrl[0][10:6] != 0;
              // RU allocation constraining for the HE_TB response
              `ifdef RW_NX_CHBW20
                mpdu_frame[i].MAC_header.ht_ctrl[0][18:12] inside {[0:8],[37:40],[53:54],61};
              `elsif RW_NX_CHBW4020
                mpdu_frame[i].MAC_header.ht_ctrl[0][18:12] inside {[0:17],[37:44],[53:56],61,62,65};
              `elsif RW_NX_CHBW804020
                mpdu_frame[i].MAC_header.ht_ctrl[0][18:12] inside {[0:67]};
              `endif
              mpdu_frame[i].MAC_header.ht_ctrl[0][11] == 0;
            }
          }

          // if frame is sent as DSSS data needs to be limited in data MPDUs
          if (is_dsss && mpdu_frame_type[i][5:4] == `DATA_MPDU) {
            mpdu_frame[i].frame_body.size() <= dsss_length;
          } else if (payload_size > 0){
            // limit MPDU size inside AMPDU
            if (payload_size > 0){
              if (mpdu_frame[i].MAC_header.frame_ctrl.protected_frame_f)
                //make frame body shorter because of protection header and footer
                mpdu_frame[i].frame_body.size() <= (payload_size - 40);
              else
                mpdu_frame[i].frame_body.size() <= payload_size;
            }
          }
        });
      end// else
    end// foreach
  endfunction : randomize_mpdu_frames

  // --------------------------------------------------------------
  // create delimiters for every MPDU frame inside A-MPDU frame,
  // except when singleton frame is created, then there is no delimiter
  // --------------------------------------------------------------
  function void create_delimiter();
    // only if there is more then one MPDU frame
    // or frame is VHT delimiters are created
    if (ppdu_format >= VHT || aggregated == 1'b1) begin
      // attached BAR doesn't need delimiter
      delimit = new[mpdu_frame.size()-bar_attached]; // create array

      foreach (delimit[i]) begin
        delimit[i]             = delimiter::type_id::create($sformatf("delimit[%0d]",i));
        delimit[i].mpdu_length = mpdu_frame[i].size();
        delimit[i].eof         = ((ppdu_format >= VHT) && !aggregated) ? 1 : 0;
        // calculate CRC for delimiter
        delimit[i].crc         = delimiter_crc_func({delimit[i].mpdu_length[11:0],
                                                     delimit[i].mpdu_length[13:12],
                                                     delimit[i].reserved,
                                                     delimit[i].eof});
      end//for
    end//if
  endfunction : create_delimiter

  // --------------------------------------------------------------
  // create blank delimters that can occur during trasmision,
  // blank delimiters are added after every delimiter and MPDU frame
  // --------------------------------------------------------------
  function void create_blank_delimiter();
    blank_delimiter blank_del;
    int             size;

    blank_delimit.delete();

    // add blank delimiters only if A-MPDU frame
    if (aggregated == 1'b1 || (ppdu_format >= VHT && mpdu_frame.size() > 1)) begin
      for (int i=0; i < mpdu_frame.size()-bar_attached; i++) begin
        // calculate number of blank delimiters needed for minimum MPDU start
        // spacing
        size = $ceil((min_mpdu_start_spacing - mpdu_frame[i].size)/4.0);
        if (size > 0) begin
          blank_del = new();
          blank_del.blank_delimiter_num = size;
          blank_delimit[i] = blank_del;
        end//if
      end//foreach
    end//aggregated
  endfunction : create_blank_delimiter

  // --------------------------------------------------------------
  // add padding bytes where needed inside A-MPDU
  // --------------------------------------------------------------
  function void add_padding();
    int       num_of_pads = 0;
    int       up_limit = 0;

    padding.delete();
    // add padding for EOF when sending aggregated frame
    up_limit = mpdu_frame.size()-bar_attached;

    if (aggregated == 1'b1 || ppdu_format >= VHT) begin
      for (int i=0; i < up_limit; i++) begin
        // when size of MPDU+delimiter is not power of 4 add padding
        num_of_pads = (mpdu_frame[i].size() + delimit[i].size()) % 4;
        num_of_pads = (num_of_pads == 0) ? 0 : 4 - num_of_pads;
        if (num_of_pads != 0) begin
          padding[i] = num_of_pads; //store number of padding needed
        end
      end//for
    end//if
  endfunction : add_padding

  // --------------------------------------------------------------
  // set MAC addresses in each MPDU
  // @RA - receiver address
  // @TA - transmitter address
  // @DA - destination address
  // @SA - source address
  // @BSSID - basic service set indetifier
  // @amsdu - indication if A-MSDU
  // --------------------------------------------------------------
  virtual function void set_MAC_addr(mac_address_t RA = 0, mac_address_t TA = 0,
                                     mac_address_t DA = 0, mac_address_t SA = 0,
                                     mac_address_t BSSID = 0, bit amsdu = 0);

     foreach (mpdu_frame[i]) begin
       mpdu_frame[i].set_MAC_addr(.RA(RA), .TA(TA),
                                  .DA(DA), .SA(SA),
                                  .BSSID(BSSID), .amsdu(amsdu));
       // after any change of values inside MPDU frame FCS needs to be
       // recalculated
       if (mpdu_frame_type[i] != ERROR_FRAME)
         mpdu_frame[i].recalculate_fcs();
     end
  endfunction : set_MAC_addr

  //------------------------------------------------------------
  // get MAC address, RA or TA
  // @ RA - 1 return RA address, 0 return TA address
  //------------------------------------------------------------
  virtual function mac_address_t get_MAC_addr(bit RA = 0);
    foreach (mpdu_frame_type[i]) begin
      if (mpdu_frame_type[i] != ERROR_FRAME)
        return mpdu_frame[i].get_MAC_addr(RA);
    end//foreach

    return 0;
  endfunction : get_MAC_addr

  //------------------------------------------------------------
  // function for setup of duration/ID field
  //------------------------------------------------------------
  virtual function void set_MAC_duration(duration_t dur);
    foreach (mpdu_frame[i]) begin
      mpdu_frame[i].set_MAC_duration(dur);
      // after any change of values inside MPDU frame FCS needs to be
      // recalculated
      if (mpdu_frame_type[i] != ERROR_FRAME)
        mpdu_frame[i].recalculate_fcs();
    end
  endfunction : set_MAC_duration

  virtual function duration_t get_MAC_duration();
    foreach (mpdu_frame_type[i]) begin
      if (mpdu_frame_type[i] != ERROR_FRAME)
        return mpdu_frame[i].MAC_header.duration_id;
    end//foreach

    return 0;
  endfunction : get_MAC_duration

  //------------------------------------------------------------
  // function for setting the SN in the A-MPDU, based on TID
  //------------------------------------------------------------
  virtual function void set_MAC_seq_num(int SSN);

    foreach (mpdu_frame[i]) begin
      // after any change of values inside MPDU frame FCS needs to be
      // recalculated
      if (mpdu_frame_type[i] != ERROR_FRAME) begin
        mpdu_frame[i].MAC_header.seq_ctrl[0].sequence_number_f = SSN + i;
        mpdu_frame[i].recalculate_fcs();
      end
    end
  endfunction : set_MAC_seq_num

  //------------------------------------------------------------
  // get sequence number from MPDU sequence control
  //------------------------------------------------------------
  function int get_MAC_seq_num(int frame_num = 0);
    if (mpdu_frame[frame_num].MAC_header.seq_ctrl.size()) begin
      return mpdu_frame[frame_num].MAC_header.seq_ctrl[0].sequence_number_f;
    end
    else begin
      `uvm_warning(get_type_name(),"No sequence control in MAC header!")
      return -1;
    end
  endfunction : get_MAC_seq_num

  //------------------------------------------------------------
  // get TID from QoS
  //------------------------------------------------------------
  function bit [3:0] get_MAC_tid_num();
    foreach (mpdu_frame_type[i]) begin
      if (mpdu_frame_type[i] != ERROR_FRAME) begin

        if (mpdu_frame[i].MAC_header.qos_ctrl.size()) begin
          return mpdu_frame[i].MAC_header.qos_ctrl[0].tid_f;
        end
        else begin
          `uvm_warning(get_type_name(),"No TID in MAC header!")
          return 0;
        end

      end//if not error
    end//foreach
  endfunction : get_MAC_tid_num

  //------------------------------------------------------------
  // set TID from QoS
  //------------------------------------------------------------
  function void set_MAC_tid_num(bit [3:0] tid);
    foreach (mpdu_frame[i]) begin
      // after any change of values inside MPDU frame FCS needs to be
      // recalculated
      if (mpdu_frame_type[i] != ERROR_FRAME) begin
        mpdu_frame[i].MAC_header.qos_ctrl[0].tid_f = tid;
        mpdu_frame[i].recalculate_fcs();
      end
    end
  endfunction : set_MAC_tid_num

  //------------------------------------------------------------
  // get function of QoS control ACK policy field
  //------------------------------------------------------------
  virtual function bit [1:0] get_qos_ack_policy();
    foreach (mpdu_frame_type[i]) begin
      if (mpdu_frame_type[i] != ERROR_FRAME)
        return mpdu_frame[i].get_qos_ack_policy();
    end//foreach

    return 0;
  endfunction : get_qos_ack_policy

  //------------------------------------------------------------
  // function for setup of QoS control filed ACK policy field
  //------------------------------------------------------------
  virtual function void set_qos_ack_policy(bit [1:0] ack_pol);
    foreach (mpdu_frame[i]) begin
      mpdu_frame[i].set_qos_ack_policy(ack_pol);
      // after any change of values inside MPDU frame FCS needs to be
      // recalculated
      mpdu_frame[i].recalculate_fcs();
    end
  endfunction : set_qos_ack_policy

  //------------------------------------------------------------
  // function set HT control
  //------------------------------------------------------------
  virtual function void set_ht_control(ht_control_s htctrl);
    foreach (mpdu_frame[i]) begin
      mpdu_frame[i].set_ht_control(htctrl);
      // after any change of values inside MPDU frame FCS needs to be
      // recalculated
      mpdu_frame[i].recalculate_fcs();
    end
  endfunction : set_ht_control

  //------------------------------------------------------------
  // function get HT control
  //------------------------------------------------------------
  virtual function ht_control_s get_ht_control();
    foreach (mpdu_frame_type[i]) begin
      if (mpdu_frame_type[i] != ERROR_FRAME)
        return mpdu_frame[i].get_ht_control();
    end//foreach

    return 0;
  endfunction : get_ht_control

  //------------------------------------------------------------
  // @ret - 1 - frame is protected
  //        0 - frame not protected
  //------------------------------------------------------------
  function bit is_protected();
    foreach (mpdu_frame_type[i]) begin
      if (mpdu_frame_type[i] != ERROR_FRAME)
        return mpdu_frame[i].is_protected();
    end//foreach

    return 0;
  endfunction : is_protected

  // --------------------------------------------------------------
  // compare two A-MPDU objects
  // --------------------------------------------------------------
  virtual function bit do_compare(uvm_object rhs, uvm_comparer comparer);
    AMPDU_frame that;
    bit         frm_diff;

    if (!$cast(that, rhs)) begin
      `uvm_fatal(get_type_name(), $sformatf("Failed cast from 'rhs' to 'that'"))
    end

    do_compare = super.do_compare(rhs, comparer);
    // to be able to compare both frames should have valid content
    if (this.no_valid_frames || that.no_valid_frames) begin
      return do_compare;
    end

    // case when testing PHY standalone and WLAN reception, matlab can
    // generate corrupted data so 2 A-MPDU can differ, 'physical' variable is
    // used to indicate status of fcsOK (Matlab flag)
`ifdef STANDALONE_PHY
    frm_diff = (comparer.physical == 0) ? 1 : 0; // there is FCS error
`else
    frm_diff = (this.delimit.size() != that.delimit.size() && comparer.physical == 0) ? 1 : 0;
`endif

    if (!frm_diff) begin
`ifdef STANDALONE_PHY
      // in standalone PHY TB compare only A-MPDU payload data
      foreach (ampdu_payload[i]) begin
        do_compare &= comparer.compare_field($sformatf("ampdu_payload[%0d]",i),
        this.ampdu_payload[i], that.ampdu_payload[i], $bits(this.ampdu_payload[i]));
      end
`else
      // MPDU TYPE
      if (this.mpdu_frame_type.size()-this.bar_attached !== that.mpdu_frame_type.size()-that.bar_attached) begin
        `uvm_error(get_type_name(), $sformatf("Number of MPDU frame types differ (expected: %0d got: %0d)",
        this.mpdu_frame_type.size(), that.mpdu_frame_type.size()))
        do_compare = 0;
      end

      foreach (mpdu_frame_type[i]) begin
        // don't compare attached BAR and ERROR_FRAME
        if (!(aggregated && mpdu_frame_type[i] == BLOCK_ACK_REQUEST) && that.mpdu_frame_type[i] != ERROR_FRAME)
          do_compare &= comparer.compare_field($sformatf("mpdu_frame_type[%0d]",i),
                        this.mpdu_frame_type[i], that.mpdu_frame_type[i], $bits(this.mpdu_frame_type[i]));
      end

      // MPDU
      if (this.mpdu_frame.size()-this.bar_attached !== that.mpdu_frame.size()-that.bar_attached) begin
        `uvm_error(get_type_name(), $sformatf("Number of MPDU frames differ (expected: %0d got: %0d)",
        this.mpdu_frame.size(), that.mpdu_frame.size()))
        do_compare = 0;
      end

      foreach (mpdu_frame[i]) begin
        // don't compare attached BAR and ERROR_FRAME
        if (!(aggregated && mpdu_frame_type[i] == BLOCK_ACK_REQUEST) && that.mpdu_frame_type[i] != ERROR_FRAME)
          do_compare &= comparer.compare_object($sformatf("mpdu_frame[%0d]",i),
                        this.mpdu_frame[i], that.mpdu_frame[i]);
      end

      // DELIMITERS
      if (this.delimit.size() !== that.delimit.size()) begin
        `uvm_error(get_type_name(), $sformatf("Number of delimiters differ (expected: %0d got: %0d)",
        this.delimit.size(), that.delimit.size()))
        do_compare = 0;
      end

      foreach (delimit[i]) begin
        do_compare &= comparer.compare_object($sformatf("delimit[%0d]",i),
                      this.delimit[i], that.delimit[i]);
      end

      // BLANK DELIMITERS
      if (this.blank_delimit.size() !== that.blank_delimit.size()) begin
        `uvm_error(get_type_name(), $sformatf("Number of blank delimiters differ (expected: %0d got: %0d)",
        this.blank_delimit.size(), that.blank_delimit.size()))
        do_compare = 0;
      end

      foreach (blank_delimit[i]) begin
        do_compare &= comparer.compare_object($sformatf("blank_delimit[%0d]",i),
                      this.blank_delimit[i], that.blank_delimit[i]);
      end

      // PADDING
      if (this.padding.size() !== that.padding.size()) begin
        `uvm_error(get_type_name(), $sformatf("Number of padding differs (expected: %0d got: %0d)",
        this.padding.size(), that.padding.size()))
        do_compare = 0;
      end

      foreach (padding[i]) begin
        do_compare &= comparer.compare_field($sformatf("padding[%0d]",i),
                      this.padding[i], that.padding[i], $bits(this.padding[i]));
      end
`endif//STANDALONE_PHY
    end// if frm_diff == 0

    return do_compare;
  endfunction : do_compare

  // --------------------------------------------------------------
  // print A-MPDU objects
  // --------------------------------------------------------------
  function void do_print(uvm_printer printer);
    super.do_print(printer);

    if (no_valid_frames)
      printer.print_int("No valid frames", no_valid_frames, 1, UVM_HEX);

  endfunction : do_print

  // --------------------------------------------------------------
  // function that clears all dynamic arrays created
  // --------------------------------------------------------------
  virtual function void clear_all_arrays();
`ifdef STANDALONE_PHY
    ampdu_payload.delete();
`else
    mpdu_frame_type.delete();
    mpdu_frame.delete();
    delimit.delete();
    blank_delimit.delete();
    padding.delete();
    bar_attached = 0;
`endif
  endfunction : clear_all_arrays

  //---------------------------------------------
  // prepare frame data for Rx when encryption is
  // set in MPDU frame. Function will prepare data
  // that is encrypted for Rx path
  //---------------------------------------------
  virtual function void setup_rx_encrypted_data();
    // create security engine
    security_wrap = new();
    // randomize security structures
    assert (security_wrap.randomize());

    foreach (mpdu_frame[i]) begin
      // push down securuty object hadle
      mpdu_frame[i].security_wrap = security_wrap;
      mpdu_frame[i].setup_rx_encrypted_data();
    end
    // need to re-calculate delimiters because MPDU size will change
    create_delimiter();
    add_padding();
    create_blank_delimiter();
  endfunction : setup_rx_encrypted_data

  //---------------------------------------------
  // encrypt MPDU frame if protection is set
  //---------------------------------------------
  virtual function void encrypt(bit ra);
    // create security engine
    security_wrap = new();
    // randomize security structures
    assert (security_wrap.randomize());

    foreach (mpdu_frame[i]) begin
      // push down securuty object hadle
      mpdu_frame[i].security_wrap = security_wrap;
      mpdu_frame[i].encrypt(ra);
    end
    // need to re-calculate delimiters because MPDU size will change
    create_delimiter();
    add_padding();
    create_blank_delimiter();
  endfunction : encrypt

  //---------------------------------------------
  // decrypt MPDU frame if protection is set
  //---------------------------------------------
  virtual function void decrypt(bit ra);
    // create security engine
    security_wrap = new();

    foreach (mpdu_frame[i]) begin
      // push down securuty object hadle
      mpdu_frame[i].security_wrap = security_wrap;
      mpdu_frame[i].decrypt(ra);
    end
  endfunction : decrypt

  //------------------------------------------------------------
  // custom copy function
  //------------------------------------------------------------
  function void do_copy(uvm_object rhs);
    AMPDU_frame rhs_;
    int         pad, i;

    if (!$cast(rhs_,rhs)) begin
      `uvm_fatal(get_type_name(), "do_copy cast failed!")
    end
    super.do_copy(rhs);

`ifdef STANDALONE_PHY
    ampdu_payload = new[rhs_.ampdu_payload.size()];
    foreach (rhs_.ampdu_payload[i]) ampdu_payload[i] = rhs_.ampdu_payload[i];
`else
    aggregated = rhs_.aggregated;
    ppdu_format = rhs_.ppdu_format;
    is_dsss = rhs_.is_dsss;
    tx_frame = rhs_.tx_frame;
    bar_attached = rhs_.bar_attached;
    no_valid_frames = rhs_.no_valid_frames;
    security_wrap = rhs_.security_wrap;

    mpdu_frame_type = new[rhs_.mpdu_frame_type.size()];
    foreach (rhs_.mpdu_frame_type[i]) mpdu_frame_type[i] = rhs_.mpdu_frame_type[i];

    create_mpdu_frames();
    foreach (rhs_.mpdu_frame[i]) mpdu_frame[i].copy(rhs_.mpdu_frame[i]);

    delimit = new[rhs_.delimit.size()];
    foreach (rhs_.delimit[i]) begin
      delimit[i] = new();
      delimit[i].copy(rhs_.delimit[i]);
    end

    if (rhs_.padding.first(pad)) do
      padding[pad] = rhs_.padding[pad];
    while (rhs_.padding.next(pad));

    if (rhs_.blank_delimit.first(i)) do begin
      blank_delimit[i] = new();
      blank_delimit[i].copy(rhs_.blank_delimit[i]);
    end while (rhs_.blank_delimit.next(i));

`endif//STANDALONE_PHY
  endfunction : do_copy


endclass : AMPDU_frame

`endif //AMPDU_FRAME_SV
