//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_HEADER_SV
`define MAC_HEADER_SV

class MAC_header extends uvm_object;

  rand frame_control_s        frame_ctrl;
  rand duration_t             duration_id;
  rand mac_address_t          addr[];
  rand sequence_control_s     seq_ctrl[];
  rand qos_control_s          qos_ctrl[];
  rand ht_control_s           ht_ctrl[];

  `uvm_object_utils_begin(MAC_header)
    `uvm_field_int      (frame_ctrl , UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int      (duration_id, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(addr       , UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(seq_ctrl   , UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(qos_ctrl   , UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(ht_ctrl    , UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
  `uvm_object_utils_end

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_mac_addr_num {
    addr.size() >= 1;
    addr.size() <= 4;
  }

  constraint c_duration_id {
    soft duration_id[15] == 0;
    soft duration_id < 16'd5000;
  }

  constraint c_qos_ctrl_present {
    qos_ctrl.size() inside {[0:1]};
  }

  constraint c_ht_ctrl_present {
    ht_ctrl.size() inside {[0:1]};
  }

  constraint c_seq_ctrl_present {
    seq_ctrl.size() inside {[0:1]};
  }

  constraint c_protocol_version {
    frame_ctrl.protocol_version_f == 2'b00;
  }
  //---------------------------------------------
  // end of constraints
  //---------------------------------------------

  function new (string name = "MAC_header");
    super.new(name);
  endfunction

  function void post_randomize();
  endfunction : post_randomize

  //------------------------------------------------------------
  // return size of MAC header in bytes
  //------------------------------------------------------------
  function int size();
    return (
        $bits(frame_control_s)
      + $bits(duration_t)
      + (addr.size() * $bits(mac_address_t))
      + (seq_ctrl.size() * $bits(sequence_control_s))
      + (qos_ctrl.size() * $bits(qos_control_s))
      + (ht_ctrl.size() * $bits(ht_control_s))
    ) / 8;
  endfunction : size

  //------------------------------------------------------------
  // put MAC header data bytes to array
  //------------------------------------------------------------
  function void put2array(ref int index, ref octet_t data[]);

    // store content inside array from MAC header
    data[index++] = {frame_ctrl[ 7:0]};
    data[index++] = {frame_ctrl[15:8]};
    data[index++] = {duration_id[ 7:0]};
    data[index++] = {duration_id[15:8]};
    for (int i=0;i<`MAC_ADDR_SIZE(addr.size());i++) begin
      // MAC address is contained of 6 ostets
      data[index++] = {addr[i][ 7: 0]};
      data[index++] = {addr[i][15: 8]};
      data[index++] = {addr[i][23:16]};
      data[index++] = {addr[i][31:24]};
      data[index++] = {addr[i][39:32]};
      data[index++] = {addr[i][47:40]};
    end//for
    for (int i=0;i<seq_ctrl.size();i++) begin
      // MAC seq_ctrl is contained of 2 ostets
      data[index++] = {seq_ctrl[i][ 7:0]};
      data[index++] = {seq_ctrl[i][15:8]};
    end//for
    // 4th MAC address can only apear after sequence control
    if (addr.size() == 4) begin
      // MAC address is contained of 6 ostets
      data[index++] = {addr[3][ 7: 0]};
      data[index++] = {addr[3][15: 8]};
      data[index++] = {addr[3][23:16]};
      data[index++] = {addr[3][31:24]};
      data[index++] = {addr[3][39:32]};
      data[index++] = {addr[3][47:40]};
    end
    for (int i=0;i<qos_ctrl.size();i++) begin
      // MAC qos_ctrl is contained of 2 ostets
      data[index++] = {qos_ctrl[i][ 7:0]};
      data[index++] = {qos_ctrl[i][15:8]};
    end//for
    for (int i=0;i<ht_ctrl.size();i++) begin
      // MAC ht_ctrl is contained of 2 ostets
      data[index++] = {ht_ctrl[i][ 7: 0]};
      data[index++] = {ht_ctrl[i][15: 8]};
      data[index++] = {ht_ctrl[i][23:16]};
      data[index++] = {ht_ctrl[i][31:24]};
    end//for

  endfunction : put2array

  //------------------------------------------------------------
  // custom print function
  //------------------------------------------------------------
  function void do_print(uvm_printer printer);
    ht_control_s   ht;
    vht_control_s  vht;
    he_control_s   he;

    super.do_print(printer);

    printer.print_string("frame control", $sformatf("%p",frame_ctrl));
    printer.print_int("duration", duration_id, $bits(duration_t), UVM_DEC);
    printer.print_string("address", $sformatf("%p",addr));
    if (seq_ctrl.size())
      printer.print_string("sequence control", $sformatf("%p",seq_ctrl));
    if (qos_ctrl.size())
      printer.print_string("QoS control", $sformatf("%p",qos_ctrl));
    if (ht_ctrl.size()) begin
      // print HT control structure
      if (ht_ctrl[0][0] == 0) begin
        ht = ht_ctrl[0];
        printer.print_string("HT control", $sformatf("%p",ht));
      end
      // print VHT control structure
      else if (ht_ctrl[0][1] == 0) begin
        vht = ht_ctrl[0];
        printer.print_string("VHT control", $sformatf("%p",vht));
      end
      // print HE control structure
      else begin
        he = ht_ctrl[0];
        printer.print_string("HE control", $sformatf("%p",he));
      end
    end
  endfunction : do_print


  //------------------------------------------------------------
  // custom compare function
  //------------------------------------------------------------
  virtual function bit do_compare(uvm_object rhs, uvm_comparer comparer);
    MAC_header that;

    if (!$cast(that, rhs)) begin
      `uvm_fatal(get_type_name(), $sformatf("Failed cast from 'rhs' to 'that'"))
    end

    do_compare = super.do_compare(rhs, comparer);
    do_compare &= comparer.compare_field("frame_ctrl", this.frame_ctrl, that.frame_ctrl, $bits(this.frame_ctrl));
    do_compare &= comparer.compare_field("duration_id", this.duration_id, that.duration_id, $bits(this.duration_id));
    foreach(addr[i]) begin
      do_compare &= comparer.compare_field($sformatf("addr[%0d]",i), this.addr[i], that.addr[i], $bits(this.addr[i]));
    end
    foreach(seq_ctrl[i]) begin
      do_compare &= comparer.compare_field($sformatf("seq_ctrl[%0d]",i), this.seq_ctrl[i], that.seq_ctrl[i], $bits(this.seq_ctrl[i]));
    end
    foreach(qos_ctrl[i]) begin
      do_compare &= comparer.compare_field($sformatf("qos_ctrl[%0d]",i), this.qos_ctrl[i], that.qos_ctrl[i], $bits(this.qos_ctrl[i]));
    end
    foreach(ht_ctrl[i]) begin
      do_compare &= comparer.compare_field($sformatf("ht_ctrl[%0d]",i), this.ht_ctrl[i], that.ht_ctrl[i], $bits(this.ht_ctrl[i]));
    end
    return do_compare;
  endfunction : do_compare


  //------------------------------------------------------------
  // custom copy function
  //------------------------------------------------------------
  function void do_copy(uvm_object rhs);
    MAC_header rhs_;

    if (!$cast(rhs_,rhs)) begin
      `uvm_fatal(get_type_name(), "do_copy cast failed!")
    end
    super.do_copy(rhs);

    frame_ctrl = rhs_.frame_ctrl;
    duration_id = rhs_.duration_id;
    addr = new[rhs_.addr.size()];
    foreach (rhs_.addr[i]) addr[i] = rhs_.addr[i];
    seq_ctrl = new[rhs_.seq_ctrl.size()];
    foreach (rhs_.seq_ctrl[i]) seq_ctrl[i] = rhs_.seq_ctrl[i];
    qos_ctrl = new[rhs_.qos_ctrl.size()];
    foreach (rhs_.qos_ctrl[i]) qos_ctrl[i] = rhs_.qos_ctrl[i];
    ht_ctrl = new[rhs_.ht_ctrl.size()];
    foreach (rhs_.ht_ctrl[i]) ht_ctrl[i] = rhs_.ht_ctrl[i];
  endfunction : do_copy

endclass : MAC_header

`endif// MAC_HEADER_SV
