//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MPDU_FRAME_SV
`define MPDU_FRAME_SV

class MPDU_frame extends uvm_object;

  rand MAC_header         MAC_header;
  rand octet_t            frame_body[];
       FCS_t              FCS; // checksum is calculated in post_randomize phase

  // content of MAC frame merged into array for FCS calculations
  octet_t                 frame[];
  octet_t                 enc_frame_body[];
  FCS_t                   eFCS; // checksum for encrypted frame
  int                     frame_size;
  security_wrapper_engine security_wrap; // handle to security engine
  // when generating frame with error
  bit                     fcs_error;


  `uvm_object_utils_begin(MPDU_frame)
    `uvm_field_object   (MAC_header, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_array_int(frame_body, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int      (FCS, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_array_int(enc_frame_body, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int      (eFCS, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int      (fcs_error, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
  `uvm_object_utils_end

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_frame_body_size {
    soft frame_body.size() >= 0;
    soft frame_body.size() <= `MAX_MPDU_SIZE-`MAX_MAC_HEADER_SIZE;
  }

  constraint c_MAC_header_frame_ctrl {
  /* left empty (placeholder)
   * it will be overriden in child class with specific
   * constraint values for frame control
   * */
   soft MAC_header.frame_ctrl.protected_frame_f dist {1'b0:/ 20, 1'b1:/80};
  }
  //---------------------------------------------
  // end of constraints
  //---------------------------------------------

  function new (string name = "MPDU_frame");
    super.new(name);

    MAC_header = frame_model_pkg::MAC_header::type_id::create("MAC_header");
  endfunction

  //------------------------------------------------------------
  // post randomize function is used for calculating FCS
  // so it needs to be called when frame is generated (all fields)
  //------------------------------------------------------------
  function void post_randomize();
    custom_post_randomize();
    prepare_frame_array();
    FCS = fcs_calc_func(frame_size, frame);
    // put FCS data to frame array, extend frame size by 4 bytes
    // for FCS and append rest of frame bytes to it
    frame = new[frame_size+4](frame);
    frame[frame_size  ] = FCS[ 7: 0];
    frame[frame_size+1] = FCS[15: 8];
    frame[frame_size+2] = FCS[23:16];
    frame[frame_size+3] = FCS[31:24];
    frame_size += 4;
  endfunction : post_randomize

  //------------------------------------------------------------
  // post monitor function is used for recreating frame and frame_size
  // from collected data in monitor. Must be called in monitor if frame
  // and MPDU_frame.size() is to be used on collected data
  //------------------------------------------------------------
  virtual function void post_monitor();
    prepare_frame_array();
    // copy FCS data to frame array, extend frame size by 4 bytes
    // for FCS and append rest of frame bytes to it
    frame = new[frame_size+4](frame);
    frame[frame_size  ] = FCS[ 7: 0];
    frame[frame_size+1] = FCS[15: 8];
    frame[frame_size+2] = FCS[23:16];
    frame[frame_size+3] = FCS[31:24];
    frame_size += 4;
  endfunction : post_monitor

  //------------------------------------------------------------
  // re-calculate FCS on frame, this function is used when some
  // fields are changed after randomization is done, so FCS has changed
  //------------------------------------------------------------
  virtual function void recalculate_fcs();
    // calculate eFCS
    if (MAC_header.frame_ctrl.protected_frame_f) begin
      prepare_frame_array(1);
      eFCS = fcs_calc_func(frame_size, frame);
    end
    else begin
      eFCS = 0;
    end
    prepare_frame_array();
    FCS = fcs_calc_func(frame_size, frame);
    // put FCS data to frame array, extend frame size by 4 bytes
    // for FCS and append rest of frame bytes to it
    frame = new[frame_size+4](frame);
    frame[frame_size  ] = FCS[ 7: 0];
    frame[frame_size+1] = FCS[15: 8];
    frame[frame_size+2] = FCS[23:16];
    frame[frame_size+3] = FCS[31:24];
    frame_size += 4;
  endfunction : recalculate_fcs

  //------------------------------------------------------------
  // returns MPDU frame size in bytes
  //------------------------------------------------------------
  virtual function int size();
    `uvm_info(get_type_name(), $sformatf("MPDU frame size is %d",frame_size), UVM_FULL)
    return frame_size;
  endfunction : size

  //------------------------------------------------------------
  // this function should be used when derived classes need some
  // post randomization setup and calculations
  //------------------------------------------------------------
  virtual function void custom_post_randomize();
  endfunction : custom_post_randomize

  //------------------------------------------------------------
  // function used for storing MAC frame content to one array
  // then FCS calculation can be done with it
  //------------------------------------------------------------
  protected function void prepare_frame_array(bit encrypted = 0);
    int index = 0;

    frame.delete(); // free memory
    // determine size of MPDU frame
    if (encrypted)
      frame_size = MAC_header.size() + enc_frame_body.size();
    else
      frame_size = MAC_header.size() + frame_body.size();
    // create frame array
    frame = new[frame_size];
    MAC_header.put2array(index, frame);

    if (encrypted) begin
      // store encrypted frame body data
      foreach (enc_frame_body[i]) frame[index++] = enc_frame_body[i];
    end
    else begin
      // store frame body data
      foreach (frame_body[i]) frame[index++] = frame_body[i];
    end
  endfunction : prepare_frame_array

  //------------------------------------------------------------
  // 1st function for setting valid MAC address
  // @addr - value of MAC address to be set
  // @id   - value of address position in MAC frame,
  //         id = 1 means address 1
  //         ...
  //         id = 4 means address 4
  //************************************************************
  // 2nd function for setting MAC addresses
  // @RA - receiver address
  // @TA - transmitter address
  // @DA - destination address
  // @SA - source address
  // @BSSID - basic service set indetifier
  // @amsdu - indication if A-MSDU
  //------------------------------------------------------------
  virtual function void set_MAC_addr_by_idx(mac_address_t addr, int id);
    if (id > MAC_header.addr.size())
      `uvm_error(get_type_name(),
      $sformatf("ID of MAC address position is not valid, id=%0d, MAC addr size=%0d",id, MAC_header.addr.size()))
    else
      MAC_header.addr[id-1] = addr;
  endfunction : set_MAC_addr_by_idx

  virtual function void set_MAC_addr(mac_address_t RA = 0, mac_address_t TA = 0,
                                     mac_address_t DA = 0, mac_address_t SA = 0,
                                     mac_address_t BSSID = 0, bit amsdu = 0);

     // for conrol frames set addresses
     if (MAC_header.frame_ctrl.type_f == `CONTROL_MPDU) begin
       case (MAC_header.frame_ctrl.subtype_f)
         `CTS,`ACK,`CONTROL_WRAPPER: begin
           MAC_header.addr[0] = (RA == 0) ? MAC_header.addr[0] : RA;
         end
         default: begin
           MAC_header.addr[0] = (RA == 0) ? MAC_header.addr[0] : RA;
           MAC_header.addr[1] = (TA == 0) ? MAC_header.addr[1] : TA;
         end
       endcase
     end
     // for management and data addresses
     else begin
       case ({MAC_header.frame_ctrl.to_DS_f,MAC_header.frame_ctrl.from_DS_f})
         2'b00: begin
           MAC_header.addr[0] = (RA == 0) ? MAC_header.addr[0] : RA;
           MAC_header.addr[1] = (TA == 0) ? MAC_header.addr[1] : TA;
           MAC_header.addr[2] = (BSSID == 0) ? MAC_header.addr[2] : BSSID;
         end
         2'b01: begin
           MAC_header.addr[0] = (RA == 0) ? MAC_header.addr[0] : RA;
           MAC_header.addr[1] = (TA == 0) ? MAC_header.addr[1] : TA;
           MAC_header.addr[2] = (amsdu)
                                ? ((BSSID == 0) ? MAC_header.addr[2] : BSSID)
                                : ((SA == 0) ? MAC_header.addr[2] : SA);
         end
         2'b10: begin
           MAC_header.addr[0] = (RA == 0) ? MAC_header.addr[0] : RA;
           MAC_header.addr[1] = (TA == 0) ? MAC_header.addr[1] : TA;
           MAC_header.addr[2] = (amsdu)
                                ? ((BSSID == 0) ? MAC_header.addr[2] : BSSID)
                                : ((DA == 0) ? MAC_header.addr[2] : DA);
         end
         2'b11: begin
           MAC_header.addr[0] = (RA == 0) ? MAC_header.addr[0] : RA;
           MAC_header.addr[1] = (TA == 0) ? MAC_header.addr[1] : TA;
           MAC_header.addr[2] = (amsdu)
                                ? ((BSSID == 0) ? MAC_header.addr[2] : BSSID)
                                : ((DA == 0) ? MAC_header.addr[2] : DA);
           MAC_header.addr[3] = (amsdu)
                                ? ((BSSID == 0) ? MAC_header.addr[3] : BSSID)
                                : ((SA == 0) ? MAC_header.addr[3] : SA);
         end
       endcase
     end
  endfunction : set_MAC_addr

  //------------------------------------------------------------
  // get MAC address, RA or TA
  // @ RA - 1 return RA address, 0 return TA address
  //------------------------------------------------------------
  virtual function mac_address_t get_MAC_addr(bit RA = 0);
    return (RA) ? MAC_header.addr[0] : MAC_header.addr[1];
  endfunction : get_MAC_addr

  //------------------------------------------------------------
  // function for setup of duration/ID field
  //------------------------------------------------------------
  virtual function void set_MAC_duration(duration_t dur);
    MAC_header.duration_id = dur;
  endfunction : set_MAC_duration

  //------------------------------------------------------------
  // get function of QoS control ACK policy field
  //------------------------------------------------------------
  virtual function bit [1:0] get_qos_ack_policy();
    return MAC_header.qos_ctrl[0].ack_policy_f;
  endfunction : get_qos_ack_policy

  //------------------------------------------------------------
  // function for setup of QoS control filed ACK policy field
  //------------------------------------------------------------
  virtual function void set_qos_ack_policy(bit [1:0] ack_pol);
    if (MAC_header.qos_ctrl.size())
      MAC_header.qos_ctrl[0].ack_policy_f = ack_pol;
  endfunction : set_qos_ack_policy

  //------------------------------------------------------------
  // function set HT control
  //------------------------------------------------------------
  virtual function void set_ht_control(ht_control_s htctrl);
    if (MAC_header.ht_ctrl.size())
      MAC_header.ht_ctrl[0] = htctrl;
  endfunction : set_ht_control

  //------------------------------------------------------------
  // function get HT control
  //------------------------------------------------------------
  virtual function ht_control_s get_ht_control();
    return (MAC_header.ht_ctrl.size()) ? MAC_header.ht_ctrl[0] : 0;
  endfunction : get_ht_control

  //------------------------------------------------------------
  // @ret - 1 - frame is protected
  //        0 - frame not protected
  //------------------------------------------------------------
  function bit is_protected();
    return MAC_header.frame_ctrl.protected_frame_f;
  endfunction : is_protected

  //------------------------------------------------------------
  // custom print function
  //------------------------------------------------------------
  function void do_print(uvm_printer printer);
    super.do_print(printer);

  endfunction : do_print

  //------------------------------------------------------------
  // custom compare function
  //------------------------------------------------------------
  virtual function bit do_compare(uvm_object rhs, uvm_comparer comparer);
    MPDU_frame that;

    if (!$cast(that, rhs)) begin
      `uvm_fatal(get_type_name(), $sformatf("Failed cast from 'rhs' to 'that'"))
    end

    do_compare = super.do_compare(rhs, comparer);
    do_compare &= comparer.compare_field("FCS", this.FCS, that.FCS, $bits(this.FCS));
    do_compare &= comparer.compare_object("MAC_header", this.MAC_header, that.MAC_header);
    foreach (frame_body[i]) begin
      //$display("frame_body[%0d] 'h%2h 'h%2h",i,this.frame_body[i], that.frame_body[i]);
      do_compare &= comparer.compare_field($sformatf("frame_body[%0d]",i), this.frame_body[i], that.frame_body[i], $bits(this.frame_body[i]));
    end
    foreach (enc_frame_body[i]) begin
      //$display("enc_frame_body[%0d] 'h%2h 'h%2h",i,this.enc_frame_body[i], that.enc_frame_body[i]);
      do_compare &= comparer.compare_field($sformatf("enc_frame_body[%0d]",i), this.enc_frame_body[i], that.enc_frame_body[i], $bits(this.enc_frame_body[i]));
    end
    do_compare &= comparer.compare_field("eFCS", this.eFCS, that.eFCS, $bits(this.eFCS));

    return do_compare;
  endfunction : do_compare

  //------------------------------------------------------------
  // custom copy function
  //------------------------------------------------------------
  virtual function void do_copy(uvm_object rhs);
    MPDU_frame rhs_;

    if (!$cast(rhs_,rhs)) begin
      `uvm_fatal(get_type_name(), "do_copy cast failed!")
    end
    super.do_copy(rhs);

    MAC_header.copy(rhs_.MAC_header);
    frame_body = new[rhs_.frame_body.size()];
    foreach (rhs_.frame_body[i]) frame_body[i] = rhs_.frame_body[i];
    FCS = rhs_.FCS;

    enc_frame_body = new[rhs_.enc_frame_body.size()];
    foreach (rhs_.enc_frame_body[i]) enc_frame_body[i] = rhs_.enc_frame_body[i];
    eFCS = rhs_.eFCS;

    frame_size = rhs_.frame_size;
    frame = new[rhs_.frame.size()];
    foreach (rhs_.frame[i]) frame[i] = rhs_.frame[i];

    fcs_error = rhs_.fcs_error;
  endfunction : do_copy

  //------------------------------------------------------------
  // encrypt frame
  //------------------------------------------------------------
  virtual function void encrypt(bit ra);
    if (MAC_header.frame_ctrl.protected_frame_f) begin
      // execute encryption
      security_wrap.encrypt_frame(ra, MAC_header, frame_body, enc_frame_body);

      // frame can be encrypted only if security is not NULL KEY
      if (!(security_wrap.security_type inside {NULL_KEY,NO_SECURITY})) begin
        // calculate FCS of plain data frame w/o encryption
        recalculate_fcs();
      end
      else if (security_wrap.security_type == NO_SECURITY) begin
        // when NO_SECURITY is set frame can't be protected
        MAC_header.frame_ctrl.protected_frame_f = 0;
        // calculate FCS of plain data frame w/o encryption
        recalculate_fcs();
      end
    end
  endfunction : encrypt

  //------------------------------------------------------------
  // decrypt frame
  //------------------------------------------------------------
  virtual function void decrypt(bit ra);
    if (MAC_header.frame_ctrl.protected_frame_f) begin
      // when we need to decrypt received frame that means that
      // monitor collected and stored data in frame body, so this
      // represents encrypted frame that needs to be decrypted
      enc_frame_body = new[frame_body.size()](frame_body);
      // execute decryption, frame_body will become decrypted frame now
      security_wrap.decrypt_frame(ra, MAC_header, frame_body, enc_frame_body);
      recalculate_fcs();
    end
  endfunction : decrypt

  //---------------------------------------------
  // prepare frame data for Rx when encryption is
  // set in MPDU frame. Function will prepare data
  // that is encrypted for Rx path
  //---------------------------------------------
  virtual function void setup_rx_encrypted_data();
    encrypt(0); // first execute encryption

    if (MAC_header.frame_ctrl.protected_frame_f) begin
      // put FCS data to frame array, extend frame size by 4 bytes
      // for FCS and append rest of frame bytes to it
      if (security_wrap.security_type != NO_SECURITY) begin
        prepare_frame_array(1);
        frame = new[frame_size+4](frame);
        frame[frame_size  ] = eFCS[ 7: 0];
        frame[frame_size+1] = eFCS[15: 8];
        frame[frame_size+2] = eFCS[23:16];
        frame[frame_size+3] = eFCS[31:24];
        frame_size += 4;
      end
    end
  endfunction : setup_rx_encrypted_data

endclass : MPDU_frame

`endif// MPDU_FRAME_SV
