//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : container class for PPDU preamble
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef MU_MIMO_PPDU_FRAME_SV
`define MU_MIMO_PPDU_FRAME_SV

class MU_MIMO_PPDU_frame extends uvm_object;

  rand int                    num_of_users;
  rand format_mod_e           ppdu_format;
       PPDU_preamble_header   preamble_header;
  rand AMPDU_frame            ampdu_frame[];
  rand bit                    tx_frame;  // indication that frame is for TX
  // for MU-MIMO frames longest frame is sent first
       int                    user_order[`MAX_USER_NUM];

  `uvm_object_utils_begin(MU_MIMO_PPDU_frame)
    `uvm_field_enum(format_mod_e, ppdu_format, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(num_of_users, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_object(preamble_header, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_array_object(ampdu_frame, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(tx_frame, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
  `uvm_object_utils_end

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  // maximum number of users is 4
  constraint c_user_num {
    num_of_users >= 0;
    num_of_users <= `MAX_USER_NUM;
    ampdu_frame.size() == num_of_users;
  }

  function new (string name = "MU_MIMO_PPDU_frame");
    super.new(name);

    // initialize user order like
    for (int i=0; i<`MAX_USER_NUM; i++)
      user_order[i] = i;
  endfunction

  //---------------------------------------------
  // return total number of bytes in frame
  //---------------------------------------------
  function int size();
    int accu;

    accu = 0;
    foreach (ampdu_frame[i])
      accu += ampdu_frame[i].size();

    return accu;
  endfunction : size

  //---------------------------------------------
  // create A-MPDU frame
  //---------------------------------------------
  function void post_randomize();
    foreach (ampdu_frame[i]) begin
      ampdu_frame[i] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[%d]",i));
    end
    create_preamble_header();
  endfunction : post_randomize

  //---------------------------------------------
  // calculate legacy length and HT length for
  // header and preamble
  //---------------------------------------------
  virtual function void calc_leg_ht_length();
    int       user;
    int       Nsymb_int;
    bit [6:0] mcs;
    int       txtime;
    real      tSymb;
    real      tPe;
    real      mid_calc;
    HESIGB_s  HESIGB;

    // when NDP frame is created there will be no payload
    if (num_of_users == 0) return;

    case (ppdu_format)
      NON_HT: preamble_header.leg_length = ampdu_frame[0].size();
      NON_HT_DUP_OFDM: preamble_header.leg_length = ampdu_frame[0].size();
      HT_MF: begin
        preamble_header.user_header[0].ht_length_f = ampdu_frame[0].size();
        // calculate duration of frame
        preamble_header.leg_length = $ceil(((ComputeTimeOnAirAC (
                                               .length_i     (preamble_header.user_header[0].ht_length_f),
                                               .preType_i    (preamble_header.format_mod),
                                               .mcsIndex_i   (preamble_header.user_header[0].mcs_f),
                                               .shortGI_i    (preamble_header.gi_type[0]),
                                               .chBW_i       (preamble_header.ch_bw),
                                               .stbc_i       (preamble_header.stbc),
                                               .extNSS_i     (preamble_header.num_extn_ss),
                                               .woPreamble_i (0),
                                               .band5G_i     (preamble_header.band5G),
                                               .multiUser_i  (0),
                                               .nSTSTotal_i  (preamble_header.user_header[0].num_sts_f),
                                               .debug_i      (0)
                                                 /*     signal extension   */
                                           ) - ((preamble_header.band5G) ? 0:6))-20) / 4.0) * 3 - 3;
      end
      HT_GF: begin
        preamble_header.leg_length = 0;
        preamble_header.user_header[0].ht_length_f = ampdu_frame[0].size();
      end
      VHT: begin
        // store HT length per user header
        user = calc_longest_user_dur();
        foreach (preamble_header.user_header[i]) begin
          preamble_header.user_header[i].ht_length_f = ampdu_frame[i].size();
        end
        if (preamble_header.stbc == 0 ) begin
          mcs = {preamble_header.user_header[user].num_sts_f,
                 preamble_header.user_header[user].mcs_f[3:0]};
        end
        else begin
          mcs = {((preamble_header.user_header[user].num_sts_f + 1)/2) - 1,
                   preamble_header.user_header[user].mcs_f[3:0]};
        end
        // calculate duration of frame
        preamble_header.leg_length = $ceil((ComputeTimeOnAirAC (
                                         .length_i     (preamble_header.user_header[user].ht_length_f),
                                         .preType_i    (preamble_header.format_mod),
                                         .mcsIndex_i   (mcs),
                                         .shortGI_i    (preamble_header.gi_type[0]),
                                         .chBW_i       (preamble_header.ch_bw),
                                         .stbc_i       (preamble_header.stbc),
                                         .extNSS_i     (preamble_header.num_extn_ss),
                                         .woPreamble_i (0),
                                         .band5G_i     (preamble_header.band5G),
                                         .multiUser_i  ((num_of_users > 1) ? 1 : 0),
                                         .nSTSTotal_i  (preamble_header.user_header[user].num_sts_f),
                                         .debug_i      (0)
                                        ) - 20) / 4.0) * 3 - 3;

      end
      HE_SU: begin
        preamble_header.user_header_he[0].he_length_f = ampdu_frame[0].size();
        // calculate duration of frame
        preamble_header.leg_length = $ceil((ComputeTimeOnAirAX (
                                         .length_i          (preamble_header.user_header_he[0].he_length_f),
                                         .preType_i         (preamble_header.format_mod),
                                         .mcsIndex_i        (preamble_header.user_header_he[0].mcs_f),
                                         .giType_i          (preamble_header.gi_type),
                                         .ruType_i          (get_ru_type_he_su(preamble_header.ch_bw)),
                                         .heLtfType_i       (preamble_header.he_ltf_type),
                                         .numHeLtf_i        (preamble_header.num_he_ltf),
                                         .dcm_i             (preamble_header.dcm),
                                         .packetExtension_i (preamble_header.user_header_he[0].pkt_ext_f),
                                         .heTbLength_i      (0),
                                         .triggerMethod_i   (0),
                                         .doppler_i         (preamble_header.doppler),
                                         .mma_i             (preamble_header.midamble_periodicity),
                                         .stbc_i            (preamble_header.stbc),
                                         .woPreamble_i      (0),
                                         .heSigB_i          (HESIGB),
                                         .debug_i           (0)
                                         ) -20) / 4.0) * 3 - 3 - 2;  // the last parameter is m = 2 for HE_SU
      end
      HE_MU: begin
        foreach (preamble_header.user_header_he[i]) begin
          preamble_header.user_header_he[i].he_length_f = ampdu_frame[i].size();
        end

        user = calc_longest_user_dur();
        // store HESIGB struture fields
        HESIGB.mcs             = preamble_header.mcs_sig_b;
        HESIGB.dcm             = preamble_header.dcm_sig_b;
        HESIGB.compressed_mode = preamble_header.sig_b_comp_mode;
        HESIGB.ru_allocation   = preamble_header.ru_allocation;

        preamble_header.leg_length = $ceil((ComputeTimeOnAirAX (
                                         .length_i          (preamble_header.user_header_he[user].he_length_f),
                                         .preType_i         (preamble_header.format_mod),
                                         .mcsIndex_i        (preamble_header.user_header_he[user].mcs_f),
                                         .giType_i          (preamble_header.gi_type),
                                         .ruType_i          (get_ru_type_from_ru_allocation(preamble_header.ru_allocation,
                                                                                            preamble_header.user_header_he[user].dut_location_f)),
                                         .heLtfType_i       (preamble_header.he_ltf_type),
                                         .numHeLtf_i        (preamble_header.num_he_ltf),
                                         .dcm_i             (preamble_header.dcm),
                                         .packetExtension_i (preamble_header.user_header_he[user].pkt_ext_f),
                                         .heTbLength_i      (0),
                                         .triggerMethod_i   (0),
                                         .doppler_i         (preamble_header.doppler),
                                         .mma_i             (preamble_header.midamble_periodicity),
                                         .stbc_i            (preamble_header.stbc),
                                         .woPreamble_i      (0),
                                         .heSigB_i          (HESIGB),
                                         .debug_i           (0)
                                         ) -20) / 4.0) * 3 - 3 - 1;  // the last parameter is m = 1 for HE_MU;
      end
      HE_EXT_SU: begin
        preamble_header.user_header_he[0].he_length_f = ampdu_frame[0].size();
        // calculate duration of frame
        preamble_header.leg_length = $ceil((ComputeTimeOnAirAX (
                                         .length_i          (preamble_header.user_header_he[0].he_length_f),
                                         .preType_i         (preamble_header.format_mod),
                                         .mcsIndex_i        (preamble_header.user_header_he[0].mcs_f),
                                         .giType_i          (preamble_header.gi_type),
                                         .ruType_i          (get_ru_type_he_su(preamble_header.ch_bw)),
                                         .heLtfType_i       (preamble_header.he_ltf_type),
                                         .numHeLtf_i        (preamble_header.num_he_ltf),
                                         .dcm_i             (preamble_header.dcm),
                                         .packetExtension_i (preamble_header.user_header_he[0].pkt_ext_f),
                                         .heTbLength_i      (0),
                                         .triggerMethod_i   (0),
                                         .doppler_i         (preamble_header.doppler),
                                         .mma_i             (preamble_header.midamble_periodicity),
                                         .stbc_i            (preamble_header.stbc),
                                         .woPreamble_i      (0),
                                         .heSigB_i          (HESIGB),
                                         .debug_i           (0)
                                         ) -20) / 4.0) * 3 - 3 - 1;  // the last parameter is m = 1 for HE_EXT_SU
      end
      HE_TB: begin
        preamble_header.user_header_he[0].he_length_f = ampdu_frame[0].size();
      end
    endcase
  endfunction : calc_leg_ht_length

  //---------------------------------------------
  // determine which user has longest time on air
  // in MU-MIMO frames
  //---------------------------------------------
  virtual function int calc_longest_user_dur();
    int longest_dur = 0;
    int time_on_air = 0;
    int index = -1;
    bit [6:0] mcs;
    HESIGB_s  HESIGB;

    // initialize user order like
    for (int i=0; i<`MAX_USER_NUM; i++)
      user_order[i] = i;

    if (preamble_header.stbc == 0 ) begin
      mcs = {preamble_header.user_header[0].num_sts_f,
      preamble_header.user_header[0].mcs_f[3:0]};
    end
    else begin
      mcs = {((preamble_header.user_header[0].num_sts_f + 1)/2) - 1,
      preamble_header.user_header[0].mcs_f[3:0]};
    end
    foreach (ampdu_frame[user]) begin
      if (ppdu_format == VHT) begin
        time_on_air = $ceil((ComputeTimeOnAirAC (
                                           .length_i     (preamble_header.user_header[user].ht_length_f),
                                           .preType_i    (preamble_header.format_mod),
                                           .mcsIndex_i   (mcs),
                                           .shortGI_i    (preamble_header.gi_type[0]),
                                           .chBW_i       (preamble_header.ch_bw),
                                           .stbc_i       (preamble_header.stbc),
                                           .extNSS_i     (preamble_header.num_extn_ss),
                                           .woPreamble_i (0),
                                           .band5G_i     (preamble_header.band5G),
                                           .multiUser_i  ((num_of_users > 1) ? 1 : 0),
                                           .nSTSTotal_i  (preamble_header.user_header[user].num_sts_f),
                                           .debug_i      (0)
                                          ) - 20) / 4.0) * 3 - 3;
      end
      else if (ppdu_format == HE_MU) begin
        // store HESIGB struture fields
        HESIGB.mcs             = preamble_header.mcs_sig_b;
        HESIGB.dcm             = preamble_header.dcm_sig_b;
        HESIGB.compressed_mode = preamble_header.sig_b_comp_mode;
        HESIGB.ru_allocation   = preamble_header.ru_allocation;

        time_on_air = $ceil((ComputeTimeOnAirAX (
                                         .length_i          (preamble_header.user_header_he[user].he_length_f),
                                         .preType_i         (preamble_header.format_mod),
                                         .mcsIndex_i        (preamble_header.user_header_he[user].mcs_f),
                                         .giType_i          (preamble_header.gi_type),
                                         .ruType_i          (get_ru_type_from_ru_allocation(preamble_header.ru_allocation,
                                                                                            preamble_header.user_header_he[user].dut_location_f)),
                                         .heLtfType_i       (preamble_header.he_ltf_type),
                                         .numHeLtf_i        (preamble_header.num_he_ltf),
                                         .dcm_i             (preamble_header.dcm),
                                         .packetExtension_i (preamble_header.user_header_he[user].pkt_ext_f),
                                         .heTbLength_i      (0),
                                         .triggerMethod_i   (0),
                                         .doppler_i         (preamble_header.doppler),
                                         .mma_i             (preamble_header.midamble_periodicity),
                                         .stbc_i            (preamble_header.stbc),
                                         .woPreamble_i      (0),
                                         .heSigB_i          (HESIGB),
                                         .debug_i           (0)
                                         ) -20) / 4.0) * 3 - 3 - 1;  // the last parameter is m = 1 for HE_MU
      end

      `uvm_info(get_type_name(), $sformatf("TimeOnAir for user %0d is %0d",user,time_on_air), UVM_DEBUG)
      if (time_on_air > longest_dur) begin
        longest_dur = time_on_air;
        index = user;
      end//if
    end//foreach

    // on primary channel user is with longest duration
    // just switch places with longest and first user;
    // Only in Tx switch user order
    if (tx_frame) begin
      user_order[0] = index;
      user_order[index] = 0;
    end
    preamble_header.user_order = user_order;
    `uvm_info(get_type_name(), $sformatf("User order %p",user_order), UVM_DEBUG)

    return index;
  endfunction : calc_longest_user_dur

  //---------------------------------------------
  // calculate maximum HT length of (V)HT PPDU frame
  // PL = (Nsymb * Ndbps - 22) / 8
  // Nsymb - number of symbols, max(TXTIME) = 5 [ms]
  //         symbol duration = 4/12.8 [us]
  // Ndbps - number of data bits per OFDM symbol
  //---------------------------------------------
  virtual function int max_payload_len(int user);
    int  num_of_bits_in_symbol;
    int  Nss;
    int  ret;
    real symb_dur;
    int  total_dur;
    int  num_mpdu;
    int  nsymb;
    int  length;
    real he_ltf_dur;
    int  nma;
    real midamble_dur;

    if (ppdu_format <= VHT) begin
      // get Nss
      Nss = get_num_ss_func(.txMCS   (preamble_header.user_header[user].mcs_f),
                            .txFormat(preamble_header.format_mod),
                            .txChBW  (preamble_header.ch_bw),
                            .txnSTS  (preamble_header.user_header[user].num_sts_f),
                            .txSTBC  (preamble_header.stbc));
      // get number of bits pre symbol from truthtable
      num_of_bits_in_symbol = get_bits_in_one_symbol_func(.txNumSSFn (Nss),
                                                          .txFormatFn(preamble_header.format_mod),
                                                          .txLRateFn (preamble_header.leg_rate),
                                                          .txChBWFn  (preamble_header.ch_bw),
                                                          .txMCSFn   (preamble_header.user_header[user].mcs_f)
                                                         );
      symb_dur = 4;
      total_dur = 5000; //5ms
    end
    else if (ppdu_format inside {HE_SU,HE_EXT_SU})begin //HE frames

      // get number of bits pre symbol from truthtable
      num_of_bits_in_symbol = get_bits_in_one_symbol_func(.txNumSSFn (preamble_header.user_header_he[user].nss_f),
                                                          .txFormatFn(preamble_header.format_mod),
                                                          .txLRateFn (preamble_header.leg_rate),
                                                          .txChBWFn  (preamble_header.ch_bw),
                                                          .txMCSFn   (preamble_header.user_header_he[user].mcs_f),
                                                          .txDCMFn   (preamble_header.dcm),
                                                          .txRUType  (get_ru_type_he_su(preamble_header.ch_bw))
                                                         );

      symb_dur = 12.8 + (gi_type_to_real(preamble_header.gi_type));
      // maximum HELTF = 64us and PE = 16us.
      total_dur = 5000 - 80; //5ms
      total_dur -= (ppdu_format == HE_SU) ? `HE_SU_PRE_DUR : `HE_ER_SU_PRE_DUR;
      // if doppler present subtract midamble time on air
      if (preamble_header.doppler == 1'b1) begin
        length = $ceil((total_dur / symb_dur) * num_of_bits_in_symbol - 22);
        //             LENGTH + TAIL + NSERVICE
        nsymb = $ceil((length +  6   +   16) / real'(num_of_bits_in_symbol));
        he_ltf_dur = (preamble_header.num_he_ltf+1) * (2**preamble_header.he_ltf_type * 3.2);
        nma = $ceil((nsymb-1)/real'(2**preamble_header.midamble_periodicity*10))-1;
        nma = (nma < 0) ? 0 : nma;
        midamble_dur = nma * he_ltf_dur;
        total_dur -= midamble_dur;

        `uvm_info(get_type_name(),$sformatf("Nsym = %0d, NMA = %0d, midamble_dur = %0f, total_dur = %0d",
          nsymb,nma,midamble_dur,total_dur),UVM_DEBUG)
      end
    end
    else if (ppdu_format == HE_MU) begin
      // get number of bits pre symbol from truthtable
      num_of_bits_in_symbol = get_bits_in_one_symbol_func(.txNumSSFn (preamble_header.user_header_he[user].nss_f),
                                                          .txFormatFn(preamble_header.format_mod),
                                                          .txLRateFn (preamble_header.leg_rate),
                                                          .txChBWFn  (preamble_header.ch_bw),
                                                          .txMCSFn   (preamble_header.user_header_he[user].mcs_f),
                                                          .txDCMFn   (preamble_header.dcm),
                                                          .txRUType  (get_ru_type_from_ru_allocation(preamble_header.ru_allocation,
                                                                                                     preamble_header.user_header_he[user].dut_location_f))
                                                         );

      symb_dur = 12.8 + (gi_type_to_real(preamble_header.gi_type));
      // subtract HESIG + HELTF + PE duration, to make sure we don't exceed
      // limitation. Value is calculated as maximum possible HESIGB = 64us,
      // HELTF = 64us and PE = 16us.
      total_dur = 5000 - `HE_MU_PRE_DUR - 144; //5ms
      // if doppler present subtract midamble time on air
      if (preamble_header.doppler == 1'b1) begin
        length = $ceil((total_dur / symb_dur) * num_of_bits_in_symbol - 22);
        //             LENGTH + TAIL + NSERVICE
        nsymb = $ceil((length +  6   +   16) / real'(num_of_bits_in_symbol));
        he_ltf_dur = (preamble_header.num_he_ltf+1) * (2**preamble_header.he_ltf_type * 3.2);
        nma = $ceil((nsymb-1)/real'(2**preamble_header.midamble_periodicity*10))-1;
        nma = (nma < 0) ? 0 : nma;
        midamble_dur = nma * he_ltf_dur;
        total_dur -= midamble_dur;

        `uvm_info(get_type_name(),$sformatf("Nsym = %0d, NMA = %0d, midamble_dur = %0f, total_dur = %0d",
          nsymb,nma,midamble_dur,total_dur),UVM_DEBUG)
      end
    end
    // HE_TB
    else begin
      // get number of bits pre symbol from truthtable
      num_of_bits_in_symbol = get_bits_in_one_symbol_func(.txNumSSFn (preamble_header.user_header_he[user].nss_f),
                                                          .txFormatFn(preamble_header.format_mod),
                                                          .txLRateFn (preamble_header.leg_rate),
                                                          .txChBWFn  (preamble_header.ch_bw),
                                                          .txMCSFn   (preamble_header.user_header_he[user].mcs_f),
                                                          .txDCMFn   (preamble_header.dcm),
                                                          .txRUType  (get_ru_type(preamble_header.ru_allocation))
                                                         );

      symb_dur = 12.8 + (gi_type_to_real(preamble_header.gi_type));
      // maximum HELTF = 64us and PE = 16us.
      total_dur = 5000 - `HE_TB_PRE_DUR - 80; //5ms
    end

    ret = $ceil(((total_dur / symb_dur) * num_of_bits_in_symbol - 22) / 8);
    `uvm_info(get_type_name(), $sformatf("Maximum payload length in HT/VHT/HE frames: %0d bytes", ret), UVM_DEBUG)

    // saturate maximum number of payload bytes
    return ((ret < (`MAX_AMPDU_PAYLOAD_SIZE/num_of_users))
             ? ret
             : (`MAX_AMPDU_PAYLOAD_SIZE/num_of_users));

  endfunction : max_payload_len

  //---------------------------------------------
  // determine the limit of MPDU number in A_MPDU
  // so payload length does not exxeed maximum
  // length for VHT duration
  //---------------------------------------------
  virtual function int max_number_of_mpdu(int len);
    int sum;

    sum = $ceil(real'(len) / `MAX_MPDU_SIZE);
    // AMPDU should have more then 1 MPDU
    return (sum == 1) ? 2 : sum;
  endfunction : max_number_of_mpdu

  //---------------------------------------------
  // calculate minimum MPDU start spacing in number
  // of octets
  //---------------------------------------------
  virtual function int calc_min_mpdu_start_spacing(int user=0);
    int  NDBPS;
    int  Nss;
    real data_rate;
    real Tsymb;
    int  ret;

    case (ppdu_format)
      HT_MF,HT_GF,VHT: begin
        // get Nss
        Nss = get_num_ss_func(.txMCS   (preamble_header.user_header[user].mcs_f),
                              .txFormat(preamble_header.format_mod),
                              .txChBW  (preamble_header.ch_bw),
                              .txnSTS  (preamble_header.user_header[user].num_sts_f),
                              .txSTBC  (preamble_header.stbc));
        // get number of bits pre symbol from truthtable
        NDBPS = get_bits_in_one_symbol_func(.txNumSSFn (Nss),
                                            .txFormatFn(preamble_header.format_mod),
                                            .txLRateFn (preamble_header.leg_rate),
                                            .txChBWFn  (preamble_header.ch_bw),
                                            .txMCSFn   (preamble_header.user_header[user].mcs_f)
                                           );

        // symbol duration, long or short
        Tsymb = (preamble_header.gi_type == 2'b00) ? 4.0 : 3.6;
        data_rate = (1 / Tsymb) * NDBPS;
        ret = $ceil(preamble_header.get_tmmss() * data_rate / 8.0);
      end
      HE_SU,HE_EXT_SU: begin
        // get number of bits pre symbol from truthtable
        NDBPS = get_bits_in_one_symbol_func(.txNumSSFn (preamble_header.user_header_he[user].nss_f),
                                            .txFormatFn(preamble_header.format_mod),
                                            .txLRateFn (preamble_header.leg_rate),
                                            .txChBWFn  (preamble_header.ch_bw),
                                            .txMCSFn   (preamble_header.user_header_he[user].mcs_f),
                                            .txDCMFn   (preamble_header.dcm),
                                            .txRUType  (get_ru_type_he_su(preamble_header.ch_bw))
                                           );
        // symbol duration
        Tsymb = 12.8;
        Tsymb += (preamble_header.gi_type == 2'b00) ? 0.8:
                 (preamble_header.gi_type == 2'b01) ? 1.6:
                                                      3.2;
        data_rate = (1 / Tsymb) * NDBPS;
        ret = $ceil(preamble_header.get_tmmss() * data_rate / 8.0);
      end
      HE_MU: begin
        // get number of bits pre symbol from truthtable
        NDBPS = get_bits_in_one_symbol_func(.txNumSSFn (preamble_header.user_header_he[user].nss_f),
                                            .txFormatFn(preamble_header.format_mod),
                                            .txLRateFn (preamble_header.leg_rate),
                                            .txChBWFn  (preamble_header.ch_bw),
                                            .txMCSFn   (preamble_header.user_header_he[user].mcs_f),
                                            .txDCMFn   (preamble_header.dcm),
                                            .txRUType  (get_ru_type_from_ru_allocation(preamble_header.ru_allocation,
                                                                                       preamble_header.user_header_he[user].dut_location_f))
                                           );
        // symbol duration
        Tsymb = 12.8;
        Tsymb += (preamble_header.gi_type == 2'b00) ? 0.8:
                 (preamble_header.gi_type == 2'b01) ? 1.6:
                                                      3.2;
        data_rate = (1 / Tsymb) * NDBPS;
        ret = $ceil(preamble_header.get_tmmss() * data_rate / 8.0);
      end
      default : ret = 0;
    endcase

    `uvm_info(get_type_name(),
    $sformatf("calc_min_mpdu_start_spacing: Tsymb = %0.1f, data rate = %0.1f, N = %0d",Tsymb,data_rate,ret),UVM_DEBUG)

    return ret;
  endfunction : calc_min_mpdu_start_spacing

  //---------------------------------------------
  // create preamble and header in relative to format and modulation
  //---------------------------------------------
  virtual function void create_preamble_header();
    case (ppdu_format)
      NON_HT: preamble_header          = NON_HT_PPDU::type_id::create("preamble_header");
      NON_HT_DUP_OFDM: preamble_header = NON_HT_DUP_OFDM_PPDU::type_id::create("preamble_header");
      HT_MF: preamble_header           = HT_MF_PPDU::type_id::create("preamble_header");
      HT_GF: preamble_header           = HT_GF_PPDU::type_id::create("preamble_header");
      VHT: preamble_header             = VHT_PPDU::type_id::create("preamble_header");
      HE_SU: preamble_header           = HE_SU_PPDU::type_id::create("preamble_header");
      HE_MU: preamble_header           = HE_MU_PPDU::type_id::create("preamble_header");
      HE_EXT_SU: preamble_header       = HE_EXT_SU_PPDU::type_id::create("preamble_header");
      HE_TB: preamble_header           = HE_TRIG_PPDU::type_id::create("preamble_header");
    endcase
  endfunction : create_preamble_header

  //---------------------------------------------
  // prepare frame data for Rx when encryption is
  // set in MPDU frame. Function will prepare data
  // that is encrypted for Rx path
  //---------------------------------------------
  virtual function void setup_rx_encrypted_data();
    // set flag that frame is sent in HT
    uvm_config_db #(format_mod_e)::set(null, "", "FORMATMOD", ppdu_format);

    foreach (ampdu_frame[i])
      ampdu_frame[i].setup_rx_encrypted_data();
  endfunction : setup_rx_encrypted_data

  //---------------------------------------------
  // encrypt MPDU frame if protection is set
  //---------------------------------------------
  virtual function void encrypt(bit ra);
    // set flag that frame is sent in HT
    uvm_config_db #(format_mod_e)::set(null, "", "FORMATMOD", ppdu_format);

    foreach (ampdu_frame[i])
      ampdu_frame[i].encrypt(ra);
  endfunction : encrypt

  //---------------------------------------------
  // decrypt MPDU frame if protection is set
  //---------------------------------------------
  virtual function void decrypt(bit ra);
    // set flag that frame is sent in HT
    uvm_config_db #(format_mod_e)::set(null, "", "FORMATMOD", ppdu_format);

    foreach (ampdu_frame[i])
      ampdu_frame[i].decrypt(ra);
  endfunction : decrypt

  //---------------------------------------------
  // custom compare function
  //---------------------------------------------
  virtual function bit do_compare(uvm_object rhs, uvm_comparer comparer);
    MU_MIMO_PPDU_frame that;
    int index;

    if (!$cast(that, rhs)) begin
      `uvm_fatal(get_type_name(), $sformatf("Failed cast from 'rhs' to 'that'"))
    end

    do_compare = super.do_compare(rhs, comparer);

    // TODO: Update, when PHY supports NON-HT-DUP-OFDM
`ifndef MAC_STANDALONE
    if (this.ppdu_format != NON_HT_DUP_OFDM && tx_frame == 0)
      do_compare &= comparer.compare_field("ppdu_format", this.ppdu_format, that.ppdu_format, $bits(this.ppdu_format));
    else if (tx_frame == 1)
      do_compare &= comparer.compare_field("ppdu_format", this.ppdu_format, that.ppdu_format, $bits(this.ppdu_format));
`elsif
      do_compare &= comparer.compare_field("ppdu_format", this.ppdu_format, that.ppdu_format, $bits(this.ppdu_format));
`endif

    do_compare &= comparer.compare_object("preamble_header", this.preamble_header, that.preamble_header);
    // Tx MU-MIMO frame
    if (tx_frame == 1 && num_of_users > 1) begin
      do_compare &= comparer.compare_field("num_of_users", this.num_of_users, that.num_of_users, $bits(this.num_of_users));
      foreach (ampdu_frame[i]) begin
        index = user_order[i];
        do_compare &= comparer.compare_object($sformatf("ampdu_frame[%0d]",i), this.ampdu_frame[index], that.ampdu_frame[i]);
      end
    end
    else if (tx_frame == 0 && num_of_users > 1) begin
      // only 1 user for MU-MIMO Rx is supported
      index = preamble_header.mu_mimo_userid;
      do_compare &= comparer.compare_field("num_of_users", 'h1, that.num_of_users, $bits(this.num_of_users));
      do_compare &= comparer.compare_object($sformatf("ampdu_frame[%0d]", index), this.ampdu_frame[index], that.ampdu_frame[0]);
    end
    else begin
      do_compare &= comparer.compare_field("num_of_users", this.num_of_users, that.num_of_users, $bits(this.num_of_users));
      foreach (ampdu_frame[i]) begin
        do_compare &= comparer.compare_object($sformatf("ampdu_frame[%0d]",i), this.ampdu_frame[i], that.ampdu_frame[i]);
      end
    end

    return do_compare;
  endfunction : do_compare

endclass : MU_MIMO_PPDU_frame

`endif //MU_MIMO_PPDU_FRAME_SV
