//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : container class for PPDU preamble
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef PPDU_FRAME_SV
`define PPDU_FRAME_SV

class PPDU_frame extends MU_MIMO_PPDU_frame;

  rand ppdu_frame_kind_e      kind;
  // helper variables
       int                    max_mpdu_count;
  rand int                    max_he_tb_length;   // set maximum HE lenght for HE-TB frame
  rand bit                    set_tb_length;      // preset HE-TB lenght
  rand bit                    beamformed;         // beamformed frame
  rand bit                    set_high_data_rate; // force high worst case data rate
  rand bit                    ppdu_force_20mhz;   // force frame to be CHBW20

  `uvm_object_utils_begin(PPDU_frame)
    `uvm_field_enum(ppdu_frame_kind_e, kind, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
  `uvm_object_utils_end

  //---------------------------------------------
  // constraints
  //---------------------------------------------

  constraint c_set_tb_length{
    soft set_tb_length == 0;
  }

  constraint c_set_high_data_rate {
    soft set_high_data_rate == 0;
  }

  constraint c_force_20mhz {
    soft ppdu_force_20mhz == 0;
  }

  constraint c_ppdu_frame {
    solve kind before ppdu_format;
    solve tx_frame before ppdu_format;

    if (ppdu_force_20mhz) {
      ppdu_format != NON_HT_DUP_OFDM;
    }

`ifdef STANDALONE_PHY
  `ifndef RW_NX_VHT_EN
    ppdu_format != VHT;                 //VHT not supported in Modem
  `endif//not RW_NX_VHT_EN
    if (!tx_frame) ppdu_format != HE_TB; //HE-TB not supported on RX path in Modem
`endif//not STANDALONE_MAC

    if (tx_frame) ppdu_format != HE_MU; //HE-MU not supported on Tx path
    // ------------------------------------------------------------------------
    // NOTE: HT-GF is not supported
    // ------------------------------------------------------------------------
    ppdu_format != HT_GF;
    // ------------------------------------------------------------------------

    // If IP supports 20MHz bandwith only, NON_HT_DUP_OFDM is not possible
`ifdef RW_NX_CHBW20
    ppdu_format != NON_HT_DUP_OFDM;
`endif

    if (kind == SINGLETON){
      num_of_users == 1;
      ppdu_format inside {NON_HT, NON_HT_DUP_OFDM, // legacy
                          HT_GF, HT_MF,            // HT
                          VHT,                     // VHT
                          HE_SU, HE_TB, HE_MU};    // HE

    } else if (kind == AGGREGATED){
`ifdef FULL_WLAN_SYSTEM
      if (ppdu_format == HE_MU)
        num_of_users > 1; //spread more users over RU
      else
        num_of_users == 1;
`else
      num_of_users == 1;
`endif
      ppdu_format inside {HT_GF, HT_MF,        // HT
                          VHT,                 // VHT
                          HE_MU, HE_SU, HE_TB};// HE

    } else if (kind == MU_MIMO){
      num_of_users > 1;
      ppdu_format inside {VHT,    // VHT
                          HE_MU}; // HE

    } else if (kind == NDP){
      num_of_users == 0;
      ampdu_frame.size() == 0;
      ppdu_format inside {HT_MF,  // HT
                          VHT,    // VHT
                          HE_SU}; // HE
    }
  }


  function new (string name = "PPDU_frame");
    super.new(name);
  endfunction

  //---------------------------------------------
  // create A-MPDU frame
  //---------------------------------------------
  function void post_randomize();
    // call super post randomize to create all objects
    super.post_randomize();

    // randomize preamble and header
    assert (preamble_header.randomize() with {
      if (tx_frame) {
        preamble_header.tr == TX;
      } else {
        preamble_header.tr == RX;
      }
      // force CHBW20
      if (ppdu_force_20mhz) {
        preamble_header.ch_bw == 3'b000;
      }

// HE-MU RU with more users
`ifdef FULL_WLAN_SYSTEM
      preamble_header.ru_with_more_users == (kind == AGGREGATED && ppdu_format == HE_MU);
`endif//FULL_WLAN_SYSTEM

      // constraint number of user headers
      if (kind == NDP) {
        if (ppdu_format inside {HT_MF, VHT}) {
          preamble_header.user_header.size() == 1;
          if (ppdu_format == VHT) {
            preamble_header.user_header[0].mcs_f == 0;
          }
          preamble_header.user_header[0].num_sts_f == 1;
          preamble_header.user_header[0].fec_coding_f == 0;
        } else {
          preamble_header.user_header_he.size() == 1;
          preamble_header.user_header_he[0].mcs_f == 0;
          preamble_header.user_header_he[0].nss_f inside {1, 3}; //NSTS 2 or 4
          preamble_header.he_ltf_type inside {2'b01, 2'b10};
          preamble_header.txop_duration == 127; //set to UNSPECIFIED
        }
        preamble_header.partial_aid == 0;
        preamble_header.group_id    == 0;
        preamble_header.gi_type     == 0;
        preamble_header.stbc        == 0;
        preamble_header.is_ndp      == 1;
        preamble_header.beamformed  == 0;
      } else if (kind == AGGREGATED || kind == MU_MIMO || ppdu_format == VHT){
        preamble_header.is_ndp == 0;
        preamble_header.aggregated == 1'b1;
        if (ppdu_format inside {HE_SU, HE_MU, HE_TB, HE_EXT_SU}) {
          preamble_header.user_header_he.size() == num_of_users;
        } else {
          preamble_header.user_header.size() == num_of_users;
        }
        // when transmiting beamformed frame
        if (kind == MU_MIMO) {
          soft preamble_header.beamformed == 1;
        } else if (ppdu_format == VHT && tx_frame) {
`ifndef RW_BFMER_EN
          preamble_header.beamformed == 0;
`else
          preamble_header.beamformed == local::beamformed;
`endif
        } else {
          preamble_header.beamformed == 0;
        }
      } else { // SINGLETON
        preamble_header.is_ndp == 0;
        preamble_header.aggregated == 1'b0;
        if (ppdu_format inside {HE_SU, HE_MU, HE_TB, HE_EXT_SU}) {
          preamble_header.user_header_he.size() == 1;
        } else {
          preamble_header.user_header.size() == 1;
        }
      }
    });

    // force highest data rate for this frame
    if (set_high_data_rate)
      preamble_header.set_max_data_rate();

    // randomize A-MPDU frame and apply constraints
    foreach (ampdu_frame[i]) begin
`ifdef STANDALONE_PHY
      // for standalone PHY only payload is created,
      // no MPDU frames
      case (kind)
        SINGLETON:
          assert (ampdu_frame[i].randomize() with {
            if ((local::ppdu_format == NON_HT) && (preamble_header.leg_rate inside {[0:3]})) {
              ampdu_payload.size() < `MAX_MPDU_SIZE_DSSS;
              ampdu_payload.size() >= `MAX_MAC_HEADER_SIZE_DSSS;
            } else {
              ampdu_payload.size() < max_payload_len(i);
              ampdu_payload.size() < `MAX_MPDU_SIZE;
              ampdu_payload.size() >= `MAX_MAC_HEADER_SIZE;
            }
          });
        AGGREGATED, MU_MIMO:
          assert (ampdu_frame[i].randomize() with {
            ampdu_payload.size() < max_payload_len(i);
            ampdu_payload.size() > `MAX_MAC_HEADER_SIZE;
          });
        NDP:
          assert (ampdu_frame[i].randomize() with {
            ampdu_payload.size() == 0;
          });
      endcase
`else
      // calculate maximum payload lenght
      if (set_tb_length == 1 && ppdu_format == HE_TB) begin
        if (kind == AGGREGATED)
          max_mpdu_count = max_number_of_mpdu(max_he_tb_length);
        else
          max_mpdu_count = 1;
        ampdu_frame[i].payload_size = $floor(max_he_tb_length / max_mpdu_count);
      end
      else if (!(ppdu_format inside {NON_HT,NON_HT_DUP_OFDM}) && kind inside {AGGREGATED,MU_MIMO}) begin
        max_mpdu_count = max_number_of_mpdu(max_payload_len(i));
        ampdu_frame[i].payload_size = $floor(max_payload_len(i) / max_mpdu_count) - `MAX_MAC_HEADER_SIZE;
      end
      else if (ppdu_format inside {HE_SU,HE_EXT_SU,HE_MU}) begin
        ampdu_frame[i].payload_size = max_payload_len(i) - `MAX_MAC_HEADER_SIZE;
      end

      // -------------------------------------------------------------------
      // calculate minimum MPDU start spacing
      // -------------------------------------------------------------------
      ampdu_frame[i].min_mpdu_start_spacing = calc_min_mpdu_start_spacing(i);

      // set DSSS flag if needed
      if ((ppdu_format == NON_HT) && (preamble_header.leg_rate inside {[0:3]})) begin
        ampdu_frame[0].is_dsss = 1;
      end else begin
        ampdu_frame[0].is_dsss = 0;
      end

      case (kind)
        SINGLETON:
          assert (ampdu_frame[i].randomize() with {
            // set PPDU format in A-MPDU
            ampdu_frame[i].ppdu_format == local::ppdu_format;
            ampdu_frame[i].mpdu_frame_type.size() == 1;
            ampdu_frame[i].aggregated == 0;
          });
        AGGREGATED:
          assert (ampdu_frame[i].randomize() with {
            // set PPDU format in A-MPDU
            ampdu_frame[i].ppdu_format == local::ppdu_format;
            // for frame to be aggregated number of MPDUs
            // has to be larger then 1
            ampdu_frame[i].aggregated == 1;
            ampdu_frame[i].mpdu_frame_type.size() >  1;
            ampdu_frame[i].mpdu_frame_type.size() <= max_mpdu_count;
          });
        MU_MIMO:
          assert (ampdu_frame[i].randomize() with {
            // set PPDU format in A-MPDU
            ampdu_frame[i].ppdu_format == local::ppdu_format;
            ampdu_frame[i].mpdu_frame_type.size() > 1;
            ampdu_frame[i].mpdu_frame_type.size() <= max_mpdu_count;
            ampdu_frame[i].aggregated == 1;
          });
        NDP: begin end
          // in NDP kind of frame there is no data
          // so there is no need for randomization
          // of A-MPDU frame
      endcase
`endif//STANDALONE_PHY
      // set flag to indicate if frame is for TX
      ampdu_frame[i].tx_frame = tx_frame;
    end //foreach


    // calling calc_leg_ht_length for an NDP frame will cause a crash
    // as there is no ampdu_frame
    if (kind != NDP && ppdu_format != HE_TB) begin
      calc_leg_ht_length();
    end else if (kind == NDP) begin
      preamble_header.leg_length = 0;
      if (ppdu_format <= VHT)
        preamble_header.user_header[0].ht_length_f = 0;
      else
        preamble_header.user_header_he[0].he_length_f = 0;
    end

  endfunction : post_randomize

  //------------------------------------------------------------
  // @ret - 1 - frame is protected
  //        0 - frame not protected
  //------------------------------------------------------------
  function bit is_protected(int user=0);
`ifndef STANDALONE_PHY
    if (kind != NDP)
      return ampdu_frame[user].is_protected();
    else
      return 0;
`else
    return 0;
`endif
  endfunction : is_protected

  //---------------------------------------------
  // custom compare function
  //---------------------------------------------
  virtual function bit do_compare(uvm_object rhs, uvm_comparer comparer);
    PPDU_frame that;

    if (!$cast(that, rhs)) begin
      `uvm_fatal(get_type_name(), $sformatf("Failed cast from 'rhs' to 'that'"))
    end

    do_compare = super.do_compare(rhs, comparer);
`ifdef STANDALONE_MAC
    if (!((kind == MU_MIMO) && (tx_frame == 0))) begin
      // for MU-MIMO Rx, the MAC-PHY monitor will see a SINGLETON/AGGREGATED VHT frame on MAC-PHY IF
      do_compare &= comparer.compare_field("kind", this.kind, that.kind, $bits(this.kind));
    end
`endif
    return do_compare;
  endfunction : do_compare

  //------------------------------------------------------------
  // custom copy function
  //------------------------------------------------------------
  function void do_copy(uvm_object rhs);
    PPDU_frame rhs_;

    if (!$cast(rhs_,rhs)) begin
      `uvm_fatal(get_type_name(), "do_copy cast failed!")
    end
    super.do_copy(rhs);

    kind = rhs_.kind;
    tx_frame = rhs_.tx_frame;

    num_of_users = rhs_.num_of_users;
    ppdu_format = rhs_.ppdu_format;
    create_preamble_header();
    preamble_header.copy(rhs_.preamble_header);

    ampdu_frame = new[rhs_.ampdu_frame.size()];
    foreach (rhs_.ampdu_frame[i]) begin
      ampdu_frame[i] = AMPDU_frame::type_id::create($sformatf("ampdu_frame[%d]",i));
      ampdu_frame[i].copy(rhs_.ampdu_frame[i]);
    end

    user_order = rhs_.user_order;
  endfunction : do_copy

endclass : PPDU_frame

`endif //PPDU_FRAME_SV
