//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef TRIGGER_FRAME_SV
`define TRIGGER_FRAME_SV

class TRIGGER_frame extends control_MPDU_frame;

  rand trigger_type_e                 trigger_type;
  rand common_info_s                  common_info;
  rand user_info_s                    user_info[];
  rand octet_t                        trigger_dependent_common_info[];
  rand octet_t                        trigger_dependent_user_info[];
  rand trigger_dependent_user_info_s  basic_trig_dep_usr_info[];
  rand user_info_ndp_feedback_s       user_info_ndp_feedback[];
  rand octet_t                        padding[];

  rand bar_control_s                  bar_ctrl;
  rand bar_variant_e                  bar_variant;
  rand octet_t                        bar_information[];
  rand int unsigned                   MAX_MCS;

  `uvm_object_utils_begin(TRIGGER_frame)
    `uvm_field_enum     (trigger_type_e, trigger_type, UVM_DEFAULT)
    `uvm_field_int      (common_info, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(user_info, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(trigger_dependent_common_info, UVM_DEFAULT | UVM_NOPRINT | UVM_NOCOMPARE)
    `uvm_field_array_int(trigger_dependent_user_info, UVM_DEFAULT | UVM_NOPRINT | UVM_NOCOMPARE)
    `uvm_field_array_int(basic_trig_dep_usr_info, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(user_info_ndp_feedback, UVM_DEFAULT | UVM_NOPRINT | UVM_NOCOMPARE)
    `uvm_field_int      (bar_ctrl, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOPRINT)
    `uvm_field_enum     (bar_variant_e, bar_variant, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOPRINT)
    `uvm_field_array_int(bar_information, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOPRINT)
  `uvm_object_utils_end

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_MAC_header_subtype {
    MAC_header.frame_ctrl.subtype_f == `TRIGGER;
  }

  constraint c_MAC_address {
    MAC_header.addr.size() == 2;
  }

  constraint c_bar_variant {
    // TODO: Add MULTI-TID BAR, Extended Compressed BAR and GCR BAR when supported
    bar_variant == COMPRESSED_BAR;
  }

  constraint c_padding {
    padding.size() == 2;
    foreach (padding[i])
      padding[i] == 8'hFF;
  }

  constraint c_bar_ctrl_and_bar_information {
    solve bar_variant before bar_ctrl;
    solve bar_variant before bar_information;

    bar_variant == COMPRESSED_BAR -> {
      bar_ctrl.bar_type_f == COMPRESSED_BAR;
      bar_ctrl.bar_ack_policy_f == 1;
      bar_information.size() == 2;
      // Fragment number
      bar_information[0][3:0] == 0;
    }
    bar_variant == EXT_COMPRESSED_BAR -> {
      bar_ctrl.bar_type_f == EXT_COMPRESSED_BAR;
      bar_ctrl.bar_ack_policy_f == 1;
      bar_information.size() == 2;
      // Fragment number
      bar_information[0][3:0] == 0;
    }
    bar_variant == MULTI_TID_BAR -> {
      bar_ctrl.bar_type_f == MULTI_TID_BAR;
      bar_ctrl.bar_ack_policy_f == 1;
      bar_information.size() == (bar_ctrl.tid_info_f + 1) * 4;
      // Fragment number
      foreach(bar_information[i]){
        if (i%4 == 0)
          bar_information[i+2][3:0] == 0;
      }
    }
    bar_variant == GCR_BAR -> {
      bar_ctrl.bar_type_f == GCR_BAR;
      bar_ctrl.bar_ack_policy_f == 1;
      bar_ctrl.tid_info_f == 0;
      bar_information.size() == 8;
      bar_information[0][3:0] == 0;
    }
  }

  constraint c_solve_before {
    solve trigger_type before common_info;
    solve bar_variant before trigger_type;
    solve common_info before trigger_dependent_common_info;
    solve common_info before trigger_dependent_user_info;
    solve common_info before user_info;
  }

  constraint c_user_info_size {
    soft user_info.size() == 1;
  }

  constraint c_common_info {
    common_info.trigger_type_f == trigger_type;
`ifdef RW_NX_CHBW20
    common_info.bw_f == 0;
`elsif RW_NX_CHBW4020
    common_info.bw_f inside {0,1};
`elsif RW_NX_CHBW804020
    common_info.bw_f inside {0,1,2};
`endif

`ifdef RW_TXRX_1X1
    common_info.stbc_f == 0;
`elsif RW_TXRX_2X2
    common_info.stbc_f inside {0,1};
`endif

    common_info.mumimo_ltf_mode_f == 0;
    if (common_info.doppler_f)
      common_info.num_of_he_ltf_symbols_f inside {0,1,2,4,5,6};
    else
      common_info.num_of_he_ltf_symbols_f inside {0,1,2,3,4};
    common_info.pre_fec_padding_f[1:0] == 0;
    common_info.spatial_reuse_f == 16'hffff;
    common_info.doppler_f inside {0,1};
    common_info.ap_tx_power_f inside {[0:60]};
    soft common_info.length_f inside {[500:3000]};
    common_info.length_f % 3 == 1;
    common_info.gi_and_ltf_type_f inside {[1:2]};

    if (common_info.trigger_type_f == MU_RTS) {
      common_info.cs_required_f == 1;
    } else if (common_info.trigger_type_f == BMF_REPORT_POLL) {
      if (common_info.length_f > 76)
        common_info.cs_required_f == 1;
    } else if (common_info.trigger_type_f inside {BASIC_TRIGGER,
                                                  MU_BAR,
                                                  BANDWIDTH_QUERY_REPORT_POLL,
                                                  GCR_MU_BAR}) {
      if (  !(common_info.trigger_type_f inside {MU_BAR, GCR_MU_BAR} && common_info.length_f <= 418)
        ||  !(common_info.length_f <= 76)) {
        common_info.cs_required_f == 1;
      }
    }
  }

  constraint c_user_info {

    `ifdef RW_NX_1024QAM_EN
      MAX_MCS == 11;
    `elsif RW_NX_256QAM_EN
      MAX_MCS == 9;
    `else
      MAX_MCS == 7;
    `endif

    foreach(user_info[i]){
      user_info[i].mcs_f <= MAX_MCS;
      user_info[i].target_rssi_f inside {[0:90],127};
      user_info[i].ss_allocation_f == 0;
      // constrain RU allocation
      if (common_info.bw_f == 0) {
        if (common_info.trigger_type_f != MU_RTS) {
          user_info[i].ru_allocation_f[7:1] inside {[0:8],[37:40],[53:54],61};
        } else {
          user_info[i].ru_allocation_f[7:1] == 61;
        }
      } else if (common_info.bw_f == 1) {
        if (common_info.trigger_type_f != MU_RTS) {
          user_info[i].ru_allocation_f[7:1] inside {[0:17],[37:44],[53:56],61,62,65};
        } else {
          user_info[i].ru_allocation_f[7:1] inside {61,62,65};
        }
        user_info[i].ru_allocation_f[7:1] != 4;
        user_info[i].ru_allocation_f[7:1] != 13;
      } else {
        user_info[i].ru_allocation_f[7:1] inside {[0:67]};
      }

      user_info[i].ru_allocation_f[0] == 0;
      if (user_info[i].mcs_f inside {0,1,3,4}) {
        user_info[i].dcm_f inside { [0:1] };
      } else {
        user_info[i].dcm_f == 0;
      }
      // DCM and STBC can't be applied together
      if (common_info.stbc_f == 1'b1) {
        user_info[i].dcm_f == 0;
      }
      if (get_ru_type(user_info[i].ru_allocation_f) < RU242) {
        user_info[i].mcs_f inside { [0:9] };
      } else {
`ifndef RW_NX_LDPC_ENC
        user_info[i].mcs_f inside { [0:9] };
      }
      user_info[i].coding_type_f == 0;
`else
      if (user_info[i].coding_type_f == 1)
        user_info[i].mcs_f inside { [0:11] };
      else
        user_info[i].mcs_f inside { [0:9] };
      }
`endif
    }
  }

  constraint c_field_sizes {
    if (common_info.trigger_type_f == BASIC_TRIGGER) {
      trigger_dependent_common_info.size() == 0;
      trigger_dependent_user_info.size() == 1;
      user_info_ndp_feedback.size() == 0;
      basic_trig_dep_usr_info.size() == 1;
    }
    else if (common_info.trigger_type_f == BMF_REPORT_POLL) {
      trigger_dependent_common_info.size() == 0;
      trigger_dependent_user_info.size() == user_info.size();
      user_info_ndp_feedback.size() == 0;
      basic_trig_dep_usr_info.size() == 0;
    }
    else if (common_info.trigger_type_f == MU_BAR) {
      trigger_dependent_common_info.size() == 0;
      basic_trig_dep_usr_info.size() == 0;
      if (bar_variant == COMPRESSED_BAR) {
        trigger_dependent_user_info.size() == 4;
      }
      else if (bar_variant == EXT_COMPRESSED_BAR) {
        trigger_dependent_user_info.size() == 4;
      }
      else if (bar_variant == MULTI_TID_BAR) {
        trigger_dependent_user_info.size() == 2 + (bar_ctrl.tid_info_f + 1) * 4;
      }
      else if (bar_variant == GCR_BAR) {
        trigger_dependent_user_info.size() == 10;
      }
      user_info_ndp_feedback.size() == 0;
    }
    else if (common_info.trigger_type_f == MU_RTS) {
      trigger_dependent_common_info.size() == 0;
      trigger_dependent_user_info.size() == 0;
      user_info_ndp_feedback.size() == 0;
      basic_trig_dep_usr_info.size() == 0;
    }
    else if (common_info.trigger_type_f == BUFFER_STATUS_REPORT_POLL) {
      trigger_dependent_common_info.size() == 0;
      trigger_dependent_user_info.size() == 0;
      user_info_ndp_feedback.size() == 0;
      basic_trig_dep_usr_info.size() == 0;
    }
    else if (common_info.trigger_type_f == GCR_MU_BAR) {
      trigger_dependent_common_info.size() == 1;
      trigger_dependent_user_info.size() == 0;
      user_info_ndp_feedback.size() == 0;
      basic_trig_dep_usr_info.size() == 0;
    }
    else if (common_info.trigger_type_f == BANDWIDTH_QUERY_REPORT_POLL) {
      trigger_dependent_common_info.size() == 0;
      trigger_dependent_user_info.size() == 0;
      user_info_ndp_feedback.size() == 0;
      basic_trig_dep_usr_info.size() == 0;
    }
    else if (common_info.trigger_type_f == NDP_FEEDBACK_REPORT_POLL) {
      trigger_dependent_common_info.size() == 0;
      trigger_dependent_user_info.size() == 0;
      user_info_ndp_feedback.size() == 1;
      basic_trig_dep_usr_info.size() == 0;
    }
  }

  constraint c_frame_body_size {
    // Fixed fields:
    // Common Info 8, Padding 2, User Info 5, FCS 4
    frame_body.size() == ( 10
                         + user_info.size()*5
                         + trigger_dependent_common_info.size()
                         + trigger_dependent_user_info.size()
                         + user_info_ndp_feedback.size());
  }
  //---------------------------------------------
  // end of constraints
  //---------------------------------------------

  function new (string name = "TRIGGER_frame");
    super.new(name);
  endfunction

  function void custom_post_randomize();
    int offset;
    // set
    common_info.ldpc_extra_symbol_segment_f = (get_ru_type(user_info[0].ru_allocation_f) >= RU484);
    // Common info subfield
    frame_body[0][3:0]  = common_info.trigger_type_f;
    {frame_body[1],
    frame_body[0][7:4]} = common_info.length_f;
    frame_body[2][0]    = common_info.more_tf_f;
    frame_body[2][1]    = common_info.cs_required_f;
    frame_body[2][3:2]  = common_info.bw_f;
    frame_body[2][5:4]  = common_info.gi_and_ltf_type_f;
    frame_body[2][6]    = common_info.mumimo_ltf_mode_f;
    frame_body[2][7]    = common_info.num_of_he_ltf_symbols_f[0];
    frame_body[3][1:0]  = common_info.num_of_he_ltf_symbols_f[2:1];
    frame_body[3][2]    = common_info.stbc_f;
    frame_body[3][3]    = common_info.ldpc_extra_symbol_segment_f;
    frame_body[3][7:4]  = common_info.ap_tx_power_f[3:0];
    frame_body[4][1:0]  = common_info.ap_tx_power_f[5:4];
    frame_body[4][3:2]  = common_info.pre_fec_padding_f;
    frame_body[4][4]    = common_info.pe_disambiguity_f;
    frame_body[4][7:5]  = common_info.spatial_reuse_f[2:0];
    frame_body[5]       = common_info.spatial_reuse_f[10:3];
    frame_body[6][4:0]  = common_info.spatial_reuse_f[15:11];
    frame_body[6][5]    = common_info.doppler_f;
    frame_body[6][7:6]  = common_info.he_sig_a_reserved_f[1:0];
    frame_body[7][6:0]  = common_info.he_sig_a_reserved_f[8:2];

    offset = $bits(common_info_s)/8;

    // If trigger dependent common info exists
    if (trigger_dependent_common_info.size()) begin
      frame_body[8] = trigger_dependent_common_info[0];
    end
    else begin
      if (trigger_dependent_user_info.size()) begin
        foreach(user_info[i]) begin
          frame_body[8  + 6*i]       = user_info[i].aid12_f[7:0];
          frame_body[9  + 6*i][3:0]  = user_info[i].aid12_f[11:8];
          frame_body[9  + 6*i][7:4]  = user_info[i].ru_allocation_f[3:0];
          frame_body[10 + 6*i][3:0] = user_info[i].ru_allocation_f[7:4];
          frame_body[10 + 6*i][4]   = user_info[i].coding_type_f;
          frame_body[10 + 6*i][7:5] = user_info[i].mcs_f[2:0];
          frame_body[11 + 6*i][0]   = user_info[i].mcs_f[3];
          frame_body[11 + 6*i][1]   = user_info[i].dcm_f;
          frame_body[11 + 6*i][7:2] = user_info[i].ss_allocation_f;
          frame_body[12 + 6*i]      = user_info[i].target_rssi_f;
          if (trigger_type != MU_BAR)
            frame_body[13 + 6*i] = trigger_dependent_user_info[i];
        end
        // determine offset bytes
        offset += ($bits(user_info_s)/8)*user_info.size() + trigger_dependent_user_info.size();
        if (trigger_type != MU_BAR) begin
          foreach (padding[i])
            frame_body[offset++] = padding[i];
        end
        else begin
          case(bar_variant)
            BASIC_BAR :           begin
                                    `uvm_error(get_type_name(),$sformatf("BASIC BAR is not supported"))
                                  end
            COMPRESSED_BAR :      begin
                                    frame_body[13 + 5*(user_info.size()-1)][0]    = bar_ctrl.bar_ack_policy_f;
                                    frame_body[13 + 5*(user_info.size()-1)][4:1]  = bar_ctrl.bar_type_f;
                                    frame_body[13 + 5*(user_info.size()-1)][7:5]  = bar_ctrl.reserved_f[2:0];
                                    frame_body[14 + 5*(user_info.size()-1)][3:0]  = bar_ctrl.reserved_f[6:3];
                                    frame_body[14][7:4] = bar_ctrl.tid_info_f;
                                    frame_body[15] = bar_information[0];
                                    frame_body[16] = bar_information[1];
                                    {frame_body[17],frame_body[18]} = 32'hffff;
                                  end
            EXT_COMPRESSED_BAR :  begin
                                    {frame_body[14],frame_body[15]} = 32'hffff;
                                  end
            MULTI_TID_BAR :       begin
                                    {frame_body[14],frame_body[15]} = 32'hffff;
                                  end
            GCR_BAR :             begin
                                    {frame_body[14],frame_body[15]} = 32'hffff;
                                  end
            default :             begin
                                    `uvm_error(get_type_name(),$sformatf("Unknown BAR variant"))
                                  end
          endcase
        end
      end
      else begin
        if (!user_info_ndp_feedback.size()) begin
          foreach(user_info[i]) begin
            frame_body[8  + 5*i]       = user_info[i].aid12_f[7:0];
            frame_body[9  + 5*i][3:0]  = user_info[i].aid12_f[11:8];
            frame_body[9  + 5*i][7:4]  = user_info[i].ru_allocation_f[3:0];
            frame_body[10 + 5*i][3:0]  = user_info[i].ru_allocation_f[7:4];
            frame_body[10 + 5*i][4]    = user_info[i].coding_type_f;
            frame_body[10 + 5*i][7:5]  = user_info[i].mcs_f[2:0];
            frame_body[11 + 5*i][0]    = user_info[i].mcs_f[3];
            frame_body[11 + 5*i][1]    = user_info[i].dcm_f;
            frame_body[11 + 5*i][7:2]  = user_info[i].ss_allocation_f;
            frame_body[12 + 5*i]       = user_info[i].target_rssi_f;
          end
          offset += ($bits(user_info_s)/8)*user_info.size();
          foreach (padding[i])
            frame_body[offset++] = padding[i];
        end
        else begin
          frame_body[8]       = user_info_ndp_feedback[0].starting_aid_f[7:0];
          frame_body[9][3:0]  = user_info_ndp_feedback[0].starting_aid_f[11:8];
          frame_body[9][7:4]  = user_info_ndp_feedback[0].reserved_f[3:0];
          frame_body[10][4:0] = user_info_ndp_feedback[0].reserved_f[8:4];
          frame_body[10][7:5] = user_info_ndp_feedback[0].feedback_type_f[2:0];
          frame_body[11][0]   = user_info_ndp_feedback[0].feedback_type_f[3];
          frame_body[11][7:1] = user_info_ndp_feedback[0].reserved2_f;
          frame_body[12][6:0] = user_info_ndp_feedback[0].target_rssi_f;
          frame_body[12][7]   = user_info_ndp_feedback[0].multiplexing_flag_f;
          offset += ($bits(user_info_ndp_feedback_s)/8)*user_info_ndp_feedback.size();
          foreach (padding[i])
            frame_body[offset++] = padding[i];
        end
      end
    end

  endfunction : custom_post_randomize

  function void post_monitor();
    trigger_type_e  trig_type;
    bar_variant_e   bar_variant;
    int             user_info_size;
    int             trigger_dependent_common_info_present;

    super.post_monitor();

    common_info.trigger_type_f   = trigger_type_e'(frame_body[0][3:0]);
    trigger_type = common_info.trigger_type_f;
    common_info.length_f         = { frame_body[1],
                                    frame_body[0][7:4]};
    common_info.more_tf_f         = frame_body[2][0];
    common_info.cs_required_f     = frame_body[2][1];
    common_info.bw_f              = frame_body[2][3:2];
    common_info.gi_and_ltf_type_f = frame_body[2][5:4];
    common_info.mumimo_ltf_mode_f = frame_body[2][6];
    common_info.num_of_he_ltf_symbols_f[0]    = frame_body[2][7];
    common_info.num_of_he_ltf_symbols_f[2:1]  = frame_body[3][1:0];
    common_info.stbc_f                        = frame_body[3][2];
    common_info.ldpc_extra_symbol_segment_f   = frame_body[3][3];
    common_info.ap_tx_power_f[3:0]            = frame_body[3][7:4];
    common_info.ap_tx_power_f[5:4]            = frame_body[4][1:0];
    common_info.pre_fec_padding_f             = frame_body[4][3:2];
    common_info.pe_disambiguity_f             = frame_body[4][4];
    common_info.spatial_reuse_f[2:0]          = frame_body[4][7:5];
    common_info.spatial_reuse_f[10:3]         = frame_body[5];
    common_info.spatial_reuse_f[15:11]        = frame_body[6][4:0];
    common_info.doppler_f                     = frame_body[6][5];
    common_info.he_sig_a_reserved_f[1:0]      = frame_body[6][7:6];
    common_info.he_sig_a_reserved_f[8:2]      = frame_body[7][6:0];

    if (trigger_type == GCR_MU_BAR)
      trigger_dependent_common_info_present   = 1;
    else
      trigger_dependent_common_info_present   = 0;

    user_info_size = (frame_body.size() - 10 - trigger_dependent_common_info_present)/5;
    user_info = new[user_info_size];

    case(common_info.trigger_type_f)
      BASIC_TRIGGER,
      BMF_REPORT_POLL:
                                  begin
                                    trigger_dependent_user_info = new[user_info_size];

                                    foreach (user_info[i]) begin
                                      user_info[i].aid12_f[7:0]      = frame_body[8 + 6*i];
                                      user_info[i].aid12_f[11:8]     = frame_body[9 + 6*i][3:0];
                                      user_info[i].ru_allocation_f   = {frame_body[10 + 6*i][3:0],
                                                                        frame_body[9 + 6*i][7:4]};
                                      user_info[i].coding_type_f     = frame_body[10 + 6*i][4];
                                      user_info[i].mcs_f             = {frame_body[11 + 6*i][0],
                                                                        frame_body[10 + 6*i][7:5]};
                                      user_info[i].dcm_f             = frame_body[11 + 6*i][1];
                                      user_info[i].ss_allocation_f   = frame_body[11 + 6*i][7:2];
                                      user_info[i].target_rssi_f     = frame_body[12 + 6*i];
                                      trigger_dependent_user_info[i] = frame_body[13 + 6*i];
                                    end

                                    if (common_info.trigger_type_f == BASIC_TRIGGER)
                                      basic_trig_dep_usr_info = new[trigger_dependent_user_info.size()](trigger_dependent_user_info);
                                  end
      MU_BAR :                    begin
                                    foreach (user_info[i]) begin
                                      user_info[i].aid12_f[7:0]     = frame_body[8 + 5*i];
                                      user_info[i].aid12_f[11:8]    = frame_body[9 + 5*i][3:0];
                                      user_info[i].ru_allocation_f  = {frame_body[10 + 5*i][3:0],
                                                                       frame_body[9 + 5*i][7:4]};
                                      user_info[i].coding_type_f    = frame_body[10 + 5*i][4];
                                      user_info[i].mcs_f            = {frame_body[11 + 5*i][0],
                                                                       frame_body[10 + 5*i][7:5]};
                                      user_info[i].dcm_f            = frame_body[11 + 5*i][1];
                                      user_info[i].ss_allocation_f  = frame_body[11 + 5*i][7:2];
                                      user_info[i].target_rssi_f    = frame_body[12 + 5*i];
                                    end
                                    trigger_dependent_user_info = new[1];
                                    trigger_dependent_user_info[0] = frame_body[8 + 5*user_info_size];
                                    if (common_info.trigger_type_f == MU_BAR) begin
                                      bar_variant = bar_variant_e'{frame_body[8 + 5*user_info_size][4],frame_body[8 + 5*user_info_size][3],
                                                                   frame_body[8 + 5*user_info_size][2],frame_body[8 + 5*user_info_size][1]};
                                      bar_ctrl.bar_ack_policy_f     = frame_body[8 + 5*user_info_size][0];
                                      bar_ctrl.bar_type_f           = {frame_body[8 + 5*user_info_size][4],frame_body[8 + 5*user_info_size][3],
                                                                       frame_body[8 + 5*user_info_size][2],frame_body[8 + 5*user_info_size][1]};
                                      bar_ctrl.reserved_f           = {frame_body[9 + 5*user_info_size][3:0],frame_body[8 + 5*user_info_size][7:5]};
                                      bar_ctrl.tid_info_f           = frame_body[9 + 5*user_info_size][7:4];
                                      bar_information[0]            = frame_body[10 + 5*user_info_size];
                                      bar_information[1]            = frame_body[11 + 5*user_info_size];
                                    end
                                  end
      MU_RTS,
      GCR_MU_BAR :                  begin
                                      foreach (user_info[i]) begin
                                        user_info[i].aid12_f[7:0]     = frame_body[8 + i];
                                        user_info[i].aid12_f[11:8]    = frame_body[9 + i][3:0];
                                        user_info[i].ru_allocation_f  = {frame_body[10 + i][3:0],
                                                                         frame_body[9 + i][7:4]};
                                        user_info[i].coding_type_f    = frame_body[10 + i][4];
                                        user_info[i].mcs_f            = {frame_body[11 + i][0],
                                                                         frame_body[10 + i][7:5]};
                                        user_info[i].dcm_f            = frame_body[11 + i][1];
                                        user_info[i].ss_allocation_f  = frame_body[11 + i][7:2];
                                        user_info[i].target_rssi_f    = frame_body[12 + i];
                                      end
                                    end
      BUFFER_STATUS_REPORT_POLL,
      BANDWIDTH_QUERY_REPORT_POLL : begin
                                      foreach (user_info[i]) begin
                                        user_info[i].aid12_f[7:0]     = frame_body[8 + i];
                                        user_info[i].aid12_f[11:8]    = frame_body[9 + i][3:0];
                                        user_info[i].ru_allocation_f  = {frame_body[10 + i][3:0],
                                                                         frame_body[9 + i][7:4]};
                                        user_info[i].coding_type_f    = frame_body[10 + i][4];
                                        user_info[i].mcs_f            = {frame_body[11 + i][0],
                                                                         frame_body[10 + i][7:5]};
                                        user_info[i].dcm_f            = frame_body[11 + i][1];
                                        user_info[i].ss_allocation_f  = frame_body[11 + i][7:2];
                                        user_info[i].target_rssi_f    = frame_body[12 + i];
                                      end
                                    end
      NDP_FEEDBACK_REPORT_POLL :    begin
                                      user_info_ndp_feedback[0].starting_aid_f        = {frame_body[9][3:0],
                                                                                         frame_body[8]};
                                      user_info_ndp_feedback[0].reserved_f            = {frame_body[10][4:0],
                                                                                         frame_body[9][7:4]};
                                      user_info_ndp_feedback[0].feedback_type_f       = {frame_body[11][0],
                                                                                         frame_body[10][7:5]};
                                      user_info_ndp_feedback[0].reserved2_f           =  frame_body[11][7:1];
                                      user_info_ndp_feedback[0].target_rssi_f         =  frame_body[12][6:0];
                                      user_info_ndp_feedback[0].multiplexing_flag_f   =  frame_body[12][7];
                                    end
    endcase

    padding = new[2];
    foreach (padding[i])
      padding[i] = frame_body[9+5*user_info_size + i];

  endfunction : post_monitor

  virtual function void do_print(uvm_printer printer);
    super.do_print(printer);

    printer.print_string("common_info", $sformatf("%p",common_info));
    foreach (user_info[i])
      printer.print_string("user_info", $sformatf("[%0d] %p",i,user_info[i]));

    foreach (trigger_dependent_common_info[i])
      printer.print_string("trigger_dependent_common_info", $sformatf("[%0d] %p",i,trigger_dependent_common_info[i]));

    foreach (trigger_dependent_user_info[i])
      printer.print_string("trigger_dependent_user_info", $sformatf("[%0d] %p",i,trigger_dependent_user_info[i]));

    if (trigger_type == BASIC_TRIGGER) begin
      foreach(basic_trig_dep_usr_info[i])
        printer.print_string("BASIC trigger_dependent_user_info", $sformatf("[%0d] %p",i,basic_trig_dep_usr_info[i]));
    end

    foreach (user_info_ndp_feedback[i])
      printer.print_string("user_info_ndp_feedback", $sformatf("[%0d] %p",i,user_info_ndp_feedback[i]));

    if (trigger_type == MU_BAR) begin
      printer.print_string("bar_variant", $sformatf("%s",bar_variant.name()));
      printer.print_string("bar_ctrl", $sformatf("%p",bar_ctrl));

      foreach(bar_information[i])
        printer.print_int($sformatf("bar_information [%0d]",i),bar_information[i],$bits(bar_information[i]),UVM_HEX);
    end

  endfunction : do_print

  //------------------------------------------------------------
  // custom copy function
  //------------------------------------------------------------
  virtual function void do_copy(uvm_object rhs);
    TRIGGER_frame rhs_;

    if (!$cast(rhs_,rhs)) begin
      `uvm_fatal(get_type_name(), "do_copy cast failed!")
    end
    super.do_copy(rhs);

    trigger_type = rhs_.common_info.trigger_type_f;
    common_info  = rhs_.common_info;
    user_info    = new[rhs_.user_info.size()](rhs_.user_info);
    trigger_dependent_common_info = new[rhs_.trigger_dependent_common_info.size()](rhs_.trigger_dependent_common_info);
    trigger_dependent_user_info = new[rhs_.trigger_dependent_user_info.size()](rhs_.trigger_dependent_user_info);
    user_info_ndp_feedback = new[rhs_.user_info_ndp_feedback.size()](rhs_.user_info_ndp_feedback);

    if (rhs_.common_info.trigger_type_f == BASIC_TRIGGER)
      basic_trig_dep_usr_info = new[rhs_.basic_trig_dep_usr_info.size()](rhs_.basic_trig_dep_usr_info);

    if (rhs_.common_info.trigger_type_f == MU_BAR) begin
      bar_variant = rhs_.bar_variant;
      bar_ctrl = rhs_.bar_ctrl;
      bar_information = new[rhs_.bar_information.size()](rhs_.bar_information);
    end

  endfunction : do_copy


endclass : TRIGGER_frame

`endif// TRIGGER_FRAME_SV

