//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef SECURITY_WRAPPER_ENGINE_SV
`define SECURITY_WRAPPER_ENGINE_SV

class security_wrapper_engine extends uvm_object;

  //---------------------------------------------
  // Encryption variables, legacy code
  //---------------------------------------------
  wep                    wep_obj;
  tkip                   tkip_obj;
  ccmp                   ccmp_obj;
  wapi                   wapi_obj;
  gcmp                   gcmp_obj;
  //---------------------------------------------
  KeyStorageRAM             ksr;    // handle to KSR
  cipher_type_e             security_type;
  bit                       encrypted;
  bit                       decrypted;
  format_mod_e              ppdu_format;
  rand wep_security         wep_sec;
  rand tkip_security        tkip_sec;
  rand ccmp_security        ccmp_sec;
  rand wapi_security        wapi_sec;
  rand gcmp_security        gcmp_sec;

  `uvm_object_utils_begin(security_wrapper_engine)
    `uvm_field_enum(cipher_type_e, security_type, UVM_DEFAULT)
  `uvm_object_utils_end

  function new (string name = "security_wrapper_engine");
    super.new(name);

    wep_sec  = new();
    tkip_sec = new();
    ccmp_sec = new();
    wapi_sec = new();
    gcmp_sec = new();
    security_type = NO_SECURITY;

    // get DB configuration objects
    if (!uvm_config_db #(KeyStorageRAM)::get(null, "", "KeyStorageRAM", ksr))
      `uvm_fatal(get_type_name(), "Key storage RAM object missing!")

    if (!uvm_config_db #(format_mod_e)::get(null, "", "FORMATMOD", ppdu_format))
      `uvm_fatal(get_type_name(), "FORMATMOD not set in config DB")
  endfunction

  // function for reducing frame body size if after enryption his size will be
  // larger then 4096 bytes
  function void reduce_frame_size(input cipher_type_e sec, ref octet_t frame[]);
    case (sec)
      NULL_KEY, NO_SECURITY: return;
      WEP:  begin
              while (frame.size() + `WEP_HEADER_FOOTER_LEN + `MAX_MAC_HEADER_SIZE > `MAX_MPDU_SIZE) begin
                frame = new[frame.size()-1](frame);
              end
            end
      TKIP: begin
              while (frame.size() + `TKIP_HEADER_FOOTER_LEN + `MAX_MAC_HEADER_SIZE > `MAX_MPDU_SIZE) begin
                frame = new[frame.size()-1](frame);
              end
            end
      CCMP: begin
              while (frame.size() + `CCMP_HEADER_FOOTER_LEN + `MAX_MAC_HEADER_SIZE > `MAX_MPDU_SIZE) begin
                frame = new[frame.size()-1](frame);
              end
            end
      WAPI: begin
              while (frame.size() + `WAPI_HEADER_FOOTER_LEN + `MAX_MAC_HEADER_SIZE > `MAX_MPDU_SIZE) begin
                frame = new[frame.size()-1](frame);
              end
            end
      GCMP: begin
              while (frame.size() + `GCMP_HEADER_FOOTER_LEN + `MAX_MAC_HEADER_SIZE > `MAX_MPDU_SIZE) begin
                frame = new[frame.size()-1](frame);
              end
            end
    endcase
  endfunction : reduce_frame_size

  //---------------------------------------------
  // function for encryption of frame data
  // @ra - indication that we use RA or TA address
  //       for KSR search
  // @mh - MAC header structure
  // @frame - frame body data array, function will add
  //          all additional bytes needed (IV, eIV, MIC,
  //          CCMP header, ICV...)
  // @enc_frame - frame body data enccrypted
  //---------------------------------------------
  function void encrypt_frame(bit ra,
                              ref MAC_header mh,
                              ref octet_t frame[],
                              ref octet_t enc_frame[]);
    mac_address_t   dev;
    int             index;
    int             data_cnt;
    octet_t         save_frm[];

    encrypted = 1;
    //-------------------------------------------------------------
    // get RA/TA address from MAC header and the find match in
    // KeyStorageRAM so encryption could be done
    //-------------------------------------------------------------
    dev           = (ra) ? mh.addr[0] : mh.addr[1];
    index         = ksr.get_device_index(dev);

    // if device is not found in KSR don't encrypt
    if (index < 0) begin
      security_type = NO_SECURITY;
      return;
    end
    // set offset key
    wep_sec.IV.key_id_f  = ksr.keyRAM[index].offset_key_idx_f;
    tkip_sec.IV.key_id_f = ksr.keyRAM[index].offset_key_idx_f;
    ccmp_sec.IV.key_id_f = ksr.keyRAM[index].offset_key_idx_f;
    gcmp_sec.IV.key_id_f = ksr.keyRAM[index].offset_key_idx_f;

    // when frame is received if useDefKeyRAM is set when Key Index and VLANID
    // will determine which KSR index entry is used
    if (   !ra // reception
        && (   ksr.keyRAM[index].use_def_key_ram_f
            || mh.addr[0] inside {`BROADCAST_ADDR,`MULTICAST_ADDR})) begin

      index =   ksr.keyRAM[index].vlan_id_ram_f * `DEFAULT_KEY_CNT
              + ksr.keyRAM[index].offset_key_idx_f;
    end
    security_type = cipher_type_e'(ksr.keyRAM[index].ctype_ram_f);
    //-------------------------------------------------------------

    reduce_frame_size(security_type, frame);

    case (security_type)
//*****************************************************************************
      NO_SECURITY: begin
//*****************************************************************************
        // nothing to do
      end//NO_SECURITY

//*****************************************************************************
      NULL_KEY: begin
//*****************************************************************************
        // nothing to do when this ecryption is performed
      end//NULL_KEY

//*****************************************************************************
      WEP: begin
//*****************************************************************************

        wep_obj = new();
        // determine is 64bit or 128bit encryption used
        if (ksr.keyRAM[index].clen_ram_f == 2'b01)
          wep_obj.Key = new[16];
        else
          wep_obj.Key = new[8];

        wep_obj.Key[0] = wep_sec.IV[7:0];
        wep_obj.Key[1] = wep_sec.IV[15:8];
        wep_obj.Key[2] = wep_sec.IV[23:16];
        for(int i=0; i < wep_obj.Key.size()-3; i++)
          wep_obj.Key[i+3] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];

        // allocate memory for plain_data + 4 bytes for ICV
        wep_obj.plain_data = new[frame.size()];
        foreach (wep_obj.plain_data[i])
          wep_obj.plain_data[i] = frame[i];

        // calculate CRC on frame body data
        wep_obj.crc32(wep_obj.plain_data, wep_obj.ICV);
        wep_obj.plain_data = new[wep_obj.plain_data.size()+4](wep_obj.plain_data);
        wep_obj.plain_data[wep_obj.plain_data.size()-4] = wep_obj.ICV[31:24];
        wep_obj.plain_data[wep_obj.plain_data.size()-3] = wep_obj.ICV[23:16];
        wep_obj.plain_data[wep_obj.plain_data.size()-2] = wep_obj.ICV[15:8];
        wep_obj.plain_data[wep_obj.plain_data.size()-1] = wep_obj.ICV[7:0];

        wep_sec.ICV = wep_obj.ICV;

        // run encryption on data
        wep_obj.encrypt();

        // set encrypted data
        enc_frame = new[4 + frame.size() + 4]; // IV + frame body + ICV
        for (int i=0; i<4; i++)
          enc_frame[i] = wep_sec.IV[i*8 +: 8]; //set IV
        foreach (wep_obj.enc_data[i])
          enc_frame[i+4] = wep_obj.enc_data[i]; // set data + ICV

        // set plain data
        save_frm = new[frame.size()](frame);
        // extend frame body for IV and ICV
        frame = new[4 + save_frm.size() + 4];
        for (int i=0; i<4; i++)
          frame[i] = wep_sec.IV[i*8 +: 8]; //set IV
        foreach (save_frm[i])
          frame[i+4] = save_frm[i]; // set frame body data

        // store ICV
        frame[frame.size()-4] = wep_sec.ICV[31:24];
        frame[frame.size()-3] = wep_sec.ICV[23:16];
        frame[frame.size()-2] = wep_sec.ICV[15:8];
        frame[frame.size()-1] = wep_sec.ICV[7:0];

        `uvm_info(get_type_name(), $sformatf("WEP encryption done!"),UVM_LOW)

      end//WEP

//*****************************************************************************
      TKIP: begin
//*****************************************************************************

        tkip_obj = new();
        foreach (tkip_obj.TK[i])
          tkip_obj.TK[i] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];

        // Transmister Adress
        foreach(tkip_obj.TA[i])
          tkip_obj.TA[i]  = mh.addr[1][i*8 +: 8];

        // Packet Number
        tkip_obj.TSC[0] = tkip_sec.IV.tsc0_f;
        tkip_obj.TSC[1] = tkip_sec.IV.tsc1_f;
        tkip_obj.TSC[2] = tkip_sec.eIV.tsc2_f;
        tkip_obj.TSC[3] = tkip_sec.eIV.tsc3_f;
        tkip_obj.TSC[4] = tkip_sec.eIV.tsc4_f;
        tkip_obj.TSC[5] = tkip_sec.eIV.tsc5_f;

        // payload + MIC
        tkip_obj.PDU_MIC = new[frame.size() + 8];
        foreach (frame[i])
          tkip_obj.PDU_MIC[i] = frame[i];
        foreach (tkip_sec.MIC[i])
          tkip_obj.PDU_MIC[i+frame.size()] = tkip_sec.MIC[i];

        // Key scheduling
        tkip_obj.key_scheduling();

        // ICV + rc4 Encryption
        tkip_obj.TKIP_Encrypt();

        // set encrypted data
        // IV + eIV + frame body + MIC + ICV
        enc_frame = new[4 + 4 + frame.size() + 8 + 4];
        for (int i=0; i<4; i++)
          enc_frame[i] = tkip_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          enc_frame[i+4] = tkip_sec.eIV[i*8 +: 8]; //set eIV
        foreach (tkip_obj.PDU_MIC_encrypted[i])
          enc_frame[i+8] = tkip_obj.PDU_MIC_encrypted[i]; // set encrypted data

        // set plain data
        save_frm = new[frame.size()](frame);
        // extend frame body for IV + eIV + frame body + MIC + ICV
        frame = new[4 + 4 + save_frm.size() + 8 + 4];
        for (int i=0; i<4; i++)
          frame[i] = tkip_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          frame[i+4] = tkip_sec.eIV[i*8 +: 8]; //set eIV
        foreach (save_frm[i])
          frame[i+8] = save_frm[i]; // set frame body data
        foreach (tkip_sec.MIC[i])
          frame[8+save_frm.size()+i] = tkip_sec.MIC[i]; // set MIC

        for (int i=0; i<4; i++)
          tkip_sec.ICV[i*8 +: 8] = tkip_obj.ICV[i*8 +: 8]; // save ICV

        // store ICV
        frame[frame.size()-4] = tkip_sec.ICV[31:24];
        frame[frame.size()-3] = tkip_sec.ICV[23:16];
        frame[frame.size()-2] = tkip_sec.ICV[15:8];
        frame[frame.size()-1] = tkip_sec.ICV[7:0];

        `uvm_info(get_type_name(), $sformatf("TKIP encryption done!"),UVM_LOW)

      end//TKIP

//*****************************************************************************
      CCMP: begin
//*****************************************************************************

        ccmp_obj = new();
        // 128bit
        if (ksr.keyRAM[index].clen_ram_f == 2'b00) begin
          ccmp_obj.M   = 8;
          ccmp_obj.L   = 2;
          ccmp_obj.Key = new[16];
          ccmp_sec.MIC = new[8];
        end
        // 256bit
        else begin
          ccmp_obj.M   = 16;
          ccmp_obj.L   = 2;
          ccmp_obj.Key = new[32];
          ccmp_sec.MIC = new[16];
        end
        // Key
        foreach(ccmp_obj.Key[i])
          ccmp_obj.Key[i] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];
        // Mac header
        ccmp_obj.macheader = new[mh.size()];
        data_cnt = 0;
        mh.put2array(data_cnt, ccmp_obj.macheader);
        // trimm HT control octets
        if (mh.ht_ctrl.size())
          ccmp_obj.macheader = new[ccmp_obj.macheader.size()-4](ccmp_obj.macheader);

        // Packet CCMP Header
        ccmp_obj.PN[0] = ccmp_sec.IV.pn0_f;
        ccmp_obj.PN[1] = ccmp_sec.IV.pn1_f;
        ccmp_obj.PN[2] = ccmp_sec.eIV.pn2_f;
        ccmp_obj.PN[3] = ccmp_sec.eIV.pn3_f;
        ccmp_obj.PN[4] = ccmp_sec.eIV.pn4_f;
        ccmp_obj.PN[5] = ccmp_sec.eIV.pn5_f;

        // payload
        ccmp_obj.plain_data = new[frame.size()];
        foreach (ccmp_obj.plain_data[i])
          ccmp_obj.plain_data[i] = frame[i];

        // Nonce
        ccmp_obj.ConstructCCMnonce();

        // HT AAD Patching
        if (ppdu_format >= HT_MF) begin
          ccmp_obj.htMode = 1;
          if(ksr.keyRAM[index].spp_ram_f == 2'b10)
            ccmp_obj.sspCapable = 1;
        end

        // AAD
        ccmp_obj.ConstructAAD();
        // MIC
        ccmp_obj.CCMP_AES_MIC(ccmp_obj.CCMnonce, ccmp_obj.plain_data, ccmp_obj.AAD,0);
        // Enc
        ccmp_obj.CCMP_AES_Enc(ccmp_obj.CCMnonce, ccmp_obj.plain_data,0);

        // store MIC to CCMP structure
        foreach (ccmp_sec.MIC[i])
          ccmp_sec.MIC[i] = ccmp_obj.MIC[i];

        // set encrypted data
        // CCMP header + frame body + MIC
        enc_frame = new[8 + frame.size() + ccmp_sec.MIC.size()];
        for (int i=0; i<4; i++)
          enc_frame[i] = ccmp_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          enc_frame[i+4] = ccmp_sec.eIV[i*8 +: 8]; //set eIV
        foreach (ccmp_obj.enc_data[i])
          enc_frame[i+8] = ccmp_obj.enc_data[i]; // set encrypted data

        // set plain data
        save_frm = new[frame.size()](frame);
        // extend frame body for CCMP header + frame body + MIC
        frame = new[8 + save_frm.size() + ccmp_sec.MIC.size()];
        for (int i=0; i<4; i++)
          frame[i] = ccmp_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          frame[i+4] = ccmp_sec.eIV[i*8 +: 8]; //set eIV
        foreach (save_frm[i])
          frame[i+8] = save_frm[i]; // set frame body data
        foreach (ccmp_sec.MIC[i])
          frame[8+save_frm.size()+i] = ccmp_sec.MIC[i]; // set MIC

        `uvm_info(get_type_name(), $sformatf("CCMP encryption done!"),UVM_LOW)

      end//CCMP

//*****************************************************************************
      WAPI: begin
//*****************************************************************************

        wapi_obj = new();

        wapi_obj.wpi_c.wpiHeader.frameControl = mh.frame_ctrl;
        wapi_obj.wpi_c.wpiHeader.addr1 = mh.addr[0];
        wapi_obj.wpi_c.wpiHeader.addr2 = mh.addr[1];
        wapi_obj.wpi_c.wpiHeader.addr3 = mh.addr[2];
        if (mh.addr.size() == 4) begin
          wapi_obj.wpi_c.wpiHeader.addr4 = mh.addr[3];
          wapi_obj.wpi_c.wpiHeader.address4Pres = 1;
        end
        if (mh.qos_ctrl.size()) begin
          wapi_obj.wpi_c.wpiHeader.qosFrame = 1;
          wapi_obj.wpi_c.wpiHeader.qosCF = mh.qos_ctrl[0];
        end
        wapi_obj.wpi_c.wpiHeader.seqControl = mh.seq_ctrl[0];
        wapi_obj.wpi_c.wpiHeader.keyIdx = wapi_sec.key_idx;
        wapi_obj.wpi_c.wpiHeader.pn = wapi_sec.PN;
        for (int i=0; i<16; i++) begin
          wapi_obj.wpi_c.wpiHeader.wpiEncryptionKey[i*8 +: 8] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];
          wapi_obj.wpi_c.wpiHeader.wpiIntegrityKey[i*8 +: 8] = ksr.keyRAM[index].int_wpi_key_ram_f[i*8 +: 8];
        end

        wapi_obj.wpi_c.wpiHeader.pduLength = frame.size();

        wapi_obj.InitWPIStruct();

        for (int i=0; i<frame.size(); i++)
          for (int j=0; j<8; j++)
            wapi_obj.wpi_c.pduPData.vector[((frame.size()*8)-1-(i*8))-j] = frame[i][7-j];

        // encrypt
        wapi_obj.WAPIEncrypt();
        // save plain MIC
        foreach (wapi_sec.MIC[i])
          wapi_sec.MIC[i] = wapi_obj.wpi_c.pMIC[i*8 +: 8];

        // set encrypted data
        // keyIdx + rsvd + PN + frame body + MIC
        enc_frame = new[1 + 1 + 16 + frame.size() + 16];
        enc_frame[0] = wapi_sec.key_idx;
        enc_frame[1] = 8'b0; // reserved
        for (int i=0; i<16; i++)
          enc_frame[i+2] = wapi_sec.PN[i*8 +: 8]; //set PN
        // set encrypted data
        for (int i=0; i<wapi_obj.wpi_c.wpiHeader.pduLength; i++) begin
          for (int j=0; j<8; j++) begin
            enc_frame[i+18][7-j] = wapi_obj.wpi_c.pduEData[((wapi_obj.wpi_c.wpiHeader.pduLength * 8) - 1 - (i * 8)) - j];
          end
        end
        // set encrypted MIC data
        for (int i=0; i<16; i++) begin
          for (int j=0; j<8; j++) begin
            enc_frame[18 + wapi_obj.wpi_c.wpiHeader.pduLength + i][7-j] = wapi_obj.wpi_c.eMIC[((128 - 1) - (i * 8)) - j];
          end
        end

        // set plain data
        save_frm = new[frame.size()](frame);
        // keyIdx + rsvd + PN + frame body + MIC
        frame = new[1 + 1 + 16 + frame.size() + 16];
        frame[0] = wapi_sec.key_idx;
        frame[1] = 8'b0; // reserved
        for (int i=0; i<16; i++)
          frame[i+2] = wapi_sec.PN[i*8 +: 8]; //set PN
        foreach (save_frm[i])
          frame[i+18] = save_frm[i]; // set frame body data
        foreach (wapi_sec.MIC[i])
          frame[18+save_frm.size()+i] = wapi_sec.MIC[i]; // set MIC

        `uvm_info(get_type_name(), $sformatf("WAPI encryption done!"),UVM_LOW)

      end//WAPI

//*****************************************************************************
      GCMP: begin
//*****************************************************************************

        gcmp_obj = new();
        // 128bit
        if (ksr.keyRAM[index].clen_ram_f == 2'b00)
          gcmp_obj.Key = new[16];
        // 256bit
        else
          gcmp_obj.Key = new[32];

        // Key
        foreach(gcmp_obj.Key[i])
          gcmp_obj.Key[i] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];
        // Mac header
        gcmp_obj.macheader = new[mh.size()];
        data_cnt = 0;
        mh.put2array(data_cnt, gcmp_obj.macheader);
        // trimm HT control octets
        if (mh.ht_ctrl.size())
          gcmp_obj.macheader = new[gcmp_obj.macheader.size()-4](gcmp_obj.macheader);

        // Packet GCMP Header
        gcmp_obj.PN[0] = gcmp_sec.IV.pn0_f;
        gcmp_obj.PN[1] = gcmp_sec.IV.pn1_f;
        gcmp_obj.PN[2] = gcmp_sec.eIV.pn2_f;
        gcmp_obj.PN[3] = gcmp_sec.eIV.pn3_f;
        gcmp_obj.PN[4] = gcmp_sec.eIV.pn4_f;
        gcmp_obj.PN[5] = gcmp_sec.eIV.pn5_f;

        // payload
        gcmp_obj.plain_data = new[frame.size()];
        foreach (gcmp_obj.plain_data[i])
          gcmp_obj.plain_data[i] = frame[i];

        // Nonce
        gcmp_obj.ConstructGCMnonce();

        // HT AAD Patching
        if (ppdu_format >= HT_MF) begin
          gcmp_obj.htMode = 1;
          if(ksr.keyRAM[index].spp_ram_f == 2'b10)
            gcmp_obj.sspCapable = 1;
        end

        // AAD
        gcmp_obj.ConstructAAD();
        // Enc
        gcmp_obj.GCMP_Enc(gcmp_obj.GCMnonce, gcmp_obj.plain_data,0);
        // MIC
        gcmp_obj.GCMP_MIC(gcmp_obj.GCMnonce, gcmp_obj.enc_data, gcmp_obj.AAD,0);

        // store MIC to GCMP structure
        foreach (gcmp_sec.MIC[i])
          gcmp_sec.MIC[i] = gcmp_obj.MIC[i];

        // set encrypted data
        // GCMP header + frame body + MIC
        enc_frame = new[8 + frame.size() + gcmp_sec.MIC.size()];
        for (int i=0; i<4; i++)
          enc_frame[i] = gcmp_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          enc_frame[i+4] = gcmp_sec.eIV[i*8 +: 8]; //set eIV
        foreach (gcmp_obj.enc_data[i])
          enc_frame[i+8] = gcmp_obj.enc_data[i]; // set encrypted data
        foreach (gcmp_obj.MIC[i])
          enc_frame[8+gcmp_obj.enc_data.size()+i] = gcmp_obj.MIC[i]; // set MIC data

        // set plain data
        save_frm = new[frame.size()](frame);
        // extend frame body for GCMP header + frame body + MIC
        frame = new[8 + save_frm.size() + gcmp_sec.MIC.size()];
        for (int i=0; i<4; i++)
          frame[i] = gcmp_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          frame[i+4] = gcmp_sec.eIV[i*8 +: 8]; //set eIV
        foreach (save_frm[i])
          frame[i+8] = save_frm[i]; // set frame body data
        foreach (gcmp_sec.MIC[i])
          frame[8+save_frm.size()+i] = gcmp_sec.MIC[i]; // set MIC

        `uvm_info(get_type_name(), $sformatf("GCMP encryption done!"),UVM_LOW)

      end//GCMP
    endcase

  endfunction : encrypt_frame

  //---------------------------------------------
  // function for decryption of frame data
  // @ra - indication that we use RA or TA address
  //       for KSR search
  // @mh - MAC header structure
  // @frame - frame body data array, function will add
  //          all additional bytes needed (IV, eIV, MIC,
  //          CCMP header, ICV...)
  // @enc_frame - frame body data encrypted
  //---------------------------------------------
  function void decrypt_frame(bit ra,
                              ref MAC_header mh,
                              ref octet_t frame[],
                              ref octet_t enc_frame[]);

    mac_address_t   dev;
    int             index;
    int             data_cnt;
    octet_t         save_frm[];

    decrypted = 1;
    //-------------------------------------------------------------
    // get RA/TA address from MAC header and the find match in
    // KeyStorageRAM so encryption could be done
    //-------------------------------------------------------------
    dev           = (ra) ? mh.addr[0] : mh.addr[1];
    index         = ksr.get_device_index(dev);

    // if device is not found in KSR don't decrypt
    if (index < 0) begin
      security_type = NO_SECURITY;
      return;
    end

    // when frame is received if useDefKeyRAM is set then Key Index and VLANID
    // will determine which KSR index entry is used
    if (   !ra // reception
        && (   ksr.keyRAM[index].use_def_key_ram_f
            || mh.addr[0] inside {`BROADCAST_ADDR,`MULTICAST_ADDR})) begin

      index =   ksr.keyRAM[index].vlan_id_ram_f * `DEFAULT_KEY_CNT
              + ksr.keyRAM[index].offset_key_idx_f;
    end
    security_type = cipher_type_e'(ksr.keyRAM[index].ctype_ram_f);
    //-------------------------------------------------------------

    case (security_type)
//*****************************************************************************
      NO_SECURITY: begin
//*****************************************************************************
        // nothing to do
      end//NO_SECURITY

//*****************************************************************************
      NULL_KEY: begin
//*****************************************************************************
        // nothing to do when this ecryption is performed
      end//NULL_KEY

//*****************************************************************************
      WEP: begin
//*****************************************************************************

        wep_obj = new();
        // determine is 64bit or 128bit encryption used
        if (ksr.keyRAM[index].clen_ram_f == 2'b01)
          wep_obj.Key = new[16];
        else
         wep_obj.Key = new[8];

        wep_obj.Key[0] = enc_frame[0];
        wep_obj.Key[1] = enc_frame[1];
        wep_obj.Key[2] = enc_frame[2];
        for(int i=0; i < wep_obj.Key.size()-3; i++)
          wep_obj.Key[i+3] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];

        // save IV parameters
        wep_sec.IV[7:0]   = enc_frame[0];
        wep_sec.IV[15:8]  = enc_frame[1];
        wep_sec.IV[23:16] = enc_frame[2];
        wep_sec.IV[31:24] = enc_frame[3];

        // allocate memory for encoded data - IV bytes
        wep_obj.enc_data = new[enc_frame.size()-4];
        foreach (wep_obj.enc_data[i])
          wep_obj.enc_data[i] = enc_frame[i+4];

        // run decryption on data
        wep_obj.decrypt();

        // set encrypted data
        frame = new[enc_frame.size()];
        for (int i=0; i<4; i++)
          frame[i] = wep_sec.IV[i*8 +: 8]; //set IV
        foreach (wep_obj.dec_data[i])
          frame[i+4] = wep_obj.dec_data[i]; // set data + ICV

        // save ICV parameters
        wep_sec.ICV[7:0]   = frame[frame.size()-4];
        wep_sec.ICV[15:8]  = frame[frame.size()-3];
        wep_sec.ICV[23:16] = frame[frame.size()-2];
        wep_sec.ICV[31:24] = frame[frame.size()-1];

        `uvm_info(get_type_name(), $sformatf("WEP decryption done!"),UVM_LOW)

      end//WEP

//*****************************************************************************
      TKIP: begin
//*****************************************************************************

        tkip_obj = new();
        foreach (tkip_obj.TK[i])
          tkip_obj.TK[i] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];

        // Transmister Adress
        foreach(tkip_obj.TA[i])
          tkip_obj.TA[i]  = mh.addr[1][i*8 +: 8];

        // Packet Number
        tkip_obj.TSC[0] = enc_frame[2];
        tkip_obj.TSC[1] = enc_frame[0];
        tkip_obj.TSC[2] = enc_frame[4];
        tkip_obj.TSC[3] = enc_frame[5];
        tkip_obj.TSC[4] = enc_frame[6];
        tkip_obj.TSC[5] = enc_frame[7];
        // save IV and eIV parameters
        tkip_sec.IV.tsc0_f  = enc_frame[2];
        tkip_sec.IV.tsc1_f  = enc_frame[0];
        tkip_sec.eIV.tsc2_f = enc_frame[4];
        tkip_sec.eIV.tsc3_f = enc_frame[5];
        tkip_sec.eIV.tsc4_f = enc_frame[6];
        tkip_sec.eIV.tsc5_f = enc_frame[7];
        tkip_sec.IV.wep_seed_f = enc_frame[1];
        tkip_sec.IV.ext_iv_f   = enc_frame[3][5];
        tkip_sec.IV.key_id_f   = enc_frame[3][7:6];

        // payload - IV and eIV
        tkip_obj.PDU_MIC_encrypted = new[enc_frame.size()-8];
        foreach (tkip_obj.PDU_MIC_encrypted[i])
          tkip_obj.PDU_MIC_encrypted[i] = enc_frame[i + 8];

        // Key scheduling
        tkip_obj.key_scheduling();

        // ICV + rc4 Encryption
        tkip_obj.TKIP_Decrypt();

        // set encrypted data
        frame = new[enc_frame.size()];
        for (int i=0; i<4; i++)
          frame[i] = tkip_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          frame[i+4] = tkip_sec.eIV[i*8 +: 8]; //set eIV
        foreach (tkip_obj.PDU_MIC_decrypted[i])
          frame[i+8] = tkip_obj.PDU_MIC_decrypted[i]; // set decrypted data

        // save MIC to structure
        foreach (tkip_sec.MIC[i])
          tkip_sec.MIC[i] = frame[frame.size()-12+i];

        // save ICV parameters
        tkip_sec.ICV[7:0]   = frame[frame.size()-4];
        tkip_sec.ICV[15:8]  = frame[frame.size()-3];
        tkip_sec.ICV[23:16] = frame[frame.size()-2];
        tkip_sec.ICV[31:24] = frame[frame.size()-1];

        `uvm_info(get_type_name(), $sformatf("TKIP decryption done!"),UVM_LOW)

      end//TKIP

//*****************************************************************************
      CCMP: begin
//*****************************************************************************

        ccmp_obj = new();
        // 128bit
        if (ksr.keyRAM[index].clen_ram_f == 2'b00) begin
          ccmp_obj.M   = 8;
          ccmp_obj.L   = 2;
          ccmp_obj.Key = new[16];
          ccmp_sec.MIC = new[8];
        end
        // 256bit
        else begin
          ccmp_obj.M   = 16;
          ccmp_obj.L   = 2;
          ccmp_obj.Key = new[32];
          ccmp_sec.MIC = new[16];
        end
        // Key
        foreach(ccmp_obj.Key[i])
          ccmp_obj.Key[i] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];

        ccmp_obj.aes_obj.gen_Key(ccmp_obj.Key,1'b1); // setup Key

        // Mac header
        ccmp_obj.macheader = new[mh.size()];
        data_cnt = 0;
        mh.put2array(data_cnt, ccmp_obj.macheader);
        // trimm HT control octets
        if (mh.ht_ctrl.size())
          ccmp_obj.macheader = new[ccmp_obj.macheader.size()-4](ccmp_obj.macheader);

        // Packet CCMP Header
        ccmp_obj.PN[0] = enc_frame[0];
        ccmp_obj.PN[1] = enc_frame[1];
        ccmp_obj.PN[2] = enc_frame[4];
        ccmp_obj.PN[3] = enc_frame[5];
        ccmp_obj.PN[4] = enc_frame[6];
        ccmp_obj.PN[5] = enc_frame[7];

        // save CCMP header paramaters
        ccmp_sec.IV.pn0_f  = enc_frame[0];
        ccmp_sec.IV.pn1_f  = enc_frame[1];
        ccmp_sec.eIV.pn2_f = enc_frame[4];
        ccmp_sec.eIV.pn3_f = enc_frame[5];
        ccmp_sec.eIV.pn4_f = enc_frame[6];
        ccmp_sec.eIV.pn5_f = enc_frame[7];
        ccmp_sec.IV.ext_iv_f = enc_frame[3][5];
        ccmp_sec.IV.key_id_f = enc_frame[3][7:6];

        // payload - CCMP header
        ccmp_obj.enc_data = new[enc_frame.size()-8];
        foreach (ccmp_obj.enc_data[i])
          ccmp_obj.enc_data[i] = enc_frame[i+8];

        // Nonce
        ccmp_obj.ConstructCCMnonce();

        // HT AAD Patching
        if (ppdu_format >= HT_MF) begin
          ccmp_obj.htMode = 1;
          if(ksr.keyRAM[index].spp_ram_f == 2'b10)
            ccmp_obj.sspCapable = 1;
        end

        // AAD
        ccmp_obj.ConstructAAD();
        // Dec
        ccmp_obj.CCMP_AES_Dec(ccmp_obj.CCMnonce, ccmp_obj.enc_data,0);
        // MIC
        ccmp_obj.CCMP_AES_MIC(ccmp_obj.CCMnonce, ccmp_obj.dec_data, ccmp_obj.AAD,0);

        // set decrypted data
        frame = new[enc_frame.size()];
        for (int i=0; i<4; i++)
          frame[i] = ccmp_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          frame[i+4] = ccmp_sec.eIV[i*8 +: 8]; //set eIV
        foreach (ccmp_obj.dec_data[i])
          frame[i+8] = ccmp_obj.dec_data[i]; // set decrypted data

        // store MIC to CCMP structure and frame
        foreach (ccmp_sec.MIC[i]) begin
          frame[i+8+ccmp_obj.dec_data.size()] = ccmp_obj.MIC[i];
          ccmp_sec.MIC[i] = ccmp_obj.MIC[i];
        end

        `uvm_info(get_type_name(), $sformatf("CCMP decryption done!"),UVM_LOW)

      end//CCMP

//*****************************************************************************
      WAPI: begin
//*****************************************************************************

        wapi_obj = new();

        wapi_obj.wpi_c.wpiHeader.frameControl = mh.frame_ctrl;
        wapi_obj.wpi_c.wpiHeader.addr1 = mh.addr[0];
        wapi_obj.wpi_c.wpiHeader.addr2 = mh.addr[1];
        wapi_obj.wpi_c.wpiHeader.addr3 = mh.addr[2];
        if (mh.addr.size() == 4) begin
          wapi_obj.wpi_c.wpiHeader.addr4 = mh.addr[3];
          wapi_obj.wpi_c.wpiHeader.address4Pres = 1;
        end
        if (mh.qos_ctrl.size()) begin
          wapi_obj.wpi_c.wpiHeader.qosFrame = 1;
          wapi_obj.wpi_c.wpiHeader.qosCF = mh.qos_ctrl[0];
        end
        wapi_obj.wpi_c.wpiHeader.seqControl = mh.seq_ctrl[0];
        wapi_obj.wpi_c.wpiHeader.keyIdx = enc_frame[0];
        wapi_sec.key_idx = enc_frame[0];

        for (int i=0; i<16; i++) begin
          wapi_sec.PN[i*8 +: 8] = enc_frame[i+2];
          wapi_obj.wpi_c.wpiHeader.pn[i*8 +: 8] = enc_frame[i+2];
          wapi_obj.wpi_c.wpiHeader.wpiEncryptionKey[i*8 +: 8] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];
          wapi_obj.wpi_c.wpiHeader.wpiIntegrityKey[i*8 +: 8] = ksr.keyRAM[index].int_wpi_key_ram_f[i*8 +: 8];
        end

        // encrypted data - header - footer bytes
        wapi_obj.wpi_c.wpiHeader.pduLength = enc_frame.size()-18-16;

        wapi_obj.InitWPIStruct();

        for (int i=0; i<wapi_obj.wpi_c.wpiHeader.pduLength; i++)
          for (int j=0; j<8; j++)
            wapi_obj.wpi_c.pduPData.vector[((wapi_obj.wpi_c.wpiHeader.pduLength*8)-1-(i*8))-j] = enc_frame[i+18][7-j];

        // encrypt
        wapi_obj.WAPIEncrypt();

        // save plain MIC
        foreach (wapi_sec.MIC[i])
          wapi_sec.MIC[i] = wapi_obj.wpi_c.eMIC[i*8 +: 8];

        // set decrypted data
        frame = new[enc_frame.size()];
        frame[0] = wapi_sec.key_idx;
        frame[1] = 8'b0; // reserved
        for (int i=0; i<16; i++)
          frame[i+2] = wapi_sec.PN[i*8 +: 8]; //set PN
        // set encrypted data
        for (int i=0; i<wapi_obj.wpi_c.wpiHeader.pduLength; i++) begin
          for (int j=0; j<8; j++) begin
            frame[i+18][7-j] = wapi_obj.wpi_c.pduEData[((wapi_obj.wpi_c.wpiHeader.pduLength * 8) - 1 - (i * 8)) - j];
          end
        end

        wapi_obj.wpi_c.pduPData.vector = wapi_obj.wpi_c.pduEData;
        wapi_obj.WAPIEncrypt;

        // set decrypted MIC data
        for (int i=0; i<16; i++) begin
          for (int j=0; j<8; j++) begin
            frame[frame.size() - 1 - i][7-j] = wapi_obj.wpi_c.pMIC[((128 - 1) - (i * 8)) - j];
          end
        end

        `uvm_info(get_type_name(), $sformatf("WAPI decryption done!"),UVM_LOW)

      end//WAPI

//*****************************************************************************
      GCMP: begin
//*****************************************************************************

        gcmp_obj = new();
        // 128bit
        if (ksr.keyRAM[index].clen_ram_f == 2'b00)
          gcmp_obj.Key = new[16];
        // 256bit
        else
          gcmp_obj.Key = new[32];

        // allocate memory for MIC
        gcmp_sec.MIC = new[16];

        // Key
        foreach(gcmp_obj.Key[i])
          gcmp_obj.Key[i] = ksr.keyRAM[index].encr_key_ram_f[i*8 +: 8];

        gcmp_obj.aes_obj.gen_Key(gcmp_obj.Key,1'b1); // setup Key

        // Mac header
        gcmp_obj.macheader = new[mh.size()];
        data_cnt = 0;
        mh.put2array(data_cnt, gcmp_obj.macheader);
        // trimm HT control octets
        if (mh.ht_ctrl.size())
          gcmp_obj.macheader = new[gcmp_obj.macheader.size()-4](gcmp_obj.macheader);

        // Packet GCMP Header
        gcmp_obj.PN[0] = enc_frame[0];
        gcmp_obj.PN[1] = enc_frame[1];
        gcmp_obj.PN[2] = enc_frame[4];
        gcmp_obj.PN[3] = enc_frame[5];
        gcmp_obj.PN[4] = enc_frame[6];
        gcmp_obj.PN[5] = enc_frame[7];

        // save GCMP header paramaters
        gcmp_sec.IV.pn0_f  = enc_frame[0];
        gcmp_sec.IV.pn1_f  = enc_frame[1];
        gcmp_sec.eIV.pn2_f = enc_frame[4];
        gcmp_sec.eIV.pn3_f = enc_frame[5];
        gcmp_sec.eIV.pn4_f = enc_frame[6];
        gcmp_sec.eIV.pn5_f = enc_frame[7];
        gcmp_sec.IV.ext_iv_f = enc_frame[3][5];
        gcmp_sec.IV.key_id_f = enc_frame[3][7:6];

        // payload - GCMP header, minus header and MIC bytes
        gcmp_obj.enc_data = new[enc_frame.size()-8-16];
        //foreach (gcmp_obj.enc_data[i])
        for (int i=8,j=0; i<enc_frame.size()-16; i++,j++)
          gcmp_obj.enc_data[j] = enc_frame[i];

        // Nonce
        gcmp_obj.ConstructGCMnonce();

        // HT AAD Patching
        if (ppdu_format >= HT_MF) begin
          gcmp_obj.htMode = 1;
          if(ksr.keyRAM[index].spp_ram_f == 2'b10)
            gcmp_obj.sspCapable = 1;
        end

        // AAD
        gcmp_obj.ConstructAAD();
        // Dec
        gcmp_obj.GCMP_Dec(gcmp_obj.GCMnonce, gcmp_obj.enc_data,0);
        // MIC
        gcmp_obj.GCMP_MIC(gcmp_obj.GCMnonce, gcmp_obj.enc_data, gcmp_obj.AAD,0);

        // set decrypted data
        frame = new[enc_frame.size()];
        for (int i=0; i<4; i++)
          frame[i] = gcmp_sec.IV[i*8 +: 8]; //set IV
        for (int i=0; i<4; i++)
          frame[i+4] = gcmp_sec.eIV[i*8 +: 8]; //set eIV
        foreach (gcmp_obj.dec_data[i])
          frame[i+8] = gcmp_obj.dec_data[i]; // set decrypted data

        // store MIC to GCMP structure and frame
        foreach (gcmp_sec.MIC[i]) begin
          frame[i+8+gcmp_obj.dec_data.size()] = gcmp_obj.MIC[i];
          gcmp_sec.MIC[i] = gcmp_obj.MIC[i];
        end

        `uvm_info(get_type_name(), $sformatf("GCMP decryption done!"),UVM_LOW)

      end//GCMP
    endcase

  endfunction : decrypt_frame

  //------------------------------------------------------------
  // custom print function
  //------------------------------------------------------------
  function void do_print(uvm_printer printer);
    super.do_print(printer);

    case (security_type)
      NULL_KEY, NO_SECURITY: begin end
      WEP: begin
        printer.print_string("WEP->IV", $sformatf("%p",wep_sec.IV));
        printer.print_int("WEP->ICV", wep_sec.ICV, $bits(wep_sec.ICV), UVM_HEX);
      end
      TKIP: begin
        printer.print_string("TKIP->IV", $sformatf("%p",tkip_sec.IV));
        printer.print_string("TKIP->eIV", $sformatf("%p",tkip_sec.eIV));
        printer.print_string("TKIP->MIC", $sformatf("%p",tkip_sec.MIC));
        printer.print_int("TKIP->ICV", tkip_sec.ICV, $bits(tkip_sec.ICV), UVM_HEX);
      end
      CCMP: begin
        printer.print_string("CCMP->IV", $sformatf("%p",ccmp_sec.IV));
        printer.print_string("CCMP->eIV", $sformatf("%p",ccmp_sec.eIV));
        printer.print_string("CCMP->MIC", $sformatf("%p",ccmp_sec.MIC));
      end
      WAPI: begin
        printer.print_string("WAPI->KEY IDX", $sformatf("%p",wapi_sec.key_idx));
        printer.print_int("WAPI->PN", wapi_sec.PN, $bits(wapi_sec.PN), UVM_HEX);
        printer.print_string("WAPI->MIC", $sformatf("%p",wapi_sec.MIC));
      end
      GCMP: begin
        printer.print_string("GCMP->IV", $sformatf("%p", gcmp_sec.IV));
        printer.print_string("GCMP->eIV", $sformatf("%p",gcmp_sec.eIV));
        printer.print_string("GCMP->MIC", $sformatf("%p",gcmp_sec.MIC));
      end
    endcase

  endfunction : do_print

endclass : security_wrapper_engine

`endif //SECURITY_WRAPPER_ENGINE_SV
