//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : container class for PPDU preamble and header
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef HE_TRIG_PPDU_SV
`define HE_TRIG_PPDU_SV

class HE_TRIG_PPDU extends PPDU_preamble_header;

  rand int        nsts_total;

  `uvm_object_utils_begin(HE_TRIG_PPDU)
    `uvm_field_int(nsts_total, UVM_DEFAULT)
  `uvm_object_utils_end

  function new (string name = "HE_TRIG_PPDU");
    super.new(name);
  endfunction

  //---------------------------------------------
  // constraints
  //---------------------------------------------

  constraint c_preamble_header {
    //------------------------------------------------------
    //  Constraints derived from Matlab config files
    //------------------------------------------------------
    solve ru_allocation before user_header_he;
    solve nsts_total before num_he_ltf;
//NOTE: Fields left random
    //  tx_pwr_level
    //  ru_tone_set_index
    //  feedback_status
      leg_length          inside {[500:1000]};
      leg_length % 3      == 1;
      leg_rate            == 4'b1011;
      he_siga_reserved    inside {[0:511]};
      uplink_flag         == 1'b1;
      pe_duration         == 1'b0;
      // calculate total number of NSTS
      foreach (user_header_he[i]) {
        if (i==0) {
          nsts_total == user_header_he[i].nss_f;
        }else{
          nsts_total == nsts_total + user_header_he[i].nss_f;
        }
      }
      // encoding of NHeLTF
      // 0 - 1x HE-LTF
      // 1 - 2x HE-LTF
      // 3 - 4x HE-LTF
      // 5 - 6x HE-LTF
      // 7 - 8x HE-LTF
      num_he_ltf >= nsts_total;
      if (doppler) {
        num_he_ltf inside {0, 1, 3};
      } else {
        num_he_ltf inside {0, 1, 3, 5, 7};
      }
      he_ltf_mode         inside {[0:1]};
      // The LDPC_EXTRA_SYMBOL parameter is not present
      // if the RU Allocation subfield indicates less
      // than a 484-tone RU; otherwise set to 1
      if (get_ru_type(ru_allocation) < RU484)
        ldpc_extra_symbol == 0;
      else
        ldpc_extra_symbol == 1;

      starting_sts_num    == 0;
      sounding            == 0;
      bss_color           inside {[0:63]};
      uplink_flag         inside {[0:1]};
      if (doppler == 0) midamble_periodicity == 0;
      continuous_tx       == 0;
      num_tx              == tx_num_func();
      soft ch_bw          inside {2'b00, 2'b01, 2'b10}; //20/40/80MHz

//NOTE: Preamble Type is reserved for formats other than NON-HT
      preamble_type      == 0;
      format_mod         == HE_TB;
//NOTE: time_of_departure_requested is set to measure time of departure by the PHY
      time_dep_req       == 0;
      gi_type            != 2'b11;
      // if 1 antenna on TX side, no STBC
      if (num_tx == 0 && tr == TX) {
        stbc == 0;
      }
      beamformed         == 0;

//NOTE: Set beam_change to 1, if the pre-HE-STF portion of the PPDU is differently mapped from HE-LTF1
      beam_change        == 1;
      txop_duration inside { [0:127] };

      /* HE LTF TYPE = 2'b00  - 1x HE-LTF
                                GI_TYPE = 1.6 us Only for Non OFDMA MU-MIMO frame
                       2'b01  - 2x HE_LTF
                                GI_TYPE = 1.6 us
                       2'b10  - 4x HE_LTF
                                GI_TYPE = 3.2 us*/
      he_ltf_type        inside {2'b01,2'b10};
      if (he_ltf_type == 2'b01) {
        gi_type == 2'b01;
      } else if (he_ltf_type == 2'b10) {
        gi_type == 2'b10;
      }
      partial_aid        == 0;
      group_id           == 0;
      abgn_mode          == MODE_802_11AX;
      service            == 0;
      num_extn_ss        == 0;
      n_user             == user_header_he.size();
      // DCM is not applied to STBC
      if (stbc == 1){
        dcm == 0;
      }
      foreach (user_header_he[i]) {
        user_header_he[i].dut_location_f == 0;
        user_header_he[i].user_position_f == 0;
        user_header_he[i].smm_index_f == 0;
        user_header_he[i].mcs_f <= MAX_MCS; //RTL constraint

        if (get_ru_type(ru_allocation) < RU242) {
          user_header_he[i].mcs_f inside { [0:9] };
        } else {
`ifndef RW_NX_LDPC_ENC
          user_header_he[i].mcs_f inside { [0:9] };
        }
        user_header_he[i].fec_coding_f == 0;
`else
          if (get_ru_type(ru_allocation) == RU484)
            user_header_he[i].fec_coding_f == 1;

          if (user_header_he[i].fec_coding_f == 1 && get_ru_type(ru_allocation) >= RU242)
            user_header_he[i].mcs_f inside { [0:11]};
          else
            user_header_he[i].mcs_f inside { [0:9] };
        }
`endif
        if (trigger_method) {//TRS
          // set PE duration to 16us
          user_header_he[i].pkt_ext_f == 3'b100;
        } else {
          user_header_he[i].pkt_ext_f[1:0] inside {[0:3]};
          user_header_he[i].pkt_ext_f[2]   inside {0,1}; // PE disambiguity
        }
        user_header_he[i].smm_index_f inside { [0:2], [8'h80:8'hff] };
        user_header_he[i].nss_f inside { [0:3] };
        user_header_he[i].staid_f == 0;
        if (user_header_he[i].mcs_f inside {0,1,3,4}) {
          dcm inside { [0:1] };
        } else {
          dcm == 0;
        }
      }
    //-----------------------------------------------------
    // RU allocation constraints
    //-----------------------------------------------------
    ru_allocation[0] == 0;

    // 20MHz
    if (ch_bw == 2'b00) {
      ru_allocation[7:1] inside {[0:8],[37:40],[53:54],61};
    }
    // 40MHz
    else if (ch_bw == 2'b01) {
`ifdef RW_NX_CHBW20
      if (primary_channel inside {0,2})
        ru_allocation[7:1] inside {[0:8],[37:40],[53:54]};
      else
        ru_allocation[7:1] inside {[9:17],[41:44],[55:56]};
      // RU restrictions for 20MHz operations
      ru_allocation[7:1] != 4;  // RU index 5
      ru_allocation[7:1] != 13; // RU index 14
    }
`else
      ru_allocation[7:1] inside {[0:17],[37:44],[53:56],61,62,65};
    }
`endif
    // 80MHz
    else if (ch_bw == 2'b10) {
`ifdef RW_NX_CHBW20
      if (primary_channel == 0)
        ru_allocation[7:1] inside {[0:8],[37:40],[53:54]};
      else if (primary_channel == 1)
        ru_allocation[7:1] inside {[10:12],[14:17],[42:44],56};
      else if (primary_channel == 2)
        ru_allocation[7:1] inside {[19:22],[24:26],[45:47],57};
      else
        ru_allocation[7:1] inside {[28:31],[33:36],[49:52],[59:60]};
      // RU restrictions for 20MHz operations
      //                  RU26
      ru_allocation[7:1] != 4;  // RU index 5
      ru_allocation[7:1] != 9;  // RU index 10
      ru_allocation[7:1] != 13; // RU index 14
      ru_allocation[7:1] != 18; // RU index 19
      ru_allocation[7:1] != 23; // RU index 24
      ru_allocation[7:1] != 27; // RU index 28
      ru_allocation[7:1] != 32; // RU index 33
      //                  RU52
      ru_allocation[7:1] != 41; // RU index 5
      ru_allocation[7:1] != 48; // RU index 12
      //                  RU106
      ru_allocation[7:1] != 55; // RU index 3
      ru_allocation[7:1] != 58; // RU index 6
`elsif RW_NX_CHBW4020
      if (primary_channel inside {0,1})
        ru_allocation[7:1] inside {[0:17],[37:44],[53:56],[61:62],65,67};
      else
        ru_allocation[7:1] inside {[19:36],[45:52],[57:60],[63:64],66,67};
`else
      ru_allocation[7:1] inside {[0:67]};
`endif
    }

    //-----------------------------------------------------
    //  Constraints based on the RTL configuration
    //-----------------------------------------------------
    // Indicates which antennas to be used for transmission
    `ifdef RW_TXRX_1X1
      antenna_set      == 1;
    `elsif RW_TXRX_2X2
      antenna_set      == 3;
    `endif

    `ifdef RW_NX_1024QAM_EN
      MAX_MCS == 11;
    `elsif RW_NX_256QAM_EN
      MAX_MCS == 9;
    `else
      MAX_MCS == 7;
    `endif

    foreach (user_header_he[i]) {
      user_header_he[i].nss_f == 0;
    }
    beamformed         == 0;
    `ifdef RW_NX_CHBW4020
      // matlab doesn't support STA40 CHBW80 frames
      ch_bw != 2'b10;
      ru_allocation[7:1] != 67; // RU996 excluded
    `endif
  }
  //---------------------------------------------
  // end of constraints
  //---------------------------------------------
  function void post_randomize();
    int  m;
    int  Nsym;
    real The_pre;
    int  Nma;
    int  Nhe_ltf;
    real The_ltf_sym;
    real Tsym;
    real Tma;
    int  Tpe;

    if (trigger_method == 0) begin
      m           = 2;
      Tsym        = 12.8+2**gi_type*0.8;
      Nhe_ltf     = num_he_ltf + 1;
      The_ltf_sym = 2**he_ltf_type*3.2 + 2**gi_type*0.8;
      The_pre     = 20 + Nhe_ltf*The_ltf_sym;
      Tma         = ((midamble_periodicity+1)*10)*Tsym + Nhe_ltf*The_ltf_sym;
      if (doppler)
        Nma = $floor((((leg_length + m + 3)/3.0)*4 - The_pre - (user_header_he[0].pkt_ext_f[2]+2)*Tsym)/Tma);
      else
        Nma = 0;
      // calculate number of symbols                                                                |      PE disambiguity
      Nsym        = $floor((((leg_length + m + 3)/3.0)*4 - The_pre - Nma*Nhe_ltf*The_ltf_sym)/Tsym) - user_header_he[0].pkt_ext_f[2];
      // calculate PE duration
      Tpe         = $floor((((leg_length + m + 3)/3.0)*4 - The_pre - Nsym*Tsym - Nma*Nhe_ltf*The_ltf_sym)/4.0) * 4;

      `uvm_info(get_type_name(),$sformatf("Length = %0d, Tsym = %0f, Nhe_ltf = %0d, The_ltf_sym = %0f, The_pre = %0f, Tma = %0f, Nma = %0d, Nsym = %0d, Tpe = %0f",
        leg_length, Tsym, Nhe_ltf, The_ltf_sym, The_pre, Tma, Nma, Nsym, Tpe),UVM_DEBUG)

      // disable PE disambiguity for frame which has Tpe above 16us
      if (Tpe > 16) begin
        user_header_he[0].pkt_ext_f[2] = 0;
        `uvm_info(get_type_name(),$sformatf("Tpe exceeds 16us, PE disambiguity set to 0!"),UVM_DEBUG)
      end
    end
  endfunction : post_randomize

endclass : HE_TRIG_PPDU

`endif //HE_TRIG_PPDU_SV
