//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthou $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 32750 $
// $Date: 2017-12-01 17:34:43 +0100 (Fri, 01 Dec 2017) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : container class for PPDU preamble and header
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef PPDU_PREAMBLE_HEADER_SV
`define PPDU_PREAMBLE_HEADER_SV

class PPDU_preamble_header extends uvm_object;

    rand user_header_s      user_header[];
    rand user_header_he_s   user_header_he[];
    rand bit [7:0]          tx_pwr_level;
    rand bit                sounding;         /* 1-SOUNDING, 0-NOT_SOUNDING */
    rand bit                smoothing;
    rand bit                continuous_tx;
    rand bit [2:0]          ch_bw;
    rand bit [7:0]          antenna_set;
    rand bit                preamble_type;    /* 0-short, 1-long (DSSS/CCK) */
    rand format_mod_e       format_mod;
    rand bit [1:0]          num_extn_ss;
    rand bit [11:0]         leg_length;
    rand bit [3:0]          leg_rate;
    rand bit [15:0]         service;
    rand bit                aggregated;
    rand bit                doze_not_allowed;
    rand bit [2:0]          num_tx;
    rand bit                stbc;             /* Space Time Block Coding */
    rand bit                beamformed;
    rand bit [8:0]          partial_aid;
    rand bit [5:0]          group_id;
    rand bit                l_sig_valid;
    rand bit [2:0]          chbw_in_non_ht;   /* channel bandwidth in non HT */
    rand bit                dyn_bw;           /* dynamic bandwidth */
    rand bit                first_user;       /* for MU-MIMO */
    rand bit [2:0]          n_user;           /* Number of User */
    rand bit [7:0]          rssi;             /* Receive signal strength indicator during Data */
    rand bit [7:0]          rssi_legacy;      /* Receive signal strength indicator during legacy preamble */
    rand bit [7:0]          rcpi[4];          /* Receive channel power indicator */
    rand bit [7:0]          evm[4];           /* Error vector magnitude */
    rand bit                band5G;           /* 0 - 2.4GHz, 1 - 5GHz */
    rand abgn_mode_e        abgn_mode;        /* operating mode of MAC HW */
         int                user_order[`MAX_USER_NUM]; // first element is the user with longest time on air
    rand transaction_e      tr;               /* indication which transaction is performed */
    rand bit [1:0]          mu_mimo_userid;   /* user ID inside MU-MIMO frame when Rx */
    rand bit                is_ndp;           /* indication that this preamle if part of NDP frame */
    rand bit                time_dep_req;         /* Time of Departure requested */
    rand bit                trigger_responding;
    rand bit [1:0]          gi_type;
    rand bit                uplink_flag;          /* UPLINK FLAG */
    rand bit                beam_change;          /* Beam Change */
    rand bit                dcm;                  /* Dual Carrier Modulation */
    rand bit [1:0]          he_ltf_type;          /* Type of HE-LTF */
    rand bit                doppler;              /* Doppler */
    rand bit [5:0]          bss_color;            /* BSS Color */
    rand bit                midamble_periodicity; /* If DOPPLER = 1 it indicates the midamble periodicity in number of OFDM symbols in the Data field */
    rand bit [6:0]          txop_duration;
    rand bit [3:0]          spatial_reuse[4];
    rand bit [8:0]          he_siga_reserved;     /* Indicates the Reserved field setting for HE-SIG-A2 */
    rand bit [2:0]          num_he_ltf;           /* Number of HE LTF */
    rand bit                he_ltf_mode;
    rand bit                ldpc_extra_symbol;
    rand bit [2:0]          starting_sts_num;
    rand bit                trigger_method;
    rand bit [6:0]          ru_tone_set_index;
    rand bit                feedback_status;
    rand bit [3:0]          pe_duration;
    rand bit                sig_b_comp_mode; /* SIG B Compression mode */
    rand bit                dcm_sig_b;       /* Dual Carrier Modulation on SIG B*/
    rand bit [2:0]          mcs_sig_b;       /* Modulation and coding scheme for HE-SIGB */
    rand bit [7:0]          ru_allocation;   /* RU Allocation */
    rand bit [1:0]          primary_channel; /* This indicates, which primary channel is used */
    rand int unsigned       MAX_MCS;         /* RTL constraint for maximum MCS value */
    rand bit [2:0]          tmmss;           /* Minimum MPDU start spacing */
    rand ru_type_e          ru_type;         /* RU allocation size HE-MU */
    rand bit                ru_with_more_users; /* on WLAN testing we want more users inside RU */
    rand user_header_he_s   user_header_he_sec[]; /* for secondary users in HE-MU */

  `uvm_object_utils_begin(PPDU_preamble_header)
    `uvm_field_array_int(user_header, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_array_int(user_header_he, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(ru_with_more_users, UVM_DEFAULT | UVM_NOCOMPARE)
`ifdef FULL_WLAN_SYSTEM
    `uvm_field_array_int(user_header_he_sec, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
`endif//FULL_WLAN_SYSTEM
    `uvm_field_int(tx_pwr_level, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(sounding, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(smoothing, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(continuous_tx, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(ch_bw, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(antenna_set, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(preamble_type, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_enum(format_mod_e, format_mod, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(num_extn_ss, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(leg_length, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(leg_rate, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(service, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(aggregated, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(doze_not_allowed, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(num_tx, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(stbc, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(beamformed, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(partial_aid, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(group_id, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(l_sig_valid, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(chbw_in_non_ht, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(dyn_bw, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(first_user, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(n_user, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(rssi, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(rssi_legacy, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_sarray_int(rcpi, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_sarray_int(evm, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY  | UVM_NOPRINT)
    `uvm_field_enum(transaction_e, tr, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(mu_mimo_userid, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(is_ndp, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(time_dep_req, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(trigger_responding, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(gi_type, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY)
    `uvm_field_int(uplink_flag, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(beam_change, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(dcm, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(he_ltf_type, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(doppler, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(midamble_periodicity, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(bss_color, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(txop_duration, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_sarray_int(spatial_reuse, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(he_siga_reserved, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(num_he_ltf, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(he_ltf_mode, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(ldpc_extra_symbol, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(starting_sts_num, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(trigger_method, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(ru_tone_set_index, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(feedback_status, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(pe_duration, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(sig_b_comp_mode, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(dcm_sig_b, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(mcs_sig_b, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(ru_allocation, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_enum(ru_type_e, ru_type, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(primary_channel, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
    `uvm_field_int(tmmss, UVM_DEFAULT | UVM_NOCOMPARE | UVM_NOCOPY | UVM_NOPRINT)
  `uvm_object_utils_end

  function new (string name = "PPDU_preamble_header");
    super.new(name);
  endfunction

  //---------------------------------------------
  // constraints begin
  //---------------------------------------------
  constraint c_user_header {
    solve format_mod before user_header;
    solve format_mod before user_header_he;

    if (format_mod <= VHT) {
      user_header.size() > 0;
      user_header_he.size() == 0;
    } else {
      user_header.size() == 0;
      user_header_he.size() > 0;
    }
  }

  constraint c_secondary_he_users {
`ifdef FULL_WLAN_SYSTEM
    // create additional users that will be inside secondary channel
    if (ru_with_more_users && ch_bw > 3'b000) {
      user_header_he_sec.size() inside {[2:4]};
     } else {
      user_header_he_sec.size() == 0;
     }
`else
    ru_with_more_users == 0;
    user_header_he_sec.size() == 0;
`endif//FULL_WLAN_SYSTEM
  }

  constraint c_abgn_mode {
    solve abgn_mode before band5G;
    solve format_mod before abgn_mode;

    if (format_mod == VHT){
      abgn_mode == MODE_802_11AC;
    } else if (format_mod == HT_MF || format_mod == HT_GF) {
      abgn_mode inside { MODE_802_11N_2_4GHZ,
                         MODE_802_11N_5GHZ,
                         MODE_802_11AC };
    }
    else if (format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) {
      abgn_mode == MODE_802_11AX;
    } else {
      abgn_mode inside { MODE_802_11B       ,
                         MODE_802_11A       ,
                         MODE_802_11G       ,
                         MODE_802_11N_2_4GHZ,
                         MODE_802_11N_5GHZ  ,
                         MODE_802_11AC      ,
                         MODE_802_11AX};
    }
    // when 802.11n and 802.11ac mode in 5GHz
    if (abgn_mode inside {MODE_802_11N_5GHZ, MODE_802_11AC, MODE_802_11AX}) {
      band5G == 1;
    } else {
      band5G == 0;
    }
  }

  // in transmission, antenna_set can have maximum four (4) antennae active
  // (four "1" set in entire byte)
  constraint c_antenna_set {
    $countones(antenna_set) <= 4;
  }

  // service field should be 0 except in bandwith signaling
  // Disable constraint when using bandwidth signaling
  constraint c_service {
    service == 0;
  }

  // first user is relevant for MU in VHT
  constraint c_first_user {
    if (tr == RX && format_mod == VHT && user_header.size() > 1 && mu_mimo_userid == 0)
      first_user == 1;
    else
      first_user == 0;
  }

  constraint c_trigger_responding {
`ifdef STANDALONE_PHY
    if (format_mod inside {NON_HT_DUP_OFDM, NON_HT})
      trigger_responding inside {0, 1};
    else
`endif//STANDALONE_PHY
      trigger_responding == 0;
  }

  // User ID for MU-MIMO reception
  constraint c_mumimo_userid_ix {
    solve user_header before mu_mimo_userid;
    solve user_header_he before mu_mimo_userid;
    if (tr == TX) {
      mu_mimo_userid == 0;
    } else {
      if (format_mod == VHT) {
        mu_mimo_userid < user_header.size();
      } else if (format_mod == HE_MU) {
        mu_mimo_userid < user_header_he.size();
      } else {
        mu_mimo_userid == 0;
      }
    }
  }

  constraint c_ch_bw_in_non_ht {
    solve ch_bw before chbw_in_non_ht;
    chbw_in_non_ht == ch_bw;
  }

  constraint c_gi_type {
    solve format_mod before gi_type;
    if (format_mod inside {NON_HT, NON_HT_DUP_OFDM}){
      gi_type inside {  2'b00,  // short preamble
                        2'b10}; // long preamble
    }
    else if (format_mod inside {HT_MF, HT_GF, VHT}){
      gi_type inside {  2'b00,  // long GI
                        2'b01}; // short GI
    }
    else {
      gi_type inside {  2'b00,  // 0.8us
                        2'b01,  // 1.6us
                        2'b10}; // 3.2us
    }
  }

  constraint c_pe_duration {
    pe_duration inside {  3'b000,   // 0 us
                          3'b001,   // 4 us
                          3'b010,   // 8 us
                          3'b011,   // 12 us
                          3'b100    // 16 us
    };
  }

  constraint c_trigger_method {
    // $TODO Disable TRS from randomization
    //if (format_mod != HE_TB) {
      trigger_method == 0;
    //}
  }

  constraint c_mcs_sig_b {
    mcs_sig_b inside {[0:5]};
  }

  /* Receive signal strength indicator during Data */
  constraint c_rssi {
    rssi inside {[195:255]};
  }

  /* Receive signal strength indicator during legacy preamble */
  constraint c_rssi_legacy {
    rssi_legacy inside {[195:255]};
  }

  constraint c_primary_channel {
    solve ch_bw before primary_channel;

    if (ch_bw == 3'b010)
      primary_channel inside {[0:3]};
    else if (ch_bw == 3'b001)
      primary_channel inside {[0:1]};
    else
      primary_channel == 0;
  }

  //Determines the minimum time between the start of adjacent MPDUs within an
  //A-MPDU that the STA can receive, measured at the PHY SAP. See 10.13.3.
  //Set to 0 for no restriction
  //Set to 1 for 1/4 us
  //Set to 2 for 1/2 us
  //Set to 3 for 1 us
  //Set to 4 for 2 us
  //Set to 5 for 4 us
  //Set to 6 for 8 us
  //Set to 7 for 16 us
  constraint c_tmmss {
    tmmss inside {[0:7]};
    soft tmmss == 7; // Force to 16us as it is the default SW configuration
  }

  // TXOP duration on TX will be set to UNSPECIFIED
  constraint c_txop_duration {
    if (tr == TX) txop_duration == 127;
  }

  //---------------------------------------------
  // constraints end
  //---------------------------------------------

  function string decode_leg_rate_string();
    case (leg_rate)
      4'b0000 :   return "1";
      4'b0001 :   return "2";
      4'b0010 :   return "5.5";
      4'b0011 :   return "11";
      4'b1011 :   return "6";
      4'b1111 :   return "9";
      4'b1010 :   return "12";
      4'b1110 :   return "18";
      4'b1001 :   return "24";
      4'b1101 :   return "36";
      4'b1000 :   return "48";
      4'b1100 :   return "54";
    endcase
  endfunction: decode_leg_rate_string

  virtual function void do_print(uvm_printer printer);
    spatial_reuse_e sr;

    super.do_print(printer);
    printer.print_string("leg_rate", {decode_leg_rate_string()," Mbps"});
    foreach (user_header[i])
      printer.print_string("user header", $sformatf("[%0d] %p",i,user_header[i]));
    foreach (user_header_he[i])
      printer.print_string("user_header_he", $sformatf("[%0d] %p",i,user_header_he[i]));
    foreach (user_header_he_sec[i])
      printer.print_string("user_header_he_sec", $sformatf("[%0d] %p",i,user_header_he_sec[i]));

    if (format_mod inside {HT_MF, HT_GF})
      printer.print_int("aggregated", aggregated, 1, UVM_HEX);

    if (format_mod inside {VHT, HE_MU, HE_TB})
      printer.print_int("n_user", n_user, 3, UVM_HEX);

    if (format_mod == VHT) begin
      printer.print_int("partial_aid", partial_aid, $bits(partial_aid), UVM_HEX);
      printer.print_int("group_id", group_id, $bits(group_id), UVM_HEX);
    end

    if (format_mod inside {VHT,HE_MU} && (user_header.size() > 1 || user_header_he.size() > 1))
      printer.print_int("mu_mimo_userid", mu_mimo_userid, $bits(mu_mimo_userid), UVM_HEX);

    if (format_mod inside {HE_SU, HE_EXT_SU, HE_TB, HE_MU}) begin
      printer.print_int("uplink_flag", uplink_flag, $bits(uplink_flag), UVM_HEX);
      printer.print_int("beam_change", beam_change, $bits(beam_change), UVM_HEX);
      printer.print_int("dcm", dcm, $bits(dcm), UVM_HEX);
      printer.print_int("he_ltf_type", he_ltf_type, $bits(he_ltf_type), UVM_HEX);
      printer.print_int("doppler", doppler, $bits(doppler), UVM_HEX);
      printer.print_int("midamble_periodicity", midamble_periodicity, $bits(midamble_periodicity), UVM_HEX);
      printer.print_int("bss_color", bss_color, $bits(bss_color), UVM_HEX);
      printer.print_int("txop_duration", txop_duration, $bits(txop_duration), UVM_HEX);
      printer.print_int("he_siga_reserved", he_siga_reserved, $bits(he_siga_reserved), UVM_HEX);
      if (format_mod == HE_TB) printer.print_int("ldpc_extra_symbol", ldpc_extra_symbol, $bits(ldpc_extra_symbol), UVM_HEX);
      printer.print_int("starting_sts_num", starting_sts_num, $bits(starting_sts_num), UVM_HEX);
      printer.print_int("trigger_method", trigger_method, $bits(trigger_method), UVM_HEX);
      printer.print_int("ru_tone_set_index", ru_tone_set_index, $bits(ru_tone_set_index), UVM_HEX);
      printer.print_int("feedback_status", feedback_status, $bits(feedback_status), UVM_HEX);
      if (tr == RX) printer.print_int("pe_duration", pe_duration, $bits(pe_duration), UVM_HEX);
    end

    if (format_mod == HE_TB) begin
      printer.print_int("he_ltf_mode", he_ltf_mode, $bits(he_ltf_mode), UVM_HEX);
      foreach (spatial_reuse[i]) begin
        sr = spatial_reuse_e'(spatial_reuse[i]);
        printer.print_string($sformatf("spatial_reuse[%0d]",i), sr.name());
      end
      printer.print_int("num_he_ltf", num_he_ltf, $bits(num_he_ltf), UVM_HEX);
    end
    else if (format_mod inside {HE_SU, HE_EXT_SU, HE_MU}) begin
      sr = spatial_reuse_e'(spatial_reuse[0]);
      printer.print_string($sformatf("spatial_reuse"), sr.name());
    end

    if (format_mod == HE_MU) begin
      printer.print_int("num_he_ltf", num_he_ltf, $bits(num_he_ltf), UVM_HEX);
      printer.print_int("sig_b_comp_mode", sig_b_comp_mode, $bits(sig_b_comp_mode), UVM_HEX);
      printer.print_int("dcm_sig_b      ", dcm_sig_b      , $bits(dcm_sig_b      ), UVM_HEX);
      printer.print_int("mcs_sig_b      ", mcs_sig_b      , $bits(mcs_sig_b      ), UVM_HEX);
      if (tr == RX)
        printer.print_string("RU type", $sformatf("%s",ru_type.name()));
    end

    if (format_mod inside {HE_MU, HE_TB}) begin
      printer.print_int("ru_allocation  ", ru_allocation  , $bits(ru_allocation  ), UVM_DEC);
    end

    if (tr == RX) begin
      printer.print_int("RSSI", rssi, 8, UVM_HEX);
      printer.print_int("rssi_legacy", rssi_legacy, 8, UVM_HEX);
      if (user_header.size() > 1 && format_mod == VHT)
        printer.print_int("first_user", first_user, 1, UVM_HEX);
    end

    printer.print_int("primary_channel", primary_channel, $bits(primary_channel), UVM_HEX);
    printer.print_string("Tmmss", $sformatf("%0.2f us",get_tmmss()));
  endfunction : do_print

  virtual function bit do_compare(uvm_object rhs, uvm_comparer comparer);
    PPDU_preamble_header that;
    int rx_userid;
    int tx_userid;

    if (!$cast(that, rhs)) begin
      `uvm_fatal(get_type_name(), $sformatf("Failed cast from 'rhs' to 'that'"))
    end

    do_compare = super.do_compare(rhs, comparer);
    do_compare &= comparer.compare_field("sounding", this.sounding, that.sounding, $bits(this.sounding));
    // Enable when feature is added
    //do_compare &= comparer.compare_field("time_dep_req", this.time_dep_req, that.time_dep_req, $bits(this.time_dep_req));
    if (format_mod == VHT) begin
      do_compare &= comparer.compare_field("beamformed", this.beamformed, that.beamformed, $bits(this.beamformed));
      do_compare &= comparer.compare_field("doze_not_allowed", this.doze_not_allowed, that.doze_not_allowed, $bits(this.doze_not_allowed));
    end

    if (format_mod inside {HT_MF, HT_GF})begin
      do_compare &= comparer.compare_field("smoothing", this.smoothing, that.smoothing, $bits(this.smoothing));
      do_compare &= comparer.compare_field("aggregated", this.aggregated, that.aggregated, $bits(this.aggregated));
    end

    if ((format_mod inside {VHT,HE_MU} && tr == TX) || (format_mod == HE_TB && tr == RX)) begin
      //TODO: uncomment when RTL supports this
      //do_compare &= comparer.compare_field("n_user", this.n_user, that.n_user, $bits(this.n_user));
    end


    // Check for preamble type only if it's a DSSS frame
    if (format_mod == NON_HT && leg_rate inside {[0:3]})
      do_compare &= comparer.compare_field("preamble_type", this.preamble_type, that.preamble_type, $bits(this.preamble_type));

`ifndef MAC_STANDALONE
    if (this.format_mod != NON_HT_DUP_OFDM && tr == RX) begin
      do_compare &= comparer.compare_field("ch_bw", this.ch_bw, that.ch_bw, $bits(this.ch_bw));
      do_compare &= comparer.compare_field("format_mod", this.format_mod, that.format_mod, $bits(this.format_mod));
    end
    else if (tr == TX) begin
      do_compare &= comparer.compare_field("ch_bw", this.ch_bw, that.ch_bw, $bits(this.ch_bw));
      do_compare &= comparer.compare_field("format_mod", this.format_mod, that.format_mod, $bits(this.format_mod));
    end
`elsif
      do_compare &= comparer.compare_field("ch_bw", this.ch_bw, that.ch_bw, $bits(this.ch_bw));
      do_compare &= comparer.compare_field("format_mod", this.format_mod, that.format_mod, $bits(this.format_mod));
`endif

    do_compare &= comparer.compare_field("num_extn_ss", this.num_extn_ss, that.num_extn_ss, $bits(this.num_extn_ss));

    if (format_mod inside {NON_HT,NON_HT_DUP_OFDM,HT_GF,HT_MF,VHT})
      do_compare &= comparer.compare_field("gi_type", this.gi_type[0], that.gi_type[0], $bits(this.gi_type[0]));
    else
      do_compare &= comparer.compare_field("gi_type", this.gi_type, that.gi_type, $bits(this.gi_type));

    do_compare &= comparer.compare_field("stbc", this.stbc, that.stbc, $bits(this.stbc));
    do_compare &= comparer.compare_field("partial_aid", this.partial_aid, that.partial_aid, $bits(this.partial_aid));
    do_compare &= comparer.compare_field("group_id", this.group_id, that.group_id, $bits(this.group_id));

    if (format_mod != HT_GF) begin
      // legacy rate and length fields are not present in HT-GF RXVECTOR (IEEE.802.11)
      do_compare &= comparer.compare_field("leg_rate", this.leg_rate, that.leg_rate, $bits(this.leg_rate));
      if (!(format_mod inside {VHT, HE_MU}) && tr == TX) begin
        // legacy length not present in VHT, but test that legacy rate is fixed to 6MBps
        do_compare &= comparer.compare_field("leg_length", this.leg_length, that.leg_length, $bits(this.leg_length));
      end
    end

    if (user_header.size() > 1 && tr == RX && format_mod == VHT) begin
    // first_user should be compared only for MU-MIMO. In SU this field is reserved in RXVECTOR
      do_compare &= comparer.compare_field("first_user", (mu_mimo_userid == 2'b00)?1:0, that.first_user, $bits(this.first_user));
    end

    do_compare &= comparer.compare_field("antenna_set", this.antenna_set, that.antenna_set, $bits(this.antenna_set));

    if (format_mod == HE_MU) begin
      do_compare &= comparer.compare_field($sformatf("sig_b_comp_mode"),
                                            this.sig_b_comp_mode, that.sig_b_comp_mode, $bits(sig_b_comp_mode));

      do_compare &= comparer.compare_field($sformatf("dcm_sig_b"),
                                           this.dcm_sig_b, that.dcm_sig_b, $bits(this.dcm_sig_b));

      do_compare &= comparer.compare_field($sformatf("mcs_sig_b"),
                                            this.mcs_sig_b, that.mcs_sig_b, $bits(this.mcs_sig_b));
    end

    if ((format_mod inside {HE_MU, HE_TB}) && (tr == TX)) begin
      do_compare &= comparer.compare_field($sformatf("ru_allocation"),
                                            this.ru_allocation, that.ru_allocation, $bits(this.ru_allocation));
    end

    if (user_header.size() > 1 && tr == RX && format_mod == VHT) begin
      //MU-MIMO Rx, only 1 user will be received - find this user first
      rx_userid = mu_mimo_userid;
      do_compare &= comparer.compare_field($sformatf("user_header[%0d].fec_coding_f", rx_userid),
                                            this.user_header[rx_userid].fec_coding_f,
                                            that.user_header[0].fec_coding_f,
                                            $bits(this.user_header[rx_userid].fec_coding_f)
                                          );

      do_compare &= comparer.compare_field($sformatf("user_header[%0d].mcs_f", rx_userid),
                                            this.user_header[rx_userid].mcs_f,
                                            that.user_header[0].mcs_f,
                                            $bits(this.user_header[rx_userid].mcs_f)
                                          );

      do_compare &= comparer.compare_field($sformatf("user_header[%0d].ht_length_f", rx_userid),
                                            this.user_header[rx_userid].ht_length_f,
                                            that.user_header[0].ht_length_f,
                                            $bits(this.user_header[rx_userid].ht_length_f)
                                          );

      do_compare &= comparer.compare_field($sformatf("user_header[%0d].user_position_f", rx_userid),
                                            this.user_header[rx_userid].user_position_f,
                                            rx_userid,// that.user_header[0].user_position_f,
                                            $bits(this.user_header[rx_userid].user_position_f)
                                          );

      do_compare &= comparer.compare_field($sformatf("user_header[%0d].num_sts_f", rx_userid),
                                            this.user_header[rx_userid].num_sts_f,
                                            that.user_header[0].num_sts_f,
                                            $bits(this.user_header[rx_userid].num_sts_f)
                                          );

    end
    else begin
      foreach(user_header[i]) begin
        tx_userid = user_order[i];

        do_compare &= comparer.compare_field($sformatf("user_header[%0d].fec_coding_f", i),
                                              this.user_header[tx_userid].fec_coding_f,
                                              that.user_header[i].fec_coding_f,
                                              $bits(this.user_header[i].fec_coding_f)
                                            );

        do_compare &= comparer.compare_field($sformatf("user_header[%0d].mcs_f", i),
                                              this.user_header[tx_userid].mcs_f,
                                              that.user_header[i].mcs_f,
                                              $bits(this.user_header[i].mcs_f)
                                            );

        do_compare &= comparer.compare_field($sformatf("user_header[%0d].ht_length_f", i),
                                              this.user_header[tx_userid].ht_length_f,
                                              that.user_header[i].ht_length_f,
                                              $bits(this.user_header[i].ht_length_f)
                                            );

        do_compare &= comparer.compare_field($sformatf("user_header[%0d].smm_index_f", i),
                                            this.user_header[tx_userid].smm_index_f,
                                            that.user_header[i].smm_index_f,
                                            $bits(this.user_header[i].smm_index_f)
                                          );

        // compare only when MU-MIMO
        if (format_mod == VHT && user_header.size() > 1) begin
          do_compare &= comparer.compare_field($sformatf("user_header[%0d].user_position_f", i),
                                              this.user_header[tx_userid].user_position_f,
                                              that.user_header[i].user_position_f,
                                              $bits(this.user_header[i].user_position_f)
                                            );
        end
        else if (format_mod == VHT) begin
          // in LM and HT this field is reserved in TXVECTOR
          do_compare &= comparer.compare_field($sformatf("user_header[%0d].num_sts_f", i),
                                                this.user_header[tx_userid].num_sts_f,
                                                that.user_header[i].num_sts_f,
                                                $bits(this.user_header[i].num_sts_f)
                                              );
        end //format_mod == VHT
      end // foreach
    end // if-else

    if (user_header_he.size() > 1 && tr == RX && format_mod == HE_MU) begin
      //MU-MIMO Rx, only 1 user will be received - find this user first
      rx_userid = mu_mimo_userid;
      do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].fec_coding_f", rx_userid),
                                            this.user_header_he[rx_userid].fec_coding_f,
                                            that.user_header_he[0].fec_coding_f,
                                            $bits(this.user_header_he[rx_userid].fec_coding_f)
                                          );

      do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].mcs_f", rx_userid),
                                            this.user_header_he[rx_userid].mcs_f,
                                            that.user_header_he[0].mcs_f,
                                            $bits(this.user_header_he[rx_userid].mcs_f)
                                          );

      do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].he_length_f", rx_userid),
                                            this.user_header_he[rx_userid].he_length_f,
                                            that.user_header_he[0].he_length_f,
                                            $bits(this.user_header_he[rx_userid].he_length_f)
                                          );

      do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].nss_f", rx_userid),
                                            this.user_header_he[rx_userid].nss_f,
                                            that.user_header_he[0].nss_f,
                                            $bits(this.user_header_he[rx_userid].nss_f)
                                          );
    end
    else if (format_mod inside {HE_SU, HE_MU, HE_EXT_SU, HE_TB}) begin
      foreach(user_header_he[i]) begin
        tx_userid = 0;

        do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].fec_coding_f", i),
                                              this.user_header_he[tx_userid].fec_coding_f,
                                              that.user_header_he[i].fec_coding_f,
                                              $bits(this.user_header_he[i].fec_coding_f)
                                            );

        do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].mcs_f", i),
                                              this.user_header_he[tx_userid].mcs_f,
                                              that.user_header_he[i].mcs_f,
                                              $bits(this.user_header_he[i].mcs_f)
                                            );
        if (tr == TX) begin
          do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].he_length_f", i),
                                                this.user_header_he[tx_userid].he_length_f,
                                                that.user_header_he[i].he_length_f,
                                                $bits(this.user_header_he[i].he_length_f)
                                              );
        end

        if (format_mod == HE_MU) begin
          do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].user_position_f", i),
                                                this.user_header_he[tx_userid].user_position_f,
                                                that.user_header_he[i].user_position_f,
                                                $bits(this.user_header_he[i].user_position_f)
                                              );
        end

        do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].smm_index_f", i),
                                            this.user_header_he[tx_userid].smm_index_f,
                                            that.user_header_he[i].smm_index_f,
                                            $bits(this.user_header_he[i].smm_index_f)
                                            );

        do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].nss_f", i),
                                              this.user_header_he[tx_userid].nss_f,
                                              that.user_header_he[i].nss_f,
                                              $bits(this.user_header_he[i].nss_f)
                                            );
        if (tr == TX) begin
          do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].pkt_ext_f", i),
                                                this.user_header_he[tx_userid].pkt_ext_f,
                                                that.user_header_he[i].pkt_ext_f,
                                                $bits(this.user_header_he[i].pkt_ext_f)
                                              );
        end

        if ((format_mod == HE_MU) && (tr == TX)) begin
          do_compare &= comparer.compare_field($sformatf("user_header_he[%0d].staid_f", i),
                                                this.user_header_he[tx_userid].staid_f,
                                                that.user_header_he[i].staid_f,
                                                $bits(this.user_header_he[i].staid_f)
                                              );
        end
      end // foreach
    end

    if (format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin
      do_compare &= comparer.compare_field("uplink_flag", this.uplink_flag, that.uplink_flag, $bits(this.uplink_flag));
      do_compare &= comparer.compare_field("beam_change", this.beam_change, that.beam_change, $bits(this.beam_change));
      do_compare &= comparer.compare_field("dcm", this.dcm, that.dcm, $bits(this.dcm));
      do_compare &= comparer.compare_field("he_ltf_type", this.he_ltf_type, that.he_ltf_type, $bits(this.he_ltf_type));
      do_compare &= comparer.compare_field("doppler", this.doppler, that.doppler, $bits(this.doppler));
      do_compare &= comparer.compare_field("bss_color", this.bss_color, that.bss_color, $bits(this.bss_color));
      do_compare &= comparer.compare_field("gi_type", this.gi_type, that.gi_type, $bits(this.gi_type));
      do_compare &= comparer.compare_field("txop_duration", this.txop_duration, that.txop_duration, $bits(this.txop_duration));
      if (format_mod == HE_TB) begin
        foreach (this.spatial_reuse[i])
          do_compare &= comparer.compare_field("spatial_reuse", this.spatial_reuse[i], that.spatial_reuse[i], $bits(this.spatial_reuse[i]));
        do_compare &= comparer.compare_field("trigger_method", this.trigger_method, that.trigger_method, $bits(this.trigger_method));
        do_compare &= comparer.compare_field("ru_tone_set_index", this.ru_tone_set_index, that.ru_tone_set_index, $bits(this.ru_tone_set_index));
        do_compare &= comparer.compare_field("he_ltf_mode", this.he_ltf_mode, that.he_ltf_mode, $bits(this.he_ltf_mode));
        do_compare &= comparer.compare_field("ldpc_extra_symbol", this.ldpc_extra_symbol, that.ldpc_extra_symbol, $bits(this.ldpc_extra_symbol));
        do_compare &= comparer.compare_field("feedback_status", this.feedback_status, that.feedback_status, $bits(this.feedback_status));
        do_compare &= comparer.compare_field("he_siga_reserved", this.he_siga_reserved, that.he_siga_reserved, $bits(this.he_siga_reserved));
        do_compare &= comparer.compare_field("num_he_ltf", this.num_he_ltf, that.num_he_ltf, $bits(this.num_he_ltf));
      end
      else begin
        do_compare &= comparer.compare_field("spatial_reuse", this.spatial_reuse[0], that.spatial_reuse[0], $bits(this.spatial_reuse[0]));
      end
    end

    if (tr == RX && format_mod == HE_MU) begin
      do_compare &= comparer.compare_field("ru_type", this.ru_type, that.ru_type, $bits(this.ru_type));
    end

    if (tr == TX) begin
      if (format_mod inside {NON_HT_DUP_OFDM, NON_HT})
        do_compare &= comparer.compare_field("trigger_responding", this.trigger_responding, that.trigger_responding, $bits(this.trigger_responding));
      do_compare &= comparer.compare_field("tx_pwr_level", this.tx_pwr_level, that.tx_pwr_level, $bits(this.tx_pwr_level));
      do_compare &= comparer.compare_field("continuous_tx", this.continuous_tx, that.continuous_tx, $bits(this.continuous_tx));
      do_compare &= comparer.compare_field("service", this.service, that.service, $bits(this.service));
    end

    return do_compare;
  endfunction : do_compare

  //------------------------------------------------------------
  // custom copy function
  //------------------------------------------------------------
  virtual function void do_copy(uvm_object rhs);
    PPDU_preamble_header rhs_;

    if (!$cast(rhs_,rhs)) begin
      `uvm_fatal(get_type_name(), "do_copy cast failed!")
    end
    super.do_copy(rhs);

    // create user header for HE frames
    user_header_he = new[rhs_.user_header_he.size()];
    foreach (rhs_.user_header_he[i]) user_header_he[i] = rhs_.user_header_he[i];
    user_header_he_sec = new[rhs_.user_header_he_sec.size()];
    foreach (rhs_.user_header_he_sec[i]) user_header_he_sec[i] = rhs_.user_header_he_sec[i];
    // create user header for non HE frames
    user_header = new[rhs_.user_header.size()];
    foreach (rhs_.user_header[i]) user_header[i] = rhs_.user_header[i];

    tx_pwr_level        = rhs_.tx_pwr_level;
    sounding            = rhs_.sounding;
    smoothing           = rhs_.smoothing;
    continuous_tx       = rhs_.continuous_tx;
    ch_bw               = rhs_.ch_bw;
    antenna_set         = rhs_.antenna_set;
    preamble_type       = rhs_.preamble_type;
    format_mod          = rhs_.format_mod;
    num_extn_ss         = rhs_.num_extn_ss;
    leg_length          = rhs_.leg_length;
    leg_rate            = rhs_.leg_rate;
    service             = rhs_.service;
    aggregated          = rhs_.aggregated;
    doze_not_allowed    = rhs_.doze_not_allowed;
    num_tx              = rhs_.num_tx;
    stbc                = rhs_.stbc;
    beamformed          = rhs_.beamformed;
    partial_aid         = rhs_.partial_aid;
    group_id            = rhs_.group_id;
    l_sig_valid         = rhs_.l_sig_valid;
    chbw_in_non_ht      = rhs_.chbw_in_non_ht;
    dyn_bw              = rhs_.dyn_bw;
    first_user          = rhs_.first_user;
    n_user              = rhs_.n_user;
    rssi                = rhs_.rssi;
    rssi_legacy         = rhs_.rssi_legacy;
    rcpi                = rhs_.rcpi;
    evm                 = rhs_.evm;
    band5G              = rhs_.band5G;
    abgn_mode           = rhs_.abgn_mode;
    tr                  = rhs_.tr;
    mu_mimo_userid      = rhs_.mu_mimo_userid;
    is_ndp              = rhs_.is_ndp;
    user_order          = rhs_.user_order;
    time_dep_req        = rhs_.time_dep_req;
    trigger_responding  = rhs_.trigger_responding;
    uplink_flag         = rhs_.uplink_flag;
    beam_change         = rhs_.beam_change;
    dcm                 = rhs_.dcm;
    he_ltf_type         = rhs_.he_ltf_type;
    doppler             = rhs_.doppler;
    midamble_periodicity = rhs_.midamble_periodicity;
    gi_type             = rhs_.gi_type;
    sig_b_comp_mode     = rhs_.sig_b_comp_mode;
    dcm_sig_b           = rhs_.dcm_sig_b;
    mcs_sig_b           = rhs_.mcs_sig_b;
    ru_allocation       = rhs_.ru_allocation;
    ru_type             = rhs_.ru_type;
    txop_duration       = rhs_.txop_duration;
    foreach (spatial_reuse[i]) spatial_reuse[i] = rhs_.spatial_reuse[i];
    bss_color           = rhs_.bss_color;
    he_siga_reserved    = rhs_.he_siga_reserved;
    num_he_ltf          = rhs_.num_he_ltf;
    he_ltf_mode         = rhs_.he_ltf_mode;
    ldpc_extra_symbol   = rhs_.ldpc_extra_symbol;
    starting_sts_num    = rhs_.starting_sts_num;
    trigger_method      = rhs_.trigger_method;
    ru_tone_set_index   = rhs_.ru_tone_set_index;
    feedback_status     = rhs_.feedback_status;
    pe_duration         = rhs_.pe_duration;
    dcm_sig_b           = rhs_.dcm_sig_b;
    mcs_sig_b           = rhs_.mcs_sig_b;
    primary_channel     = rhs_.primary_channel;

  endfunction : do_copy

  //----------------------------------------------------------------------
  // in relative to format mode set maximum value of MCS and CHBW
  //----------------------------------------------------------------------
  function void set_max_data_rate();
    int max_chbw;

    // determine highest value of CHBW
`ifdef RW_NX_DERIV_CHBW20ONLY
    max_chbw = 0;
`elsif RW_NX_DERIV_CHBW4020ONLY
    max_chbw = 1;
`else
    max_chbw = 2;
`endif

    // determine highest value of MCS
    case (format_mod)
      NON_HT, NON_HT_DUP_OFDM: begin
        leg_rate = 4'b1100; /*  54 Mbps */
      end
      HT_MF,HT_GF: begin
        ch_bw = max_chbw;
        user_header[0].mcs_f = 7;
      end
      VHT: begin
        ch_bw = max_chbw;
`ifdef RW_NX_DERIV_EQU_VHT
        user_header[0].mcs_f = (max_chbw == 0) ? 8 : 9;
`else
        user_header[0].mcs_f = 7;
`endif
      end
      HE_SU, HE_EXT_SU: begin
        ch_bw = max_chbw;
        dcm = 0;
        // constraint FEC to LDPC in CHBW40
`ifdef RW_NX_LDPC_ENC
        if (max_chbw == 1)
          user_header_he[0].fec_coding_f = 1;
`endif

`ifdef RW_NX_1024QAM_EN
        user_header_he[0].mcs_f = (user_header_he[0].fec_coding_f == 1) ? 11 : 9;
`elsif RW_NX_256QAM_EN
        user_header_he[0].mcs_f = 9;
`else
        user_header_he[0].mcs_f = 7;
`endif
      end
      HE_MU: begin

`ifdef RW_NX_1024QAM_EN
        user_header_he[mu_mimo_userid].mcs_f = (user_header_he[mu_mimo_userid].fec_coding_f == 1) ? 11 : 9;
`elsif RW_NX_256QAM_EN
        user_header_he[mu_mimo_userid].mcs_f = 9;
`else
        user_header_he[mu_mimo_userid].mcs_f = 7;
`endif
      end
    endcase
  endfunction : set_max_data_rate

  //----------------------------------------------------------------------
  // convert Tmmss to real
  //----------------------------------------------------------------------
  function real get_tmmss();
    real ret;

    ret = (tmmss == 3'b000) ? 0.0  :
          (tmmss == 3'b001) ? 0.25 :
          (tmmss == 3'b010) ? 0.5  :
          (tmmss == 3'b011) ? 1.0  :
          (tmmss == 3'b100) ? 2.0  :
          (tmmss == 3'b101) ? 4.0  :
          (tmmss == 3'b110) ? 8.0  :
                              16.0 ;

    return ret;
  endfunction : get_tmmss


endclass : PPDU_preamble_header

`endif //PPDU_PREAMBLE_HEADER_SV

