//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef KEYSTORAGERAM_SV
`define KEYSTORAGERAM_SV

class KeyStorageRAM extends uvm_object;

  rand int                   num_of_vlan; // number of VLANs
  rand int                   num_of_dev;  // number of devices
  rand key_storage_ram_field keyRAM[`KEYRAM_SIZE];

  // constaint array with cipher_type_e to be able to generate all type for
  // unicast and group key
`ifdef RW_ALL_CIPHER_EN
  //                                          0    1    2    3    4
  const cipher_type_e all_cipher_types[] = '{WAPI,WEP,TKIP,CCMP,GCMP};
`elsif RW_WAPI_EN
  //                                          0    1    2    3
  const cipher_type_e all_cipher_types[] = '{WAPI,WEP,TKIP,CCMP};
`elsif RW_GCMP_EN
  //                                          0    1    2    3
  const cipher_type_e all_cipher_types[] = '{WEP,TKIP,CCMP,GCMP};
`else
  //                                          0    1    2
  const cipher_type_e all_cipher_types[] = '{WEP,TKIP,CCMP};
`endif

  `uvm_object_utils_begin(KeyStorageRAM)
    `uvm_field_int(num_of_vlan, UVM_DEFAULT | UVM_NOPRINT)
    `uvm_field_int(num_of_dev,  UVM_DEFAULT | UVM_NOPRINT)
  `uvm_object_utils_end

  // --------------------------------------------
  // constraints
  // --------------------------------------------
  constraint c_vlan_number {
    soft num_of_vlan >= 2;
    soft num_of_vlan <= `NUM_OF_VAPS;
  }

  constraint c_dev_number {
    soft num_of_dev >= 10;
    soft num_of_dev <= `KEYRAM_DEV_END_ADDR-`KEYRAM_VLAN_END_ADDR;
  }

  constraint c_solve_order {
    solve num_of_vlan before num_of_dev;
  }

  constraint c_data_gen {
    foreach (keyRAM[i]) {

      // constraint VLAN number, for 1 VLAN there is 4 default key IDs
      if (i < num_of_vlan*`DEFAULT_KEY_CNT) {
        keyRAM[i].mac_addr_ram_f    == 0;
        keyRAM[i].vlan_id_ram_f     == 0;
        keyRAM[i].offset_key_idx_f  == 0;
        keyRAM[i].use_def_key_ram_f == 0;

        // make sure that all cipher type occurs in group key
        if (i < all_cipher_types.size()) {
          keyRAM[i].ctype_ram_f == all_cipher_types[i];
        }
`ifdef RW_WAPI_EN
        // make sure that WAPI group key can be only stored on address
        // power of 4
        else if (i % `DEFAULT_KEY_CNT == 0) {
          keyRAM[i].ctype_ram_f == WAPI;
        }
`endif
      }
      // constraint devices
      else if (   i >= `KEYRAM_DEV_START_ADDR
               && i <  `KEYRAM_DEV_START_ADDR + num_of_dev) {
        // make device address unicast
        keyRAM[i].mac_addr_ram_f[0] == 0;
        keyRAM[i].mac_addr_ram_f[47:1] != 0;
        keyRAM[i].vlan_id_ram_f < num_of_vlan;
        // make sure that all types of unicast keys are in KSR
        if ((i-`KEYRAM_DEV_START_ADDR) < all_cipher_types.size()) {
          keyRAM[i].ctype_ram_f == all_cipher_types[i-`KEYRAM_DEV_START_ADDR];
        }
      }
    }//foreach
  }//constraint

  function new (string name = "KeyStorageRAM");
    super.new(name);
    // create fields
    foreach (keyRAM[i])
      keyRAM[i] = new();
  endfunction

  function void post_randomize();
    int ccmp_q[$], tkip_q[$], wapi_q[$], wep_q[$], gcmp_q[$];

    // create all available indexes per encryption type and set flag that
    // indicates which cipher type exist
    for (int i=0;i<num_of_vlan*`DEFAULT_KEY_CNT;i++) begin
      case (keyRAM[i].ctype_ram_f)
        WEP : begin wep_q.push_front(i);  end
        TKIP: begin tkip_q.push_front(i); end
        CCMP: begin ccmp_q.push_front(i); end
        WAPI: begin wapi_q.push_front(i); end
        GCMP: begin gcmp_q.push_front(i); end
      endcase
    end

    foreach (keyRAM[i]) begin
      // constraint devices
      if (   i >= `KEYRAM_DEV_START_ADDR
          && i <  `KEYRAM_DEV_START_ADDR + num_of_dev) begin

        // determine VLANID and offset values with available keyIndex per
        // cipher type
        case (keyRAM[i].ctype_ram_f)
          WEP: begin
            wep_q.shuffle();
            keyRAM[i].vlan_id_ram_f    = wep_q[0] / `DEFAULT_KEY_CNT;
            keyRAM[i].offset_key_idx_f = wep_q[0] % `DEFAULT_KEY_CNT;
          end
          TKIP: begin
            tkip_q.shuffle();
            keyRAM[i].vlan_id_ram_f    = tkip_q[0] / `DEFAULT_KEY_CNT;
            keyRAM[i].offset_key_idx_f = tkip_q[0] % `DEFAULT_KEY_CNT;
          end
          CCMP: begin
            ccmp_q.shuffle();
            keyRAM[i].vlan_id_ram_f    = ccmp_q[0] / `DEFAULT_KEY_CNT;
            keyRAM[i].offset_key_idx_f = ccmp_q[0] % `DEFAULT_KEY_CNT;
          end
          WAPI: begin
            wapi_q.shuffle();
            keyRAM[i].vlan_id_ram_f    = wapi_q[0] / `DEFAULT_KEY_CNT;
            keyRAM[i].offset_key_idx_f = 0;
          end
          GCMP: begin
            gcmp_q.shuffle();
            keyRAM[i].vlan_id_ram_f    = gcmp_q[0] / `DEFAULT_KEY_CNT;
            keyRAM[i].offset_key_idx_f = gcmp_q[0] % `DEFAULT_KEY_CNT;
          end
        endcase
      end//else if
    end//foreach
  endfunction : post_randomize

  function void do_print(uvm_printer printer);
    super.do_print(printer);

    printer.print_int("number of VLANs", num_of_vlan, $bits(int), UVM_DEC);
    printer.print_int("number of devices", num_of_dev, $bits(int), UVM_DEC);
    // print only valid entries
    foreach (keyRAM[i]) begin
      if (  (i < num_of_vlan*`DEFAULT_KEY_CNT)
         || (i >= `KEYRAM_DEV_START_ADDR && i < `KEYRAM_DEV_START_ADDR+num_of_dev))
        `uvm_info(get_type_name(),$sformatf("KeyRAM[%0d] = %s",i,keyRAM[i].print()),UVM_LOW)
    end
  endfunction : do_print

  //---------------------------------------------------------------------------
  // returns matching device index inside KSR, searches
  //---------------------------------------------------------------------------
  function int get_device_index(bit [47:0] dev);

    for (int i=`KEYRAM_DEV_START_ADDR; i<`KEYRAM_DEV_END_ADDR; i++) begin
      if (keyRAM[i].mac_addr_ram_f == dev) return i;
    end

    `uvm_info(get_type_name(), $sformatf("Device (%0h) not found in KSR!",dev), UVM_LOW)
    return -1;
  endfunction : get_device_index

  //---------------------------------------------------------------------------
  // return matching device cipher type
  //---------------------------------------------------------------------------
  function int get_dev_ctype(bit [47:0] dev);
    int index;

    index = get_device_index(dev);
    if (index < 0) return index;

    return keyRAM[index].ctype_ram_f;
  endfunction : get_dev_ctype

endclass : KeyStorageRAM

`endif //KEYSTORAGERAM_SV
