//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthou $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 28315 $
// $Date: 2016-09-29 17:34:35 +0300 (Thu, 29 Sep 2016) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////

`ifndef MDM_DATA_MODEL_SV
`define MDM_DATA_MODEL_SV


class mdm_data_model extends uvm_object;

  phyData_t DataBuf [`MIMO_NR] [];     // dynamic multi-dimensional array that holds payload data

  int numUsers;                        // In a MUMIMO Tx/Rx indicates the number of users with non-zero space-time streams
  int userPos[`MIMO_NR-1:0];           // Array for mapping users to positions 0,1,2,3
  int packetLength;                    // Packet length
  int FrameCounter;                    // Counter for processed PPDU frames; used for backing up sysparam and phyPayload files for easier debugging

  // already present parameters
        bit [7:0] scramblerSeed;       // Seed of scrambler; has to be the same as in RTL
        int nTx;                       // Number of Tx antennas.
        int nRx[`MIMO_NR-1:0];         // Number of Rx antennas
        string FrameFormat;            // Operating mode:: LM: Legacy OFDM; LM11B: Legacy DSSS/CCK; MM: HT Mixed mode; AC: VHT preamble format.
        bit PreambleType11b;           // DSSS/CCK frame preamble format. 1: short; 0: long (in RTL the meaning is reversed!)
        int LegacyRate;                // Legacy frame rate as defined in L-SIG rate field; valid only if FrameFormat = LM
        int MCS[`MIMO_NR-1:0];         // MCS as defined in HT or VHT standards. Can be used to define legacy frame rate
  rand  int ChanType;                  // IEEE channel type; 0:Z, 1:A, 2:B, 3:C, 4:D, 5:E, 6:F
        int FrameBW;                   // Frame bandwith
        int RxChannelBW;               // Receiver channel bandwith in MHz (20/40/80)
        int TxChannelBW;               // Transmitter channel bandwith in MHz (20/40/80)
        int RxControlChannel;          // Primary channel location of the receiver. If RxChannelBW = 20 this should be 0.
        int TxControlChannel;          // Primary channel location of the transmitter. If TxChannelBW = 20 this should be 0.
        int htLength[`MIMO_NR-1:0];    // If set, replaces the length field. Should not be used!
        int LegacyLength;              // Length in bytes used to fill L-SIG length field
        string CarrierFreq_GHz;        // Carrier frequency
        int smoothing;                 // Used to fill the HT-SIG smoothing field.
        int NESS;                      // Number of extended streams in MM
        bit ShortGI;                   // Guard interval duration. 0: long; 1: short
        bit [1:0] STBC;                // Space Time Block Code enable. 0: no STBC, 1: STBC
        int NSTS [`MIMO_NR-1:0];       // Number of Space Time Streams
        int Tx4044FilterShift;         // DSSS/CCK 44->40 resampling filter shift
        int TxFENegShiftDel;           // Tx OFDM front-end frequency shift start index
        int Tx4044FilterInDel;         // DSSS/CCK 44->40 resampling filter input latency
        int Tx4044FilterOutDel;        // DSSS/CCK 44->40 resampling filter output latency
        bit NOTSOUNDING;               // Indicates that the frame is not a sounding frame. Fills HT-SIG sounding field. 0: sounding, 1: not sounding
  rand  int TxDigGainLin;              // Transmitter digital gain value
        bit EVM;                       // Enable EVM computation
        bit TxDump;                    // 1: Enable generation of Tx test vectors
        bit RxDump;                    // 1: Enable generation of Rx test vectors
        bit AGCDumpEn;                 // 1: Enable generation of RIU/AGC internal test vectors
        bit TXFRONTENDDumpEn;          // 1: Enable generation of Tx front-end internal test vectors
        bit HTSTFAGCEn;                // 0: disables AGC gain update in (V)HT-STF
        bit Aggregation;               // 1: PPDU contains an A-MPDU; 0: otherwise. Fills HT-SIG Aggregation field.
        bit DozeNotAllow;              // If not empty, VHT-SIG-A"TXOP_PS_NOT_ALLOWED" field is filled with this value, else it is forced to 0
        bit LDPC[`MIMO_NR-1:0];        // 0: Coder is a BCC; 1: coder is an LDPC
  rand  int TxIQPhase_deg;             // Transmitter IQ phase mismatch in degrees
  rand  int TxIQGain_dB;               // Transmitter IQ gain mismatch in dB
  rand  int RxIQPhase_deg;             // Receiver IQ phase mismatch in degrees
  rand  int RxIQGain_dB;               // Receiver IQ gain mismatch in dB
  rand  int RxDC_mv;                   // DC offset in mV
  rand  int TxDC_mv;                   // DC offset in mV
  rand  int AGCError_dB;               // Valid only if AGCModel=PFCT; defines the mismatch in dB between ADC target level and effective ADC power
  rand  int CarrierLostStartUs;        // Carrier lost start time in us
  rand  int packetStartUs;             // Packet start time in us
  rand  int ACIRxPAntdBm;              // ACI antenna power in dBm
  rand  int ACIStartUs;                // ACI start time in us
  rand  int ACIStopUs;                 // ACI stop time in us
  rand  int ACIFcMHz;                  // Frequency offset between the central frequency of the frame and the central frequency of the ACI
        string ACIOperatingMode;       // ACI frame format
  rand  int RampUpFreqkHz;             // If the ramp-up is a carrier, this parameter defines its frequency
  rand  int RampUpLength_us;           // Ramp-up time in us
  rand  int RampUpType;                // Define the shape of the ramp-up eventually added before the frame
        real RampUpRatio;              // Defines the proportion of time spent in each phase of a 2-phase ramp-up
  rand  int RampUpRatioDividend;
  rand  int RampUpRatioDivisor;
  rand  int RampUpPowDeltadB;          // Defines the power of the ramp-up relatively to the power of the frame
        bit LSIGParityErrForce;        // 1: L-SIG parity bit is corrupted
        bit HTSIGCRCErrForce;          // 1: Corrupt the (V)HT-SIG(A) CRC
        int smmIdx;                    // SU Tx: Defines the Q steering matrix type; MU-MIMO Tx: should not be set
        string RxRFName;               // RFI model to be applied: NON: basic reference model, KARST: model of KARST RF IC
        string AGCModel;               // PFCT: packet detection and gain setting forced;FXPT: p.d. and gain are results of AGC FSM processing
        string StateDescription;       // Allows selecting AGC FSM description file. Possible values: RTL; KARST_24GHz
  rand int RxPAntdBm;                  // Antenna power in dBm, valid only if RxRFName ="KARST"
        bit FixRandomSeed;             // 0: random seed;  1: RNGSeed will be used as seed value
        bit [31:0] RNGSeed;            // matlab seed value if FixRandomSeed = 1, should be svseed from simulation (non-negaive number < 2^32)
        bit AGCPrint;                  // 1: sequence of AGC states is displayed

        // new additions
        string DirName;                // Name of directory containing dumps if PrintDump is executed
        bit BFDump;                    // 1: Enambe generation of beamforming test vectors
        bit TXFREQDumpEn;              // 1: Enable generation of Tx SW internal test vectors
        bit RXTDDumpEn;                // 1: Enable generation of Rx TD internal test vectors
        bit RXFDDumpEn;                // 1: Enable generation of Rx FD internal test vectors
  rand  int SNR;                       // SNR in dB, valid only if RxRFName is"NON"
  rand  int ClockOffset_ppm;           // Sampling clock offset in ppm
  rand  int FrequencyOffset_ppm;       // Frequency offset in ppm
        int NbSucPack;                 // Number of successive frames generated.
        int SucPackIFS_us;             // Inter Frame Space in us between successive frames
        int Length[`MIMO_NR-1:0];      // Frame length in bytes
        bit BFEn;                      // Indicates a TX SU beamformed frame or a Tx/Rx MUMIMO frame.
        bit [23:0] ForceLSIG;          // If defined forces the value of L-SIG field
        bit [23:0] ForceHTSIG;         // If defined forces the value of L-SIG field
        bit CorruptSFD;                // 0: normal SFD; 1: corrupted SFD, all bits forced to 0
        int PartialAID;                // If not empty this parameter will be used to fill VHT-SIGA Partial AID field
        int GroupID;                   // If not empty this parameter will be used to fill VHT-SIGA GroupID field. Must be 2 for MU-MIMO Rx
                                       // Must not be 0 or 63 for MU-MIMO Tx
        int ForceLegacyRate;           // if not empty, the LegacyRate field is filled with this parameter
        int ForceMCS;                  // if not empty, the MCS field is filled with this parameter
        int ForceSTBC;                 // if not empty, the STBC field of HT-SIG is filled with this parameter. Otherwise the field is 0.
        bit UnsupportedMod;            // 1: Corrupt the signal and service field of the DSSS/CCK frame
        bit UnsupportedRate;           // 1: Corrupt the rate field of the DSSS/CCK frame
        bit UnsupportedService;        // 1: Corrupt the service field of the DSSS/CCK frame
        bit UnsupportedMCS;            // 1: Corrupt the MCS field
        int ACIMCS;                    // ACI frame MCS
        int ACIFrameBW;                // ACI frame bandwith
        int ACIChannelBW;              // ACI transmitte channel bandwith
        string TxModel;                // FLPT: floating point Matlab model is used;   FXPT: fixed-point Matlab model of transmitrer is used.
                                       //       Tx dump cannot be used, sim is faster        nTx must be <= 2

        int UserID;                    // MU-MIMO Rx stream selection
        int AGCDelay_us;               // Defines the delay in us between packets, valid only for AGCModel=PFCT
        string AGCDCEstim;             // NON: AGC DC estimator is disabled; FXPT: AGC DC esimator enabled according to state description
        int HTSigDeclatency_us;        // Defines the latency in us of (V)HT-SIG(A) decoding by DSP SW
        real TDCyclicRot_us;            // Defines TD cyclic shift in us to apply n the RxTFU to the legacy part of the frame
        bit FOCOMPEn;                  // 0: Disables TD frequency offset compensation (in RxTFU znc in SW-RxTD)
        bit DCCOMPEn;                  // 0: Disables DC offset compensation (in RxTFU znc in SW-RxTD)
        bit STOTDCCOMPEn;              // 0: Disables TD compensation of the sampling clock offset inducing variation of the sync value in RxTFU
        int ViterbiTraceBack;          // Viterbi trace back value

        // Calibration parameters (`NB_STA_MAX stations). All set from testcase
        int       nBeamformingSTA;          // Number of statios to test for Beamforming, comes from testcase
        bit [2:0] STAIDList [`NB_STA_MAX];    // Must contain the IDs (between 1 and NB_STA_MAX) of the stations targeted by TX, comes from testcase
                                            // STAIDList[0] is used for SU BF TX
                                            // STAIDList[0] and STAIDList[1] are used for MU TX
                                            // TAKE CARE : STA with smallest user position must be first in Matlab !
        bit [`BFSTA_NR-1:0] BFRRxAntSet;    // Rx antenna selection for beamformee.
        bit NDPEn;                          // Enable NDP Tx with storing of channel parameters.
        bit [2:0] Nc [`NB_STA_MAX];         // Nc value used in beamforming calibration: 1 or 2.
        bit [2:0] Nr;                       // Nr value used in beamforming calibration: 1 or 2.
        bit [1:0] Grouping [`NB_STA_MAX];   // Grouping value used in beamforming calibration:: 0: no grouping; 1:grouping by 2; 2: grouping by 4.
        bit       Codebook [`NB_STA_MAX];   // Codebook value used in beamforming calibration:: 0 or 1.
        int       Feedback [`NB_STA_MAX];   // Feedback value used in beamforming calibration:: 0: SU feedback type; 1: MU feedback type.
        int       RUStartIndex[`NB_STA_MAX];// The starting RU index indicates the first 26-tone RU for which the HE beamformer is requesting feedback.
        int       RUEndIndex[`NB_STA_MAX];  // The ending RU index indicates the last 26-tone RU for which the HE beamformer is requesting feedback.

        string    NcMode;                   // ?????

        bit RIFSDetEn;                 // 1: RIFS detection enabled in AGC FSM. Valid only if AGCModel is "FXPT"
        string NoisePowEst;            // PFCT: ?????, FXPT: not yet supported
        string FDOOffsetEstim;         // NON: disables FD offset (CPE/STO) estimation; FXPT: ???
        string FDOOffsetComp;          // NON: disables FD offset (CPE/STO) compensation; FXPT: ???
        string ViterbiType;            // mexPFCT: floating-point model ; mexFXPT: fixed-point model; RADIX4: ???

  // missing from XLS, yet used by libdump
        int RxClockOffset_ppm;
        int TBEBias;
        string FrameBWDet = "FXPT";
        string RxSmoothing;
        int TDDCCompInit;
//     int TDDCfInit;
        int TDDCTracking;
        int TDCyclicRotateValUs20MHz;
        int TDCyclicRotateValUs40MHz;
        int ShiftSBEn;
        int debugEn;
        int TxPhaseNoiseFc_kHz;
        int TxPhaseNoisePow_degrms;
        int RxFrequencyOffset_ppm;
        int RxPhaseNoiseFc_kHz;
        int RxPhaseNoisePow_degrms;
        int CarrierLostStopUs;
        int SMFFilterSet;
        // fIQ test enable
        bit               fIQTest;
        bit [31:0]        m_CAL_fQ[];
        bit [31:0]        m_CAL_sI[];
        bit [31:0]        m_CAL_sQ[];
        bit [31:0]        m_CAL_error[];

  // Matlab return values
  // DAC referent values (Tx path verification)
  DAC_mu_data_t TxframeOut_re [];
  DAC_mu_data_t TxframeOut_im [];

  // H memory used to compute the beamforming report from matlab model
  bit signed [`HMEM_WIDTH-1:0] bfr_hmem_re[];
  bit signed [`HMEM_WIDTH-1:0] bfr_hmem_im[];
  hmem_array                   hmem_ref_data;

  // Beamforming Report from matlab model
  bit                        bfr[];

  // Beamforming Report Sigma dB from matlab model
  bit [`BFRSIGMADB_WIDTH-1:0] bfr_sigmadB[];

  // SNR
  bit [31:0]                  snr_dB;

  // Noise variance from matlab model
  bit [`SIGMAE2_WIDTH-1:0]    sigmae2;
  bit [7:0]                   noise_variance;

  // Tx BF reports
  bit [ `TXBFRLENGTH_WIDTH-1:0] txBfrLength [`NB_STA_MAX];
  bit [`TXBFRCOMPREP_WIDTH-1:0] txBfrCompressedReport [`NB_STA_MAX];

  // Rx data
  ADC_data_t    ADCDataI [];
  ADC_data_t    ADCDataQ [];
  ADC_data_t    RFDataI [];
  ADC_data_t    RFDataQ [];
  ADC_mu_data_t ADCframe_re [];
  ADC_mu_data_t ADCframe_im [];
  ADC_mu_data_t RFframe_re [];
  ADC_mu_data_t RFframe_im [];

  // Rx Frontend from matlab model
  AGC_data_t    rxFEout20Re [];
  AGC_data_t    rxFEout20Im [];
  AGC_mu_data_t rxFEout20_re [];
  AGC_mu_data_t rxFEout20_im [];

  AGC_data_t    rxFEout20SRe [];
  AGC_data_t    rxFEout20SIm [];
  AGC_mu_data_t rxFEout20S_re [];
  AGC_mu_data_t rxFEout20S_im [];

  AGC_data_t    rxFEout40Re [];
  AGC_data_t    rxFEout40Im [];
  AGC_mu_data_t rxFEout40_re [];
  AGC_mu_data_t rxFEout40_im [];

  AGC_data_t    rxFEout80Re [];
  AGC_data_t    rxFEout80Im [];
  AGC_mu_data_t rxFEout80_re [];
  AGC_mu_data_t rxFEout80_im [];

  bit signed [6:0] rxFEout20DSSSRe [];
  bit signed [6:0] rxFEout20DSSSIm [];

  rxparameters_s   rxparams; //used for Time and Freq domain Rx blocks

  // Placeholder for PPDU frame, shall be used by modem_rx_scoreboard
  PPDU_frame golden_frame;
  // flag should be set to 1 from test case if sysParam.txt and phyPayload.txt
  // files are prepared in advance for usage. This will prevent cleaning of
  // matlab files and generating new ones.
  bit        use_preset_files = 0;
  // flag to indicate that payload file is not set, matlab will generate
  // payload content
  bit        use_matlab_payload = 0;
  // flags that indicate status of Rx data provided by Matlab
  int        fcsOK;
  int        lsigErrorcode;
  int        htsigErrorcode;
  int        hesigaErrorcode;
  int        hesigbErrorcode;
  int        vhtsigbErrorcode;
  // SIG structures used in mdm_data_model_ver2
  l_sig_s       l_sig;
  ht_sig_s      ht_sig;
  vht_siga_su_s vht_siga_su;
  vht_siga_mu_s vht_siga_mu;
  he_siga_su_s  he_siga_su;
  he_siga_tb_s  he_siga_tb;
  he_siga_mu_s  he_siga_mu;
  he_sigb_s     he_sigb;

  // fields that are used for setting custom RU parameters
  sta_list_t    sta_list;
  // flag for inserting errors in SIG signals
  // LSIG/HTSIG/VHTSIG/HESIGA
  bit           sig_error_injection = 0;
  bit           station_file_defined = 0;
  int           offset_ppm;
  // handle to register model
  REGMODEL_pkg::TOP_register_block m_regmodel;


  `uvm_object_utils(mdm_data_model)

//--------------------------------------------------------------------
// Constraints for randomized values
//--------------------------------------------------------------------
  constraint c_SNR {
    SNR >=0;
    SNR <= 40;
  }

  constraint c_RxPAntdBm {
    RxPAntdBm >= -110;
    RxPAntdBm <=   20;
  }

  constraint c_ACIRxPAntdBm {
    ACIRxPAntdBm >= -100;
    ACIRxPAntdBm <=    0;
  }

  constraint c_ACIStartUs {
    ACIStartUs >=    0;
    ACIStartUs <= 1000;
  }

  constraint c_ACIStopUs {
    ACIStopUs >=    0;
    ACIStopUs <= 1000;
  }

  constraint c_ACIFcMHz {
    ACIFcMHz >=   0;
    ACIFcMHz <= 160;
  }

  constraint c_CarrierLostStartUs {
    // set from test if error injecting is necessary
    CarrierLostStartUs == 0;

  }

  constraint c_TxDigGainLin {
    TxDigGainLin >=   0;
    TxDigGainLin <= 127;
  }

  constraint c_AGCError_dB {
    AGCError_dB >= -20;
    AGCError_dB <=  20;
  }

  constraint c_RampUpFreqkHz {
    RampUpFreqkHz >= 0;
    RampUpFreqkHz <= 100;
  }

  constraint c_RampUpLength_us {
    RampUpLength_us >=  0;
    RampUpLength_us <= 10;
  }

  constraint c_RampUpType {
    RampUpType >= 1;
    RampUpType <= 5;
  }

  constraint c_RampUpRatio {
    RampUpRatioDividend == 1;
    RampUpRatioDivisor  == 2;
  }

  constraint c_RampUpPowDeltadB {
    RampUpPowDeltadB == -15;
  }

  constraint c_ClockOffset_ppm {
    ClockOffset_ppm >= -100;
    ClockOffset_ppm <=  100;
  }

  constraint c_FrequencyOffset_ppm {
    FrequencyOffset_ppm >= -100;
    FrequencyOffset_ppm <=  100;
  }

  constraint c_TxIQPhase_deg {
    TxIQPhase_deg >= -5;
    TxIQPhase_deg <=  5;
  }

  constraint c_TxIQGain_dB {
    TxIQGain_dB >= -6;
    TxIQGain_dB <=  6;
  }

  constraint c_RxIQPhase_deg {
    RxIQPhase_deg >= -5;
    RxIQPhase_deg <=  5;
  }
  constraint c_RxIQGain_dB {
    RxIQGain_dB >= -6;
    RxIQGain_dB <=  6;
  }
  constraint c_RxDC_mv {
    RxDC_mv >= -500;
    RxDC_mv <=  500;
  }

  constraint c_TxDC_mv {
    TxDC_mv >= -500;
    TxDC_mv <=  500;
  }

  constraint c_packetStartUs {
    packetStartUs >=   10;
    packetStartUs <= 1000;
  }

  constraint c_ChanType {
//     ChanType >= 0;
//     ChanType <= 7;
    ChanType == 0;  // TODO current test use value of 0, except for BF
                    // or MU when the value is 2 and that is set from testcase directly
                    // So randomize this or not ?!

}

//--------------------------------------------------------------------
// Initialize all default values
//--------------------------------------------------------------------
  function void initialize_vars();
    numUsers            = 0; // valid only for VHT, then range is 1-4
    packetLength        = 0;
    scramblerSeed       = 8'b01011001;
    nTx                 = 1;
    FrameFormat         = "LM";
    PreambleType11b     = 0;
    LegacyRate          = 11; // 6MBps
    FrameBW             = 20;
`ifdef RW_NX_DERIV_CHBW20ONLY
    RxChannelBW         = 20;
    TxChannelBW         = 20;
`endif
`ifdef RW_NX_DERIV_CHBW4020ONLY
    RxChannelBW         = 40;
    TxChannelBW         = 40;
`endif
`ifdef RW_NX_DERIV_CHBW804020ONLY
    RxChannelBW         = 80;
    TxChannelBW         = 80;
`endif
    RxControlChannel    = 0;
    TxControlChannel    = 0;
    LegacyLength        = -1;
    CarrierFreq_GHz     = "6";
    smoothing           = 1;
    NESS                = 0;
    ShortGI             = 0;
    STBC                = 0;
    TBEBias             = 5;
    Tx4044FilterShift   = 1;
    TxFENegShiftDel     = 0;
    Tx4044FilterInDel   = 9;
    Tx4044FilterOutDel  = 1;
    NOTSOUNDING         = 1;
    EVM                 = 1;
    TxDump              = 0;
    RxDump              = 0;
    AGCDumpEn           = 0;
    TXFRONTENDDumpEn    = 0;
    HTSTFAGCEn          = 0;
    Aggregation         = 0;
    DozeNotAllow        = 0;
    ACIOperatingMode    ="NON";
    smmIdx              = 2;
    RxRFName            ="KARST";  // Forced to KARST
    AGCModel            ="PFCT";
    StateDescription    ="KARST_24GHz";
    FixRandomSeed       = 1;                          // should be forced to 1 to use svseed from simulation command line
    RNGSeed             = $get_initial_random_seed(); // get svseed value
    AGCPrint            = 0;
    DirName             ="DUMP";
    RampUpRatioDividend = 1;
    RampUpRatioDivisor  = 2;
    RampUpType          = 1;
    RampUpPowDeltadB    = -15;
    BFDump              = 0;
    TXFREQDumpEn        = 0;
    RXTDDumpEn          = 0;
    RXFDDumpEn          = 0;
    NbSucPack           = 1;
    SucPackIFS_us       = 2;
    BFEn                = 0;
    CorruptSFD          = 0;
    PartialAID          = 0;
    GroupID             = 0;
    ACIMCS              = 6;
    ACIFrameBW          = 20;
    ACIChannelBW        = 20;
    TxModel             ="FXPT";
    NDPEn               = 0; // Should be "1" according to XLS
    UserID              = 1; // should be 0
    AGCDelay_us         = 4;
    AGCDCEstim          ="FXPT";
    HTSigDeclatency_us  = 2;
    TDCyclicRot_us      = 0.3;
    FOCOMPEn            = 1;
    DCCOMPEn            = 1;
    STOTDCCOMPEn        = 1;
    RIFSDetEn           = 0;
    NoisePowEst         ="PFCT";
    FDOOffsetEstim      ="FXPT";
    FDOOffsetComp       ="FXPT";
    ViterbiType         ="mexPFCT";
    ViterbiTraceBack    = 80;
    BFRRxAntSet         = 'h0;
`ifdef RW_TXRX_1X1
    BFRRxAntSet[0]      = 1;
    BFRRxAntSet[1]      = 0;
`else
    BFRRxAntSet[0]      = 1;
    BFRRxAntSet[1]      = 1;
`endif
    RxPhaseNoiseFc_kHz     = 0;
    RxPhaseNoisePow_degrms = 0;
    TxPhaseNoiseFc_kHz     = 0;
    TxPhaseNoisePow_degrms = 0;
    CarrierLostStopUs      = 0;
    for (int i=0; i<`MIMO_NR; i++) begin
      LDPC[i]     = 0;
      nRx[i]      = 1;
      MCS[i]      = 0;
      NSTS[i]     = 0;
      Length[i]   = 0;
      userPos[i]  = i;
      htLength[i] = -1;
    end

    NcMode  = "FORCE";
    fIQTest = 1'b0;

  endfunction: initialize_vars

//--------------------------------------------------------------------
// Variables which must not be reset during test execution e.g.
// variables set from the testcase
//--------------------------------------------------------------------
  function void initialize_vars_once();
    FrameCounter = 0;  // Do not reset during run-time
    ChanType     = 0;
    nBeamformingSTA = 1;
    STAIDList[0] = 0;
    STAIDList[1] = 0;
    Nr           = `RW_NX_DERIV_NTX-1;
    for (int i=0;i<`NB_STA_MAX;i++) begin
      Nc[i]       = `RW_NX_DERIV_NRX-1;
      Grouping[i] = 0;
      Codebook[i] = 0;
      Feedback[i] = 0;
    end

    ForceLSIG           = 0;
    ForceHTSIG          = 0;
    ForceLegacyRate     = 0;
    ForceMCS            = 0;
    ForceSTBC           = 0;
    UnsupportedMod      = 0;
    UnsupportedRate     = 0;
    UnsupportedService  = 0;
    UnsupportedMCS      = 0;
    LSIGParityErrForce  = 0;
    HTSIGCRCErrForce    = 0;
  endfunction: initialize_vars_once

//--------------------------------------------------------------------
// Constructor
//--------------------------------------------------------------------
  function new (string name ="mdm_data_model");
    super.new(name);
    initialize_vars();
    initialize_vars_once();
  endfunction

//--------------------------------------------------------------------
// Returns 1 if parameter FrameFormat has a valid value, prints error
// and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_FrameFormat();
    FrameFormat = FrameFormat.toupper();
    if ( (FrameFormat =="LM") || (FrameFormat =="LM11B") || (FrameFormat =="MM") || (FrameFormat =="GF") || (FrameFormat =="AC")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid FrameFormat parameter '%s' received, should be in [LM, LM11B, MM, GF, AC]", FrameFormat))
      return 0;
    end
  endfunction: validate_FrameFormat

//--------------------------------------------------------------------
// Returns 1 if parameter ACIOperatingMode has a valid value, prints
// error and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_ACIOperatingMode();
    ACIOperatingMode = ACIOperatingMode.toupper();
    if ( (ACIOperatingMode =="LM") || (ACIOperatingMode =="LM11B") || (ACIOperatingMode =="MM") || (ACIOperatingMode =="NON")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid ACIOperatingMode parameter '%s' received, should be in [LM, LM11B, MM, NON]", ACIOperatingMode))
      return 0;
    end
  endfunction: validate_ACIOperatingMode

//--------------------------------------------------------------------
// Returns 1 if parameter RxRFName has a valid value, prints error
// and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_RxRFName();
    RxRFName = RxRFName.toupper();
    if ( (RxRFName =="NON") || (RxRFName =="KARST") ) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid RxRFName parameter '%s' received, should be in [NON, KARST]", RxRFName))
      return 0;
    end
  endfunction: validate_RxRFName

//--------------------------------------------------------------------
// Returns 1 if parameter AGCModel has a valid value, prints error
// and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_AGCModel();
    AGCModel = AGCModel.toupper();
    if ( (AGCModel =="PFCT") || (AGCModel =="FXPT")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid AGCModel parameter '%s' received, should be in [LM, LM11B, MM, NON]", AGCModel))
      return 0;
    end
  endfunction: validate_AGCModel

//--------------------------------------------------------------------
// Returns 1 if parameter StateDescription has a valid value,
// prints error and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_StateDescription();
    if ( (StateDescription.toupper() =="RTL") || (StateDescription.toupper() =="FPGA_KARST")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid StateDescription parameter '%s' received, should be in [RTL, KARST_24GHz]", StateDescription))
      return 0;
    end
  endfunction: validate_StateDescription

//--------------------------------------------------------------------
// Returns 1 if parameter TxModel has a valid value, prints error
// and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_TxModel();
    TxModel = TxModel.toupper();
    if ( (TxModel =="FLPT") || (TxModel =="FXPT")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid TxModel parameter '%s' received, should be in [FLPT, FXPT]", TxModel))
      return 0;
    end
  endfunction: validate_TxModel

//--------------------------------------------------------------------
// Returns 1 if parameter AGCDCEstim has a valid value, prints error
// and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_AGCDCEstim();
    AGCDCEstim = AGCDCEstim.toupper();
    if ( (AGCDCEstim =="NON") || (AGCDCEstim =="FXPT")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid AGCDCEstim parameter '%s' received, should be in [NON, FXPT]", AGCDCEstim))
      return 0;
    end
  endfunction: validate_AGCDCEstim

//--------------------------------------------------------------------
// Returns 1 if parameter NoisePowEst has a valid value, prints error
// and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_NoisePowEst();
    NoisePowEst = NoisePowEst.toupper();
    if ( (NoisePowEst =="PFCT") || (NoisePowEst =="FXPT")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid NoisePowEst parameter '%s' received, should be in [PFCTl, FXPT]", NoisePowEst))
      return 0;
    end
  endfunction: validate_NoisePowEst

//--------------------------------------------------------------------
// Returns 1 if parameter FDOOffsetEstim has a valid value, print
// error and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_FDOOffsetEstim();
    FDOOffsetEstim = FDOOffsetEstim.toupper();
    if ( (FDOOffsetEstim =="NON") || (FDOOffsetEstim =="FXPT")) begin
      return 1;
    end else begin
      `uvm_error(get_type_name(), $sformatf(" Invalid FDOOffsetEstim parameter '%s' received, should be in [NON, FXPT]", FDOOffsetEstim))
      return 0;
    end
  endfunction: validate_FDOOffsetEstim

//--------------------------------------------------------------------
// Returns 1 if parameter FDOOffsetComp has a valid value, prints error
// and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_FDOOffsetComp();
    FDOOffsetComp = FDOOffsetComp.toupper();
    if ( (FDOOffsetComp =="NON") || (FDOOffsetComp =="FXPT")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid FDOOffsetComp parameter '%s' received, should be in [NON, FXPT]", FDOOffsetComp))
      return 0;
    end
  endfunction: validate_FDOOffsetComp

  // Returns 1 if parameter ViterbiType has a valid value, prints error and returns 0 otherwise
  function bit validate_ViterbiType();
    if ( (ViterbiType.toupper() =="MEXPFCT") || (ViterbiType.toupper() =="MEXFXPT")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid ViterbiType parameter '%s' received, should be in [mexPFCT, mexFXPT]", ViterbiType))
      return 0;
    end
  endfunction: validate_ViterbiType


//--------------------------------------------------------------------
// Returns 1 if parameter CarrierFreq_GHz has a valid value, prints
// error and returns 0 otherwise
//--------------------------------------------------------------------
  function bit validate_CarrierFreq_GHz();
    if ( (CarrierFreq_GHz.toupper() =="2.4") || (CarrierFreq_GHz.toupper() =="6")) begin
      return 1;
    end else begin
      `uvm_fatal(get_type_name(), $sformatf(" Invalid CarrierFreq_GHz parameter '%s' received, should be in [2.4, 6]", CarrierFreq_GHz))
      return 0;
    end
  endfunction: validate_CarrierFreq_GHz

//--------------------------------------------------------------------
// Validate all parameters
//--------------------------------------------------------------------
  function void validate_params(input bit agcBypass = 0);
    bit ret;
    ret = validate_RxRFName();
    ret = validate_FrameFormat();
    ret = validate_ACIOperatingMode();
    ret = validate_AGCModel();
    ret = validate_TxModel();
    ret = validate_AGCDCEstim();
    ret = validate_NoisePowEst();
    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      ret = validate_StateDescription();
    end
    ret = validate_FDOOffsetEstim();
    ret = validate_FDOOffsetComp();
    ret = validate_ViterbiType();
    ret = validate_CarrierFreq_GHz();

    if (FrameFormat == "LM" && FrameBW != 20) begin
      `uvm_fatal(get_type_name(), $sformatf("In legacy mode the frame bandwidth can be only 20MHz!"))
    end

    if (TxChannelBW == 20 && TxControlChannel != 1) begin
      `uvm_fatal(get_type_name(), $sformatf("When TxChannelBW = 20MHz, TxControlChannel should be 1!"))
    end

    if (RxChannelBW == 20 && RxControlChannel != 1) begin
      `uvm_fatal(get_type_name(), $sformatf("When RxChannelBW = 20MHz, RxControlChannel should be 1!"))
    end

    if (STBC == 1 && NSTS[0] != 2) begin
      `uvm_fatal(get_type_name(), $sformatf("The system supports STBC = 1 only for NSTS = 2!"))
    end

//     if (FrameFormat == "GF" && /*NSTS[0] == 1*/ MCS[0] <= 7'd7 && ShortGI == 1'b1) begin
//       `uvm_fatal(get_type_name(), $sformatf("ShortGI not allowed in HT-GF mode with one spatial stream!"))
//     end

    foreach (NSTS[i]) begin
      if (nTx < NESS + NSTS[i]) begin
      `uvm_fatal(get_type_name(), $sformatf("nTx must be greater than NSTS+NESS!"))
      end
    end
  endfunction: validate_params

//--------------------------------------------------------------------
// Prints all parameters' value
//--------------------------------------------------------------------
  function void do_print(uvm_printer printer);
    super.do_print(printer);

    if (uvm_top.get_report_verbosity_level() == UVM_HIGH) begin
      printer.print_int("numUsers", numUsers, $bits(int), UVM_DEC);
      printer.print_int("scramblerSeed", scramblerSeed, $bits(int), UVM_DEC);
      printer.print_int("nTx", nTx, $bits(int), UVM_DEC);
      printer.print_string("FrameFormat", $sformatf("%p",FrameFormat));
      printer.print_int("PreambleType11b", PreambleType11b, $bits(int), UVM_DEC);
      printer.print_int("LegacyRate", LegacyRate, $bits(int), UVM_DEC);
      printer.print_int("ChanType", ChanType, $bits(int), UVM_DEC);
      printer.print_int("FrameBW", FrameBW, $bits(int), UVM_DEC);
      printer.print_int("RxChannelBW", RxChannelBW, $bits(int), UVM_DEC);
      printer.print_int("TxChannelBW", TxChannelBW, $bits(int), UVM_DEC);
      printer.print_int("RxControlChannel", RxControlChannel, $bits(int), UVM_DEC);
      printer.print_int("TxControlChannel", TxControlChannel, $bits(int), UVM_DEC);
      printer.print_int("LegacyLength", LegacyLength, $bits(int), UVM_DEC);
      printer.print_string("CarrierFreq_GHz", $sformatf("%p",CarrierFreq_GHz));
      printer.print_int("smoothing", smoothing, $bits(int), UVM_DEC);
      printer.print_int("NESS", NESS, $bits(int), UVM_DEC);
      printer.print_int("ShortGI", ShortGI, $bits(int), UVM_DEC);
      printer.print_int("STBC", STBC, $bits(int), UVM_DEC);
      printer.print_int("Tx4044FilterShift", Tx4044FilterShift, $bits(int), UVM_DEC);
      printer.print_int("TxFENegShiftDel", TxFENegShiftDel, $bits(int), UVM_DEC);
      printer.print_int("Tx4044FilterInDel", Tx4044FilterInDel, $bits(int), UVM_DEC);
      printer.print_int("Tx4044FilterOutDel", Tx4044FilterOutDel, $bits(int), UVM_DEC);
      printer.print_int("NOTSOUNDING", NOTSOUNDING, $bits(int), UVM_DEC);
      printer.print_int("TxDigGainLin", TxDigGainLin, $bits(int), UVM_DEC);
      printer.print_int("EVM", EVM, $bits(int), UVM_DEC);
      printer.print_int("TxDump", TxDump, $bits(int), UVM_DEC);
      printer.print_int("RxDump", RxDump, $bits(int), UVM_DEC);
      printer.print_int("AGCDumpEn", AGCDumpEn, $bits(int), UVM_DEC);
      printer.print_int("TXFRONTENDDumpEn", TXFRONTENDDumpEn, $bits(int), UVM_DEC);
      printer.print_int("HTSTFAGCEn", HTSTFAGCEn, $bits(int), UVM_DEC);
      printer.print_int("Aggregation", Aggregation, $bits(int), UVM_DEC);
      printer.print_int("DozeNotAllow", DozeNotAllow, $bits(int), UVM_DEC);
      printer.print_int("TxIQPhase_deg", TxIQPhase_deg, $bits(int), UVM_DEC);
      printer.print_int("TxIQGain_dB", TxIQGain_dB, $bits(int), UVM_DEC);
      printer.print_int("RxIQPhase_deg", RxIQPhase_deg, $bits(int), UVM_DEC);
      printer.print_int("RxIQGain_dB", RxIQGain_dB, $bits(int), UVM_DEC);
      printer.print_int("RxDC_mv", RxDC_mv, $bits(int), UVM_DEC);
      printer.print_int("TxDC_mv", TxDC_mv, $bits(int), UVM_DEC);
      printer.print_int("AGCError_dB", AGCError_dB, $bits(int), UVM_DEC);
      printer.print_int("CarrierLostStartUs", CarrierLostStartUs, $bits(int), UVM_DEC);
      printer.print_int("packetStartUs", packetStartUs, $bits(int), UVM_DEC);
      printer.print_int("ACIRxPAntdBm", ACIRxPAntdBm, $bits(int), UVM_DEC);
      printer.print_int("ACIStartUs", ACIStartUs, $bits(int), UVM_DEC);
      printer.print_int("ACIStopUs", ACIStopUs, $bits(int), UVM_DEC);
      printer.print_int("ACIFcMHz", ACIFcMHz, $bits(int), UVM_DEC);
      printer.print_string("ACIOperatingMode", $sformatf("%p", ACIOperatingMode));
      printer.print_int("RampUpFreqkHz", RampUpFreqkHz, $bits(int), UVM_DEC);
      printer.print_int("RampUpLength_us", RampUpLength_us, $bits(int), UVM_DEC);
      printer.print_int("RampUpType", RampUpType, $bits(int), UVM_DEC);
      printer.print_int("RampUpRatio", RampUpRatio, $bits(int), UVM_DEC);
      printer.print_int("RampUpPowDeltadB", RampUpPowDeltadB, $bits(int), UVM_DEC);
      printer.print_int("LSIGParityErrForce", LSIGParityErrForce, $bits(int), UVM_DEC);
      printer.print_int("HTSIGCRCErrForce", HTSIGCRCErrForce, $bits(int), UVM_DEC);
      printer.print_int("smmIdx", smmIdx, $bits(int), UVM_DEC);
      printer.print_string("RxRFName", $sformatf("%p",RxRFName));
      printer.print_string("AGCModel", $sformatf("%p",AGCModel));
      printer.print_string("StateDescription", $sformatf("%p",StateDescription));
      printer.print_int("RxPAntdBm", RxPAntdBm, $bits(int), UVM_DEC);
      printer.print_int("FixRandomSeed", FixRandomSeed, $bits(int), UVM_DEC);
      printer.print_int("RNGSeed", RNGSeed, $bits(int), UVM_DEC);
      printer.print_int("AGCPrint", AGCPrint, $bits(int), UVM_DEC);
      printer.print_string("DirName",$sformatf("%p", DirName));
      printer.print_int("BFDump", BFDump, $bits(int), UVM_DEC);
      printer.print_int("TXFREQDumpEn", TXFREQDumpEn, $bits(int), UVM_DEC);
      printer.print_int("RXTDDumpEn", RXTDDumpEn, $bits(int), UVM_DEC);
      printer.print_int("RXFDDumpEn", RXFDDumpEn, $bits(int), UVM_DEC);
      printer.print_int("SNR", SNR, $bits(int), UVM_DEC);
      printer.print_int("ClockOffset_ppm", ClockOffset_ppm, $bits(int), UVM_DEC);
      printer.print_int("FrequencyOffset_ppm", FrequencyOffset_ppm, $bits(int), UVM_DEC);
      printer.print_int("NbSucPack", NbSucPack, $bits(int), UVM_DEC);
      printer.print_int("SucPackIFS_us", SucPackIFS_us, $bits(int), UVM_DEC);
      printer.print_int("BFEn", BFEn, $bits(int), UVM_DEC);
      printer.print_int("ForceLSIG", ForceLSIG, $bits(int), UVM_DEC);
      printer.print_int("ForceHTSIG", ForceHTSIG, $bits(int), UVM_DEC);
      printer.print_int("CorruptSFD", CorruptSFD, $bits(int), UVM_DEC);
      printer.print_int("PartialAID", PartialAID, $bits(int), UVM_DEC);
      printer.print_int("GroupID", GroupID, $bits(int), UVM_DEC);
      printer.print_int("ForceLegacyRate", ForceLegacyRate, $bits(int), UVM_DEC);
      printer.print_int("ForceMCS", ForceMCS, $bits(int), UVM_DEC);
      printer.print_int("ForceSTBC", ForceSTBC, $bits(int), UVM_DEC);
      printer.print_int("UnsupportedMod", UnsupportedMod, $bits(int), UVM_DEC);
      printer.print_int("UnsupportedRate", UnsupportedRate, $bits(int), UVM_DEC);
      printer.print_int("UnsupportedService", UnsupportedService, $bits(int), UVM_DEC);
      printer.print_int("ACIMCS", ACIMCS, $bits(int), UVM_DEC);
      printer.print_int("ACIFrameBW", ACIFrameBW, $bits(int), UVM_DEC);
      printer.print_int("ACIChannelBW", ACIChannelBW, $bits(int), UVM_DEC);
      printer.print_string("TxModel", $sformatf("%p",TxModel));
      printer.print_int("NDPEn", NDPEn, $bits(int), UVM_DEC);
      printer.print_int("UserID", UserID, $bits(int), UVM_DEC);
      printer.print_int("AGCDelay_us", AGCDelay_us, $bits(int), UVM_DEC);
      printer.print_string("AGCDCEstim", $sformatf("%p",AGCDCEstim));
      printer.print_int("HTSigDeclatency_us", HTSigDeclatency_us, $bits(int), UVM_DEC);
      printer.print_int("TDCyclicRot_us", TDCyclicRot_us, $bits(int), UVM_DEC);
      printer.print_int("FOCOMPEn", FOCOMPEn, $bits(int), UVM_DEC);
      printer.print_int("DCCOMPEn", DCCOMPEn, $bits(int), UVM_DEC);
      printer.print_int("STOTDCCOMPEn", STOTDCCOMPEn, $bits(int), UVM_DEC);

      printer.print_string("NcMode", $sformatf("%p",NcMode));
      printer.print_int("RIFSDetEn", RIFSDetEn, $bits(int), UVM_DEC);
      printer.print_string("NoisePowEst", $sformatf("%p",NoisePowEst));
      printer.print_string("FDOOffsetEstim", $sformatf("%p",FDOOffsetEstim));
      printer.print_string("FDOOffsetComp", $sformatf("%p",FDOOffsetComp));
      printer.print_string("ViterbiType", $sformatf("%p",ViterbiType));
      printer.print_int("ViterbiTraceBack", ViterbiTraceBack, $bits(int), UVM_DEC);
      printer.print_int("nBeamformingSTA", nBeamformingSTA, $bits(int), UVM_DEC);
      foreach (BFRRxAntSet[i]) printer.print_int($sformatf("BFRRxAntSet[%1d]", i), BFRRxAntSet[i], $bits(int), UVM_DEC);
      foreach (STAIDList[i]) printer.print_int($sformatf("STAIDList[%1d]", i), STAIDList[i]+1, $bits(int), UVM_DEC);
      foreach (LDPC[i]) printer.print_int($sformatf("LDPC[%1d]", i), LDPC[i], $bits(int), UVM_DEC);
      foreach (nRx[i]) printer.print_int($sformatf("nRx[%1d]", i), nRx[i], $bits(int), UVM_DEC);
      foreach (htLength[i]) printer.print_int($sformatf("htLength[%1d]", i), htLength[i], $bits(int), UVM_DEC);
      foreach (MCS[i]) printer.print_int($sformatf("MCS[%1d]", i), MCS[i], $bits(int), UVM_DEC);
      foreach (NSTS[i]) printer.print_int($sformatf("NSTS[%1d]", i), NSTS[i], $bits(int), UVM_DEC);
      foreach (Length[i]) printer.print_int($sformatf("Length[%1d]",i), Length[i], $bits(int), UVM_DEC);
      foreach (userPos[i]) printer.print_int($sformatf("userPos[%1d]",i), userPos[i], $bits(int), UVM_DEC);
      printer.print_int("Nr", Nr, $bits(int), UVM_DEC);
      foreach (Nc[i]) printer.print_int($sformatf("Nc[%1d]", i), Nc[i], $bits(int), UVM_DEC);
      foreach (Grouping[i]) printer.print_int($sformatf("Grouping[%1d]", i), Grouping[i], $bits(int), UVM_DEC);
      foreach (Codebook[i]) printer.print_int($sformatf("Codebook[%1d]", i), Codebook[i], $bits(int), UVM_DEC);
      foreach (Feedback[i]) printer.print_int($sformatf("Feedback[%1d]", i), Feedback[i], $bits(int), UVM_DEC);
      foreach (DataBuf[i]) begin
        foreach (DataBuf[i][j])printer.print_int($sformatf("DataBuf[%1d][%1d]",i,j), DataBuf[i][j], $bits(int), UVM_HEX);
      end
    end
  endfunction : do_print

//--------------------------------------------------------------------
// Returns 1 if MU-MIMO Rx or Tx is active, 0 otherwise
//--------------------------------------------------------------------
  virtual function bit is_mumimo_active();
    if ((GroupID != 0) && (GroupID != 63) && FrameFormat == "AC") begin
      return 1;
    end else begin
      return 0;
    end
  endfunction: is_mumimo_active

//--------------------------------------------------------------------
// Returns directory path for PHY payload
//--------------------------------------------------------------------
  virtual function string get_phyPayload_dir_name();
    if (is_mumimo_active()) begin
      return {`STRINGIFY(`PAYLOAD_TXT_DIR), "/", `STRINGIFY(`MUMIMO_DIR)};
    end else begin
      return {`STRINGIFY(`PAYLOAD_TXT_DIR), "/"};
    end
  endfunction: get_phyPayload_dir_name

//--------------------------------------------------------------------
// Returns directory path for sysparam
//--------------------------------------------------------------------
  virtual function string get_sysparam_dir_name();
    if (is_mumimo_active()) begin
      return {`STRINGIFY(`SYSPARAM_TXT_DIR), "/", `STRINGIFY(`MUMIMO_DIR)};
    end else begin
      return {`STRINGIFY(`SYSPARAM_TXT_DIR), "/"};
    end
  endfunction: get_sysparam_dir_name

//--------------------------------------------------------------------
// Return PHY payload file name
//--------------------------------------------------------------------
  virtual function string get_phyPayload_file_name(int user=0);
    if (is_mumimo_active()) begin
      return {`PAYLOAD_NAME, $sformatf(`MUMIMO_SUFFIX, (user+1)), `PAYLOAD_EXT};
    end else begin
      return {`PAYLOAD_NAME, `PAYLOAD_EXT};
    end
  endfunction: get_phyPayload_file_name

//--------------------------------------------------------------------
// Bit-swaps the input data. Necessary as Matlab interprets the
// left-most bit as the LSB, while in *Verilog it is the MSB
//--------------------------------------------------------------------
  function bit [31:0] swap_bits (bit [31:0] in, int MSB = 8);
    bit [31:0] out;
    out = 'b0;
    for (int i=0; i< MSB; i++) begin
      out[i] = in[MSB-1-i];
    end
    return out;
  endfunction: swap_bits

//--------------------------------------------------------------------
// find user index with matching STAID
//--------------------------------------------------------------------
  function int find_he_mu_user(int user);
    int ret;

    if (sta_list.size() == 0) begin
      `uvm_warning(get_type_name(),$sformatf("STA list is empty, no return value. Create first STA list"))
      return -1;
    end
    // find user possition in STA list to determine which payload file
    // will matlab use in execution;
    // proper index is on location of DUT inside RU allocation
    ret = 0;
    foreach (sta_list[i]) begin
      if (sta_list[i] != 2046) begin
        if (sta_list[i] == golden_frame.preamble_header.user_header_he[user].staid_f)
          break;
        ret++;
      end//if
    end//foreach

    return ret;
  endfunction : find_he_mu_user

//--------------------------------------------------------------------
// writes a PHY payload file for user 'user'
//--------------------------------------------------------------------
  virtual function void write_payload_user(int user);
    int       pfp; // phyPayload file pointer
    int       pktLength;
    string    fpn; // phyPayload file full path and name
    int       user_he_mu; //user index in HE MUMIMO

    if (DataBuf[user].size() > 0) begin

      if (is_mumimo_active() && golden_frame.ppdu_format == HE_MU) begin
        user_he_mu = find_he_mu_user(user);
        fpn = {get_phyPayload_dir_name(),get_phyPayload_file_name(user_he_mu)};
      end
      else begin
        fpn = {get_phyPayload_dir_name(),get_phyPayload_file_name(user)};
      end

      pfp = $fopen(fpn,"w");

      // If the file cannot be opened for writing, then show a fatal error
      if (pfp == 0)
        `uvm_fatal(get_type_name(), $sformatf("Unable to open %s for writing", fpn))
      else
        `uvm_info(get_type_name(), $sformatf("Created %s",fpn), UVM_LOW)

      pktLength = DataBuf[user].size();

      for (int i = 0; i < pktLength; i = i + 1) begin
        bit [31:0] tmpBuf;
        tmpBuf = swap_bits(DataBuf[user][i], 8);
        $fwrite(pfp,"%h \n",tmpBuf[7:0]);
      end

      $fclose(pfp);

    end else begin
      if (NSTS[user] != 0) begin
        `uvm_warning(get_type_name(),
        $sformatf("No payload data found, skipping creation of %s. Note that Nsts is set for this user, so Matlab will generate a random payload!",get_phyPayload_file_name(user)))
      end else begin
        `uvm_info(get_type_name(), $sformatf("No payload, skipping creation of %s.",get_phyPayload_file_name(user)), UVM_HIGH)
      end
    end

  endfunction: write_payload_user

//--------------------------------------------------------------------
// Write all possible phyPayload.txt files
//--------------------------------------------------------------------
  function void write_payload();
    if (is_mumimo_active()) begin
      for (int i=0; i < numUsers; i++) begin
        write_payload_user(userPos[i]);
      end
    end
    else begin
      write_payload_user(0);
    end
  endfunction: write_payload

//--------------------------------------------------------------------
// writes the value of variable 'data' from position 'start' with width 'width'
// to file pointer fp in binary format, without bit swapping
//--------------------------------------------------------------------
  function void write_int_bin(int fp, int data, int start=0, int width=8);
    if (fp == 0)
      `uvm_fatal(get_type_name(), $sformatf("File pointer not valid!"))
    $fwrite(fp, "[ %1d", data[start+width-1]);
    for (int i=(start+width-2); i>(start); i--) begin
      $fwrite(fp, " %1d", data[i]);
    end
    $fwrite(fp, " %1d ]\n", data[start]);

  endfunction: write_int_bin

//--------------------------------------------------------------------
// writes values from array of ints to file pointer fp
//--------------------------------------------------------------------
  function void write_int_array(int fp, int data [`MIMO_NR-1:0], int values=`MIMO_NR);
    if (fp == 0) begin
      `uvm_fatal(get_type_name(), $sformatf("File pointer not valid!"))
    end
    if (values > `MIMO_NR) begin
      `uvm_fatal(get_type_name(), $sformatf("Parameter values (%0d) bigger than maximum allowed value (`MIMO_NR) !", values))
    end
    $fwrite(fp, "[ %1d;", data[0]);
    for (int i=1; i  < values-1; i++) begin
      $fwrite(fp, " %1d;", data[i]);
    end
    $fwrite(fp, " %1d ]\n", data[values-1]);
  endfunction: write_int_array

//--------------------------------------------------------------------
// writes values from array of bits to file pointer fp
//--------------------------------------------------------------------
  function void write_bit_array(int fp, bit data [`MIMO_NR-1:0], int values=`MIMO_NR);
    int int_data[`MIMO_NR-1:0];
    if (fp == 0) begin
      `uvm_fatal(get_type_name(), $sformatf("File pointer not valid!"))
    end
    if (values > `MIMO_NR) begin
      `uvm_fatal(get_type_name(), $sformatf("Parameter values (%0d) bigger than maximum allowed value (`MIMO_NR) !", values))
    end
    foreach (data[i]) begin
      if (!$cast(int_data[i], data[i])) begin
        `uvm_fatal(get_type_name(), $sformatf("Cast from BIT to INT failed!"))
      end
    end
    write_int_array(fp, int_data, values);
  endfunction: write_bit_array


//--------------------------------------------------------------------
// writes values from array of bits to file pointer fp
//--------------------------------------------------------------------
  function void write_bf_array(int fp, int data [`NB_STA_MAX], int values=`MIMO_NR);
    if (fp == 0) begin
      `uvm_fatal(get_type_name(), $sformatf("File pointer not valid!"))
    end
    if (values > `NB_STA_MAX) begin
      `uvm_fatal(get_type_name(), $sformatf("Parameter values (%0d) bigger than maximum allowed value (`NB_STA_MAX) !", values))
    end
    $fwrite(fp, "[ %0d;", data[0]);
    for (int i=1; i < values-1; i++) begin
      $fwrite(fp, " %0d;", data[i]);
    end
    $fwrite(fp, " %0d ]\n", data[values-1]);
  endfunction: write_bf_array


//--------------------------------------------------------------------
// Write sysParam.txt file.
//--------------------------------------------------------------------
  function void write_sysparam(direction_e direction,
                               bit agcBypass = 0,
                               bit [3:0] modemconf = 4'b100);

    int        pfs; // sysParam file pointer
    string     fsn; // sysParam file full path and name
    bit [31:0] tmpBuf;

    validate_params(agcBypass);

    fsn = {get_sysparam_dir_name(), `SYSPARAM_NAME, `SYSPARAM_EXT};

    pfs = $fopen(fsn,"w");

    // If the file cannot be opened for writing, then show a fatal error
    if (pfs == 0)
      `uvm_fatal(get_type_name(), $sformatf("Unable to open %s for writing", fsn))
    else
      `uvm_info(get_type_name(), $sformatf("Created %s",fsn), UVM_LOW)

    $fwrite(pfs,"%s\n","Start,");

    $fwrite(pfs,"%s\t","scramblerSeed,                ");
    tmpBuf =  swap_bits(scramblerSeed, 8);
    write_int_bin(pfs, tmpBuf[7:0], 1, 7); // bit 7 (bit 0 after bit swapping) is left out as it indicates the validity of the seed

    // Parameter nTx, Number of transmit chains.
    $fwrite(pfs,"%s\t","NTx,                          ");
    //$fwrite(pfs,"[ %1d ] \n",4);
    $fwrite(pfs,"[ %1d ] \n",nTx);

    $fwrite(pfs,"%s\t","NRx,                          ");
    if( (direction == Tx) && (is_mumimo_active() || (packetLength == 'h0) || BFEn == 1'b1)) begin
      // write NDP parameters
      if (packetLength == 'h0) begin
        write_int_array(pfs, nRx);
      end
      // write MU-MIMO specific parameters
      else if (is_mumimo_active()) begin
        write_int_array(pfs, nRx);
        $fwrite(pfs,"%s\t","GroupID,                      ");
        tmpBuf =  swap_bits(GroupID, 6);
        write_int_bin(pfs, tmpBuf[5:0], 0, 6);

        $fwrite(pfs,"STAIDList,                      ");
        $fwrite(pfs,"[ %0d  %0d]\n", STAIDList[0]+1, STAIDList[1]+1);
        // UserId must match one of the Tx user Positions
        $fwrite(pfs,"UserID,                         [ %0d ] \n",
        golden_frame.preamble_header.user_header[0].user_position_f+1);
        $fwrite(pfs,"TDCSDEn,                      [ 0 ] \n");
      end
      // SU Tx beamformed
      else if (BFEn == 1'b1) begin
        $fwrite(pfs,"[ %1d ] \n", nRx[0]);
        $fwrite(pfs,"STAIDList,                      ");
        $fwrite(pfs,"[ %0d ]\n", STAIDList[0]+1);
        $fwrite(pfs,"TDCSDEn,                      [ 0 ] \n");
      end
    end else begin
      $fwrite(pfs,"[ %1d ] \n",nRx[0]);
    end

    $fwrite(pfs,"%s\t","FrameFormat,                  ");
    if (FrameFormat == "LM11B") begin
      $fwrite(pfs,"[ '%s' ] \n", "LM");
    end else begin
      $fwrite(pfs,"[ '%s' ] \n", FrameFormat);
    end

    $fwrite(pfs,"%s\t","PreambleType11b,              ");
    $fwrite(pfs,"[ %1d ] \n",PreambleType11b);

    $fwrite(pfs,"%s\t","LegacyRate,                   ");
    $fwrite(pfs,"[ %1d ] \n",LegacyRate);

    $fwrite(pfs,"%s\t","MCS,                          ");
    if (!is_mumimo_active()) begin
      $fwrite(pfs,"[ %1d ] \n",MCS[0]);
    end else begin
      write_int_array(pfs, MCS);
    end

    if (LegacyLength > 0) begin
      $fwrite(pfs,"%s\t","LegacyLength,                 ");
      $fwrite(pfs,"[ %1d ] \n",LegacyLength);
    end

    $fwrite(pfs,"%s\t","ChanType,                     ");
    $fwrite(pfs,"[ %1d ] \n",ChanType);

    $fwrite(pfs,"%s\t","FrameBW,                      ");
    $fwrite(pfs,"[ %1d ] \n",FrameBW);

    $fwrite(pfs,"%s \t","RxChannelBW,                 ");
    $fwrite(pfs,"[ %1d ] \n",RxChannelBW);

    $fwrite(pfs,"%s \t","TxChannelBW,                 ");
    $fwrite(pfs,"[ %1d ] \n",TxChannelBW);

    $fwrite(pfs,"%s \t","RxControlChannel,            ");
    $fwrite(pfs,"[ %1d ] \n",RxControlChannel);

    $fwrite(pfs,"%s \t","TxControlChannel,            ");
    $fwrite(pfs,"[ %1d ] \n",TxControlChannel);

    if ((FrameFormat != "LM") && (FrameFormat != "LM11B")) begin
      $fwrite(pfs,"%s\t","htLength,                     ");
      if (!is_mumimo_active()) begin
        $fwrite(pfs,"[ %1d ] \n",htLength[0]);
      end else begin
        write_int_array(pfs, htLength);
      end
    end

    $fwrite(pfs,"%s\t","CarrierFreq_GHz,              ");
    $fwrite(pfs,"[ %1s ] \n",CarrierFreq_GHz);

    if ((GroupID == 0) || (GroupID == 63)) begin
      $fwrite(pfs,"%s\t","Smoothing,                    ");
      $fwrite(pfs,"[ %1d ] \n",smoothing);
    end

    $fwrite(pfs,"%s\t","NESS,                         ");
    $fwrite(pfs,"[ %1d ] \n",NESS);

    $fwrite(pfs,"%s\t","ShortGI,                      ");
    $fwrite(pfs,"[ %1d ] \n",ShortGI);

    $fwrite(pfs,"%s\t","STBC,                         ");
    $fwrite(pfs,"[ %1d ] \n",STBC);

    $fwrite(pfs,"%s\t","NSTS,                         ");
    if (!is_mumimo_active()) begin
      $fwrite(pfs,"[ %1d ] \n",NSTS[0]);
    end else begin
      write_int_array(pfs, NSTS);
    end

    $fwrite(pfs,"%s\t","NOTSOUNDING,                  ");
    $fwrite(pfs,"[ %1d ] \n",NOTSOUNDING);

    $fwrite(pfs,"%s\t","LDPC,                         ");
    if (!is_mumimo_active()) begin
      $fwrite(pfs,"[ %1d ] \n",LDPC[0]);
    end else begin
      write_bit_array(pfs,LDPC);
    end

    // Parameter RxRFName,
    $fwrite(pfs,"%s\t","RxRFName,                     ");
    $fwrite(pfs,"[ '%s' ] \n",RxRFName);

    // Parameter AGCModel
    $fwrite(pfs,"%s\t","AGC,                          ");
    $fwrite(pfs,"[ '%s' ] \n",AGCModel);

    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      // Parameter StateDescription,
      $fwrite(pfs,"%s\t","StateDescription,             ");
      $fwrite(pfs,"[ '%s' ] \n",StateDescription);

      // Parameter rxPAntdBm,
      $fwrite(pfs,"%s\t","RxPAntdBm,                    ");
      $fwrite(pfs,"[ %1d ] \n",RxPAntdBm);
    end else begin
      // Parameter snr
      $fwrite(pfs,"%s\t","SNR,                          ");
      $fwrite(pfs,"[ %1d ] \n",SNR);
    end

    $fwrite(pfs,"%s\t","DirName,                      ");
    $fwrite(pfs,"[ '%s' ] \n", DirName);

    if (GroupID != 0 && GroupID != 63 && direction == Rx) begin
      $fwrite(pfs,"%s\t","UserID,                       ");
      $fwrite(pfs,"[ %0d ] \n", UserID);

      $fwrite(pfs,"%s\t","Tx,                           ");
      $fwrite(pfs,"[ '%s' ] \n", TxModel);
    end

    $fwrite(pfs,"%s\t","TBEBias,                      ");
    $fwrite(pfs,"[ %1d ] \n",TBEBias);

    $fwrite(pfs,"%s\t","Tx4044FilterShift,            ");
    $fwrite(pfs,"[ %1d ] \n",Tx4044FilterShift);

    $fwrite(pfs,"%s\t","TxFENegShiftDel,              ");
    $fwrite(pfs,"[ %1d ] \n",TxFENegShiftDel);

    $fwrite(pfs,"%s\t","Tx4044FilterInDel,            ");
    $fwrite(pfs,"[ %1d ] \n",Tx4044FilterInDel);

    $fwrite(pfs,"%s\t","Tx4044FilterOutDel,           ");
    $fwrite(pfs,"[ %1d ] \n",Tx4044FilterOutDel);

    $fwrite(pfs,"%s\t","TxDigGainLin,                 ");
    $fwrite(pfs,"[ %1d ] \n",TxDigGainLin);

    $fwrite(pfs,"%s\t","EVM,                          ");
    $fwrite(pfs,"[ %1d ] \n",EVM);

    $fwrite(pfs,"%s\t","debugEn,                      ");
    `ifdef RW_DEBUG
      $fwrite(pfs,"[ %1d ] \n",1);
    `else
      $fwrite(pfs,"[ %1d ] \n",debugEn);
    `endif

    $fwrite(pfs,"%s\t","TxDump,                       ");
    $fwrite(pfs,"[ %1d ] \n",TxDump);

    $fwrite(pfs,"%s\t","RxDump,                       ");
    $fwrite(pfs,"[ %1d ] \n",RxDump);

    $fwrite(pfs,"%s\t","AGCDumpEn,                    ");
    $fwrite(pfs,"[ %1d ] \n",AGCDumpEn);

    $fwrite(pfs,"%s\t","TXFRONTENDDumpEn,             ");
    $fwrite(pfs,"[ %1d ] \n",TXFRONTENDDumpEn);

// Specific fields for analog filters imbalance compensation procedure
`ifdef RW_NX_FIQ_COMP_EN
    if (fIQTest == 1'b1) begin
      $fwrite(pfs,"%s\t","TxIQFiltCutError_MHz,       ");
      $fwrite(pfs,"[ 0; 0 ] \n");
      $fwrite(pfs,"%s\t","TxIQFiltComp,               ");
      $fwrite(pfs,"[ 'NON' ] \n");
      $fwrite(pfs,"%s\t","RxIQFiltCutError_MHz,       ");
      $fwrite(pfs,"[ 3; 3 ] \n");
      $fwrite(pfs,"%s\t","RxIQFiltComp,               ");
      $fwrite(pfs,"[ 'FXPT' ] \n");
      $fwrite(pfs,"%s\t","CalDump,                    ");
      $fwrite(pfs,"[ 1 ] \n");
      $fwrite(pfs,"%s\t","fIQEstNite,                 ");
      $fwrite(pfs,"[ %0d ] \n",35000);
      $fwrite(pfs,"%s\t","fIQNbTap,                   ");
      $fwrite(pfs,"[ 20 ] \n");
      $fwrite(pfs,"%s\t","fIQTxDel,                   ");
      $fwrite(pfs,"[ %0d ] \n",9);
      $fwrite(pfs,"%s\t","fIQRxDel,                   ");
      $fwrite(pfs,"[ %0d ] \n",9);
      $fwrite(pfs,"%s\t","fIQEstStep,                 ");
      $fwrite(pfs,"[ %0d ] \n",6);
    end
`endif // RW_NX_FIQ_COMP_EN

    $fwrite(pfs,"%s\t","HTSTFAGCEn,                   ");
    $fwrite(pfs,"[ %1d ] \n",HTSTFAGCEn);

    $fwrite(pfs,"%s\t","Aggregation,                  ");
    $fwrite(pfs,"[ %1d ] \n",Aggregation);

    $fwrite(pfs,"%s\t","DozeNotAllow,                 ");
    $fwrite(pfs,"[ %1d ] \n",DozeNotAllow);

    $fwrite(pfs,"%s\t","TxIQPhase_deg,                ");
    $fwrite(pfs,"[ %1d ] \n",TxIQPhase_deg);

    $fwrite(pfs,"%s\t","TxIQGain_dB,                  ");
    $fwrite(pfs,"[ %1d ] \n",TxIQGain_dB);

    $fwrite(pfs,"%s\t","RxIQPhase_deg,                ");
    $fwrite(pfs,"[ %1d ] \n",RxIQPhase_deg);

    $fwrite(pfs,"%s\t","RxIQGain_dB,                  ");
    $fwrite(pfs,"[ %1d ] \n",RxIQGain_dB);

    $fwrite(pfs,"%s\t","RxDC_mv,                      ");
    $fwrite(pfs,"[ %1d ] \n",RxDC_mv);

    $fwrite(pfs,"%s\t","TxDC_mv,                      ");
    $fwrite(pfs,"[ %1d ] \n",TxDC_mv);

    $fwrite(pfs,"%s\t","CarrierLostStartUs,           ");
    $fwrite(pfs,"[ %1d ] \n",CarrierLostStartUs);

    $fwrite(pfs,"%s\t","CarrierLostStopUs,            ");
    $fwrite(pfs,"[ %1d ] \n",CarrierLostStopUs);


    $fwrite(pfs,"%s\t","packetStartUs,                ");
    $fwrite(pfs,"[ %1d ] \n",packetStartUs);

    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      $fwrite(pfs,"%s\t","ACIRxPAntdBm,                 ");
      $fwrite(pfs,"[ %1d ] \n",ACIRxPAntdBm);

      $fwrite(pfs,"%s\t","ACIStartUs,                   ");
      $fwrite(pfs,"[ %1d ] \n",ACIStartUs);

      $fwrite(pfs,"%s\t","ACIStopUs,                    ");
      $fwrite(pfs,"[ %1d ] \n",ACIStopUs);

      $fwrite(pfs,"%s\t","ACIFcMHz,                     ");
      $fwrite(pfs,"[ %1d ] \n",ACIFcMHz);

      $fwrite(pfs,"%s\t","ACIOperatingMode,             ");
      $fwrite(pfs,"[ '%s' ] \n", ACIOperatingMode);

      $fwrite(pfs,"%s\t","RampUpFreqkHz,                ");
      $fwrite(pfs,"[ %1d ] \n",RampUpFreqkHz);

      $fwrite(pfs,"%s\t","RampUpLength_us,              ");
      $fwrite(pfs,"[ %1d ] \n",RampUpLength_us);

      $fwrite(pfs,"%s\t","RampUpType,                   ");
      $fwrite(pfs,"[ %1d ] \n",RampUpType);

      $fwrite(pfs,"%s\t","RampUpRatio,                  ");
      $fwrite(pfs,"[ %1.1f ] \n",RampUpRatio);

      $fwrite(pfs,"%s\t","RampUpPowDeltadB,             ");
      $fwrite(pfs,"[ %1d ] \n",RampUpPowDeltadB);
    end

    // Do not randomize. Do not specify to use defult value
    $fwrite(pfs,"%s\t","LSIGParityErrForce,           ");
    $fwrite(pfs,"[ %1d ] \n",LSIGParityErrForce);

    // Do not randomize. Do not specify to use defult value
    $fwrite(pfs,"%s\t","HTSIGCRCErrForce,             ");
    $fwrite(pfs,"[ %1d ] \n",HTSIGCRCErrForce);

    // force LSIG value
    if (ForceLSIG != 0) begin
      $fwrite(pfs,"%s\t                               [ %0d ]\n","ForceLSIG,",ForceLSIG);
    end

    // force HTSIG value
    if (ForceHTSIG != 0) begin
      $fwrite(pfs,"%s\t                               [ %0d ]\n","ForceHTSIG,",ForceHTSIG);
    end

    // SMMindex must not be written to sysparam in case of Rx SU, Tx beamformed or Tx MU frames
    if ((direction == Rx && (GroupID == 0 || GroupID == 63)) ||
        (direction == Tx && BFEn == 0 && is_mumimo_active() != 1)
       ) begin
      $fwrite(pfs,"%s\t","smmIdx,                       ");
      $fwrite(pfs,"[ %1d ] \n",smmIdx);

      // Help debug: for NDP Tx, smmIdx should be 0 (hard-coded in Matlab)
      if ( direction == Tx && Length[0] == 0 && smmIdx !== 0) begin
        `uvm_warning(get_type_name(), "NDP Tx with SMMIndex different from zero in Tx vector")
      end
    end

    $fwrite(pfs,"%s\t","FixRandomSeed,                ");
    $fwrite(pfs,"[ %1d ] \n",FixRandomSeed);

    $fwrite(pfs,"%s\t","RNGSeed,                      ");
    $fwrite(pfs,"[ %1d ] \n",RNGSeed);

    $fwrite(pfs,"%s\t","AGCPrint,                     ");
    $fwrite(pfs,"[ %1d ] \n",AGCPrint);

    $fwrite(pfs,"%s\t","ClockOffset_ppm,              ");
    $fwrite(pfs,"[ %1d ] \n",ClockOffset_ppm);

    $fwrite(pfs,"%s\t","FrequencyOffset_ppm,          ");
    $fwrite(pfs,"[ %1d ] \n",FrequencyOffset_ppm);

    // Do not randomize. Do not specify to use defult value
    if(UnsupportedRate == 1) begin
      $fwrite(pfs,"%s\t","ForceLegacyRate,            ");
      $fwrite(pfs,"[ %1d ] \n",ForceLegacyRate);
    end

    if (!is_mumimo_active() && (NDPEn == 0)) begin
      $fwrite(pfs,"%s\t","PartialAID,                   ");
      tmpBuf =  swap_bits(PartialAID, 9);
      write_int_bin(pfs, tmpBuf[8:0], 0, 9);

      $fwrite(pfs,"%s\t","GroupID,                      ");
      tmpBuf =  swap_bits(GroupID, 6);
      write_int_bin(pfs, tmpBuf[5:0], 0, 6);
    end

    if (NDPEn == 1) begin
      $fwrite(pfs,"%s\t","NDPEn,                        ");
      $fwrite(pfs,"[ %1d ] \n",NDPEn);
    end
    else if (direction == Tx && NDPEn == 0) begin
      $fwrite(pfs,"%s\t","NDPEn,                        ");
      $fwrite(pfs,"[ %1d ] \n",NDPEn);
    end

    $fwrite(pfs,"%s\t","BFDump,                       ");
    $fwrite(pfs,"[ %1d ] \n",BFDump);

    $fwrite(pfs,"%s\t","BFEn,                         ");
    $fwrite(pfs,"[ %0d ] \n", BFEn);

    if (UnsupportedMod == 1) begin
      $fwrite(pfs,"%s\t","UnsupportedMod,               ");
      $fwrite(pfs,"[ %1d ] \n",UnsupportedMod);
    end

    if (UnsupportedMCS == 1)  begin
      $fwrite(pfs,"%s\t","ForceMCS,                   ");
      $fwrite(pfs,"[ %1d ] \n",ForceMCS);
    end

    $fwrite(pfs,"%s\t","AGCDelay_us,                  ");
    $fwrite(pfs,"[ %1d ] \n",AGCDelay_us);

    // Parameter SMFFilterSet,
    $fwrite(pfs,"%s\t","SMFFilterSet,                 ");
    $fwrite(pfs,"[ 0 ] \n");

//    // Comes from testcase
   if ((NDPEn == 1 || (direction == Tx && Length[0] == 0)) && FrameFormat == "AC") begin
       // Parameter BFR RX Antenna set
       $fwrite(pfs,"BFRRxAntSet,                  \t");
       $fwrite(pfs,"[ %1d %1d ] \n", BFRRxAntSet[0], BFRRxAntSet[1]);

       // Parameter BFR Nc,
       $fwrite(pfs,"Nc,                           \t");
       if (direction == Tx) begin
          int temp_nc [`NB_STA_MAX];
          foreach (Nc[i]) begin
            if (!$cast(temp_nc[i], (Nc[i]+1))) begin
              `uvm_fatal(get_type_name(), $sformatf("Cast from BIT to INT failed!"))
            end
          end
          write_bf_array(pfs, temp_nc);
       end else begin
         $fwrite(pfs,"[ %1d ] \n", Nc[0]+1);
       end

       $fwrite(pfs,"NcMode,                       \t");
       $fwrite(pfs,"[ '%s' ] \n", NcMode);

       // Parameter Grouping,
       $fwrite(pfs,"Grouping,                     \t");
       if (direction == Tx) begin
          int temp_grouping [`NB_STA_MAX];
          foreach (Grouping[i]) begin
            if (!$cast(temp_grouping[i], Grouping[i])) begin
              `uvm_fatal(get_type_name(), $sformatf("Cast from BIT to INT failed!"))
            end
          end
          write_bf_array(pfs, temp_grouping);
       end else begin
         $fwrite(pfs,"[ %1d ] \n", Grouping[0]);
       end

       // Parameter Codebook,
       $fwrite(pfs,"Codebook,                     \t");
       if (direction == Tx) begin
          int temp_codebook [`NB_STA_MAX];
          foreach (Codebook[i]) begin
            if (!$cast(temp_codebook[i], Codebook[i])) begin
              `uvm_fatal(get_type_name(), $sformatf("Cast from BIT to INT failed!"))
            end
          end
          write_bf_array(pfs, temp_codebook);
       end else begin
         $fwrite(pfs,"[ %1d ] \n", Codebook[0]);
       end

       // Parameter Feedback,
       $fwrite(pfs,"Feedback,                     \t");
       if (direction == Tx) begin
          int temp_feedback [`NB_STA_MAX];
          foreach (Feedback[i]) begin
            if (!$cast(temp_feedback[i], Feedback[i])) begin
              `uvm_fatal(get_type_name(), $sformatf("Cast from BIT to INT failed!"))
            end
          end
          write_bf_array(pfs, temp_feedback);
       end else begin
         $fwrite(pfs,"[ %1d ] \n", Feedback[0]);
       end
   end

// Parameters missing from XLS but used by libdump
    $fwrite(pfs,"%s\t","FrameBWDet,                   ");
    $fwrite(pfs,"[ 'FXPT' ] \n");

    $fwrite(pfs,"%s\t","RxSmoothing,                  ");
    if (smoothing || (FrameFormat == "AC" && (GroupID == 0 || GroupID == 63)))
      $fwrite(pfs,"[ 'FXPT' ] \n");
    else
      $fwrite(pfs,"[ 'NON'] \n");

    $fwrite(pfs,"%s\t","TDDCCompInit,                 ");
    $fwrite(pfs,"[ 'FXPT' ] \n");

    $fwrite(pfs,"%s\t","TDDCTracking,                 ");
    $fwrite(pfs,"[ 'FXPT' ] \n");

    $fwrite(pfs,"%s\t","TDCyclicRotateValUs20MHz,     ");
    $fwrite(pfs,"[ 0.3 ] \n");

    $fwrite(pfs,"%s\t","TDCyclicRotateValUs40MHz,     ");
    $fwrite(pfs,"[ 0.3 ] \n");

`ifdef RW_NX_DERIV_EQU_SS1
    $fwrite(pfs,"%s\t","Equalizer,                    ");
    $fwrite(pfs,"[ 'FXPT_NSS1' ] \n");
`endif

    $fwrite(pfs,"%s\t","ShiftSBEn,                    ");
    $fwrite(pfs,"[ 1 ] \n");

    // Parameters that do not appear in documentation but old code
    // still uses it
    $fwrite(pfs,"%s\t","dcTrigOff,                    ");
    $fwrite(pfs,"[ 0 ] \n");




//NOTE: for revision 30140

    $fwrite(pfs,"dcTrigOff,                       [ 0 ]\n");
`ifdef RW_NX_DERIV_CHBW20ONLY
    $fwrite(pfs,"dcRelockHTTrigOff,               [ 1 ]\n");
    $fwrite(pfs,"dcHTTrigOff20,                   [ 31 ]\n");
    $fwrite(pfs,"foHTTrigOff2020,                 [ 5 ]\n");
    $fwrite(pfs,"dcVHTTrigOff20,                  [ 61 ]\n");
`elsif RW_NX_DERIV_CHBW4020ONLY
    if(modemconf=='d0 || modemconf=='d1) begin
      $fwrite(pfs,"dcRelockHTTrigOff,             [ 1 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                 [ 31 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,               [ 5 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                [ 61 ]\n");
    end
    else begin
      $fwrite(pfs,"dcRelockHTTrigOff,             [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,               [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff2040,               [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4020,               [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff4040,               [ 5 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                 [ 60 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                [ 103 ]\n");
      $fwrite(pfs,"dcHTTrigOff40,                 [ 53 ]\n");
      $fwrite(pfs,"dcVHTTrigOff40,                [ 87 ]\n");
    end
`elsif RW_NX_DERIV_CHBW804020ONLY
    if(modemconf=='d0 || modemconf=='d1) begin
      $fwrite(pfs,"dcRelockHTTrigOff,             [ 1 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                 [ 31 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,               [ 5 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                [ 61 ]\n");
    end
    else if(modemconf=='d2 || modemconf=='d3) begin
      $fwrite(pfs,"dcRelockHTTrigOff,             [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,               [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff2040,               [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4020,               [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff4040,               [ 5 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                 [ 60 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                [ 103 ]\n");
      $fwrite(pfs,"dcHTTrigOff40,                 [ 53 ]\n");
      $fwrite(pfs,"dcVHTTrigOff40,                [ 87 ]\n");
    end
    else begin
      $fwrite(pfs,"dcRelockHTTrigOff,             [ -1 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,               [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff2040,               [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4020,               [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff4040,               [ 5 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                 [ 60 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                [ 235 ]\n");
      $fwrite(pfs,"dcHTTrigOff40,                 [ 53 ]\n");
      $fwrite(pfs,"dcVHTTrigOff40,                [ 190 ]\n");
      $fwrite(pfs,"foHTTrigOff2080,               [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4080,               [ 10 ]\n");
    end
`endif


//NOTE: for revision 28517
/*
`ifdef RW_NX_DERIV_CHBW20ONLY
    $fwrite(pfs,"dcRelockHTTrigOff,                [ 1 ]\n");
    $fwrite(pfs,"dcHTTrigOff20,                    [ 9 ]\n");
    $fwrite(pfs,"foHTTrigOff2020,                  [ 5 ]\n");
    $fwrite(pfs,"dcVHTTrigOff20,                   [ 61 ]\n");
`elsif RW_NX_DERIV_CHBW4020ONLY
    if(modemconf =='d0 || modemconf =='d1)
    begin
      $fwrite(pfs,"dcRelockHTTrigOff,              [ 1 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                  [ 9 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,                [ 5 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                 [ 61 ]\n");
    end
    else
    begin
      $fwrite(pfs,"dcRelockHTTrigOff,              [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,                [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff2040,                [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4020,                [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff4040,                [ 5 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                  [ 16 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                 [ 103 ]\n");
      $fwrite(pfs,"dcHTTrigOff40,                  [ 9 ]\n");
      $fwrite(pfs,"dcVHTTrigOff40,                 [ 87 ]\n");
    end
`elsif RW_NX_DERIV_CHBW804020ONLY
    if(modemconf =='d0 || modemconf =='d1)
    begin
      $fwrite(pfs,"dcRelockHTTrigOff,              [ 1 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                  [ 9 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,                [ 5 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                 [ 61 ]\n");
    end
    else if(modemconf =='d2 || modemconf =='d3)
    begin
      $fwrite(pfs,"dcRelockHTTrigOff,              [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,                [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff2040,                [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4020,                [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff4040,                [ 5 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                  [ 16 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                 [ 103 ]\n");
      $fwrite(pfs,"dcHTTrigOff40,                  [ 9 ]\n");
      $fwrite(pfs,"dcVHTTrigOff40,                 [ 87 ]\n");
    end
    else
    begin
      $fwrite(pfs,"dcRelockHTTrigOff,              [ -1 ]\n");
      $fwrite(pfs,"foHTTrigOff2020,                [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff2040,                [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4020,                [ 3 ]\n");
      $fwrite(pfs,"foHTTrigOff4040,                [ 5 ]\n");
      $fwrite(pfs,"dcHTTrigOff20,                  [ 16 ]\n");
      $fwrite(pfs,"dcVHTTrigOff20,                 [ 243 ]\n");
      $fwrite(pfs,"dcHTTrigOff40,                  [ 9 ]\n");
      $fwrite(pfs,"dcVHTTrigOff40,                 [ 190 ]\n");
      $fwrite(pfs,"foHTTrigOff2080,                [ 1 ]\n");
      $fwrite(pfs,"foHTTrigOff4080,                [ 10 ]\n");
    end
`endif
*/
    // Parameter RxPhaseNoiseFc_kHz,Rx Phase noise Cutoff Frequency
    $fwrite(pfs,"%s\t","RxPhaseNoiseFc_kHz,           ");
    $fwrite(pfs,"[ %1d ] \n",RxPhaseNoiseFc_kHz);

    // Parameter RxPhaseNoisePow_degrms,Rx Phase Noise RMS
    $fwrite(pfs,"%s\t","RxPhaseNoisePow_degrms,       ");
    $fwrite(pfs,"[ %1d ] \n",RxPhaseNoisePow_degrms);

    // Parameter TxPhaseNoiseFc_kHz,Tx Phase noise Cutoff Frequency
    $fwrite(pfs,"%s\t","TxPhaseNoiseFc_kHz,           ");
    $fwrite(pfs,"[ %1d ] \n",TxPhaseNoiseFc_kHz);

    // Parameter TxPhaseNoisePow_degrms,Tx Phase Noise RMS
    $fwrite(pfs,"%s\t","TxPhaseNoisePow_degrms,       ");
    $fwrite(pfs,"[ %1d ] \n",TxPhaseNoisePow_degrms);

    // end clause
    $fwrite(pfs,"%s\n","End,");
    $fclose(pfs);
  endfunction: write_sysparam


//--------------------------------------------------------------------
// Removes previous sysParam and phyPayload files
//--------------------------------------------------------------------
 virtual function void cleanMatlabWorkDir();
    int dirListFp;
    string dirlist;
    string dir_name;

    `ifdef KEEP_MATLAB
      if (FrameCounter != 0) begin
        dir_name = $sformatf("%s/%03d/", `STRINGIFY(`BACKUP_DIR), FrameCounter);

        dirlist = {"mkdir -p ", dir_name, " \n"};
        `shell_exec(dirlist);

        dirlist = {"cp -rf ", get_sysparam_dir_name(), "* ", dir_name, " \n"};
        `shell_exec(dirlist);

        dirlist = {"rm -f ", get_sysparam_dir_name(), "*.txt \n"};
      `shell_exec(dirlist);

      end else begin
        // nothing to backup yet as this is the first frame. Remove files from previous sims
        dirlist = {"rm -rf ", `STRINGIFY(`BACKUP_DIR), " \n"};
        `shell_exec(dirlist);
      end
      FrameCounter++;
    `else
      dirlist = {"rm -f ", get_sysparam_dir_name(), "*.txt \n"};
      `shell_exec(dirlist);
    `endif

  endfunction: cleanMatlabWorkDir

//--------------------------------------------------------------------
// Get AMPDU from PPDU for one user
//--------------------------------------------------------------------
  function void get_PPDU_AMPDU(input PPDU_frame ppdu, int user);
    int payload_size;
    bit [31:0] delimiter;
    int base_ofset;
    int mapped_user;
    int blank_delimiter_size;

    // map proper user position
    mapped_user = (is_mumimo_active()) ? userPos[user] : 0;

    payload_size = DataBuf[mapped_user].size();

    base_ofset = 0; // used as offset for DataBuf, contains the length of previously parsed MPDUs

`ifdef STANDALONE_PHY
    foreach (ppdu.ampdu_frame[user].ampdu_payload[i]) begin
      DataBuf[mapped_user][i] = ppdu.ampdu_frame[user].ampdu_payload[i];
    end
`else
    // VHT, HE_* frames are always aggregated
    if ((ppdu.ampdu_frame[user].aggregated == 1'b1) || (ppdu.ppdu_format inside {VHT,HE_SU,HE_MU,HE_EXT_SU,HE_TB})) begin
      // A-MPDU frame
      foreach (ppdu.ampdu_frame[user].mpdu_frame[frame]) begin

        // add delimiter
        delimiter = ppdu.ampdu_frame[user].delimit[frame].get_data();
        for (int i=0; i<(ppdu.ampdu_frame[user].delimit[frame].size()); i++) begin
          DataBuf[mapped_user][i+base_ofset] = delimiter[8*i +: 8]; // start from position 8*i and use 8 bit counting up
          `uvm_info(get_type_name(), $sformatf("[get_PPDU_AMPDU] :: MPDU frame %0d, adding delimiter DataBuf[%0d][%0d] = 0x%x",
          frame, user, i+base_ofset, DataBuf[mapped_user][i+base_ofset]), UVM_DEBUG)
        end

        // add MPDU
        for (int i=0; i<(ppdu.ampdu_frame[user].mpdu_frame[frame].size()); i++) begin
          DataBuf[mapped_user][i
                               +base_ofset
                               +ppdu.ampdu_frame[user].delimit[frame].size()
                              ] = ppdu.ampdu_frame[user].mpdu_frame[frame].frame[i];

          `uvm_info(get_type_name(), $sformatf("[get_PPDU_AMPDU] :: MPDU frame %0d, adding payload DataBuf[%0d][%0d] = 0x%x",
          frame, user, (i+base_ofset+ppdu.ampdu_frame[user].delimit[frame].size()),
          DataBuf[mapped_user][i+base_ofset+ppdu.ampdu_frame[user].delimit[frame].size()]), UVM_DEBUG)
        end

        // add padding
        if (ppdu.ampdu_frame[user].padding[frame] != 0) begin
          // padding is also present, max 3 octets
          for (int i=0; i<ppdu.ampdu_frame[user].padding[frame]; i++) begin
            DataBuf[mapped_user][ i
                                 +base_ofset
                                 +ppdu.ampdu_frame[user].delimit[frame].size()
                                 +ppdu.ampdu_frame[user].mpdu_frame[frame].size()
                                ] = 0;

            `uvm_info(get_type_name(), $sformatf("[get_PPDU_AMPDU] :: MPDU frame %0d, adding padding DataBuf[%0d][%0d] = 0x%x",
            frame, user, (i+base_ofset+ppdu.ampdu_frame[user].delimit[frame].size()+ppdu.ampdu_frame[user].mpdu_frame[frame].size()),
            DataBuf[mapped_user][i+base_ofset+ppdu.ampdu_frame[user].delimit[frame].size()+ppdu.ampdu_frame[user].mpdu_frame[frame].size()]), UVM_DEBUG)
          end
        end

        //add blank delimiters
        blank_delimiter_size = 0;
        if (ppdu.ampdu_frame[user].blank_delimit.exists(frame)) begin
          delimiter = ppdu.ampdu_frame[user].blank_delimit[frame].get_data();
          blank_delimiter_size = ppdu.ampdu_frame[user].blank_delimit[frame].size();
          for (int bd=0; bd<ppdu.ampdu_frame[user].blank_delimit[frame].blank_delimiter_num; bd++) begin
            for (int i=0; i<4; i++) begin
              DataBuf[mapped_user][ i
                                   +(4*bd)
                                   +base_ofset
                                   +ppdu.ampdu_frame[user].delimit[frame].size()
                                   +ppdu.ampdu_frame[user].mpdu_frame[frame].size()
                                   +ppdu.ampdu_frame[user].padding[frame]
                                  ] = delimiter[8*i +: 8];
            end//for bytes
          end//for blank delimiter
        end

        // for the next MPDU, start filling DataBuf at this offset
        base_ofset +=   ppdu.ampdu_frame[user].mpdu_frame[frame].size()
                      + ppdu.ampdu_frame[user].delimit[frame].size()
                      + blank_delimiter_size
                      + ppdu.ampdu_frame[user].padding[frame];
      end
    end else begin
      // MPDU frame
      for (int i=0; i<(ppdu.ampdu_frame[user].mpdu_frame[0].frame.size()); i++) begin
        DataBuf[mapped_user][i] = ppdu.ampdu_frame[user].mpdu_frame[0].frame[i];
      end
      if (ppdu.ampdu_frame[user].padding[0] !=0) begin
        // padding is also present, max 3 octets
        for (int i=0; i<ppdu.ampdu_frame[user].padding[0]; i++) begin
          DataBuf[mapped_user][i+ppdu.ampdu_frame[user].mpdu_frame[0].size()] = 0;
        end
      end
    end
`endif//STANDALONE_PHY
  endfunction: get_PPDU_AMPDU

  //--------------------------------------------------------------------
  // Get payload size for one user
  //--------------------------------------------------------------------
  function int get_payload_size(input PPDU_frame ppdu, int user);
    int payload_size=0;

    if ((FrameFormat == "LM") || (FrameFormat == "LM11B")) begin
      payload_size = ppdu.preamble_header.leg_length;
    end else if (FrameFormat inside {"MM","GF","AC"}) begin
      payload_size = ppdu.preamble_header.user_header[user].ht_length_f;
    end else begin
      payload_size = ppdu.preamble_header.user_header_he[user].he_length_f;
    end

    `uvm_info(get_type_name(), $sformatf("[get_payload_size] :: User %0d payload size %0d", user, payload_size), UVM_DEBUG)
    return payload_size;
  endfunction: get_payload_size

//--------------------------------------------------------------------
// Get AMPDU from PPDU for all users
//--------------------------------------------------------------------
  function void get_PPDU_payload(input PPDU_frame ppdu);
    int payload_size;
    if ((FrameFormat == "LM") || (FrameFormat == "LM11B")) begin
      `uvm_info(get_type_name(), $sformatf("[get_PPDU_payload] :: FrameFormat %s, loading PPDU data",FrameFormat), UVM_DEBUG)
      payload_size = get_payload_size(ppdu,0);
      DataBuf[0].delete();
      DataBuf[0] = new[payload_size];
      Length[0] = payload_size;
      if (ppdu.ampdu_frame.size() == 0) begin
        `uvm_fatal(get_type_name(), $sformatf("[get_PPDU_payload] :: Missing payload !"))
      end
      get_PPDU_AMPDU(ppdu, 0);
    end else if (ppdu.kind != NDP) begin
      foreach (ppdu.ampdu_frame[i]) begin
        `uvm_info(get_type_name(), $sformatf("[get_PPDU_payload] :: FrameFormat %s, loading PPDU data for user %1d", FrameFormat, userPos[i]), UVM_DEBUG)
        payload_size = get_payload_size(ppdu, i);
        DataBuf[userPos[i]].delete();
        DataBuf[userPos[i]] = new[payload_size];
        Length[userPos[i]] = payload_size;
        get_PPDU_AMPDU(ppdu, i);
      end
    end else begin
      `uvm_info(get_type_name(), $sformatf("[get_PPDU_payload] :: Frame is an NDP frame - skipping payload collection"), UVM_LOW)
    end// if - else
  endfunction: get_PPDU_payload

//--------------------------------------------------------------------
// Loads sysParam parameters from a PPDU frame
//--------------------------------------------------------------------
  function void get_PPDU_params(input PPDU_frame ppdu, direction_e direction);

    // Group ID
    GroupID = ppdu.preamble_header.group_id;

    // Frame format
    if (ppdu.ppdu_format == NON_HT ) begin
      FrameFormat = "LM";
    end else  if (ppdu.ppdu_format == NON_HT_DUP_OFDM ) begin
      FrameFormat = "LM11B";
    end else if (ppdu.ppdu_format == HT_MF ) begin
      FrameFormat = "MM";
    end else if (ppdu.ppdu_format == HT_GF ) begin
      FrameFormat = "GF";
    end else if (ppdu.ppdu_format == VHT ) begin
      FrameFormat = "AC";
    end else if (ppdu.ppdu_format == HE_SU) begin
      FrameFormat = "HE_SU";
    end else if (ppdu.ppdu_format == HE_MU) begin
      FrameFormat = "HE_MU";
    end else if (ppdu.ppdu_format == HE_EXT_SU) begin
      FrameFormat = "HE_EXT_SU";
    end else if (ppdu.ppdu_format == HE_TB) begin
      FrameFormat = "HE_TB";
    end else begin
      `uvm_fatal(get_type_name(), $sformatf("Unknown frame format %s",ppdu.ppdu_format.name()))
    end

    // Map user position
    if (is_mumimo_active()) begin
      numUsers = (ppdu.ppdu_format >= HE_SU) ? ppdu.preamble_header.user_header_he.size() : ppdu.preamble_header.user_header.size();
      for (int index=0; index<numUsers; index++) begin
        userPos[index] = (ppdu.ppdu_format >= HE_SU) ? index : ppdu.preamble_header.user_header[index].user_position_f;
      end
    end else begin
      numUsers = 0;
      for (int index=0; index<`MIMO_NR; index++)
        userPos[index] = 0;
    end

    // Packet length
    if ((FrameFormat == "LM" ) || (FrameFormat == "LM11B")) begin
      packetLength = ppdu.preamble_header.leg_length;
    end else if ((FrameFormat == "MM" ) || (FrameFormat == "GF" )) begin
      packetLength = ppdu.preamble_header.user_header[0].ht_length_f;
      htLength[0]  = ppdu.preamble_header.user_header[0].ht_length_f;
    end else begin
      packetLength = 0;
      foreach (ppdu.preamble_header.user_header[i]) begin
        packetLength += ppdu.preamble_header.user_header[i].ht_length_f;
        htLength[userPos[i]] = ppdu.preamble_header.user_header[i].ht_length_f;
      end
    end

    // Legacy length calculated in PPDU_class
    if( FrameFormat == "LM" || FrameFormat == "LM11B" ||
      ((FrameFormat == "MM") && (direction == Tx) && ppdu.preamble_header.user_header[0].fec_coding_f == 'b1)) begin
      LegacyLength = ppdu.preamble_header.leg_length;
    end else begin
      LegacyLength = -1;
    end

    // Legacy datarate calculated in PPDU_class
    LegacyRate = ppdu.preamble_header.leg_rate;

    // Smoothing
    smoothing = ppdu.preamble_header.smoothing;

    // Aggregation
    Aggregation =  ppdu.preamble_header.aggregated;

    //SMM Index
    smmIdx = ppdu.preamble_header.user_header[0].smm_index_f;

    // FEC coding
    foreach (ppdu.preamble_header.user_header[i])
      LDPC[userPos[i]] = ppdu.preamble_header.user_header[i].fec_coding_f;

    // Partial AID
    if ( (FrameFormat == "AC") && (!is_mumimo_active()))
      PartialAID = ppdu.preamble_header.partial_aid;
    else
      PartialAID = 0;

    // Number of extended streams in MM
    if (FrameFormat == "MM")
      NESS = ppdu.preamble_header.num_extn_ss;
    else
      NESS = 0;

    // Preamble type in DSSS/CCK
    if ((FrameFormat == "LM") && (ppdu.preamble_header.leg_rate < 4'd4))
      PreambleType11b = ppdu.preamble_header.preamble_type;
    else
      PreambleType11b = 0;

    // Space Time Block enable
    STBC = ppdu.preamble_header.stbc;

    // Sounding
    NOTSOUNDING = !ppdu.preamble_header.sounding;

    // Number of space time streams
    foreach (ppdu.preamble_header.user_header[i]) begin
      NSTS[userPos[i]] = ppdu.preamble_header.user_header[i].num_sts_f+1;
    end
    foreach (ppdu.preamble_header.user_header_he[i]) begin
      NSTS[userPos[i]] = ppdu.preamble_header.user_header_he[i].nss_f+1;
    end
    // Doze not allowed
    DozeNotAllow =  ppdu.preamble_header.doze_not_allowed;

    // Mu-MIMO Rx
    if ((direction == Rx) && (is_mumimo_active())) begin
      TxModel = "FLPT";
      UserID = ppdu.preamble_header.mu_mimo_userid+1;
    end

    // Beamformed
    BFEn = ppdu.preamble_header.beamformed;

    // MCS
    if ((FrameFormat == "AC") || (FrameFormat == "MM") || (FrameFormat == "GF")) begin
      foreach (ppdu.preamble_header.user_header[i])
        MCS[userPos[i]] = ppdu.preamble_header.user_header[i].mcs_f;
    end else if ((FrameFormat == "LM") && (ppdu.preamble_header.leg_rate < 4'd4)) begin
      MCS[0] = ppdu.preamble_header.leg_rate + 1;
    end else begin
      for (int index=0; index<`MIMO_NR; index++)
        MCS[index] = 0;
    end

    //Short guard interval
    ShortGI = ppdu.preamble_header.gi_type[0];

    //Number of transmit / receive chains
    // nRX == `RW_NX_DERIV_NRX -1
    // nTx == `RW_NX_DERIV_NTX -1
    if (ppdu.kind == NDP) begin
      nTx = Nr+1;
    end else if ((direction == Rx) && is_mumimo_active()) begin
      nTx =  4;
    end else if((direction == Rx) && (`RW_NX_DERIV_NTX != `RW_NX_DERIV_NRX)) begin
      nTx =  NSTS[0];
    end else if ((direction == Rx) && STBC && (`RW_NX_DERIV_NTX == 1)) begin
      nTx =  ppdu.preamble_header.num_tx+2;
    end else begin
      nTx =  ppdu.preamble_header.num_tx+1;  // num_tx = `RW_NX_DERIV_NTX-1; coding 3'b0 -> 1 antenna
    end


    if(direction == Tx) begin
      // Tx
      if (packetLength == 0) begin
      // DUT NDP Tx   : NRx must match the Nc that will be used for the report
        for (int staidx=0; staidx<nBeamformingSTA; staidx++) begin
           nRx[staidx] = Nc[staidx]+1;
        end
      end else if (is_mumimo_active()) begin
        //WARNING STA with smallest user position must be first in Matlab !
           nRx[0] = Nc[STAIDList[0]]+1;
           nRx[1] = Nc[STAIDList[1]]+1;
      // SU, not beamformed Tx
      end else if (BFEn == 1'b1) begin
        // DUT SU BF Tx : NRx lmust match the Nc that was used for the report
        nRx[0] = Nc[0]+1;
      end else begin
        nRx[0] = nTx;// Force nRx=nTx to use channel Z
      end
    end else begin
      // Rx
      nRx[0] = `RW_NX_DERIV_NRX;
    end

    // Carrier frequency
    if ((FrameFormat == "LM") && (ppdu.preamble_header.leg_rate < 4'd4))
      // DSSS/CCK
      CarrierFreq_GHz = "2.4";
    else
      CarrierFreq_GHz = "6";

    //FrameBW
    if (ppdu.preamble_header.ch_bw == 0) begin
      FrameBW = 20;
    end else if (ppdu.preamble_header.ch_bw == 1) begin
      FrameBW = 40;
    end else if (ppdu.preamble_header.ch_bw == 2) begin
      FrameBW = 80;
    end else if (ppdu.preamble_header.ch_bw == 3) begin
      FrameBW = 160;
      `uvm_fatal(get_type_name(), $sformatf("[get_PPDU_params] :: Unsupported channel bandwith parameter %2d", ppdu.preamble_header.ch_bw))
    end

  endfunction: get_PPDU_params


//--------------------------------------------------------------------
// Sets various sysParam parameters based on testcase
//--------------------------------------------------------------------
  function void set_sys_params(direction_e direction,
                               bit agcBypass,
                               bit [3:0] modemconf,
                               bit [1:0] controlChannel);

    //NoisePowEst, currently only PFCT supported
    NoisePowEst = "PFCT";

    SNR       = 40;   //TODO should be randomized instead!
    RxPAntdBm = -40;  //TODO should be randomized instead!

   // Parameter ControlChannel.
    // Config Bw 20
    if(RxChannelBW == 20) begin
      RxControlChannel = 1;
      TxControlChannel = 1;
    end else if (RxChannelBW == 40) begin  // Config Bw 40
      // Upper channel
      if(controlChannel == 2'b10) begin
        RxControlChannel = 1;
        TxControlChannel = 1;
      end else if(controlChannel == 2'b01) begin
        // Lower Channel
        RxControlChannel = -1;
        TxControlChannel = -1;
      end else  begin
        `uvm_fatal(get_type_name(), "[set_sys_params] :: Config Bw40 and controlChannel is neither 2'b01 or 2'b10 !")
      end
    end else if (RxChannelBW == 80) begin // Config Bw 80
      // Lower channel
      if(controlChannel == 2'b00) begin
        RxControlChannel = -2;
        TxControlChannel = -2;
      end else if(controlChannel == 2'b01) begin
        // Lower Channel
        RxControlChannel = -1;
        TxControlChannel = -1;
      end else if(controlChannel == 2'b10) begin
        // Upper channel
        RxControlChannel = 1;
        TxControlChannel = 1;
      end else if(controlChannel == 2'b11) begin
        // Upper Channel
        RxControlChannel = 2;
        TxControlChannel = 2;
      end
      else
      begin
         `uvm_fatal(get_type_name(), "[set_sys_params] :: Config Bw80 and controlChannel is neither 2'b00 or 2'b01 or 2'b10 or 2'b11 !")
      end
    end

    // Parameter RampUpRatio
    RampUpRatio = real'(RampUpRatioDividend) / real'(RampUpRatioDivisor);


    if (direction == Rx && packetLength == 0) begin
      NDPEn  = 1;
      BFDump = 1;
    end else begin
      NDPEn  = 0;
      BFDump = 0;
    end

    // Specific fields for NDP, MU-MIMO Tx and beamforming
    //WARNING STA with smallest user position must be first in Matlab, testcase must take this into account when setting STAIDLIST!
  //  if ((NOTSOUNDING == 0) || (direction == Tx && packetLength == 0)) begin
    if ((direction == Tx) && (packetLength == 0)) begin
      // Tx NDP
      NDPEn = 1;
      BFDump = 1;
    end
    else if (direction == Tx) begin
      // Tx not NDP
      NDPEn = 0;
      if (is_mumimo_active()) begin
      // Tx MU-MIMO
        BFDump = 1;
      end else if (BFEn)begin
        // Tx SU beamformed
        BFDump = 1;
      end else begin
        // Tx SU not beamformed
        BFDump = 0;
      end
    end

    // Parameter HTSTFAGCEn,
    if((direction == Rx) && (`AGC_ON == 1) && (agcBypass == 0)) begin
      HTSTFAGCEn = 1;
    end else begin
      HTSTFAGCEn = 0;
    end

    // AGC related params
    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      RxRFName          = "KARST";
      AGCModel          = "FXPT";
      StateDescription  = "FPGA_KARST";
    end else begin
      RxRFName          = "NON";
      AGCModel          = "PFCT";
    end

    // Enable Matlab Tx dumps
    TxDump = 1;

    // Enable Matlab Rx dumps
    if(direction == Rx) begin
      RxDump = 1;
    end else begin
      RxDump = 0;
    end

    // Parameter debugEn, if set the Matlab library dumps the intermediate signals.
    debugEn = 1;

    // Parameter AGCDumpEn, if set the Matlab library dumps the AGC intermediate signals.
    if((direction == Rx) && (`AGC_ON == 1) && (agcBypass == 0)) begin
      AGCDumpEn = 1;
    end else begin
      AGCDumpEn = 0;
    end

    // Parameter TXFRONTENDDumpEn, if set the Matlab library dumps the FE intermediate signals.
    if(direction == Tx) begin
      TXFRONTENDDumpEn = 1;
    end else begin
      TXFRONTENDDumpEn = 0;
    end

    // Parameter AGCPrint
    if (direction == Tx) begin
      AGCPrint = 0;
    end else if (uvm_top.get_report_verbosity_level() >= UVM_HIGH) begin
      // if UVM verbosity is equal or higher than UVM_HIGH display AGC debug messages from matlab
      AGCPrint = 1;
    end else begin
      // also enable RW-style debug messages
 `ifdef RW_DEBUG
      AGCPrint = 1;
 `else
      AGCPrint = 0;
 `endif
    end


  // Parameter Tx4044FilterShift for DSSS-CCK TX shift
  Tx4044FilterShift = 0;

  // Parameter TxFENegShiftDel,
  TxFENegShiftDel = 7;

  // Parameter Tx4044FilterInDel,
  Tx4044FilterInDel = 9;

  // Parameter Tx4044FilterOutDel,
  Tx4044FilterOutDel = 1;

  // Parameter TXIQ DIGITAL GAIN
  if (direction == Tx) begin
      TxDigGainLin = 32; //TODO should be randomized, but the randomized value MUST be written to RTL register!
  end else begin
      TxDigGainLin = 32;
  end

  // Reset noiseDuration, if AGC is used then this is set to 10 us
  // else its reset to 0
  if ((`AGC_ON == 1'b1) && (agcBypass == 0)) begin
    packetStartUs      = 16;
  end else begin
    packetStartUs      = 16;
  end


  if ( is_mumimo_active() || BFEn || (NDPEn == 1)  ) begin
    ChanType = 2;
  end else begin
    ChanType = 0;
  end

  if ((`AGC_ON == 0) || (agcBypass == 1)) begin
    `uvm_info(get_type_name(), $sformatf("[set_sys_params] :: AGC is disabled!"), UVM_LOW)
  end

endfunction: set_sys_params

//-------------------------------------------------------------------
  // Calculate the number of spatial streams based on txMCS.
//-------------------------------------------------------------------
  function int get_nss_func(int MCS, string FrameFormat, int FrameBW, int NSTS, int STBC);

    int NSS;

    NSS = 0; // Default value

    if ((FrameFormat == "MM") || (FrameFormat == "GF")) begin
       if (((MCS>=7'd0 ) & (MCS<=7'd7)) |
           ((MCS==7'd32) & (FrameBW == 1)))
         NSS = 1;
       else if (((MCS>=7'd8)  & (MCS<=7'd15)) |
                ((MCS>=7'd33) & (MCS<=7'd38)))
         NSS = 2;
       else if (((MCS>=7'd16) & (MCS<=7'd23)) |
                ((MCS>=7'd39) & (MCS<=7'd52)))
         NSS = 3;
       else if (((MCS>=7'd24) & (MCS<=7'd31)) |
                ((MCS>=7'd53) & (MCS<=7'd76)))
         NSS = 4;
    end
    else if (FrameFormat == "AC") begin
       if (STBC==2'b00)
         NSS = NSTS;
       else if (STBC==2'b01)
         NSS = (NSTS)/2;
    end

    return NSS;
  endfunction: get_nss_func


//-------------------------------------------------------------------
// This function reads Matlab to get H memory content
//-------------------------------------------------------------------
  virtual function void get_h_mem_data();
    int        data_size;       // returned Matlab array size
    bit [31:0] data_read_re[];
    bit [31:0] data_read_im[];

    // Dump BF HFFT
    data_size = getDataSize_v("BF","HFFT");
    data_read_re  = new[data_size];
    data_read_im  = new[data_size];
    bfr_hmem_re   = new[data_size];
    bfr_hmem_im   = new[data_size];

    getComplexData_v(data_read_re,data_read_im,"BF","HFFT");
    for (int j=0;j<data_size;j=j+1) begin
      bfr_hmem_re[j] = data_read_re[j];
      bfr_hmem_im[j] = data_read_im[j];
    end
  endfunction : get_h_mem_data

//-------------------------------------------------------------------
// This function reads Matlab outputs to generate beamforming outputs
//-------------------------------------------------------------------
  virtual function void setup_bf_data(direction_e direction, bit agcBypass);

    int        data_size;       // returned Matlab array size
    bit [31:0] data_read_re[];
    bit [31:0] data_read_im[];

    // fetch from Matlab content of H-memory
    get_h_mem_data();

    // In case of Beamformer scenario, dump the beamforming report
    // for future use
    if (direction == Tx) begin
      for (int idx=0; idx<nBeamformingSTA; idx++) begin
        `uvm_info(get_type_name(), $sformatf("[setup_bf_data] :: Dumping BF report for STATION %0d", idx+1), UVM_DEBUG)
        data_size     = getDataSize_v("BF",$psprintf("report_sta%0d", idx+1));
        data_read_re  = new[data_size];
        bfr           = new[data_size];
        getData_v(data_read_re,"BF",$psprintf("report_sta%0d", idx+1));
        for (int j=0;j<data_size;j=j+1) begin
           bfr[j] = data_read_re[j];
        end
        txBfrLength[idx] = data_size;
        for (int i=0; i< data_size; i=i+1) begin
          txBfrCompressedReport[idx][i] = bfr[i] & 1'b1;
        end
      end

    end else begin
      // Report used in RX - backward compatible
      `uvm_info(get_type_name(), $sformatf("[setup_bf_data] :: Dumping BF report"), UVM_DEBUG)
      data_size     = getDataSize_v("BF","report");
      data_read_re  = new[data_size];
      bfr           = new[data_size];
      getData_v(data_read_re,"BF","report");
      for (int j=0;j<data_size;j=j+1) begin
         bfr[j] = data_read_re[j];
      end

    end

    if ((`AGC_ON == 0) || (agcBypass == 1)) begin
      //AGC BYPASSED
      //Dump BF sig_dB
      data_size = getDataSize_v("AGC","sigma2_dB");
      data_read_re     = new[data_size];
      bfr_sigmadB      = new[data_size];

      getData_v(data_read_re,"AGC","sigma2_dB");
      for (int j=0;j<data_size;j=j+1) begin
        bfr_sigmadB[j] = data_read_re[j];
      end

      if (htLength[0] !=0 &&
          ((MCS[0] > 7 && (FrameFormat == "MM" || FrameFormat == "GF" || FrameFormat == "AC")) ||
           (FrameFormat == "AC" && get_nss_func(MCS[0], FrameFormat, FrameBW, NSTS[0], STBC) == 2)
          ))  begin
        //"Dump Equalyzer sigmae2
        data_size = getDataSize_v("EQUALIZER","sigmae2");
        data_read_re        = new[data_size];
        getData_v(data_read_re,"EQUALIZER","sigmae2");
        sigmae2 = data_read_re[0];
      end
      else begin
        sigmae2 = 0;
      end
    end

  endfunction : setup_bf_data

//--------------------------------------------------------------------
// This function reads Matlab outputs to generate RX FrontEnd outputs and
// RX configuration parameters
//--------------------------------------------------------------------
  virtual function void setup_rx_data(rf_gpio_t rf_gpio,
                                      bit       agcBypass = 0,
                                      bit [3:0] modemconf = 4'b100);

    int        i,j;               // for loop indexes
    int        max_agc_data;      // max value in for loops when agc is disabled
    int        data_size;         // returned Matlab array size
    int        RFdata_size;       // returned Matlab array size
    int        mu_mimo_idx;
    bit [31:0] matlabRXVector1[];
    bit [31:0] data_read_re[];
    bit [31:0] data_read_im[];
    bit [31:0] data_read_rssi[];

    bit [31:0] karst_gain_idx[];
    bit [31:0] karst_matlab_cmd[];  // remnant of old code, unused, DPI code still needs it

    karst_gain_idx = new[2];
    karst_matlab_cmd = new[1];
    karst_matlab_cmd[0] = 32'hFFFFFFFF;
    karst_gain_idx[0] = {26'd0, rf_gpio[5:0]};
    karst_gain_idx[1] = {26'd0, rf_gpio[5:0]};

    // Matlab Rx Vector Dump
    if(FrameFormat inside {"AC","MM"}) begin
      // don't get HtLength from matlab for error scenario
      matlabRXVector1 = new[25];
      getData_v(matlabRXVector1,"RXVECTOR", "rxvector1");
      if (is_mumimo_active()) begin
        mu_mimo_idx = golden_frame.preamble_header.mu_mimo_userid;
        golden_frame.preamble_header.user_header[mu_mimo_idx].ht_length_f = matlabRXVector1[2];
        golden_frame.preamble_header.user_header[mu_mimo_idx].num_sts_f = matlabRXVector1[10]-1;
        golden_frame.preamble_header.leg_length = matlabRXVector1[0];
      end
      else begin
        golden_frame.preamble_header.user_header[userPos[0]].ht_length_f = matlabRXVector1[2];
        golden_frame.preamble_header.user_header[userPos[0]].num_sts_f = matlabRXVector1[10]-1;
        golden_frame.preamble_header.leg_length = matlabRXVector1[0];
      end
    end

    // In case of a DSSS frame set the AntennaSet to 1
    if (FrameFormat inside {"LM", "LM11B"}
        && golden_frame.preamble_header.leg_rate inside {[0:3]}) begin
      golden_frame.preamble_header.antenna_set = 1;
    end

`ifndef RW_MUMIMO_RX_EN
    // when AGC is bypassed antenna set can't be updated, so it is set to
    // 0 because in RxVector collected on MAC-PHY interface will be 0
    if (agcBypass) begin
      golden_frame.preamble_header.antenna_set = 0;
    end
`endif

    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      data_size    = getDataSize_v("AGC", "dsp_rxADC");
      // Initialize ADC data
      data_read_re   = new[data_size];
      data_read_im   = new[data_size];
      ADCDataI       = new[data_size];
      ADCDataQ       = new[data_size];
      getComplexData_v(data_read_re,data_read_im, "AGC", "dsp_rxADC");
      for (j=0;j<data_size;j=j+1) begin
        ADCDataI[j] = data_read_re[j];
        ADCDataQ[j] = data_read_im[j];
      end

      // Initialize RF data
      data_read_rssi = new[data_size];
      `uvm_info(get_type_name(), $sformatf("[setup_rx_data] :: SetGAIN : hash idx = %1d",karst_gain_idx[0]), UVM_DEBUG)
      RFdata_size=setGain_v(data_read_re,data_read_im,data_read_rssi,0,karst_gain_idx,karst_matlab_cmd);

      // Extraction for each antenna
      ADCframe_re = new[data_size];
      ADCframe_im = new[data_size];

      for (j=0; j < `ANTENNA_NR; j++) begin
        for (i=0; i<(data_size/`RW_NX_DERIV_NRX) ; i++) begin
          ADCframe_re[i][j] = ADCDataI[i+j*(data_size/(`RW_NX_DERIV_NRX))];
          ADCframe_im[i][j] = ADCDataQ[i+j*(data_size/(`RW_NX_DERIV_NRX))];
        end
      end

      // Initialize RF data
      data_read_re   = new[RFdata_size];
      data_read_im   = new[RFdata_size];
      RFDataI        = new[RFdata_size];
      RFDataQ        = new[RFdata_size];
      data_read_rssi = new[RFdata_size];

      RFdata_size=setGain_v(data_read_re,data_read_im,data_read_rssi,int'('d16),karst_gain_idx,karst_matlab_cmd);
      for (i=0;i<RFdata_size;i=i+1) begin
        RFDataI[i] = data_read_re[i];
        RFDataQ[i] = data_read_im[i];
      end

      foreach (RFDataI[i])
        `uvm_info(get_type_name(), $sformatf("[setup_rx_data] :: RFDataI[%0d]=0x%x RFDataQ[%0d]=0x%x", i, RFDataI[i], i, RFDataQ[i]), UVM_DEBUG)

      // ADC extraction
      RFframe_re  = new[RFdata_size];
      RFframe_im  = new[RFdata_size];

      for (j=0; j < `ANTENNA_NR; j++) begin
        for (i=0; i<(RFdata_size/`RW_NX_DERIV_NRX) ; i++) begin
          RFframe_re[i][j] = RFDataI[i+j*(RFdata_size/(`RW_NX_DERIV_NRX))];
          RFframe_im[i][j] = RFDataQ[i+j*(RFdata_size/(`RW_NX_DERIV_NRX))];
        end
      end

    end

    ////////////////////////////
    // RXFRONTEND dump
    ////////////////////////////
    if ((`AGC_ON == 0) || (agcBypass == 1)) // AGC BYPASSED
    begin
      // RXFRONTEND dump
      `ifdef RW_NX_DERIV_PATH1
        if (`RW_NX_DERIV_NRX>=2) begin
          max_agc_data = 2;
        end else begin
          max_agc_data = 1;
        end
      `else
        max_agc_data = 1;
      `endif

      if (LegacyRate > 3 || (FrameFormat != "LM" && FrameFormat != "LM11B")) begin
        data_size    = getDataSize_v("RXFRONTEND", "rxFEoutSync20");
        data_read_re = new[data_size];
        data_read_im = new[data_size];
        rxFEout20Re  = new[data_size];
        rxFEout20Im  = new[data_size];
        getComplexData_v(data_read_re,data_read_im, "RXFRONTEND", "rxFEoutSync20");
        for (i=0;i<data_size;i=i+1) begin
          rxFEout20Re[i] = data_read_re[i];
          rxFEout20Im[i] = data_read_im[i];
        end

        rxFEout20_re = new[data_size/`RW_NX_DERIV_NRX];
        rxFEout20_im = new[data_size/`RW_NX_DERIV_NRX];

        for (j=0; j<max_agc_data; j++) begin
          for (i=0; i<(data_size/`RW_NX_DERIV_NRX); i++) begin
            rxFEout20_re[i][j] = rxFEout20Re[i+j*(data_size/(`RW_NX_DERIV_NRX))];
            rxFEout20_im[i][j] = rxFEout20Im[i+j*(data_size/(`RW_NX_DERIV_NRX))];
          end
        end
      end else if (FrameFormat == "LM" || FrameFormat == "LM11B") begin
        data_size     = getDataSize_v("RXFRONTEND", "rxFEoutSyncDSSS");
        data_read_re  = new[data_size];
        data_read_im  = new[data_size];
        rxFEout20DSSSRe = new[data_size];
        rxFEout20DSSSIm = new[data_size];
        getComplexData_v(data_read_re,data_read_im, "RXFRONTEND", "rxFEoutSyncDSSS");
        for (i=0;i<data_size;i=i+1)
        begin
          rxFEout20DSSSRe[i] = data_read_re[i];
          rxFEout20DSSSIm[i] = data_read_im[i];
        end
      end

      if ((RxChannelBW >= 40) && !(modemconf=='d0 || modemconf=='d1) && (LegacyRate > 3 || (FrameFormat != "LM" && FrameFormat != "LM11B"))) begin
        data_size     = getDataSize_v("RXFRONTEND", "rxFEoutSync20s");
        data_read_re  = new[data_size];
        data_read_im  = new[data_size];
        rxFEout20SRe  = new[data_size];
        rxFEout20SIm  = new[data_size];
        getComplexData_v(data_read_re,data_read_im, "RXFRONTEND", "rxFEoutSync20s");
        for (i=0;i<data_size;i=i+1)
        begin
          rxFEout20SRe[i] = data_read_re[i];
          rxFEout20SIm[i] = data_read_im[i];
        end

        rxFEout20S_re = new[data_size/`RW_NX_DERIV_NRX];
        rxFEout20S_im = new[data_size/`RW_NX_DERIV_NRX];

        for (j=0; j<max_agc_data; j++) begin
          for (i=0; i<(data_size/`RW_NX_DERIV_NRX); i++) begin
            rxFEout20S_re[i][j] = rxFEout20SRe[i+j*(data_size/(`RW_NX_DERIV_NRX))];
            rxFEout20S_im[i][j] = rxFEout20SIm[i+j*(data_size/(`RW_NX_DERIV_NRX))];
          end
        end

        data_size    = getDataSize_v("RXFRONTEND", "rxFEoutSync40");
        data_read_re = new[data_size];
        data_read_im = new[data_size];
        rxFEout40Re  = new[data_size];
        rxFEout40Im  = new[data_size];
        getComplexData_v(data_read_re,data_read_im, "RXFRONTEND", "rxFEoutSync40");
        for (i=0;i<data_size;i=i+1)
        begin
          rxFEout40Re[i] = data_read_re[i];
          rxFEout40Im[i] = data_read_im[i];
        end


        rxFEout40_re = new[data_size/`RW_NX_DERIV_NRX];
        rxFEout40_im = new[data_size/`RW_NX_DERIV_NRX];

        for (j=0; j<max_agc_data; j++) begin
          for (i=0; i<(data_size/`RW_NX_DERIV_NRX); i++) begin
            rxFEout40_re[i][j] = rxFEout40Re[i+j*(data_size/(`RW_NX_DERIV_NRX))];
            rxFEout40_im[i][j] = rxFEout40Im[i+j*(data_size/(`RW_NX_DERIV_NRX))];
          end
        end
      end

      if (RxChannelBW == 80 && !(modemconf=='d0 || modemconf=='d1) && !(modemconf=='d2 || modemconf=='d3) && (LegacyRate > 3 || (FrameFormat != "LM" && FrameFormat != "LM11B")))
      begin
        data_size    = getDataSize_v("RXFRONTEND", "rxFEoutSync80");
        data_read_re = new[data_size];
        data_read_im = new[data_size];
        rxFEout80Re  = new[data_size];
        rxFEout80Im  = new[data_size];
        getComplexData_v(data_read_re,data_read_im, "RXFRONTEND", "rxFEoutSync80");
        for (i=0;i<data_size;i=i+1)
        begin
          rxFEout80Re[i] = data_read_re[i];
          rxFEout80Im[i] = data_read_im[i];
        end

        rxFEout80_re = new[data_size/`RW_NX_DERIV_NRX];
        rxFEout80_im = new[data_size/`RW_NX_DERIV_NRX];

        for (j=0; j<max_agc_data; j++) begin
          for (i=0; i<(data_size/`RW_NX_DERIV_NRX); i++) begin
            rxFEout80_re[i][j] = rxFEout80Re[i+j*(data_size/(`RW_NX_DERIV_NRX))];
            rxFEout80_im[i][j] = rxFEout80Im[i+j*(data_size/(`RW_NX_DERIV_NRX))];
          end
        end

      end
    end

    //Dump SNR
    data_size = getDataSize_v("AGC","noise_eval_SNR");
    data_read_re = new[data_size];
    getData_v(data_read_re,"AGC","noise_eval_SNR");
    snr_dB = data_read_re[0];

  endfunction: setup_rx_data


//-----------------------------------------------------------------
// This function has to be called when AGC is ON (not bypassed from a test)
// and PHY RTL changes the value of rf_gpio
//-----------------------------------------------------------------
  function void update_rx_data(rf_gpio_t  rf_gpio,
                               bit        agcBypass = 0,
                               bit [31:0] AGCCount = 0);

    int        data_size;
    bit [31:0] data_read_re[];
    bit [31:0] data_read_im[];
    bit [31:0] data_read_rssi[];

    bit [31:0] karst_gain_idx[];
    bit [31:0] karst_matlab_cmd[];  // remnant of old code, unused, DPI code still needs it

    karst_gain_idx = new[2];
    karst_matlab_cmd = new[1];
    karst_matlab_cmd[0] = 32'hFFFFFFFF;
    karst_gain_idx[0] = {26'd0, rf_gpio[5:0]};
    karst_gain_idx[1] = {26'd0, rf_gpio[5:0]};


    if (rf_gpio == 'h0) begin
      `uvm_fatal(get_type_name(), $sformatf("Setting rf_gpio = 0 will cause a Matlab crash!"))
    end

    `uvm_info(get_type_name(), $sformatf("DEBUG: rf_gpio = %0d AGCCount = %0d agcBypass = %1d", rf_gpio, AGCCount, agcBypass ), UVM_DEBUG)

    // Initialize RF data
    data_size      = RFDataI.size();
    data_read_re   = new[data_size];
    data_read_im   = new[data_size];
    RFDataI        = new[data_size];
    RFDataQ        = new[data_size];
    data_read_rssi = new[data_size];

    data_size = setGain_v(data_read_re, data_read_im,data_read_rssi, AGCCount, karst_gain_idx, karst_matlab_cmd);

    for (int i=0; i<data_size; i++) begin
      RFDataI[i] = data_read_re[i];
      RFDataQ[i] = data_read_im[i];
    end

    // ADC extraction
    RFframe_re  = new[data_size];
    RFframe_im  = new[data_size];

    for (int j=0; j < `ANTENNA_NR; j++) begin
      for (int i=0; i<(data_size/`RW_NX_DERIV_NRX) ; i++) begin
        RFframe_re[i][j] = RFDataI[i+j*(data_size/(`RW_NX_DERIV_NRX))];
        RFframe_im[i][j] = RFDataQ[i+j*(data_size/(`RW_NX_DERIV_NRX))];
      end
    end


  endfunction: update_rx_data

//-----------------------------------------------------------------
// This function creates a TX Frame by reading Matlab outputs
// and generate both input patterns and outputs reference patterns
//-----------------------------------------------------------------
  virtual function void setup_tx_data();
    int i,j;              // for loop indexes
    int data_size;        // returned Matlab array size

    bit signed [`DACWIDTH-1:0]  DAC_ref_re_out[];
    bit signed [`DACWIDTH-1:0]  DAC_ref_im_out[];

    bit [31:0] data_read_re[];
    bit [31:0] data_read_im[];

    `uvm_info(get_type_name(), $sformatf("Debug ELMO"), UVM_DEBUG)

    // clear existing data
    TxframeOut_re.delete();
    TxframeOut_im.delete();

    // DAC
    data_size = getDataSize_v("TXFRONTEND", "txDAC");

    DAC_ref_re_out   = new[data_size];
    DAC_ref_im_out   = new[data_size];
    data_read_re     = new[data_size];
    data_read_im     = new[data_size];

    getComplexData_v(data_read_re,data_read_im, "TXFRONTEND", "txDAC");
    `uvm_info(get_type_name(), $sformatf("Debug ELMO"), UVM_DEBUG)

    for (i=0;i<data_size;i=i+1) begin
      DAC_ref_re_out[i] = data_read_re[i];
      DAC_ref_im_out[i] = data_read_im[i];
    end

    ////////////////////////////////////
    // FRAME GENERATION               //
    ////////////////////////////////////

    if((FrameFormat== "LM") && (LegacyRate < 'd4)) begin
      // for DSSS, store samples in position for Tx path 0
      TxframeOut_re = new[data_size];
      TxframeOut_im = new[data_size];

      for (i=0; i<(data_size) ; i++) begin
        TxframeOut_re[i][0] = DAC_ref_re_out[i];
        TxframeOut_im[i][0] = DAC_ref_im_out[i];
        if (nTx > 1) begin
          // If system has two antennae, the same data is expected on both!
          TxframeOut_re[i][1] = DAC_ref_re_out[i];
          TxframeOut_im[i][1] = DAC_ref_im_out[i];
        end

      end
    end else begin
      // by default matlab generates patterns for 4 antennae (`ANTENNA_NR)
      TxframeOut_re = new[data_size/nTx];
      TxframeOut_im = new[data_size/nTx];

      `uvm_info(get_type_name(), $sformatf("ELMO DEBUG GEN"), UVM_DEBUG)
      for (j=0; j < nTx; j++) begin
        for (i=0; i<(data_size/nTx) ; i++) begin
          TxframeOut_re[i][j] = DAC_ref_re_out[i+j*(data_size/nTx)];
          TxframeOut_im[i][j] = DAC_ref_im_out[i+j*(data_size/nTx)];
        end // for i
      end  // for j
    end // else
  endfunction: setup_tx_data

//--------------------------------------------------------------------
// Executes Matlab DPI
//--------------------------------------------------------------------
  virtual task execute(direction_e direction = Tx,
                       PPDU_frame  ppdu,
                       rf_gpio_t   rf_gpio = `GPIO_WIDTH'd27,
                       bit         agcBypass = 0);

    bit [3:0]   modemconf;
    bit [1:0]   controlChannel;
    bit [31:0]  rdata;

    // check input data to prevent unknown crashes
    if (rf_gpio == 'h0) begin
      `uvm_fatal(get_type_name(), $sformatf("Setting rf_gpio = 0 will cause a Matlab crash!"))
    end

    //get register values
    rdata = m_regmodel.get_mirrored_reg_value("MDMCONF");
    modemconf = rdata[3:0]; //CHBW
    rdata = m_regmodel.get_mirrored_reg_value("RWNXMACSTATICCONFIG");
    controlChannel = rdata[1:0]; //PSSELECT

    // Reset variables
    initialize_vars();

    // store frame locally
    golden_frame = new();
    golden_frame.copy(ppdu);

    // Create sysParam.txt and phyPayload.txt pattern files
    get_PPDU_params(golden_frame, direction);
    set_sys_params(direction, agcBypass, modemconf, controlChannel);
    get_PPDU_payload(golden_frame);
    `uvm_info(get_type_name(), $sformatf("Matlab configuration (direction = %s):\n%s", direction.name(), this.sprint()), UVM_HIGH)

    if (!use_preset_files) begin
      // Remove old patterns
      cleanMatlabWorkDir();
      // and write new ones
      if (golden_frame.kind != NDP) write_payload();
      write_sysparam(direction, agcBypass, modemconf);
    end
    else begin
      `uvm_info(get_type_name(),$sformatf("Matlab will be re-run with preset SysParam and Payload files"),UVM_LOW)
    end

    #1;

    // Call matlab model with the path to sysParam and phyPayload
    // In case of NDP matlab Rx model is always called to generate the bfr.
    `uvm_info(get_type_name(), $sformatf("Executing Matlab (revision %0d)",getRevision_v()), UVM_LOW)

     setup_testcase_v(get_sysparam_dir_name());

    // Get data from Matlab
    if (direction == Tx && (packetLength != 0 || FrameFormat != "AC")) begin
      setup_tx_data();
    end else if (direction == Tx && ( packetLength == 0)) begin
      // Valid only for VHT
      setup_tx_data;
      setup_bf_data(direction,agcBypass);
    end else begin
      setup_rx_data(rf_gpio, agcBypass, modemconf);
      if ((NDPEn == 1 && FrameFormat == "AC" )) begin
        setup_bf_data(direction, agcBypass);
      end
`ifdef RW_NX_FIQ_COMP_EN
      if (fIQTest == 1'b1)
        setup_fiq_data();
`endif
    end

  endtask: execute

//--------------------------------------------------------------------
// This task reads from Matlab fIQ parameters
//--------------------------------------------------------------------
`ifdef RW_NX_FIQ_COMP_EN
  task setup_fiq_data();
    int data_size;

    //getBlockList_v("CAL") ;

    // Get Matlab dumps for fIQ estimation
    data_size           = getDataSize_v("CAL", "fQ");
    m_CAL_fQ            = new[data_size];
    getData_v(m_CAL_fQ, "CAL", "fQ");

    data_size           = getDataSize_v("CAL", "sI");
    m_CAL_sI            = new[data_size];
    getData_v(m_CAL_sI, "CAL", "sI");

    data_size           = getDataSize_v("CAL", "sQ");
    m_CAL_sQ            = new[data_size];
    getData_v(m_CAL_sQ, "CAL", "sQ");

    data_size           = getDataSize_v("CAL", "error");
    m_CAL_error         = new[data_size];
    getData_v(m_CAL_error, "CAL", "error");

  endtask : setup_fiq_data
`endif // RW_NX_FIQ_COMP_EN

//--------------------------------------------------------------------
// Copies reference data (DAC samples) to caller
//--------------------------------------------------------------------
  virtual function void get_ref_data(ref DAC_mu_data_t DAC_data_re[],
                                     ref DAC_mu_data_t DAC_data_im[]);

    if (TxframeOut_re.size() > 0) begin
      // create DAC_data_re and copy contents of TxframeOut_re
      DAC_data_re = new[TxframeOut_re.size()](TxframeOut_re);
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_ref_data] :: No data to copy - run task execute() first!"))
    end
    if (TxframeOut_im.size() > 0) begin
      // create DAC_data_im and copy contents of TxframeOut_im
      DAC_data_im = new[TxframeOut_im.size()](TxframeOut_im);
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_ref_data] :: No data to copy - run task execute() first!"))
    end
  endfunction: get_ref_data

//--------------------------------------------------------------------
// Copies reference data (TXCORE samples) to caller
//--------------------------------------------------------------------
  virtual function void get_txcore_data(ref TXCORE_data_s fe_data[]);
  endfunction : get_txcore_data

//--------------------------------------------------------------------
// Copies reference data (TXCORE samples) to caller
//--------------------------------------------------------------------
  virtual function void get_txmapper_data(ref TXMAP_data_s map_data[]);
  endfunction : get_txmapper_data

//--------------------------------------------------------------------
// Copies golden data (ADC samples) to caller
//--------------------------------------------------------------------
  virtual function void get_golden_data(ref ADC_mu_data_t RF_data_re[],
                                        ref ADC_mu_data_t RF_data_im[]);

    bit [31:0] data[];
    int        size;

    //-------------------------------------------------------------------------
    // get list of error codes
    //-------------------------------------------------------------------------
    //getBlockList_v("RXPSDU");
    size = getDataSize_v("RXPSDU", "fcsOK");
    data = new[size];
    getData_v(data, "RXPSDU", "fcsOK", "REAL");
    fcsOK = data[0];
    size = getDataSize_v("RXPSDU", "lsigErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "lsigErrorcode", "REAL");
    lsigErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "htsigErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "htsigErrorcode", "REAL");
    htsigErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "hesigaErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "hesigaErrorcode", "REAL");
    hesigaErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "hesigbErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "hesigbErrorcode", "REAL");
    vhtsigbErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "vhtsigbErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "vhtsigbErrorcode", "REAL");
    vhtsigbErrorcode = data[0];

    `uvm_info(get_type_name(),
    $sformatf("[get_golden_data] fcsOK (%0d),lsigErrorcode (%0d),htsigErrorcode (%0d), hesigaErrorcode (%0d), hesigbErrorcode (%0d), vhtsigbErrorcode (%0d)",
    fcsOK,lsigErrorcode,htsigErrorcode, hesigaErrorcode, hesigbErrorcode, vhtsigbErrorcode),UVM_LOW)
    //-------------------------------------------------------------------------

    if (RFframe_re.size() > 0) begin
      // create RF_data_re and copy contents of RFframe_re
      RF_data_re = new[RFframe_re.size()](RFframe_re);
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_golden_data] :: No data to copy - run task execute() first!"))
    end
    if (RFframe_im.size() > 0) begin
      // create RF_data_im and copy contents of RFframe_im
      RF_data_im = new[RFframe_im.size()](RFframe_im);
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_golden_data] :: No data to copy - run task execute() first!"))
    end
  endfunction: get_golden_data

//--------------------------------------------------------------------
// Copies golden data (ADC samples) to caller in case AGC is disabled
//--------------------------------------------------------------------
  virtual function void get_raw_data(ref AGC_mu_data_t rxFEout20_re [],
                                     ref AGC_mu_data_t rxFEout20_im [],
                                     ref AGC_mu_data_t rxFEout20S_re[],
                                     ref AGC_mu_data_t rxFEout20S_im[],
                                     ref AGC_mu_data_t rxFEout40_re [],
                                     ref AGC_mu_data_t rxFEout40_im [],
                                     ref AGC_mu_data_t rxFEout80_re [],
                                     ref AGC_mu_data_t rxFEout80_im [],
                                     ref bit [`BFRSIGMADB_WIDTH-1:0] bfrSigmadB[],
                                     ref bit [`SIGMAE2_WIDTH-1:0]    sigmae2,
                                     ref bit [31:0] snr_dB,
                                     ref bit [7:0]  noise_variance);

    bit [31:0] data[];
    int        size;

    //-------------------------------------------------------------------------
    // get list of error codes
    //-------------------------------------------------------------------------
    //getBlockList_v("RXPSDU");
    size = getDataSize_v("RXPSDU", "fcsOK");
    data = new[size];
    getData_v(data, "RXPSDU", "fcsOK", "REAL");
    fcsOK = data[0];
    size = getDataSize_v("RXPSDU", "lsigErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "lsigErrorcode", "REAL");
    lsigErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "htsigErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "htsigErrorcode", "REAL");
    htsigErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "hesigaErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "hesigaErrorcode", "REAL");
    hesigaErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "hesigbErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "hesigbErrorcode", "REAL");
    hesigbErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "vhtsigbErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "vhtsigbErrorcode", "REAL");
    vhtsigbErrorcode = data[0];

    `uvm_info(get_type_name(),
    $sformatf("[get_golden_data] fcsOK (%0d),lsigErrorcode (%0d),htsigErrorcode (%0d), hesigaErrorcode (%0d), hesigbErrorcode (%0d), vhtsigbErrorcode (%0d)",
    fcsOK,lsigErrorcode,htsigErrorcode, hesigaErrorcode, hesigbErrorcode, vhtsigbErrorcode),UVM_LOW)
    //-------------------------------------------------------------------------


    if (this.rxFEout20_re.size() > 0) begin
      rxFEout20_re = new[this.rxFEout20_re.size()](this.rxFEout20_re);
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_raw_data] :: Array rxFEout20_re empty - run task execute() first!"))
    end
    if (this.rxFEout20_im.size() > 0) begin
      rxFEout20_im = new[this.rxFEout20_im.size()](this.rxFEout20_im);
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_raw_data] :: Array rxFEout20_im empty - run task execute() first!"))
    end

    if ((`BW_PARAM_CONFIG == 40) || (`BW_PARAM_CONFIG == 80)) begin
      if (this.rxFEout20S_re.size() > 0) begin
        rxFEout20S_re = new[this.rxFEout20S_re.size()](this.rxFEout20S_re);
      end
      if (this.rxFEout20S_im.size() > 0) begin
        rxFEout20S_im = new[this.rxFEout20S_im.size()](this.rxFEout20S_im);
      end
    end

    if ((`BW_PARAM_CONFIG == 40) || (`BW_PARAM_CONFIG == 80)) begin
      if (this.rxFEout40_re.size() > 0) begin
        rxFEout40_re = new[this.rxFEout40_re.size()](this.rxFEout40_re);
      end
      if (this.rxFEout40_im.size() > 0) begin
        rxFEout40_im = new[this.rxFEout40_im.size()](this.rxFEout40_im);
      end
    end

    if (`BW_PARAM_CONFIG == 80) begin
      if (this.rxFEout80_re.size() > 0) begin
        rxFEout80_re = new[this.rxFEout80_re.size()](this.rxFEout80_re);
      end
      if (this.rxFEout80_im.size() > 0) begin
        rxFEout80_im = new[this.rxFEout80_im.size()](this.rxFEout80_im);
      end
    end

    if (bfr_sigmadB.size() > 0) begin
      bfrSigmadB = new[bfr_sigmadB.size()](bfr_sigmadB);
    end
    sigmae2 = this.sigmae2;
    noise_variance = this.noise_variance;

    snr_dB = this.snr_dB;

  endfunction: get_raw_data

//--------------------------------------------------------------------
// Copies BF report to caller
//--------------------------------------------------------------------
  function void get_bf_report(ref bit beamfrm_report[]);
    if (bfr.size() > 0) begin
      // create beamforming report data and copy content
      beamfrm_report = new[bfr.size()](bfr);
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_bf_report] :: No BF report to copy - run task execute() first!"))
    end
  endfunction: get_bf_report

//--------------------------------------------------------------------
// Copies BF H memory to caller
//--------------------------------------------------------------------
  function hmem_array get_bfr_hmem();

    if (hmem_ref_data.size() > 0) begin
      // create H memory data and copy content
      return hmem_ref_data;
    end else begin
      `uvm_error(get_type_name(), $sformatf("[get_bfr_hmem] :: No BF H memory to copy - run task execute() first!"))
      return hmem_ref_data;
    end
  endfunction : get_bfr_hmem

//--------------------------------------------------------------------
// Copies Rx stream that is expected on MAC-PHY interface, contained of
// RxVector1 - Payload bytes - RxVector2
//--------------------------------------------------------------------
  virtual function void get_rx_stream(ref bit [31:0] rxvector1[],
                                      ref bit [31:0] payload[]);

    int        size;
    int        byte_cnt;
    bit [7:0]  invert;
    bit [31:0] payload_bits[];
    bit [31:0] data[];

    // clear data
    rxvector1.delete();
    payload.delete();

    //-------------------------------------------------------------------------
    // get list of error codes
    //-------------------------------------------------------------------------
    size = getDataSize_v("RXPSDU", "fcsOK");
    data = new[size];
    getData_v(data, "RXPSDU", "fcsOK", "REAL");
    fcsOK = data[0];
    size = getDataSize_v("RXPSDU", "lsigErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "lsigErrorcode", "REAL");
    lsigErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "htsigErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "htsigErrorcode", "REAL");
    htsigErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "hesigaErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "hesigaErrorcode", "REAL");
    hesigaErrorcode = data[0];
    size = getDataSize_v("RXPSDU", "hesigbErrorcode");
    data = new[size];
    getData_v(data, "RXPSDU", "hesigbErrorcode", "REAL");
    hesigbErrorcode = data[0];

    //-------------------------------------------------------------------------

    // get RxVector1 data
    if (fcsOK == 1 &&
        (  (golden_frame.ppdu_format inside {NON_HT,NON_HT_DUP_OFDM} && lsigErrorcode == 0)
        || (golden_frame.ppdu_format == HT_GF && htsigErrorcode == 0)
        || (golden_frame.ppdu_format inside {HT_MF,VHT} && lsigErrorcode == 0 && htsigErrorcode == 0))
    ) begin
      size = getDataSize_v("RXVECTOR", "rxvector1");
      if (size == 0) begin
        `uvm_error(get_type_name(), $sformatf("[get_rx_stream] No Rx vector1 - run task execute() first!"))
      end
      rxvector1 = new[size];
      getData_v(rxvector1, "RXVECTOR", "rxvector1", "REAL");
    end

    // get Rx payload data
    if (golden_frame.kind == NDP) begin
      `uvm_info(get_type_name(),$sformatf("[get_rx_stream] No data in NDP frame"),UVM_LOW)
    end
    else if (golden_frame.kind == SINGLETON && golden_frame.ppdu_format != VHT && fcsOK == 1) begin
      size = getDataSize_v("RXPSDU", "rx_psdu_dec_with_fcs");
      payload = new[size];
      getData_v(payload, "RXPSDU", "rx_psdu_dec_with_fcs", "REAL");

      // Matlab generates inverted bits
      for (int i=0; i<payload.size(); i++) begin
        invert[7] = payload[i][0];
        invert[6] = payload[i][1];
        invert[5] = payload[i][2];
        invert[4] = payload[i][3];
        invert[3] = payload[i][4];
        invert[2] = payload[i][5];
        invert[1] = payload[i][6];
        invert[0] = payload[i][7];
        payload[i]  = invert;
      end
    end
    else if (fcsOK == 1) begin
      size = getDataSize_v("RXPSDU", "rx_psdu_bit_with_fcs");
      payload_bits = new[size];
      getData_v(payload_bits, "RXPSDU", "rx_psdu_bit_with_fcs", "REAL");
      // convert to bytes
      payload = new[size/8];
      byte_cnt = 0;
      for (int i=0; i<payload_bits.size(); i=i+8) begin
        payload[byte_cnt][0] = payload_bits[i+0][0];
        payload[byte_cnt][1] = payload_bits[i+1][0];
        payload[byte_cnt][2] = payload_bits[i+2][0];
        payload[byte_cnt][3] = payload_bits[i+3][0];
        payload[byte_cnt][4] = payload_bits[i+4][0];
        payload[byte_cnt][5] = payload_bits[i+5][0];
        payload[byte_cnt][6] = payload_bits[i+6][0];
        payload[byte_cnt][7] = payload_bits[i+7][0];
        byte_cnt++;
      end
    end

  endfunction : get_rx_stream

//--------------------------------------------------------------------
// Sets new gain value, recalculates ADC data and copies new data to
// caller
//--------------------------------------------------------------------
  virtual function void set_gain(ref ADC_mu_data_t RF_data_re[],
                                 ref ADC_mu_data_t RF_data_im[],
                                 input rf_gpio_t rf_gpio,
                                 input bit AGCBypass = 0,
                                 input bit [31:0] AGCCount);

    update_rx_data(rf_gpio, AGCBypass, AGCCount);
    RF_data_re = new[RFframe_re.size()](RFframe_re);
    RF_data_im = new[RFframe_im.size()](RFframe_im);
  endfunction: set_gain

//--------------------------------------------------------------------
// get de-inter referent data
//--------------------------------------------------------------------
  virtual function soft_bits_array get_soft_bits(bit compressed);
  endfunction : get_soft_bits

//--------------------------------------------------------------------
// get dcm input referent data
//--------------------------------------------------------------------
  virtual function dcm_data_array get_dcm_input_samples();
  endfunction : get_dcm_input_samples

//--------------------------------------------------------------------
// get dcm output referent data
//--------------------------------------------------------------------
  virtual function dcm_data_array get_dcm_output_samples();
  endfunction : get_dcm_output_samples

//--------------------------------------------------------------------
// get de-inter referent data
//--------------------------------------------------------------------
  virtual function soft_bits_array get_deint_soft_bits();
  endfunction : get_deint_soft_bits

//--------------------------------------------------------------------
// get Viterbi referent data
//--------------------------------------------------------------------
  virtual function viterbi_array get_viterbi_samples();
  endfunction : get_viterbi_samples

//--------------------------------------------------------------------
// get LDPC referent data
//--------------------------------------------------------------------
  virtual function ldpc_array get_ldpc_samples();
  endfunction : get_ldpc_samples

//--------------------------------------------------------------------
// get LDPC data referent
//--------------------------------------------------------------------
  virtual function ldpc_data_array get_ldpc_data_samples();
  endfunction : get_ldpc_data_samples

//--------------------------------------------------------------------
// set Rx paramters for TXRx block interface signals
//--------------------------------------------------------------------
  virtual function void set_rxparams();
  endfunction : set_rxparams

//--------------------------------------------------------------------
// get Rx paramters for TXRx block interface signals
//--------------------------------------------------------------------
  virtual function rxparameters_s get_rxparams();
    return rxparams;
  endfunction : get_rxparams

//--------------------------------------------------------------------
// get FFT in interface signals
//--------------------------------------------------------------------
  virtual function fft_data_array get_fft_data_in();
  endfunction : get_fft_data_in

//--------------------------------------------------------------------
// get FFT out interface signals
//--------------------------------------------------------------------
  virtual function fft_data_array get_fft_data_out();
  endfunction : get_fft_data_out

//--------------------------------------------------------------------
// get value of field from matlab block
//--------------------------------------------------------------------
  virtual function void get_block_field_value(ref bit [31:0] value[],
                                              input string block_name,
                                              input string field_name,
                                              input string type_name = "REAL",
                                              input bit    debug = 0);
    int size;

    if (debug) getBlockList_v(block_name);
    size = getDataSize_v(block_name, field_name);
    value.delete();
    value = new[size];
    if (size)
      getData_v(value, block_name, field_name, type_name);
    else
      `uvm_warning(get_type_name(),$sformatf("Block %s field %s has size %0d",block_name,field_name,size))

    if (debug)
      `uvm_info(get_type_name(),
      $sformatf("Value of %s field from %s block: %p",field_name,block_name,value),UVM_LOW)
  endfunction : get_block_field_value

endclass : mdm_data_model

`endif //MDM_DATA_MODEL_SV
