//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : sradulovic
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 30649 $
// $Date: 2017-06-06 11:33:55 +0200 (Tue, 06 Jun 2017) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : UVM RAL register block for TOP
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
///////////////////////////////////////////////////////////////////////////////

`ifndef __TOP_REG_BLOCK_SV__
`define __TOP_REG_BLOCK_SV__


//-------------------------------------------------------------------
// TOP_register_block
//--------------------------------------------------------------------
class TOP_register_block extends REGMODEL_register_block;
  `uvm_object_utils(TOP_register_block)

  // for register access
  uvm_status_e               status;
  uvm_reg_field              fields[$];

  // -------------------------------------------------------------------
  // add callback instance here
  // -------------------------------------------------------------------

  //---new--------------------------------------------------------------
  function new(string name = "TOP_register_block");
    super.new(name);
  endfunction

  //---build------------------------------------------------------------
  virtual function void build();
    super.build();
    // create callback
    // get all fields for Trigger frame
    MACPL.RXHETRIGCOMMONINFOREG_reg.get_fields(fields);
    foreach(fields[i]) begin
      fields[i].set_compare(UVM_CHECK);
      fields[i].set_volatility(0);
    end
      
    MACPL.RXHETRIGUSERINFOREG_reg.get_fields(fields);
    foreach(fields[i]) begin
      fields[i].set_compare(UVM_CHECK);
      fields[i].set_volatility(0);
    end

    //REGMODEL_register_map.set_check_on_read(1);
  endfunction : build

  //---connect----------------------------------------------------------
  virtual function void connect();
    // connect callback with register field
  endfunction : connect

  //--------------------------------------------------------------------
  // helper wrapper functions and tasks for easy acces to registers and fileds
  //--------------------------------------------------------------------

  //--------------------------------------------------------------------
  // resolve reg_block and register from field_name (which is the only
  // compulsory parameter)
  // @return 0 - OK
  //         1 - ERROR
  //--------------------------------------------------------------------
  function bit get_field_identity(ref string        block_name,
                                  ref uvm_reg_block reg_block,
                                  ref string        register_name,
                                  ref uvm_reg       register,
                                  input string      field_name,
                                  ref uvm_reg_field field);

    if (block_name == "") begin
      if (register_name == "") begin
        // neither block_name or regsiter_name are known
        field = this.get_field_by_name(field_name);
        if (field == null) begin
          `uvm_error(get_type_name(), $sformatf("Unable to locate register field %s",  field_name))
          return 1;
        end else begin
          register = field.get_parent();
          if (register == null) begin
            `uvm_error(get_type_name(), $sformatf("Unable to locate register for field %s", field_name))
            return 1;
          end
          register_name = register.get_name();
          reg_block = register.get_parent();
          if (reg_block == null) begin
            `uvm_error(get_type_name(), $sformatf("Unable to locate register block for %s.%s",register_name, field_name))
            return 1;
          end
          block_name = reg_block.get_name();
        end
      end else begin
        //register_name is known, block_name is unknown
        register =  this.get_reg_by_name(register_name);
        if (register == null) begin
          `uvm_error(get_type_name(), $sformatf("Unable to locate register %s", register_name))
          return 1;
        end else begin
          reg_block = register.get_parent();
          if (reg_block == null) begin
            `uvm_error(get_type_name(), $sformatf("Unable to locate register block for register %s",register_name))
            return 1;
          end
          block_name = reg_block.get_name();
          field =register.get_field_by_name(field_name);
          if (field == null) begin
            `uvm_error(get_type_name(), $sformatf("Unable to locate register field %s.%s.%s",block_name, register_name, field_name))
            return 1;
          end
        end
      end
    end

    if (register_name == "") begin
      reg_block = this.get_block_by_name(block_name);
      if (reg_block == null) begin
        `uvm_error(get_type_name(), $sformatf("Unable to locate register block %s",block_name))
        return 1;
      end else begin
        field =reg_block.get_field_by_name(field_name);
        if (field == null) begin
          `uvm_error(get_type_name(), $sformatf("Unable to locate register field %s in block %s", field_name, block_name))
          return 1;
        end else begin
          register = field.get_parent();
          if (register == null) begin
            `uvm_error(get_type_name(), $sformatf("Unable to locate register %s", register_name))
            return 1;
          end
          register_name = register.get_name();
        end
      end
    end

    // this.get_field_by_name() can return the field at once, but in case there are two or more fields with
    // the same name in different registers, we use block_name and register_name to pinpoint the desired register
    reg_block = this.get_block_by_name(block_name);
    if (reg_block == null) begin
      `uvm_error(get_type_name(), $sformatf("Unable to locate register block %s",block_name))
      return 1;
    end else begin
      register = reg_block.get_reg_by_name(register_name);
      if (register == null) begin
        `uvm_error(get_type_name(), $sformatf("Unable to locate register %s.%s", block_name, register_name))
        return 1;
      end else begin
        field = register.get_field_by_name(field_name);
        if (field == null) begin
          `uvm_error(get_type_name(), $sformatf("Unable to locate register field %s.%s.%s",block_name, register_name, field_name))
          return 1;
        end
      end
    end

    return 0; //field found
  endfunction: get_field_identity

  //--------------------------------------------------------------------
  // resolve reg_block and register from register_name (which is the
  // only compulsory parameter)
  // @return 0 - OK
  //         1 - ERROR
  //--------------------------------------------------------------------
  function bit get_reg_identity(ref string        block_name,
                                ref uvm_reg_block reg_block,
                                input string      register_name,
                                ref uvm_reg       register);

    if (block_name == "") begin
      register =  this.get_reg_by_name(register_name);
      if (register == null) begin
        `uvm_error(get_type_name(), $sformatf("Unable to locate register %s", register_name))
        return 1;
      end
      reg_block = register.get_parent();
      if (reg_block == null) begin
        `uvm_error(get_type_name(), $sformatf("Unable to locate register block for register %s",register_name))
        return 1;
      end
      block_name = reg_block.get_name();
    end else begin
      reg_block = this.get_block_by_name(block_name);
      if (reg_block == null) begin
        `uvm_error(get_type_name(), $sformatf("Unable to locate register block %s",block_name))
        return 1;
      end else begin
        register = reg_block.get_reg_by_name(register_name);
        if (register == null) begin
          `uvm_error(get_type_name(), $sformatf("Unable to locate register %s.%s", block_name, register_name))
          return 1;
        end
      end
    end

    return 0; //register found
  endfunction: get_reg_identity

  //--------------------------------------------------------------------
  // get value of specific register field. register_name and
  // register_block are optional
  //--------------------------------------------------------------------
  task get_field_value(ref uvm_reg_data_t data,
                       input string       field_name,
                       string             register_name="",
                       string             block_name=""  );

    uvm_reg_block reg_block;
    uvm_reg       register;
    uvm_reg_field field;
    bit  [7:0]    lsb_pos;
    bit  [7:0]    fld_width;
    bit [31:0]    mask;

    if (get_field_identity(block_name, reg_block, register_name, register, field_name, field))
      return; //field not found
    lsb_pos  = field.get_lsb_pos();
    fld_width = field.get_n_bits();
    mask = ((2**fld_width)-1) << lsb_pos;
    register.read(status, data);
    data = (data & mask) >> lsb_pos;

    `uvm_info(get_type_name(),
    $sformatf("Value of register field %s.%s.%s is 0x%4x (lsb_pos: %2d fld_width: %2d mask: 0x%4x)",
    block_name, register_name, field_name, data, lsb_pos, fld_width, mask), UVM_DEBUG)
  endtask: get_field_value

  //--------------------------------------------------------------------
  // get value of specific register field. If block_name or register_name
  // is not specified, the code will try to guess them
  //--------------------------------------------------------------------
  task set_field_value(uvm_reg_data_t dataw,
                       string         field_name,
                       string         register_name="",
                       string         block_name="" );

    uvm_reg_block reg_block;
    uvm_reg         register;
    uvm_reg_field   field;
    bit  [7:0]      lsb_pos;
    bit  [7:0]      fld_width;
    uvm_reg_data_t  mask, data, old_data;

    if (get_field_identity(block_name, reg_block, register_name, register, field_name, field))
      return;//field not found
    lsb_pos  = field.get_lsb_pos();
    fld_width = field.get_n_bits();
    mask = ((2**fld_width)-1) << lsb_pos;
    register.read(status, data);
    old_data = data;
    data = (data & ~mask) | ((dataw << lsb_pos) & mask);
    register.write(status, data);

    `uvm_info(get_type_name(),
    $sformatf("Writing 0x%0x to register field %s.%s.%s (new data: 0x%8x old data: 0x%8x lsb_pos: %2d fld_width: %2d mask: 0x%8x)",
    dataw, block_name, register_name, field_name, data, old_data, lsb_pos, fld_width, mask), UVM_DEBUG)
  endtask: set_field_value

  //--------------------------------------------------------------------
  // set register value
  //--------------------------------------------------------------------
  task set_reg_value(uvm_reg_data_t dataw, string register_name, string block_name="");
    uvm_reg_block reg_block;
    uvm_reg       register;

    if (get_reg_identity(block_name, reg_block, register_name, register))
      return; //register not found
    register.write(status, dataw);
    `uvm_info(get_type_name(),
    $sformatf("Writing 0x%0x to register %s.%s",dataw, block_name, register_name), UVM_DEBUG)
  endtask: set_reg_value

  //--------------------------------------------------------------------
  // get register value
  //--------------------------------------------------------------------
  task get_reg_value(ref uvm_reg_data_t data, input string register_name, string block_name="");
    uvm_reg_block reg_block;
    uvm_reg       register;

    if (get_reg_identity(block_name, reg_block, register_name, register))
      return; //register not found
    register.read(status, data);
    `uvm_info(get_type_name(),
    $sformatf("Read 0x%0x from register %s.%s",data, block_name, register_name), UVM_DEBUG)
  endtask: get_reg_value

  //--------------------------------------------------------------------
  // get mirrored register value
  //--------------------------------------------------------------------
  function bit [31:0] get_mirrored_reg_value(string register_name, string block_name="");
    uvm_reg_block reg_block;
    uvm_reg       register;
    bit [31:0]    ret;

    if (get_reg_identity(block_name, reg_block, register_name, register))
      return 0; //register not found
    ret = register.get_mirrored_value();
    `uvm_info(get_type_name(),
    $sformatf("Read mirrored value 0x%0x from register %s.%s",ret, block_name, register_name), UVM_DEBUG)

    return ret;
  endfunction : get_mirrored_reg_value

  //--------------------------------------------------------------------
  // predict a field value in register
  //--------------------------------------------------------------------
  function void predict_field_value(uvm_reg_data_t value,
                                    string field_name,
                                    uvm_predict_e kind = UVM_PREDICT_READ,
                                    string register_name="",
                                    string block_name="");
    uvm_reg_block   reg_block;
    uvm_reg_field   field;
    uvm_reg         register;

    if (get_field_identity(block_name, reg_block, register_name, register, field_name, field))
      return;

    assert(field.predict(.value(value), .kind(kind), .path(UVM_BACKDOOR)));
    `uvm_info(get_type_name(),
    $sformatf("Predicted value 0x%0x for field %s, in %s register", value, field_name, register_name), UVM_DEBUG)
    
  endfunction

endclass : TOP_register_block
`endif // __TOP_REG_BLOCK_SV__
