//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_CORE_HSU_ENC_SEQ_SV
`define MAC_CORE_HSU_ENC_SEQ_SV

  typedef struct{
                 bit[31:0] start_address; // Start address of the message (or part of the message) in the shared RAM
                 bit[15:0] length;        // Length of the message in bytes
                 bit       first_buf;     // The first chunk of the message
                 bit       last_buf;      // The last  chunk of the message
                 } tkip_mem_s;

//************************************************************
// Represents memory content for both TKIP and CMAC mode
//************************************************************
class hsu_mem_layout_tkip_cmac;

  int unsigned   buffer_length_max = 65535; // Maximum length of the message
  int unsigned   buffer_length_min = 100;   // Minimal length of the message
  tkip_mem_s     tkip_mem_q[$];

  // Dynamic arrays for randomization
  rand bit[31:0] start_address[];
  rand bit[15:0] length[];

  rand int unsigned  buffer_length; // Length of the message

  // Number of "subbuffers", depends on the original buffer length
  rand int unsigned num_of_sub_bufs;

  // Message (data bytes)
  rand bit           [7:0]  message[];
  // Keys (only key[0]/key[1] are used in TKIP mode)
  rand bit           [31:0]  keys[$];

  // 0 -> TKIP, 1-> CMAC
  bit cmac_mode = 1'b0;

  constraint num_of_sub_bufs_c {
                                (buffer_length < 4 )          -> num_of_sub_bufs == 1;
                                (buffer_length inside {[4:16]} )        -> num_of_sub_bufs inside {[1:3]};
                                (buffer_length inside {[17:100]} )      -> num_of_sub_bufs inside {[1:8]};
                                (buffer_length inside {[101:1000]} )    -> num_of_sub_bufs inside {[1:16]};
                                (buffer_length inside {[1001:65535]})   -> num_of_sub_bufs inside {[1:buffer_length/1001]};
                                }


  constraint buffer_length_c {
                              buffer_length inside {[buffer_length_min:buffer_length_max]};
                             }

  constraint order_c {
                      solve  buffer_length before  num_of_sub_bufs;
                      solve length, start_address before  num_of_sub_bufs;
                      }

  // Specifies sizes of rand arrays
  constraint size_of_tkip_mem_q_c {
                                   start_address.size()     ==  num_of_sub_bufs;
                                   length.size()            ==  num_of_sub_bufs;
                                  }

  // Length of all "subbuffers"
  constraint tkip_buffs_len_c {
                                // Length of all subbuffers should be equal to the totall lenght of the message
                                length.sum() with (int'(item)) ==  buffer_length;
                                // None of the subbuffers should have 0 length
                                foreach (length[j])
                                  length[j] != 0;

                                // Equal probability for odd and even length of all subbuffers (including the last one)
                                foreach (length[j])
                                  if(cmac_mode == 1'b0)
                                    length[j][0] dist { 1'b0 := 1, 1'b1 := 1 }; // Only for TKIP mode

                                // Size is multiple of 4 for all chunks except the last one !!! Different
                                foreach (length[j])
                                  if(j < length.size()-1 && cmac_mode == 1'b1)
                                    length[j][2:0] == 3'b000;                                       // Only for CMAC mode

                                // Specifies presence of certain chunks (1/3 are valid only for TKIP mode)
                                (buffer_length inside {[101:65535]} && !cmac_mode) -> 1 inside {length};
                                (buffer_length inside {[101:65535]} && !cmac_mode) -> 3 inside {length};
                              }


  constraint order_start_address_c {
                                    solve  buffer_length,length before  start_address;
                                   }

  constraint start_address_c {
                              // Address of the next subbuffer (including the possible hole)
                              foreach (start_address[j])
                                if(j > 0) start_address[j] > start_address[j-1] + length[j-1];

                               foreach (start_address[j])
                                 start_address[j] inside {[0:256*1024-buffer_length]};

                                // Start address is aligned (is it obvious?)
                                foreach (start_address[j])
                                  start_address[j][1:0] == 2'b00;
                             }

  // Message
  constraint message_size_c  { message.size() == buffer_length; }

  // Message data represents "counter"
  constraint message_counter_debug_c { foreach(message[k]) message[k] == k[7:0]; }

  // Keys
  constraint keys_size_c     { keys.size() == 4; }
  // For the purpose of debug
  constraint keys_not_zero_c { foreach(keys[k]) keys[k] != 32'h0; }
  // ??? constraint keys_unique_c   { foreach(keys[k]) unique {keys[k]}; }


  function new (input bit[15:0] len_min = 1, len_max  = 65535, cmac = 1'b0);
    buffer_length_min    = len_min;
    buffer_length_max    = len_max;
    cmac_mode            = cmac;
  endfunction : new

  function bit[7:0] get_msg_byte(input bit[15:0] adr);
    return message[adr];
  endfunction : get_msg_byte

  function int get_msg_size();
    return message.size();
  endfunction : get_msg_size

  function void display_mem_layout();
    `uvm_info("Debug", $sformatf(" Total buffer size %d. Num of chunks %d. CMAC %d.",
    buffer_length, num_of_sub_bufs, cmac_mode), UVM_HIGH)
    foreach(tkip_mem_q[i]) begin
      `uvm_info("Debug", $sformatf(" Subbuffer %d. Start Address 0x%8x Length %d.",
      i, tkip_mem_q[i].start_address, tkip_mem_q[i].length), UVM_HIGH)
    end
  endfunction : display_mem_layout


  function void post_randomize();
    foreach(length[i]) begin
    // Create the queue
    tkip_mem_q[i].length        = length[i];
    tkip_mem_q[i].start_address = start_address[i];
    // Clear buffer flags (for first and last buffers)
    tkip_mem_q[i].first_buf = 1'b0;
    tkip_mem_q[i].last_buf  = 1'b0;
    end
    // Mark first and last buffer(s)
    tkip_mem_q[0].first_buf = 1'b1;                  // First
    tkip_mem_q[tkip_mem_q.size()-1].last_buf = 1'b1; // Last
  endfunction : post_randomize


  function bit[7:0] get_message_byte(input bit[15:0] adr);
    if(adr > buffer_length - 1) begin
      `uvm_error("Debug", $sformatf("Attempt to read non-init. SRAM location Addres %d. Max Address %d.", adr, buffer_length - 1))
     end
     return message[adr];
  endfunction : get_message_byte

  // Returns one of the keys (key0...key3)
  function bit[31:0] get_key_word(input bit[1:0] adr);
    return keys[adr];
  endfunction : get_key_word

endclass : hsu_mem_layout_tkip_cmac


//$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$


class mac_core_hsu_enc_seq extends wlan_seq_base;

  `uvm_object_utils(mac_core_hsu_enc_seq)

   hsu_mem_layout_tkip_cmac hsu_mem_layout;
   bit          [63:0] mic_from_model; // MIC calculated by DPI C function(s)

   // Allows to avoid the following warning during runtime
   // Randomization constraint has this warning:
   // Warning: The generated size 63176 of the queue/dynamic array is extremely large.
   bit                 short_msg_buffer = 1'b0;
   int unsigned        msg_buffer_max_len = (short_msg_buffer) ? 16383 : 65535;
   int unsigned        msg_buffer_min_len = 100;

   // Number of test iterations for TKIP/CMAC
   int unsigned        num_of_tkip_iterations = 6;
   int unsigned        num_of_cmac_iterations = 6;

  function new (string name = "mac_core_hsu_enc_seq");
    super.new (name);
  endfunction : new

   virtual task prepare_message_subbuffers(
                                           hsu_mem_layout_tkip_cmac arg
                                          );
    bit[31:0] tmp;
    bit[31:0] byte_cnt = 0;
    bit[31:0] address;
    foreach(arg.tkip_mem_q[k]) begin
    // For each subbuffer ...
      for(int unsigned i = 0; i < arg.tkip_mem_q[k].length / 4; i++) begin
        tmp[7:0]   = arg.get_message_byte(byte_cnt + i*4 + 0);
        tmp[15:8]  = arg.get_message_byte(byte_cnt + i*4 + 1);
        tmp[23:16] = arg.get_message_byte(byte_cnt + i*4 + 2);
        tmp[31:24] = arg.get_message_byte(byte_cnt + i*4 + 3);
        address    = arg.tkip_mem_q[k].start_address + (i << 2);
        `uvm_info("Debug", $sformatf("Prepare message in SRAM Address 0x%8x. Data 0x%8x.", address, tmp), UVM_HIGH)
        write_word_sram(address, tmp);
      end
      if(arg.tkip_mem_q[k].length % 4 != 0) begin
        address = (arg.tkip_mem_q[k].length > 3) ? address + 4 : arg.tkip_mem_q[k].start_address;
        tmp        = '0;
        tmp[7:0]   = (arg.tkip_mem_q[k].length % 4 > 0) ?  arg.get_message_byte(byte_cnt + (arg.tkip_mem_q[k].length / 4)*4 + 0) : '0;
        tmp[15:8]  = (arg.tkip_mem_q[k].length % 4 > 1) ?  arg.get_message_byte(byte_cnt + (arg.tkip_mem_q[k].length / 4)*4 + 1) : '0;
        tmp[23:16] = (arg.tkip_mem_q[k].length % 4 > 2) ?  arg.get_message_byte(byte_cnt + (arg.tkip_mem_q[k].length / 4)*4 + 2) : '0;
        `uvm_info("Debug", $sformatf("Prepare message in SRAM Address 0x%8x. Data 0x%8x.", address, tmp), UVM_HIGH)
        write_word_sram(address, tmp);
      end
        byte_cnt += arg.tkip_mem_q[k].length; // Increment pointer
    end // foreach(arg.tkip_mem_q[k])
  endtask : prepare_message_subbuffers


   virtual task process_message_subbuffers(
                                            hsu_mem_layout_tkip_cmac  arg,
                                            input bit            cmac = 1'b0
                                          );
    bit[31:0] key;
    bit[31:0] tmp;
    foreach(arg.tkip_mem_q[k]) begin
      // For each subbuffer ...
      tmp  = '0;

      if(arg.tkip_mem_q[k].first_buf == 1'b1) begin
        // Load keys
        key = arg.get_key_word(0);
        m_regmodel.set_reg_value(key,           "KEY_TAB0");
        key = arg.get_key_word(1);
        m_regmodel.set_reg_value(key,           "KEY_TAB1");
        if(cmac) begin
          // Load keys
          key = arg.get_key_word(2);
          m_regmodel.set_reg_value(key,           "KEY_TAB2");
          key = arg.get_key_word(3);
          m_regmodel.set_reg_value(key,           "KEY_TAB3");
        end
        tmp[4] = 1'b1;  // First
      end
      if(arg.tkip_mem_q[k].last_buf == 1'b1) begin
        tmp[5] = 1'b1;  // Last
      end
      tmp[8] = (cmac) ? 1'b1 : 1'b0; // TKIP or CMAC mode
      tmp[0] = 1'b1;  // Start
      `uvm_info("Debug", $sformatf(" Processing subbuffer %d Start Address 0x%x  Length %d. First Buffer %b Last Buffer %b. CMAC %b",
                                    k, arg.tkip_mem_q[k].start_address, arg.tkip_mem_q[k].length, arg.tkip_mem_q[k].first_buf,
                                    arg.tkip_mem_q[k].last_buf, tmp[8]), UVM_HIGH)
      m_regmodel.set_reg_value(arg.tkip_mem_q[k].start_address, "SOURCE_ADDR");
      m_regmodel.set_reg_value(arg.tkip_mem_q[k].length,         "LENGTH");

      m_regmodel.set_reg_value(tmp, "CONTROL"); // Start HSU operation
      wait_for_hsu_done();
      read_hsu_remaining();
      if(!arg.tkip_mem_q[k].last_buf) begin
        save_corrupt_restore_hsu_remaining();
      end
      read_hsu_mic((arg.tkip_mem_q[k].last_buf) ? 1'b1 : 1'b0); // Perform comparioson only after the last buffer processing
      hsu_done_clear();
    end // foreach(arg.tkip_mem_q[k])
  endtask : process_message_subbuffers

  // *****************************************
  // Wait until DONE_SET is set
  // *****************************************
  virtual task wait_for_hsu_done();
    uvm_reg_data_t data;
    do begin
      m_regmodel.get_reg_value(data, "STATUS_SET");
      `uvm_info(get_type_name(), $sformatf("STATUS_SET = %b.", data), UVM_HIGH)
    end
    while (data[0] != 1'b1);
  endtask : wait_for_hsu_done

  // *****************************************
  // Read MIC
  // *****************************************
  virtual task read_hsu_mic(input bit perform_comparison = 1'b1);
    uvm_reg_data_t data_l, data_h;
      m_regmodel.get_reg_value(data_l, "MIC_TAB0");
      `uvm_info(get_type_name(), $sformatf("MIC_TAB0 = 0x%8x.", data_l), UVM_HIGH)
      m_regmodel.get_reg_value(data_h, "MIC_TAB1");
      `uvm_info(get_type_name(), $sformatf("MIC_TAB1 = 0x%8x.", data_h), UVM_HIGH)
      if(perform_comparison == 1'b1) begin
        if({data_h[31:0],data_l[31:0]} != mic_from_model[63:0]) begin
          `uvm_error(get_type_name(), $sformatf("MIC calculated by the reference model does not match one calculated by RTL. 0x%16x. 0x%16x.", {data_h[31:0],data_l[31:0]}, mic_from_model[63:0]))
        end
        else begin
          `uvm_info(get_type_name(), "MIC comparison PASSED.", UVM_HIGH)
        end
      end // if(perform_comparison == 1'b1)
  endtask : read_hsu_mic


  // *****************************************
  // Read remaining
  // *****************************************
  virtual task read_hsu_remaining();
    uvm_reg_data_t data;
    m_regmodel.get_reg_value(data, "REMAINING");
    // Decode and display
    `uvm_info(get_type_name(), $sformatf("Remaining LENGTH %2d. BYTE2 0x%2x. BYTE1 0x%2x. BYTE0 0x%2x.", data[25:24], data[23:16], data[15:8], data[7:0]), UVM_HIGH)
  endtask : read_hsu_remaining


  // *****************************************
  // Read remaining
  // *****************************************
  virtual task save_corrupt_restore_hsu_remaining();
    uvm_reg_data_t data;
    // Save
    m_regmodel.get_reg_value(data, "REMAINING");
    // Corrupt
    m_regmodel.set_reg_value(~data[31:0], "REMAINING");
    // Restore
    m_regmodel.set_reg_value(data[31:0], "REMAINING");
  endtask : save_corrupt_restore_hsu_remaining



  // *****************************************
  // Acknowledge one ore more interrupt flag(s)
  // *****************************************
  virtual task hsu_done_clear();
    m_regmodel.set_reg_value(32'h1, "STATUS_CLEAR");
  endtask : hsu_done_clear

  // *****************************************
  //
  // *****************************************
  virtual task body();
    super.body();

    repeat(num_of_tkip_iterations) begin
      hsu_mem_layout = new(msg_buffer_min_len, msg_buffer_max_len, 0); // Min/Max
      assert(hsu_mem_layout.randomize());
      hsu_mem_layout.display_mem_layout();
      prepare_message_subbuffers(hsu_mem_layout);

      // Prepare message
      $msg_init(hsu_mem_layout.get_msg_size());
      for(int unsigned i = 0; i < hsu_mem_layout.get_msg_size(); i++) begin
        $msg_append(hsu_mem_layout.get_msg_byte(i[15:0]));
      end

      $tkip(hsu_mem_layout.get_key_word(0),
            hsu_mem_layout.get_key_word(1),
            hsu_mem_layout.get_key_word(2),
            hsu_mem_layout.get_key_word(3),
            hsu_mem_layout.get_msg_size(),
            mic_from_model[31:0],
            mic_from_model[63:32]);

      `uvm_info(get_type_name(), $sformatf("PRECALCULATED MIC for TKIP = 0x%x.", mic_from_model), UVM_HIGH)
      process_message_subbuffers(hsu_mem_layout);
    end // repeat(num_of_tkip_iterations)

    // TKIP - special case (for coverage)
    // Start HSU operation with LENGHT equal to 0
    m_regmodel.set_reg_value(32'h0,         "LENGTH");  // Length is 0
    m_regmodel.set_reg_value(32'h31,       "CONTROL"); // Start
    wait_for_hsu_done();
    hsu_done_clear();
   //################################################

    repeat(num_of_cmac_iterations) begin
      hsu_mem_layout = new(msg_buffer_min_len, msg_buffer_max_len, 1); // Min/Max
      assert(hsu_mem_layout.randomize());
      hsu_mem_layout.display_mem_layout();
      prepare_message_subbuffers(hsu_mem_layout);

      // Prepare message
      $msg_init(hsu_mem_layout.get_msg_size());
      for(int unsigned i = 0; i < hsu_mem_layout.get_msg_size(); i++) begin
        $msg_append(hsu_mem_layout.get_msg_byte(i[15:0]));
      end

      $cmac(hsu_mem_layout.get_key_word(0),
            hsu_mem_layout.get_key_word(1),
            hsu_mem_layout.get_key_word(2),
            hsu_mem_layout.get_key_word(3),
            hsu_mem_layout.get_msg_size(),
            mic_from_model[31:0],
            mic_from_model[63:32]);

      `uvm_info(get_type_name(), $sformatf("PRECALCULATED MIC for CMAC = 0x%x.", mic_from_model), UVM_HIGH)
      process_message_subbuffers(hsu_mem_layout, 1);
    end // repeat(num_of_cmac_iterations)

    `uvm_info(get_type_name(), "HSU test is completed.", UVM_LOW)
  endtask : body

endclass : mac_core_hsu_enc_seq

`endif // MAC_CORE_HSU_ENC_SEQ_SV
