//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_CORE_PTA_BT_SEQ_SV
`define MAC_CORE_PTA_BT_SEQ_SV


class mac_core_pta_bt_seq extends mac_core_seq_base;

  `uvm_object_utils(mac_core_pta_bt_seq)

  PPDU_frame                 data_frame;
  PPDU_frame                 cts_frame;
  PPDU_frame                 ack_frame;

  mac_ctrl_info_1_s          mac_ctrl_info_1;
  mac_ctrl_info_2_s          mac_ctrl_info_2;

  int                        ksr_index;
  int                        frame_num;
  rand nav_prot_frm_e        prot;
  rand expect_ack_e          expect_ack;
  rand access_category_e     ac;

  bit [31:0]                 rx_buff_rd_ptr;
  bit                        filtered_frame;
  bit [3:0]                  pti;
  bit [3:0]                  bt_pti;

  coex_bt_seq_item           bt_item;
  int                        idle_time;
  bit                        basic_priority = 0;
  int                        chan_margin;

  function new (string name = "mac_core_pta_bt_seq");
    super.new (name);
  endfunction : new

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_prot_ack {
    prot == NO_PROT;
    ac inside {AC_BE,AC_BK,AC_VI,AC_VO};
    expect_ack == NO_ACK;
  }
  //---------------------------------------------

  virtual task body();
    uvm_reg_data_t data;
    super.body();

    //--------------------------------------------
    // determine device and BSS address
    //--------------------------------------------
    create_keystorageRAM();

    create_dev_and_bss_addr();

    // prepare SRAM for Rx
    prepare_sram_for_rx(1000, 1);

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);
    m_regmodel.set_field_value(1'b1,"pta_enable","CONFIG");
    m_regmodel.set_field_value(basic_priority,"basic_priority","CONFIG");
    m_regmodel.set_field_value(1'b1,"chan_enable","CONFIG");
    m_regmodel.set_field_value(1'b1,"pti_enable", "CONFIG");
    m_regmodel.set_field_value(1'b1,"no_sim_tx", "CONFIG");
    m_regmodel.set_field_value(1'b1,"no_sim_rx", "CONFIG");
    m_regmodel.set_field_value(1,"wlan_pti_mode","CONFIG");
    m_regmodel.set_field_value(1'b1, "coexEnable", "COEXCONTROLREG"); // Enable coex
    m_regmodel.set_field_value(1'b0, "coexEnable", "COEXCONTROLREG"); // Enable coex
    m_regmodel.set_field_value(1'b1, "coexEnable", "COEXCONTROLREG"); // Enable coex
    m_regmodel.set_field_value(1'b1, "coexAutoPTIAdjEnable", "COEXCONTROLREG");
    m_regmodel.set_field_value(1'b1, "coexPHYTxAbortEnable", "COEXCONTROLREG");
    m_regmodel.set_field_value(1'b1, "coexPHYRxAbortEnable", "COEXCONTROLREG");
    frame_num = $urandom_range(10,20);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Transmit Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      m_regmodel.set_field_value($urandom_range(84,12), "coexWlanChanFreq", "COEXCONTROLREG");   // WLAN Channel center frequency(12-48MHz)
      chan_margin = $urandom_range(63,0);
      m_regmodel.set_field_value(chan_margin, "chan_margin", "CONFIG");

      pti = $urandom_range(15,0);
      m_cfg.m_sram_cfg.pti = pti;
      data_frame = new("data_frame");
      cts_frame  = new("cts_frame");
      ack_frame  = new("ack_frame");

      // randomize fields in sequence
      assert (this.randomize());

      //--------------------------------
      // create data frame for Tx
      //--------------------------------

      assert (data_frame.randomize() with {
        kind == SINGLETON;
        tx_frame == 1;
        ppdu_format inside {NON_HT, HT_MF, VHT, HE_SU, HE_MU};
      });

      data_frame.ampdu_frame[0].clear_all_arrays();
      assert (data_frame.ampdu_frame[0].randomize() with {
        data_frame.ampdu_frame[0].ppdu_format == data_frame.ppdu_format;
        data_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        data_frame.ampdu_frame[0].mpdu_frame_type[0][5:4] == `DATA_MPDU;
        data_frame.ampdu_frame[0].aggregated == 0;
      });

      ksr_index = get_random_ksr_index(data_frame);
      // set MAC addresses
      data_frame.ampdu_frame[0].set_MAC_addr(
        .TA    (dev_addr),
        .RA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
        .BSSID (bss_addr)
      );
      // encrypt frame if protection bit is set
      data_frame.encrypt(.ra(1));
      data_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(),
      $sformatf("Data frame:%s", data_frame.sprint()), UVM_LOW)

      // configure MAC control 1 field for THD
      mac_ctrl_info_1.prot_frm_dur_f   = 16'd0;
      mac_ctrl_info_1.write_ack_f      = 0;
      mac_ctrl_info_1.low_rate_retry_f = 0;
      mac_ctrl_info_1.lstp_prot_f      = 0;
      mac_ctrl_info_1.lstp_f           = 0;
      mac_ctrl_info_1.expect_ack_f     = expect_ack; //normal ACK

      // configure MAC control 2 field for THD
      mac_ctrl_info_2.dont_generate_mh_f = 0;
      mac_ctrl_info_2.dont_encrypt_f     = 0;
      mac_ctrl_info_2.dont_touch_fc_f    = 0;
      mac_ctrl_info_2.dont_touch_dur_f   = 0;
      mac_ctrl_info_2.dont_touch_qos_f   = 0;
      mac_ctrl_info_2.dont_touch_htc_f   = 0;
      mac_ctrl_info_2.dont_touch_tsf_f   = 0;
      mac_ctrl_info_2.dont_touch_dtim_f  = 0;
      mac_ctrl_info_2.dont_touch_fcs_f   = 0;
      mac_ctrl_info_2.under_ba_setup_f   = 0;
      mac_ctrl_info_2.num_blank_delimit_f = 0;
      mac_ctrl_info_2.int_en_f            = 1;

      // determine type of NAV protection frame exchange
      `uvm_info(get_type_name(),$sformatf("NAV protection %s",prot.name()),UVM_LOW)
      // expected ACK
      `uvm_info(get_type_name(),$sformatf("Expected ACK: %s",expect_ack.name()),UVM_LOW)
      // access category
      `uvm_info(get_type_name(),$sformatf("Access category %s",ac.name()),UVM_LOW)

      // Store PPDU frame in SRAM for trasmit
      wr_frame_to_sram(data_frame,
                       mac_ctrl_info_1,
                       mac_ctrl_info_2,
                       prot,
                       32'h80);

      //--------------------------------
      // create CTS frame for Rx
      //--------------------------------
      assert (cts_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      cts_frame.ampdu_frame[0].clear_all_arrays();
      assert (cts_frame.ampdu_frame[0].randomize() with {
        cts_frame.ampdu_frame[0].ppdu_format == cts_frame.ppdu_format;
        cts_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        cts_frame.ampdu_frame[0].mpdu_frame_type[0] == CTS;
        cts_frame.ampdu_frame[0].aggregated == 0;
      });
      cts_frame.calc_leg_ht_length();

      // set MAC addresses
      cts_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      //--------------------------------
      // create ACK frame for Rx
      //--------------------------------
      assert (ack_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      ack_frame.ampdu_frame[0].clear_all_arrays();
      assert (ack_frame.ampdu_frame[0].randomize() with {
        ack_frame.ampdu_frame[0].ppdu_format == ack_frame.ppdu_format;
        ack_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        ack_frame.ampdu_frame[0].mpdu_frame_type[0] == ACK;
        ack_frame.ampdu_frame[0].aggregated == 0;
      });
      ack_frame.calc_leg_ht_length();

      // set MAC addresses
      ack_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      // set Tx interrupt masks for AC
      set_ac_interrupts(ac);

      set_DMA_for_tx(ac);

      fork
        //------------------------------------------------
        // simulate Tx data frame
        //------------------------------------------------
        begin : RST_CTS_DATA_TX
          phy_core_tx();
        end
        //------------------------------------------------
        // simulate BT TX or RX
        //------------------------------------------------
        begin
          bt_item = coex_bt_seq_item::type_id::create("bt_item");
          assert(bt_item.randomize() with {
            bt_item.cmd inside {BT_SET_TX, BT_SET_RX};
            bt_item.sig_value inside {[0:78]};
          })

          set_coex_bt_channel(bt_item.sig_value);
          set_coex_bt_bw($urandom_range(1,0));    // Set BW to 0 - 1MHz
                                                  // Set BW to 1 - 2MHz
          if (bt_item.cmd == BT_SET_TX) begin
            // BT TX
            set_coex_bt_tx(1'b1);               // Set BT TX field to active state
            set_coex_bt_event(1'b1);            // Activate BT event
            bt_pti = $urandom_range(15,0);      // Get a random value for BT PTI
            set_coex_bt_pti(bt_pti);
            idle_time = $urandom_range(600,300);
            insert_idle_cycles(idle_time);      // Insert idle cycles to simulate TX
            set_coex_bt_tx(1'b0);               // Deassert BT TX
            set_coex_bt_event(1'b0);            // Deassert BT event
            idle_time = $urandom_range(600,300);
            insert_idle_cycles(idle_time);
          end
          else if (bt_item.cmd == BT_SET_RX) begin
            // BT_RX
            set_coex_bt_rx(1'b1);               // Set BT RX field to active state
            set_coex_bt_event(1'b1);            // Activate BT event
            bt_pti = $urandom_range(15,0);      // Get a random value for BT PTI
            set_coex_bt_pti(bt_pti);
            idle_time = $urandom_range(600,300);
            insert_idle_cycles(idle_time);      // Insert idle cycles to simulate RX
            set_coex_bt_rx(1'b0);               // Deassert BT RX
            set_coex_bt_event(1'b0);            // Deassert BT event
            idle_time = $urandom_range(600,300);
            insert_idle_cycles(idle_time);
          end

        end
        fork
          wait_mac_tx_trigger(ac);
          begin
          //wait abort_transmission
            wait (m_cfg.m_mac_phy_cfg.m_phy_cfg.aborted_transmission == 1'b1);
            if(m_cfg.m_sram_cfg.pti <= bt_pti)begin
              set_coex_bt_pti(m_cfg.m_sram_cfg.pti - 1);
            end
            m_cfg.m_mac_phy_cfg.m_phy_cfg.aborted_transmission = 1'b0;
            //stop_pooling = 1;
            phy_core_tx();

          end
        join_any
      join
      // Change basic priority
      if ($urandom_range(3,1)) begin
        basic_priority = ~basic_priority;
        m_regmodel.set_field_value(basic_priority,"basic_priority","CONFIG");
      end

      // read THD status from SRAM
      rd_tx_status_from_sram(m_cfg.m_sram_cfg.thd_head_ptr);

      insert_idle_cycles(20);
      // clear interrupt flag
      clear_ac_interrupts(ac);

    end // for loop

  endtask : body


endclass : mac_core_pta_bt_seq

`endif // MAC_CORE_PTA_BT_SEQ_SV
