//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_CORE_RX_CTRL_FRAME_SEQ_SV
`define MAC_CORE_RX_CTRL_FRAME_SEQ_SV


class mac_core_rx_ctrl_frame_seq extends mac_core_seq_base;

  `uvm_object_utils(mac_core_rx_ctrl_frame_seq)

  PPDU_frame                 ctrl_frame;
  bit [31:0]                 rx_buff1_rd_ptr;
  bit [31:0]                 rx_buff2_rd_ptr;
  int                        ksr_index;
  int                        frame_num;
  bit                        filtered_frame = 0;
  bit                        disableRxBuffer2;
  rand bit [47:0]            unicast_addr;

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_unicast_addr {
    unicast_addr[0]     == 0;
    unicast_addr[47:1]  != 0;
    unicast_addr        != dev_addr;
  }

  function new (string name = "mac_core_rx_ctrl_frame_seq");
    super.new (name);

  endfunction : new

  virtual task body();
    super.body();

    //--------------------------------------------
    // determine device and BSS address
    //--------------------------------------------
    create_keystorageRAM();

    create_dev_and_bss_addr();

    // set Rx interrupt masks
    m_regmodel.set_field_value(1'b1, "maskrxBuffer1Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masktimerRxTrigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "maskrxBuffer2Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masterTxRxIntEn", "TXRXINTUNMASKREG"); // Enables interrupt generation

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    frame_num = $urandom_range(10,20);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Receive Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      assert(this.randomize());

      ctrl_frame = new("ctrl_frame");
      // after MAC received 10 frames, once again prepare SRAM for Rx
      // because RHDs are filled
      if ((loop % 10) == 0) begin
        prepare_sram_for_rx(1000);
        prepare_sram_priority_rx(1000);
        rx_buff1_rd_ptr = m_regmodel.get_mirrored_reg_value("RXBUF1RDPTRREG") + rhd_header;
        rx_buff2_rd_ptr = m_regmodel.get_mirrored_reg_value("RXBUF2RDPTRREG") + rhd_header;
      end

      //--------------------------------
      // create data frame for Rx
      //--------------------------------
      assert (ctrl_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      ctrl_frame.ampdu_frame[0].clear_all_arrays();
      assert (ctrl_frame.ampdu_frame[0].randomize() with {
        ctrl_frame.ampdu_frame[0].ppdu_format == ctrl_frame.ppdu_format;
        ctrl_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        ctrl_frame.ampdu_frame[0].mpdu_frame_type[0][5:4] == `CONTROL_MPDU;
        // avoid generating these frames
        !(ctrl_frame.ampdu_frame[0].mpdu_frame_type[0] inside {
          TRIGGER,
          BEAMFORMING_REPORT_POLL,
          VHT_NDP_ANNOUNCEMENT});
        ctrl_frame.ampdu_frame[0].aggregated == 0;
      });

      ksr_index = get_random_ksr_index(ctrl_frame);
      // set MAC addresses
      if (ctrl_frame.ampdu_frame[0].mpdu_frame_type[0] inside {ACK, CTS, BLOCK_ACK, CONTROL_WRAPPER}) begin
        ctrl_frame.ampdu_frame[0].set_MAC_addr(
          .RA    (unicast_addr),
          .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f)
        );
      end
      else begin
        ctrl_frame.ampdu_frame[0].set_MAC_addr(
          .RA    (dev_addr),
          .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f)
        );
      end
      ctrl_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(),
      $sformatf("PPDU_frame:\n%s", ctrl_frame.sprint()), UVM_LOW)

      // receive ACK and wait for interrupt Rx trigger
      fork
        // if frames if going to be received trigger will occur,
        // if frames if filtered event will occur
        fork
          wait_mac_rx_trigger();
          rx_frame_filtered();
        join_any

        begin
          phy_core_rx(ctrl_frame);
          // if there is no response from MAC wait for sync
          if (responde_to_frame(ctrl_frame, dev_addr)) begin
            // PHY sends frame to MAC core
            phy_core_tx();// RX ACK
          end
          else begin
            // PHY sends frame to MAC core
            no_resp_trigger.wait_trigger();
          end
        end
      join
      // only if frame was stored in SRAM do read
      if (!stop_pooling) begin
        `uvm_info(get_type_name(),
        "MAC core received frame and stored it to SRAM", UVM_LOW)

        m_regmodel.get_field_value(rdata, "disableRxBuffer2");
        disableRxBuffer2 = rdata[0];
        // determine in which buffer was frame stored
        // buffer 2 is priority
        // buffer 1 others
        if (is_priority_rx_frame(ctrl_frame) && !disableRxBuffer2) begin
          rd_frame_from_sram(rx_buff2_rd_ptr);
          insert_idle_cycles(5);
          get_next_rhd_ptr(2, rx_buff2_rd_ptr, rx_buff2_rd_ptr);
        end
        else begin
          rd_frame_from_sram(rx_buff1_rd_ptr);
          insert_idle_cycles(5);
          get_next_rhd_ptr(1, rx_buff1_rd_ptr, rx_buff1_rd_ptr);
        end
      end

      // clear interrupt flag
      m_regmodel.set_field_value(1'b1, "clearrxBuffer1Trigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "cleartimerRxTrigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "clearrxBuffer2Trigger", "TXRXINTEVENTCLEARREG");

      // To give a time for interrupt comparison
      insert_idle_cycles(10);

    end //for

  endtask : body


endclass : mac_core_rx_ctrl_frame_seq

`endif // MAC_CORE_RX_CTRL_FRAME_SEQ_SV
