//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_CORE_RX_NDP_SU_CALIB_SEQ_SV
`define MAC_CORE_RX_NDP_SU_CALIB_SEQ_SV


class mac_core_rx_ndp_su_calib_seq extends mac_core_seq_base;

  `uvm_object_utils(mac_core_rx_ndp_su_calib_seq)

  PPDU_frame                 ctrl_frame;
  PPDU_frame                 ndp_frame;
  VHT_NDP_ANNOUNCEMENT_frame ndpa;
  bit [31:0]                 rx_buff_rd_ptr;
  int                        ksr_index;
  int                        frame_num;
  //--------------------------------------------------------------------
  // variables for NDP
  //--------------------------------------------------------------------
  rand bit  [2:0] Nr;                         // Nr Value To Test
  rand bit  [2:0] Nc[`NB_STA_MAX];            // Nc Value To Test
  rand bit  [1:0] Grouping[`NB_STA_MAX];      // Grouping Value To Test
  rand bit        FeedbackType[`NB_STA_MAX];  // Feedback Type Value To Test
  rand bit        Codebook[`NB_STA_MAX];      // Codebook Value To Test

  rand bit [2:0]  num_of_stations;   // number of stations, used for BF Tx test
       bit [15:0] AID;

  //--------------------------------------------------------------------
  // Constraints
  //--------------------------------------------------------------------

`ifdef RW_MUMIMO_RX_EN
    constraint c_Nr {
      Nr inside {1,2};
    }
`else//RW_MUMIMO_RX_EN
    constraint c_Nr {
      Nr == 1;
    }
`endif//RW_MUMIMO_RX_EN

  constraint c_FeedbackType {
      foreach (FeedbackType[i]) FeedbackType[i] == 0;
  }

  constraint c_Nc {
    solve Nr before Nc;
`ifdef RW_TXRX_1X1
    foreach (Nc[i]) Nc[i] == 0;
`else
    foreach (Nc[i]) Nc[i] <= 1;
`endif
  }

  constraint c_Grouping {
    foreach (Grouping[i]) Grouping[i] inside {[0:2]};
  }

  constraint c_num_of_stations {
    num_of_stations == 1;
  }
  //--------------------------------------------------------------------

  function new (string name = "mac_core_rx_ndp_su_calib_seq");
    super.new (name);
  endfunction : new

  function void print_ndp_params(int i=0);
    bit [1:0]  ch_bw;
    ch_bw = ctrl_frame.preamble_header.ch_bw;
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("Reception of VHT NDP Sounding procedure"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nr                : %2d", (Nr+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nc                : %2d", (Nc[i]+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Grouping          : %2d", Grouping[i]      ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Feedback type     : %s" , (FeedbackType[i]==1'b0)? "SU" : "MU"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Codebook          : %2d", Codebook[i]   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Channel BW        : %2d", ch_bw   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
  endfunction: print_ndp_params

  virtual task body();
    super.body();

    //--------------------------------------------
    // determine device and BSS address
    //--------------------------------------------
    create_keystorageRAM();

    create_dev_and_bss_addr();

    // set Rx interrupt masks
    m_regmodel.set_field_value(1'b1, "maskrxBuffer1Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masktimerRxTrigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "maskrxBuffer2Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masterTxRxIntEn", "TXRXINTUNMASKREG"); // Enables interrupt generation
    // configure Beamformee register
    m_regmodel.set_field_value(3'b100, "bfrFormatMod", "BFMEECONTROLREG"); //VHT
    m_regmodel.set_field_value(1'h1, "bfmeeEnable", "BFMEECONTROLREG");
    // set AID value for this receiving station
    AID = $urandom();
    m_regmodel.set_field_value(AID, "aid", "BCNCNTRL2REG");

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    // generate BFM report when needed
    phy_bfm_report_gen();

    frame_num = $urandom_range(10,20);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Receive Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      ctrl_frame = new("ctrl_frame");
      ndp_frame  = new("ndp_frame");
      ndpa       = new("ndpa");

      // randomize fields in sequence
      assert (this.randomize());

      // overwrite register BFMEE with randomized parameters
      m_regmodel.set_field_value(FeedbackType[0], "bfmeeMUSupport", "BFMEECONTROLREG");
      m_regmodel.set_field_value(Codebook[0], "bfmeeCodebook", "BFMEECONTROLREG");
      m_regmodel.set_field_value(Grouping[0], "bfmeeGrouping", "BFMEECONTROLREG");
      m_regmodel.set_field_value(Nr, "bfmeeNr", "BFMEECONTROLREG");
      m_regmodel.set_field_value(Nc[0], "bfmeeNc", "BFMEECONTROLREG");

      // after MAC received 10 frames, once again prepare SRAM for Rx
      // because RHDs are filled
      if ((loop % 5) == 0) begin
        prepare_sram_for_rx(10000);
        m_regmodel.get_reg_value(rdata, "RXBUF1RDPTRREG");
        rx_buff_rd_ptr = rdata + rhd_header;
      end

      //--------------------------------
      // create NDPA frame for Rx
      //--------------------------------
      // create NDPA MPDU
      assert (ndpa.randomize() with {
        sta_info.size() == num_of_stations;
        sta_info[0].nc_index_f == Nc[0];
        sta_info[0].feedback_type_f == FeedbackType[0];
        sta_info[0].aid12_f == AID[11:0]; // 12 LSB bits
      });

      assert (ctrl_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == VHT;
        tx_frame == 0;
      });

      ctrl_frame.ampdu_frame[0].clear_all_arrays();
      assert (ctrl_frame.ampdu_frame[0].randomize() with {
        ctrl_frame.ampdu_frame[0].ppdu_format == ctrl_frame.ppdu_format;
        ctrl_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        ctrl_frame.ampdu_frame[0].mpdu_frame_type[0] == VHT_NDP_ANNOUNCEMENT;
        ctrl_frame.ampdu_frame[0].aggregated == 0;
      });
      ctrl_frame.ampdu_frame[0].mpdu_frame[0] = ndpa;
      ctrl_frame.ampdu_frame[0].create_delimiter();
      ctrl_frame.ampdu_frame[0].add_padding();

      ksr_index = get_random_ksr_index(ctrl_frame);
      // set MAC addresses
      ctrl_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr),
        .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
        .BSSID (bss_addr)
      );
      ctrl_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(),
      $sformatf("NDPA:\n%s", ctrl_frame.sprint()), UVM_LOW)
      print_ndp_params();

      //--------------------------------
      // create NDP frame for Rx
      //--------------------------------
      assert (ndp_frame.randomize() with {
        kind == NDP;
        ppdu_format == VHT;
        tx_frame == 0;
      });
      // channel bandwidth needs to be same as in NDPA
      ndp_frame.preamble_header.ch_bw = ctrl_frame.preamble_header.ch_bw;

      `uvm_info(get_type_name(),
      $sformatf("NDP:\n%s", ndp_frame.sprint()), UVM_LOW)

      // receive ACK and wait for interrupt Rx trigger
      fork
        // if frames if going to be received trigger will occur,
        // if frames if filtered event will occur
        fork
          wait_mac_rx_trigger();
          rx_frame_filtered();
        join_any

        begin
          // PHY sends frame to MAC core
          phy_core_rx(ctrl_frame);
          insert_idle_cycles(16);
          phy_core_rx(ndp_frame);
          phy_core_tx(); // RX Beamforming report
        end
      join

      // only if frame was stored in SRAM do read
      if (!stop_pooling) begin
        `uvm_info(get_type_name(),
        "MAC core received frame and stored it to SRAM", UVM_LOW)
        rd_frame_from_sram(rx_buff_rd_ptr);
        // get next RHD pointer value if frame was not filtered
        get_next_rhd_ptr(1, rx_buff_rd_ptr, rx_buff_rd_ptr);
        rd_frame_from_sram(rx_buff_rd_ptr);
        // get next RHD pointer value if frame was not filtered
        get_next_rhd_ptr(1, rx_buff_rd_ptr, rx_buff_rd_ptr);
      end

      insert_idle_cycles(20);

      // clear interrupt flag
      m_regmodel.set_field_value(1'b1, "clearrxBuffer1Trigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "cleartimerRxTrigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "clearrxBuffer2Trigger", "TXRXINTEVENTCLEARREG");

      // To give a time for interrupt comparison
      insert_idle_cycles(30);

    end //for

  endtask : body


endclass : mac_core_rx_ndp_su_calib_seq

`endif // MAC_CORE_RX_NDP_SU_CALIB_SEQ_SV
