//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_CORE_TX_MUMIMO_FRAME_SEQ_SV
`define MAC_CORE_TX_MUMIMO_FRAME_SEQ_SV


class mac_core_tx_mumimo_frame_seq extends mac_core_seq_base;

  `uvm_object_utils(mac_core_tx_mumimo_frame_seq)

  PPDU_frame                 data_frame;
  PPDU_frame                 cts_frame;
  PPDU_frame                 ack_frame;
  PPDU_frame                 ba_frame;
  BLOCK_ACK_frame            ba_mpdu;

  mac_ctrl_info_1_s          mac_ctrl_info_1;
  mac_ctrl_info_2_s          mac_ctrl_info_2;

  int                        ksr_index;
  int                        frame_num;
  bit [31:0]                 user1_addr;
  rand nav_prot_frm_e        prot;
  rand expect_ack_e          expect_ack;
  rand access_category_e     ac;

  function new (string name = "mac_core_tx_mumimo_frame_seq");
    super.new (name);
  endfunction : new

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_prot_ack {
    prot inside {NO_PROT, SELF_CTS};
    ac   inside {AC_VI, AC_VO};
    expect_ack == NO_ACK;
  }
  //---------------------------------------------

  virtual task body();
    super.body();

    //--------------------------------------------
    // determine device and BSS address
    //--------------------------------------------
    create_keystorageRAM(1,3,20);

    // prevent WAPI encryption type
    foreach(m_ksr.keyRAM[i]) begin
      if (m_ksr.keyRAM[i].ctype_ram_f == WAPI) begin
        m_ksr.keyRAM[i].ctype_ram_f = CCMP;
        m_ksr.keyRAM[i].clen_ram_f = 2'b00;
        m_ksr.keyRAM[i].use_def_key_ram_f = 0;
      end
    end

    // overwrite keyStorageRAM with new cosntraints
    overwrite_keystorageRAM();

    create_dev_and_bss_addr();

    // prepare SRAM for Rx, with smaller Rx segment
    prepare_sram_for_rx(1000, 1);
    prepare_sram_priority_rx(1000);
    // don't accept control frames
    m_regmodel.set_field_value(1'b0, "acceptBA",  "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptNotExpectedBA", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptBAR", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptRTS", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptCTS", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptACK", "RXCNTRLREG");

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    frame_num = $urandom_range(5,15);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Transmit Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      data_frame = new("data_frame");
      cts_frame  = new("cts_frame");
      ack_frame  = new("ack_frame");
      ba_frame   = new("ba_frame");
      ba_mpdu    = new("ba_mpdu");

      // randomize fields in sequence
      assert (this.randomize());

      //--------------------------------
      // create data frame for Tx
      //--------------------------------

      assert (data_frame.randomize() with {
        kind == MU_MIMO;
        num_of_users == 2;
        tx_frame == 1;
      });

      foreach (data_frame.ampdu_frame[i]) begin
        ksr_index = get_random_ksr_index(data_frame, i);
        // set MAC addresses
        data_frame.ampdu_frame[i].set_MAC_addr(
          .TA    (dev_addr),
          .RA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
          .BSSID (bss_addr)
        );
      end//foreach

      foreach (data_frame.preamble_header.user_header[i]) begin
        if (data_frame.preamble_header.ch_bw == 0 && data_frame.preamble_header.user_header[i].mcs_f == 7'd9) begin
          data_frame.preamble_header.user_header[i].mcs_f = $urandom_range(0,8);
        end
      end

      // encrypt frame if protection bit is set
      data_frame.encrypt(.ra(1));
      data_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(), $sformatf("Data frame:%s", data_frame.sprint()), UVM_LOW)

      // configure MAC control 1 field for THD
      mac_ctrl_info_1.prot_frm_dur_f   = 16'd0;
      mac_ctrl_info_1.write_ack_f      = 0;
      mac_ctrl_info_1.low_rate_retry_f = 0;
      mac_ctrl_info_1.lstp_prot_f      = 0;
      mac_ctrl_info_1.lstp_f           = 0;
      mac_ctrl_info_1.expect_ack_f     = expect_ack; //normal ACK

      // configure MAC control 2 field for THD
      mac_ctrl_info_2.dont_generate_mh_f = 0;
      mac_ctrl_info_2.dont_encrypt_f     = 0;
      mac_ctrl_info_2.dont_touch_fc_f    = 0;
      mac_ctrl_info_2.dont_touch_dur_f   = 0;
      mac_ctrl_info_2.dont_touch_qos_f   = 0;
      mac_ctrl_info_2.dont_touch_htc_f   = 0;
      mac_ctrl_info_2.dont_touch_tsf_f   = 0;
      mac_ctrl_info_2.dont_touch_dtim_f  = 0;
      mac_ctrl_info_2.dont_touch_fcs_f   = 0;
      mac_ctrl_info_2.under_ba_setup_f   = 0;
      mac_ctrl_info_2.num_blank_delimit_f = 0;
      mac_ctrl_info_2.int_en_f            = 1;

      // determine type of NAV protection frame exchange
      `uvm_info(get_type_name(),$sformatf("NAV protection %s",prot.name()),UVM_LOW)
      // expected ACK
      `uvm_info(get_type_name(),$sformatf("Expected ACK: %s",expect_ack.name()),UVM_LOW)
      // access category
      `uvm_info(get_type_name(),$sformatf("Access category %s",ac.name()),UVM_LOW)

      // Store PPDU frame in SRAM for trasmit
      wr_frame_to_sram(data_frame,
                       mac_ctrl_info_1,
                       mac_ctrl_info_2,
                       prot,
                       32'h80);

      //--------------------------------
      // create CTS frame for Rx
      //--------------------------------
      assert (cts_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      cts_frame.ampdu_frame[0].clear_all_arrays();
      assert (cts_frame.ampdu_frame[0].randomize() with {
        cts_frame.ampdu_frame[0].ppdu_format == cts_frame.ppdu_format;
        cts_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        cts_frame.ampdu_frame[0].mpdu_frame_type[0] == CTS;
        cts_frame.ampdu_frame[0].aggregated == 0;
      });
      cts_frame.calc_leg_ht_length();

      // set MAC addresses
      cts_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      //--------------------------------
      // create ACK frame for Rx
      //--------------------------------
      assert (ack_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      ack_frame.ampdu_frame[0].clear_all_arrays();
      assert (ack_frame.ampdu_frame[0].randomize() with {
        ack_frame.ampdu_frame[0].ppdu_format == ack_frame.ppdu_format;
        ack_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        ack_frame.ampdu_frame[0].mpdu_frame_type[0] == ACK;
        ack_frame.ampdu_frame[0].aggregated == 0;
      });
      ack_frame.calc_leg_ht_length();

      // set MAC addresses
      ack_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      //--------------------------------
      // create BLOCK_ACK frame for Rx
      //--------------------------------
      assert (ba_mpdu.randomize() with {
        ba_variant == COMP_BA;
        starting_seq_num == data_frame.ampdu_frame[0].get_MAC_seq_num();
      });
      assert (ba_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      ba_frame.ampdu_frame[0].clear_all_arrays();
      assert (ba_frame.ampdu_frame[0].randomize() with {
        ba_frame.ampdu_frame[0].ppdu_format == ba_frame.ppdu_format;
        ba_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        ba_frame.ampdu_frame[0].mpdu_frame_type[0] == BLOCK_ACK;
        ba_frame.ampdu_frame[0].aggregated == 0;
      });
      ba_frame.ampdu_frame[0].mpdu_frame[0] = ba_mpdu;
      ba_frame.calc_leg_ht_length();

      // set MAC addresses
      ba_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      // set Tx interrupt masks for AC
      set_ac_interrupts(ac);

      set_DMA_for_tx(ac);

      fork
        begin : RST_CTS_DATA_TX
          //------------------------------------------------
          // collect Tx RTS/CTS protection frames
          //------------------------------------------------
          if (prot != NO_PROT) begin
            phy_core_tx();
          end

          //------------------------------------------------
          // send CTS
          //------------------------------------------------
          if (prot inside {RTS_CTS, RTS_CTS_QAP}) begin
            create_ppdu_response_frame(cts_frame);
            phy_core_rx(cts_frame);
          end

          //------------------------------------------------
          // collect Tx data frame
          //------------------------------------------------
          phy_core_tx();

          //------------------------------------------------
          // send ACK or BLOACK ACK or receive BLOCK ACK REQ
          //------------------------------------------------
          if (expect_ack == NORMAL_ACK) begin
            create_ppdu_response_frame(ack_frame);
            phy_core_rx(ack_frame);
          end
          else if (expect_ack == COMPRESSED_BA) begin
            create_ppdu_response_frame(ba_frame);
            phy_core_rx(ba_frame);
          end
          else if (expect_ack == NO_ACK) begin
            phy_core_tx(); // collect BLOCK ACK REQ primary user
            insert_idle_cycles(1);
            phy_core_tx(); // collect BLOCK ACK REQ secondary user
          end
        end//RST_CTS_DATA_TX

        wait_mac_tx_trigger(ac);
      join

      // read THD status from SRAM for primary user
      rd_tx_status_from_sram(
        .head     (m_cfg.m_sram_cfg.thd_head_ptr),
        .skip_athd((expect_ack == NO_ACK) ? 0 : 1) // read A-THD if BAR is sent
      );
      // read THD status from SRAM for secondary user
      read_word_sram(
        m_cfg.m_sram_cfg.thd_head_ptr+`USER1_AMPDU_THD_PTR, user1_addr);

      rd_tx_status_from_sram(
        .head     (user1_addr),
        .skip_athd((expect_ack == NO_ACK) ? 0 : 1) // read A-THD if BAR is sent
      );

      insert_idle_cycles(20);
      // clear interrupt flag
      clear_ac_interrupts(ac);

    end // for loop

  endtask : body


endclass : mac_core_tx_mumimo_frame_seq

`endif // MAC_CORE_TX_MUMIMO_FRAME_SEQ_SV
