//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_CORE_TX_NDP_SEQ_SV
`define MAC_CORE_TX_NDP_SEQ_SV


class mac_core_tx_ndp_seq extends mac_core_seq_base;

  `uvm_object_utils(mac_core_tx_ndp_seq)

  PPDU_frame                 data_frame;
  PPDU_frame                 cts_frame;
  PPDU_frame                 bfr_frame;
  PPDU_frame                 ndp_frame;
  PPDU_frame                 ndpa_frame;

  mac_ctrl_info_1_s          mac_ctrl_info_1;
  mac_ctrl_info_2_s          mac_ctrl_info_2;

  int                        ksr_index;
  int                        frame_num;
  bit [31:0]                 thd_addr_q[$]; // save THD address
  rand nav_prot_frm_e        prot;
  rand expect_ack_e          expect_ack;
  rand access_category_e     ac;

  function new (string name = "mac_core_tx_ndp_seq");
    super.new (name);
  endfunction : new

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_prot_ack {
    prot == NO_PROT;
    ac inside {AC_BE,AC_BK,AC_VI,AC_VO};
    expect_ack == NORMAL_ACK;
  }
  //---------------------------------------------

  virtual task body();
    super.body();

    //--------------------------------------------
    // determine device and BSS address
    //--------------------------------------------
    create_keystorageRAM();

    create_dev_and_bss_addr();

    // prepare SRAM for Rx, with smaller Rx segment
    prepare_sram_for_rx(1000, 1);
    prepare_sram_priority_rx(1000);

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    frame_num = $urandom_range(10,20);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Transmit Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      data_frame = new("data_frame");
      cts_frame  = new("cts_frame");
      bfr_frame  = new("bfr_frame");
      ndp_frame  = new("ndp_frame");
      ndpa_frame = new("ndpa_frame");

      // randomize fields in sequence
      assert (this.randomize());

      //--------------------------------
      // create NDPA frame for Rx
      //--------------------------------
      assert (ndpa_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == VHT;
        tx_frame == 0;
      });

      ndpa_frame.ampdu_frame[0].clear_all_arrays();
      assert (ndpa_frame.ampdu_frame[0].randomize() with {
        ndpa_frame.ampdu_frame[0].ppdu_format == ndpa_frame.ppdu_format;
        ndpa_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        ndpa_frame.ampdu_frame[0].mpdu_frame_type[0] == VHT_NDP_ANNOUNCEMENT;
        ndpa_frame.ampdu_frame[0].aggregated == 0;
      });

      ksr_index = get_random_ksr_index(ndpa_frame);
      // set MAC addresses
      ndpa_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
        .TA    (dev_addr),
        .BSSID (bss_addr)
      );
      ndpa_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(),
      $sformatf("NDPA:\n%s", ndpa_frame.sprint()), UVM_LOW)

      // configure MAC control 1 field for THD
      mac_ctrl_info_1.prot_frm_dur_f   = $urandom_range(500, 1000);
      mac_ctrl_info_1.write_ack_f      = 0;
      mac_ctrl_info_1.low_rate_retry_f = 0;
      mac_ctrl_info_1.lstp_prot_f      = 0;
      mac_ctrl_info_1.lstp_f           = 0;
      mac_ctrl_info_1.expect_ack_f     = bit'(NO_ACK);

      // configure MAC control 2 field for THD
      mac_ctrl_info_2.dont_generate_mh_f = 0;
      mac_ctrl_info_2.dont_encrypt_f     = 0;
      mac_ctrl_info_2.dont_touch_fc_f    = 0;
      mac_ctrl_info_2.dont_touch_dur_f   = 1;
      mac_ctrl_info_2.dont_touch_qos_f   = 0;
      mac_ctrl_info_2.dont_touch_htc_f   = 0;
      mac_ctrl_info_2.dont_touch_tsf_f   = 0;
      mac_ctrl_info_2.dont_touch_dtim_f  = 0;
      mac_ctrl_info_2.dont_touch_fcs_f   = 0;
      mac_ctrl_info_2.under_ba_setup_f   = 0;
      mac_ctrl_info_2.num_blank_delimit_f = 0;
      mac_ctrl_info_2.int_en_f            = 1;

      m_cfg.m_sram_cfg.use_cfg_pt_ptr = 0;

      // Store PPDU frame in SRAM for trasmit
      wr_frame_to_sram(ndpa_frame,
                       mac_ctrl_info_1,
                       mac_ctrl_info_2,
                       prot,
                       32'h80);

      // save THD address for future read of status information
      thd_addr_q.push_front(m_cfg.m_sram_cfg.thd_head_ptr);

      // set Tx interrupt masks for AC
      set_ac_interrupts(ac);
      set_DMA_for_tx(ac);

      //--------------------------------
      // create NDP frame for Rx
      //--------------------------------
      assert (ndp_frame.randomize() with {
        kind == NDP;
        tx_frame == 1;
      });
      // channel bandwidth needs to be same as in NDPA
      ndp_frame.preamble_header.ch_bw = ndpa_frame.preamble_header.ch_bw;

      `uvm_info(get_type_name(),
      $sformatf("NDP:\n%s", ndp_frame.sprint()), UVM_LOW)

      // configure MAC control 1 field for THD
      mac_ctrl_info_1.prot_frm_dur_f   = 16'd0;
      mac_ctrl_info_1.write_ack_f      = 0;
      mac_ctrl_info_1.low_rate_retry_f = 0;
      mac_ctrl_info_1.lstp_prot_f      = 0;
      mac_ctrl_info_1.lstp_f           = 0;
      mac_ctrl_info_1.expect_ack_f     = bit'(NORMAL_ACK);

      // configure MAC control 2 field for THD
      mac_ctrl_info_2.dont_generate_mh_f = 0;
      mac_ctrl_info_2.dont_encrypt_f     = 0;
      mac_ctrl_info_2.dont_touch_fc_f    = 0;
      mac_ctrl_info_2.dont_touch_dur_f   = 0;
      mac_ctrl_info_2.dont_touch_qos_f   = 0;
      mac_ctrl_info_2.dont_touch_htc_f   = 0;
      mac_ctrl_info_2.dont_touch_tsf_f   = 0;
      mac_ctrl_info_2.dont_touch_dtim_f  = 0;
      mac_ctrl_info_2.dont_touch_fcs_f   = 0;
      mac_ctrl_info_2.under_ba_setup_f   = 0;
      mac_ctrl_info_2.num_blank_delimit_f = 0;
      mac_ctrl_info_2.int_en_f            = 1;

      // link NDPA and NDP
      write_word_sram(
        m_cfg.m_sram_cfg.thd_head_ptr+`NEXT_ATOMIC_FRAME_OFFSET,
        m_cfg.m_sram_cfg.thd_curr_ptr
      );

      m_cfg.m_sram_cfg.use_cfg_pt_ptr = 1;

      // Store PPDU frame in SRAM for trasmit
      wr_frame_to_sram(ndp_frame,
                       mac_ctrl_info_1,
                       mac_ctrl_info_2,
                       prot,
                       m_cfg.m_sram_cfg.thd_curr_ptr);

      // save THD address for future read of status information
      thd_addr_q.push_front(m_cfg.m_sram_cfg.thd_head_ptr);

      //--------------------------------
      // create BF report frame for Rx
      //--------------------------------
      assert (bfr_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == VHT;
        tx_frame == 0;
      });

      bfr_frame.ampdu_frame[0].clear_all_arrays();
      assert (bfr_frame.ampdu_frame[0].randomize() with {
        bfr_frame.ampdu_frame[0].ppdu_format == bfr_frame.ppdu_format;
        bfr_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        bfr_frame.ampdu_frame[0].mpdu_frame_type[0] == BEAMFORMING_REPORT;
        bfr_frame.ampdu_frame[0].aggregated == 0;
      });
      // encrypt BFR
      bfr_frame.encrypt(.ra(1));
      bfr_frame.calc_leg_ht_length();

      // set MAC addresses
      bfr_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      `uvm_info(get_type_name(),
      $sformatf("Beamformig report for MAC:%s", bfr_frame.sprint()), UVM_LOW)


      fork
        begin : RST_CTS_DATA_TX
          //------------------------------------------------
          // collect Tx RTS/CTS protection frames
          //------------------------------------------------
          if (prot != NO_PROT) begin
            phy_core_tx();
          end

          //------------------------------------------------
          // send CTS
          //------------------------------------------------
          if (prot inside {RTS_CTS, RTS_CTS_QAP}) begin
            create_ppdu_response_frame(cts_frame);
            phy_core_rx(cts_frame);
          end

          //------------------------------------------------
          // collect NDPA frame
          //------------------------------------------------
          phy_core_tx();

          //------------------------------------------------
          // collect NDP frame
          //------------------------------------------------
          phy_core_tx();

          //------------------------------------------------
          // send Beamforming report
          //------------------------------------------------
          if (expect_ack == NORMAL_ACK) begin
            create_ppdu_response_frame(bfr_frame);
            phy_core_rx(bfr_frame);
          end
        end

        wait_mac_tx_trigger(ac);
      join

      insert_idle_cycles(20);
      // read THD status from SRAM
      while (thd_addr_q.size() != 0)
        rd_tx_status_from_sram(thd_addr_q.pop_back());

      // clear interrupt flag
      clear_ac_interrupts(ac);

    end // for loop

  endtask : body


endclass : mac_core_tx_ndp_seq

`endif // MAC_CORE_TX_NDP_SEQ_SV
