//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MODEM_NDP_RX_SEQ_SV
`define MODEM_NDP_RX_SEQ_SV


class modem_ndp_rx_seq extends modem_seq_base;

  `uvm_object_utils(modem_ndp_rx_seq)

  //--------------------------------------------------------------------
  // variables for NDP
  //--------------------------------------------------------------------
  rand bit  [2:0] Nr;                         // Nr Value To Test
  rand bit  [2:0] Nc[`NB_STA_MAX];            // Nc Value To Test
  rand bit  [1:0] Grouping[`NB_STA_MAX];      // Grouping Value To Test
  rand bit        FeedbackType[`NB_STA_MAX];  // Feedback Type Value To Test
  rand bit        Codebook[`NB_STA_MAX];      // Codebook Value To Test
  rand bit [2:0]  num_of_stations;            // number of stations, used for BF Tx test
  rand bit [1:0]  ch_bw;
  rand bit [5:0]  ruindex_start;              // RU start index
  rand bit [5:0]  ruindex_end;                // RU end index
  format_mod_e    custom_format;              // format mode of frame, set from test case
  bit             is_he;                      // indication that HE frame is tested
  bit             insert_non_ht;              // insert NON-HT frame after NDP

  //--------------------------------------------------------------------
  // Constraints
  //--------------------------------------------------------------------

`ifdef RW_MUMIMO_RX_EN
    constraint c_Nr {
      Nr inside {1, 3};
    }
    constraint c_FeedbackType {
      foreach (FeedbackType[i]) FeedbackType[i] inside {0,1};
    }
`else//RW_MUMIMO_RX_EN
    constraint c_Nr {
      Nr inside {1, 3};
    }
    constraint c_FeedbackType {
      foreach (FeedbackType[i]) FeedbackType[i] == 0;
    }
`endif//RW_MUMIMO_RX_EN

  constraint c_Nc {
    solve Nr before Nc;
`ifdef RW_TXRX_1X1
    foreach (Nc[i]) Nc[i] == 0;
`else
    foreach (Nc[i]) Nc[i] <= 1;
`endif
  }

  constraint c_Grouping {
    foreach (Grouping[i]) {
      if (custom_format == VHT)
        Grouping[i] inside {[0:2]};
      else//For HE grouping is 1bit wide
        Grouping[i] inside {[0:1]};
    }
  }

  constraint c_channel_bw {
    ch_bw == channel_bw_func(custom_format);
  }

  constraint c_num_of_stations {
    num_of_stations == 1;
  }

  constraint c_ru_index {
    solve ch_bw before ruindex_start;
    solve ch_bw before ruindex_end;

    if (ch_bw == 2'b00) {
      ruindex_start  inside {[0:8]}; // RU start index
      ruindex_end    inside {[0:8]}; // RU end index
    } else if (ch_bw == 2'b01) {
      ruindex_start inside {[0:17]}; // RU start index
      ruindex_end   inside {[0:17]}; // RU end index
    } else {
      ruindex_start inside {[0:36]}; // RU start index
      ruindex_end   inside {[0:36]}; // RU end index
    }
    // RU start index must be less or equal to RU end index
    ruindex_start <= ruindex_end;
    // don't randomize if non HE
    if (custom_format < HE_SU) {
      ruindex_start == 0;
      ruindex_end == 0;
    } else if (custom_format == HE_SU) {
      if (ch_bw == 2'b00) {
        ruindex_start == 0;
        ruindex_end == 8;
      } else if (ch_bw == 2'b01) {
        ruindex_start == 0;
        ruindex_end == 17;
      } else {
        ruindex_start == 0;
        ruindex_end == 36;
      }
    }

    foreach(Grouping[i]) {
      if (4**(Grouping[i] + 1) == 16)
        Codebook[i] == 1;
    }

  }

  function new (string name = "modem_ndp_rx_seq");
    super.new (name);
  endfunction : new

  function void print_ndp_params(int i=0);
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("Reception of %s NDP Sounding procedure", (is_he)?"HE-SU":"VHT"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nr                : %2d", (Nr+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nc                : %2d", (Nc[i]+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Grouping          : %2d", Grouping[i]      ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Feedback type     : %s" , (FeedbackType[i]==1'b0)? "SU" : "MU"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Codebook          : %2d", Codebook[i]   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Channel BW        : %2d", ch_bw   ), UVM_LOW)
   if (is_he) begin
     `uvm_info(get_type_name(), $sformatf("   RUStartIndex      : %2d", ruindex_start   ), UVM_LOW)
     `uvm_info(get_type_name(), $sformatf("   RUEndIndex        : %2d", ruindex_end   ), UVM_LOW)
   end
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
  endfunction: print_ndp_params

  function void set_ndp_params();
    m_mdm_cfg.Nr = Nr;
    m_mdm_cfg.nBeamformingSTA = num_of_stations;
    for (int i=0; i<num_of_stations; i++) begin
      m_mdm_cfg.Nc[i]           = Nc[i];
      m_mdm_cfg.Grouping[i]     = Grouping[i];
      m_mdm_cfg.Feedback[i]     = FeedbackType[i];
      m_mdm_cfg.Codebook[i]     = Codebook[i];
      m_mdm_cfg.RUStartIndex[i] = ruindex_start;
      m_mdm_cfg.RUEndIndex[i]   = ruindex_end;
    end
  endfunction : set_ndp_params


  virtual task body();
    super.body();

    agcBypass = m_cfg.m_radio_cfg.m_rui_cfg.agcBypass; // set from testcase
    is_he = (custom_format >= HE_SU) ? 1'b1 : 1'b0;    // set HE frame flag

    // fork off gain update task
    fork  : GAIN_UPDATE
      begin  // thread 1
        if ((`AGC_ON == 1) && (agcBypass == 0)) refresh_RF_data(); // call AGC update task only if necessary
      end
    join_none

    // start generating noise on ADC inputs
    drive_noise_adc();

    // set number of frames MAC driver should receive,
    // NDPA + NDP
    if (insert_non_ht == 1)
      num_rx_frames = 3;
    else
      num_rx_frames = 2;
    
    frame_num = $urandom_range(5, 10);

    for (int loop_num=1; loop_num <= frame_num; loop_num++) begin
      insert_idle_cycles(1);

      `uvm_info(get_type_name(), $sformatf("Receiving frame number %0d of %0d",loop_num, frame_num ), UVM_LOW)

      if (insert_non_ht == 1) begin
        // create NDPA PPDU frame
        frame = PPDU_frame::type_id::create("frame");
        assert (frame.randomize() with {
          kind == SINGLETON;
          ppdu_format == HE_SU;
          tx_frame == 0;
        });
        frame.preamble_header.ch_bw = ch_bw;
        if (ch_bw >= 2'b01)
          frame.preamble_header.user_header_he[0].fec_coding_f = 1;

        drive_phy_adc(frame);
        // Set the PRIMARYIND register
        m_regmodel.set_reg_value(frame.preamble_header.primary_channel,"PRIMARYIND","PHYCONFIG");
      end

      // randomize NDP parameters and set them in mdm_data_model class
      assert (this.randomize());
      print_ndp_params();
      set_ndp_params();

      // configure beamforming block
      configure_bfmee_ctrl(Nc[0], Nr, Grouping[0], FeedbackType[0], Codebook[0], ch_bw, is_he, ruindex_start, ruindex_end);

      // create NDP PPDU frame
      frame = PPDU_frame::type_id::create("frame");
      assert (frame.randomize() with {
        kind == NDP;
        ppdu_format == custom_format;
        tx_frame == 0;
      });
      frame.preamble_header.ch_bw = ch_bw;
      // constraint FEC to LDPC in CHBW40 and CHBW80 for HE frames
      if (custom_format == HE_SU) begin
        if (ch_bw >= 2'b01)
          frame.preamble_header.user_header_he[0].fec_coding_f = 1;
        // set NSTS same as Nr
        frame.preamble_header.user_header_he[0].nss_f = Nr;
      end
      else if (custom_format == VHT) begin
        // set NSTS same as Nr
        frame.preamble_header.user_header[0].num_sts_f = Nr;
      end
      `uvm_info(get_type_name(), $sformatf("Frame configuration :\n%s", frame.sprint()), UVM_LOW)

      // Set the PRIMARYIND register
      m_regmodel.set_reg_value(frame.preamble_header.primary_channel,"PRIMARYIND","PHYCONFIG");

      drive_phy_adc(frame);

      fork
        // START the Beamforming process
        begin
          // start beamforming proces
          bfmee_start();

          // read beamforming report from bfmee memory
          read_bfm_report(Nc[0], Nr, Grouping[0], FeedbackType[0], Codebook[0], ch_bw, is_he, ruindex_start, ruindex_end);

          if (!insert_non_ht) begin
            insert_idle_cycles(16);
            // create NDP PPDU frame
            frame = PPDU_frame::type_id::create("frame");
            assert (frame.randomize() with {
              kind == SINGLETON;
              ppdu_format == custom_format;
              tx_frame == 0;
            });

            frame.preamble_header.ch_bw = ch_bw;
            // constraint FEC to LDPC in CHBW40 and CHBW80 for HE frames
            if (custom_format == HE_SU && ch_bw >= 2'b01) begin
              frame.preamble_header.user_header_he[0].fec_coding_f = 1;
            end
            else if (custom_format == VHT && ch_bw == 2'b00) begin
              // prevent unsupported MCS
              if (frame.preamble_header.user_header[0].mcs_f == 'd9)
                frame.preamble_header.user_header[0].mcs_f = 'd8;
            end
            frame.calc_leg_ht_length();
            `uvm_info(get_type_name(), $sformatf("Frame configuration :\n%s", frame.sprint()), UVM_LOW)

            drive_phy_adc(frame);
          end
          
        end

        // Receive a NON-HT frame
        begin
          if (insert_non_ht == 1) begin
            //insert_idle_cycles(16);

            //reception of a NON-HT frame
            frame = PPDU_frame::type_id::create("frame");
            assert (frame.randomize() with {
              ppdu_format == NON_HT;
              tx_frame == 0;
            });

            drive_phy_adc(frame);
          end
        end
      join
     

      if (insert_non_ht == 1) begin
        // if MAC driver hasn't finished unlock it
        wait (mac_seq_completed == 1'b1);
        insert_idle_cycles(16);
        // BFR Report TX
        frame = PPDU_frame::type_id::create("frame");
        assert (frame.randomize() with {
          ppdu_format == HE_SU;
          tx_frame == 1;
        });
        frame.calc_leg_ht_length();
        tx_ppdu_frame(frame);
      end

      insert_idle_cycles(30);

    end // for loop_num
  endtask : body

endclass : modem_ndp_rx_seq


`endif //MODEM_NDP_RX_SEQ_SV
