//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MODEM_SEQ_BASE_SV
`define MODEM_SEQ_BASE_SV


class modem_seq_base extends wlan_seq_base;

  `uvm_object_utils(modem_seq_base)

  //--------------------------------------------------------------------
  // sequences needed for running test
  //--------------------------------------------------------------------
  mac_rx_frame_seq              mac_rx_seq;
  mac_tx_frame_seq              mac_tx_seq;

  //--------------------------------------------------------------------
  // Misc. variables
  //--------------------------------------------------------------------
  int             frame_num;          // Number of frames to receive
  int             num_rx_frames;      // Number of frames MAC driver should
                                      // receive before de-asserting rxReq
  bit             mac_seq_completed;  // Indicates mac_rx_seq terminated properly
  //--------------------------------------------------------------------

  function new (string name = "modem_seq_base");
    super.new (name);

    mac_seq_completed = 1'b1;
    num_rx_frames = 1;
  endfunction : new

  virtual task pre_body();
  endtask : pre_body

  virtual task body();
    super.body();
  endtask : body

  virtual task post_body();
  endtask : post_body

  //---------------------------------------------
  // common functions and tasks
  //---------------------------------------------

  //---------------------------------------------
  // Drive transmision on MAC-PHY interface, for
  // modem standalone
  //---------------------------------------------
  task tx_ppdu_frame(ref PPDU_frame frm);
    mac_tx_seq = mac_tx_frame_seq::type_id::create("mac_tx_seq");
    mac_tx_seq.frame = frm;
    m_cfg.ppdu_tx_list.push_front(frm); // add PPDU frame to TX list, will be used by modem_tx_scoreboard
    mac_tx_seq.start(m_vsqr.m_mac_phy_vsqr.m_mac_sqr);
  endtask : tx_ppdu_frame

  //---------------------------------------------
  // drive PHY inputs
  //---------------------------------------------
  virtual task drive_phy_adc(ref PPDU_frame frm,
                             input bit dont_drive_adc = 0,
                             input bit dont_drive_noise = 0);

    // create sequences
    mac_rx_seq        = mac_rx_frame_seq::type_id::create("mac_rx_seq");
    adc_gen_seq       = modem_stream_gen_samples_seq#(`ADCWIDTH)::type_id::create("adc_gen_seq");
    adc_noagc_gen_seq = rui_gen_samples_seq::type_id::create("adc_noagc_gen_seq");

    if (mac_seq_completed == 1'b1) begin
      // fork off: mac_rx_seq to start frame reception and to put the PHY in Rx mode
      fork: START_MAC_RX
        begin
          mac_seq_completed = 1'b0;
          mac_rx_seq.frame = frame;
          mac_rx_seq.frame_num = num_rx_frames;
          mac_rx_seq.start(m_vsqr.m_mac_phy_vsqr.m_mac_sqr);
          mac_seq_completed = 1'b1;
        end
      join_none
    end

    // wait 5us
    if (!dont_drive_noise && (`AGC_ON == 1 && agcBypass == 0))
      insert_idle_cycles(5);

    // use the generated PPDU frame to calculate samples for ADC using the matlab model
    rf_gpio = m_cfg.m_radio_cfg.m_radio_ctrl_cfg.karst_gain;
    m_mdm_cfg.execute(direction, frm, rf_gpio, agcBypass);
    // trigger event that new PPDU is executed
    ->m_cfg.new_frame_mdm_exe;

    // skipp driving ADC samples
    if (dont_drive_adc) return;

    // get appropriate data depending on if AGC is enabled/disabled
    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      m_mdm_cfg.get_golden_data(adc_gen_seq.RFdata_re, adc_gen_seq.RFdata_im);
    end else begin
      m_mdm_cfg.get_raw_data(adc_noagc_gen_seq.rxFEout20_re,  adc_noagc_gen_seq.rxFEout20_im,
                             adc_noagc_gen_seq.rxFEout20S_re, adc_noagc_gen_seq.rxFEout20S_im,
                             adc_noagc_gen_seq.rxFEout40_re,  adc_noagc_gen_seq.rxFEout40_im,
                             adc_noagc_gen_seq.rxFEout80_re,  adc_noagc_gen_seq.rxFEout80_im,
                             adc_noagc_gen_seq.bfrSigmadB,    adc_noagc_gen_seq.sigmae2,
                             adc_noagc_gen_seq.snr_dB,        adc_noagc_gen_seq.noise_variance
                            );
      // get RXPARAMTERS
      adc_noagc_gen_seq.rxparams = m_mdm_cfg.get_rxparams();
    end

    // call appropriate sequence depending on if AGC is enabled/disabled
    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      // stop noise
      if (adc_noise_seq != null) begin
        adc_noise_seq.stop_noise = 1;
      end
      // Send calculated ADC samples using modem_stream_agent
      adc_gen_seq.start(m_vsqr.m_adc_sqr);
    end else begin
      fork
        adc_noagc_gen_seq.start(m_vsqr.m_rui_sqr);
        // read FFT out memory if this path is forced
        read_fft_out_mem();
        // write FFT input memory if this path is forced
        write_fft_in_mem();
      join
    end

    // start generating noise on ADC inputs
    if(!dont_drive_noise)
      drive_noise_adc();

  endtask : drive_phy_adc

  //---------------------------------------------
  // BFMEE configuration
  //---------------------------------------------
  task configure_bfmee_ctrl(bit [1:0] nc,
                            bit [1:0] nr,
                            bit [1:0] grouping,
                            bit       feedback,
                            bit       codebook,
                            bit [1:0] ch_bw,
                            bit       he = 0,
                            bit [5:0] ruindex_start = 0,
                            bit [5:0] ruindex_end = 0);

    mac_bf_cmd_seq mac_bfmee_cmd;
`ifdef RW_BFMEE_EN
    mac_bfmee_cmd = mac_bf_cmd_seq::type_id::create("mac_bfmee_cmd");
    mac_bfmee_cmd.seq_cmd = BFMEE_CFG;
    mac_bfmee_cmd.seq_Nc = nc;
    mac_bfmee_cmd.seq_Nr = nr;
    mac_bfmee_cmd.seq_grouping = grouping;
    mac_bfmee_cmd.seq_feedback_type = feedback;
    mac_bfmee_cmd.seq_codebook = codebook;
    mac_bfmee_cmd.seq_ch_bw = ch_bw;
    mac_bfmee_cmd.seq_he = he;
    mac_bfmee_cmd.seq_ruindex_start = ruindex_start;
    mac_bfmee_cmd.seq_ruindex_end = ruindex_end;
    mac_bfmee_cmd.start(m_vsqr.m_bf_vsqr.m_mac_bf_sqr);
    `uvm_info(get_type_name(), $sformatf("Configure BFMEE inputs"), UVM_LOW)
`else
    `uvm_info(get_type_name(), $sformatf("RW_BFMEE_EN not defined task will do nothing"), UVM_LOW)
`endif
  endtask : configure_bfmee_ctrl

  //---------------------------------------------
  // start beamforming
  //---------------------------------------------
  task bfmee_start();
    mac_bf_cmd_seq mac_bfmee_cmd;

`ifdef RW_BFMEE_EN
    mac_bfmee_cmd = mac_bf_cmd_seq::type_id::create("mac_bfmee_cmd");
    mac_bfmee_cmd.seq_cmd = BFMEE_CTRL;
    mac_bfmee_cmd.start(m_vsqr.m_bf_vsqr.m_mac_bf_sqr);
    `uvm_info(get_type_name(), $sformatf("Start BFMEE generation"), UVM_LOW)
`else
    `uvm_info(get_type_name(), $sformatf("RW_BFMEE_EN not defined task will do nothing"), UVM_LOW)
`endif
  endtask : bfmee_start

  //---------------------------------------------
  // read beamforming report from BFMEE memory
  //---------------------------------------------
  task read_bfm_report(bit [1:0] nc,
                       bit [1:0] nr,
                       bit [1:0] grouping,
                       bit       feedback,
                       bit       codebook,
                       bit [1:0] ch_bw,
                       bit       he = 0,
                       bit [5:0] ruindex_start = 0,
                       bit [5:0] ruindex_end = 0);
    int sample_num;
    mac_bf_cmd_seq mac_bfmee_cmd;

`ifdef RW_BFMEE_EN
    if (he)
     sample_num = he_bfm_report_size_estimator(.Nr(nr),
                                               .Nc(nc),
                                               .ch_bw(ch_bw),
                                               .grouping(grouping),
                                               .codebook(codebook),
                                               .feedbacktype(feedback),
                                               .ru_start_index(ruindex_start),
                                               .ru_end_index(ruindex_end),
                                               .debug(0));
    else
     sample_num = bfm_report_size_estimator(.Nr(nr),
                                            .Nc(nc),
                                            .ch_bw(ch_bw),
                                            .grouping(grouping),
                                            .codebook(codebook),
                                            .feedbacktype(feedback),
                                            .debug(0));

    `uvm_info(get_type_name(), $sformatf("Read beamforming report, sample num: %0d bytes",sample_num), UVM_LOW)
    mac_bfmee_cmd = mac_bf_cmd_seq::type_id::create("mac_bfmee_cmd");
    mac_bfmee_cmd.seq_cmd = BFMEE_REPORT;
    mac_bfmee_cmd.seq_bfm_sample_num = sample_num;
    mac_bfmee_cmd.start(m_vsqr.m_bf_vsqr.m_mac_bf_sqr);
`else
    `uvm_info(get_type_name(), $sformatf("RW_BFMEE_EN not defined task will do nothing"), UVM_LOW)
`endif
  endtask : read_bfm_report

  //---------------------------------------------
  // read H memory
  //---------------------------------------------
  task read_h_mem(bit [1:0] ch_bw, bit [2:0] num_rx, bit [2:0] num_ss);
    h_mem_wr_rd_seq h_mem_wr_rd;

    h_mem_wr_rd = h_mem_wr_rd_seq::type_id::create("h_mem_wr_rd");
    h_mem_wr_rd.seq_cmd    = H_MEM_RD;
    h_mem_wr_rd.seq_ch_bw  = ch_bw;
    h_mem_wr_rd.seq_num_rx = num_rx;
    h_mem_wr_rd.seq_num_ss = num_ss;
    h_mem_wr_rd.start(m_vsqr.m_bf_vsqr.m_h_mem_sqr);
    `uvm_info(get_type_name(), $sformatf("Read from H memory"), UVM_LOW)
  endtask : read_h_mem

  //---------------------------------------------
  // During Rx test case it can occur that modem
  // will not provide Rx end on MAC-PHY interface
  // so MAC driver can get stuck on waiting RxEnd.
  // This task will unlock driver and prevent hanging
  // of MAC driver.
  //---------------------------------------------
  task unlock_mac_driver();
    m_cfg.m_mac_phy_cfg.m_mac_cfg.stop_wating_rx_end = 1'b1;
    insert_idle_cycles(1);
    m_cfg.m_mac_phy_cfg.m_mac_cfg.stop_wating_rx_end = 1'b0;
    `uvm_info(get_type_name(), $sformatf("MAC driver has been unlocked in reception"), UVM_LOW)
  endtask : unlock_mac_driver

  //----------------------------------------------------------------------
  // Read out FFT output memory, task used when Time domain is tested
  //----------------------------------------------------------------------
  virtual task read_fft_out_mem();
    mem_read_seq    fft_out_rd_seq;
    rxparameters_s  rxparams;

    if (m_cfg.m_fft_out_cfg.is_active == UVM_ACTIVE && m_cfg.has_fft_out_agent) begin
      // fetch needed parameters for FFT sequence
      rxparams = m_mdm_cfg.get_rxparams();
      if (rxparams.format_mod_f == -1) return;
      fft_out_rd_seq = mem_read_seq::type_id::create("fft_out_rd_seq");
                                  // add data number of symbols with preamble
                                  // number of symbols
      fft_out_rd_seq.read_nsymb  =   rxparams.nsymb_f
                                   + get_preamble_nsymb(format_mod_e'(rxparams.format_mod_f));
      fft_out_rd_seq.format_mod  = format_mod_e'(rxparams.format_mod_f);
                                    // HE-TB
      fft_out_rd_seq.ru_type     = (rxparams.format_mod_f == 4'h8) ? get_ru_type(0) :
                                    // HE-SU, HE-EXT-SU
                                    (rxparams.format_mod_f inside {4'h5,4'h7})? get_ru_type_he_su(rxparams.cbw_f) :
                                    // HE-MU
                                    (rxparams.format_mod_f == 4'h6) ? ru_type_e'(rxparams.rutype_f) :
                                    // other frame formats
                                    RU242;
      fft_out_rd_seq.he_ltf_type = rxparams.he_ltf_type_f;
      fft_out_rd_seq.ch_bw       = rxparams.cbw_f;
      fft_out_rd_seq.num_he_sig_b= rxparams.he_sigb_f;
      fft_out_rd_seq.num_he_ltf  = rxparams.num_he_ltf_f;
      fft_out_rd_seq.dut_location= rxparams.ruindex_f;
      fft_out_rd_seq.start(m_vsqr.m_fft_out_sqr);
    end
  endtask : read_fft_out_mem

  //----------------------------------------------------------------------
  // Write to FFT intput memory, task used when Freq domain is tested
  //----------------------------------------------------------------------
  virtual task write_fft_in_mem();
    mem_write_seq    fft_in_wr_seq;
    rxparameters_s   rxparams;

    if (m_cfg.m_fft_in_cfg.is_active == UVM_ACTIVE && m_cfg.has_fft_in_agent) begin
      // fetch needed parameters for FFT sequence
      rxparams = m_mdm_cfg.get_rxparams();
      if (rxparams.format_mod_f == -1) return;
      fft_in_wr_seq = mem_write_seq::type_id::create("fft_in_wr_seq");
      // fetch FFT input data for sequence
      fft_in_wr_seq.fft_in_data  = m_mdm_cfg.get_fft_data_in();
      fft_in_wr_seq.format_mod   = format_mod_e'(rxparams.format_mod_f);
      fft_in_wr_seq.he_ltf_type  = rxparams.he_ltf_type_f;
      fft_in_wr_seq.num_he_ltf   = rxparams.num_he_ltf_f;
      fft_in_wr_seq.num_he_sig_b = rxparams.he_sigb_f;
      fft_in_wr_seq.gi_type      = rxparams.gi_type_f;
      fft_in_wr_seq.ch_bw        = rxparams.cbw_f;
      fft_in_wr_seq.dut_location = rxparams.dut_location_f;
      fft_in_wr_seq.start(m_vsqr.m_fft_in_sqr);
    end
  endtask : write_fft_in_mem

endclass : modem_seq_base


`endif //MODEM_SEQ_BASE_SV
