//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef WLAN_AXI_READ_WRITE_SEQ_SV
`define WLAN_AXI_READ_WRITE_SEQ_SV

  typedef struct {
     bit[14:0] lli_sram_adr;
     bit[31:0] src_adr;
     bit[31:0] dst_adr;
     bit[15:0] num_of_bytes;
     bit[3:0]  irq;
     bit[3:0]  cnt;      // Not used for the moment
     bit[31:0] nxt_adr;
     bit       first;
     bit       last;
  } lli_s;


class wlan_axi_read_write_seq extends wlan_seq_base;

  `uvm_object_utils(wlan_axi_read_write_seq)

  axi_slave_memory m_axi_slv_mem;

  lli_s lli_dma0_q[$]; // Downstream DMA
  lli_s lli_dma1_q[$]; // Downstream DMA
  lli_s lli_dma2_q[$]; // Upstream DMA
  lli_s lli_dma3_q[$]; // Upstream DMA

  function new (string name = "wlan_axi_read_write_seq");
    super.new (name);
  endfunction : new

  virtual task check_dma_stat();
    uvm_reg_data_t data;
    uvm_reg_data_t data_prev;
    bit            first_read_is_done;
    bit            done_fl;
    do begin
      // insert_idle_cycles(1);
      m_regmodel.get_reg_value(data, "DMA_STATUS");
      if(data_prev != data || first_read_is_done == 1'b0) begin
        `uvm_info(get_type_name(), $sformatf("DMA_STATUS = %b.", data), UVM_HIGH)
        first_read_is_done = 1'b1;
      end
      data_prev = data;
      done_fl = ~(|data[29:28]); // Upstream & downstream are not busy
    end
    while (done_fl != 1'b1);
  endtask : check_dma_stat

  // Init Sahred RAM.
  // This procedure allows to get rid of problems with X state in the DMA control logic,
  // but takes a too much time (slows down a simualtion significantly)
  virtual task init_sram();
    for(int unsigned i = 0; i < 65536; i++) begin
      write_word_sram({i[29:0], 2'b00}, 32'h0);
    end
  endtask : init_sram

  virtual task handle_queue_of_llis(input lli_s lli_q[$], input bit[1:0] dma_channel);
    lli_s lli;
    bit   last_lli;
    bit[14:0] lli_start_adr;
    string dma_reg_name;
    // Write chain of LLI to Shared RAM
    foreach(lli_q[j]) begin
      lli = lli_q[j];
      last_lli = (lli_q.size() == j-1);
      prepare_lli(lli.lli_sram_adr,
                  lli.src_adr,
                  lli.dst_adr,
                  lli.num_of_bytes,
                  (lli.last) ? 1'b1 : 1'b0,
                  dma_channel /*lli.irq*/,
                  '0,
                  '0,
                  lli.nxt_adr); // IRQ should be written only for the LAST LLI

      if(lli.first) lli_start_adr = lli.lli_sram_adr;
    end

    // Start LLI processing
    dma_reg_name = (dma_channel == 4) ? "CH_LLI_ROOT4" :
                   (dma_channel == 3) ? "CH_LLI_ROOT3" :
                   (dma_channel == 2) ? "CH_LLI_ROOT2" :
                   (dma_channel == 1) ? "CH_LLI_ROOT1" : "CH_LLI_ROOT0";

    m_regmodel.set_reg_value(lli_start_adr, dma_reg_name);
    // Wait for the end of transfer and clear IRQ status flag
    wait_for_lli_irq(1 << dma_channel);
    ack_dma_irq(0,0, 1 << dma_channel);

  endtask : handle_queue_of_llis


  virtual task config_axi_burst_length_limit(input bit[3:0] axi_burst_length);
    m_regmodel.set_reg_value({ 28'b0, axi_burst_length}, "AXI_BURST_LENGTH_LIMIT");
  endtask : config_axi_burst_length_limit

  // Init memory block in AXI slave RAM
  virtual task init_axi_mem_blk(input bit[31:0] address, input bit[31:0] block_length);
    bit [63:0]  mem_val;

    for(int i = 0; i < block_length; i++) begin
      mem_val = { address, i[31:0] };
      m_axi_slv_mem.write_mem(address + (i << 2), mem_val);
    end
  endtask : init_axi_mem_blk

  virtual task randomize_lli(ref lli_s lli_q[$], input bit upstream); // upstream -> Write AXI Mem, downstream -> Read AXI Mem
    lli_s        lli_tmp;
    int unsigned lli_queue_size;
    bit[14:0]    lli_start_address;
    lli_queue_size    = $urandom_range(1, 8);
    lli_start_address = $urandom_range(/*0*/ 'h4, 'h0100 - ((lli_queue_size+1) * 'h10)); // Where LLI is located -> lower part of SRAM
    lli_start_address = {lli_start_address[14:2], 2'b0}; // Word address
    for(int unsigned i = 0; i < lli_queue_size; i++) begin
      lli_tmp.lli_sram_adr = lli_start_address + 32'h10 * i ;  //
      lli_tmp.src_adr      = (upstream) ? ($urandom_range('h0100+1, 10000) & 32'hFFFFFFF8) : ($urandom_range(0, 16*1024-1)  & 32'hFFFFFFF8); //  READ SRAM ADDRESS RANGE : READ AXI ADDRESS RANGE
      lli_tmp.dst_adr      = (upstream) ? ($urandom_range(0, 16*1024-1) & 32'hFFFFFFF8)    : ($urandom_range('h0100+1, 10000) & 32'hFFFFFFF8); //  WRITE AXI ADDRESS RANGE : WRITE SRAM ADDRESS RANGE
      lli_tmp.num_of_bytes = $urandom_range(100, 65535);
      lli_tmp.irq          = 0;
      lli_tmp.cnt          = 0;
      lli_tmp.nxt_adr      = (i == lli_queue_size-1) ? 32'h0 : lli_start_address + 32'h10 * (i + 1); // Must be overwritten
      lli_tmp.first        = (i == 0) ? 1'b1 : 1'b0;
      lli_tmp.last         = (i == lli_queue_size - 1) ? 1'b1 : 1'b0;
      lli_q[i] = lli_tmp;
    end
      lli_q.shuffle();
  endtask : randomize_lli

  // ---------------------------------------------------------------------------------------
  // Wait until one ore more interrupt flag(s) are set
  // ---------------------------------------------------------------------------------------
  virtual task wait_for_dma_ch_irq(bit[2:0] dma_ch);
    uvm_reg_data_t data;
    do begin
      m_regmodel.get_reg_value(data, "INT_RAWSTATUS");
      `uvm_info(get_type_name(), $sformatf("INT_RAWSTATUS = %b.", data), UVM_HIGH)
    end
    while (data[20 + dma_ch] != 1'b1);
  endtask : wait_for_dma_ch_irq


  virtual task body();
    super.body();

    // AXI memory model
    if (!uvm_config_db #(axi_slave_memory)::get(null, "", "m_axi_slv_mem", m_axi_slv_mem))
      `uvm_fatal(get_type_name(), "AXI memory model missing!")

    init_axi_agent();

    // Init Sahred RAM. -> Not used, replaced by the init. of next LLI location
    // init_sram();

    // Init AXI memory
    init_axi_mem_blk(0, 32'h2000);

    // Change maximum length of AXI burst (ARLEN/AWLEN)
    config_axi_burst_length_limit(4'b1111);

    // Delay
    insert_idle_cycles(1);

    `uvm_info(get_type_name(), "Start of READ/WRITE", UVM_LOW)

    randcase

      1 : begin
        // Read from AXI memory
        randomize_lli(lli_dma2_q, 0);
        handle_queue_of_llis(lli_dma2_q, 2);
        wait_for_dma_ch_irq(2);

        // Write to AXI memory
        randomize_lli(lli_dma0_q, 1);
        handle_queue_of_llis(lli_dma0_q, 0);
        wait_for_dma_ch_irq(0);
      end


      1 : begin
        // Read from AXI memory
        randomize_lli(lli_dma3_q, 0);
        handle_queue_of_llis(lli_dma3_q, 3);
        wait_for_dma_ch_irq(3);

        // Write to AXI memory
        randomize_lli(lli_dma1_q, 1);
        handle_queue_of_llis(lli_dma1_q, 1);
        wait_for_dma_ch_irq(1);
      end
    endcase // randcase

    m_regmodel.set_field_value(1'b1, "AGCFSMRESET", "RWNXAGCCNTL"); //enable memory access

    `uvm_info(get_type_name(), "Generate downstream data transfer PHY", UVM_LOW)
    downstream_dma_transfer(.src_adr (32'h0100),
                            .dst_adr (`PHYCONFIGBASEADDRESS),
                            // Number of bytes to transfer
                            .num_of_bytes (16'h2000),
                            .dma_channel(2)
                            );

    `uvm_info(get_type_name(), "Generate upstream data transfer PHY", UVM_LOW)
    upstream_dma_transfer(.src_adr (`PHYCONFIGBASEADDRESS+32'h0a00),
                          .dst_adr (32'h0a00),
                          // Number of bytes to transfer
                          .num_of_bytes (16'h2000),
                          .dma_channel(0)
                         );

    `uvm_info(get_type_name(), "Generate downstream data transfer PHY", UVM_LOW)
    downstream_dma_transfer(.src_adr (32'h0100),
                            .dst_adr (`PHYCONFIGBASEADDRESS),
                            // Number of bytes to transfer
                            .num_of_bytes (16'h200),
                            .dma_channel(3)
                            );

    `uvm_info(get_type_name(), "Generate upstream data transfer PHY", UVM_LOW)
    upstream_dma_transfer(.src_adr (`PHYCONFIGBASEADDRESS),
                          .dst_adr (32'h0a00),
                          // Number of bytes to transfer
                          .num_of_bytes (16'h200),
                          .dma_channel(1)
                         );

`ifdef RW_EMBEDDED_LA
    `uvm_info(get_type_name(), "Generate downstream data transfer LA", UVM_LOW)
    downstream_dma_transfer(.src_adr (32'h0100),
                            .dst_adr (`LABASEADDRESS),
                            // Number of bytes to transfer
                            .num_of_bytes (16'hFFF),
                            .dma_channel(3)
                            );

    `uvm_info(get_type_name(), "Generate upstream data transfer LA", UVM_LOW)
    upstream_dma_transfer(.src_adr (`LABASEADDRESS),
                          .dst_adr (32'h0a00),
                          // Number of bytes to transfer
                          .num_of_bytes (16'hFFF),
                          .dma_channel(1)
                         );
`endif//RW_EMBEDDED_LA

    // AGC MEMORY ACCESS
    `uvm_info(get_type_name(), "Generate downstream data transfer AGC", UVM_LOW)
    downstream_dma_transfer(.src_adr (($urandom_range('h0100+1, 10000) & 32'hFFFFFFF4)),
                            .dst_adr (`AGCMEMBASEADDR),
                            // Number of bytes to transfer
                            .num_of_bytes (16'h0FFF),
                            .dma_channel(2)
                            );

    `uvm_info(get_type_name(), "Generate upstream data transfer AGC", UVM_LOW)
    upstream_dma_transfer(.src_adr (`AGCMEMBASEADDR),
                          .dst_adr (($urandom_range(0, 16*1024-1) & 32'hFFFFFFF4)),
                          // Number of bytes to transfer
                          .num_of_bytes (16'h0FFF),
                          .dma_channel(0)
                         );

    // MIDSTREAM ACCESS
    m_cfg.mac_dma_check_en = 1'b0; //disable checking for midstream

    `uvm_info(get_type_name(), "Generate midstream data transfer PHY", UVM_LOW)
    midstream_dma_transfer(.src_adr (32'h100),
                           .dst_adr (`PHYCONFIGBASEADDRESS),
                           .num_of_bytes (16'd2000)
                           );

    midstream_dma_transfer(.src_adr (`PHYCONFIGBASEADDRESS),
                           .dst_adr (32'h800),
                           .num_of_bytes (16'd4400)
                           );

    `uvm_info(get_type_name(), "Generate midstream data transfer AGC memory", UVM_LOW)
    midstream_dma_transfer(.src_adr (32'h3a0),
                           .dst_adr (`AGCMEMBASEADDR),
                           .num_of_bytes (16'd1500)
                           );

    midstream_dma_transfer(.src_adr (`AGCMEMBASEADDR),
                           .dst_adr (32'h400),
                           .num_of_bytes (16'd1400)
                           );

`ifdef RW_EMBEDDED_LA
    `uvm_info(get_type_name(), "Generate midstream data transfer LA", UVM_LOW)
    midstream_dma_transfer(.src_adr (32'h0100),
                           .dst_adr (`LABASEADDRESS),
                           // Number of bytes to transfer
                           .num_of_bytes (16'hFFF)
                           );

    `uvm_info(get_type_name(), "Generate midstream_dma_transfer data transfer LA", UVM_LOW)
    midstream_dma_transfer(.src_adr (`LABASEADDRESS),
                           .dst_adr (32'h0a00),
                           // Number of bytes to transfer
                           .num_of_bytes (16'hFFF)
                          );
`endif//RW_EMBEDDED_LA

    `uvm_info(get_type_name(), "Platform AXI Read/Write test is completed.", UVM_LOW)
  endtask : body

endclass : wlan_axi_read_write_seq


`endif // WLAN_AXI_READ_WRITE_SEQ_SV
