//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef WLAN_BF_MU_RX_HE_SEQ_SV
`define WLAN_BF_MU_RX_HE_SEQ_SV


class wlan_bf_mu_rx_he_seq extends wlan_seq_base;

  `uvm_object_utils(wlan_bf_mu_rx_he_seq)

  PPDU_frame                 ndpa_frame;
  PPDU_frame                 ndp_frame;
  PPDU_frame                 trigger_frame;
  TRIGGER_frame              trig_frame;
  HE_NDP_ANNOUNCEMENT_frame  ndpa;
  PPDU_frame                 data_frame;

  bit [31:0]                 rx_buff1_rd_ptr;
  bit [31:0]                 rx_buff2_rd_ptr;
  int                        ksr_index;
  int                        frame_num;
  int                        rhd_cnt;
  int                        rpd_cnt;
  int                        STAID;
  bit [15:0]                 AID;
  bit [3:0]                  TID;
  bit [11:0]                 SSN;
  bit [1:0]                  primary_channel;
  he_control_s               he_ctrl;
  int                        bfm_report_size;
  int                        txtime;
  int                        bfr_he_len;
  int                        bfr_leg_len;
  HESIGB_s                   HESIGB;
  bit [1:0]                  gi_type;
  bit [1:0]                  he_ltf_type;
  bit [3:0]                  num_he_ltf;
  bit                        mma;
  int                        user;

  //--------------------------------------------------------------------
  // variables for NDP
  //--------------------------------------------------------------------
  rand bit  [2:0] Nr;                         // Nr Value To Test
  rand bit  [2:0] Nc[`NB_STA_MAX];            // Nc Value To Test
  rand bit        Grouping[`NB_STA_MAX];      // Grouping Value To Test
  rand bit        FeedbackType[`NB_STA_MAX];  // Feedback Type Value To Test
  rand bit        Codebook[`NB_STA_MAX];      // Codebook Value To Test
  rand bit [5:0]  ruindex_start;              // RU start index
  rand bit [5:0]  ruindex_end;                // RU end index
  rand int        num_of_stations;            // number of stations, used for BF Tx test
  rand bit [1:0]  calib_ch_bw;

  //--------------------------------------------------------------------
  // Constraints
  //--------------------------------------------------------------------

`ifdef RW_MUMIMO_RX_EN
  constraint c_Nr {
    Nr inside {1, 3};
  }
`else
  constraint c_Nr {
    Nr == 1;
  }
`endif // RW_MUMIMO_RX_EN
  constraint c_FeedbackType {
    foreach (FeedbackType[i]) FeedbackType[i] == 1;
  }

  constraint c_Nc {
    solve Nr before Nc;
`ifdef RW_TXRX_1X1
    foreach (Nc[i]) Nc[i] == 0;
`else
    foreach (Nc[i]) Nc[i] <= 1;
`endif
  }

  constraint c_channel_bw {
    calib_ch_bw == channel_bw_func(HE_SU);
  }

  constraint c_num_of_stations {
    num_of_stations inside {[2:4]};
  }

  constraint c_ru_index {
    solve calib_ch_bw before ruindex_start;
    solve calib_ch_bw before ruindex_end;

    if (calib_ch_bw == 2'b00) {
      ruindex_start  inside {[0:8]}; // RU start index
      ruindex_end    inside {[0:8]}; // RU end index
    } else if (calib_ch_bw == 2'b01) {
      ruindex_start inside {[0:17]}; // RU start index
      ruindex_end   inside {[0:17]}; // RU end index
    } else {
      ruindex_start inside {[0:36]}; // RU start index
      ruindex_end   inside {[0:36]}; // RU end index
    }
    // RU start index must be less or equal to RU end index
    ruindex_start <= ruindex_end;

    foreach(Grouping[i]) {
      if (4**(Grouping[i] + 1) == 16)
        Codebook[i] == 1;
    }
  }


  function new (string name = "wlan_bf_mu_rx_he_seq");
    super.new (name);
  endfunction : new

  function void print_ndp_params(int i=0);
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("Reception of HE NDP Sounding procedure user %0d", i), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nr                : %2d", (Nr+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nc                : %2d", (Nc[i]+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Grouping          : %2d", Grouping[i]      ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Feedback type     : %s" , (FeedbackType[i]==1'b0)? "SU" : "MU"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Codebook          : %2d", Codebook[i]   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Channel BW        : %2d", calib_ch_bw   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   RUStartIndex      : %2d", ruindex_start   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   RUEndIndex        : %2d", ruindex_end   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
  endfunction: print_ndp_params

  function void set_ndp_params();
    m_mdm_cfg.Nr = Nr;
    m_mdm_cfg.nBeamformingSTA = num_of_stations;
    for (int i=0; i<num_of_stations; i++) begin
      m_mdm_cfg.Nc[i]           = Nc[i];
      m_mdm_cfg.Grouping[i]     = Grouping[i];
      m_mdm_cfg.Feedback[i]     = FeedbackType[i];
      m_mdm_cfg.Codebook[i]     = Codebook[i];
      m_mdm_cfg.RUStartIndex[i] = ruindex_start;
      m_mdm_cfg.RUEndIndex[i]   = ruindex_end;
      print_ndp_params(i);
    end
  endfunction : set_ndp_params

  virtual task body();
    super.body();

    // Maximum Nss supported set to 4
    m_regmodel.set_field_value(3'd3, "RXNSSMAX", "RXMODE", "PHYCONFIG");
    m_regmodel.set_field_value(0, "OFDMONLY", "RWNXAGCCNTL", "RIUKARST");

    agcBypass = m_cfg.m_radio_cfg.m_rui_cfg.agcBypass; // set from testcases

    // fork off gain update task
    fork  : GAIN_UPDATE
      begin // thread 1
        if ((`AGC_ON == 1) && (agcBypass == 0)) refresh_RF_data(); // call AGC update task only if necessary
      end

      drive_noise_adc();
    join_none

    //-----------------------------------------
    // determine device and BSS address
    //-----------------------------------------
    create_keystorageRAM();

    create_dev_and_bss_addr();

    // set RX interrupt masks
    m_regmodel.set_field_value(1'b1, "maskrxBuffer1Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masktimerRxTrigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "maskrxBuffer2Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masterTxRxIntEn", "TXRXINTUNMASKREG"); // Enables interrupt generation
    m_regmodel.set_field_value(1'b0, "disableRxBuffer2");
    m_regmodel.set_field_value(1'b1, "acceptTriggerSWFrames", "HECONFIGREG");
    m_regmodel.set_field_value(1'b1, "acceptTriggerHWFrames", "HECONFIGREG");
    // configure Beamformee register
    m_regmodel.set_field_value(3'b101, "bfrFormatMod", "BFMEECONTROLREG"); //HE-SU
    m_regmodel.set_field_value(1'h1, "bfmeeEnable", "BFMEECONTROLREG");

    // set AID value for this receiving station
    AID = $urandom_range(1,2007);
    m_regmodel.set_field_value(AID, "aid", "BCNCNTRL2REG");
    `uvm_info(get_type_name(),$sformatf("The AID of the STA is %d", AID), UVM_LOW)

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    ndpa_frame    = new("ndpa_frame");
    ndp_frame     = new("ndp_frame");
    ndpa          = new("ndpa");
    trigger_frame = new("trigger_frame");
    trig_frame    = new("trig_frame");

    // prevent checking of TSF reg
    m_cfg.check_mac_tsf_reg = 0;

    // randomize fields in sequence
    assert (this.randomize());
    set_ndp_params();

    prepare_sram_for_rx(10000);
    prepare_sram_priority_rx(5000);
    rx_buff1_rd_ptr = m_regmodel.get_mirrored_reg_value("RXBUF1RDPTRREG") + rhd_header;
    rx_buff2_rd_ptr = m_regmodel.get_mirrored_reg_value("RXBUF2RDPTRREG") + rhd_header;

    STAID = $urandom_range(0, num_of_stations - 1);
    `uvm_info(get_type_name(), $sformatf("STAID used in NDPA: %0d",STAID), UVM_LOW)
    // overwrite register BFMEE with randomized parameters
    m_regmodel.set_field_value(FeedbackType[STAID], "bfmeeMUSupport", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Codebook[STAID], "bfmeeCodebook", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Grouping[STAID], "bfmeeGrouping", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Nr, "bfmeeNr", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Nc[STAID], "bfmeeNc", "BFMEECONTROLREG");

    //--------------------------------
    // create NDPA frame for Rx
    //--------------------------------

    assert (ndpa_frame.randomize() with {
      kind == SINGLETON;
      ppdu_format == HE_SU;
      tx_frame == 0;
    });
    ndpa_frame.preamble_header.ch_bw = calib_ch_bw;

    if (calib_ch_bw >= 2'b01) begin
      ndpa_frame.preamble_header.user_header_he[0].fec_coding_f = 1;
    end

    // create NDPA MPDU
    assert (ndpa.randomize() with {
      ch_bw                          == local::calib_ch_bw;
      sta_info.size()                == num_of_stations;
      sta_info[STAID].nc_index_f         == Nc[STAID];
      sta_info[STAID].codebook_size_f    == Codebook[STAID];
      sta_info[STAID].feedback_type_ng_f[0] == FeedbackType[STAID];
      sta_info[STAID].feedback_type_ng_f[1] == Grouping[STAID]; //Ng
      sta_info[STAID].aid11_f            == AID[10:0]; // 11 LSB bits
      sta_info[STAID].disambiguation_f   == AID[11];   // MSB bit
      sta_info[STAID].ru_end_index_f     == ruindex_end;   //Partial BW info
      sta_info[STAID].ru_start_index_f   == ruindex_start; //Partial BW info
    });

    ndpa_frame.ampdu_frame[0].clear_all_arrays();
    assert (ndpa_frame.ampdu_frame[0].randomize() with {
      ndpa_frame.ampdu_frame[0].ppdu_format == ndpa_frame.ppdu_format;
      ndpa_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
      ndpa_frame.ampdu_frame[0].mpdu_frame_type[0] == HE_NDP_ANNOUNCEMENT;
      ndpa_frame.ampdu_frame[0].aggregated == 0;
    });
    ndpa_frame.ampdu_frame[0].mpdu_frame[0] = ndpa;
    ndpa_frame.ampdu_frame[0].create_delimiter();
    ndpa_frame.ampdu_frame[0].add_padding();

    ksr_index = get_random_ksr_index(ndpa_frame);
    // set MAC addresses
    ndpa_frame.ampdu_frame[0].set_MAC_addr(
      .RA    (`BROADCAST_ADDR),
      .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
      .BSSID (bss_addr)
    );
    ndpa_frame.calc_leg_ht_length();


    `uvm_info(get_type_name(),
    $sformatf("NDPA:\n%s", ndpa_frame.sprint()), UVM_LOW)

    //--------------------------------
    // create NDP frame for Rx
    //--------------------------------
    assert (ndp_frame.randomize() with {
      kind == NDP;
      ppdu_format == HE_SU;
      tx_frame == 0;
    });
    // channel bandwidth needs to be same as in NDPA
    ndp_frame.preamble_header.ch_bw = ndpa_frame.preamble_header.ch_bw;
    ndp_frame.preamble_header.user_header_he[0].nss_f = Nr;
    // constraint FEC to LDPC in CHBW40
    if (ndp_frame.preamble_header.ch_bw >= 3'b001) begin
      ndp_frame.preamble_header.user_header_he[0].fec_coding_f = 1;
    end

    `uvm_info(get_type_name(),
    $sformatf("NDP:\n%s", ndp_frame.sprint()), UVM_LOW)

    //--------------------------------
    // create BFRP Trigger frame
    //--------------------------------

    assert (trig_frame.randomize() with {
      trigger_type == BMF_REPORT_POLL;
      common_info.bw_f == ndp_frame.preamble_header.ch_bw;
      user_info.size() == num_of_stations;
    });
    trig_frame.user_info[STAID].aid12_f = AID[11:0];
    trig_frame.user_info[STAID].ru_allocation_f[7:1] = $urandom_range(int'(ruindex_start), int'(ruindex_end));


    // set maximum MCS supported
    m_regmodel.set_field_value(trig_frame.MAX_MCS, "maxMCSInHETB");

    // Get the right size of the BFM, to set correctly the
    // length in the Trigger frame
    bfm_report_size = he_bfm_report_size_estimator( .Nr(Nr),
                                                    .Nc(Nc[STAID]),
                                                    .ch_bw(calib_ch_bw),
                                                    .grouping(Grouping[STAID]),
                                                    .codebook(Codebook[STAID]),
                                                    .feedbacktype(FeedbackType[STAID]),
                                                    .ru_start_index(ruindex_start),
                                                    .ru_end_index(ruindex_end),
                                                    .debug(0));

    `uvm_info(get_type_name(),$sformatf("Expected beamforming report size = %0d",bfm_report_size),UVM_LOW)
    // calculate size of BFR frame size which will be in HE-TB
    // MAC header + category and action + HE MUMIMO cotrol + BFR size
    // + delimiter + FCS
    bfr_he_len = 28 + 2 + 5 + bfm_report_size + 4 + 4;
    // add padding bytes
    bfr_he_len += (bfr_he_len % 4 != 0) ? (4 - (bfr_he_len % 4)) : 0;
    bfr_he_len += 10; //add more dummy bytes to avoid phy_err flag
    // determine NumHELTF and midamble_periodicity
    if (trig_frame.common_info.doppler_f)
      {num_he_ltf,mma} = (trig_frame.common_info.num_of_he_ltf_symbols_f == 0) ? {4'h0,1'b0} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 1) ? {4'h1,1'b0} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 2) ? {4'h3,1'b0} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 4) ? {4'h0,1'b1} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 5) ? {4'h1,1'b1} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 6) ? {4'h3,1'b1} : {4'h0,1'b0};
    else
      {num_he_ltf,mma} = (trig_frame.common_info.num_of_he_ltf_symbols_f == 0) ? {4'h0,1'b0} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 1) ? {4'h1,1'b0} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 2) ? {4'h3,1'b0} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 3) ? {4'h5,1'b0} :
                         (trig_frame.common_info.num_of_he_ltf_symbols_f == 4) ? {4'h7,1'b0} : {4'h0,1'b0};

    {gi_type, he_ltf_type} = (trig_frame.common_info.gi_and_ltf_type_f == 0) ? {2'b01, 2'b00} :
                             (trig_frame.common_info.gi_and_ltf_type_f == 1) ? {2'b01, 2'b01} :
                             (trig_frame.common_info.gi_and_ltf_type_f == 2) ? {2'b10, 2'b10} : {2'b00 ,2'b00};

    txtime = ComputeTimeOnAirAX (.length_i          (bfr_he_len),
                                 .preType_i         (HE_TB),
                                 .mcsIndex_i        (trig_frame.user_info[STAID].mcs_f),
                                 .giType_i          (gi_type),
                                 .ruType_i          (get_ru_type(trig_frame.user_info[STAID].ru_allocation_f)),
                                 .heLtfType_i       (he_ltf_type),
                                 .numHeLtf_i        (num_he_ltf),
                                 .dcm_i             (trig_frame.user_info[STAID].dcm_f),
                                 .packetExtension_i (trig_frame.common_info.pre_fec_padding_f),
                                 .heTbLength_i      (0),
                                 .triggerMethod_i   (0),
                                 .doppler_i         (trig_frame.common_info.doppler_f),
                                 .mma_i             (mma),
                                 .stbc_i            (trig_frame.common_info.stbc_f),
                                 .woPreamble_i      (0),
                                 .heSigB_i          (HESIGB),
                                 .debug_i           (0));

    // calculate L_LENGTH
    bfr_leg_len = $ceil((txtime - 20)/4.0) * 3 - 3 - 2;
    bfr_leg_len += (bfr_leg_len % 3 == 2) ? 2 :
                   (bfr_leg_len % 3 == 0) ? 1 : 0;
    // set L_LENGTH for trigger frame
    trig_frame.common_info.length_f = bfr_leg_len;
    trig_frame.common_info.ldpc_extra_symbol_segment_f = (get_ru_type(trig_frame.user_info[STAID].ru_allocation_f) >= RU484);
    // create frame body with new field values
    trig_frame.custom_post_randomize();

    assert (trigger_frame.randomize() with {
      kind == SINGLETON;
      tx_frame == 0;
      ppdu_format == HE_SU;
    });

    trigger_frame.ampdu_frame[0].clear_all_arrays();
    assert (trigger_frame.ampdu_frame[0].randomize() with {
      trigger_frame.ampdu_frame[0].ppdu_format == trigger_frame.ppdu_format;
      trigger_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
      trigger_frame.ampdu_frame[0].mpdu_frame_type[0] == TRIGGER;
      trigger_frame.ampdu_frame[0].aggregated == 0;
    });
    trigger_frame.ampdu_frame[0].mpdu_frame[0] = trig_frame;
    trigger_frame.ampdu_frame[0].create_delimiter();
    trigger_frame.ampdu_frame[0].add_padding();

    // set MAC addresses
    trigger_frame.ampdu_frame[0].set_MAC_addr(
      .RA    (`BROADCAST_ADDR),
      .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f)
    );
    //trigger_frame.setup_rx_encrypted_data();
    trigger_frame.calc_leg_ht_length();

    `uvm_info(get_type_name(), $sformatf("BMFR TRIGGER frame:\n%s", trigger_frame.sprint()), UVM_LOW)

    // receive ACK and wait for interrupt Rx trigger
    fork : RECEIVE_NDPA_NDA_MU_CALIBRATIO
      // if frames if going to be received trigger will occur,
      // if frames if filtered event will occur
      fork
        wait_mac_rx_trigger();
        rx_frame_filtered();
      join_any

      begin
        // PHY sends frame to MAC core
        drive_phy_adc(ndpa_frame,0,1);
        insert_idle_cycles(10);
        drive_phy_adc(ndp_frame);
        wait_rx_end();
      end
    join

    // only if frame was stored in SRAM do read
    if (!stop_pooling) begin
      `uvm_info(get_type_name(),
      "MAC core received frame and stored it to SRAM", UVM_LOW)
      rd_frame_from_sram(rx_buff1_rd_ptr);
      // get next RHD pointer value if frame was not filtered
      get_next_rhd_ptr(1, rx_buff1_rd_ptr, rx_buff1_rd_ptr);
      rd_frame_from_sram(rx_buff1_rd_ptr);
      // get next RHD pointer value if frame was not filtered
      get_next_rhd_ptr(1, rx_buff1_rd_ptr, rx_buff1_rd_ptr);
    end

    insert_idle_cycles(2);
    // clear interrupt flag
    m_regmodel.set_field_value(1'b1, "clearrxBuffer1Trigger", "TXRXINTEVENTCLEARREG");
    m_regmodel.set_field_value(1'b1, "cleartimerRxTrigger", "TXRXINTEVENTCLEARREG");
    m_regmodel.set_field_value(1'b1, "clearrxBuffer2Trigger", "TXRXINTEVENTCLEARREG");

    // receive ACK and wait for interrupt Rx trigger
    fork : RECEIVE_BFR_TRIGGER_FRAME
      // if frames if going to be received trigger will occur,
      // if frames if filtered event will occur
      fork
        wait_mac_rx_trigger();
        rx_frame_filtered();
      join_any

      begin
        drive_phy_adc(trigger_frame);
        // --------------------------------------------------------------------
        // read precompensation status registers in order to mirror there value
        // since there values are used in MDM STA setup (to avoid backdoor
        // access)
        // --------------------------------------------------------------------
        fork
          begin
            wait_rx_end();
            m_regmodel.get_reg_value(rdata,"RXCFOEST");
            m_regmodel.get_reg_value(rdata,"RXSFOEST");
          end
        // --------------------------------------------------------------------
          begin
            wait_tx_end();// wait BFR
          end
        join
      end
    join

    // only if frame was stored in SRAM do read
    if (!stop_pooling) begin
      `uvm_info(get_type_name(),
      "MAC core received frame and stored it to SRAM", UVM_LOW)
      rd_frame_from_sram(rx_buff2_rd_ptr);
      // get next RHD pointer value if frame was not filtered
      get_next_rhd_ptr(2, rx_buff2_rd_ptr, rx_buff2_rd_ptr);
    end

    insert_idle_cycles(1);
    // clear interrupt flag
    m_regmodel.set_field_value(1'b1, "clearrxBuffer1Trigger", "TXRXINTEVENTCLEARREG");
    m_regmodel.set_field_value(1'b1, "cleartimerRxTrigger", "TXRXINTEVENTCLEARREG");
    m_regmodel.set_field_value(1'b1, "clearrxBuffer2Trigger", "TXRXINTEVENTCLEARREG");

    insert_idle_cycles(20);

    frame_num = $urandom_range(3,5);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Receive Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      data_frame = new("data_frame");
      //--------------------------------
      // create data frame for Rx
      //--------------------------------
      assert (data_frame.randomize() with {
        kind == AGGREGATED;
        ppdu_format == HE_MU;
        tx_frame == 0;
      });

      user = data_frame.preamble_header.mu_mimo_userid;
      m_regmodel.set_reg_value(data_frame.preamble_header.user_header_he[user].staid_f,"HE_STAID_TAB0","PHYCONFIG");

      // Set the PRIMARYIND register
      if (data_frame.ppdu_format >= HE_SU) begin
        c_channel_bw.constraint_mode(0);
        // find RU allocation from HE control
        he_ctrl = get_ht_control(data_frame, user);
        // set proper RU allocation
        set_he_ctrl_ru_allocation(data_frame, he_ctrl, user);
        // set primary channle in register
        primary_channel = get_primary_channel(he_ctrl.ctrl_info_f.ru_allocation_f);
        m_regmodel.set_reg_value(primary_channel,"PRIMARYIND","PHYCONFIG");
        `uvm_info(get_type_name(),$sformatf("Primary channel set to %0d, for RU allocation %0d",
        primary_channel, he_ctrl.ctrl_info_f.ru_allocation_f),UVM_LOW)
      end

      ksr_index = get_random_ksr_index(data_frame, user);

      // set MAC addresses
      data_frame.ampdu_frame[user].set_MAC_addr(
        .RA    (dev_addr),
        .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
        .BSSID (bss_addr)
      );

      // get the TID for every frame and update TID table
      TID = data_frame.ampdu_frame[user].get_MAC_tid_num();
      SSN = data_frame.ampdu_frame[user].get_MAC_seq_num();
      for (int i = 0; i < m_cfg.TID_table.size(); i++) begin
        if (m_cfg.TID_table[i].TID == TID && ksr_index == m_cfg.TID_table[i].ksr_index) begin
          SSN = m_cfg.TID_table[i].SSN + `WINDOWSIZE + 2;
          data_frame.ampdu_frame[user].set_MAC_seq_num(SSN);
          break;
        end
      end

      // Write ksr_index into TID_table
      m_cfg.ksr_index = ksr_index;

      data_frame.setup_rx_encrypted_data();
      data_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(),
      $sformatf("PPDU_frame:\n%s", data_frame.sprint()), UVM_LOW)

      // get number of MPDUs in A-MPDU and byte size
      rhd_cnt = data_frame.ampdu_frame[user].mpdu_frame_type.size()*4*`RHD_LEN;
      rpd_cnt = data_frame.ampdu_frame[user].size();
      // prepare SRAM for Rx
      prepare_sram_for_rx(rhd_cnt + rpd_cnt);
      m_regmodel.get_reg_value(rdata, "RXBUF1RDPTRREG");
      rx_buff1_rd_ptr = rdata + rhd_header;

      // receive ACK and wait for interrupt Rx trigger
      fork
        // if frames if going to be received trigger will occur,
        // if frames if filtered event will occur
        begin
          fork
            wait_mac_rx_trigger();
            rx_frame_filtered();
          join_any
        end
        // if there is no response from MAC wait for sync
        if (responde_to_frame(data_frame, dev_addr)) begin
          // PHY sends frame to MAC core
          drive_phy_adc(data_frame);
          // --------------------------------------------------------------------
          // read precompensation status registers in order to mirror there value
          // since there values are used in MDM STA setup (to avoid backdoor
          // access)
          // --------------------------------------------------------------------
          fork
            begin
              wait_rx_end();
              m_regmodel.get_reg_value(rdata,"RXCFOEST");
              m_regmodel.get_reg_value(rdata,"RXSFOEST");
            end
          // --------------------------------------------------------------------
            begin
              wait_tx_end();// wait ACK
            end
          join
          drive_noise_adc();
        end
        else begin
          // PHY sends frame to MAC core
          drive_phy_adc(data_frame);
          no_resp_trigger.wait_trigger();
        end
      join

      // only if frame was stored in SRAM do read
      if (!stop_pooling) begin
        `uvm_info(get_type_name(),
        "MAC core received frame and stored it to SRAM", UVM_LOW)
        rd_frame_from_sram(rx_buff1_rd_ptr);
      end

      // clear interrupt flag
      m_regmodel.set_field_value(1'b1, "clearrxBuffer1Trigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "cleartimerRxTrigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "clearrxBuffer2Trigger", "TXRXINTEVENTCLEARREG");

      // To give a time for interrupt comparison
      insert_idle_cycles(10);

    end //for

  endtask : body

endclass : wlan_bf_mu_rx_he_seq


`endif //WLAN_BF_MU_RX_HE_SEQ_SV



